package Direct::YT::monitor_stats::mobile_multiplier_pct;

=head1 NAME

    Direct::YT::monitor_stats::mobile_multiplier_pct - метрики по использованию мобильных коэффициентов

    Мобильные объявления
    https://st.yandex-team.ru/DIRECT-39705
    
    Количество активных кампаний с установленным понижающим мобильным коэффициентом (сюда считаем если есть не только на кампанию, но и хоть одна активная группа с коэфф)
    Количество активных групп с установленным понижающим мобильным коэффициентов
    Количество активных кампаний с установленным повышающим мобильным коэффициентом (сюда считаем если есть не только на кампанию, но и хоть одна активная группа с коэфф)
    Распределение количества активных групп в диапазонах заданных ограничений в % от всех активных групп (группа наследует коэфф кампании, если в ней не задано своего)
    Диапазоны: -50 -40, -39 -30, -29 -20, -19 -10, -10 не задан (не включая), не задан (не включая) 50%, 51% 100%, 101% 200%, 201% 300%, 301% 600%, 601% 900%, 901% 1200%
    Аналогичное распределение по количеству активных кампаний в % от всех активных (смотрим только на настройку на уровне кампании)

=cut

use strict;
use warnings;
use utf8;
use feature 'state';

use parent 'Direct::YT::monitor_stats::base';

my @ranges = (
    {from => -50, to => -40},
    {from => -39, to => -30},
    {from => -29, to => -20},
    {from => -19, to => -10},
    {from => -9, to => -1},
    {from => 1, to => 50},
    {from => 51, to => 100},
    {from => 101, to => 200},
    {from => 201, to => 300},
    {from => 301, to => 600},
    {from => 601, to => 900},
    {from => 901, to => 1200},
    );
my %serv_types = (
    all => "все",
    manager => "менеджерские",
    agency => "агентские",
    self => "самоходные",
    );
    
sub prefix {
    return 'mobile.mobile_multiplier_pct_';
}

sub metrics {
    my %types = (
        all_camps => "всего активных кампаний",
        negative_camps => "активные кампании с установленным понижающим мобильным коэффициентом",
        positive_camps => "активные кампании с установленным повышающим мобильным коэффициентом",
        all_groups => "всего активных групп",
        negative_groups => "активные группы с установленным понижающим мобильным коэффициентом",
        positive_groups => "активные группы с установленным повышающим мобильным коэффициентом",
        );
    for my $range (@ranges) {
        $types{"range_$range->{from}_$range->{to}_groups"} = "групп с ограничением: $range->{from} .. $range->{to}";
        $types{"range_$range->{from}_$range->{to}_camps"} = "кампаний с ограничением: $range->{from} .. $range->{to}";
    }
    
    my %ret;
    while(my ($type, $desc) = each %types) {
        for my $serv (keys %serv_types) {
            $ret{"${type}_$serv"} = "$desc ($serv_types{$serv})";
        }
    }

    return \%ret;
}

sub used_fields {
    return {
        campaigns => [qw/statusActive AgencyUID ManagerUID mobile_multiplier_pct sum sum_spent wallet_sum wallet_sum_spent/],
        phrases => [qw/statusModerate mobile_multiplier_pct/],
        banners => [qw/statusActive/],
        hierarchical_multipliers => [qw/is_enabled multiplier_pct/],
    };
}

sub on_adgroup {
    my ($self, $adgroup) = @_;
    my $camp = $adgroup->{campaign};
    my $cid = $camp->{cid};
    my $pid = $adgroup->{pid};

    return unless ($camp->{sum}||0) + ($camp->{wallet_sum}||0) > ($camp->{sum_spent}||0) + ($camp->{wallet_sum_spent}||0)
        && $camp->{statusActive} eq 'Yes'
        && $adgroup->{statusModerate} eq 'Yes'
        && (grep {$_->{statusActive} eq 'Yes'} @{$adgroup->{banners}});

    $self->state($camp)->{is_good} = 1;

    my $serv_type = $camp->{AgencyUID} ? 'agency' : $camp->{ManagerUID} ? 'manager' : 'self';

    for my $serv ('all', $serv_type) {
        my $group_pct = $adgroup->{hierarchical_multipliers}{mobile_multiplier}{multiplier_pct};

        $self->sum("all_groups_$serv" => 1);
        if (defined $group_pct && $group_pct < 100) {
            $self->sum("negative_groups_$serv" => 1);
        } elsif (defined $group_pct && $group_pct > 100) {
            $self->sum("positive_groups_$serv" => 1);
        }

        if (my $range = _range($group_pct)) {
            $self->sum("range_$range->{from}_$range->{to}_groups_$serv" => 1);
        }
    }
}


sub on_campaign {
    my ($self, $camp) = @_;
    return unless $self->state($camp)->{is_good};

    my $serv_type = $camp->{AgencyUID} ? 'agency' : $camp->{ManagerUID} ? 'manager' : 'self';

    for my $serv ('all', $serv_type) {
        my $camp_pct = $camp->{hierarchical_multipliers}{mobile_multiplier}{multiplier_pct};

        $self->sum("all_camps_$serv" => 1);
        if (defined $camp_pct && $camp_pct < 100) {
            $self->sum("negative_camps_$serv" => 1);
        } elsif (defined $camp_pct && $camp_pct > 100) {
            $self->sum("positive_camps_$serv" => 1);
        }

        if (my $range = _range($camp_pct)) {
            $self->sum("range_$range->{from}_$range->{to}_camps_$serv" => 1);
        }
    }
}


sub _range {
    my ($pct) = @_;
    state $pct2range;
    if (!$pct2range) {
        for my $range (@ranges) {
            $pct2range->{$_} = $range for ($range->{from}+100) .. ($range->{to}+100);
        }
    }
    return defined $pct && $pct2range->{$pct} ? $pct2range->{$pct} : undef;
}

1;
