package Direct::YaAgency;

use Direct::Modern;
use Direct::TurboLandings;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Balance;
use Yandex::DateTime qw/now/;

use RBACDirect;
use RBAC2::Extended;

use CampaignTools;
use PrimitivesIds;

use Settings;
use feature 'state';

our $UID //= 1;
our $PRODUCTS //= $Settings::YaAgency_PRODUCTS;

our $ALLOWED_PRODUCT_TYPES //= [sort {$a <=> $b} keys %$PRODUCTS];
our $YA_AGENCY_NOTIFICATION_EMAIL_TO //= 'nadiano@yandex-team.ru, stitov@yandex-team.ru, client-service@yandex-team.ru';

state $SERVICE_ID = $Settings::SERVICEID{ya_agency};
state $SELECT = q/SELECT ClientID, yaOrderId as id, product_type, yaOrderStatus as status, created, LastChange FROM yandex_agency_orders/;

=head2 create_order

       Создание заявки на обслуживание клиента Я.Агенством

       create_order(product_type => ..., ClientID =>  ...)
              product_type - идентификатор тарифа из @$ALLOWED_PRODUCT_TYPES
              ClientID - идентификатор клиента

=cut

sub create_order {
    my %in = @_;
    my ($product_type, $client_id) = @in{qw/product_type ClientID/};

    die 'Invalid product type: '.$product_type unless exists $PRODUCTS->{$product_type};

    do_insert_into_table(PPC(ClientID => $client_id), 'yandex_agency_orders',
                          {ClientID => $client_id,
                           product_type => $product_type,
                           yaOrderStatus => 'New',
                     });
    return;
}

=head2 get_order

       Получение по ClientID или yaOrderID заявки на обслуживание клиента Я.Агенством

       $order = get_order(ClientID => ...)
       $order = get_order(yaOrderID => ...)

       Для найденной заявки возвращает hashref c полями:
              id => yaOrderID, ClientID => ClientID, product_type => id_тарифа, status => New|Sent|paid|Completed,
              created => dtime, LastChange => dtime

=cut

sub get_order {
    my %in = @_;
    my ($client_id, $order_id) = @in{qw/ClientID yaOrderID/};

    die 'One of ClientID or OrderID required' unless ($client_id || $order_id);
    return $client_id ? _get_order_by_client_id($client_id) : _get_order_by_id($order_id);
}

=head2 send_order

       Отправить заявку в баланс

              $res = send_order(ClientID => ...);
              $res = send_order(ClientID => ..., product_type => ...);

       Если передан product_type - используется переданное значение тарифа, в противном случае - сохраненное.
       Отправленной заявке выставляется yaOrderID, статус меняется на Sent, выставляется актуальное значение тарифа.

       Возвращает hashref на структуру с полями:
        ok  => 1 если заявку удалось отправить и 0 в противном случае,
        url => возвращенный балансом url счета

=cut

sub send_order {
    my %in = @_;

    my ($client_id, $product_type) = @in{qw/ClientID product_type/};
    die 'ClientID required' unless ($client_id);

    my $order_info = get_order(ClientID => $client_id);
    return {ok => 0} unless $order_info->{status} && $order_info->{status} eq 'New';

    my $product_id = $product_type // $order_info->{product_type};
    return {ok => 0} unless $PRODUCTS->{$product_id};

    $order_info->{product_type} = $product_id;

    #yaOrderID будем брать из пространства cid, чтобы исключить пересечение
    my $yaOrderId = Yandex::DBShards::get_new_id('cid', ClientID => $client_id);


    my $order = {
        ServiceID => $SERVICE_ID,
        ProductID => $product_id,
        ServiceOrderID => $yaOrderId,
        ClientID => $client_id,
        unmoderated => 0,
        Text => 'Настройка Яндекс.Директа',
    };
    my $ok = balance_create_update_orders($UID, [$order]);
    unless ($ok) {
        warn 'Error while order sending';
        return {ok => 0};
    }

    my $url = balance_create_request($UID, $client_id,
              [{ServiceID => $SERVICE_ID, ServiceOrderID => $yaOrderId, Qty => 1}],
              {DenyPromocode => 1, Overdraft => 0, UiType => 'std'}
    );

    do_in_transaction {
        do_insert_into_table(PPC(ClientID => $client_id), 'clients_options', {
                      is_ya_agency_client => 1,
                      ClientID => $client_id,
               },
               on_duplicate_key_update => 1,
               key => 'ClientID'
        );

        do_update_table(PPC(ClientID => $client_id), 'yandex_agency_orders',
            {
                yaOrderStatus=> 'Sent',
                yaOrderId => $yaOrderId,
                product_type => $order_info->{product_type},
                LastChange__dont_quote => 'NOW()',
            },
            where => {ClientID => $client_id}
        );
    };
    return {ok => $ok, url => $url};
}

=head2 finalise_order

       закрыть заявку и рассервисировать кампании клиента

              $ok = finalise_order(ClientID => ...);

       Возвращает 1 если заявку удалось закрыть и 0 в противном случае

=cut

sub finalise_order {
    my %in = @_;

    my ($client_id, $product_type, $force_zero) = @in{qw/ClientID product_type force_zero/};
    die 'ClientID required' unless ($client_id);

    my $order_info = get_order(ClientID => $client_id);
    return 0 unless $order_info->{status};
    
    my $ok;
    if ($order_info->{status} eq 'Paid') {
        #Если заказ оплачен - перед закрытием его в балансе делаем рассервисирование
        $ok = _unservice_client($client_id);
        return 0 unless $ok;
    }
    elsif($order_info->{status} eq 'Completed'){
        #Если заказ завершен - в баланс можно отправить только нулевую открутку
        unless ($force_zero){
            warn sprintf ('Couldn`t finalise completed order %s', $order_info->{id});
            return 0;
        }
    }
    else {
        #При других статусах заказа его нельзя финализировать
        warn sprintf ('order_id %s - finalisation prohibited for status "%s"', @$order_info{qw/id status/});
        return 0;
    }

    my $balance_result = _close_balance([$order_info->{id}], $force_zero);
    unless ($balance_result->{$order_info->{id}}){
        warn sprintf ('Couldn`t close balance for order_id %s', $order_info->{id});
        return 0;
    }
    
    do_update_table(PPC(ClientID => $client_id), 'clients_options', { is_ya_agency_client => 0, }, where => { ClientID => $client_id });

    do_update_table(PPC(ClientID => $client_id), 'yandex_agency_orders',
        {
            yaOrderStatus=> $order_info->{status} eq 'Completed' ? 'Resurrected' : 'Completed',
            LastChange__dont_quote => 'NOW()',
        },
        where => {ClientID => $client_id}
    );

    return 1;
}


sub _close_balance {
    my ( $order_ids, $force_zero_bucks ) = @_;
    $order_ids = [$order_ids] unless ref $order_ids;
    
    my @orders;
    foreach my $id (@$order_ids){
        push @orders, {
                ServiceID => $SERVICE_ID,
                ServiceOrderID => $id,
                dt => now()->strftime("%Y%m%d%H%M%S"),
                Bucks => $force_zero_bucks ? 0 : 1,
                Stop => 1,
            };
    }
   
    my @res = balance_update_campaigns(\@orders);
    return $res[0]->{$SERVICE_ID};
}

=head2 get_product_id_by_frontend_id

    По идентификатору выбранного тарифа, пришедшему с фронтенда возвращает product_id

=cut


sub get_product_id_by_frontend_id {
    my ($frontend_id) = @_;

    return $Settings::YaAgency_PRODUCT_BY_FRONTEND_ID->{$frontend_id}
}

sub _get_order_by_client_id {
    my ($client_id) = @_;
    return get_one_line_sql(PPC(ClientID => $client_id),[
        $SELECT, WHERE => {ClientID => $client_id}
    ]);
}

sub _get_order_by_id {
    my ($id) = @_;
    return get_one_line_sql(PPC(shard => 'all'),[
        $SELECT, WHERE => {yaOrderId=> $id}
    ]);
}

sub _unservice_client {
       my ($client_id) = @_;

       my $cids = get_one_column_sql(PPC(ClientID => $client_id),
              [q/SELECT cid FROM campaigns c JOIN users u ON (c.uid = u.uid)/,
                     WHERE => {'u.ClientID' => $client_id}
              ]
       );

       my $rbac = RBAC2::Extended->get_singleton($UID);

       my $has_serviced_campaigns;
       foreach my $cid (@$cids){
              next unless RBACDirect::rbac_is_scampaign($rbac, $cid);
              $has_serviced_campaigns //= 1;

              my $errcode = RBACDirect::rbac_move_scamp_to_nscamp($rbac, $cid);
              return if $errcode;

              CampaignTools::campaign_manager_changed($rbac, $UID, $cid, 0);
       }

       unless ($has_serviced_campaigns) {
              my $uids = get_uids(ClientID => $client_id);
              foreach my $uid (@$uids){
                     my $manager_uids = RBACDirect::rbac_get_managers_of_client($rbac, $uid);
                     RBACDirect::rbac_unbind_manager($rbac, $_, $uid) foreach @$manager_uids;
              }
       }
        
       Direct::TurboLandings::mass_refresh_metrika_grants_for_all_client_counters([$client_id]);
       do_update_table(PPC(ClientID => $client_id), 'campaigns', { statusBsSynced => 'No' }, where => { cid => $cids }) if @$cids;

       return 1;
}

1;
