package DirectCache;

# $Id$

=head1 NAME

    DirectCache

=head1 DESCRIPTION

    Управляет кешированием данных в директе.
    Позволяет кешировать результаты вызовов функций для каждого набора аргументов. 
    Время жизни кеша по умолчанию не ограничено, возможно его ограничение одним запросом. 
    В основном представляет из себя централизующую обёртку надо модулем Memoize.

=head2 EXAMPLE

    my $dcache = new DirectCache(groups => ['lemmer']);
    $dcache->start_caching_for_group('lemmer');
    $dcache->flush_cache_for_group('lemmer');
    $dcache->on_request_end;

=head2 CAVEATS

    Функции коректно кешируются только при обращении к ним по полному имени.
    Если использовать экспортированную копию, кеширование работать не будет.

    use DirectCache;
    my $direct_cache = new DirectCache(groups => ['camp_type']);
    # в этом случае кеш использоваться НЕ будет
    get_camp_mediatype(12345) for (1..20);
    # а в этом — будет
    Primitives::get_camp_mediatype(12345) for (1..20);

=cut

use strict;
use warnings;
use utf8; 

use Campaign::Types;
use Yandex::DBShards;
use Stat::Tools;
use Rbac;
use Direct::TargetingCategories;
use RBACDirect;

use Memoize;

=head2 CACHE GROUPS

    Группа для кеширования — это совокупность функций, результаты выполнения которых надо кешировать,
    и правил, по которым надо управлять кешем этих функций.

=cut

my %CACHE_GROUPS = (
    lemmer => {
        functions_to_cache => ['Yandex::MyGoodWords::MyGetGoodWords'],
        flush_every_request => 1,
    },
    forecast_db_queries => {
        functions_to_cache => ['PhraseText::get_ctr_data'],
    },
    stat_date_format => {
        functions_to_cache => ['Stat::Tools::get_date_subperiod', 'Stat::Tools::format_date'],
        flush_every_request => 1,
    }
);

=head2 new

    Конструктор. Принимает опциональный именованный параметр:
        groups — ссылка на список групп для кеширования. Указание здесь групп эквивалентно вызову start_caching_for_group.
    Возвращает ссылку на сконструированный объект.

    my $dcache = new DirectCache(groups => ['lemmer']);

=cut

sub new {
    my($class, %O) = @_;

    my $self = bless({}, $class);
    $self->{_cached_groups} = {};

    if ($O{groups}) {
        for my $group (@{$O{groups}}) {
            $self->start_caching_for_group($group);
        }
    }

    return $self;
}

=head2 start_caching_for_group

    Включает кеширование для группы с указанным именем.

    $dcache->start_caching_for_group('lemmer');

=cut

sub start_caching_for_group {
    my ($self, $group) = @_;

    die "no such group '$group'" unless exists $CACHE_GROUPS{$group};
    return if $self->{_cached_groups}{$group};

    my $functions_to_cache = $CACHE_GROUPS{$group}{functions_to_cache};
    if ($functions_to_cache) {
        for my $function (@$functions_to_cache) {
           Memoize::memoize($function);
        }
    }

    $self->{_cached_groups}{$group} = 1;
}

=head2 stop_caching_for_group

    Выключает кеширование для группы с указанным именем.

    $dcache->stop_caching_for_group('lemmer');

=cut

sub stop_caching_for_group {
    my ($self, $group) = @_;

    die "no such group '$group'" unless exists $CACHE_GROUPS{$group};
    return unless $self->{_cached_groups}{$group};

    my $functions_to_cache = $CACHE_GROUPS{$group}{functions_to_cache};
    if ($functions_to_cache) {
        for my $function (@$functions_to_cache) {
           Memoize::unmemoize($function);
        }
    }

    delete $self->{_cached_groups}{$group};
}

=head2 flush_cache_for_group

    Принудительно сбрасывает кеш для группы с указанным именем.

    $dcache->flush_cache_for_group('lemmer');

=cut

sub flush_cache_for_group {
    my ($self, $group) = @_;

    die "no such group '$group'" unless exists $CACHE_GROUPS{$group};
    return unless $self->{_cached_groups}{$group};

    my $functions_to_cache = $CACHE_GROUPS{$group}{functions_to_cache};
    if ($functions_to_cache) {
        for my $function (@$functions_to_cache) {
            eval {
                Memoize::flush_cache($function);
            };
            warn $@ if $@;
        }
    }
}

=head2 on_request_end

    Процедура, которую надо вызывать при окончании обработки каждого запроса,
    если используются группы с параметром flush_every_request.

    $dcache->on_request_end;

=cut

sub on_request_end {
    my ($self) = @_;

    for my $group (grep { $CACHE_GROUPS{$_}{flush_every_request} } keys %{$self->{_cached_groups}}) {
        $self->flush_cache_for_group($group);
    }
    Campaign::Types::clear_cache();
    Yandex::DBShards::clear_cache();
    Stat::Tools::clear_banner_info_cache();
    Rbac::clear_cache();
    Client::clear_cache();
    Client::ClientFeatures::clear_cache();
    Direct::TargetingCategories::clear_cache();
    RBACDirect::clear_cache();
}

sub DESTROY {
    my ($self) = @_;

    for my $group (keys %{$self->{_cached_groups}}) {
        $self->stop_caching_for_group($group);
    }
}

1;
