package DirectRedis;

use Direct::Modern;
use Carp;
use JSON;
use List::Util qw/shuffle/;
use Time::HiRes qw//;
use JSON qw//;

use Redis::Cluster;
use Redis;

use Yandex::Trace qw//;
use Yandex::DBTools qw/get_db_config/;

use Settings;

our @EXPORT_OK = qw/get_redis/;

our $CHECK_INTERVAL = 3;
our $REDIS_CACHE_NAME //= 'redis_cache';

my %CACHE;

=head2 my $redis_cluster = get_redis()

    Получить синглтон клиента к Директовскому Redis::Cluster
    Раз в CHECK_INTERVAL проверяется, изменился ли конфиг.
    Если изменился - пересоздётся Redis::Cluster

=cut
sub get_redis {
    my $name = shift || 'redis';
    my $cache = $CACHE{$name} //= {};
    if (!$cache->{redis} || Time::HiRes::time() - $cache->{check_time} >= $CHECK_INTERVAL) {
        my $config = get_db_config($name);
        my $config_json = JSON::to_json($config, {canonical => 1});

        if (!$cache->{redis} || $cache->{config_json} ne $config_json) {
            undef $cache->{redis};
            $cache->{redis} = _create_redis($config);
        }

        $cache->{check_time} = Time::HiRes::time();
        $cache->{config_json} = $config_json;
    }
    bless {name => $name};
}


=head2 my $redis = get_cache_redis()

    Получить инстанс редиса, предназначенный для хранения протухающих данных

=cut
sub get_cache_redis {
    return get_redis($REDIS_CACHE_NAME);
}

sub _create_redis {
    my ($config) = @_;

    # добавляем порт, если он не указан в хосте
    my @hosts = map {/:/ ? $_ : "$_:$config->{port}"}
        ref $config->{host} ? @{$config->{host}} : ($config->{host});

    # redis-cluster при попытке коннекта перебирает хосты по-очереди
    # на одном сервере у нас может быть размежено несколько шардов redis
    # если сервер недоступен - недоступны сразу все редисы
    # чтобы мы пытались коннектиться в разные сервера - далем хитрую рандомизацию
    my %hosts_cnt;
    @hosts = map {$_->[0]} 
        sort {$a->[1] <=> $b->[1] || $a->[2] <=> $b->[2]} 
        map {/(.*):/; [$_, ++$hosts_cnt{$1}, rand]} 
        shuffle
        @hosts;

    my $type = $config->{redis_type} // 'cluster';
    my $password = $config->{pass} // '';

    if ($type eq 'cluster') {
        return Redis::Cluster->new(
            server         => \@hosts,
            refresh        => 30,
            max_redirects  => 5,
            max_queue_size => 10,
            allow_slave    => 0,
            cnx_timeout    => $config->{connect_timeout},
            cnx_tries      => $config->{connect_tries} // 3, # https://st.yandex-team.ru/DIRECT-66814
            reconnect      => 1,
            every          => 100_000,
            conservative_reconnect => 1,
            read_timeout   => 5,
            write_timeout  => 5,
            (length $password ? (password => $password) : ()),
            );
    } elsif ($type eq 'single') {
        return Redis->new(
            server         => $hosts[0],
            cnx_timeout    => $config->{connect_timeout},
            reconnect      => 1,
            every          => 100_000,
            conservative_reconnect => 1,
            read_timeout   => 5,
            write_timeout  => 5,
            (length $password ? (password => $password) : ()),
            );
    } else {
        die "Unsupported redis_type: $type";
    }
}

# оборачиваем все вызовы для трейсинга
our $AUTOLOAD;
sub AUTOLOAD {
    my ($self, @args) = @_;

    (my $method = $AUTOLOAD) =~ s/.*:://;

    my $profile = Yandex::Trace::new_profile("redis:$method");

    my $reqid = Yandex::Trace::current_span_id()//'(reqid:0)';
    local $SIG{__DIE__} = sub {
        print STDERR Carp::longmess("REDIS ERROR: method=$method, reqid=$reqid, args=".JSON::to_json(\@args, { allow_unknown => 1 }).", error=".$_[0]);
    };

    no strict 'refs';
    return $CACHE{$self->{name}}->{redis}->$method(@args);
}

1;
