package DoCmd::Base;

# $Id$

=head1 NAME
    
    DoCmd::Base - базовый модуль для написания MVC контроллеров

=head1 SYNOPSIS

    package DoCmd::Something;
    use base qw/DoCmd::Base/;

    sub cmd_editSomething
        :Cmd(editSomething)
        :Rbac(Code => rbac_internal_use_only, Role => super)
    {
        ...
    }

    1;

=head1 DESCRIPTION


=cut

use strict;
use warnings;

use Carp;
use Attribute::Handlers;

use RBAC2::DirectChecks qw//;
use DoCmd::FormCheck;

our %cmds;
our %blocking_cmds;
our %CheckCSRF;
our %CheckBySchema;
our %CmdDescriptions;
our %cmd_no_auth;
our %parallel_limit;
our %captcha;
our %no_captcha;
our %predefine_vars;
our %allow_blocked_client;

# максимальная длина названия контроллера
our $MAX_CMD_LENGTH ||= 32;

# переменные нужны для передачи данных между атрибутами одной функции
# чтобы не передавать, можно было бы всё реализовать в одном атрибуте,
# но это, кажется, ещё хуже
my $tmp_coderef;
my @tmp_aliases;

=head2 :Cmd

    Атрибут Cmd позволяет регистрировать контроллер и определять
    одну или несколько команд, для обработки которых нужно вызывать
    эту функцию.
    Если вам понадобилось указывать несколько алиасов - подумайте,
    возможно просто стоит сделать несколько контроллеров.

=cut

# BEGIN нужен из-за того, что под mod_perl нет фазы INIT
# По этой же причине мы не можем получить имя текущей функции,
# которое было бы удобно использовать для определения одной команды
sub Cmd :ATTR(BEGIN) {
    my ($package, $symbol, $referent, $attr, $data, $phase, $filename, $linenum) = @_;
    # Получаем алиасы из параметров
    my @aliases = !defined $data
        ? ()
        : ref($data) eq 'ARRAY'
        ? @$data
        : $data;
    # Регистрируем функцию в хеше
    for my $name (@aliases) {
        $name =~ s/^cmd_//;
        if (length($name) > $MAX_CMD_LENGTH) {
            die "Too long cmd: '$name'";
        } elsif (exists $cmds{$name}) {
            die "Handler for $name redefined";
        }
        $cmds{$name} = $referent;
    }
    # копируем данные для передачи между атрибутами
    $tmp_coderef = $referent;
    @tmp_aliases = @aliases;
}

=head2 :Rbac

    Атрибут определяет проверку прав доступа на исполнение контроллера
    На вход должен получать "хэш". Выполнение разрешается, если выполнены все условия.
    Возможные ключи хэша:

=over

=item Code
    
    Имя функции из модуля RBAC2::DirectChecks.
    Пример:
    :Rbac(Code => rbac_cmd_internal_networks_only)

    можно указать несколько, при этом каждая должна вернуть 0
    :Rbac(Code => [rbac_cmd_internal_networks_only, rbac_cmd_check_client_login])

=item Role

    Используется для контроллеров, которые могут исполнять только некоторые роли пользователей
    Пример:
    :Rbac(Role => super)
    :Rbac(Role => [super, manager, agency])

=item ExceptRole

    Используется для контроллеров, которые могут исполнять все роли кроме указанных.
    Нельзя использовать одновременно с Role, и без указания дополнительных проверок (Code, Perm или PerlClear), на что есть отдельный unit-test.

    Пример:
    :Rbac(Code => rbac_cmd_user_allow_edit_camps, ExceptRole => media)

==item Perm

    Проверяется наличие права у пользователя и принадлежность объекта пользователю.
    Для медиапланеров и сапортов принадлежность объекта не проверяется.
    Пример:
    :Rbac(Perm => SystemReport)

=item PermClear

    Проверяется только наличие права у пользователя.
    Пример:
    :Rbac(PermClear => SystemReport)

=item CampKind

    Аналогично CampType, но для надтипов кампаний (см. Campaign::Types)
    Должно выполняться хотя бы одно условие, чтобы запрос отработал

    Пример:
    :CampKind(web_edit => 1)
    :CampKind(web_edit => 1, web_super_view => [super])

=item AllowDevelopers

    если есть такой ключ, и есть ограничения по ролям (Role, ExceptRole) то суперридерам с настройкой "разработчик"
    все равно позволяем независимо от того запрещено ли суперридерам или нет.
    у простых суперридеров всё по старому.

=item Cmd

    Если указать директиву Cmd, то указанные проверки будут применяться только для некоторых алиасов команд.
    В этом случае нужно указывать несколько атрибутов :Rbac
    Пример:
    sub cmd_doit
        :Cmd(editRep, makeRep, viewRep)
        :Rbac(Cmd => [editRep, makeRep], Role => [super])
        :Rbac(Cmd => [viewRep], Role => [super, manager])
    {
    }

=back

=cut

sub Rbac :ATTR(BEGIN) {
    my ($package, $symbol, $referent, $attr, $data, $phase, $filename, $linenum) = @_;
    my %data = @$data;
    die "Attribute :Rbac must be used only after :Cmd" if !exists $data{Cmd} && (!defined $tmp_coderef || $tmp_coderef ne $referent);
    my $rbac = {};
    my @local_aliases = @tmp_aliases;
    while(my ($key, $val) = each %data) {
        if ($key eq 'Cmd') {
            @local_aliases = ($val);
        } elsif ($key eq 'Code') {
            my @rbac_check_functions = ref($val) eq 'ARRAY' ? (@$val) : ($val);
            for my $rbac_check_function (@rbac_check_functions) {
                my $code_ref = RBAC2::DirectChecks->can($rbac_check_function) || die "No such function: RBAC2::DirectChecks::$rbac_check_function";
                push @{$rbac->{Code}}, $code_ref;
            }

        } elsif ($key eq 'Perm') {
            $rbac->{Perm} = $val;
        } elsif ($key eq 'PermClear') {
            $rbac->{PermClear} = $val;
        } elsif ($key eq 'Role') {
            $rbac->{Role} = $val;
        } elsif ($key eq 'ExceptRole') {
            $rbac->{ExceptRole} = $val;
        } elsif ($key eq 'CampKind') {
           $rbac->{CampKind} = $val;
        } elsif ($key eq 'AllowForLimitedClients') {
            $rbac->{AllowForLimitedClients} = 1;
        } elsif ($key eq 'AllowDevelopers') {
            $rbac->{AllowDevelopers} = 1;
        } elsif ($key eq 'AllowReadonlyReps') {
            $rbac->{AllowReadonlyReps} = 1;
        } else {
            die "unknown key '$key'";
        }
    }

    for my $cmd (@local_aliases) {
        die "Duplicate :Rbac declaration for $cmd" if exists $RBAC2::DirectChecks::cmds{$cmd};
        $RBAC2::DirectChecks::cmds{$cmd} = $rbac;
    }
}

=head2 :Lock

    Обозначение блокирующих функций

    :Lock(Cid => "campaign_old")  -  ID кампании  в переменной FORM{campaign_old}
    :Lock(1)  - проверка блокировки кампании FORM{cid}
    :Lock(0)  - неблокирующая операция

=cut

sub Lock :ATTR(BEGIN) {
    my ($package, $symbol, $referent, $attr, $data, $phase, $filename, $linenum) = @_;


    for my $cmd (@tmp_aliases) {
        $blocking_cmds{$cmd} = $data;
    }
}

=head2 :NoAuth

    Методы, доступные без авторизации

    :NoAuth

=cut

sub NoAuth :ATTR(BEGIN) {
    my ($package, $symbol, $referent, $attr, $data, $phase, $filename, $linenum) = @_;

    for my $cmd (@tmp_aliases) {
        $cmd_no_auth{$cmd} = 1;
    }

    # хорошо бы залочить %cmd_no_auth, но непонятно, в какой момент
}

=head2 :RequireParam

    Проверка на обязательные параметры
    :RequireParam(param_name => 'param type')

    типы (хранятся в DoCmd::FormCheck):
      Cid: номер кампании
      Cids: номерa кампаний (несколько одинаковых параметров), превращаются в ccылку на массив
      Require: параметр должен присутствовать (имя любое)

    Пример:
    :RequireParam(cid => 'Cids')

=cut

sub RequireParam :ATTR(BEGIN) {
    my ($package, $symbol, $referent, $attr, $data, $phase, $filename, $linenum) = @_;

    my %data = @$data;
    die "Attribute :RequireParam must be used only after :Cmd" if !exists $data{Cmd} && (!defined $tmp_coderef || $tmp_coderef ne $referent);

    my %check_params = @$data;

    for my $cmd (@tmp_aliases) {
        while (my ($param_key, $param_type) = each %check_params) {
            die "Duplicate :RequireParam declaration for $cmd / $param_key" if exists $DoCmd::FormCheck::cmds_param_checks{$cmd}->{$param_key};
            $DoCmd::FormCheck::cmds_param_checks{$cmd}->{$param_key} = $param_type;
        }
    }
}

#-------------------------------------------------------------------------------

=head2 :CheckCSRF

    Check cmd for CSRF

=cut

sub CheckCSRF :ATTR(BEGIN) {
    my ($package, $symbol, $referent, $attr, $data, $phase, $filename, $linenum) = @_;

    for my $cmd (@tmp_aliases) {
        $CheckCSRF{$cmd} = 1;
    }
}

#-------------------------------------------------------------------------------

=head2 :Description

    Descriptions for cmds

=cut

sub Description :ATTR(BEGIN) {
    my ($package, $symbol, $referent, $attr, $data, $phase, $filename, $linenum) = @_;

    for my $cmd (@tmp_aliases) {
        $CmdDescriptions{$cmd} = ref $data eq 'ARRAY' ? $data->[0] : $data;
    }
}

=head2 :ParallelLimit

    Ограничение количества одновременно выполняемых запросов (Num штук)
    для клуча, собранного из параметров, указанных в Key
    
    В Key возможны параметры: UID, uid, FORM.param, FORM(хэш от всех cgi параметров)
    cmd (название контроллера) добавляется автоматически

    Если указан параметр LogOnly и он истиннен - записываем информацию о неудачной 
    блокировке в лог, но продолжаем выполнять контроллер

    Примеры:
    :ParallelLimit(Num => 3, Key => [UID, FORM.cid])

=cut

sub ParallelLimit :ATTR(BEGIN) {
    my ($package, $symbol, $referent, $attr, $data, $phase, $filename, $linenum) = @_;
    die "Attribute :ParallelLimit must be used only after :Cmd" if !defined $tmp_coderef || $tmp_coderef ne $referent;
    my %data = @$data;
    my $limit = {};
    my $cmd_text = join(', ', @tmp_aliases);
    while(my ($key, $val) = each %data) {
        if ($key eq 'Num') {
            croak "Incorrect :ParallelLimit(Num) for $cmd_text" if $val !~ /^\d+$/;
            $limit->{Num} = $val;
        } elsif ($key eq 'Key') {
            my @vals = ref($val) eq 'ARRAY' ? (@$val) : ($val);
            croak "Incorrect :ParallelLimit(Key) for $cmd_text" if !@vals;
            $limit->{Key} = \@vals;
        } elsif ($key eq 'LogOnly') {
            $limit->{$key} = $val;
        } else {
            croak "Unknown key :ParallelLimit($key) for $cmd_text" if $val !~ /^\d+$/;
        }
    }
    croak "Incomplete :ParallelLimit for $cmd_text" if !exists $limit->{Num} || !exists $limit->{Key};
    for my $cmd (@tmp_aliases) {
        push @{$parallel_limit{$cmd}}, $limit;
    }
}

=head2 :Captcha

    Подключаем для указанного контроллера Plack::CaptchaChecker
    В качестве ключа пока поддерживаются UID, IP
    При указании DynamicLimits параметры будут вычислены в соответствующей функции

    :Captcha(Key => [IP], Freq => 10, Interval => 86400, MaxReq => 100)
    :Captcha(Key => [UID], DynamicLimits => Captcha::calc_forecast)

=cut

sub Captcha :ATTR(BEGIN) {
    my ($package, $symbol, $referent, $attr, $data, $phase, $filename, $linenum) = @_;
    die "Attribute :Captcha must be used only after :Cmd" if !defined $tmp_coderef || $tmp_coderef ne $referent;
    my %data = @$data;
    my $params = {};
    my $cmd_text = join(', ', @tmp_aliases);
    my @captcha_params = qw/Freq Interval MaxReq/;
    while (my ($key, $val) = each %data) {
        if ($key eq 'Key') {
            my @vals = ref($val) eq 'ARRAY' ? (@$val) : ($val);
            croak "Incorrect :Captcha(Key) for $cmd_text" if !@vals;
            $params->{Key} = \@vals;
        } elsif (grep {$key eq $_} @captcha_params, 'DynamicLimits') {
            $params->{$key} = $val;
        }
    }
    die "no Key in :Captcha for $cmd_text" unless $params->{Key};
    if ($params->{DynamicLimits}) {
        if (my @present = grep {defined $params->{$_}} @captcha_params) {
            die sprintf ":Captcha for $cmd_text uses DynamicLimits, remove %s", join(', ', @present);
        }
    } elsif (my @missing = grep {!defined $params->{$_}} @captcha_params) {
        die sprintf "Specify %s in :Captcha for $cmd_text or use DynamicLimits", join(', ', @missing);
    }
    for my $cmd (@tmp_aliases) {
        die "Incorrect usage: both of :NoCaptcha and :Captcha specified for $cmd" if $no_captcha{$cmd};
        push @{$captcha{$cmd}}, $params;
    }
}

=head2 :NoCaptcha

    Внести указанный контроллер в исключения для показа капчи.
    Предназначен для контроллеров, которые отвечают json-данными (ajax),
    т.к. в большинстве случаев фронтенд не готов к получение капчи вместо json-данных и все ломается.

=cut

sub NoCaptcha :ATTR(BEGIN) {
    my ($package, $symbol, $referent, $attr, $data, $phase, $filename, $linenum) = @_;
    die "Attribute :NoCaptcha must be used only after :Cmd" if !defined $tmp_coderef || $tmp_coderef ne $referent;;

    for my $cmd (@tmp_aliases) {
        die "Incorrect usage: both of :NoCaptcha and :Captcha specified for $cmd" if $captcha{$cmd};
        $no_captcha{$cmd} = 1;
    }
}

=head2 :PredefineVars

    Указать переменные, которые должны быть предопределены в контроллере (в $vars)

    :PredefineVars(qw/campaign_agency_contacts/)

=cut

sub PredefineVars :ATTR(BEGIN) {
    my ($package, $symbol, $referent, $attr, $data, $phase, $filename, $linenum) = @_;

    # Получаем имена переменных из параметров
    my @params = !defined $data
        ? ()
        : ref($data) eq 'ARRAY'
        ? @$data
        : $data;

    for my $cmd (@tmp_aliases) {
        $predefine_vars{$cmd} = \@params;
    }
}

=head2 :AllowBlockedClient

    Контроллеры которые разрешены для заблокированных (statusBlocked=Yes) пользователей

=cut

sub AllowBlockedClient :ATTR(BEGIN) {
    $allow_blocked_client{$_} = 1 foreach @tmp_aliases;
}

=head2 :CheckBySchema

    Проверить входные и/или выходные данные контроллера на соответствие заданной json-схеме.

=over

=item Cmd
    
    Если контроллер имеет несколько алиасов, с помощью этого параметра можно ограничить выполнение проверок.
    При указанном Cmd проверки на соответствие схеме будут выполнятся только для перечисленных алиасов
    Пример:
        :CheckBySchema(Cmd => [showCampMultiEdit, showCampMultiEditLight], Input => warn)
        
=item Input

    Задает уровень проверки входных параметров контроллера:
        none : проверка на соответствие схеме не производится
        warn : проверка производится, если данные не соответствуют схеме - выводится сообщение на STDERR, обработка не прерывается
        check : проверка производится, если данные не соответствуют схеме - обработка данных прерывается и выдается сообщение об ошибке
        crop : из датасета убираются элементы не покрытые схемой, затем проверка производится также как для check
    Пример:
    :CheckBySchema(Input => check)

=item Output

    Задает уровень проверки выходных данных контроллера:
        none : проверка на соответствие схеме не производится
        warn : проверка производится, если данные не соответствуют схеме - выводится сообщение на STDERR, обработка не прерывается
        check : проверка производится, если данные не соответствуют схеме - обработка данных прерывается и выдается сообщение об ошибке
        crop : проверка производится также как для check, в случае успешной проверки из датасета убираются элементы не покрытые схемой
    Пример:
    :CheckBySchema(Output => warn)
    
=cut


sub CheckBySchema :ATTR(BEGIN) {
    my ($package, $symbol, $referent, $attr, $data, $phase, $filename, $linenum) = @_;

    my %data = @$data;
    die "Attribute :CheckBySchema must be used only after :Cmd" if !exists $data{Cmd} && (!defined $tmp_coderef || $tmp_coderef ne $referent);
    
    my @local_aliases = @tmp_aliases;
    my ($input, $output);
    my $allowed_checks = {map {$_ => 1} qw/none warn check crop/};
    
    @local_aliases = (@{$data{Cmd}}) if exists $data{Cmd};
    $input  = $data{Input} // 'none';
    $output = $data{Output} // 'none';

    foreach ($input, $output) {
        die 'Unknown check type: '.$_ unless $allowed_checks->{$_}
    }
    
    delete @data{qw/Cmd Input Output/};
    die 'Unknown keys: '.join(', ', keys %data) if %data;

    for my $cmd (@local_aliases) {
        die 'Duplicate :CheckBySchema declaration for '.$cmd if exists $CheckBySchema{$cmd};
        $CheckBySchema{$cmd} = {input => $input, output => $output};
    }
    
    return
}

1;
