package DoCmd::FormCheck;

# $Id$

=head1 NAME

    FormCheck - проверка форм

=head1 DESCRIPTION

=cut

use strict;
use warnings;

use Yandex::I18n;
use Tools;
use Campaign;
use Campaign::Types;
use BannersCommon qw/check_json_ajax_update_struct check_json_ajax_apply_unglued_minus_words_struct/;
use Settings;
use Yandex::Validate qw/is_valid_float is_valid_int is_valid_id/;
use List::MoreUtils qw/any all none/;

use base  qw/Exporter/;

our @EXPORT= qw/
    check_required_params
/;

use utf8;

our %cmds_param_checks; # filled in DoCmd::Base::RequireParam

our %CMD_FORM_ERRORS = (
    # errors for rbac_check_require_param()
    1 => iget_noop('Ошибка: требуется параметр'), # это общая ошибка, лучше указывать конкретный параметр (как 2)
    2 => iget_noop('Ошибка: не задан номер кампании'),
    3 => iget_noop('Ошибка: неверный номер кампании: %s'),
    4 => iget_noop('Ошибка: не задан номер объявления'),
    5 => iget_noop('Ошибка: неверный номер объявления: %s'),
    6 => iget_noop('Ошибка: не заданы логины клиентов'),
    7 => iget_noop('Ошибка: отсутствуют обязательные параметры'),
    8 => iget_noop('Ошибка: параметр имеет неверный формат'),
    9 => iget_noop('Ошибка: не задан номер группы'),
    10 => iget_noop('Ошибка: неверный номер группы: %s'),
    11 => iget_noop('Ошибка: неверный геотаргетинг: %s'),
    12 => iget_noop('Ошибка: не задан текст уточнения'),
    13 => iget_noop('Ошибка: не задан id уточнения'),
);


=head2 check_required_params_for_cmd

    Проверка параметров в форме для заданной команды

=cut

sub check_required_params_for_cmd
{
    my ($cmd, $form) = @_;

    return 0 if ref($cmds_param_checks{$cmd}) ne 'HASH';

    my @errors = eval { check_required_params($form, %{ $cmds_param_checks{$cmd} }) };

    if ( $@ ) {
        die "unknown param type in $cmd :RequireParam($@)"
    }

    return @errors; 
}


#...............................................................................
sub is_positiv_int
{
    my $int = shift;
    return defined $int && $int =~ m/^\d+$/ && $int > 0 ? 1 : 0;
}

sub vld_struct_strategy {
    
    my ($param, $form) = @_;
    Campaign::strategy_struct_sample($form->{$param});
}

sub vld_struct_banner_statuses {
    my ($param, $form) = @_;
    return {
        map {$_ => {statusShow => 1}} ref $form->{$param} eq 'HASH' ? keys %{$form->{$param}} : () 
    };
}

# ajaxUpdatePhrases: хеш с данными об изменениях во фразах
sub vld_simple_ajaxUpdatePhrases {
    my ($param_name, $form) = @_;
    return 0 unless defined $form->{$param_name};

    return !BannersCommon::check_json_ajax_update_struct($form->{$param_name});
}

sub vld_simple_ajaxApplyRejectCorrection {
    my ($param_name, $form) = @_;
    return !BannersCommon::check_json_ajax_apply_unglued_minus_words_struct($form->{$param_name});
}

sub vld_simple_ajaxPrepareStatPhrases {
    my ($param_name, $form) = @_;

    my $struct = $form->{$param_name};
    return 1 unless ref($struct) eq 'ARRAY';
    return 1 unless all { ref($_) eq 'HASH' } @{$struct};
    
    return 0;
}

# --------------------------------------------------------------------

=head2 vld_simple_ajaxSaveRetargetingCond

проверка условия ретаргетинга при сохранении

пример:
  $VAR1 = {
          'condition' => [
                              {
                                'goals' => [
                                             {
                                               'goal_id' => '15045410',
                                               'time' => '1',
                                               'goal_type' => 'goal'
                                             },
                                             {
                                               'goal_id' => '15045411',
                                               'time' => '5'
                                               'goal_type' => 'goal'
                                             },
                                             {
                                               'goal_id' => '15045410',
                                               'time' => '1'
                                               'goal_type' => 'segment'
                                             }
                                           ],
                                'type' => 'not'
                              },
                              {
                                'goals' => [
                                             {
                                               'goal_id' => '15045412',
                                               'time' => '4'
                                               'goal_type' => 'goal'
                                             },
                                             {
                                               'goal_id' => '15045412',
                                               'time' => '1'
                                               'goal_type' => 'segment'
                                             }
                                           ],
                                'type' => 'not'
                              }
                            ],
          'condition_desc' => 'dwedwedew',
          'condition_name' => 'dewdwedewd'
        };

=cut

sub vld_simple_ajaxSaveRetargetingCond {
    my ($param_name, $form) = @_;

    return 8 if ref($form->{$param_name}) ne 'HASH';
    return 8 unless all {defined $form->{$param_name}->{$_}} qw/condition condition_desc condition_name/;

    return 8 unless ref($form->{$param_name}->{condition}) eq 'ARRAY'
                    && @{$form->{$param_name}->{condition}}
                    && all {ref($_->{goals}) eq 'ARRAY'
                            && defined $_->{type}
                           }
                       @{$form->{$param_name}->{retargeting}};

    for my $goals (map {@{$_->{goals}}} @{$form->{$param_name}->{retargeting}}) {
        return 8 unless ref($goals) eq 'HASH';
        return 8 unless 2 == scalar keys %$goals;
        return 8 unless is_positiv_int($goals->{goal_id}) && is_positiv_int($goals->{time});
        return 8 unless defined $goals->{goal_type} && $goals->{goal_type} =~ /^(goal|segment|audience)$/;
    }

    return 0;
}

# --------------------------------------------------------------------

=head2 vld_simple_ajaxEditAdGroupRetargeting

сохранение условий ретаргетинга в баннер

пример:
  {
    pid1 => {
        edited => {
            ret_id1 => {
                price_context => 0.01,
                autobudgetPriority => 1,
                is_suspended => 0,
            },
            ret_id2 => {
                price_context => 0.01,
                autobudgetPriority => 1,
                is_suspended => 0,
            }, ... 
        },
        deleted => [ret_id3, ret_id4, ...]
    },
    pid2 => {
        ...
    }
  }

=cut

sub vld_simple_ajaxEditAdGroupRetargeting {
    my ($param_name, $form) = @_;

    my $new_retargetings = $form->{$param_name};

    return 8 if ref($new_retargetings) ne 'HASH';
    return 8 unless all {m/^\d+$/} keys %{ $new_retargetings };

    for my $pid (keys %{ $new_retargetings }) {
        return 8 if ref($new_retargetings->{$pid}) ne 'HASH';
        return 8 unless all {m/^(edited|deleted|main_bid)$/} keys %{ $new_retargetings->{$pid} };

        if (exists $new_retargetings->{$pid}->{edited}) {
            my $banner_edit = $new_retargetings->{$pid}->{edited};

            return 8 unless all {m/^\d+$/} keys %$banner_edit;
            for my $ret_id (keys %$banner_edit) {
                my $one_ret = $banner_edit->{$ret_id};
                return 8 unless all {m/^(price_context|autobudgetPriority|is_suspended)$/} keys %$one_ret;

                return 8 if exists $one_ret->{price_context}
                            && $one_ret->{price_context} !~ /^\d+(\.\d+)?$/;
                return 8 if exists $one_ret->{autobudgetPriority}
                            && $one_ret->{autobudgetPriority} !~ /^(1|3|5)$/;
                return 8 if exists $one_ret->{is_suspended}
                            && $one_ret->{is_suspended} !~ /^(0|1)$/;
            }
        }

        if (exists $new_retargetings->{$pid}->{deleted}) {
            return 8 unless ref($new_retargetings->{$pid}->{deleted}) eq 'ARRAY';
            return 8 unless scalar(@{ $new_retargetings->{$pid}->{deleted} }) >= 1;
            return 8 unless all {m/^\d+$/} @{ $new_retargetings->{$pid}->{deleted} };
        }

        if (exists $new_retargetings->{$pid}->{main_bid}) {
            return 8 if ref $new_retargetings->{$pid}->{main_bid};
            return 8 unless is_valid_id($new_retargetings->{$pid}->{main_bid});
        }
    }

    return 0;
}

# --------------------------------------------------------------------

=head2 vld_simple_ajaxEditAdGroupRelevanceMatches

сохранение условий беcфразного таргетинга в баннер

пример:
  {
    pid1 => {
        edited => {
            bid_id1 => {
                price => 0.01,
                autobudgetPriority => 1,
                is_suspended => 0,
            },
            bid_id2 => {
                price => 0.01,
                autobudgetPriority => 1,
                is_suspended => 0,
            }, ...
        },
        deleted => [bid_id3, bid_id4, ...]
    },
    pid2 => {
        ...
    }
  }

=cut

sub vld_simple_ajaxEditAdGroupRelevanceMatches {
    my ($param_name, $form) = @_;

    my $new_relevance_matches = $form->{$param_name};

    return 0 unless defined $new_relevance_matches;

    return 8 if ref($new_relevance_matches) ne 'HASH';
    return 8 unless all {m/^\d+$/} keys %{ $new_relevance_matches };

    for my $pid (keys %{ $new_relevance_matches }) {
        return 8 if ref($new_relevance_matches->{$pid}) ne 'HASH';
        return 8 unless all {m/^(added|edited|deleted|main_bid)$/} keys %{ $new_relevance_matches->{$pid} };

        if (exists $new_relevance_matches->{$pid}->{edited}) {
            my $banner_edit = $new_relevance_matches->{$pid}->{edited};

            return 8 unless all {m/^\d+$/} keys %$banner_edit;
            for my $bid_id (keys %$banner_edit) {
                my $one_bid = $banner_edit->{$bid_id};
                return 8 unless all {m/^(price|price_context|autobudgetPriority|is_suspended)$/} keys %$one_bid;

                return 8 if exists $one_bid->{price}
                    && $one_bid->{price} !~ /^\d+(\.\d+)?$/;
                return 8 if exists $one_bid->{price_conext}
                    && $one_bid->{price_context} !~ /^\d+(\.\d+)?$/;
                return 8 if exists $one_bid->{autobudgetPriority}
                    && $one_bid->{autobudgetPriority} !~ /^(1|3|5)$/;
                return 8 if exists $one_bid->{is_suspended}
                    && $one_bid->{is_suspended} !~ /^(0|1)$/;
            }
        }

        if (exists $new_relevance_matches->{$pid}->{deleted}) {
            return 8 unless ref($new_relevance_matches->{$pid}->{deleted}) eq 'ARRAY';
            return 8 unless scalar(@{ $new_relevance_matches->{$pid}->{deleted} }) >= 1;
            return 8 unless all {m/^\d+$/} @{ $new_relevance_matches->{$pid}->{deleted} };
        }

        if (exists $new_relevance_matches->{$pid}->{main_bid}) {
            return 8 if ref $new_relevance_matches->{$pid}->{main_bid};
            return 8 unless is_valid_id($new_relevance_matches->{$pid}->{main_bid});
        }
    }

    return 0;
}

# --------------------------------------------------------------------

=head2 vld_simple_ajaxReplaceGoalsInRetargetings

Замена целей во всех условиях ретаргетинга

пример:
    [
        {old_goal_id => 123, new_goal_id => 124},
        {old_goal_id => 125, new_goal_id => 126},
        {old_goal_id => 127, new_goal_id => 128},
        {old_goal_id => 129, new_goal_id => 130},
        ...
    ]

=cut

sub vld_simple_ajaxReplaceGoalsInRetargetings {
    my ($param_name, $form) = @_;

    my $param = $form->{$param_name};

    return 8 if ref($param) ne 'ARRAY';
    for my $row (@$param) {
        return 8 unless ref($row) eq 'HASH';
        return 8 unless exists $row->{old_goal_id} && exists $row->{new_goal_id};
        return 8 unless $row->{old_goal_id} =~ /^\d+$/ && $row->{new_goal_id} =~ /^\d+$/;
    }

    return 0;
}

# --------------------------------------------------------------------

=head2 vld_simple_ajaxEditShowConditions

AJAX редактирование условий показа на группе.
Формат структуры см. в DoCmdShowConditions::cmd_ajaxEditShowConditions

Упрощенная проверка, основные правила находятся в DoCmd::Checks

=cut

sub vld_simple_ajaxEditShowConditions {
    my (undef, $form) = @_;

    my $param_name = $form->{cmd} eq 'ajaxEditDynamicConditions' ? 'json_adgroup_dynamic_conditions' : 'json_adgroup_performance_filters'; 
    my $show_conds_data = $form->{$param_name};

    # Должен быть хеш с идентификаторами групп в качестве ключей
    return 8 unless ref($show_conds_data) eq 'HASH';
    return 8 unless all { /^(\d+)$/ && $1 > 0 } keys %$show_conds_data;

    for my $adgroup_id (keys %$show_conds_data) {
        # Должен быть хеш с ключами "edited" или "deleted"
        return 8 unless ref($show_conds_data->{$adgroup_id}) eq 'HASH';
        return 8 unless all { /^(?:edited|deleted)$/ } keys %{ $show_conds_data->{$adgroup_id} };

        # Проверим содержимое "edited" структуры
        if (exists $show_conds_data->{$adgroup_id}->{edited}) {
            my $show_conds_edited = $show_conds_data->{$adgroup_id}->{edited};

            # Должен быть хеш с идентификаторами условий показа в качестве ключей
            return 8 unless ref($show_conds_edited) eq 'HASH';
            return 8 unless all { /^(\d+)$/ && $1 > 0 } keys %$show_conds_edited;
            return 8 unless all { ref($_) eq 'HASH' } values %$show_conds_edited;
        }

        # Проверим содержимое "deleted" структуры
        if (exists $show_conds_data->{$adgroup_id}->{deleted}) {
            my $show_conds_deleted = $show_conds_data->{$adgroup_id}->{deleted};

            # Должен быть массив с идентификаторами условий показа
            return 8 unless ref($show_conds_deleted) eq 'ARRAY';
            return 8 unless all { /^(\d+)$/ && $1 > 0 } @$show_conds_deleted;
        }
    }

    return 0;
}

# --------------------------------------------------------------------

=head2 vld_simple_day_budget

Параметры дневного бюджета
примеры:
    {"set": true, "sum": 32.32, "show_mode": "default"}
    {"set": true, "sum": 32.32, "show_mode": "stretched"}
    {"set": false}

=cut

sub vld_simple_day_budget {
    my ($param_name, $form) = @_;

    my $param = $form->{$param_name};

    return 8 if ref($param) ne 'HASH';
    return 8 unless defined $param->{set} && $param->{set} =~ /^(true|false)$/;

    # это JSON::XS::Boolean в булевском контексте ведет себя как 0/1, в строковом как true/false
    if ($param->{set}) {
        return 8 unless defined $param->{show_mode} && $param->{show_mode} =~ /^(default|stretched)$/;
        return 8 unless defined $param->{sum} && is_valid_float($param->{sum});
    }

    return 0;
}

=head2 vld_simple_adgroups

Проверяет, что кампания, переданная в форме имеет groups, а те в свою очередь banners и phrases/retargetings/dynamic_conditions

Пример:
    json_groups => [{
        banners => [...],
        phrases => [...],
        retargetings => [...],
        dynamic_conditions => [...],
    }, {
        ...
    }]

=cut

sub vld_simple_adgroups {
    my ($param_name, $form) = @_;

    my $adgroups = $form->{$param_name};

    return 7 unless defined $adgroups;
    return 8 if ref($adgroups) ne 'ARRAY';

    my $is_light = $form->{is_light};
    foreach my $adgroup (@$adgroups) {
        return 8 if ref($adgroup) ne 'HASH';

        # Сами объявления должны быть всегда
        return 8 if !defined $adgroup->{banners} || ref($adgroup->{banners}) ne 'ARRAY';

        if ($is_light) {
            # Если только редактирование текстов объявлений - то phrases/retargetings/dynamic_conditions не должны передаваться
            return 8 if any { defined $adgroup->{$_} } qw/phrases retargetings dynamic_conditions/;
        } else {
            # Иначе что-нибудь одно все же должно быть
            return 8 if none { defined $adgroup->{$_} && ref($adgroup->{$_}) eq 'ARRAY' } qw/phrases retargetings dynamic_conditions target_interests relevance_match/;
        }
    }

    return 0;
}

=head2 vld_simple_saveAdGroups

Проверяет переданную в JSON структуру групп.
Из условий показа -- проверка пока работает для dynamic_conditions.

Пример структуры:
    json_groups => [{
        adgroup_id => 1,
        banners => [...],
        dynamic_conditions => [...],
    }, {
        ...
    }]

=cut

sub vld_simple_saveAdGroups {
    my ($param_name, $form) = @_;

    my $adgroups = $form->{$param_name};

    return 7 unless defined $adgroups;
    return 8 if ref($adgroups) ne 'ARRAY';

    foreach my $adgroup (@$adgroups) {
        return 8 if ref($adgroup) ne 'HASH';

        # Баннеры
        return 8 if exists $adgroup->{banners} && ref($adgroup->{banners}) ne 'ARRAY';
        for my $banner (@{$adgroup->{banners} // []}) {
            return 8 if ref($banner) ne 'HASH';

            # Визитка & Сайтлинки
            return 8 if $banner->{has_vcard} && ref($banner->{vcard}) ne 'HASH';
            return 8 if exists $banner->{sitelinks} && ref($banner->{sitelinks}) ne 'ARRAY';
        }

        # Условия нацеливания
        return 8 if exists $adgroup->{dynamic_conditions} && ref($adgroup->{dynamic_conditions}) ne 'ARRAY';
        for my $dyn_cond (@{$adgroup->{dynamic_conditions} // []}) {
            return 8 if ref($dyn_cond) ne 'HASH';
        }

        # Перфоманс фильтры
        return 8 if exists $adgroup->{performance_filters} && ref($adgroup->{performance_filters}) ne 'ARRAY';
        for my $perf_filter (@{$adgroup->{performance_filters} // []}) {
            return 8 if ref($perf_filter) ne 'HASH';
        }
    }

    return 0;
}


# --------------------------------------------------------------------

=head2 vld_simple_getGeoRestrictions

cmd=getGeoRestrictions

пример:
    [
        {requestId => 123, text => 'banner text1', geo => 1},
        {requestId => 124, text => 'banner text2', geo => 187},
        ...
    ]

https://st.yandex-team.ru/DIRECT-37521

=cut

sub vld_simple_getGeoRestrictions {
    my ($param_name, $form) = @_;

    my $param = $form->{$param_name};

    return 8 if ref($param) ne 'ARRAY';
    return 8 if ! @$param;

    for my $row (@$param) {
        return 8 unless defined $row->{requestId}
                     && defined $row->{text}
                     && length($row->{text}) > 0
                     && defined $row->{geo}
                     && length($row->{geo}) > 0;
        return 8 unless is_positiv_int($row->{requestId});
        return 8 if grep {! is_valid_int($_)} split /,/, $row->{geo};
    }

    return 0;
}

# --------------------------------------------------------------------

=head2 vld_simple_list_banners_param

cmd=ajaxValidateStreet, cmd=ajaxCheckUrl

пример:
    [
        {requestId => 123, country => , city => , street => , house => , build => , apart => , metro => },
        {requestId => 124, ...},
        ...
    ]

=cut

sub vld_simple_list_banners_param {
    my ($param_name, $form) = @_;

    my $param = $form->{$param_name};

    return 8 if ref($param) ne 'ARRAY';
    return 8 if ! @$param;

    for my $row (@$param) {
        return 8 unless defined $row->{requestId} && is_positiv_int($row->{requestId});
    }

    return 0;
}

# --------------------------------------------------------------------

=head2 vld_simple_banners_texts

cmd=cmd_ajaxCheckModDoc

пример:
    [
        {
            domain => "..."
            , title => "..."
            , title_extension => "..."
            , body => "..."
            , phrases => ["...", "...", ...]
        },
        {...},
        ...
    ]

=cut

sub vld_simple_banners_texts {
    my ($param_name, $form) = @_;

    my $param = $form->{$param_name};

    return 8 if ref($param) ne 'ARRAY';
    return 8 if ! @$param;

    for my $row (@$param) {
        return 8 unless defined $row->{domain};
        return 8 unless defined $row->{title};
        return 8 unless defined $row->{body};
        return 8 unless ref($row->{phrases}) eq 'ARRAY' && @{$row->{phrases}};
    }

    return 0;
}

# --------------------------------------------------------------------
# Cid: номер кампании
sub vld_simple_Cid {
    my ($param_name, $form) = @_;
    
    return 2 if ! defined $form->{$param_name};
    return(3, $form->{$param_name}) if ! is_positiv_int($form->{$param_name});
    return 0;
}

# требуется параметр
sub vld_simple_Require {
    my ($param_name, $form) = @_;

    return 1 if ! defined $form->{$param_name};
    return 0;
}

# Cids: несколько номеров кампании
sub vld_simple_Cids {
    my ($param_name, $form) = @_;

    return 2 if ! defined $form->{$param_name};

    my @cids = split /\s*,\s*/, $form->{$param_name};
    for my $cid (@cids) {
        return(3, $cid) if ! is_positiv_int($cid);
    }

    $form->{$param_name} = \@cids; # save cids arrayref to $FORM{cid}

    return 0;
}

# Cids: несколько номеров кампании, или отсутствие cid
sub vld_simple_CidsMaybe {
    my ($param_name, $form) = @_;

    return 0 if ! defined $form->{$param_name};

    my @cids = split /\s*,\s*/, $form->{$param_name};
    for my $cid (@cids) {
        return(3, $cid) if ! is_positiv_int($cid);
    }

    $form->{$param_name} = \@cids; # save cids arrayref to $FORM{cid}

    return 0;
}

# Bid: номер объявления
sub vld_simple_Bid {
    my ($param_name, $form) = @_;

    return 4 if ! defined $form->{$param_name};
    return(5, $form->{$param_name}) if ! is_positiv_int($form->{$param_name});

    return 0;
}

# Bids: несколько номеров объявлений
sub vld_simple_Bids {
    my ($param_name, $form) = @_;

    return 4 if ! defined $form->{$param_name};

    my @bids = split /\s*,\s*/, $form->{$param_name};
    for my $bid (@bids) {
        return(5, $bid) if ! is_positiv_int($bid);
    }

    $form->{$param_name} = \@bids; # save bids arrayref to $FORM{bid}

    return 0;
}

# Pid: номеров группы
sub vld_simple_Pid {
    my ($param_name, $form) = @_;

    return 9 if ! defined $form->{$param_name};
    return(10, $form->{$param_name}) if ! is_positiv_int($form->{$param_name});

    return 0;
}

# Pids: несколько номеров групп
sub vld_simple_Pids {
    my ($param_name, $form) = @_;
    
    return 9 if ! defined $form->{$param_name};

    my @pids = split /\s*,\s*/, $form->{$param_name};
    for my $pid (@pids) {
        return(10, $pid) unless is_positiv_int($pid);
    }

    $form->{$param_name} = \@pids;

    return 0;
}

# ClientLogin: архивация/разархивация - требуется параметр client_login
sub vld_simple_ClientLogin {
    my ($param_name, $form) = @_;

    return 6 if ! defined $form->{$param_name};
    return 0;
}

# geo: список регионов через запятую
sub vld_simple_Geo {
    my ($param_name, $form) = @_;

    return 1 if ! defined $form->{$param_name};

    for my $geo (split /,/, $form->{$param_name}) {
        return(11, $geo) unless is_valid_int($geo);
    }

    return 0;
}

# --------------------------------------------------------------------

=head2 vld_simple_banner_additions

Список дополнений на баннер
Пока один обязательный ключ: callouts

{
    callouts: [
            {
                callout_text       : "callout text",
            }, {...}, ...
        ]
}

=cut

sub vld_simple_banner_additions {
    my ($param_name, $form) = @_;

    my $param = $form->{$param_name};
    return 7 if ! defined $param;

    return 8 if ref($param) ne 'HASH';
    return 8 if ref($param->{callouts}) ne 'ARRAY';

    for my $row (@{$param->{callouts}}) {
        return 12 if ref($row) ne "HASH";
        return 12 if $form->{cmd} eq "saveBannersAdditions" && ! defined $row->{callout_text};
        return 13 if $form->{cmd} =~ /^(deleteBannersAdditions)$/ && ! defined $row->{additions_item_id};
    }

    return 0;
}

=head2 check_required_params

    Проверка необходимых параметров в форме, для вызова из произвольного места

    Параметры: 
        позиционный $form -- ссылка на хеш с формой

        далее -- хеш ( имя_поля_в_форме => тип проверки по CHECK_SUBS)

    Результат: 
        если все в порядке -- возвращает 0
        если не в порядке -- выполнение прерывается, пользователю отдается страница с ошибкой

    Примеры: 
        @form_check_result = check_required_params(\%FORM, cid => 'Cid', bid => 'Bid')

=cut 

{

my %KIND;
{
    no strict 'refs';
    my $_PACKAGE = __PACKAGE__ . '::';
    foreach (keys %{$_PACKAGE}) {
        next unless /^vld_([^_]+)_(.+)$/;
        $KIND{$1} = {} unless exists $KIND{$1};
        $KIND{$1}->{$2} = $_PACKAGE->{$_};
    }
    use strict;
}

sub check_required_params
{
    my ($form, %check) = @_;
    
    # может оказаться, что алфавитный порядок для проверок не подходит, 
    # тогда надо делать кастомный
    for my $name (sort keys %check) {
        my $args = $check{$name};
        my ($type, $rule, %params) = ref $args ? @$args : ('simple', $args);

        if (exists $params{camps} && $form->{cid}) {
            my $type = get_camp_type(cid => $form->{cid});
            next unless any {$_ eq $type} @{$params{camps}}
        }
        
        if (exists $KIND{$type} && exists $KIND{$type}->{$rule}) {
            my @results;
            if ($type eq 'simple') {
                @results = $KIND{$type}->{$rule}->($name, $form);
            } elsif ($type eq 'struct') {
                my $struct = $KIND{$type}->{$rule}->($name, $form);
                @results = validate_structure($form->{$name}, $struct) ? 7 : 0;
            }
            return @results if $results[0];
        } else {
            die "$name => [$type $rule]";
        }
    }
    
    return 0;
}
}


1;
