package DoCmd::Public;

use warnings;
use strict;

=pod

    $Id$

=head1 NAME

    DoCmd::Public

=head1 DESCRIPTION

    Модуль с контроллерами доступными без авторизации

=cut

use base qw/DoCmd::Base/;

use feature qw/state/;

use UNIVERSAL;
use Encode;
use List::Util qw/max min sum/;
use List::MoreUtils qw/any all none/;
use POSIX qw(strftime);
use URI::Escape qw/uri_escape_utf8/;
use Template;
use JSON;
use Plack::Util;
use CSRFCheck;

use Direct::Validation::MinusWords qw/validate_keyword_minus_words/;
use Direct::Validation::Keywords qw//;

use Yandex::Validate;
use Yandex::DBTools;
use Yandex::HashUtils;
use Yandex::I18n;
use Yandex::IDN;
use Yandex::Blackbox;
use Yandex::Log::Messages;
use Yandex::SendMail qw/sendmail/;
use Yandex::OAuth;
use Yandex::HTTP;
use Yandex::TimeCommon qw/str_round_day/;

use Settings;
use geo_regions;
use PhraseText;
use Suggestions;
use EnvTools;
use TextTools;
use HttpTools;
use IpTools;
use Tools;
use ADVQ6;
use TTTools;
use GeoTools;
use Direct::ResponseHelper;
use Direct::Validation::MinusWords;
use YandexOffice;
use XLSCampImport;
use Direct::Template;
use Surveys;
use Survey::TNS;
use CSRFCheck;
use Intapi::CLegacy::News;

#Для showDnaPb
use Apache::AuthAdwords;
use RBAC2::Extended;
use RBAC2::DirectChecks;
use PrimitivesIds;
use Primitives;
use Property;
use Client;
use DirectCache;

#для ajaxInfoblockRequest
use Campaign::Types;
use RBACElementary;
use RBACDirect;
use Try::Tiny;
use TvmChecker;

use utf8;

{
my $template;
sub configure_template($$) {
    my ($r, $form) = @_;

    Direct::Template::save_dynamic_predefine({reqid => $r->env->{reqid}});
    Direct::Template::put_csp_nonce($r->env->{reqid});

    if (ref($template) !~ /Template/) {
        my $is_beta = is_beta();
        my %CONFIG = (
            PLUGIN_BASE => 'Yandex::Template::Plugin',
            ENCODING => 'utf8',
            EVAL_PERL => 1,
            INCLUDE_PATH => $Settings::TT_INCLUDE_PATH,
            COMPILE_EXT  => '.ttc',
            COMPILE_DIR  => "/tmp/tt-cache-$<",
            CACHE_TTL => $is_beta ? 1 : 3600,
            INTERPOLATE  => 0, # expand "$var" in plain text
            POST_CHOMP   => 1, # cleanup whitespace
            PRE_DEFINE   => {
                support        => 'support@direct.yandex.ru',
                help           => 'http://direct.yandex.ru/help/',
                is_beta        => $is_beta,
                is_production  => is_production(),
                DOCUMENT_ROOT  => "$Settings::ROOT/data/",

                # functions
                uri_escape                 => \&uri_escape_utf8,
                string2html                => \&string2html,
                round2s                    => \&round2s,
                floor                      => \&floor,
                ceil                       => \&ceil,
                get_plot_link              => \&TTTools::get_plot_link,
                dumper                     => \&TTTools::dumper,
                get_sort_table_header      => \&TTTools::get_sort_table_header,
                get_sort_table_all_headers => \&TTTools::get_sort_table_all_headers,
                sort_table_data            => \&TTTools::sort_table_data,
                format_price               => \&TTTools::format_price,
                format_units               => \&TTTools::format_units,
                format_int                 => \&TTTools::format_int,
                format_date                => \&TTTools::format_date,
                format_href                => \&TTTools::format_href,
                format_file_size           => \&TTTools::format_file_size,
                get_word_for_digit         => \&get_word_for_digit,
                get_geo_names              => \&GeoTools::get_geo_names,
                get_current_url_params     => \&TTTools::get_current_url_params,
                iget                       => \&iget,
                iget_noop                  => \&iget_noop,
                trim_float                 => \&TTTools::trim_float,
                console_log                => \&TTTools::console_log,
                json_dump                  => \&TTTools::json_dump,
                truncate                   => \&TextTools::truncate_text,
                is_targeting_in_region     => \&GeoTools::is_targeting_in_region,

                # required by lego
                js_quote                   => \&js_quote,
                js_encode                  => \&js_quote,
                url_quote                  => \&url_quote,
                generate_id                => \&TTTools::generate_id,
                is_string                  => \&TTTools::is_string,

                hash_copy                  => \&Yandex::HashUtils::hash_copy,
            },
    
            FILTERS => {
                js                         => \&js_quote,
                url                        => \&url_quote,
                idn_to_ascii               => \&Yandex::IDN::idn_to_ascii,
                typograph                  => \&TTTools::typograph,
            }
        );
        $template = Template->new(\%CONFIG);
    }

    my $scheme = http_server_scheme($r);
    my $server_host = http_server_host($r);

    my $SCRIPT = $scheme . '://' . $server_host . '/public';

    my $user_region = http_geo_region($r);
    my $yandex_domain = yandex_domain($r);
    my $is_direct = is_direct($server_host);
    my $lang = lang_auto_detect($r);
    my $SCRIPT_OBJECT = {
        protocol => $scheme,
        host => $server_host,
        path => '/public', 
    };

    my %TT_CONFIG_DYN_PRE_DEFINE = (
        SCRIPT => $SCRIPT,
        script => $SCRIPT,
        SCRIPT_OBJECT => $SCRIPT_OBJECT,
        server_name => $server_host,
        yandex_domain => $yandex_domain,
        passport_domain => get_passport_domain($r),
        scheme => $scheme,
        FORM => $form,
        save_time_after_tt => sub { return; }, # устарело
        get_url => sub {return TTTools::get_url($SCRIPT, undef, @_)},
        geo_region => $user_region,
        officecity => get_officecity_by_geo_footer($user_region, $yandex_domain),
        is_direct => $is_direct,
        lang => $lang,
        tune_secret_key => HttpTools::get_tune_secret_key($r, undef),
        infoblock_teasers_url => $Settings::INFOBLOCK_TEASERS_URL,

        # required by BEM
        cmd      => $form->{cmd},
        COOKIES  => get_all_cookies($r),
        uatraits => HttpTools::get_uatraits(http_get_header($r, 'User-Agent')),
        get_relative_url => sub {return TTTools::get_url('/registered/main.pl', undef, @_)},
    );

    # set dynamic variables in Template
    $template->context()->stash()->update(\%TT_CONFIG_DYN_PRE_DEFINE);
    Direct::Template::add_dynamic_predefine($r->env->{reqid}, \%TT_CONFIG_DYN_PRE_DEFINE);
    $template->context()->stash()->update({
        csp_nonce => Direct::Template::get_csp_nonce($r->env->{reqid}),
    });

    Yandex::I18n::init_i18n($lang);

    return $template;
}
}

sub do_direct_public_cmd
{
    my ($r, $multiform) = @_;

    my %FORM = %{multiform2directform($multiform)};
    my $cur_step = $FORM{cmd} || '-';

    # Если существует стоп-файл - ничего не делаем, отдыхаем
    if (-f $Settings::WEB_STOP_FLAG) {
        die "Stop file $Settings::WEB_STOP_FLAG founded";
    }

    if ( !exists $DoCmd::Base::cmds{$cur_step} || 
        !$DoCmd::Base::cmd_no_auth{$cur_step}
    ){
        return respond_text($r, "Operation not permitted or not available yet.\n");
    }

    local $Yandex::TVM2::APP_ID;      $Yandex::TVM2::APP_ID = $Settings::TVM2_APP_ID{web};
    local $Yandex::TVM2::SECRET_PATH; $Yandex::TVM2::SECRET_PATH = $Settings::TVM2_SECRET_PATH{web};
    local $Yandex::Blackbox::BLACKBOX_USE_TVM_CHECKER = \&TvmChecker::use_tvm;

    my $template = configure_template($r, \%FORM);
    my $SCRIPT = $template->context->stash->get('SCRIPT');

    Tools::_save_vars(0);

    $r->env->{trace}->method("$cur_step");
    $r->env->{trace}->tags('PublicCmd');
    set_response_csp_header($r, $r->env->{reqid}, $cur_step, Direct::Template::get_csp_nonce($r->env->{reqid}));
    my $resp = $DoCmd::Base::cmds{$cur_step}->({
            R            => $r,
            FORM         => \%FORM, 
            SCRIPT       => $SCRIPT,
            TEMPLATE     => $template,
            vars         => {},
        });

    # по умолчанию - устанавливаем заголовки, запрещающие кэширование
    # но если кто-то установил $r->env->{no_cache} = 0 - заголовки не ставим (повторяем логику апача)
    if (!defined $r->env->{no_cache} || $r->env->{no_cache}) {
        Plack::Util::header_set($resp->[1], 'Pragma' => "no-cache");
        Plack::Util::header_set($resp->[1], 'Cache-control' => "no-cache, no-store, max-age=0, must-revalidate");
    }
    Plack::Util::header_set($resp->[1], 'Expires' => strftime("%a, %d %b %Y %H:%M:%S GMT", gmtime));

    return $resp;
}

sub display_captcha {
    my ($env, $vars) = @_;

    $env->{trace}->method("showCaptcha");
    $env->{trace}->tags('PublicCmd');

    my $r = Plack::UTF8Request->new($env);

    if (($r->headers->header('Accept') || '') =~ /text\/html/
        && $r->param('cmd') !~ /ajax/
    ) {
        if (UNIVERSAL::isa($vars->{FORM}, 'Hash::MultiValue')) {
            $vars->{FORM} = multiform2directform($vars->{FORM});
        }
        my $template = configure_template($r, {});
        return respond_template($r, $template, 'show_captcha.html', $vars);
    } else {
        # это либо ajax, либо робот
        my $response = to_json(hash_cut $vars, qw/captcha_id captcha_url/);
        if ($INC{'Apache2/RequestUtil.pm'}) {
            # Apache2.2 не знает статус 429
            # это станет ненужным после перехода на Apache2.4 или на другой Plack сервер
            Apache2::RequestUtil->request->status_line("$Direct::ResponseHelper::AJAX_CAPTCHA_STATUS $Direct::ResponseHelper::AJAX_CAPTCHA_STATUS_MESSAGE");
        }
        return [
            $Direct::ResponseHelper::AJAX_CAPTCHA_STATUS,
            # вообще-то следовало бы $r->headers->header('Origin'), но у старых Firefox странности с его выставлением, а yandex_domain дает подходящую логику
            ['Content-Type' => 'application/json', 'Access-Control-Allow-Origin' => 'https://'.yandex_domain($r)],
            [Encode::is_utf8($response) ? Encode::encode_utf8($response) : $response],
        ];
    }
}


=head2 cmd_ajaxGetSuggestion

    Подсказки ключевых слов

    Параметры 
        srcPhrases -- исходные фразы, через запятую
        brief      -- не возвращать подробную информацию по фразам (принимает значение "yes")
        currency   -- валюта, в которой отдать цены (нужна только если получаются полные данные по фразам с ценами)

    Для полу-прогнозной промо-страницы есть доп. параметры, для одновременного запроса статистики показов (advq) _по исходным фразам_
        get_stat -- флаг "выдать кроме подсказок еще и статистику"
        geo -- регионы, для которых выдать статистику показов
        predefined_phrase -- флаг "фраза из рекламного баннера, надо кешировать"

    Цены по фразам возвращаются только для суперов/менеджеров/медиапланеров и только если в запросе есть параметр currency и нет brief.

    Пример: 
    curl 'http://beta.direct.yandex.ru:8086/public?cmd=ajaxGetSuggestion&srcPhrases=nokia,пицца'

=cut

sub cmd_ajaxGetSuggestion :Cmd(ajaxGetSuggestion)
    :Description('подсказки ключевых слов')
    :Rbac(Code => rbac_cmd_allow_all)
    :NoAuth
    # атрибут Captcha работает только при вызове через /registered,
    # ограничения вызова через /public описаны в PSGIApp/Public.pm
    :Captcha(Key => [UID], DynamicLimits => 'Captcha::wordstat')
    :Captcha(Key => [IP], Freq => 100, Interval => 3600, MaxReq => 4000)
{
    my ($r, $login_rights) = @{$_[0]}{
      qw/R   LOGIN_RIGHTS/};

    # вообще-то следовало бы $r->headers->header('Origin'), но у старых Firefox странности с его выставлением, а yandex_domain дает подходящую логику
    set_response_header($r, 'Access-Control-Allow-Origin', 'https://'.yandex_domain($r));

    my %FORM = %{$_[0]{FORM}};
    my $BEST_PHRASES_CNT = 50;
    # разбираем параметрыe

    my $src_phrases_text = defined $FORM{srcPhrases} ? $FORM{srcPhrases} : '';
    $src_phrases_text =~ s/\[|\]//g;
    process_phrase_brackets($src_phrases_text); # раскрываем скобки типа "(автомобилей | автомашин) (ВАЗ | ГАЗ)"
    my $src_phrases = [grep {$_} split(/[ ]*[,\n][ ]*/, $src_phrases_text)];

    my $special_user = any {$login_rights->{"${_}_control"}} qw/super superreader manager media support/;
    my $options = { 
        unfit_phrases => [split(/[,\n]+/,$FORM{unfitPhrases}||'')],
        # обычному пользователю не отдаем полную информацию
        full => !($FORM{brief} && $FORM{brief} eq 'yes') && $special_user && $FORM{currency},
        iteration => is_valid_int($FORM{iteration}, 0) ? $FORM{iteration} :0,
        currency => $FORM{currency} || 'YND_FIXED',
    };
    $options->{count} = $FORM{n} if $FORM{n};

    # обычному пользователю не выдаем подсказки по большим наборам фраз
    # чем больше у пользователя фраз, тем меньше подсказок выдаем
    unless ($special_user) {
        my $n = max(0, $BEST_PHRASES_CNT - scalar @$src_phrases); 
        $options->{count} = $n unless defined $options->{count} && $options->{count} < $n;
        $options->{count} = 0 if length($src_phrases_text) > 3072;
    }

    # обычному пользователю не выдаем подсказки с слишком большими показами
    $options->{incfactor} = 5 unless ($special_user);

    # для страницы "Подсказки по словам" (для медиапланеров, менеджеров и т.п.) отдаем все без ограничений
    if (($FORM{nolimits} || '') eq 'yes' && $special_user ) {
        $options->{users_links_lim} = 0;
        $options->{no_inner_links_filter} = 1;
    }

    my $suggest = Suggestions::get_bm_suggestion($src_phrases, $options);

    # если требуется -- добавляем статистику из advq
    if ($FORM{get_stat}){
        my $geo = $FORM{geo} || 0;
        if ($FORM{predefined_phrase} && @$src_phrases == 1 && $geo !~ /,/){
            # одна фраза из get-параметров, протой регион -- работаем через кеш, чтобы не задолбить ADVQ запросами из тизера и других рекламных баннеров
            my $phrase = $src_phrases->[0];
            $suggest->{stat}->{$phrase} = advq_one_phrase_shows_cached($geo, $phrase);
        } else {
            # DIRECT-30788: get_stat используется только на морде в блоке «Ваш товар ищут»
            # а там показывается статистика только по одной фразе.
            # Чтобы не поощрять всяких злонамеренных пользователей, даже не считаем статистику по "лишним" фразам.
            $src_phrases =[ (sort @$src_phrases)[0] ];
            
            my $raw_stat= advq_get_stat( $src_phrases, $geo );
            for my $p (@{$raw_stat || []}){
                next if ref $p ne 'HASH' || ! $p->{phrase};
                $suggest->{stat}->{$p->{phrase}} = $p->{count} || 0;
            }
            for my $ph (@$src_phrases){
                # если по фразе нет данных -- пишем, что у нее -1 показов
                $suggest->{stat}->{$ph} = -1 if !exists $suggest->{stat}->{$ph} || !defined $suggest->{stat}->{$ph};
            }
        }
    }

    return respond_json($r, $suggest);
}


=head2 advq_one_phrase_shows_cached

    Прогноз показов для одной фразы с одним регионом. 
    Кеширующая функция.

    Проверок, что параметры достаточно простые для кеширования -- НЕ делает, 
    о них должна позаботиться вызывающая сторона. 

    Использовать осторожно, только если ОЧЕНЬ надо, 
    и следить за тем, чтобы не слишком много фраз попадало в кеш. 

    TODO перенести в более подходящий модуль. 
      Вопрос только, в какой: ADVQ не должен бы знать про сущестование БД,
      Forecast, Forecast::Autobudget??
    TODO Возможно, потом наладить очистку кеша от старых значений

=cut
sub advq_one_phrase_shows_cached
{
    my ($geo, $phrase) = @_;

    $geo ||= 0;

    my $count = get_one_field_sql(PPCDICT, 
        ["select shows from advq_cache", 
        where => { 
            phrase => $phrase, 
            geo => $geo,
        } 
        ]
    );

    if( !defined $count ){
        my $stat = advq_get_stat( [$phrase], $geo ) || [];
        $count = $stat->[0]->{count} if @$stat && ref $stat->[0] eq 'HASH';
        $count ||= 0;
        do_insert_into_table(PPCDICT, 
            "advq_cache", 
            {
                phrase => $phrase, 
                geo => $geo,
                shows => $count,
            },
            ignore => 1,
        );
    }

    return $count;
}


sub cmd_ajaxPhraseStat :Cmd(ajaxPhraseStat)
    :Description('статистика показов ключевых слов')
    :Rbac(Code => rbac_cmd_allow_all)
    :Captcha(Key => [UID], DynamicLimits => 'Captcha::calc_forecast')
    :Captcha(Key => [IP, 'cb:Captcha::has_paid_key_part'], DynamicLimits => 'Captcha::calc_forecast_ip')
    :NoAuth
{
    my ($r) = @{$_[0]}{
      qw/R/};
    my %FORM = %{$_[0]{FORM}};

    # вообще-то следовало бы $r->headers->header('Origin'), но у старых Firefox странности с его выставлением, а yandex_domain дает подходящую логику
    set_response_header($r, 'Access-Control-Allow-Origin', 'https://'.yandex_domain($r));

    my $w1 = "";
    if( exists $FORM{w1} ) {
        $w1 = $FORM{w1};
        $w1 =~ s/\s+/ /g;
        $w1 =~ s/^\s//ig;
    }

    my $minus_phrases = $FORM{json_mw1};
    my $geo =  $FORM{geo};
    my $is_mobile = $FORM{is_mobile} // 0;
    
    my $correct_words = $w1;
    #При проверке полученных минус-фраз не проверяем общую длинну, т.к. фразы просуммированы по группе и кампании
    if ($minus_phrases && @$minus_phrases && Direct::Validation::MinusWords::validate_group_minus_words($minus_phrases, undef, max_overall_length => -1)->is_valid()){
        $correct_words = ADVQ6::format_advq_query($w1, minus_words => $minus_phrases, remove_intersection => 1);
    }

    my $lang = yandex_domain($r) =~ /\.com\.tr$/ ? 'tr' : 'ru';
    my $stat = ADVQ6::advq_get_stat( [$correct_words], $geo || 0, devices => $is_mobile ? [qw/phone tablet/] : ['all'],
                                    lang => $lang, video_advq => $FORM{video_advq},
                                    collections_advq => $FORM{collections_advq},
    );
    my $count = $stat && ref($stat) eq 'ARRAY' && ref(@{$stat}[0]) eq 'HASH' ? @{$stat}[0]->{count}:0;

    return respond_text($r, $count);
}


=head2 cmd_getReqid

    Возвращает reqid текущего запроса. 
    Для разметки логов при тестировании и разных замерах: 
    запрашиваем getReqid, запоминаем reqid, потом набор тестовых запросов и снова getReqid. 
    В логах (nginx, profile, БД, ...) тестовые запросы легко выбрать по условию "после первого запомненного reqid, но до второго".

    Можно обращаться без авторизации: curl 'http://beta.direct.yandex.ru:8086/public?cmd=getReqid'

    Можно стандартным образом, с авторизацией и типовым url: http://beta.direct.yandex.ru:8086/registered/main.pl?cmd=getReqid

=cut

sub cmd_getReqid :Cmd(getReqid)
    :Description('reqid текущего запроса')
    :Rbac(Code => rbac_cmd_internal_networks_only)
    :NoAuth
{
    my ($r, $vars) = @{$_[0]}{
      qw/R   vars/};

    # непоняно, пригодится ли в продакшене. Пока закрываем, будет надо -- можно разрешить
    error("Operation not permitted or not available yet...") if is_production();
    
    my $reqid;
    if ( $r && $r->isa('Plack::Request') ) {
        $reqid = $r->env->{reqid};
    } else {
        $reqid = $_[0]->{c}->{reqid};
    }

    return respond_text($r, $reqid);
}

=head2 cmd_exportSampleCamp

    Пример xls файла для управления кампаний через Excel.
    Ссылка публичная, достпна в http://help.yandex.ru/direct/?id=1111746

=cut

sub cmd_exportSampleCamp :Cmd(exportSampleCamp)
    :Description('выгрузка примера xls файла для управления кампаниями')
    :Rbac(Code => rbac_cmd_allow_all)
    :NoAuth
{
    my $r = $_[0]->{R};
    my $c = $_[0]->{c};
    my %FORM = %{$_[0]{FORM}};

    my $format = $FORM{format} && $FORM{format} eq 'xlsx' ? 'xlsx' : 'xls';  
    my $host = http_server_host($r);
    my $xls = camp_snapshot2excel(get_sample_camp_snapshot($host),
        host => $host,
        format => $format,
        show_callouts => 1,
    ); 
    return respond_data($r, $xls, ":${format}" => "direct_example.${format}");
}

=head2 cmd_sendTokenToTechSales

    Так как Горыныч внутреннее приложение
    ссылку на него нельзя установить в качестве callback url
    для OAuth сервера, поэтому и существует эта ручка.
    Коды авторизаци от пользователей Горыныча приходят на неё,
    она преобразует код в авторизационный токен,
    получает имя пользователя и отправляет эти данные Горынычу

=cut

sub cmd_sendTokenToTechSales :Cmd(sendTokenToTechSales)
    :Description('отправка токена Горынычу')
    :Rbac(Code => rbac_cmd_allow_all)
    :NoAuth
{
    my ($r, $login_rights) = @{$_[0]}{
      qw/R   LOGIN_RIGHTS/};

    my $log = Yandex::Log::Messages->new();

    my $template = configure_template($r, {});
    my $p = $r->parameters;
    my $code = $p->get('code');
    my $state = $p->get('state');
    error(iget('Не передан код авторизации')) unless $code;
        
    my $params = {
        code => $code,
        client_id => $Settings::GORYNYCH_DATA{CLIENT_ID},
        client_secret => $Settings::GORYNYCH_DATA{CLIENT_SECRET},
    };

    my $error;
    eval {
        my $token_data = oa_get_token_by_code($params);
        
        my $result;
        my $token = $token_data->{access_token};
        if (!$token) {
            $log->out(to_json($token_data));
            die "there is no access_token in oa_get_token_by_code result\n";
        }

        my $userinfo = bb_oauth_token($token, $r->address, $Settings::API_SERVER_PATH, [$BB_LOGIN]);
        if (($userinfo->{status} // '') ne 'VALID'){
            $error = iget('Не удалось найти клиента с таким токеном');
            die "userinfo status isn't 'VALID'";
        }

        my $login = $userinfo->{'dbfield'}{$BB_LOGIN};
        $result = {
            login => $login,
            token => $token,
            state => $state || ''
        };
        my $res = submit_form("POST", $Settings::GORYNYCH_DATA{AUTH_URL}, $result);

        if ($res->is_error) {
            my $email_to = $Settings::GORYNYCH_DATA{EMAIL};
            my $email_from = $Yandex::SendMail::MAIL_FOR_ALERTS;
            my $subj = 'Token';
            delete $result->{token}; # не светим токены в рассылках и логах
            my $mail_content = to_json($result);
            sendmail($email_to, $email_from, $subj, \$mail_content);
            $log->out("Error: $mail_content, ".$res->status_line.", ".$res->content =~ s/\Q$token\E/XXX/r);
            die "Gorynych error: ".$res->status_line;
        }
    };

    if ($@) {
        $log->out("Error: $@");
        $error //= iget('Ошибка при получении доступа, повторите попытку');
        error($error);
    } else {
        my $msg = iget('По вопросам подключения модулей обращайтесь
                        к вашему персональному менеджеру.
                        Редактировать доступ приложения к вашим рекламным
                        кампаниям вы можете на странице
                        <a href="https://oauth.yandex.ru/client/my">
                        oauth.yandex.ru/client/my</a>');
        my $title = iget('Система управления модулями Custom Solutions
                          успешно получила доступ к программному интерфейсу (API) Яндекс.Директа');
        return respond_template($r, $template, 'blank.html', {msg => $msg, hash => {title => $title}});
    }
}


=head2 cmd_getOfficeContacts

Возвращает строку с контактами офиса из футера (для Документации).

Параметры:

    * lang
    * domain
    * geo_id

=cut

sub cmd_getOfficeContacts :Cmd(getOfficeContacts)
    :Description('строка с контактами клиентского офиса')
    :Rbac(Code => rbac_cmd_internal_networks_only)
    :NoAuth
{
    my ($r, $vars) = @{$_[0]}{
      qw/R   vars/};
    my %FORM = %{$_[0]{FORM}};

    my $lang = $vars->{lang} = $FORM{lang} || 'ru';
    Yandex::I18n::init_i18n($lang);

    my $geo_id = $FORM{geo_id} || $geo_regions::MOSCOW;
    my $domain = $FORM{domain} || 'ru';
    $domain =~ s/ ^ .* \byandex\. //xms;

    my $direct_geo_id = GeoTools::get_direct_region_from_geo($geo_id);
    $vars->{office_city} = get_officecity_by_geo_footer($direct_geo_id, "yandex.$domain");

    return respond_bem($r, $r->env->{reqid}, $vars, source => 'data3');
}


=head2 cmd_showSurveySuccess

Страница приземления после прохождения опроса TNS.

Показываем без авторизации, ошибки молча игнорируем.

Параметры:

    * user_id
    * survey_id
    * chk

=cut

sub cmd_showSurveySuccess :Cmd(showSurveySuccess)
    :Description('приземление после прохождения опроса')
    :Rbac(Code => rbac_cmd_allow_all)
    :NoAuth
{
    my ($r, $vars) = @{$_[0]}{
      qw/R   vars/};
    my %FORM = %{$_[0]{FORM}};
    $vars->{display_name} = 0;

    return respond_bem($r, $r->env->{reqid}, $vars, source => 'data3')  if !$FORM{survey_id} || !$FORM{user_id} || !$FORM{chk};

    my $chk = Survey::TNS->get_user_checksum($FORM{survey_id} . $FORM{user_id});
    return respond_bem($r, $r->env->{reqid}, $vars, source => 'data3')  if $chk ne $FORM{chk};

    my $surveys = Surveys::get_surveys( condition => {type => 'tns'} );
    my ($survey_id) = map {$_->{survey_id}} grep {$_->{ext_survey_id} eq $FORM{survey_id}} @$surveys;
    return respond_bem($r, $r->env->{reqid}, $vars, source => 'data3')  if !$survey_id;

    eval {
        Surveys::complete_user_survey($survey_id, ext_user_id => $FORM{user_id});
    };

    return respond_bem($r, $r->env->{reqid}, $vars, source=>'data3');
}

# ручка -- переписанные на perl direct-handlers (часть, которая работает с инфоблоком) и direct-infoblock
sub cmd_ajaxInfoblockRequest :Cmd(ajaxInfoblockRequest)
:Description('запрос к инфоблоку')
    :Rbac(Code => rbac_cmd_allow_all)
    :NoAuth
{
    my ($r, $vars) = @{$_[0]}{
      qw/R   vars/};
    my %FORM = %{$_[0]{FORM}};

    my $req = decode_json($r->raw_body);
    my $uid = $req->{uid};
    my $euid = $req->{euid};
    if (!$euid) {
        my $session_id = get_cookie($r, 'Session_id');
        if (!$session_id) {
            return respond_text($r, 'Invalid euid');
        }
        my $bb_res = bb_sessionid($session_id, http_remote_ip($r), http_server_host($r));
        if ($bb_res->{valid} && $bb_res->{uid}) {
            $euid = $bb_res->{uid};
        }
    }
    my $region = {
        'com.tr' => 'tr',
        com => 'en',
        ua => 'ua',
        by => 'by',
        kz => 'kz',
    }->{ HttpTools::yandex_tld_domain($r) } || 'ru';
    my $lang = lang_auto_detect($r);
    $lang = 'uk' if $lang eq 'ua';
    my @res;

    unless (check_csrf_token($req->{csrf_token}, $euid)) {
        return respond_text($r, 'Invalid token');
    }

    my %userinfo;
    for my $u ($uid, $euid) {
        next if exists $userinfo{$u};
        $userinfo{$u} = rbac_who_is_detailed(undef, $u);
        my $role = $userinfo{$u}->{role};
        $userinfo{$u}->{has_own_infoblock} = $role eq 'client' || $role eq 'subclient';
        if ($role eq 'client') {
            my $chief_uid = rbac_get_chief_rep_of_client_rep($u);
            $userinfo{$u}->{has_campaigns} = get_one_field_sql(PPC(uid => $chief_uid), ['SELECT 1 FROM campaigns', where => { uid => $chief_uid, type => get_camp_kind_types('web_edit_base'), statusEmpty__ne => 'Yes' }]);
        }
    }

    my ($can_view, $can_edit);
    if ($uid == $euid) {
        $can_view = 1;
        $can_edit = 1;
    } elsif (rbac_is_owner(undef, $euid, $uid)) {
        $can_view = 1;
        $can_edit = $userinfo{$euid}->{role} eq 'manager' && !$userinfo{$euid}->{is_any_teamleader} || !$userinfo{$euid}->{is_internal_user};
    }

    my $infoblock_uid = $uid;
    if ($uid != $euid && all { my $role = $userinfo{$_}->{role}; $role eq 'client' || $role eq 'subclient' } ($uid, $euid)) {
        # комментарий из python-infoblock:
        # особый случай для взаимодействия главного представителя и младшего представителя.
        # должны отображать инфоблок главного представителя, даже если просили для младшего от имени старшего.
        $infoblock_uid = $euid;
    }

    for my $queue_item (@{ $req->{queue} }) {
        my $result = {};
        my ($error, $error_msg);
        try {
            if ($queue_item->{type} eq 'require-events') {
                die { msg => "User with euid=$euid cannot view news for uid=$uid" } unless $can_view;

                my $infoblock_state;
                if ($userinfo{$infoblock_uid}->{has_own_infoblock} && $userinfo{$infoblock_uid}->{has_campaigns}) {
                    $infoblock_state = get_one_line_sql(PPC(uid => $infoblock_uid), 'SELECT notifications, news_enabled, block_enabled, last_updated, teasers_enabled FROM infoblock_state WHERE uid = ?', $infoblock_uid);
                    if (!$infoblock_state) {
                        $infoblock_state = {
                            block_enabled => 1,
                            news_enabled => 1,
                            teasers_enabled => 0,
                        };
                    }
                } else {
                    $infoblock_state = {
                        block_enabled => 0,
                        news_enabled => 0,
                        teasers_enabled => 0,
                    };
                }
                my $notifications;
                if ($infoblock_state->{notifications}) {
                    $notifications = decode_json($infoblock_state->{notifications});
                } else {
                    $notifications = { news => { items => {} }, teasers => { items => {} }};
                }
                my $news_state = $notifications->{news}->{items};
                my $langregion = "${lang}_${region}";
                my @sorted_ext_news_ids;
                my $preloaded_news = {};

                if ($infoblock_state->{news_enabled}) {

                    my $actual_news_period_days = 0;
                    my $min_news_date;
                    if ($infoblock_state->{last_updated}) {
                        $min_news_date = str_round_day($infoblock_state->{last_updated});
                    } else {
                        $min_news_date = strftime("%Y-%m-%d", gmtime(time() - $actual_news_period_days * 86400));
                    }
                    my $new_news = get_all_sql(PPCDICT, ["SELECT ext_news_id, lang, region FROM adv_news_items", where => { news_date__ge => $min_news_date }]);
                    for my $news_item (@$new_news) {
                        my $ext_news_id = $news_item->{ext_news_id};
                        my $langregion_for_this_news = $news_item->{lang} . '_' . $news_item->{region};
                        my $has_this_news_in_this_langregion = exists $news_state->{$ext_news_id} && any { $_ eq $langregion_for_this_news } @{ $news_state->{$ext_news_id}->{langsregions} };
                        if (! $has_this_news_in_this_langregion) {
                            $infoblock_state->{last_updated} = undef;   # форсировать сохранение в базу
                            # hushed, скорее всего, не используется фронтов и точно не меняется в базе, должно быть можно оторвать
                            $news_state->{$ext_news_id} //= { langsregions => [], status => 'new', hushed => JSON::false };
                            push @{ $news_state->{$ext_news_id}->{langsregions} }, $langregion_for_this_news;
                        }
                    }

                    # упорядочивать новости нужно по дате; пока ext_news_id имеет формат d-YYYY-MM-DD-..., сортировка по нему работает
                    for my $ext_news_id (reverse sort keys %$news_state) {
                        my $langsregions = $news_state->{$ext_news_id}->{langsregions};
                        if (any { $_ eq $langregion } @$langsregions) {
                            push @sorted_ext_news_ids, $ext_news_id;
                        }
                    }

                    my $preloaded_news_limit = 5;
                    my $news_to_preload_count = min($preloaded_news_limit, scalar @sorted_ext_news_ids);
                    my @ext_news_ids_to_preload = @sorted_ext_news_ids[0 .. $news_to_preload_count-1];
                    if (@ext_news_ids_to_preload) {
                        $preloaded_news = _get_news_data(\@ext_news_ids_to_preload, $lang, $region);
                    }
                }

                $result->{news}->{items} = [];
                # timeout -- не саое удачное название (можно подумать, что отсчитывается от даты новости), но лучше не придумал. И на момент появления используется только в одном месте
                my $news_expiration_timeout_days = 1;
                my $news_expiration_time = strftime("%Y-%m-%d %H:%M:%S", gmtime(time() - $news_expiration_timeout_days * 86400));
                for my $ext_news_id (@sorted_ext_news_ids) {
                    my $status = $news_state->{$ext_news_id}->{status};
                    # статус 'old' остался от какой-то предыдущей инкарнации инфоблока, может остаться у каких-то клиентов в базе, но на новые новости не выставляется
                    next if $status eq 'old';

                    my ($date) = $ext_news_id =~ /^d-([0-9]{4}-[0-9]{2}-[0-9]{2})/;
                    my $is_expired = "$date 00:00:00" lt $news_expiration_time;
                    my $is_new = $status eq 'new' && !$is_expired;
                    my $is_read = $status eq 'read' || $is_expired;

                    my $news_item_for_response = {
                        ext_news_id => $ext_news_id,
                        read => $is_read,
                        new => $is_new,
                        # seen и visited были помечены как "Backwards compatible" в python-инфоблоке, возможно, можно их не передавать
                        seen => !$is_new,
                        visited => 0,
                    };
                    if ($preloaded_news->{$ext_news_id}) {
                        $news_item_for_response->{data} = $preloaded_news->{$ext_news_id};
                    }
                    $news_item_for_response->{$_} = $news_item_for_response->{$_} ? JSON::true : JSON::false for qw/read new seen visited/;
                    push @{ $result->{news}->{items} }, $news_item_for_response;
                }
                $result->{lang} = $lang;
                $result->{region} = $region;
                $result->{show_news} = $infoblock_state->{news_enabled};
                # при выключенном block_enabled ходим в базу, хотя это не нужно; так было в оригинальном python-infoblock
                $result->{show_block} = $result->{show_news} && $infoblock_state->{block_enabled};
                ### для обратной совместимости
                $result->{show_teasers} = 0;
                $result->{expose} = 0;  # https://st.yandex-team.ru/DIRECT-73713#5a2abbc2ee3a1a001b7ae8d3
                $result->{last_opened} = undef; # https://st.yandex-team.ru/DIRECT-73713#5a2abbc2ee3a1a001b7ae8d3
                ###
                $result->{$_} = $result->{$_} ? JSON::true : JSON::false for qw/show_news show_block show_teasers expose/;

                if ($result->{show_block} && !$infoblock_state->{last_updated}) {
                    $infoblock_state->{notifications} = encode_json($notifications);
                    $infoblock_state->{last_updated} = strftime('%Y-%m-%d %H:%M:%S', gmtime);
                    do_insert_into_table(PPC(uid => $infoblock_uid), 'infoblock_state', {
                        uid => $infoblock_uid,
                        notifications   => $infoblock_state->{notifications},
                        block_enabled   => $infoblock_state->{block_enabled},
                        news_enabled    => $infoblock_state->{news_enabled},
                        teasers_enabled => $infoblock_state->{teasers_enabled},
                        last_updated    => $infoblock_state->{last_updated},
                    }, on_duplicate_key_update => 1, key => 'uid');
                }
            } elsif ($queue_item->{type} eq 'update-news-state') {
                die { msg => "User with euid=$euid cannot change news status for uid=$uid" } unless $can_edit;

                my $data = $queue_item->{data};
                my $new_status;
                # сейчас выставляется только read
                if ($data->{read}) {
                    $new_status = 'read';
                } else {
                    # до переписывания python-инфоблока был ещё closed, но для новостей closed не имеет смысла
                    die { msg => 'Unknown news state: read property required' };
                }
                my $ext_news_id = $data->{ext_news_id};
                my $infoblock_state = get_one_line_sql(PPC(uid => $infoblock_uid), 'SELECT notifications, last_updated FROM infoblock_state WHERE uid = ?', $infoblock_uid);
                die "infoblock state does not exists yet for uid=$infoblock_uid" unless $infoblock_state;
                my $notifications = decode_json($infoblock_state->{notifications});
                my $news_state = $notifications->{news}->{items};
                my $news_data = {};
                if (exists $news_state->{$ext_news_id}) {
                    $news_state->{$ext_news_id}->{status} = $new_status;
                    $infoblock_state->{notifications} = encode_json($notifications);
                    $infoblock_state->{last_updated} = strftime('%Y-%m-%d %H:%M:%S', gmtime);
                    do_update_table(PPC(uid => $infoblock_uid), 'infoblock_state', hash_cut($infoblock_state, ['notifications', 'last_updated']), where => {uid => $infoblock_uid});
                    # нужна ли фронту новость на самом деле?
                    my $news_data = (_get_news_data([$ext_news_id], $lang, $region))->{$ext_news_id};
                } else {
                    # комментарий из python-infoblock:
                    # ОБРАТНАЯ СОВМЕСТИМОСТЬ:
                    # если про новость ничего не было известно,
                    # то молча игнорируем запрос.
                }
                $result = {
                    lang => $lang,
                    region => $region,
                    ext_news_id => $ext_news_id,
                    data => $news_data,
                    new => $new_status eq 'new',
                    read => $new_status eq 'read',
                    # комментарий в python-infoblock: Obsolete
                    visited => $new_status eq 'old',
                    seen => $new_status ne 'new',
                };
                $result->{$_} = $result->{$_} ? JSON::true : JSON::false for qw/new read visited seen/;
            ### update-last-opened, update-teaser-state, update-teaser-exposed не должны вызываться, но в коде фронта остались рудименты. Поэтому оставляем заглушки.
            } elsif ($queue_item->{type} eq 'update-last-opened') {
                $result = {};
            } elsif ($queue_item->{type} eq 'update-teaser-state') {
                $result = {};
            } elsif ($queue_item->{type} eq 'update-teaser-exposed') {
                $result = {};
            ###
            } else {
                die { msg => "request type '$queue_item->{type}' not implemented" };
            }
        } catch {
            $error = shift;
            if (ref $error eq 'HASH') {
                $error_msg = $error->{msg};
            } else {
                $error_msg = 'infoblock error';
                warn "infoblock error: $error";
            }
        };
        push @res, { uniqId => $queue_item->{uniqId}, $error ? (error=>{message=>$error_msg}) : (result=>$result) };
    }
    return respond_json($r, \@res);
}

sub cmd_ajaxGetNewsById :Cmd(ajaxGetNewsById)
:Description('получение новости для инфоблока')
    :Rbac(Code => rbac_cmd_allow_all)
    :NoAuth
{
    my ($r, $vars) = @{$_[0]}{
      qw/R   vars/};
    my %FORM = %{$_[0]{FORM}};

    my $euid;
    my $session_id = get_cookie($r, 'Session_id');
    my $bb_res = bb_sessionid($session_id, http_remote_ip($r), http_server_host($r));
    if ($bb_res->{valid} && $bb_res->{uid}) {
        $euid = $bb_res->{uid};
    }
    my $region = {
        'com.tr' => 'tr',
        com => 'en',
        ua => 'ua',
        by => 'by',
        kz => 'kz',
    }->{ HttpTools::yandex_tld_domain($r) } || 'ru';
    my $lang = lang_auto_detect($r);
    $lang = 'uk' if $lang eq 'ua';

    unless (check_csrf_token($FORM{csrf_token}, $euid)) {
        return respond_text($r, 'Invalid token');
    }

    my $res;

    # TODO заменить обращение в intapi на вызов _get_news_data
    my $news_service = Intapi::CLegacy::News->new();
    my $params = {
            region => $region,
            lang => $lang,
            ext_news_id => $FORM{ext_news_id},
        };
    my $news_service_result = eval { $news_service->get_news_by_id($params) };
    if (!$news_service_result) {
        warn $@;
        $res = { error => 'internal server error' };
    } else {
        $res = $news_service_result;
    }
    return respond_json($r, $res);
}

#Функции, добавляющие в dataset указанной страницы дополнительные данные
my $_ENRICHERS = {
    freelancers => \&_fill_page_meta_for_freelancer_card,
};

=head2 cmd_dumb

Контролер для страницы превью креативов. Используется в модерации и в Толоке.

=cut

sub cmd_creativePreview :Cmd(creativePreview)
    :Description('страница превью креативов')
    :Rbac(Code => rbac_cmd_allow_all)
    :PredefineVars(qw//)
    :NoAuth
{
    my ($r, $vars) = @{$_[0]}{
        qw/R   vars/};

    my %FORM = %{$_[0]{FORM}};
    $vars->{is_public} = 1;
    $vars->{current_url} = '/public/creative-preview?'.TTTools::get_current_url_params(\%FORM);

    $vars->{scheme} = http_server_scheme($r);
    $vars->{host}   = http_server_host($r);
    $vars->{request_method} = $ENV{REQUEST_METHOD};

    return respond_bem($r, $r->env->{reqid}, $vars, source=>'data3');
}

=head2
Контролер для страницы превью видео (из видео конструктора). Используется в сендбоксной таске чтобы снять по нему видео.
=cut
sub cmd_videoEditorPreview :Cmd(videoEditorPreview)
    :Description('контроллер для превью видео без авторизации')
    :PredefineVars(qw//)
    :Rbac(Code => rbac_cmd_allow_all)
    :NoAuth
{
    my ($r, $vars) = @{$_[0]}{
        qw/R   vars/};
    my %FORM = %{$_[0]{FORM}};
    $vars->{is_public} = 1;
    $vars->{current_url} = '/public/video-editor-preview?'.TTTools::get_current_url_params(\%FORM);
    $vars->{scheme} = http_server_scheme($r);
    $vars->{host}   = http_server_host($r);
    $vars->{request_method} = $ENV{REQUEST_METHOD};
    return respond_bem($r, $r->env->{reqid}, $vars, source=>'data3');
}

=head2 cmd_showDna

Контроллер для нового интерфейса без авторизации

=cut

sub cmd_showDnaPb
    :Cmd(showDnaPb)
    :Description('единый контроллер для нового интерфейса без авторизации')
    :PredefineVars(qw//)
    :Rbac(Code => rbac_cmd_allow_all)
    :NoAuth
{
    my ($r, $vars) = @{$_[0]}{
      qw/R   vars/};

    my $direct_cache = new DirectCache(groups => []);

    my %FORM = %{$_[0]{FORM}};
    $vars->{grid_allowed} = 1;
    $vars->{is_public} = 1;
    $vars->{svn_info} = TTTools::get_svn_info();
    $vars->{current_url} = '/public/?'.TTTools::get_current_url_params(\%FORM);

    $vars->{scheme} = http_server_scheme($r);
    $vars->{host}   = http_server_host($r);

    #TVM-тикет нужен для получения фич из intapi
    local $Yandex::TVM2::APP_ID;      $Yandex::TVM2::APP_ID //= $Settings::TVM2_APP_ID{web};
    local $Yandex::TVM2::SECRET_PATH; $Yandex::TVM2::SECRET_PATH //= $Settings::TVM2_SECRET_PATH{web};
    no warnings 'once';
    local $Yandex::Blackbox::BLACKBOX_USE_TVM_CHECKER = \&TvmChecker::use_tvm;

    state $enriching_mode_prop = Property->new('ENRICH_DATA_AT_SHOW_DNA_PB');
    my $enriching_mode = $enriching_mode_prop->get(60) // 0;

    my $is_internal_ip = is_internal_ip(http_remote_ip($r));
    $vars->{is_internal_ip} = $is_internal_ip;

    my $operator_uid = Apache::AuthAdwords::auth2($r) // 0;

    #Если зашел авторизованный пользователь - достанем его данные для отрисовки шапки
    if ($operator_uid > 0) {
        my $rbac;

        $vars->{UID} = $vars->{uid} = $operator_uid;
        $vars->{uname} = $r->{_LOGIN};
        $vars->{display_name} = $r->{_DISPLAY_NAME};
        $vars->{address_list} = $r->{_ADDRESS_LIST};
        $vars->{blackbox_users} = $r->{_BLACKBOX_USERS};
        $vars->{allow_more_users} = $r->{_ALLOW_MORE_USERS};
        $vars->{is_internal_ip} = is_internal_ip(http_remote_ip($r));

        eval { $rbac = RBAC2::Extended->get_singleton( $operator_uid ) };
        if ($rbac){
            my $operator_info = Primitives::get_operator_info_for_do_cmd($operator_uid);

            my $rbac_login_rights;
            my $rbac_login_result = rbac_login_check($rbac, {
                    UID => $operator_uid,
                    ClientID => $operator_info->{ClientID},
                    user_info => $operator_info,
                    is_internal_ip => $is_internal_ip,
                }, \$rbac_login_rights);

            if ($rbac_login_result == 0 && $rbac_login_rights->{ClientID}) {
                $vars->{login} = $FORM{ulogin} || get_login(uid => $operator_uid);
                $vars->{user_login} = $vars->{ulogin} = $vars->{uname} = $operator_info->{login};
                $vars->{user_email} = $operator_info->{email};
                $vars->{user_fio} = $operator_info->{FIO};

                my $uid_small;
                my $uid_info;
                if ($FORM{ulogin}) {
                    $uid_small = get_uid_by_login2($FORM{ulogin});
                    if (
                        defined $uid_small
                            && $uid_small
                            && $operator_uid != $uid_small
                            && (($rbac_login_rights->{agency_control} && rbac_who_is($rbac, $uid_small) ne 'empty')
                            || $rbac_login_rights->{limited_support_control}
                            || $rbac_login_rights->{is_any_client}
                            || $rbac_login_rights->{role} =~ /empty/
                        )
                            && ! rbac_is_owner($rbac, $operator_uid, $uid_small)
                    )
                    {
                        # на стороне интерфейса пустой список преобразуется в ошибку
                        $vars->{features_enabled_for_client_all} = [];
                    }

                    $uid_info = get_user_info($uid_small);

                    $vars->{uid} = $uid_small;
                }

                # Получение фичей для оператора и клиента
                my $custom_abt_info_soft_timeout;
                my $custom_abt_info_num_attempts;
                state $custom_abt_info_soft_timeout_property //= Property->new('SHOW_DNA_ABT_INFO_SOFT_TIMEOUT');
                state $custom_abt_info_num_attempts_property //= Property->new('SHOW_DNA_ABT_INFO_NUM_ATTEMPTS');
                $custom_abt_info_soft_timeout = $custom_abt_info_soft_timeout_property->get(120);
                $custom_abt_info_num_attempts = $custom_abt_info_num_attempts_property->get(120);
                local $Direct::Feature::SOFT_TIMEOUT = $custom_abt_info_soft_timeout if $custom_abt_info_soft_timeout;
                local $Direct::Feature::NUM_ATTEMPTS = $custom_abt_info_num_attempts if $custom_abt_info_num_attempts;
                state $features_parallel_request_property //= Property->new('FEATURES_PARALLEL_REQUEST');
                my $features_parallel_request = $features_parallel_request_property->get(120);
                my $operator_client_id = $operator_info->{ClientID};
                my @client_ids = ();
                if ($operator_client_id) {
                    push @client_ids, $operator_client_id;
                }
                if ($features_parallel_request && defined $uid_info && $uid_info->{ClientID}) {
                    push @client_ids, $uid_info->{ClientID};
                }
                my $ab_info = Client::ClientFeatures::get_abt_info_parallel(\@client_ids);
                if ($operator_client_id) {
                    $vars->{ab_boxes} = [split /;/, ($ab_info->{$operator_client_id}->{boxes} // '')];
                    $vars->{ab_boxes_crypted} = $ab_info->{$operator_client_id}->{boxes_crypted};
                    $vars->{is_fast_show_dna} = Client::ClientFeatures::is_velocity_fast_show_dna_enabled($operator_client_id) || (exists $FORM{fast_show_dna});
                }
                
                $vars->{grid_allowed} = Client::ClientFeatures::has_grid_allowed_feature($rbac_login_rights->{ClientID});
                $vars->{login_rights} = $rbac_login_rights;

                #нужен для страницы видеоконструктора
                $vars->{video_constructor_enabled} = Client::ClientFeatures::has_video_constructor_enabled_direct_feature($rbac_login_rights->{ClientID});
                # нужен для включения создания видео с нуля в видеоконструкторе
                $vars->{video_constructor_create_from_scratch_enabled} =
                        Client::ClientFeatures::has_video_constructor_create_from_scratch_enabled_direct_feature($rbac_login_rights->{ClientID});
                #нужен для доступа к фидам в видеоконструкторе
                $vars->{video_constructor_feed_enabled} =
                        Client::ClientFeatures::has_video_constructor_feed_enabled_direct_feature($rbac_login_rights->{ClientID});

                #добавляем во все контроллеры список фичей доступных оператору и клиенту
                my @operator_features = map {uc $_} @{Client::ClientFeatures::allowed_features($rbac_login_rights->{ClientID})};
                $vars->{features_enabled_for_operator_all} = \@operator_features;

                if (defined $uid_info && $uid_info->{ClientID} != 0) {
                    my @client_features = map {uc $_} @{Client::ClientFeatures::allowed_features($uid_info->{ClientID})};
                    $vars->{features_enabled_for_client_all} ||= \@client_features;
                }
                $vars->{features_enabled_for_client_all} ||= $vars->{features_enabled_for_operator_all};

                $vars->{is_uc_grid_design_enabled} = Client::ClientFeatures::has_uc_grid_design_enabled($rbac_login_rights->{ClientID});

                $vars->{real_login} = $r->{_REAL_LOGIN};

                if (my $perms = Rbac::get_perminfo(uid => $operator_uid)) {
                    $vars->{AgencyID} = $perms->{agency_client_id};
                }
            }
        }
        else{
            #Пользователь есть в паспорте, но отсутствует в Директе
            $vars->{user_login} = $r->{_LOGIN};
            $vars->{has_no_client_id} = 1;
        }
    }

    $vars->{csrf_token} = get_csrf_token($operator_uid);

    $vars->{login_rights} //= {role => ''};

    my @nodes = split qr{/}, ($FORM{context} // '');

    my $is_enriching_allowed = $enriching_mode =~ /by_flag/i ? $FORM{_enrich_data} : $enriching_mode;

    if ( (my $enricher = $_ENRICHERS->{shift @nodes}) && $is_enriching_allowed ){
        $vars = $enricher->(\@nodes, $vars);
        if ($vars->{ERROR_NOT_FOUND}) {
            $direct_cache->on_request_end;
            return respond_http_error($r, 404);
        }
    }

    $direct_cache->on_request_end;
    return respond_bem($r,  $r->env->{reqid}, $vars, source=>'data3');
}

sub _fill_page_meta_for_freelancer_card {
    my ($subnodes, $vars) = @_;
    
    if (!@$subnodes) {
        #Страница со списком фрилансеров
        $vars->{page_title} = iget('Специалисты по настройке Яндекс.Директа');
        $vars->{page_description} = iget('Размещение рекламы всегда можно доверить профессионалам — специалистам по рекламным технологиям Яндекса, которые прошли проверку знаний и получили именной электронный сертификат. Здесь вы можете выбрать исполнителя для настройки, ведения или аудита ваших рекламных кампаний в нужном регионе.');
    } else {
        #Страница фрилансера, первый элемент в subnodes должен быть логином
        my $fr_login = shift @$subnodes;
        my $fr_uid = get_uid_by_login($fr_login);
        return {ERROR_NOT_FOUND => 1} unless $fr_uid && RBACDirect::is_freelancer($fr_uid);

        my $fr_info = get_freelancer_info($fr_uid, (image => 'orig'));
        $vars->{page_title} = join ' ', @$fr_info{qw/first_name second_name/}, iget('- Специалист по настройке Яндекс.Директа') ;
        $vars->{page_description} = $fr_info->{brief};
        $vars->{page_meta} = {
            'og:title' => _get_meta(property => $vars->{page_title}),
            'og:url'   => _get_meta(property =>(sprintf '%s://%s/dna/freelancers/%s', @$vars{qw/scheme host/}, $fr_login)),
            'og:type' =>  _get_meta(property => 'profile'),
            'og:image'=>  _get_meta(property =>$fr_info->{avatar_url}),
	        'og:locale' => _get_meta(property => Yandex::I18n::get_locale(Yandex::I18n::current_lang())),
	        'og:description' => _get_meta(property => $vars->{page_description}),
            'profile:first_name' => _get_meta(property => $fr_info->{first_name}),
            'profile:last_name'  => _get_meta(property => $fr_info->{second_name}),
            'profile:username'   => _get_meta(property => $fr_login),
        };

    }

    return $vars;
}

sub _get_meta($$) {
    my ($type, $content) = @_;

    return {type => $type, content => $content}
}

our $NEWS_CACHE_TIMEOUT = 60;   # секунд
# логика из Intapi::CLegacy::News, от которого планируется избавиться
sub _get_news_data($$$) {
    my ($ext_news_ids, $lang, $region) = @_;

    if (exists $Settings::NEWS_REGION_ALIAS{$region}) {
        $region = $Settings::NEWS_REGION_ALIAS{$region};
    }
    my $regions_for_lang_info = $Settings::NEWS_LANG_REGIONS{$lang};
    die "Unknown feed <$region>-<$lang>" if !$regions_for_lang_info || none { $region eq $_ } @{ $regions_for_lang_info->{region} // [] };

    # кэширование было в Intapi::CLegacy::News
    # Насколько оно обосновано (запросы в ppcdict из инфоблока -- единицы процентов из общего потока по количеству), непонятно. Кроме этого, эффективность кэширования внутри процесса зависит от конфигурации веб-сервера.
    # С другой стороны, не хочется читать по многу раз одни и те же данные (предположительно, большая часть загружаемых новостей -- последние N недавних), которые редко меняются. Если хотим серьёзно подойти к кэшированию, имеет смысл попробовать Redis
    state %news_cache;
    my @news_ids_to_fetch;
    my %res;
    for my $id (@$ext_news_ids) {
        my $key = "$region$lang$id";
        if (exists $news_cache{$key} && time() < $news_cache{$key}->{expire_time}) {
            # В результате возвращается ссылка на структуру из кэша. Никаких гарантий на то, что вызывающая сторона не поменяет данные, хранящиеся в структуре (и тем самым попортит кэш), не даётся. Пока функция вызывается только в паре мест внутри этого же модуля и это можно отследить. При необходимости можно использовать Storable::dclone или аналогичное
            $res{$id} = $news_cache{$key}->{data};
        } else {
            delete $news_cache{$key};
            push @news_ids_to_fetch, $id;
        }
    }
    my $news_from_db = get_all_sql(PPCDICT, ['SELECT ext_news_id, data FROM adv_news_items', where => { region => $region, lang => $lang, ext_news_id => \@news_ids_to_fetch }]);
    for my $news_item (@$news_from_db) {
        my $id = $news_item->{ext_news_id};
        my $key = "$region$lang$id";
        my $data = hash_cut from_json($news_item->{data}), qw/date content/;
        $news_cache{$key} = { expire_time => time() + $NEWS_CACHE_TIMEOUT, data => $data };
        $res{$id} = $data;
    }
    return \%res;
}

1;
