package DoCmdAdGroup;

use Direct::Modern;

use base qw/DoCmd::Base/;

use Settings;

use Direct::ResponseHelper;
use BannerFlags;
use PrimitivesIds;
use Tools;
use HttpTools;
use TextTools;
use CampaignTools;
use Currencies;
use VCards;
use Client qw//;
use Campaign qw/get_camp_info/;
use Campaign::Types qw/get_camp_supported_adgroup_types/;
use User qw/get_user_data/;
use BannersCommon qw//;
use Tag qw//;
use URLDomain qw//;
use Promocodes qw/get_promocode_domains/;

use Models::AdGroup;
use Models::Banner;
use Models::Campaign qw//;
use Campaign::Types qw/get_camp_type get_camp_kind_types camp_kind_in/;
use FilterSchema;
use Client;
use Geo qw/get_common_geo_for_camp set_extended_geo_to_camp/;

use Yandex::DBShards qw/SHARD_IDS/;
use Yandex::DBTools;
use Yandex::HashUtils;
use Yandex::ListUtils qw/xdiff/;
use Yandex::Validate;
use Yandex::ScalarUtils qw/str/;
use Yandex::I18n;
use List::Util qw/first/;
use List::MoreUtils qw/all any none uniq part/;
use DoCmd::Checks;
use Direct::Market qw/fill_in_groups_rating/;
use Direct::Validation::UserData qw/check_user_data/;
use Direct::ValidationResult;
use Direct::Errors::Messages;
use Direct::AdGroups2::Smart;
use Direct::AdGroups2;
use Direct::Creatives qw/get_used_creative_ids/;
use Direct::Feeds;
use Direct::Banners;
use Direct::PerformanceFilters;
use Direct::Model::AdGroupPerformance;
use Direct::Validation::DynamicConditions;
use DoCmdAdGroup::Helper;
use Direct::Model::BidRelevanceMatch::Helper;
use HierarchicalMultipliers;

use MinusWords;

=head2 cmd_editAdGroups

    Первый шаг создания/редактирования группы (на текущий момент только перфоманс)

=cut

sub cmd_editAdGroups
    :Cmd(
        addAdGroupsPerformance, editAdGroupsPerformance
    )
    #:CheckBySchema(Input => 'warn', Output => 'warn')
    :Rbac(Code => rbac_cmd_by_owners, CampKind => {web_edit => 1})
    :RequireParam(cid => 'Cid')
    :Lock(1)
    :Description('Редактирование нескольких объявлений')
    :PredefineVars(qw/campaign_agency_contacts enable_cpm_deals_campaigns enable_content_promotion_video_campaigns enable_recommendations is_feature_smart_at_search_enabled/)
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c    vars/};
    my %FORM = %{$_[0]{FORM}};

    my $cid = $FORM{cid};
    my $pids = [grep { $_ > 0 } @{get_num_array_by_str($FORM{adgroup_ids})}];
    my $bids = defined($FORM{bids} // $FORM{bid}) ? [grep { $_ > 0 } @{get_num_array_by_str($FORM{bids} // $FORM{bid})}] : undef;

    my $action = $FORM{cmd} =~ /^(.+)?AdGroups/ ? $1 : '';
    if ($action eq 'edit' && !@$pids) {
        error(
            iget('Ошибка: не задан номер группы.'),
            {return_to => { href => $SCRIPT . '?cmd=showCamps' . str($FORM{uid_url}), text => iget('вернуться к списку кампаний') }}
        );
    }

    # не указана опция without_multipliers, т.к. корректировки нужны ниже по коду
    my $campaign = get_camp_info($cid, $c->client_chief_uid, with_strategy => 1);
    $campaign->{tags} = Tag::get_all_campaign_tags($cid);
    my $supported_adgroup_types = get_camp_supported_adgroup_types(type => $campaign->{type});
    my $adgroup_type = $FORM{cmd} =~ /^(.+)?AdGroups(.+)$/ ? lc($2) : '';
    unless (any { $_ eq $adgroup_type } @$supported_adgroup_types) {
        error(
            iget('Ошибка: Данную группу объявлений нельзя сохранить в текущей кампании (не совместимый тип кампании)'),
            {return_to => { href => $SCRIPT . '?cmd=showCamps' . str($FORM{uid_url}), text => iget('вернуться к списку кампаний') }}
        );
    }

    if ($campaign->{archived} && $campaign->{archived} eq 'Yes') {
        error(
            iget('Объявления недоступны для редактирования, поскольку кампания № %d перенесена в архив', $campaign->{cid}),
            {return_to => { href => $SCRIPT . '?cmd=showCamps' . str($FORM{uid_url}), text => iget('вернуться к списку кампаний') }}
        );
    }

    # Признак единого geo на кампанию
    $campaign->{common_geo_set} = 0;
    my $geo_for_new_adgroup; # Регион для новой группы
    my $common_geo = get_common_geo_for_camp($cid);
    if (defined $common_geo) {
        $geo_for_new_adgroup = $common_geo; # Тут модификация региона не нужна
        $campaign->{common_geo} = GeoTools::modify_translocal_region_before_show($common_geo, {ClientID => $c->client_client_id});
        $campaign->{common_geo_set} = 1;
    }
    $campaign = set_extended_geo_to_camp($campaign, $c->client_client_id);
    my ($adgroups, $feeds);
    if ($action eq 'edit') {
        $adgroups = Direct::AdGroups2->get_by(adgroup_id => $pids, adgroup_type => $adgroup_type, extended => 1, with_tags => 1, with_multipliers => 1)->items;
        my $pid2filters = Direct::PerformanceFilters->get_by(adgroup_id => $pids, with_additional => 1)->items_by('adgroup_id');

        my $pid2banners = Direct::Banners->get_by(
            adgroup_id => $pids,
            banner_type => $adgroup_type,
            $bids ? (filter => {bid => $bids, statusArch => 'No'}) : (),
        )->items_by('adgroup_id');
        for my $adgroup (@$adgroups) {
            $adgroup->performance_filters($pid2filters->{$adgroup->id} || []);
            $adgroup->banners($pid2banners->{$adgroup->id} || []);
        }
        $feeds = Direct::Feeds->get_with_categories($c->client_client_id, active_or_used => 1, adgroups => $adgroups)->items;
    } else {
        $feeds = Direct::Feeds->get_with_categories($c->client_client_id, active => 1)->items;
        $adgroups = [Direct::Model::AdGroupPerformance->new(
            id => 0,
            campaign_id => $cid,
            adgroup_name => '',
            geo => $geo_for_new_adgroup,
            hierarchical_multipliers => {},
            banners_count => 0,
            archived_banners_count => 0,
            feed_id => 0,
            performance_filters => [],
            banners => [],
        )];
    }
    $campaign->{has_archived_banners_are_opened_for_edit} = !!($FORM{banner_status} && $FORM{banner_status} eq 'all'
                                                                && any { $_->archived_banners_count > 0 } @$adgroups);

    my $all_retargeting_conditions = Retargeting::get_retargeting_conditions(ClientID => $c->client_client_id, with_campaigns => 1);

    # prepare data for template
    my @camp_groups;
    state $transformation = {
        id => 'adgroup_id',
        archived_banners_count => 'banners_arch_quantity',
        banners_count => 'banners_quantity',
        adgroup_name => 'group_name'
    };

    my $is_adgroups_copy_action = ($FORM{is_groups_copy_action} || 0) ? 1 : 0;
    my @pids = grep { $_ > 0 } map { $_->id } @$adgroups;
    my $adgroup_creative_ids = {};
    if (@pids) {
        $adgroup_creative_ids = get_used_creative_ids(\@pids);
    }
    for my $group_model (@$adgroups) {
        my $group = $group_model->to_hash;
        $group->{used_creative_ids} = $adgroup_creative_ids->{$group_model->id} || [];
        $group->{is_group_copy_action} = $is_adgroups_copy_action;
        while (my ($old_key, $new_key)  = each %$transformation) {
            $group->{$new_key} = delete $group->{$old_key}
        }
        for my $banner (@{$group->{banners}}) {
            $banner->{bid} = delete $banner->{id};
            $banner->{creative}->{creative_id} = delete $banner->{creative}->{id};
            $banner->{BannerID} = delete $banner->{bs_banner_id};
        }
        $group->{performance_filters} = [map {
            hash_merge $_->to_template_hash, {retargeting => $all_retargeting_conditions->{ $_->ret_cond_id // 0 }}
        } @{$group_model->performance_filters}];
        push @camp_groups, $group;
    }

    BannersCommon::modify_groups_geo_for_translocal_before_show(\@camp_groups, {ClientID => $c->client_client_id});
    $campaign->{groups} = \@camp_groups;

    $vars->{campaign} = $campaign;
    $vars->{feeds} = [map { $_->to_template_hash } @$feeds];
    $vars->{tags_allowed} = get_user_data($c->client_chief_uid, [qw/tags_allowed/])->{tags_allowed};
    $vars->{is_groups_copy_action} = $is_adgroups_copy_action;
    $vars->{all_retargeting_conditions} = $all_retargeting_conditions;
    $vars->{filter_schema_performance} = FilterSchema->new(filter_type => 'performance')->iget_schema('compiled');
    # Проверяем тип кампании, что доступны Я.Аудитории.
    $vars->{is_audience_enabled} = camp_kind_in(type => $campaign->{mediaType}, 'allow_audience') ? 1 : 0;

    if ($vars->{enable_cpm_deals_campaigns}){
        $vars->{new_deals_count} = Client::get_count_received_deals($c->login_rights->{ClientID});
    }

    $vars->{is_first_camp} = $FORM{from_newCamp} ? Primitives::is_first_camp_under_wallet($cid, $c->client_client_id) : 0;

    $vars->{features_enabled_for_client} //= {};
    hash_merge $vars->{features_enabled_for_client}, Client::ClientFeatures::get_features_enabled_for_client(
        $c->client_client_id,
        [qw/cpc_device_modifiers support_chat cpm_video_device_modifiers
            is_demography_bid_modifier_unknown_age_allowed mobile_os_bid_modifier_enabled/]);

    return respond_bem($r, $c->reqid, $vars, source=>'data3');
}

=head2 cmd_saveAdGroups

Сохранение групп объявлений (финальный шаг)

=cut

sub _can_create_new_adgroup {
    my ($r, $c, $login_rights, $cid) = @_;

    my $camp_type = get_camp_type(cid => $cid);

    if ($camp_type eq 'dynamic' && !$login_rights->{super_control}) {
        # Для турецких клиентов создание первого динамического объявлений доступно только суперу
        return 0 if !CampaignTools::is_dynamic_allowed($c->client_client_id, $login_rights, yandex_domain($r));
    } elsif ($camp_type eq 'performance') {
        my $is_next_performance_adgroup = sub { get_one_field_sql(PPC(cid => $cid), q{
            SELECT 1 FROM phrases g JOIN adgroups_performance gperf ON (gperf.pid = g.pid) WHERE g.cid = ? LIMIT 1
        }, $cid) };

        if (
            !$is_next_performance_adgroup->()
            && !CampaignTools::is_performance_allowed($c->client_client_id, $login_rights, yandex_domain($r))
        ) {
            return 0;
        }
    }

    return 1;
}

sub _respond_error {
    my ($need_json, $r) = @_;

    if ($need_json) {
        return sub {
            my ($errors) = @_;
            if (ref $errors) {
                return respond_json($r, {errors => $errors});
            } else {
                return respond_json($r, {errors => {
                    groups => { generic_errors => [ {text => $errors} ] }
                }});
            }
        }
    }
    return sub {
        my ($error_text, $go_back_link) = @_;
        return error($error_text, $go_back_link);
    }
}

sub cmd_saveAdGroups
    :Cmd(saveDynamicAdGroups, saveMobileAdGroups, savePerformanceAdGroups, saveTextAdGroups, bannersMultiSave, saveMediaAdGroups)
    :Rbac(
        Code => [rbac_cmd_by_owners, rbac_cmd_user_allow_edit_camps],
        ExceptRole => [media, superreader],
        CampKind => {web_edit_base => 1},
    )
    :RequireParam(cid => 'Cid', json_groups => 'saveAdGroups')
    :Lock(1)
    :CheckCSRF
    :Description('Сохранение групп объявлений в кампании')
    :PredefineVars(qw/is_featureTurboLandingEnabled is_feature_smart_at_search_enabled/)
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $cvars, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS  vars    c/};
    my %FORM = %{$_[0]{FORM}};

    my $cid = $FORM{cid};
    my $client_id = $c->client_client_id;
    my $form_adgroups = $FORM{json_groups};
    my $save_as_draft = $FORM{save_draft};
    my $is_adgroups_copy_action = !!$FORM{is_groups_copy_action};

    # pressed "back": go to prev step (editCamp&new_camp=1)
    if (defined $FORM{btn_back}) {
        return redirect($r, $SCRIPT, {
            cmd => 'editCamp',
            continue_creating => 1,
            cid => $cid,
            uid_url =>$FORM{uid_url}
        });
    }

    my $retpath = $FORM{retpath} || "${SCRIPT}?cmd=showCamp&cid=${cid}".str($FORM{uid_url});
    my $back_to_campaign_page = {return_to => {href => $retpath, text => iget('вернуться на страницу кампании')}};

    my $camp_type = get_camp_type(cid => $cid);
    my $is_text = $camp_type eq 'text';
    my $is_dynamic = $camp_type eq 'dynamic';
    my $is_mobile_content = $camp_type eq 'mobile_content';
    my $is_performance = $camp_type eq 'performance';
    my $is_mcbanner = $camp_type eq 'mcbanner';
    my $is_cpm_banner = $camp_type eq 'cpm_banner';
    my $is_cpm_deals = $camp_type eq 'cpm_deals';
    # Признак "лекого" редактирования (только текстов)
    my $is_light = $FORM{is_light};
    my $need_json_response = $is_performance || $is_dynamic || $is_mcbanner || $is_cpm_banner || $is_cpm_deals;
    my $respond_error = _respond_error($need_json_response, $r);

    my $context_relevance_match_enabled_for_campaign = Campaign::has_context_relevance_match_feature($camp_type, $client_id);
    $cvars->{context_relevance_match_enabled_for_campaign} = $context_relevance_match_enabled_for_campaign;

    # При добавлении новой группы мы ожидаем определённый формат входных данных
    my $is_new_adgroup = any { !$_->{adgroup_id} } @$form_adgroups;
    return $respond_error->(iget('Неверные входные данные')) if $is_new_adgroup && @$form_adgroups > 1;

    # Проверка возможности созданий новой группы
    if ($is_new_adgroup && !_can_create_new_adgroup($r, $c, $login_rights, $cid)) {
        return $respond_error->(iget('Ошибка! Вам недоступна работа с данным типом групп.'), $back_to_campaign_page);
    }

    for my $form_adgroup (@$form_adgroups) {
        if ($form_adgroup->{hierarchical_multipliers}) {
            $form_adgroup->{hierarchical_multipliers} = HierarchicalMultipliers::unwrap_expression_multipliers(
                $form_adgroup->{hierarchical_multipliers}
            );
        }
    }

    local $LogTools::context{cid} = $cid;

    # Пропускаем частичные пустые группы (без условий показа)
    return $respond_error->(iget('Ошибка: группы объявлений не найдены.'), $back_to_campaign_page) if !@$form_adgroups;

    my $vr;
    $vr = Direct::ValidationResult->new() if $need_json_response;

    my $has_extended_relevance_match = Campaign::has_context_relevance_match_feature($camp_type, $c->{client_client_id});
    for my $form_adgroup (@$form_adgroups) {
        my $vr_item;
        $vr_item = $vr->next if $vr;
        if (!@{$form_adgroup->{banners} // []}) {
            my $error = iget('Ошибка: объявления не найдены.');
            $vr_item ? $vr_item->add(banners => error_ReqField($error)) : return $respond_error->($error, $back_to_campaign_page);
        }
        # Ожидаю $vr_item defined
        if ($is_cpm_banner || $is_cpm_deals) {
            if (!exists $form_adgroup->{retargetings} && !exists $form_adgroup->{keywords} && !exists $form_adgroup->{minus_words}) {
                $vr_item->add_generic(error_RequiredAtLeastOneOfParameters(iget(
                    'Один из параметров %s должен быть указан', 'retargetings, keywords, minus_words')));
            }
            if (exists $form_adgroup->{retargetings} && (exists $form_adgroup->{keywords} || exists $form_adgroup->{minus_words})) {
                $vr_item->add_generic(error_PossibleOnlyOneParameter(iget(
                    'Только один параметр из %s может быть указан', 'retargetings, (keywords|minus_words)')));
            }
        }
    }
    return $respond_error->(
            Direct::ValidationResult::convert_to_hash($vr)->{objects_results},
            $back_to_campaign_page
        ) if $vr && !$vr->is_valid;

    my $chief_user = User::get_user_data($c->client_chief_uid, [qw/uid tags_allowed/]);

    # Поддержка сохранения группы с первого шага
    # условие на $FORM{auto_price} по-идее больше не нужно, т.к. в продакшене этот параметр всегда присылается со значением 1
    # однако автотесты все еще работают в режиме без auto_price. Пока тесты в процессе починки, оставляем условие
    my $is_auto_price = !!$FORM{auto_price} && !$is_light && ($is_text || $is_mobile_content || $is_mcbanner || $is_cpm_banner || $is_cpm_deals);

    my ($search_options, $options);
    if ($is_auto_price) {
        if (!$is_new_adgroup) {
            my @pids = grep { is_valid_id($_)} map { $_->{adgroup_id } } @$form_adgroups;
            my @bids = map { my $group = $_; grep { is_valid_id($_) } map { $_->{bid} } @{$group->{banners}} } @$form_adgroups;
            $search_options = {
                pid => \@pids,
                (scalar @bids ? (bid => \@bids) : ())
            };
        } else {
            $search_options = {no_groups => 1};
        }
        $options = {
            not_use_own_stat => $is_adgroups_copy_action || $is_new_adgroup,
            get_auction => ($is_cpm_banner || $is_cpm_deals) ? 0 : 1,
            detailed_retargeting_warnings => 1,
            count_banners_types => 1,
        };
    } else {
        $search_options = {no_groups => 1};
        $options = {detailed_retargeting_warnings => 1};
    }

    # не указана опция without_multipliers, т.к. корректировки нужны ниже по коду
    my $campaign = Models::Campaign::get_user_camp_gr($c->client_chief_uid, $cid, $search_options, $options);

    if ($campaign->{mediaType} eq 'mobile_content' && !$campaign->{mobile_app_id}) {
        # мобильное приложение может быть не заполнено в кампании при создании кампании через API или Excel
        # но для дальнейшей работы на сайте приложение необходимо заполнить на странице редактирования кампании
        return redirect($r, $SCRIPT, {cmd => 'editCamp', cid => $cid, ulogin => $FORM{ulogin}});
    }

    my $promocode_domains = get_promocode_domains($cid);

    if ($is_auto_price) {
        my $error_vars = DoCmdAdGroup::Helper::prepare_and_validate_for_save_adgroup(
            campaign => $campaign,
            form_adgroups => $form_adgroups,
            uid => $uid,
            client_id => $client_id,
            login_rights => $login_rights,
            is_new_adgroup => $is_new_adgroup,
            is_adgroups_copy_action => $is_adgroups_copy_action,
            camp_banners_domain => $FORM{camp_banners_domain},
            operator_uid => $UID,
            promocode_domains => $promocode_domains
        );
        if ($error_vars) {
            hash_merge $error_vars, hash_cut $cvars, qw/is_featureTurboLandingEnabled
            context_relevance_match_enabled_for_campaign is_feature_smart_at_search_enabled/;

            $error_vars->{campaign}->{hierarchical_multipliers} = HierarchicalMultipliers::wrap_expression_multipliers(
                $error_vars->{campaign}->{hierarchical_multipliers}
            );
            for my $group (@{$error_vars->{campaign}->{groups}}) {
                $group->{hierarchical_multipliers} = HierarchicalMultipliers::wrap_expression_multipliers(
                    $group->{hierarchical_multipliers}
                );
            }
            return respond_bem($r, $c->reqid, $error_vars, source => 'data3');
        }
    }

    # Приведем form_adgroups к виду для AdGroups2::Smart
    for my $form_adgroup (@$form_adgroups) {
        delete $form_adgroup->{pid};
        $form_adgroup->{adgroup_name} = delete($form_adgroup->{group_name}) if exists $form_adgroup->{group_name};
        $form_adgroup->{minus_words} = MinusWords::polish_minus_words_array($form_adgroup->{minus_words}) if $form_adgroup->{minus_words};

        for my $form_banner (@{$form_adgroup->{banners}}) {
            my $has_vcard = delete $form_banner->{has_vcard};
            # подпорка подпорки - не добавляем/удаляем undef в mcbanner (как у image_ad), а сразу требуем от фронта его отсутствия
            $form_banner->{vcard} = undef if !$has_vcard && ($form_banner->{ad_type} // '') ne 'mcbanner';

            # Для легкого редактирования нет: визитки
            delete @{$form_banner}{qw/vcard/} if $is_light || ($form_banner->{ad_type} // '') eq 'image_ad';

            if ($is_text) {
                $form_banner->{is_mobile} //= ($form_banner->{banner_type} // '') eq 'mobile';
                delete $form_banner->{banner_type};
            }

            if ($is_performance) {
                $form_banner->{creative_id} //= $form_banner->{creative}->{creative_id} if $form_banner->{creative};
            }

            # TODO: Не передевать из шаблонов hash_flags для НЕ РМП
            delete $form_banner->{hash_flags} if !$is_mobile_content;
            # временная очистка пользовательских данных в случае пустых значений в объектах image_ad и creative DIRECT-58615
            if ($form_banner->{creative}) {
                delete $form_banner->{creative} unless $form_banner->{creative}->{creative_id};
            }
            if ($form_banner->{image_ad} && ($form_banner->{ad_type} // '') ne 'mcbanner') {
                delete $form_banner->{image_ad} unless $form_banner->{image_ad}->{hash};
            }
        }

        if ($is_text || $is_mobile_content) {   # TODO DIRECT-67003 подумать про param12
            $form_adgroup->{keywords} //= $form_adgroup->{phrases} if exists $form_adgroup->{phrases}; # TODO: delete
            for my $keyword (@{$form_adgroup->{keywords} // []}) {
                delete @{$keyword}{qw/param1 param2/};
            }
        }

        if ($is_dynamic) {
            for my $dyn_cond (@{$form_adgroup->{dynamic_conditions} // []}) {
                for my $rule (@{$dyn_cond->{condition}}) {
                    next if ($rule->{type} // '') ne 'any';
                    delete $rule->{kind} if exists $rule->{kind} && !$rule->{kind};
                    delete $rule->{value} if exists $rule->{value} && !$rule->{value};
                }
            }
        }

        if ($is_performance) {
           # Добавляем autobudgetPriority по умолчанию (т.к. он не приходит из интерфейса)
           if ($campaign->{strategy}->{name} eq 'autobudget_roi') {
               $_->{autobudgetPriority} = 3 for @{$form_adgroup->{performance_filters}};
           }

           # чтобы понимать, что это явное редактирование, DIRECT-83200
           $form_adgroup->{field_to_use_as_name} //= "";
        }

        if ($is_mobile_content) {
            # Получаем данные по мобильному контенту
            if (!$form_adgroup->{mobile_content}->{mobile_content_id}) {
                MobileContent::ajax_mobile_content_info($form_adgroup->{store_content_href}, $client_id);
            }
        }

        if ($is_mcbanner) {
            $form_adgroup->{keywords} = delete($form_adgroup->{phrases});
            for my $keyword (@{$form_adgroup->{keywords} // []}) {
                delete @{$keyword}{qw/param1 param2 price_context/};
                $keyword->{id} = 0 unless exists $keyword->{id};
            }
        }

        if ($is_cpm_banner || $is_cpm_deals) {
            $form_adgroup->{keywords} = delete($form_adgroup->{phrases});
            for my $keyword (@{$form_adgroup->{keywords} // []}) {
                delete @{$keyword}{qw/param1 param2 autobudgetPriority price/};
                $keyword->{id} = 0 unless exists $keyword->{id};
            }
        }

        # Для легкого редактирования нет минус-слов и гео
        delete @{$form_adgroup}{qw/minus_words geo/} if $is_light;
        if (!$is_light && !$form_adgroup->{geo}){
            my $error = iget('Не заданы регионы показа');
            #DIRECT-68015, временное решение - пока фронтенд не отображает для ДО простые generic'и
            $error = {groups => {array_errors => [{object_errors => {geo => [{type => $error, description => $error}]}}]}} if $is_dynamic;
            return $respond_error->($error, $back_to_campaign_page);
        }

        # Проставляем ставку по-умолчанию для беcфразного таргетинга
        my $is_different_places = $campaign->{strategy}->{name} eq 'different_places' ? 1 : 0;
        my $rm_need_set_price_context = $has_extended_relevance_match
            && ($campaign->{platform} eq 'context'
            || ($campaign->{platform} eq 'both' && $is_different_places));
        foreach my $relevance_match ( @{$form_adgroup->{relevance_match} // []} ) {
            # для существующих автотаргетингов, price уже должен быть установлен, а price_context может быть пересчитан
            next if ($relevance_match->{bid_id} && !$rm_need_set_price_context);
            if (scalar(@{$form_adgroup->{phrases}})) {
                if (!$relevance_match->{price} > 0) {
                    $relevance_match->{price} = Direct::Model::BidRelevanceMatch::Helper::calc_price(
                        [ map { $_->{price} } @{$form_adgroup->{phrases}} ],
                        $campaign->{currency}
                    );
                }

                if ($rm_need_set_price_context){
                    $relevance_match->{price_context} = Direct::Model::BidRelevanceMatch::Helper::calc_average_price(
                        Direct::Model::BidRelevanceMatch::Helper::extract_ctx_prices($form_adgroup->{phrases}),
                        $campaign->{currency},
                        $campaign->{ContextPriceCoef}
                    ) unless $relevance_match->{price_context} > 0;
                }
            }
            else {
                $relevance_match->{price} = get_currency_constant($campaign->{currency}, 'DEFAULT_PRICE');
                $relevance_match->{price_context} = get_currency_constant($campaign->{currency}, 'DEFAULT_PRICE') if $has_extended_relevance_match;
            }
            $relevance_match->{autobudgetPriority} = 3;
        }
    }

    # Проверка валидности структуры $form_adgroups
    {;
        my $pre_validation_result;
        if ($is_text) {
            $pre_validation_result = check_user_data(
                $form_adgroups,
                [1, "ArrayRef[HashRef]", {
                    %DoCmd::Checks::ADGROUP_TEXT,
                    banners => [1, "ArrayRef[HashRef]", \%DoCmd::Checks::BANNER_TEXT],
                    keywords => [1, "ArrayRef[HashRef]", \%DoCmd::Checks::KEYWORD],
                    retargetings => [1, 'ArrayRef[HashRef]'],
                    relevance_match => [0, 'ArrayRef[HashRef]', \%DoCmd::Checks::RELEVANCE_MATCH],

                    ($is_light ? (
                        adgroup_id => "Id",
                        geo => [0, "Str"],
                        minus_words => [0, "ArrayRef[Str]"],
                        tags => [0, $DoCmd::Checks::ADGROUP_TEXT{tags}],
                        keywords => [undef],
                        retargetings => [undef],
                    ) : ()),
                }], {
                    user => $chief_user,
                    campaign => $campaign,
                }
            );
        } elsif ($is_dynamic) {
            $pre_validation_result = check_user_data(
                $form_adgroups,
                [1, "ArrayRef[HashRef]", {
                    %DoCmd::Checks::ADGROUP_DYNAMIC,
                    banners => [1, "ArrayRef[HashRef]", \%DoCmd::Checks::BANNER_DYNAMIC],
                    dynamic_conditions => [sub { $is_light ? undef : 1 }, "ArrayRef[HashRef]", \%DoCmd::Checks::DYNAMIC_CONDITION],

                    ($is_light ? (
                        adgroup_id => "Id",
                        geo => [0, "Str"],
                        minus_words => [0, "Str"],
                        tags => [0, $DoCmd::Checks::ADGROUP_DYNAMIC{tags}],
                        banners => [1, "ArrayRef[HashRef]", {%DoCmd::Checks::BANNER_DYNAMIC, vcard => [undef]}],
                    ) : ()),
                }], {
                    user => $chief_user,
                    campaign => $campaign,
                    login_rights => $login_rights,
                }
            );
        } elsif ($is_mobile_content) {
            $pre_validation_result = check_user_data(
                $form_adgroups,
                [1, "ArrayRef[HashRef]", {
                    %DoCmd::Checks::ADGROUP_MOBILE,
                    banners => [1, "ArrayRef[HashRef]", \%DoCmd::Checks::BANNER_MOBILE],
                    phrases => [1, 'ArrayRef[HashRef]', \%DoCmd::Checks::KEYWORD],
                    relevance_match => [0, 'ArrayRef[HashRef]', \%DoCmd::Checks::RELEVANCE_MATCH],
                    retargetings => [1, 'ArrayRef[HashRef]'],
                    target_interests => [0, 'ArrayRef[HashRef]', \%DoCmd::Checks::TARGET_INTERESTS],

                    ($is_light ? (
                        adgroup_id => "Id",
                        geo => [0, "Str"],
                        minus_words => [0, "ArrayRef[Str]"],
                        tags => [0, $DoCmd::Checks::ADGROUP_MOBILE{tags}],
                        phrases => [undef],
                        retargetings => [undef],
                    ) : ()),
                }], {
                    user => $chief_user,
                    campaign => $campaign,
                }
            );
        } elsif ($is_performance) {
            $pre_validation_result = check_user_data(
                $form_adgroups,
                [1, "ArrayRef[HashRef]", {
                    %DoCmd::Checks::ADGROUP_PERFORMANCE,
                    banners => [1, "ArrayRef[HashRef]", \%DoCmd::Checks::BANNER_PERFORMANCE],
                    performance_filters => [1, "ArrayRef[HashRef]", \%DoCmd::Checks::PERFORMANCE_FILTER],
                }], {
                    user => $chief_user,
                    campaign => $campaign,
                }
            );
        } elsif ($is_mcbanner) {
            $pre_validation_result = check_user_data(
                $form_adgroups,
                [1, "ArrayRef[HashRef]", {
                    %DoCmd::Checks::ADGROUP_MCBANNER,
                    banners => [1, "ArrayRef[HashRef]", \%DoCmd::Checks::BANNER_MCBANNER],
                    keywords => [1, "NonEmptyArrayRef[HashRef]", \%DoCmd::Checks::KEYWORD_WITHOUT_PRICES],
                    retargetings => [undef],
                    relevance_match => [undef],
                    target_interests => [undef],
                    dynamic_conditions => [undef],
                    performance_filters => [undef],
                    phrases => [undef],
                }], {
                    user => $chief_user,
                    campaign => $campaign,
                }
            );
        } elsif ($is_cpm_banner || $is_cpm_deals) {
            $pre_validation_result = check_user_data(
                $form_adgroups,
                [1, "ArrayRef[HashRef]", {
                    %DoCmd::Checks::ADGROUP_CPM_BANNER,
                    banners => [1, "ArrayRef[HashRef]", \%DoCmd::Checks::BANNER_CPM_BANNER],
                    keywords => [0, "ArrayRef[HashRef]", \%DoCmd::Checks::KEYWORD_CPM_BANNER],
                    retargetings => [0, 'ArrayRef[HashRef]'],
                    minus_words => [0, 'ArrayRef[Str]'],
                    relevance_match => [undef],
                    target_interests => [undef],
                    dynamic_conditions => [undef],
                    performance_filters => [undef],
                    phrases => [undef],
                }], {
                    user => $chief_user,
                    campaign => $campaign,
                }
            );
        }

        # Если есть ошибки пред-валидации, то выведем их в STDERR (для диагностики), а клиенту отдадим стандартное сообщение
        if (!$pre_validation_result->is_valid) {
            warn join("\n", @{$pre_validation_result->get_error_descriptions});
            return $respond_error->(iget('Неверные входные данные'), $back_to_campaign_page);
        }
    }

    # Сохранение группы с первого шага
    if ($is_auto_price) {
        my $error = DoCmdAdGroup::Helper::calc_prices_for_save_adgroup($campaign, $form_adgroups, $is_adgroups_copy_action);
        if ($error) {
            return $respond_error->($error);
        }
    }

    # Флаг применения пост-модерации, если объявление редактирует менеджер и у пользователя разрешена пост-модерация
    my $post_moderate = $login_rights->{manager_control} && User::get_one_user_field($c->client_chief_uid, 'statusPostmoderate') eq 'Yes';

    if ($is_adgroups_copy_action) {
        # Заполним данные для копирования CTR
        if (($is_text || $is_mobile_content || $is_mcbanner || $is_cpm_banner || $is_cpm_deals) && any {$login_rights->{role} eq $_} qw/super manager placer agency/) {
            for my $keyword (map { @{$_->{keywords} // []} } @$form_adgroups) {
                if ($login_rights->{role} =~ /^(empty|client)$/) {
                    delete $keyword->{id};
                } else {
                    $keyword->{ctr_source_id} = delete $keyword->{id} if $keyword->{id};
                }
            }
        }

        Models::AdGroup::on_copy_adgroup($_) for @$form_adgroups;
        # Подчищаем после функции выше.
        # Она кривовата и падает если что не так.
        delete $_->{adgroup_id} for @$form_adgroups;
        delete $_->{id} for map { @{$_->{keywords} // []} } @$form_adgroups;
    }

    # Если поставлен новый видимый домен "для всех объявлений в кампании", то сначала обновляем баннеры только на предмет этого домена.
    # Это могут делать только внутренние роли, и предполагается, что делать будут не часто.
    # (!) Только для текстовых кампаний
    if ($FORM{camp_banners_domain} && URLDomain::can_edit_domain($login_rights) && !$is_light && $is_text) {
        my $banners = Direct::Banners::Text->get_by(campaign_id => $cid, filter => {'b.statusArch' => 'No', 'b.href__is_not_null' => 1})->items;
        for (@$banners) {
            $_->domain(smartstrip2($FORM{camp_banners_domain}));
            $_->status_bs_synced('No');
        }
        Direct::Model::BannerText::Manager->new(items => $banners)->update();
    }

    for my $adgroup (@$form_adgroups) {
        $adgroup->{geo} = '' unless $adgroup->{geo};
        $adgroup->{adgroup_type} = $adgroup->{cpm_banners_type} if $adgroup->{cpm_banners_type};
        for my $banner (@{$adgroup->{banners}}) {
            $banner->{domain} //= undef if $banner->{href};
        }
    }

    my $smart = Direct::AdGroups2::Smart->from_user_data(
        $c->client_chief_uid, $cid, $form_adgroups, operator_uid => $UID,
        save_as_draft => $save_as_draft, post_moderate => $post_moderate, replace_show_conditions => !$is_light
    );
    $smart->validate;

    unless ($smart->is_valid) {
        return $need_json_response
            ? $respond_error->($smart->get_errors_response)
            : error(join("\n", @{$smart->errors}, @{$smart->validation_result->get_error_descriptions}), $back_to_campaign_page);
    }

    # Применение изменений в БД
    my $ret = $smart->apply;

    # DIRECT-75232
    if ($FORM{popupMode}) {
        if ($is_new_adgroup && @$ret) {
            my $adgroup_id = $ret->[0]->{adgroup}->id;
            $retpath = "${SCRIPT}?cmd=dnaSaveSuccess&adgroupId=${adgroup_id}";
        } else {
            $retpath = "${SCRIPT}?cmd=dnaSaveSuccess";
        }
    }

    $cvars->{features_enabled_for_client} //= {};
    hash_merge $cvars->{features_enabled_for_client}, Client::ClientFeatures::get_features_enabled_for_client(
        $client_id, [qw/cpc_device_modifiers cpm_video_device_modifiers mobile_content_cpc_video_allowed
        mobile_os_bid_modifier_enabled/]);

    return $need_json_response ? respond_json($r, {result => {location => $retpath}}) : redirect($r, $retpath);
}

=head2 getAdGroup

     Получить группу(с баннерами, но без фраз) по pid
     count - число баннеров возвращаемых за один раз (при отсутствии возвращаются все баннеры)
     page - номер запрашиваемой "страницы" с баннерами

=cut

sub cmd_getAdGroup
    :Cmd(getAdGroup)
    :RequireParam(adgroup_id => 'Pid')
    :Rbac(Code => rbac_cmd_by_owners, AllowReadonlyReps => 1)
    :Description('Получить группы по pid')
    :AllowBlockedClient
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c/};
    my %FORM = %{$_[0]{FORM}};

    my ($items_per_page, $page);
    # items per page
    if (is_valid_id($FORM{count})) {
        $items_per_page = $FORM{count};
        $page = is_valid_id($FORM{page}) ? $FORM{page} : 1;
    }

    my ($groups) = Models::AdGroup::get_groups_gr({
        pid => $FORM{adgroup_id},
        (defined $FORM{arch_banners} && $FORM{arch_banners} =~ /^0|1$/ ? (arch_banners => $FORM{arch_banners}) : ()),
        adgroup_types => get_camp_supported_adgroup_types(cid => get_cid(pid => $FORM{adgroup_id})),
    }, {
        banner_limit => $items_per_page,
        banner_offset => ($page - 1) * $items_per_page,
        banner_order_by_status => 1,
        get_phrases_statuses => 1,
        pass_empty_groups => 1,
    });
    my $group = $groups->[0];
    if ($group) {
        $group->{count} = $group->{total_banners};
        my $camp_data = CampaignQuery->get_campaign_data( cid => $group->{cid}, ['archived', 'status_click_track']);
        Models::Banner::fill_in_can_delete_banner($group->{banners});

        foreach my $banner (@{$group->{banners}}) {
            # TODO: перенести в Model::Banner::compile_banner_params?
            if ($banner->{flags}) {
                $banner->{hash_flags} = BannerFlags::get_banner_flags_as_hash($banner->{flags});
            }
            separate_vcard($banner);
        }
        BannerFlags::get_retargeting_warnings_flags($group, detailed => 1);
        BannersCommon::fill_in_text_before_moderation($group->{banners});

        $group->{is_camp_archived} = $camp_data->{archived} eq 'Yes' ? 1 : 0;
        $group->{campaign} = hash_cut $camp_data, qw/status_click_track/;
        # выкидываем поля, которые не нужны на страницах, где это вызывается.
        delete $group->{$_} foreach qw/phrases retargetings minus_words tags geo/;
    }

    fill_in_groups_rating([$group]);

    if (@{$group->{banners}}) {
        my $permalinks = [uniq map { $_->{permalink} } grep { $_->{permalink} } @{$group->{banners}}];
        $group->{organizations} = Direct::BannersPermalinks::get_organizations_info($permalinks, HttpTools::lang_auto_detect($r));
    }

    $group->{hierarchical_multipliers} = HierarchicalMultipliers::wrap_expression_multipliers(
        $group->{hierarchical_multipliers}
    );


    return respond_json($r, $group || {});
}

sub cmd_setBannersStatuses
    :Cmd(setBannersStatuses)
    :RequireParam(adgroup_id => 'Pid', json_statuses => [struct => 'banner_statuses'])
    :RequireParam(cid => 'CidsMaybe')
    :Rbac(Code => rbac_cmd_by_owners, CampKind => {web_edit_base => 1}, ExceptRole => [media, superreader])
    :CheckCSRF
    :Description('изменение статуса объявлений')
{

    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS/};
    my %FORM = %{$_[0]{FORM}};

    my $statuses = $FORM{json_statuses};

    my (@banners_to_stop, @banners_to_resume);
    foreach my $bid (keys %$statuses) {
        next unless $bid && $bid =~ /^\d+$/;
        if ($statuses->{$bid}->{statusShow} && $statuses->{$bid}->{statusShow} eq 'Yes') {
            push @banners_to_resume, $bid;
        } else {
            push @banners_to_stop, $bid;
        }
    }

    if (@banners_to_resume) {
        Models::AdGroup::resume_groups($FORM{adgroup_id}, {
            uid => $login_rights->{client_chief_uid},
            bid => \@banners_to_resume
        });
    }

    if (@banners_to_stop) {
        Models::AdGroup::stop_groups($FORM{adgroup_id}, {
            uid => $login_rights->{client_chief_uid},
            bid => \@banners_to_stop
        });
    }

    return respond_json($r, {success => scalar(@banners_to_resume) || scalar(@banners_to_stop)});
}


# ---------------------------------------------------------------------------------------------------------------------------
sub cmd_stopBanner
    :Cmd(stopBanner)
    :RequireParam(bid => 'Bids')
    :RequireParam(cid => 'CidsMaybe')
    :Rbac(Code => [rbac_cmd_by_owners, rbac_cmd_user_allow_edit_camps_extended], CampKind => {web_edit_base => 1}, ExceptRole => [media, superreader])
    :CheckCSRF
    :Description('остановка объявления')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS/};
    my %FORM = %{$_[0]{FORM}};

    my $bids = $FORM{bid};
    my $stop_result = Models::AdGroup::stop_groups(get_pids(bid => $bids), {
        uid => $login_rights->{client_chief_uid},
        bid => $bids
    });
    if (defined $FORM{json}) {
        my %processed_bids = map { $_ => 1 } map { @$_ } values $stop_result;
        my ($updated_bids, $skipped_bids) = part { exists $processed_bids{$_} ? 0 : 1} @$bids;
        return respond_json($r, { updated_objects => $updated_bids || [], skipped_objects => $skipped_bids || []});
    } else {
        return redirect($r, $FORM{retpath} || $SCRIPT."?cmd=showCamp&cid=$FORM{cid}[0]$FORM{uid_url}");
    }
}

# ---------------------------------------------------------------------------------------------------------------------------
sub cmd_resumeAdGroup :Cmd(resumeAdGroup)
    :RequireParam(adgroup_ids => 'Pids')
    :RequireParam(cid => 'CidsMaybe')
    :Rbac(Code => [rbac_cmd_by_owners, rbac_cmd_user_allow_edit_camps_extended], CampKind => {web_edit_base => 1}, ExceptRole => [media, superreader])
    :CheckCSRF
    :Description('Запуск ad-group')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS/};
    my %FORM = %{$_[0]{FORM}};

    my $resume_result = Models::AdGroup::resume_groups($FORM{adgroup_ids}, { uid => $login_rights->{client_chief_uid} });

    return respond_to($r,
        json => sub {
            my ($updated_groups, $skipped_groups) = part { exists $resume_result->{$_} ? 0 : 1} @{$FORM{adgroup_ids}};
            return { updated_objects => $updated_groups || [], skipped_objects => $skipped_groups || []};
        },
        any  => sub {
            my $cid = $FORM{cid}[0];
            my $go_to_tab = get_one_field_sql(PPC(cid => $cid),
                "select count(*) from banners where cid = ? and statusShow = 'No' and statusArch = 'No'", $cid) ? '&tab=off' : '';
            return redirect($r, $FORM{retpath} || $SCRIPT."?cmd=showCamp&cid=$cid$FORM{uid_url}$go_to_tab");
        }
    );
}

# ---------------------------------------------------------------------------------------------------------------------------
sub cmd_resumeBanner :Cmd(resumeBanner)
    :RequireParam(bid => 'Bids')
    :RequireParam(cid => 'CidsMaybe')
    :Rbac(Code => [rbac_cmd_by_owners, rbac_cmd_user_allow_edit_camps_extended], CampKind => {web_edit_base => 1}, ExceptRole => [media, superreader])
    :CheckCSRF
    :Description('запуск объявления')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS/};
    my %FORM = %{$_[0]{FORM}};

    my $bids = $FORM{bid};
    my $resume_result = Models::AdGroup::resume_groups(get_pids(bid => $bids), {
        uid => $login_rights->{client_chief_uid},
        bid => $bids
    });

    if (defined $FORM{json}) {
        my %processed_bids = map { $_ => 1 } map { @$_ } values $resume_result;
        my ($updated_bids, $skipped_bids) = part { exists $processed_bids{$_} ? 0 : 1} @$bids;
        return respond_json($r, { updated_objects => $updated_bids || [], skipped_objects => $skipped_bids || []});
    } else {
        my $cid = $FORM{cid}[0];
        my $go_to_tab = get_one_field_sql(PPC(cid => $cid),
            "select count(*) from banners where cid = ? and statusShow = 'No' and statusArch = 'No'", $cid) ? '&tab=off' : '';

        return redirect($r, $FORM{retpath} || $SCRIPT."?cmd=showCamp&cid=$cid$FORM{uid_url}$go_to_tab");
    }
}

# ---------------------------------------------------------------------------------------------------------------------------
sub cmd_stopAdGroup
    :Cmd(stopAdGroup)
    :RequireParam(adgroup_ids => 'Pids')
    :RequireParam(cid => 'CidsMaybe')
    :Rbac(Code => [rbac_cmd_by_owners, rbac_cmd_user_allow_edit_camps_extended], CampKind => {web_edit_base => 1}, ExceptRole => [media, superreader])
    :CheckCSRF
    :Description('Остановка ad-group')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS/};
    my %FORM = %{$_[0]{FORM}};

    my $stop_result = Models::AdGroup::stop_groups($FORM{adgroup_ids}, {uid => $login_rights->{client_chief_uid}});
    return respond_to($r,
        json => sub {
            my ($updated_groups, $skipped_groups) = part { exists $stop_result->{$_} ? 0 : 1} @{$FORM{adgroup_ids}};
            return { updated_objects => $updated_groups || [], skipped_objects => $skipped_groups || []};
        },
        any  => sub {
            return redirect($r, $FORM{retpath} || $SCRIPT."?cmd=showCamp&cid=$FORM{cid}[0]$FORM{uid_url}");
        }
    );
}

=head2 cmd_saveAdGroupTags

    Сохранение меток на группу объявлений

=cut

sub cmd_saveAdGroupTags :Cmd(saveAdGroupTags)
    :RequireParam(cid => 'Cid')
    :Rbac(Code => [rbac_cmd_by_owners, rbac_cmd_user_allow_edit_camps],  ExceptRole => media, CampKind => {web_edit_base => 1})
    :CheckCSRF
    :Description('сохранить метки для группы объявлений')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c/};
    my %FORM = %{$_[0]{FORM}};

    my $tags_allowed = get_user_data($c->client_chief_uid, [qw/tags_allowed/])->{tags_allowed};
    if ($tags_allowed ne 'Yes') {
        return respond_json($r, {errors=>[iget('Невозможно установить метки')]});
    }

    my $tags_ids = get_num_array_by_str($FORM{tags_ids} || "");
    my @new_tags = (defined($FORM{new_tags})) ? split '\s*,\s*', $FORM{new_tags} : ();
    my $save_camp_tags_only = $FORM{save_camp_tags_only} || 0;

    my $pids = get_num_array_by_str($FORM{adgroup_ids} || '');
    $pids = [uniq @$pids];
    my $pids_count = scalar @$pids;

    my @errors = ();
    # Проверяем, что количество установленавливаемых меток не больше разрешенного количества.
    my $new_tags_count = scalar(@new_tags);
    my $error = Tag::check_banner_count_limit(scalar(@{$tags_ids}) + $new_tags_count);
    # Проверяем, что общее количество меток не больше разрешенного количества.
    $error = Tag::check_camp_count_limit(Tag::get_all_campaign_tags_count($FORM{cid}) + $new_tags_count) unless $error || !$new_tags_count;
    push @errors, $error if ($error);

    push @errors, grep {$_} map {Tag::check_tag_text($_)} @new_tags;

    my $tags_groups = {};
    if ($pids_count > 0 && !$save_camp_tags_only && !scalar(@errors)) {
    # При редактировании используется след. логика:
    # Если тег выбран, значит он должен стать выбранным для всех групп.
    # Если тег не выбран, то: a) если были все выбраны до этого, то удалить для всех;
    # б) если НЕ был выбран для всех, тогда ничего не делать с ними.
    # Одна группа - это частный случай мультиредактирования.

        $tags_groups = Tag::get_tags_groups($pids);
        $tags_groups->{$_} = $pids foreach @$tags_ids;

        my $tags_ids_hash = {map {$_ => 1} @{$tags_ids}};
        my @to_delete = grep {
            $pids_count == @{$tags_groups->{$_}}
                && !$tags_ids_hash->{$_}
        } keys %$tags_groups;
        delete @{$tags_groups}{@to_delete} if @to_delete;

        my $error = Tag::validate_adgroup_tags($FORM{cid}, $tags_ids);
        push @errors, $error if $error;
    }

    if (!scalar(@errors)) {
        # Если нет ошибок, сохраняем новые метки и добавляем их идентификаторы в $tag_ids
        my $added_tags = Tag::add_campaign_tags($FORM{cid}, \@new_tags, return_inserted_tag_ids=>1);
        for (@{$added_tags}) {
            $tags_groups->{$_->{tag_id}} = $pids;
        }

        Tag::save_tags_groups($pids, $tags_groups) unless $save_camp_tags_only;

        return respond_json($r, {new_tags_as_hash => {map {$_->{name} => {tag_id => $_->{tag_id}}} @{$added_tags}}});
    } else {
        return respond_json($r, {errors=>\@errors});
    }
}

=head2 cmd_deleteAdGroup

    Удаление групп объявлений полность (баннеры, условия показа, метки и т.д.)

=cut

sub cmd_deleteAdGroup :Cmd(deleteAdGroup)
    :RequireParam(cid => 'Cid', adgroup_ids => 'Pids')
    :Rbac(Code => rbac_cmd_by_owners, ExceptRole => [media, superreader])
    :CheckCSRF
    :Description('удаление групп объявлений')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS/};
    my %FORM = %{$_[0]{FORM}};

    my $err = Models::AdGroup::delete_groups([map { {cid => $FORM{cid}, pid => $_} } @{$FORM{adgroup_ids}}]);
    if (defined $err) {
        warn "delete_groups error: $err";
    }

    return respond_to($r,
                      json => sub {
                          if (!defined $err) {
                              return [{status => 'success'}];
                          } else {
                              return [{status => 'error', msg => $err}];
                          }
                      },
                      any  => sub {
                          return redirect($r, "$SCRIPT?cmd=showCamp&cid=$FORM{cid}$FORM{uid_url}");
                      }
        );
}

=head2 cmd_deleteAdGroups

    Массовое удаление групп объявлений полностью (баннеры, условия показа, метки и т.д.)
    Контроллер предназначен для массового удаления групп объявлений из Нового интерфейса

=cut

sub cmd_deleteAdGroups :Cmd(deleteAdGroups)
    :RequireParam(adgroup_ids => 'Pids')
    :Rbac(Code => rbac_cmd_by_owners, ExceptRole => [media, superreader])
    :CheckCSRF
    :Description('массовое удаление групп объявлений')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS/};
    my %FORM = %{$_[0]{FORM}};

    my $request_adgroup_ids = [uniq @{$FORM{adgroup_ids}}];
    my $limit = $Settings::MAX_AD_GROUPS_PER_UPDATE_REQUEST;
    if (@$request_adgroup_ids > $limit) {
        return respond_json($r, {status => 'error',
            error_message => "Allowed to delete no more than $limit adGroups in a single request"
        });
    }

    my $response;
    eval {
        my $adgroup_id2cid = get_pid2cid(pid => $request_adgroup_ids);
        my $exists_adgroup_ids = [keys %$adgroup_id2cid];

        my $main_banner_ids_by_pid = Primitives::get_main_banner_ids_by_pids($exists_adgroup_ids);
        my $main_banners = Models::Banner::get_banners_for_delete({bid => [values %$main_banner_ids_by_pid]});
        my %adgroup_id_by_bid = reverse %$main_banner_ids_by_pid;

        # Надо оставить только те группы, которые можем удалить целиком, иначе может получиться,
        # что удалили не все объявления: смотри метод Models::Banner::delete_banners
        # Группу можно удалять, если можем удалить все объявления этой группы
        # Для экономии времени проверяем только возможность удаления главного объявления группы.
        # Предполагаем, что если можно удалить главное объявление, то и остальные объявления тоже можно удалять.
        my $skipped_adgroup_ids = [map {$adgroup_id_by_bid{$_->{bid}}} grep { has_delete_banner_problem($_) } @$main_banners];
        my $adgroup_ids_for_delete = xdiff($exists_adgroup_ids, $skipped_adgroup_ids);

        # будет undef или текст про первый запрещенный баннер к удалению: Models::Banner::delete_banners
        # отдаем в ответ для информативности - фронт не использует для отрисовки
        my $error = Models::AdGroup::delete_groups([ map {{ cid => $adgroup_id2cid->{$_}, pid => $_ }} @$adgroup_ids_for_delete ]);

        my $not_deleted_adgroup_ids = get_one_column_sql(PPC(pid => $exists_adgroup_ids),
            ["select pid from phrases", where => {pid => SHARD_IDS}]);
        my $deleted_adgroup_ids = xdiff($exists_adgroup_ids, $not_deleted_adgroup_ids);
        my $request_skipped_adgroup_ids = xdiff($request_adgroup_ids, $deleted_adgroup_ids);

        $response = {
            status => 'success',
            updated_objects => $deleted_adgroup_ids,
            skipped_objects => $request_skipped_adgroup_ids,
        };
        $response->{error_message} = $error if defined $error;
    };
    if ($@) {
        return respond_json($r, {status => 'error'});
    }

    return respond_json($r, $response);
}

=head2 cmd_ajaxValidateDynamicConditions
    Проверка условий нацеливания
=cut
sub cmd_ajaxValidateDynamicConditions :Cmd(ajaxValidateDynamicConditions)
    :Rbac(Code => rbac_cmd_by_owners, ExceptRole => [])
    :Description('Проверка условий нацеливания')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c) = @{$_[0]}{
    qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS c/};
    my %FORM = %{$_[0]{FORM}};
    my $cid = $FORM{cid};
    my $groups = $FORM{json_groups} // [];
    my $client_id = $c->client_client_id;
    my $campaign = defined $cid ? Models::Campaign::get_user_camp_gr($c->client_chief_uid, $cid, {no_groups => 1, without_multipliers => 1}) : undef;
    return respond_json($r, {errors => {generic_errors => [{description => iget('Нет доступа к кампании %s', $cid)}]}}) unless $campaign;
    my $vr = Direct::ValidationResult->new();
    my $smart = Direct::AdGroups2::Smart->new(campaign => $campaign,);
    foreach my $adgroup_data (@$groups){
        my $adgroup = Direct::Model::AdGroupDynamic->new(
            id => $adgroup_data->{id} // 0,
        client_id =>  $client_id,
        campaign_id => $cid,
        defined $adgroup_data->{feed_id} ? (feed_id => $adgroup_data->{feed_id}) : (main_domain => $adgroup_data->{domain} )
            );
        my $dynamic_conditions = $smart->prepare_dyn_conds_from_user_data($adgroup, $adgroup_data->{dynamic_conditions});
        $vr->next()->add( dynamic_condition => Direct::Validation::DynamicConditions::validate_dynamic_conditions_for_adgroup($dynamic_conditions, [], $campaign));
    }
    return respond_json($r,
            $vr->is_valid ?
                  {'status' => 'ok'}
                : {errors => {groups => $vr->convert_vr_for_frontend(groups => { map { $_ => 1 } qw/groups banners performance_filters/ })}}
            );
}


1;
