package DoCmdAdGroup::Helper;

use Direct::Modern;

use Settings;

use List::Util qw/first/;
use List::MoreUtils qw/any pairwise/;

use Lang::Unglue;
use Yandex::HashUtils;
use Yandex::Clone qw/yclone/;
use Yandex::URL;
use Yandex::I18n qw/iget/;
use Yandex::DBTools;

use Campaign::Types;
use Client;
use Common qw/calc_adgroups_prices/;
use Currencies;
use GeoTools;
use Models::AdGroup;
use Models::PhraseTools;
use OrgDetails;
use PhrasePrice qw/phrase_price_context/;
use PhrasePrice qw/validate_phrase_price validate_cpm_price/;
use PhraseText;
use Primitives;
use Retargeting;
use Sitelinks;
use TextTools qw/html2string/;
use URLDomain qw/divide_href_protocol/;
use User;
use VCards;
use Geo qw/set_extended_geo_to_camp/;

use Direct::AdGroups2::MobileContent;
use Direct::CanvasCreatives;
use Direct::ImageFormats;
use Direct::TargetingCategories;
use Direct::VideoAdditions;
use Direct::Model::BidRelevanceMatch::Helper;

=head2 prepare_and_validate_for_save_adgroup

     Для ТГО/РМП/ГО кампании поддерживаем сохранение группы минуя второй шаг (назначение ставок)
     Поскольку паралельно поддерживается и классический путь сохранения группы, то пока мы не можем изменить
     формат данных которые приходят с фронта. Поэтому нам необходимо подготовить данные в вид пригодный 
     для cmd_saveAdGroups и провести их валидацию.
     
     Если все хорошо - возращаем false, если валидация завершилась ошибкой, возращаем $vars с ошибками.

=cut

sub prepare_and_validate_for_save_adgroup {
    my %args = @_;
    my (
        $campaign,
        $form_adgroups,
        $uid,
        $client_id,
        $login_rights,
        $is_new_adgroup,
        $is_adgroups_copy_action,
        $camp_banners_domain
    ) = @args{qw/campaign form_adgroups uid client_id login_rights is_new_adgroup is_adgroups_copy_action camp_banners_domain/};

    my $is_cpm_banner = $campaign->{type} eq 'cpm_banner';
    my $is_cpm_deals = $campaign->{type} eq 'cpm_deals';
    my $min_price = Currencies::get_currency_constant($campaign->{currency}, ($is_cpm_banner || $is_cpm_deals) ? 'MIN_CPM_PRICE' : 'MIN_PRICE');

    my $is_text = $campaign->{type} eq 'text';
    my $is_mobile_content = $campaign->{type} eq 'mobile_content';
    my $is_mcbanner = $campaign->{type} eq 'mcbanner';
    
    my $supported_adgroup_type = get_camp_supported_adgroup_types(cid => $campaign->{cid})->[0];
    foreach my $group (@$form_adgroups) {
        $group->{geo} = '' unless $group->{geo};
        $group->{adgroup_id} = 0 unless $group->{adgroup_id};
        $group->{pid} = $group->{adgroup_id};
        $group->{adgroup_type} = $supported_adgroup_type;
        if ($supported_adgroup_type eq 'mobile_content') {
            foreach my $banner (@{$group->{banners}}) {
                $banner->{adgroup_type} = $supported_adgroup_type;
            }
        }
    }

    my $all_retargeting_conditions = Retargeting::get_retargeting_conditions(ClientID => $client_id, with_campaigns => 1);
    my $target_categories;
    if ($is_mobile_content) {
        $target_categories = Direct::TargetingCategories->get_rmp_interests->items_by;
    }
    my $old_groups_hash = { map {$_->{pid} => $_} @{$campaign->{groups}} };
    my $exists_banners_type = {
        map { $_->{bid} => $_->{banner_type} } $is_new_adgroup ? () : map { @{$_->{banners}} } @{$campaign->{groups}}
    };

    my @mobile_content_ids = map { (ref $_->{mobile_content} eq 'HASH' ? $_->{mobile_content}->{mobile_content_id} : undef) || () } @$form_adgroups;
    my $mobile_contents = Direct::AdGroups2::MobileContent->get_client_mobile_contents($campaign->{ClientID}, \@mobile_content_ids);
    Models::AdGroup::fill_absent_adgroup_fields($form_adgroups, $campaign->{groups}, $mobile_contents);

    $campaign->{groups} = $form_adgroups;

    if ($args{operator_uid}) {
        my $vcards_to_validate = [map { $_->{vcard} } grep { $_->{has_vcard} && $_->{vcard} } 
                                  map { @{ $_->{banners} } } @{ $campaign->{groups} }];
        VCards::set_vcards_validation_result($vcards_to_validate, 
                {operator_uid => $args{operator_uid}, client_id => $client_id, campaign_id => $campaign->{cid}});
    }

    my $has_errors = 0;
    foreach my $group ( @{$campaign->{groups}} ) {
        delete @{$group}{qw/status/} if $is_adgroups_copy_action;

        my $old_group = $old_groups_hash->{$group->{adgroup_id}} || {};
        $group->{cid} = $campaign->{cid};
        if (defined $group->{geo} && $group->{geo} gt '') {
            $group->{geo} = GeoTools::refine_geoid($group->{geo}, undef, { ClientID => $client_id });
        }
        $group->{phrases} = $group->{keywords} if $group->{keywords};
        Phrase::get_phrases_form(
            group                 => $group,
            old_phrases           => yclone($old_group->{phrases}),
            new_phrases           => $group->{phrases},
            is_groups_copy_action => $is_adgroups_copy_action,
        );

        my @creative_ids = map {
            $_->{ad_type} eq 'image_ad' && $_->{creative}
                ? $_->{creative}->{creative_id}
                : ()
        } @{$group->{banners}};
        my %creative_text = map {
            my $creative = $_->to_template_hash;
            ($creative->{creative_id} => $creative->{creative_text})
        } @{Direct::CanvasCreatives->get_by(creative_id => \@creative_ids, $uid)->items};

        # Проверяем банер
        my $template_fields = BannerTemplates::get_banners_template_fields($group->{banners}, $group->{phrases});
        my @b_minus_geo;
        foreach my $banner (@{$group->{banners}}) {
            my $bid = $banner->{bid};

            if ($bid == 0) {
                # обновляем group_banners_types с учетом нового баннера
                my $type_count = first {$_->{banner_type} eq $banner->{ad_type}} @{$group->{group_banners_types}};
                unless ($type_count) {
                    $type_count = {
                        banner_type => $banner->{ad_type},
                        count       => 0,
                    };
                    push @{$group->{group_banners_types}}, $type_count;
                }
                $type_count->{count}++;
            }

            if ($banner->{minus_geo}) {
                push @b_minus_geo, @{$banner->{minus_geo}};
            }

            # Проставляем шаблонные поля если есть необходимость.
            hash_merge $banner, $template_fields->{$bid};
            $banner->{url_protocol} = clear_protocol($banner->{url_protocol});
            $banner->{href} = URLDomain::clear_banner_href($banner->{href}, $banner->{url_protocol});

            # для валидации
            $banner->{banner_minus_words} = $group->{minus_words};

            if ($banner->{ad_type} eq 'image_ad' && $banner->{creative}) {
                $banner->{creative}->{creative_text} = $creative_text{$banner->{creative}->{creative_id}};
            }
        }
        if (@b_minus_geo) {
            my ($new_geo, $excl) = GeoTools::exclude_region($group->{geo}, \@b_minus_geo, { ClientID => $client_id });
            $group->{effective_geo} = $new_geo;
            $group->{minus_geo} = join ',', @$excl;
        }

        hash_copy $group, $campaign, qw(campaign_minus_words);

        my $old_relevance_matches = { map {; $_->{bid_id} => $_} @{$old_group->{relevance_match}} };
        foreach my $relevance_match (@{$group->{relevance_match} // []}) {
            if ($relevance_match->{bid_id}) {
                $relevance_match = $old_relevance_matches->{$relevance_match->{bid_id}} if $old_relevance_matches->{$relevance_match->{bid_id}};
            } else {
                if ($is_adgroups_copy_action) {
                    $relevance_match->{is_suspended} = 0;
                } else {
                    # для новых условий с фронта приходит пустая строка, если у нас $is_auto_price == 1, то
                    # выкидываем price из хеша, чтобы проверка в check_user_data прошла
                    delete $relevance_match->{price} unless $relevance_match->{price};
                }
            }
        }

        # обрабатываем пришедшие условия ретаргетинга
        my $retargetings_error = Retargeting::get_retargetings_form(
            group                      => $group,
            old_retargetings           => $old_group->{retargetings},
            new_retargetings           => $group->{retargetings},
            all_retargeting_conditions => $all_retargeting_conditions,
            geo                        => $group->{geo},
            MIN_PRICE                  => $min_price,
            is_groups_copy_action      => $is_adgroups_copy_action,
        );

        my $target_interests_error;
        if ($is_mobile_content) {
            $target_interests_error = Retargeting::get_target_interests_form(
                $group, $target_categories, $old_group->{target_interests}, $min_price, $is_adgroups_copy_action,
            );
        }

        $group->{group_name} = $old_group->{group_name} unless ($group->{group_name});
        $group->{currency} = $campaign->{currency};

        next if $is_mcbanner || $is_cpm_banner || $is_cpm_deals; # validate_group знает только про ТГО м РМП, валидация mcbanner, cpm_banner и cpm_deals в AdGroups2::Smart

        # validate_group не только проверяет переданную структуру, но и модифицирует еще in-place
        # изменения несовместимы с основным кодом cmd_saveAdGroups, поэтому в проверку отдаем копию объекта
        my $group_copy = yclone($group);
        foreach my $banner (@{$group_copy->{banners}}) {
            #Быстрые ссылки
            $banner->{sitelinks} = Sitelinks::clear_sitelinks_href($banner->{sitelinks});
            # удаление url_protocol
            $banner->{sitelinks} = Sitelinks::get_clean_sitelinks_list($banner->{sitelinks});
        }
        my $group_errors = validate_group($group_copy,
            {
                use_banner_with_flags     => 1,
                ClientID                  => $client_id,
                login_rights              => $login_rights,
                check_href_available      => 1,
                exists_banners_type       => $exists_banners_type,
                skip_price_check          => 1,
                exists_ret_conds          => $all_retargeting_conditions,
                exists_group_retargetings => { map { $_->{ret_id} => 1 } @{$old_group->{retargetings}} },
            }
        ) || {};

        if ($retargetings_error) {
            $group_errors->{retargetings} ||= [];
            push @{$group_errors->{retargetings}}, $retargetings_error;
        }

        if ($target_interests_error) {
            push @{$group_errors->{target_interests} //= []}, $target_interests_error;
        }

        if (%$group_errors) {
            $group = $group_copy;
            $group->{errors} = $group_errors;
            $has_errors = 1;
        } elsif($is_text) {
            # validate_group подписывает домен (если он был измененен и пользователь имел на это правл), поэтому
            # необходимо скопировать подпись в изначальную структуру
            pairwise {
                my ($banner, $copy_banner) = ($a, $b);
                hash_copy($banner, $copy_banner, qw/domain domain_sign domain_redir domain_redir_sign/);
            } @{$group->{banners}}, @{$group_copy->{banners}};
        }
    }

    # Выводим ошибку, если надо
    if ($has_errors) {
        my @banner_video_additions = ();
        foreach my $adgroup (@{$campaign->{groups}}) {
            for my $banner (@{$adgroup->{banners}}){
                # удаляем ненужные экземпляры визиточных значений: если показывать страницу с ошибкой -- должна остаться только $banner->{vcard}
                delete $banner->{$_} for @$VCARD_FIELDS_FORM;
                OrgDetails::separate_org_details($banner->{vcard});
                $banner->{url_protocol} = get_protocol($banner->{href});
                Sitelinks::divide_sitelinks_href($banner->{sitelinks});
                if ($campaign->{type} eq 'text' && $banner->{video_resources}
                    && $banner->{video_resources}->{resource_type} // '' eq 'creative') {
                    push @banner_video_additions, $banner->{video_resources};
                }
            }
        }

        Models::AdGroup::update_phrases_shows_forecast($campaign->{groups});
        Models::AdGroup::get_groups_auction($campaign->{groups}, {});

        $campaign->{OPTIONS} = get_camp_options($campaign->{cid});
        $campaign->{images} = BannerImages::get_campaign_images($campaign->{cid});
        $campaign = set_extended_geo_to_camp($campaign, $client_id);

        if ( $is_mobile_content && $campaign->{mobile_app_id} ) {
            my $intapi_endpoint = JavaIntapi::GetSingleMobileApp->new(
                client_id => $client_id,
                mobile_app_id => $campaign->{mobile_app_id},
            );

            $campaign->{selected_mobile_app} = $intapi_endpoint->call;
        }

        my $vars = {has_errors => $has_errors,
            campaign => $campaign,
            is_groups_copy_action => $is_adgroups_copy_action,
            all_retargeting_conditions => $all_retargeting_conditions,
            tags_allowed => get_user_data($uid, [qw/tags_allowed/]),
            is_audience_enabled => camp_kind_in(cid => $campaign->{cid}, 'allow_audience') ? 1 : 0,
            camp_banners_domain => $camp_banners_domain,
        };

        # Подгрузим предыдущий баннер кампании, чтобы работала ссылка "скопировать из предыдущего объявления"
        $vars->{prev_banner} = get_last_banner_info(cid => $campaign->{cid}, structured_res => 1, uid => $uid) if $is_new_adgroup;

        $vars->{interest_categories} = Direct::TargetingCategories->build_category_tree('rmp_interest') if $is_mobile_content;

        my $client_options = get_client_data($client_id, [qw/auto_video/]);
        $vars->{auto_video} = $client_options->{auto_video} ? 1 : 0;

        if ($campaign->{type} eq 'text' && $vars->{auto_video}) {
            #генерируем одно нейтральное (без категорий) видео дополнение
            my $generated_video_additions = [];
            my $generate_count = 1;
            my $locale = Yandex::I18n::get_locale(Yandex::I18n::current_lang());
            if ($is_new_adgroup || scalar @banner_video_additions == 0) {
                my $va_conditions = {
                    banner_type => $campaign->{type},
                    category_ids => [],
                    count => $generate_count,
                    locale => $locale,
                };
                $generated_video_additions = eval {
                    Direct::VideoAdditions->generate_video_additions($client_id, [$va_conditions]);
                } || [ ];
                print STDERR "error: $@" if ($@);
                $generated_video_additions = [map { $_->to_template_hash }
                    @{Direct::VideoAdditions->get_video_additions_from_bs($client_id, $generated_video_additions)}];
            } else {
                $generated_video_additions = [ splice @banner_video_additions, 0, $generate_count ];
            }

            $vars->{generated_video_creatives} = [ map { hash_cut $_, qw/id name live_preview_url resource_type/ }
                @{$generated_video_additions} ];
        }

        $vars->{cmd} = $is_text ? 'showCampMultiEdit' : 'editAdGroupsMobileContent';
        return $vars;
    }

    my $cumulative_ctr = {};
    if ($is_adgroups_copy_action && any {$login_rights->{role} eq $_} qw/super manager placer agency/) {
        my @ph_ids = map { map { $_->{id} } @{$_->{phrases}} } @{$campaign->{groups}};
        my $history = BS::History::get_keywords_with_history({ph_ids => \@ph_ids, cid => $campaign->{cid}});
        $cumulative_ctr = { map { $_->{id} => $_ } @$history };
    }

    # Теперь находим цены
    foreach my $group ( @{$campaign->{groups}} ) {
        my $old_group = $old_groups_hash->{$group->{adgroup_id}} || {};
        my $old_group_phrases_hash = {map {$_->{id} => $_} grep {$_->{id}} @{$old_group->{phrases}}};
        for my $ph (@{$group->{phrases}}) {
            my $old_ph = $old_group_phrases_hash && $ph->{id} && $old_group_phrases_hash->{$ph->{id}};
            $ph->{phrase_old} = $old_ph->{phrase}  if $old_ph;

            if (
                phrase_should_null_phrase_id($ph) ||
                    ($is_adgroups_copy_action &&
                        ($old_ph && $ph->{norm_phrase} ne $old_ph->{norm_phrase} || $login_rights->{role} =~ /^(empty|client)$/)
                    )
            ) {
                delete @$ph{qw/id PhraseID PriorityID shows Shows clicks Clicks ctr Ctr/};
            } elsif ($is_adgroups_copy_action && $cumulative_ctr->{$ph->{id}}) {
                $ph->{phraseIdHistory} = $cumulative_ctr->{$ph->{id}}->{phraseIdHistory}
            }
            ensure_phrase_have_props($ph); # нужно для unglue_phrases
        }
        my $unglued_phrases_count = unglue_phrases([$group]);
        if ($unglued_phrases_count) {
            for my $ph (grep { $_->{phrase_unglued_suffix} } @{$group->{phrases}}) {
                $ph->{phrase} .= $ph->{phrase_unglued_suffix};
            }
        }
    }

    unless ($is_mcbanner || $is_cpm_banner || $is_cpm_deals) { # для ГО нет смысла ходить в торги, ставка либо дефолтная, либо единая
        my $context_coef = $campaign->{strategy}->{name} ne 'different_places' && $campaign->{ContextPriceCoef};
        calc_adgroups_prices($campaign,
            not_use_common_stat => $is_adgroups_copy_action || $is_new_adgroup,
            ContextPriceCoef => $context_coef,
        );
    }

    return '';
}

=head2 calc_prices_for_save_adgroup

     Только ТГО/РМП/ГО кампании.
     Поскольку второго шага при сохранении группы нет, то нет и ставок, проставлям здесь ставки для 
     всех условий показа в группах.
     
     Если все хорошо - возращаем fasle, если валидация завершилась ошибкой, возращаем $vars с ошибками.

=cut

sub calc_prices_for_save_adgroup {

    my ($campaign, $form_adgroups, $is_adgroups_copy_action) = @_;
    
    my @form_adgroups_auto_price = grep { $_->{auto_price} } @$form_adgroups;
    return unless @form_adgroups_auto_price;

    my $is_mcbanner = $campaign->{type} eq 'mcbanner';
    my $is_cpm_banner = $campaign->{type} eq 'cpm_banner';
    my $is_cpm_deals = $campaign->{type} eq 'cpm_deals';
    my $is_cpm_yndx_frontpage = $campaign->{type} eq 'cpm_yndx_frontpage';

    my $min_price;
    if ($is_cpm_yndx_frontpage) {
        $min_price = Currencies::get_currency_constant($campaign->{currency}, 'MIN_CPM_FRONTPAGE_PRICE');
    } else {
        $min_price = Currencies::get_currency_constant($campaign->{currency}, ($is_cpm_banner || $is_cpm_deals) ? 'MIN_CPM_PRICE' : 'MIN_PRICE');
    }
    my $default_price = Currencies::get_currency_constant($campaign->{currency}, 'DEFAULT_PRICE');
    my $has_extended_relevance_match = Campaign::has_context_relevance_match_feature($campaign->{type}, $campaign->{ClientID});

    for my $form_adgroup (@form_adgroups_auto_price) {
        if ($campaign->{strategy} && $campaign->{strategy}->{is_autobudget}) {
            foreach my $condition (qw/keywords relevance_match retargetings target_interests/) {
                next unless $form_adgroup->{$condition};
                foreach (@{$form_adgroup->{$condition}}) {
                    $_->{autobudgetPriority} //= 3;
                }
            }
        } else {
            if (!$form_adgroup->{auto_price}->{auto}) {
                my ($single_price, $single_price_context);
                if ($is_cpm_banner || $is_cpm_deals) {
                    $single_price = 0;
                    $single_price_context = $form_adgroup->{auto_price}->{single_price};
                    if (defined $single_price_context) {
                        my $error = validate_cpm_price($single_price_context, $campaign->{currency}, dont_support_comma => 1);
                        return $error if $error;
                    }
                } else {
                    $single_price = $form_adgroup->{auto_price}->{single_price};
                    if (my $error = validate_phrase_price($single_price, $campaign->{currency}, dont_support_comma => 1)) {
                        return $error;
                    }

                    if ($campaign->{strategy}->{name} ne 'different_places') {
                        # необходимо для того, чтобы в логах ppclog_price попала ставка на сеть с коэффициентом
                        if ($campaign->{ContextPriceCoef}) {
                            $single_price_context = phrase_price_context($single_price, $campaign->{ContextPriceCoef}, $campaign->{currency});
                        } else {
                            $single_price_context = $min_price;
                        }
                    } else {
                        $single_price_context = $single_price;
                    }
                }

                # если есть единая ставка, то ставим ее в:
                foreach my $keyword (
                    grep {
                        $is_adgroups_copy_action # при копировании группы, игнорируя существующие ставки
                        || !$_->{id} # для новых фраз
                        || $_->{phrase} ne $_->{phrase_old} # при малейшем изменении фразы (см DIRECT-70148)
                    } @{$form_adgroup->{keywords} // []}
                ) {
                    return iget('Не задана цена за тыс. показов') if ($is_cpm_banner || $is_cpm_deals) && !defined $single_price_context;
                    $keyword->{price} = $single_price;
                    $keyword->{price_context} = $single_price_context;
                }
                # Проставляем ставку по-умолчанию для беcфразного таргетинга
                foreach my $relevance_match ( @{$form_adgroup->{relevance_match} // []} ) {
                    # не выставляем единую ставку для существующих БТ (bid_id!=0) (кроме копирования группы, при копировании заменяем на единую)
                    next if $relevance_match->{bid_id};
                    $relevance_match->{price} = $single_price;
                    $relevance_match->{price_context} = $single_price_context;
                }
                foreach my $condition (qw/retargetings target_interests/) {
                    next unless $form_adgroup->{$condition};
                    foreach (grep { $is_adgroups_copy_action || !$_->{ret_id} } @{$form_adgroup->{$condition}}) {
                        return iget('Не задана цена за тыс. показов') if ($is_cpm_banner || $is_cpm_deals) && !defined $single_price_context;
                        $_->{price_context} = ($is_cpm_banner || $is_cpm_deals) ? $single_price_context : $single_price;
                    }
                }
            } else {
                # Для ГО-кампаний ставим дефолтную ставку на фразы без ставки
                # (т.е. при значительном измении текста фразы, когда id сброситься - ставки сохраниться DIRECT-70237)
                if ($is_mcbanner) {
                    foreach my $keyword (grep { !$_->{price} } @{$form_adgroup->{keywords} // []}) {
                        $keyword->{price} = $default_price
                    }
                }
                foreach my $condition (qw/retargetings target_interests/) {
                    next unless $form_adgroup->{$condition};
                    foreach (@{$form_adgroup->{$condition}}) {
                        $_->{price_context} //= $min_price;
                    }
                }

                foreach my $group ( @{$campaign->{groups}} ) {
                    foreach my $relevance_match ( @{$group->{relevance_match} // []} ) {
                        # не расчитываем ставку для существующих БТ (bid_id!=0) (кроме копирования группы, в этом случае все равно пересчитываем)
                        next if $relevance_match->{bid_id};
                        if (scalar(@{$group->{keywords}})) {
                            if ($campaign->{platform} ne 'context') {
                                $relevance_match->{price} = Direct::Model::BidRelevanceMatch::Helper::calc_price(
                                    [ map { $_->{price} } @{$group->{keywords}} ], $campaign->{currency}
                                )
                            }
                            if ($has_extended_relevance_match
                                && ($campaign->{platform} eq 'context'
                                || $campaign->{platform} eq 'both' && $campaign->{strategy}->{name} eq 'different_places')
                            ) {
                                $relevance_match->{price_context} = Direct::Model::BidRelevanceMatch::Helper::calc_average_price(
                                    Direct::Model::BidRelevanceMatch::Helper::extract_ctx_prices($form_adgroup->{phrases}),
                                    $campaign->{currency},
                                    $campaign->{ContextPriceCoef}
                                );
                            } 
                        }
                        else {
                            $relevance_match->{price} = $default_price;
                            $relevance_match->{price_context} = $default_price if $has_extended_relevance_match;
                        }
                        
                        $relevance_match->{autobudgetPriority} = 3;
                    }
                }
            }
        }
    }

    return '';
}

=head2 get_last_banner_info

    Возвращает хеш с информацией об объявлении, которое последним редактировалось в указанной кампании

    Какое-то время существовала параллельно с, а потом полностью заменила get_last_camp_contactinfo

    Параметры именованные
        cid              обязательно, номер кампании, из которой выбрать объявление
        uid              обязательно, если в кампании есть креативы
        except_bid       опционально, если указан -- это объявление будет пропущено
        with_phone_only  опционально, флаг, если взведен -- будут выбираться только объявления с контактной информацией
        structured_res   опционально, флаг, если взведен -- будет возвращен структурированный хеш ( {body => '...', title => '...', vcard => {...}, ... } ) 
                                            иначе (по умолчанию) -- плоский хеш ( {body => '...', title => '...', phone => '...', street => '...', ... } ) 
                                            со временем хорошо бы сделать везде обработку именно структурированных представлений объявления

    Типичное использование: 
        my $prev_banner = get_last_banner_info( $dbh, cid => $cid);

=cut 

sub get_last_banner_info {
    my (%OPT) = @_;
    return {} unless $OPT{cid};

    my $sql = sprintf <<EOS, join ',', map {"vc.$_"} @$VCARD_FIELDS;
        SELECT
            b.bid, b.body, b.title, b.title_extension,
            b.href, b.domain, b.sitelinks_set_id,
            if(b.banner_type = 'image_ad', 'image_ad', 'text') as ad_type,
            im.image_id, bp.creative_id,
            %s
        FROM
            phrases g
            JOIN banners b ON g.pid = b.pid
            LEFT JOIN vcards vc ON b.vcard_id = vc.vcard_id
            LEFT JOIN images im ON b.bid = im.bid
            LEFT JOIN banners_performance bp ON b.bid = bp.bid
EOS
    
    my $info = get_one_line_sql(PPC(cid => $OPT{cid}), [$sql, WHERE => {
        'g.cid' => $OPT{cid},
        'g.adgroup_type' => 'base',
        ($OPT{except_bid}
            ? ('b.bid__ne' => $OPT{except_bid})
            : ()),
        ($OPT{with_phone_only}
            ? ('vc.phone__is_not_null' => 1)
            : ()),
        'b.statusArch' => 'No',
    }, 'ORDER BY b.LastChange DESC LIMIT 1']);
    if (my $sitelinks_set_id = $info->{sitelinks_set_id}) {
        $info->{sitelinks} = Sitelinks::get_sitelinks_by_set_id($sitelinks_set_id);
        Sitelinks::divide_sitelinks_href($info->{sitelinks});
    }

    html2string($info->{body});
    html2string($info->{title});
    html2string($info->{title_extension});
    hash_merge $info, divide_href_protocol($info->{href});

    if ( $info->{bid} ) {
        hash_merge($info, parse_phone($info->{phone})) if $info->{phone};
        $info->{worktimes} = get_worktimes_array( $info->{worktime} ) if defined $info->{worktime};

        if ($info->{ad_type} eq 'image_ad') {
            $info->{image_ad} = Direct::ImageFormats->get_by(banner_id => $info->{bid})->items->[0]->to_template_hash if $info->{image_id};
            $info->{creative} = Direct::Creatives::get_creatives_with_type([qw/canvas html5_creative/], $OPT{uid}, [$info->{creative_id}])->[0]->to_template_hash if $info->{creative_id};
        }
    
        if ( $OPT{structured_res} ){ 
            # хорошо ли здесь выставлять тип объявления? А где лучше?
            $info->{with_ci} = $info->{phone} ? 1 : 0;
            $info->{with_href} = $info->{href} ? 1 : 0;

            if ($info->{with_ci}) {
                $info->{vcard} = hash_cut $info, @$VCARD_FIELDS_FORM, 'worktimes'; 
                delete $info->{$_} for @$VCARD_FIELDS_FORM, 'worktimes'; 
            }
        }

        $info->{if_banner_found}=1; # аккуратно избавиться
        return $info;
    }

    return {};
}

1;
