package DoCmdApiCertification;

#  $Id$

=head1 NAME

    DoCmdApiCertification

=head1 DESCRIPTION

    контроллеры для работы с API сертификатами

=cut

use warnings;
use strict;
use utf8;

use MIME::Types;
use List::Util qw/first/;
use Storable qw/dclone/;
use Path::Tiny;
use base qw/DoCmd::Base/;
use Carp;

use Yandex::HashUtils;
use Yandex::DateTime;
use Yandex::I18n;

use Settings;
use User;
use Yandex::DBTools;
use Yandex::DBShards;
use Direct::ResponseHelper;
use Direct::PredefineVars;
use APICommon qw/:subs/;
use PrimitivesIds;
use Template::Plugin::DocsL10n;
use LogTools;

use API::App::Access;
use API::App::AccessRegistry;
use API::App::Request::Test;
use API::App::Request::Normal;
use API::App::Request;
use API::App::Request::CSV;
use API::App::Request::Tools qw/
    get_request_info
    get_request_info_by_cr_id
    send_mail_to_support
    prepare_request_object
    prepare_response_object
    mix_ulogin
/;
use API::App::Request::MassMethods qw/   
    app_requests_count
    app_requests_get_by_user
    app_requests_get_by_date
    app_requests_search
/;
use API::App::Request::History qw/
    app_request_history_get
    app_request_history_get_last/;

=head2 cmd_apiCertificationRequestList

    Cписок заявок пользователя на сертификацию приложений API 

=cut

sub cmd_apiCertificationRequestList:Cmd(apiCertificationRequestList) 
    :Rbac(Code => [rbac_cmd_by_owners, rbac_can_edit_api_settings], AllowForLimitedClients => 1)
    :Description('список заявок на сертификацию приложений API')
{
    my ($r, $SCRIPT, $template, $c, $cvars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   c vars/};
    my %FORM = %{$_[0]{FORM}};
    
    my $vars = get_user_api_params($c->{rbac}, $c->{uid}, $c->{UID});
    my $results = app_requests_get_by_user($c->{uid});
    $vars->{requests} = $results;

    $vars->{enable_cpm_deals_campaigns} = Direct::PredefineVars::_enable_cpm_deals_campaigns($c);
    if ($vars->{enable_cpm_deals_campaigns}){
        $vars->{new_deals_count} = Client::get_count_received_deals($c->login_rights->{ClientID});
    }
    $vars->{enable_content_promotion_video_campaigns} = Direct::PredefineVars::_enable_content_promotion_video_campaigns($c);
    $vars->{enable_cpm_yndx_frontpage_campaigns} = Direct::PredefineVars::_enable_cpm_yndx_frontpage_campaigns($c);

    $vars->{features_enabled_for_operator_all} = $cvars->{features_enabled_for_operator_all};
    $vars->{features_enabled_for_client_all} = $cvars->{features_enabled_for_client_all};

    return respond_bem($r, $c->reqid, $vars, source => 'data3');
}
    
=head2 cmd_ajaxApiCertificationSetStatus

    Изменить статус заявки

=cut

sub cmd_ajaxApiCertificationSetStatus:Cmd(ajaxApiCertificationSetStatus) 
    :Rbac(Role => [super, support])
    :CheckCSRF
    :Description('изменить статус заявки')
{
    my ($r, $SCRIPT, $template, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   c/};
    my %FORM = %{$_[0]{FORM}};
    
    my $application_id = $FORM{application_id} or croak('Need application_id');
    my $request = get_request_info($c->{uid}, $application_id);
    if ($request) {
        my $cert_req = API::App::Request->new(%$request);
        $cert_req->status(ucfirst($FORM{status}));
        my $manager_login = get_login(uid => $c->{UID});
        $cert_req->manager_login($manager_login);
        $cert_req->update_for_super();
        _set_app_status($cert_req, $FORM{status});
        return respond_json($r, {result => 1});
    }
    return respond_json($r, {result => 0});
}


sub cmd_ajaxApiAccessSetMinVersion :Cmd(ajaxApiAccessSetMinVersion)
    :Rbac(Role => [super, support])
    :CheckCSRF
    :Description('установить способ поддержки приложения в старых версиях API')
{
    my ($r, $SCRIPT, $template, $FORM) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   FORM/};

    my $application_id = $FORM->{application_id} or croak('Need application_id');

    my $accesses = API::App::AccessRegistry->new([$application_id]);
    my $access = $accesses->get($application_id) or croak("No access found for $application_id");

    my $min_api_version_override = $FORM->{min_api_version_override};
    croak('Need min_api_version_override') unless defined $min_api_version_override;
    unless ($min_api_version_override eq '0' || $min_api_version_override eq '4' || $min_api_version_override eq '104') {
        croak("Invalid min_api_version_override: $min_api_version_override");
    }

    $access->set_min_api_version_override($min_api_version_override);
    $access->resend_app_access_to_sandbox();

    return respond_json($r, { result => 0 });
}


sub _set_app_status($$) {
    my ($cert_req, $status) = @_;
    my $last_version = app_request_history_get_last($cert_req->uid, $cert_req->application_id);
    my $access = API::App::Access->new(
        application_id => $cert_req->application_id,
        manager_login => $cert_req->manager_login,
    );
    if ($last_version) {
        if ($last_version->{access_type} eq 'test'
            && $cert_req->access_type eq 'normal'
            && $status eq 'approve'
        ) {
            $access->add('normal');
        }
    } else {
        if ($status eq 'approve') {
            $access->add($cert_req->access_type);
        } elsif ($status eq 'reject') {
            $access->delete();
        }
    }
}

# получить тект оферты для отрисовки в шаблоне
sub _get_offer_text {
    my $doc_rel_path = Template::Plugin::DocsL10n::get_doc_path('api_developer_offer.html', Yandex::I18n::current_lang());
    my $doc_abs_path = $Template::Plugin::DocsL10n::DOC_ROOT . '/' . $doc_rel_path;

    return path($doc_abs_path)->slurp_utf8;
}    

=head2 cmd_apiCertificationGetHistory

    Возвращает историю измененний заявки
    Принимает: application_id и
               create_time -время создания версии

=cut

sub cmd_apiCertificationGetHistory:Cmd(apiCertificationGetHistory) 
    :Rbac(Role => [super, support, superreader])
    :Description('изменить статус заявки')
{
    my ($r, $SCRIPT, $template, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   c/};
    my %FORM = %{$_[0]{FORM}};
    my ($vars);
    
    my $application_id = $FORM{application_id} or croak('Need application_id');
    my $history = app_request_history_get($c->{uid}, $application_id);
    if (scalar @$history) {
        my $version = dclone first { $_->{create_time} eq $FORM{create_time} } @$history;
    
        my $user_data = get_user_data($c->{uid}, [qw/login email/]);
        hash_merge($version, $user_data);

        $version->{api_logins_list} = prepare_logins_list($version->{api_logins_list}, $SCRIPT)
            if $version->{api_logins_list};
        $version->{login} = prepare_logins_list($version->{login}, $SCRIPT)->[0];
        $version->{history} = $history;
        $vars->{request} = $version;
    } else {
        $vars->{request} = {};
    }
    return respond_bem($r, $c->reqid, $vars, source => 'data3');
}
    
=head2 cmd_ajaxApiCertificationAddComment

    Добавить комментарий

=cut

sub cmd_ajaxApiCertificationAddComment:Cmd(ajaxApiCertificationAddComment) 
    :Rbac(Role => [super, support])
    :CheckCSRF
    :Description('добавить комментарий к заявке')
{
    my ($r, $SCRIPT, $template, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   c/};
    my %FORM = %{$_[0]{FORM}};

    my $application_id = $FORM{application_id} or croak('Need application_id');
    my $request = get_request_info($c->{uid}, $application_id);
    if ($request) {
        my $cert_req = API::App::Request->new(%$request);
        if ($FORM{comment}) {
            $cert_req->comment($FORM{comment});
            $cert_req->update_for_super();
            return respond_json($r, {result => 1});
        }
    }
    return respond_json($r, {result => 0});
}

=head2 cmd_ajaxApiCertificationValidateId

    Проверить валидность application_id

=cut

sub cmd_ajaxApiCertificationValidateId:Cmd(ajaxApiCertificationValidateId) 
    :Rbac(Code => [rbac_cmd_by_owners])
    :Description('добавить комментарий к заявке')
{
    my ($r, $SCRIPT, $template, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   c/};
    my %FORM = %{$_[0]{FORM}};
    my $application_id = $FORM{application_id} or croak('Need application_id');
    my $result = validate_app_id($application_id, $c->{uid});

    return respond_json($r, $result);

}
    
=head2 cmd_ajaxApiCertificationDeleteFile

    Удалить один из файлов приложенных к заявке 

=cut

sub cmd_ajaxApiCertificationDeleteFile:Cmd(ajaxApiCertificationDeleteFile) 
    :Rbac(Code => [rbac_cmd_by_owners])
    :CheckCSRF
    :Description('удалить файл')
{
    my ($r, $SCRIPT, $template, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   c/};
    my %FORM = %{$_[0]{FORM}};
    
    my $application_id = $FORM{application_id} or croak('Need application_id');
    my $name = $FORM{name} or croak('Need deleted file name');
    
    my $request = get_request_info($c->{uid}, $application_id);
    if ($request) {
        my $cert_req = API::App::Request::Normal->new(%$request);
        $cert_req->specifications_and_screenshots->delete($name);
        $cert_req->update();
        return respond_json($r, {result => 1});
    }
    return respond_json($r, {result => 0});
}

=head2 cmd_apiCertificationAddRequest

    Добавление заявки на сертификацию приложений API

=cut

sub cmd_apiCertificationAddRequest:Cmd(apiCertificationAddRequest) 
    :Rbac(Code => [rbac_cmd_by_owners])
    :CheckCSRF
    :Description('добавление заявки на сертификацию приложений API')
{
    my ($r, $SCRIPT, $template, $c, $cvars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   c vars/};
    my %FORM = %{$_[0]{FORM}};
    my $request;
    
    if ($FORM{save}) {
        if ($FORM{accept}) {
            $FORM{uid} = $c->{uid};
            $request = prepare_request_object(\%FORM);
            my $cert_req;
            if ($FORM{access_type} eq 'test') {
                $cert_req = API::App::Request::Test->new(%$request);
            } elsif ($FORM{access_type} eq 'normal') {
                $cert_req = API::App::Request::Normal->new(%$request);
            }
            $cert_req->add();

            my $saved_request = prepare_response_object($cert_req->to_hash());
            send_mail_to_support($saved_request, $c->{uid}, $SCRIPT);
            return redirect($r,"$SCRIPT?cmd=apiCertificationRequestList");
        } else {
            croak('Oferta not accepted');
        }
    } else {
        my $vars = get_user_api_params($c->{rbac}, $c->{uid}, $c->{UID});
        my $application_ids = Yandex::OAuth::oa_get_clients_own_apps($c->{uid}, ['direct:api']);                  
        my @application_list;
        for my $application_id (@$application_ids) {
            my $app_info = Yandex::OAuth::oa_get_app_info($application_id);
            push @application_list, {application_id => $application_id,
                                     name => $app_info->{name}
                                    }
        }
        $request->{application_ids} = \@application_list;
        $vars->{request} = $request;

        $vars->{features_enabled_for_operator_all} = $cvars->{features_enabled_for_operator_all};
        $vars->{features_enabled_for_client_all} = $cvars->{features_enabled_for_client_all};

        $vars->{api_developer_offer_text} = _get_offer_text();
        return respond_bem($r, $c->reqid, $vars, source=>'data3');
    }
}

=head2 cmd_apiCertificationEditRequest

    Редактирование заявки на сертификацию приложений API

=cut

sub cmd_apiCertificationEditRequest:Cmd(apiCertificationEditRequest) 
    :Rbac(Code => [rbac_cmd_by_owners])
    :CheckCSRF
    :Description('редактирование заявки на сертификацию приложений API')
{
    my ($r, $SCRIPT, $template, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   c/};
    my %FORM = %{$_[0]{FORM}};

    my ($request, $cert_req);
    my $application_id = $FORM{application_id} or croak('Need application_id');

    if ($FORM{save}) {
        $FORM{uid} = $c->{uid};
        $request = prepare_request_object(\%FORM);
        if ($request->{access_type} eq 'test') {
            $cert_req = API::App::Request::Test->new(%$request);
        } elsif ($request->{access_type} eq 'normal') {
            $cert_req = API::App::Request::Normal->new(%$request);
        }
        my $new_id = $cert_req->update();

        my $saved_request = prepare_response_object($cert_req->to_hash());
        send_mail_to_support($saved_request, $c->{uid}, $SCRIPT);
        return redirect($r,"$SCRIPT?cmd=apiCertificationRequestList");

    } else {
        my $vars = get_user_api_params($c->{rbac}, $c->{uid}, $c->{UID});
        $request = get_request_info($c->{uid}, $application_id);
        if ($request) {
            if ($FORM{access_type} eq 'test') {
                $cert_req = API::App::Request::Test->new(%$request);
            } elsif ($FORM{access_type} eq 'normal') {
                $cert_req = API::App::Request::Normal->new(%$request);
            }
            $vars->{request} = prepare_response_object($cert_req->to_hash());
        }

        $vars->{api_developer_offer_text} = _get_offer_text();
        return respond_bem($r, $c->reqid, $vars, source=>'data3');
    }
}
    
=head2 cmd_apiCertificationEditRequest

    Редактирование заявки на сертификацию приложений API

=cut

sub cmd_apiCertificationUpgradeRequest:Cmd(apiCertificationUpgradeRequest) 
    :Rbac(Code => [rbac_cmd_by_owners])
    :CheckCSRF
    :Description('преобразование заявки на сертификацию приложений API из тестовой в полную')
{
    my ($r, $SCRIPT, $template, $c, $cvars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   c vars/};
    my %FORM = %{$_[0]{FORM}};

    my ($request, $cert_req);
    my $application_id = $FORM{application_id} or croak('Need application_id');

    my $vars = get_user_api_params($c->{rbac}, $c->{uid}, $c->{UID});
    $request = get_request_info($c->{uid}, $application_id);

    $vars->{features_enabled_for_operator_all} = $cvars->{features_enabled_for_operator_all};
    $vars->{features_enabled_for_client_all} = $cvars->{features_enabled_for_client_all};

    if ($request) {
        $cert_req = API::App::Request::Test->new(%$request);
         $vars->{request} = prepare_response_object($cert_req->to_hash());
    }

    $vars->{api_developer_offer_text} = _get_offer_text();
    return respond_bem($r, $c->reqid, $vars, source => 'data3');
}

=head2 cmd_ajaxCertificationDeleteRequest

    Удаление заявки на сертификацию приложений API

=cut

sub cmd_ajaxCertificationDeleteRequest:Cmd(ajaxCertificationDeleteRequest) 
    :Rbac(Code => [rbac_cmd_by_owners])
    :CheckCSRF
    :Description('удаление заявки на сертификацию приложений API')
{
    my ($r, $SCRIPT, $template, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   c/};
    my %FORM = %{$_[0]{FORM}};
    
    my $application_id = $FORM{application_id} or croak('Need application_id');
    
    my $request = get_request_info($c->{uid}, $application_id);
    if ($request) {
        my $cert_req = API::App::Request->new(%$request);
        $cert_req->delete();
        return respond_json($r, {result => 1});
    } 
    return respond_json($r, {result => 0});
}   

=head2 cmd_apiCertificationGetFile

    Возвращает файл со спецификацией или скриншотом по его имени

=cut

sub cmd_apiCertificationGetFile:Cmd(apiCertificationGetFile) 
    :Rbac(Code => [rbac_cmd_by_owners])
    :Description('редактирование заявки на сертификацию приложений API')
{
    my ($r, $SCRIPT, $template, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   c/};
    my %FORM = %{$_[0]{FORM}};

    my $application_id = $FORM{application_id} or croak('Need application_id');
    my $name = $FORM{name} or croak('Need deleted file name');
    $name =~ s(\r|\n|\x00)()g;
    my $request = get_request_info($c->{uid}, $application_id);
    if ($request) {
        my $cert_req = API::App::Request::Normal->new(%$request);
        my $specification = $cert_req->specifications_and_screenshots->get($name);
        if ($specification) {
            my $mime_type = MIME::Types->new()->mimeTypeOf($name);
            return respond_data($r, $specification->content, $mime_type, $name);
        } else {
            return respond_http_error($r, 404);
        }
    } else {
        return respond_http_error($r, 404);
    } 
}
 
=head2 cmd_apiCertificationRequestsByDate

    Cписок заявок на сертификацию приложений API
    за день, неделю или месяц

=cut

sub cmd_apiCertificationRequestsByDate:Cmd(apiCertificationRequestsByDate) 
    :Rbac(Role => [super, support, superreader], AllowDevelopers => 1)
    :Description('список заявок на сертификацию приложений API')
{
    my ($r, $SCRIPT, $template, $c, $cvars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   c vars/};
    my %FORM = %{$_[0]{FORM}};
    my ($vars, @result);
    
    my $start_date = now();
    my $end_date = now();

    if ($FORM{period} eq 'today') {
        $start_date->subtract( days => 1 );
    } elsif ($FORM{period} eq 'week'){
        $start_date->subtract( weeks => 1 );
    } elsif ($FORM{period} eq 'month'){
        $start_date->subtract( months => 1 );
    } else {
        error( iget("Неизвестный период: %s", $FORM{period}) );
    }

    $vars->{features_enabled_for_operator_all} = $cvars->{features_enabled_for_operator_all};
    $vars->{features_enabled_for_client_all} = $cvars->{features_enabled_for_client_all};

    my $requests = app_requests_get_by_date($start_date, $end_date, $FORM{page}||1, $FORM{objects_on_page});
    mix_ulogin($requests->{result});
    mix_access_and_history($requests->{result});
    $vars->{requests} = $requests;
    
    return respond_bem($r, $c->reqid, $vars, source => 'data3');
}
=head2 cmd_apiCertificationSearchRequest

    Ищет заявку по дате, id или названию приложения

=cut

sub cmd_apiCertificationSearchRequest:Cmd(apiCertificationSearchRequest) 
    :Rbac(Role => [super, support, superreader], AllowDevelopers => 1)
    :Description('показать заявку на сертификацию приложений API')
{
    my ($r, $SCRIPT, $template, $c, $cvars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   c vars/};
    my %FORM = %{$_[0]{FORM}};
    my ($vars, @result);
    my $requests;
    if ($FORM{query_string}) {
        $requests = app_requests_search($FORM{query_string}, $FORM{page}||1, $FORM{objects_on_page});
    } else {
        $requests = {result => []};
    }
    mix_ulogin($requests->{result});
    mix_access_and_history($requests->{result});
    $vars->{requests} = $requests;

    $vars->{features_enabled_for_operator_all} = $cvars->{features_enabled_for_operator_all};
    $vars->{features_enabled_for_client_all} = $cvars->{features_enabled_for_client_all};

    return respond_bem($r, $c->reqid, $vars, source => 'data3');
}
    
=head2 cmd_apiCertificationEditRequest

    Показать заявку на сертификацию приложений API

=cut

sub cmd_apiCertificationShowRequest:Cmd(apiCertificationShowRequest) 
    :Rbac(Role => [super, support, superreader], AllowDevelopers => 1)
    :Description('показать заявку на сертификацию приложений API')
{
    my ($r, $SCRIPT, $template, $c, $cvars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   c vars/};
    my %FORM = %{$_[0]{FORM}};
    my ($vars, $app_info, $cert_req);
    my $application_id = $FORM{application_id} or croak('Need application_id');
    my $cr_id = $FORM{app_request_id};

    my $request = $cr_id ? get_request_info_by_cr_id($cr_id) : get_request_info($c->{uid}, $application_id);

    if ($request) {
        if ($FORM{access_type} eq 'normal') {
            $cert_req = API::App::Request::Normal->new(%$request);
        } else {
            $cert_req = API::App::Request::Test->new(%$request);
        }
        $request = prepare_response_object($cert_req->to_hash());
        my $user_data = get_user_data($c->{uid}, [qw/login email/]);
        hash_merge($request, $user_data);

        $request->{api_logins_list} = prepare_logins_list($request->{api_logins_list}, $SCRIPT)
            if $request->{api_logins_list};

        $request->{login} = prepare_logins_list($request->{login}, $SCRIPT)->[0]
            if $request->{login};

        my $accesses = API::App::AccessRegistry->new([$request->{application_id}]);
        my $access = $accesses->get($request->{application_id});
        $request->{app_access} = $access ? $access->to_hash() : undef;

        $request->{history} = app_request_history_get($c->{uid}, $application_id);
        mix_ulogin([$request]);
        $vars->{request} = $request;

        $vars->{features_enabled_for_operator_all} = $cvars->{features_enabled_for_operator_all};
        $vars->{features_enabled_for_client_all} = $cvars->{features_enabled_for_client_all};

        return respond_bem($r, $c->reqid, $vars, source => 'data3');

    } else {
        return respond_http_error($r, 404);
    }
}

=head2 cmd_ajaxApiAppSetStatus

    Изменить статус заявки
    
=cut

sub cmd_ajaxApiAppSetStatus:Cmd(ajaxApiAppSetStatus) 
    :Rbac(Role => [super, support])
    :CheckCSRF
    :Description('изменить статус заявки')
{
    my ($r, $SCRIPT, $template, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   c/};
    my %FORM = %{$_[0]{FORM}};
    
    my $application_id = $FORM{application_id} or croak('Need application_id');
    my $access = API::App::Access->new(
        application_id => $application_id,
        manager_login => get_login(uid => $c->{UID}),
        comment => $FORM{comment},
        check_time => now(),
    );
    if ($FORM{status} eq 'approve'){
        $access->access_type($FORM{access_type});
        $access->add();
    } elsif ($FORM{status} eq 'reject'){
        $access->delete();
    } else {
        croak('Неизвестный статус '.$FORM{status});
    }
    return respond_json($r, {
        ulogin => $FORM{ulogin},
        application_id => $application_id,
        access_type => $access->access_type,
        status => $FORM{status},
    });
}

=head2 cmd_ajaxGenerateCSVReport

    Сгенерировать CSV отчет по заявкам на сертификацию
    
=cut

sub cmd_ajaxGenerateCSVReport:Cmd(ajaxGenerateCSVReport) 
    :Rbac(Role => [super, support])
    :Description('сгенерировать CSV отчет')
{
    my ($r, $SCRIPT, $template, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   c/};
    
    my $report = API::App::Request::CSV->new();

    return respond_json($r, {result => $report->generate()});
}
    
    
=head2 cmd_ajaxGetCSVReport

    Получить CSV отчет по заявкам на сертификацию
    
=cut

sub cmd_ajaxGetCSVReport:Cmd(ajaxGetCSVReport) 
    :Rbac(Role => [super, support])
    :Description('получить CSV отчет')
{
    my ($r, $SCRIPT, $template, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   c/};
    
    my $report = API::App::Request::CSV->new();
    my $mime_type = MIME::Types->new()->mimeTypeOf($report->file_name);
    return respond_data($r, $report->get(), $mime_type, $report->file_name);
}

=head2 cmd_ajaxResendAppAccessToSandbox

    Переотправить доступ в песочницу
    
=cut

sub cmd_ajaxResendAppAccessToSandbox:Cmd(ajaxResendAppAccessToSandbox) 
    :Rbac(Role => [super, support])
    :CheckCSRF
    :Description('переотправить доступ в песочницу')
{
    my ($r, $SCRIPT, $template, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   c/};
    my %FORM = %{$_[0]{FORM}};
    
    my $application_id = $FORM{application_id} or croak('Need application_id');
    my $accesses = API::App::AccessRegistry->new([$application_id]);
    my $access = $accesses->get($application_id);
    $access->resend_app_access_to_sandbox();

    return respond_json($r, {result => 1});
}

=head2 validate_app_id

    Проверяет валидность application_id

=cut

sub validate_app_id{
    my ($application_id, $uid) = @_;
    my ($app_info, @errors);
    
    eval {$app_info = Yandex::OAuth::oa_get_app_info($application_id);};
    push @errors, ['application_id', iget('Приложения с таким id не найдено')]
        if $@;
    LogTools::log_messages('CertificationRequest', $@) if $@;

    unless ($app_info->{scope}{'direct:api'}) {
        push @errors, ['application_id', iget('У этого приложения нет доступа в Директ')];
    }
    my $is_exists = get_one_field_sql(PPCDICT,
                ["select 1 from api_app_certification_request",
                 where => {application_id => $application_id,
                           uid => $uid}]);
    push @errors, ['application_id', iget('Заявка для этого приложения уже создана Вами')]
        if $is_exists;
    return {errors => \@errors};
}

sub prepare_logins_list($$){
    my ($logins_list, $SCRIPT) = @_;

    my @logins_info;
    my @logins_list = split(/\s+/, $logins_list);
    my $login2uid = get_login2uid(login => \@logins_list);

    for my $login (@logins_list){
        my $item = {login => $login};
        $item->{link} = "$SCRIPT?cmd=modifyUser&ulogin=$login"
            if $login2uid->{$login};

        $item->{approved_requests_count} = app_requests_count($login2uid->{$login}, status => 'Approve');

        push @logins_info, $item;
    }
    return \@logins_info;
}

sub mix_access_and_history {
    my $requests = shift;
    my @application_ids = map {$_->{application_id}} @$requests;
    my $accesses = API::App::AccessRegistry->new(\@application_ids);
    my @accesses_managers_logins = grep {$_} map {$_->manager_login} $accesses->list();
    my @managers_logins = grep {$_} map {$_->{manager_login}} @$requests;
    my $login2domain_login = {};
    if (scalar @managers_logins or scalar @accesses_managers_logins) {
        $login2domain_login = get_internal_user_logins([@managers_logins, @accesses_managers_logins]);
    }
    for my $request (@$requests){
        my $access = $accesses->get($request->{application_id});
        if ($access) {
            my $app_access = $access->to_hash();
            $app_access->{manager_domain_login} = $login2domain_login->{$access->manager_login}
                if $access->manager_login;
            $request->{app_access} = $app_access;
        }
        $request->{manager_domain_login} = $login2domain_login->{$request->{manager_login}}
            if $request->{manager_login};
        $request->{history} = app_request_history_get($request->{uid}, $request->{application_id});
    }
    return;
}

sub get_internal_user_logins {
    my $logins = shift;
    my $login2uid = get_login2uid(login => $logins);
    my $uid2domain_login = get_hash_sql(PPC(uid => [values %$login2uid]),
        ['select uid, domain_login from internal_users',
            where => {uid => SHARD_IDS}, 'limit 1'
        ]);
    my %uid2login = reverse %$login2uid;
    return {map { $uid2login{$_} => $uid2domain_login->{$_} } keys %$uid2domain_login}
}

1;
