package DoCmdBannersAdditions;

=head1 DESCRIPTION

Контроллеры для работы с дополнениями баннеров.

=cut

use Direct::Modern;
use base qw/DoCmd::Base/;

use Settings;
use Yandex::I18n;
use Yandex::DBTools;
use Yandex::ListUtils qw/xuniq/;

use Direct::ResponseHelper;
use Direct::BannersAdditions;
use Direct::Model::AdditionsItemCallout;
use Direct::Validation::BannersAdditions;

use Rbac;

=head2 cmd_getBannersAdditions

Выдача уточнений по клиенту, с листалкой,
параметры: ulogin, additions_type, limit, offset.
в ответе список callouts как в превью баннера
    {
        callouts: [
            {
                additions_item_id  : 16,
                callout_text       : "callout text",
                status_moderate    : "Sent"
            }, {...}, ...
        ]
      }

=cut

sub cmd_getBannersAdditions
    :Cmd(getBannersAdditions)
    :Rbac(Code => rbac_cmd_by_owners)
    :Description('Получение списка дополнений баннеров')
    :RequireParam(additions_type => 'Require', limit => 'Require', offset => 'Require')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c   vars/};
    my %FORM = %{$_[0]{FORM}};

    return respond_json($r, {success => 0, error => "additions_type is not valid"}) unless $FORM{additions_type} =~ /^(callout)$/;

    my $items_callouts = Direct::BannersAdditions->get_by(
        additions_type => $FORM{additions_type},
        client_id => $c->client_client_id,
        limit => $FORM{limit},
        offset => $FORM{offset},
        sort => [{id => "asc"}],
    )->items_callouts();

    my $result = {success => 1, callouts => [map {$_->to_template_hash} @$items_callouts]};
    return respond_json($r, $result);
}

=head2 cmd_saveBannersAdditions

Сохранение дополнений баннеров на клиента

=cut

sub cmd_saveBannersAdditions
    :Cmd(saveBannersAdditions)
    :Rbac(Code => rbac_cmd_by_owners, ExceptRole => [superreader, limited_support])
    :CheckCSRF
    :Description('Сохранение дополнений баннеров')
    :RequireParam(json_banner_additions => 'banner_additions')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c   vars/};
    my %FORM = %{$_[0]{FORM}};
    my $client_perminfo = Rbac::get_perminfo(uid => $uid);
    my $client_role = $client_perminfo->{role} || $Rbac::ROLE_EMPTY;
    if ($client_role ne $Rbac::ROLE_CLIENT && $client_role ne $Rbac::ROLE_EMPTY) {
        return error_json($r, {error => iget("Выбранный пользователь не является клиентом")});
    }

    my $banner_additions_form = $FORM{json_banner_additions};

    my $items_callouts = [];
    if (defined $banner_additions_form->{callouts}) {
        for my $row (@{$banner_additions_form->{callouts}}) {
            push @$items_callouts, Direct::Model::AdditionsItemCallout->new(client_id => $c->client_client_id, callout_text => $row->{callout_text});
        }
    }

    my $banner_additions = Direct::BannersAdditions->new(items_callouts => $items_callouts);
    my $validate_result = Direct::Validation::BannersAdditions::validate_callouts($items_callouts);

    if (! $validate_result->is_valid()) {
        my $vr_front = $validate_result->convert_vr_for_frontend();
        return respond_json($r, {success => 0, callouts => $vr_front});
    }

    $banner_additions->save;
    my $callouts_result = [map {$_->to_template_hash} @$items_callouts];

    return respond_json($r, {success => 1, callouts => $callouts_result});
}

=head2 cmd_deleteBannersAdditions

Удаление уточнений на клиенте

параметры: ulogin, json_banner_additions.

json_banner_additions:
    {
        callouts: [
            {
                additions_item_id  : 16,
                callout_text       : "callout text",
                status_moderate    : "Sent"
            }, {...}, ...
        ]
      }

В каждом уточнении обязателен только additions_item_id

=cut

sub cmd_deleteBannersAdditions
    :Cmd(deleteBannersAdditions)
    :Rbac(Code => rbac_cmd_by_owners, ExceptRole => [superreader, limited_support])
    :CheckCSRF
    :Description('Удаление уточнений')
    :RequireParam(json_banner_additions => 'banner_additions')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c   vars/};
    my %FORM = %{$_[0]{FORM}};

    my $banner_additions_form = $FORM{json_banner_additions};

    my $items_callouts = [];
    if (defined $banner_additions_form->{callouts}) {
        for my $row (@{$banner_additions_form->{callouts}}) {
            push @$items_callouts, Direct::Model::AdditionsItemCallout->new(id => $row->{additions_item_id}, client_id => $c->client_client_id);
        }
    }

    my $validate_result = Direct::Validation::BannersAdditions::validate_callouts_owner($items_callouts, $c->client_client_id);

    if (! $validate_result->is_valid()) {
        my $vr_front = $validate_result->convert_vr_for_frontend();
        return respond_json($r, {success => 0, callouts => $vr_front});
    }

    my $banner_additions = Direct::BannersAdditions->new(items_callouts => $items_callouts);
    $banner_additions->delete();

    my $result = {success => 1, callouts => [map {$_->to_template_hash} @$items_callouts]};
    return respond_json($r, $result);
}

=head2 cmd_remoderateBannersAdditions

Перемодерация уточнений на клиенте

параметры: ulogin, cid, adgroup_ids (список pid через запятую).

=cut

sub cmd_remoderateBannersAdditions
    :Cmd(remoderateBannersAdditions, moderateAcceptBannersAdditions)
    :Rbac(Cmd => remoderateBannersAdditions, Role => [super, superreader, support, limited_support, placer], CampKind => {web_edit => 1}, AllowDevelopers => 1 )
    :Rbac(Cmd => moderateAcceptBannersAdditions, Perm => AutoModerate, CampKind => {web_edit => 1}, AllowDevelopers => 1)
    :CheckCSRF
    :Description('Перемодерация уточнений')
    :RequireParam(cid => 'Cid', adgroup_ids => 'Pids')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c   vars/};
    my %FORM = %{$_[0]{FORM}};

    my $banner_additions = Direct::BannersAdditions->get_by(
        additions_type => 'callout',
        client_id => $c->client_client_id,
        adgroup_id => $FORM{adgroup_ids}
    );

    if (@{$banner_additions->items_callouts()}) {
        my $items_callouts_for_update = [];

        for my $items_callout (xuniq {$_->id} @{$banner_additions->items_callouts()}) {
            $items_callout->status_moderate('Ready');
            push @$items_callouts_for_update, $items_callout;
        }

        my $new_banner_additions = Direct::BannersAdditions->new(items_callouts => $items_callouts_for_update);
        $new_banner_additions->save;

        # переотправка на ручную модерацию
        my $moderate_type = $FORM{cmd} eq 'moderateAcceptBannersAdditions' ? 'auto' : 'pre';
        do_mass_insert_sql(PPC(pid => $FORM{adgroup_ids}), 'INSERT IGNORE moderate_additions(additions_item_id, moderate_type) VALUES %s', [
            map {[$_, $moderate_type]}
            map {$_->id}
            @$items_callouts_for_update
        ]);
    }

    my $result = {success => 1};

    if (detect_respond_format($r) eq 'json') {
        return respond_json($r, $result);
    } else {
        return redirect($r, $SCRIPT, {cmd => 'showCamp', cid => $FORM{cid}, ulogin => $FORM{ulogin}});
    }
}

1;
