package DoCmdCurrencyConvert;

use strict;
use warnings;

use base qw/DoCmd::Base/;

use List::Util qw/sum/;
use List::MoreUtils qw/none uniq any firstval/;

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::TimeCommon;
use Yandex::Validate;
use Yandex::SendSMS qw/sms_check_user/;
use Yandex::HashUtils;
use Yandex::ScalarUtils;
use Yandex::ListUtils;
use Yandex::IDN qw/is_valid_email/;
use Yandex::Balance qw/balance_get_direct_balance/;
use Yandex::I18n;

use Primitives;
use PrimitivesIds;
use Client;
use Client::ConvertToRealMoney;
use Client::CurrencyTeaserData;
use Client::NDSDiscountSchedule;
use Currencies;
use Currency::Rate;
use TextTools;
use Direct::ResponseHelper qw(respond_to respond_template error redirect message);
use Common qw(:globals :subs);
use Campaign;
use Campaign::Types;
use Property;
use RBACDirect ();
use RBACElementary ();

use utf8;

=head3 _check_convert_start_datetime2ts

    Конвертирует дату и время начала перехода из формата, передаваемого с клиента, в unix timestamp.
    Попутно делает проверки всех условий, налагаемых на дату начала перехода, поэтому функция
    вряд ли пригодна для использования где-то вне cmd_moveToRealMoney.

    Возвращает timestamp если всё хорошо или undef, если была какая-то ошибка.

=cut

sub _check_convert_start_datetime2ts {
    my ($date, $hour, $minute) = @_;

    if (none {$minute == $_} (0, 30)) {
        # время можно указывать с гранулярностью в полчаса
        return undef;
    }

    my $datetime_str = "$date $hour:$minute:00";
    my $convert_time = eval { mysql2unix($datetime_str) };

    if (!$convert_time) {
        # дата и время не распарсились
        return undef;
    }

    my $now_ts = time();
    if ($convert_time < $now_ts) {
        # указанное время уже прошло
        return undef;
    }

    if ($convert_time - $now_ts > $Client::ConvertToRealMoney::SECONDS_WE_ALLOW_TO_CHOOSE_CONVERT_DATE_IN_FUTURE) {
        # дату можно выбрать на ограниченное время вперёд
        # на клиенте это значение проверяется и меняется в data/block/b-convert-page/b-convert-page.js
        return undef;
    }

    return $convert_time;
}

=head2 cmd_moveToRealMoney

    Переход из у.е. в реальную валюту

    Без параметра move показываем клиенту страницу с данным о том, что станет с суммами на кампании после перевода (из Баланса),
    и добиваемся всех необходимых для перевода данных + окончательное подтверждение перехода.
    С параметром move => 1 ставим заявку на переход в очередь.

=cut

sub cmd_moveToRealMoney :Cmd(moveToRealMoney)
    :CheckCSRF
    :Rbac(Code => rbac_cmd_by_owners, ExceptRole => [superreader, media, limited_support])
    :Description('переход на настоящие деньги')
    :AllowBlockedClient
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c   vars/};
    my %FORM = %{$_[0]{FORM}};

    if (Property->new($Client::CURRENCY_CONVERT_TEASER_DISABLED_PROPERTY_NAME)->get()) {
        error(iget("Переход в реальные валюты временно недоступен"));
    }

    my ($currency, $country, $email) = @FORM{qw/
         currency   country   email/};
    die "Invalid currency: " . str($currency) unless $currency && $currency ne 'YND_FIXED' && is_valid_currency($currency);
    die "Invalid country: " . str($country) unless $country && is_valid_int($country, 1);

    my $client_id = $c->client_client_id;
    die "ClientID missing for uid $uid" unless $client_id;

    my $client_nds = get_client_NDS($client_id, fetch_missing_from_balance => 1, rbac => $rbac, fetch_for_ynd_fixed_too => 1);
    my $client_currencies = get_client_currencies($client_id);

    # обновляем кешированный список пар страна-валюта, чтобы can_convert_to_real_money проверял по [почти] актуальному
    # список может изменяться при оплатах в других сервисах без нашего ведома
    my $agency_id = Primitives::get_client_first_agency($client_id);
    my $res = eval {
        Client::CurrencyTeaserData::fetch_client_multicurrency_teaser_data($client_id, $agency_id, balance_timeout => 120);
        return 1;
    };
    if (!$res) {
        my $msg = $@;
        # если обновить не получилось, то проверим нет ли у нас
        # свежих ($COUNTRY_CURRENCY_LAST_UPDATE_INTERVAL_DAYS) закешированнх данных
        my $has_fresh_data = get_one_field_sql(PPC(ClientID => $client_id), "SELECT 1 FROM client_teaser_data_lastupdate WHERE ClientID = ? AND last_update > DATE_SUB(NOW(), INTERVAL $Client::ConvertToRealMoney::COUNTRY_CURRENCY_LAST_UPDATE_INTERVAL_DAYS DAY) LIMIT 1", $client_id);
        if ($has_fresh_data) {
            # ok
        } else {
            die $msg;
        }
    }

    my @country_currencies;
    my $can_convert = Client::can_convert_to_real_money(
        ClientID => $client_id,
        NDS => $client_nds,
        client_currencies => $client_currencies,
        client_chief_uid => $c->client_chief_uid,
        country_currency_ref => \@country_currencies,
        ignore_nds_absence => 1,
        agency_id => $agency_id,
    );
    die "can_convert_to_real_money returned $can_convert" if defined $can_convert;
    die "Client $client_id cannot be converted into currency $currency and country $country" if none {$_->{currency} eq $currency && $_->{region_id} == $country} @country_currencies;
    return error(iget('Заказ конвертации клиентом невозможен')) if $login_rights->{client_have_agency} && none { $login_rights->{"${_}_control"} } qw(agency manager support super);

    my $country_set_in_balance = 1;
    if (!defined $client_nds) {
        # у клиента нет выбранной страны в Балансе, получем НДС по выбранной им в тизере стране и повторяем проверки
        $country_set_in_balance = 0;
        # NB! совсем правильно учитывать здесь и в can_convert_to_real_money еще и резидентство клиентов
        # для агентстких нерезидентов нужны неагентсткие страны/валюты/ндс.
        # пока не учитываем, так как такие случаи крайне редки
        if ($agency_id) {
            $client_nds = Client::NDSDiscountSchedule::get_country_today_nds_for_agency($country);
        } else {
            $client_nds = Client::NDSDiscountSchedule::get_country_today_nds_not_for_agency($country);
        }
        if (defined $client_nds) {
            $can_convert = Client::can_convert_to_real_money(
                ClientID => $client_id,
                NDS => $client_nds,
                client_currencies => $client_currencies,
                client_chief_uid => $c->client_chief_uid,
                country_currency_ref => \@country_currencies,
                agency_id => $agency_id,
            );
            die "can_convert_to_real_money for ClientID $client_id with NDS $client_nds for country $country returned error $can_convert" if defined $can_convert;
        } else {
            die "for ClientID $client_id there is not NDS nor in database nor got by country $country";
        }
    }

    my $convert_type = Client::ConvertToRealMoney::get_convert_type($client_id, $currency, $client_nds, \@country_currencies);

    my $now_ts = time();
    my $convert_start_ts;
    if ($convert_type eq 'MODIFY') {
        $convert_start_ts = Client::ConvertToRealMoney::get_closest_modify_convert_start_ts($now_ts);
    }

    # проверяем и конвертируем переданные клиентом данные
    # в виде "культурных" ошибок на странице показываем далеко не все
    my (%errors, $convert_time);
    if ($FORM{move}) {
        # проверяем email. указывать его не обязательно
        if (defined $email && $email ne '') {
            if (!is_valid_email($email)) {
                $errors{invalid_email} = 1;
            }
        } else {
            $email = undef; # если email не указан, в таблицу всегда пишем NULL
        }

        if ($convert_type eq 'COPY') {
            # при переходе с копированием кампаний можно указать дату и время начала перехода
            if ($FORM{convert_time} && $FORM{convert_time} eq 'now') {
                $convert_time = time();
            } elsif ($FORM{convert_time} && $FORM{convert_time} eq 'in_time') {
                # данные приходят в таком виде: convert_hour = 06 convert_minute = 30 convert_day = 2012-09-18
                $convert_time = _check_convert_start_datetime2ts($FORM{convert_day}, $FORM{convert_hour}, $FORM{convert_minute});
            }
        } elsif ($convert_type eq 'MODIFY') {
            $convert_time = $convert_start_ts;
        } else {
            die "Unknown convert type: $convert_type";
        }

        if (!$convert_time) {
            $errors{invalid_convert_time} = 1;
        }
    }

    $vars->{new_currency} = $currency;
    $vars->{convert_type} = $convert_type;

    $vars->{country} = $country;
    my $region = firstval {$_->{region_id} == $country} @geo_regions::COUNTRY_REGIONS;
    $vars->{country_name} = $region->{'name_' . Yandex::I18n::current_lang()};

    # чтобы не путать людей какого числа произойдёт переход, вместо честной полночи 24.09.2012 00:00 показываем 23.09.2012 23:59
    if ($convert_type eq 'MODIFY') {
        $vars->{convert_start_datetime} = unix2human($convert_start_ts - 60);
    }

    my $client_discount = get_client_discount($client_id);
    $vars->{discount} = $client_discount;

    if ($FORM{move} && !%errors) {
        # клиент подтвердил выбор страны и валюты прописываем выбранную валюту в Балансе и у нас
        my $is_error = update_client_id($UID, $client_id, { REGION_ID => $country });
        die "Error updating client $client_id data in Balance" if $is_error;
        create_update_client({client_data => {
            ClientID => $client_id,
            country_region_id => $country,
        }});

        # изменение страны вызывает изменение списка доступных пар страна-валюта и графика НДС
        my $update_success = eval {
            my $agency_id = Primitives::get_client_first_agency($client_id);
            Client::CurrencyTeaserData::fetch_client_multicurrency_teaser_data($client_id, $agency_id, balance_timeout => 120);
            return 1;
        };
        if (!$update_success) {
            warn "WARNING: country-currency refresh after setting country failed for ClientID $client_id, skipping: $@";
        }
        Client::NDSDiscountSchedule::sync_nds_schedule_for_clients([$client_id], rbac => $rbac);

        do_in_transaction {
            # ставим заявку на переход в очередь
            do_insert_into_table(PPC(ClientID => $client_id), 'currency_convert_queue', {
                ClientID => $client_id,
                uid => $uid,
                convert_type => $convert_type,
                new_currency => $currency,
                country_region_id => $country,
                email => $email,
                start_convert_at => unix2mysql($convert_time),
                state => 'NEW',
                in_state_since__dont_quote => 'NOW()',
            });

            # записываем дату перехода
            do_insert_into_table(PPC(ClientID => $client_id), 'client_currency_changes', {
                ClientID => $client_id,
                currency_from => 'YND_FIXED',
                currency_to => $currency,
                date => unix2mysql(ts_round_day($convert_time)),
            });
        };

        if ($convert_type eq 'MODIFY') {
            # показываем страницу о том, что заявка принята. интерфейс при этом не блокируется
            return respond_template($r, $template, 'convert/convert-in-progress.tt2', $vars);
        } elsif ($convert_type eq 'COPY') {
            # т.к. сразу после добавления в очередь у клиента заблокируется интерфейс, не очень важно куда его отправить
            return respond_to($r,
                json => [{ success => "1" }],
                any => sub { redirect($r, $SCRIPT, {cmd => 'showCamps', uid_url => $FORM{uid_url}}) },
            );            
        } else {
            die "Unknown convert type: $convert_type";
        }
    } else {
        # показываем страницу настроек перехода (возможно, с ошибкой)
        # список кампаний и денег до/после показываем только при переходе копированием
        # при переходе без копирования важно только наличие/отсутствие кампаний (причём, это более частый случай)
        my $campaigns_count;
        if ($convert_type eq 'MODIFY') {
            my $campaigns_data = get_user_camps($c->client_chief_uid, mediaType => get_camp_kind_types_and('web_edit_base', 'currency_convert'), archived => 'No', only_count => 1);
            $campaigns_count = $campaigns_data->{count};
        } else {
            my $all_wallets = get_all_wallet_camps(client_chief_uid => $c->client_chief_uid);
            my $has_enabled_wallets = (any { $_->{is_enabled} } @$all_wallets) ? 1 : 0;
            $vars->{has_enabled_wallets} = $has_enabled_wallets;

            my @campaigns;
            my $compensate_sum_converted = 0;
            # несмотря на то, что конвертируем мы в том числе и архивные кампании (всегда для MODIFY и по галочке для COPY),
            # на странице перехода их не показываем
            my $order_by = ($has_enabled_wallets) ? ['cid'] : ['total DESC', 'cid'];
            my $campaigns_data = get_user_camps($c->client_chief_uid, mediaType => get_camp_kind_types_and('copyable', 'with_currency'), archived => 'No', order_by => $order_by);
            if (@{$campaigns_data->{campaigns}}) {
                # получаем от Баланса суммы на кампании в выбранной валюте
                my $orders_balance_data = balance_get_direct_balance(ClientID => $client_id, CurrencyCode => $currency);
                my %order_balance;
                if ($orders_balance_data && @$orders_balance_data) {
                    %order_balance = map { $_->{ORDER_ID} => ($_->{CURRENT_SUM} - $_->{COMPLETION_SUM}) } @$orders_balance_data;
                }
                for my $campaign (@{$campaigns_data->{campaigns}}, @{$campaigns_data->{wallet_campaigns}}) {
                    $campaign->{sum_rest} = $campaign->{sum} - $campaign->{sum_spent};
                    if ($has_enabled_wallets && $campaign->{sum_rest} < 0) {
                        # на кампании перетрата, которая ещё не компенсирована с общего счёта
                        # Баланс в Balance.GetDirectBalance такие не учитывает (т.к. реальные деньги с кошелька ешё не списались)
                        # поэтому компенсируем руками с долей погрешности на конвертацию суммы и неактуальность наших данных
                        $compensate_sum_converted += convert_currency($campaign->{sum_rest} || 0, $campaign->{currency}, $currency, with_nds => 0);
                    }
                    $campaign->{sum_rest_converted} = Currencies::remove_nds($order_balance{$campaign->{cid}} || 0, $client_nds);
                    if ($campaign->{sum_rest_converted} < 0) {
                        # непокрытые деньгами фишки дарятся и не конвертируются
                        $campaign->{sum_rest_converted} = 0;
                    }

                    if ($campaign->{mediaType} ne 'wallet') {
                        # $campaign->{strategy} используется в Campaign::campaign_strategy и попадает в $campaign->{strategy}->{name}
                        $campaign->{strategy} = Campaign::campaign_strategy($campaign);
                        push @campaigns, $campaign;
                    }
                }
            }

            $vars->{campaigns} = \@campaigns;
            $campaigns_count = scalar(@campaigns);

            my @all_campaigns = (@campaigns, @{$campaigns_data->{wallet_campaigns}});
            $vars->{total_sum} = round2s(sum 0, map {$_->{sum_rest}} @all_campaigns);
            $vars->{total_sum_converted} = round2s(sum $compensate_sum_converted, map {$_->{sum_rest_converted}} @all_campaigns);
            if ($vars->{total_sum_converted} < 0) {
                # непокрытые деньгами фишки дарятся и не конвертируются
                $vars->{total_sum_converted} = 0;
             }

            $vars->{current_date} = unix2human($now_ts, '%Y-%m-%d');
        }
        $vars->{has_campaigns} = ($campaigns_count > 0) ? 1 : 0;

        $vars->{phone_for_sms} = sms_check_user($uid, $c->user_ip);
        $vars->{errors} = \%errors;

        return respond_template($r, $template, 'convert/convert.tt2', $vars);
    }
}

=head2 cmd_massMoveToRealMoney

    Массовый заказ перехода из у.е. в реальную валюту для всех субклиентов агентств

    Без параметра move показываем клиенту страницу с данным о количестве конвертируемых клиентов и их валютах
    + кнопка окончательное подтверждение перехода.
    С параметром move => 1 ставим заявки на переход в очередь.

=cut

sub cmd_massMoveToRealMoney :Cmd(massMoveToRealMoney)
    :CheckCSRF
    :Rbac(Code => rbac_can_massMoveToRealMoney)
    :Description('массовый переход на настоящие деньги')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c   vars/};
    my %FORM = %{$_[0]{FORM}};

    if (Property->new($Client::CURRENCY_CONVERT_TEASER_DISABLED_PROPERTY_NAME)->get()) {
        error(iget("Переход в реальные валюты временно недоступен"));
    }

    my $agency_id = $c->client_client_id;
    die "ClientID missing for uid $uid" unless $agency_id;

    my $clids = RBACDirect::rbac_get_subclients_clientids($rbac, $uid);
    my $clid2currencies = mass_get_client_currencies($clids);
    $clids = [ grep { $clid2currencies->{$_}->{work_currency} eq 'YND_FIXED' } @$clids ];
    my $clients = get_all_sql(PPC(ClientID => $clids), ['
        SELECT ctfmt.ClientID
             , cfcc.country_region_id
             , cfcc.currency
        FROM clients_to_force_multicurrency_teaser ctfmt
        LEFT JOIN currency_convert_queue q ON ctfmt.ClientID = q.ClientID
        LEFT JOIN client_firm_country_currency cfcc ON ctfmt.ClientID = cfcc.ClientID
     ', WHERE => {
            'ctfmt.ClientID' => SHARD_IDS,
            'q.ClientID__is_null' => 1,
        }, '
        GROUP BY ctfmt.ClientID
        HAVING
                COUNT(DISTINCT cfcc.country_region_id) = 1
            AND COUNT(DISTINCT cfcc.currency) = 1
    ']);

    my $convert_start_ts = Client::ConvertToRealMoney::get_closest_modify_convert_start_ts();
    # чтобы не путать людей какого числа произойдёт переход, вместо честной полночи 24.09.2012 00:00 показываем 23.09.2012 23:59
    $vars->{convert_start_datetime} = unix2human($convert_start_ts - 60);

    my %currency2client_count;
    if (@$clients) {
        my $clids = [ map { $_->{ClientID} } @$clients ];
        my $clid2nds = mass_get_client_NDS($clids, fetch_for_ynd_fixed_too => 1);
        my $clid2first_agency = Primitives::mass_get_client_first_agency($clids);
        my $clients_data = mass_get_clients_data($clids, [qw/allow_create_scamp_by_subclient/]);
        my $clid2chief_uid = {};
        for my $clids_chunk (chunks($clids, 5_000)) {
            # разбиваем на чанки, чтобы не падать по памяти при большом количестве клиентов
            hash_merge($clid2chief_uid, RBACElementary::rbac_get_chief_reps_of_clients($clids_chunk));
        }

        # по факту, уже продублировали почти все проверки из Client::can_convert_to_real_money в SQL запросе выше, поэтому её не используем
        my @convert_requests;
        my $clients_count = 0;
        for my $client (@$clients) {
            my $client_id = $client->{ClientID};

            my $client_nds = $clid2nds->{$client_id};
            next unless defined $client_nds;

            my $is_free_client = $clients_data->{$client_id}->{allow_create_scamp_by_subclient};
            next if $is_free_client;

            my $currency = $client->{currency};
            my $country = $client->{country_region_id};

            $currency2client_count{$currency}++;
            $clients_count++;

            if ($FORM{move}) {
                # единственность стран и валют и то, что они такие -- проверили в исходном SQL-запросе
                my @country_currencies = ({region_id => $country, currency => $currency});
                my $convert_type = Client::ConvertToRealMoney::get_convert_type($client_id, $currency, $client_nds, \@country_currencies);
                push @convert_requests, {
                    ClientID => $client_id,
                    uid => undef, # SMS никому не пишем
                    convert_type => $convert_type,
                    new_currency => $currency,
                    country_region_id => $country,
                    email => undef, # отдельных писем не пишем, пишем только общее письмо по окончанию конвертации всех поставленных клиентов
                    start_convert_ts => $convert_start_ts,
                };
            }

            last if $clients_count >= $Client::ConvertToRealMoney::MASS_CONVERT_BUTTON_CLIENTS_LIMIT;
        }

        if (@convert_requests) {
            Client::ConvertToRealMoney::mass_queue_currency_convert(\@convert_requests, ignore => 1, AgencyID => $agency_id);
        }

    }

    if ($FORM{move}) {
        my $retpath = $SCRIPT . '?cmd=showClients' . str($FORM{uid_url});
        return message(
            iget('Конвертация клиентов в валюту платежа успешно запланирована. Вы получите уведомление по завершении'),
            { return_to => { href => $retpath, text => iget('Вернуться к списку клиентов') } },
        );
    }

    $vars->{clients_count_by_currency} = \%currency2client_count;

    return respond_template($r, $template, 'convert/convert.tt2', $vars);
}

=head2 cmd_currencyConvertInProgress

    Страница с информацией о том, что идёт переход в настоящие деньги и интерфейс заблокирован

=cut

sub cmd_currencyConvertInProgress :Cmd(currencyConvertInProgress)
    :Rbac(Code => rbac_cmd_by_owners, ExceptRole => [media])
    :Description('идёт переход в настоящие деньги')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c   vars/};

    my $client_id = get_clientid(uid => $uid);
    die "ClientID missing for client $uid" unless $client_id;

    my $convert_request_data = get_one_line_sql(PPC(ClientID => $client_id), 'SELECT start_convert_at, convert_started_at, new_currency, email, uid, convert_type FROM currency_convert_queue WHERE ClientID = ?', $client_id);
    die "no convert request found for ClientID $client_id" unless $convert_request_data && ref($convert_request_data) eq 'HASH' && %$convert_request_data;
    hash_copy $vars, $convert_request_data, qw/new_currency email convert_type/;

    $vars->{phone_for_sms} = sms_check_user($convert_request_data->{uid}, $c->user_ip);
    $vars->{campaigns_working} = ($convert_request_data->{convert_type} eq 'MODIFY') ? 1 : 0;
    $vars->{interface_blocked} = 1;

    my $convert_started_at = mysql2unix ( check_mysql_date($convert_request_data->{convert_started_at}) ? $convert_request_data->{convert_started_at} : $convert_request_data->{start_convert_at} );
    my $convert_finish_ts = $convert_started_at + Client::ConvertToRealMoney::get_client_convert_duration_forecast($c->client_chief_uid, $convert_request_data->{convert_type});
    $vars->{convert_finish_forecast} = unix2mysql($convert_finish_ts);

    return respond_template($r, $template, 'convert/convert-in-progress.tt2', $vars);
}

=head2 cmd_currencyConvertSuccess

    Страница успешного перехода в реальную валюту.
    Показываем пользователю её в ответ на любой запрос пока не нажмёт на ней кнопку "показать кампании".
    На ней сколько денег было в у.е. и сколько стало в реальной валюты. Суммарно и по каждой кампании.

=cut

sub cmd_currencyConvertSuccess :Cmd(currencyConvertSuccess)
    :Rbac(Code => rbac_cmd_by_owners, ExceptRole => [media])
    :Description('переход в настоящие деньги успешно завершён')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c   vars/};

    my $client_id = $c->client_client_id;
    my $client_nds = get_client_NDS($client_id);
    my $client_discount = get_client_discount($client_id);

    my $convert_request_data = get_one_line_sql(PPC(ClientID => $client_id), 'SELECT new_currency, convert_started_at, convert_finished_at, start_convert_at, convert_type FROM currency_convert_queue WHERE ClientID = ?', $client_id);
    die "no convert request found for ClientID $client_id" unless $convert_request_data && ref($convert_request_data) eq 'HASH' && %$convert_request_data;

    my $convert_type = $convert_request_data->{convert_type};
    hash_copy $vars, $convert_request_data, qw/new_currency convert_started_at convert_finished_at convert_type/;

    my $oldcid2equivalence_data = get_hashes_hash_sql(PPC(ClientID => $client_id), '
        SELECT ccmc.old_cid, ccmc.new_cid, ccmc.money_before, ccmc.money_after
        FROM currency_convert_money_correspondence ccmc
        LEFT JOIN campaigns c ON c.cid = ccmc.old_cid
        WHERE
                ccmc.ClientID = ?
            AND ccmc.was_archived = 0
            AND IFNULL(c.statusEmpty, "No") = "No"
    ', $client_id);

    my @campaigns;
    my @cids = keys %$oldcid2equivalence_data;
    my $campaign_data = get_camp_info(\@cids, $c->client_chief_uid, short => 1);
    for my $campaign (@$campaign_data) {
        my $equivalence_data = $oldcid2equivalence_data->{$campaign->{cid}};
        if ($convert_type eq 'COPY') {
            $campaign->{new_cid} = $equivalence_data->{new_cid};
        }
        $campaign->{sum_rest} = $equivalence_data->{money_before} || 0; # это фишки, которые уже без НДС
        my $sum_rest_converted_without_nds = Currencies::remove_nds($equivalence_data->{money_after} || 0, $client_nds);
        $campaign->{sum_rest_converted} = Currencies::add_bonus($sum_rest_converted_without_nds, $client_discount);
        $campaign->{sum_rest_converted_bonus} = Currencies::calc_bonus($sum_rest_converted_without_nds, $client_discount);
        push @campaigns, $campaign if $campaign->{type} ne 'wallet';
    }
    @campaigns = xsort { \($_->{sum_rest}+0) } @campaigns;
    $vars->{campaigns} = \@campaigns;
    $vars->{interface_blocked} = ($convert_type eq 'COPY') ? 1 : 0;

    $vars->{total_sum} = sum 0, map {$_->{sum_rest}} @$campaign_data;
    $vars->{total_sum_converted} = sum 0, map {$_->{sum_rest_converted}} @$campaign_data;
    $vars->{total_sum_converted_bonus} = sum 0, map {$_->{sum_rest_converted_bonus}} @$campaign_data;

    # multicurrency: добавить более реальный расчёт времени окончания
    my $convert_finish_time = mysql2unix($convert_request_data->{start_convert_at}) + 60 * 60;
    $vars->{convert_finish_forecast} = unix2mysql($convert_finish_time);

    return respond_template($r, $template, 'convert/convert-success.tt2', $vars);
}

1;
