package DoCmdFeed;

=head1 DESCRIPTION

Контроллеры для работы с фидами (динамических медийных объявлений).

=cut

use Direct::Modern;
use base qw/DoCmd::Base/;

use Settings;
use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::I18n;
use Yandex::HashUtils qw/hash_merge/;

use File::Basename qw/basename/;

use Client qw/get_client_limits/;
use Direct::Feeds;
use Direct::ResponseHelper;
use Direct::Validation::Feeds qw/validate_feed is_valid_feed_filename is_valid_feed_business_type/;
use TextTools qw/get_num_array_by_str smartstrip2/;
use HttpTools qw/yandex_domain/;
use CampaignTools;

=head2 cmd_saveFeed

Сохранение фида

=cut
sub cmd_saveFeed
    :Cmd(saveFeed)
    :Rbac(Code => rbac_modify_feeds, ExceptRole => [limited_support])
    :CheckCSRF
    :Description('Сохранение фида')
    :RequireParam(name => 'Require')
    :ParallelLimit(Num => 1, Key => [uid])
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c   vars/};
    my %FORM = %{$_[0]{FORM}};

    return error(iget('Ошибка! Вам недоступна работа с данным разделом.'))
        if (!CampaignTools::is_performance_allowed($c->client_client_id, $login_rights, yandex_domain($r)) &&
            !CampaignTools::is_dynamic_allowed($c->client_client_id, $login_rights, yandex_domain($r)));

    my $callback = $FORM{callback};
    my $is_json = $FORM{json};

    my $feed;
    my $is_new_feed = !$FORM{feed_id};
    if ($is_new_feed) {
        if (Direct::Feeds->check_add_limit(client_id => $c->client_client_id)) {
            my $error = iget('Превышено ограничение на количество фидов');
            return $is_json ? respond_json_or_jsonp($r, {error => {message => $error}}, $callback) : error($error, {});
        }
        $feed = Direct::Model::Feed->new(client_id => $c->client_client_id);
    } else {
        $feed = Direct::Feeds->get_by($c->client_client_id, id => int($FORM{feed_id}))->items->[0];
        if (!$feed) {
            my $error = iget('Фид не найден');
            return $is_json ? respond_json_or_jsonp($r, {error => {message => $error}}, $callback) : error($error, {});
        }
    }

    for my $field (qw/name email login password/) {
        next unless exists $FORM{$field};
        my $stripped = smartstrip2($FORM{$field});
        $FORM{$field} = (defined $stripped && length $stripped) ? $stripped : undef;
    }

    $feed->name($FORM{name});
    $feed->email($FORM{email}) if exists $FORM{email};

    $feed->is_remove_utm($FORM{is_remove_utm} ? 1 : 0);

    if ($is_new_feed) {
        my $business_type = $FORM{business_type};
        if (!is_valid_feed_business_type($business_type)) {
            my $error = iget("Тип бизнеса не поддерживается");
            return $is_json ? respond_json_or_jsonp($r, {error => {message => $error}}, $callback) : error($error, {});
        }
        $feed->business_type($business_type);
    }

    if ($FORM{source} eq 'file') {
        $feed->source('file');
        if ($FORM{feed_file}) {
            # Проверяем здесь, чтобы не тянуть до валидации огромный кусок данных, который может оказаться
            # никому не нужным.
            my $limits = get_client_limits($c->client_client_id);
            my $fh = $FORM{feed_file};
            my $file_size = (stat($fh))[7];
            my $error;

            if ($file_size > $limits->{feed_max_file_size}) {
                $error = iget("Превышено ограничение на размер файла фида");
            } elsif (!is_valid_feed_filename("$FORM{feed_file}")) {
                $error = iget("Допустимы только xls-, xlsx -, xml-, yml-, csv-, tsv-, zip- и gz-файлы");
            }
            if ($error) {
                return $is_json ? respond_json_or_jsonp($r, {error => {message => $error}}, $callback) : error($error, {});
            }
            read($fh, my $content, $file_size);
            $feed->filename(basename("$FORM{feed_file}"));
            $feed->content($content);
        }
    } else {
        $feed->source('url');
        $feed->filename($FORM{url});
        $feed->url($FORM{url});
        $feed->login($FORM{login});

        # На всякий случай принудительно сбрасываем пароль, если поменялся url или имя пользователя.
        if (defined $FORM{password} || $feed->is_url_changed || $feed->is_login_changed) {
            $feed->set_password($FORM{password});
        }
    }

    my $vr = validate_feed($feed);
    if (!$vr->is_valid) {
        $vr->process_descriptions(
            url => { field => iget("'ссылка на фид'") },
            filename => { field => iget("'имя файла'") },
            email => { field => iget("'электронная почта'") },
            login => { field => iget("'логин'") },
        );
        my $error = join "\n", @{$vr->get_error_descriptions};
        return $is_json ? respond_json_or_jsonp($r, {error => {message => $error}}, $callback) : error($error, {});
    }

    Direct::Feeds->new(items => [$feed])->save;

    return $is_json ? respond_json_or_jsonp($r, {result => 'ok'}, $callback) : redirect($r, $SCRIPT, {cmd => 'showFeeds'});
}

=head2 cmd_showFeedHistory

Страничка с историей по фиду. Данные самой истории загружаются через ajaxGetFeedHistory.

=cut

sub cmd_showFeedHistory
    :Cmd(showFeedHistory)
    :Rbac(Code => rbac_can_read_client_data)
    :Description('История загрузок фида')
    :RequireParam(feed_id => 'Require')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c   vars/};
    my %FORM = %{$_[0]{FORM}};

    # Проверка на существование и владение
    my $feed = Direct::Feeds->get_by($c->client_client_id, id => $FORM{feed_id})->items->[0];

    if (!$feed) {
        error(iget('Фид не найден'), {});
    }

    $vars->{feed} = $feed->to_template_hash;

    if (!$feed->{fetch_errors_count}) {
        error(iget('Детализация по фиду отсутствует'), {});
    }

    return respond_bem($r, $c->reqid, $vars, source=>'data3');
}

=head2 cmd_ajaxGetFeedHistory

Данные о загрузках фида

=cut
sub cmd_ajaxGetFeedHistory
    :Cmd(ajaxGetFeedHistory)
    :Rbac(Code => rbac_can_read_client_data)
    :Description('История загрузок фида - данные')
    :RequireParam(feed_id => 'Require')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c   vars/};
    my %FORM = %{$_[0]{FORM}};

    my $feed = Direct::Feeds->get_by($c->client_client_id, id => $FORM{feed_id})->items->[0];
    return respond_http_error($r, 404) unless $feed;

    my $page = int($FORM{page} // 1);
    $page = 1 if $page < 1;

    my $count = int($FORM{count} // 50);
    $count = 50 if $count < 10 || $count > 50;

    my ($feed_history, $total) = Direct::Feeds->get_history($feed->client_id, $feed->id, limit => $count, offset => ($page - 1) * $count);
    my @feed_history_simplified;
    for my $history_item (@$feed_history) {
        my $hash = $history_item->to_hash;
        delete $hash->{_parse_results_json_compressed};
        push @feed_history_simplified, $hash;
    }

    my $data = {
        total_count => $total,
        feed => $feed->to_template_hash,
        feed_history => \@feed_history_simplified,
    };

    return respond_json($r, {result => $data});
}

=head2 cmd_showFeeds

Страничка со списком фидов клиента.

=cut
sub cmd_showFeeds
    :Cmd(showFeeds)
    :Rbac(Code => rbac_can_read_client_data)
    :Description('Страничка со списком фидов клиента')
    :PredefineVars(qw/enable_cpm_deals_campaigns enable_content_promotion_video_campaigns enable_cpm_yndx_frontpage_campaigns/)
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c, $vars, $operator_info) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c   vars operator_info/};
    my %FORM = %{$_[0]{FORM}};

    return error(iget('Ошибка! Вам недоступна работа с данным разделом.'))
        if (!CampaignTools::is_performance_allowed($c->client_client_id, $login_rights, yandex_domain($r)) &&
            !CampaignTools::is_dynamic_allowed($c->client_client_id, $login_rights, yandex_domain($r)));

    $vars->{allow_edit_feeds} = RBAC2::DirectChecks::rbac_modify_feeds($rbac, {UID => $UID, uid => $uid, rights => $login_rights}, undef, $operator_info) ? 0 : 1;
    $vars->{client_limits} = get_client_limits($c->client_client_id);
    $vars->{client_feed_count} = get_one_field_sql(PPC(ClientID => $c->client_client_id), [
        "select count(*) from feeds", where => { ClientID => SHARD_IDS }
    ]);
    if ($vars->{enable_cpm_deals_campaigns}){
        $vars->{new_deals_count} = Client::get_count_received_deals($c->login_rights->{ClientID});
    }
    $vars->{features_enabled_for_client} //= {}; 
    hash_merge $vars->{features_enabled_for_client}, Client::ClientFeatures::get_features_enabled_for_client(
        $c->client_client_id, [qw/support_chat/]);

    hash_merge $vars->{features_enabled_for_client}, Client::ClientFeatures::get_features_enabled_for_client(
        $c->login_rights->{ClientID}, [qw/is_grid_enabled is_hide_old_show_camps is_show_dna_by_default/]);

    return respond_bem($r, $c->reqid, $vars, source => 'data3');
}

=head2 cmd_ajaxGetFeeds

Список фидов клиента

Параметры:
    count=NNN -- отдавать конкретное кол-во (по умолчанию 50)
    search= -- запрос по точному совпадению с указанным feed_id, или по префиксу в name (комбинация id и name для интерфейса)
    limit, offset - параметры для пагинации
    sort -- поле для сортировки
    reverse=1 -- сортировать в обратном порядке
    with_categories=1 -- добавлять категории к фидам

    Проверка прав: в запросе Direct::Feeds->get_by() передается ClientID клиента, и фиды выберутся только по этому клиенту
    Сам клиент проверяется стандартно через rbac_cmd_by_owners()

=cut

sub cmd_ajaxGetFeeds
    :Cmd(ajaxGetFeeds)
    :Rbac(Code => rbac_can_read_client_data)
    :Description('Список фидов клиента')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c   vars/};
    my %FORM = %{$_[0]{FORM}};

    my $page = int($FORM{page} // 1);
    $page = 1 if $page < 1;

    my $count = int($FORM{count} // 50);
    $count = 50 if $count < 10 || $count > 50;

    my %get_params = (
        total_count => 1,
        limit => $count,
        offset => ($page - 1) * $count,
        with_campaigns => 1,
    );

    if (my $search = $FORM{search}) {
        $get_params{filter} = {_OR => [feed_id => $search, name__contains => $search]};
    }

    if (my $sort_field = $FORM{sort}) {
        $get_params{sort}->{$sort_field} = $FORM{reverse} ? 'desc': 'asc';
    }

    my $feeds = [];
    if ($FORM{with_categories}) {
        $get_params{adgroup_ids} = [$FORM{adgroup_ids} =~ /\d+/g] if $FORM{adgroup_ids};
        $feeds = Direct::Feeds->get_with_categories($c->client_client_id, %get_params);
    } else {
        $feeds = Direct::Feeds->get_by($c->client_client_id, %get_params);
    }

    my $data = {
        total_count => $feeds->total,
        items => [map { $_->to_template_hash } @{$feeds->items}],
    };

    return respond_json($r, {result => $data});
}

=head2 cmd_ajaxDeleteFeeds

Удаление фидов

=cut
sub cmd_ajaxDeleteFeeds
    :Cmd(ajaxDeleteFeeds)
    :Rbac(Code => rbac_modify_feeds, ExceptRole => [limited_support])
    :CheckCSRF
    :Description('Удаление фидов')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c   vars/};
    my %FORM = %{$_[0]{FORM}};

    my $data = Direct::Feeds->delete_unused($c->client_client_id, get_num_array_by_str($FORM{feeds_ids}));

    return respond_json($r, {result => $data});
}

1;
