package DoCmdInternalAds;
use Direct::Modern;

use parent 'DoCmd::Base';

use JSON qw( decode_json encode_json );
use List::Util qw( none );
use List::MoreUtils qw( uniq );
use Try::Tiny;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::HTTP qw( http_parallel_request );
use Yandex::I18n;
use Yandex::Validate qw( is_valid_id );

use Campaign qw( del_camp mass_is_camp_deletable_by_hash resume_camp stop_camp );
use Campaign::Types;
use Common qw( get_user_camps_by_sql unarc_camp );
use Direct::ResponseHelper;
use InternalAdManagers;
use Settings;
use Wallet qw();

my %ACTIVE_CAMPAIGNS_FILTER = (
    'c.archived' => 'No',
    'c.statusEmpty' => 'No',
    'c.statusModerate__ne' => 'New',
    'c.statusShow' => 'Yes',
);

my %ACTIVE_ADS_FILTER = (
    'b.statusArch' => 'No',
    'b.statusModerate__ne' => 'New',
    'b.statusShow' => 'Yes',
);

my %CAMPAIGNS_ID_FIELD_NAME_BY_CRITERIA = (
    'cids' => 'cid',
    'order_ids' => 'OrderID',
);

my %ADS_ID_FILTER_BY_CRITERIA = (
    'bids' => 'b.bid__int',
    'banner_ids' => 'b.BannerID__int',
);

sub cmd_showInternalAdsSearchPage :Cmd(showInternalAdsSearchPage)
    :Description('страница поиска материалов внутренней рекламы')
    :Rbac(Role => [super, superreader, internal_ad_admin, internal_ad_manager, internal_ad_superreader])
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   vars   c/};

    return respond_bem($r, $c->reqid, $vars, source => 'data3');
}

sub cmd_internalAdsSearch :Cmd(internalAdsSearch)
    :Description('страница результатов поиска материалов внутренней рекламы')
    :Rbac(Role => [super, superreader, internal_ad_admin, internal_ad_manager, internal_ad_superreader])
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   vars   c/};
    my %FORM = %{$_[0]{FORM}};

    my $object_type = $FORM{object_type};
    unless ($object_type) {
        error( iget('Непонятно, что искать') );
    }

    my $criteria = $FORM{criteria};
    unless ( $criteria ) {
        error( iget('Не задано условие поиска') );
    }

    # что здесь $product_client_ids: undef, если можно всё, или массив ClientID доступных продуктов для фильтрации
    my ( $product_client_id_access, $product_client_ids, $readonly_role );
    if ( $login_rights->{internal_ad_manager_control} ) {
        $product_client_id_access = InternalAdManagers::get_accessible_products( $login_rights->{ClientID} );
        $product_client_ids = [ keys %$product_client_id_access ];
    } elsif ( $login_rights->{superreader_control} || $login_rights->{internal_ad_superreader_control} ) {
        $readonly_role = 1;
    }

    if ( $object_type eq 'campaigns' ) {
        my $result = _search_campaigns( $FORM{campaigns_criteria_type}, $criteria, {
            exact_match => $FORM{exact_match},
            include_inactive_campaigns => $FORM{include_inactive_campaigns},
            product_client_ids => $product_client_ids,
            product_client_id_access => $product_client_id_access,
            readonly_role => $readonly_role,
        } );
        $vars->{campaigns} = $result->{campaigns};
        $vars->{warnings} = $result->{warnings};
        $vars->{info_for_empty_products} = $result->{info_for_empty_products};
    } elsif ( $object_type eq 'adgroups' ) {
        my $result = _search_adgroups( $FORM{adgroups_criteria_type}, $criteria, {
            exact_match => $FORM{exact_match},
            include_inactive_campaigns => $FORM{include_inactive_campaigns},
            product_client_ids => $product_client_ids,
        } );
        $vars->{adgroups} = $result->{adgroups};
        $vars->{warnings} = $result->{warnings};
    } elsif ( $object_type eq 'ads' ) {
        my $result = _search_ads( $FORM{ads_criteria_type}, $criteria, {
            exact_match => $FORM{exact_match},
            include_inactive_campaigns => $FORM{include_inactive_campaigns},
            include_inactive_ads => $FORM{include_inactive_ads},
            product_client_ids => $product_client_ids,
        } );
        $vars->{ads} = $result->{ads};
        $vars->{warnings} = $result->{warnings};
    } else {
        error( iget( 'Нет такого типа объекта: %s', $object_type ) );
    }

    return respond_bem($r, $c->reqid, $vars, source => 'data3');
}

sub cmd_ajaxInternalAdsCampaignAction :Cmd(ajaxInternalAdsCampaignAction)
    :Description('действие с кампанией внутренней рекламы')
    :Rbac(Role => [super, internal_ad_admin, internal_ad_manager])
    :CheckCSRF
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   vars   c/};
    my %FORM = %{$_[0]{FORM}};

    my $cid = $FORM{cid};
    unless ($cid) {
        return error_to( $r, iget( "Не передан необходимый параметр: %s", "cid" ) );
    }

    my $action = $FORM{action};
    unless ($action) {
        return error_to( $r, iget( "Не передан необходимый параметр: %s", "action" ) );
    }

    # что здесь $product_client_ids: undef, если можно всё, или массив ClientID доступных продуктов для фильтрации
    my ( $product_client_id_access, $product_client_ids );
    if ( $login_rights->{internal_ad_manager_control} ) {
        $product_client_id_access = InternalAdManagers::get_accessible_products( $login_rights->{ClientID} );
        $product_client_ids = [ keys %$product_client_id_access ];
    }

    my $initial_search_result = _search_campaigns( cids => $cid, {
        product_client_ids => $product_client_ids,
        product_client_id_access => $product_client_id_access,
    })->{campaigns};

    unless (@$initial_search_result) {
        return error_to( $r, iget( "Нет такой кампании: %s", $cid ) );
    }

    my $campaign = $initial_search_result->[0];
    my $product_client_id = $campaign->{ClientID};
    my $no_rights = sub { return error_to( $r, iget( 'Нет прав на кампанию %s', $cid ) ) };
    if ( defined $product_client_id_access ) {
        if ( !exists $product_client_id_access->{$product_client_id} ) {
            return $no_rights->();
        }
        
        my $access = $product_client_id_access->{$product_client_id};
        if ( $access->{type} ne 'internal_ad_publisher' ) {
            return $no_rights->();
        }
        
        if ( my @place_ids = @{ $access->{place_ids} } ) {
            if ( none { $_ == $campaign->{place_id} } @place_ids ) {
                return $no_rights->();
            }
        }
    }

    if ( $action eq 'resume' ) {
        my $result = resume_camp( $campaign->{uid}, $cid,
            has_operator_super_control => $login_rights->{super_control},
            has_operator_manager_control => $login_rights->{manager_control});
        if ($result) {
            return error_to( $r, $result );
        }
    } elsif ( $action eq 'stop' ) {
        my $result = stop_camp( $campaign->{uid}, $cid,
            has_operator_super_control => $login_rights->{super_control},
            has_operator_manager_control => $login_rights->{manager_control});
        if ($result) {
            return error_to( $r, $result );
        }
    } elsif ( $action eq 'archive' ) {
        my $result = Wallet::arc_camp( undef, $UID, $campaign->{uid}, [$cid] );
        if ( $result->{result} && $result->{error} ) {
            return error_to( $r, $result->{error} );
        }
    } elsif ( $action eq 'unarchive' ) {
        if ( $campaign->{archived} eq 'No' ) {
            return error_to( $r, iget('Кампания не в архиве') );
        }

        unarc_camp( $campaign->{uid}, $cid, UID => $UID );
    } elsif ( $action eq 'delete' ) {
        my $result = del_camp( undef, $cid, $campaign->{uid}, UID => $UID );
        if ($result) {
            return respond_to($r,
                json => [{ result => [] }],
                any  => sub {
                    return redirect($r, $SCRIPT, {cmd => 'showInternalAdsSearchPage'}) unless $FORM{ulogin};
                    return redirect($r, '/dna/grid/campaigns', { ulogin => $FORM{ulogin} });
                }
            );
        } else {
            return error_to( $r, iget("Не удалось удалить кампанию") );
        }
    } else {
        return error_to( $r, iget( "Нет такого действия: %s", $action ) );
    }

    my $search_result = _search_campaigns( cids => $cid, {
        product_client_ids => $product_client_ids,
        product_client_id_access => $product_client_id_access,
    })->{campaigns};

    return respond_to( $r,
        json => [{ result => $search_result }],
        any  => sub {
            return redirect($r, $SCRIPT, {cmd => 'showInternalAdsSearchPage'}) unless $FORM{ulogin};
            return redirect($r, '/dna/grid/campaigns', { ulogin => $FORM{ulogin} });
        }
    );
}

sub _get_words_from_search_phrase {
    my ($phrase) = @_;

    my @words;
    while ( $phrase =~ /(\w+)/g ) {
        push @words, $1;
    }

    return @words;
}

sub _construct_sql_condition_for_fulltext_search {
    my ( $phrase, $field_name, $exact_match ) = @_;

    my @condition;
    if ( $exact_match ) {
        push @condition, "${field_name}__like" => '%' . sql_quote_like_pattern($phrase) . '%';
    } else {
        my @words = _get_words_from_search_phrase($phrase);
        push @condition, '_OR', [ map { ( "${field_name}__like" => '%' . sql_quote_like_pattern($_) . '%' ) } @words ];
    }

    return @condition;
}

sub _search_campaigns {
    my ( $criteria_type, $criteria, $params ) = @_;

    unless ( $criteria_type ) {
        error( iget('Не задано условие поиска') );
    }

    my @warnings;

    my %sql_where = ( 'c.type' => get_camp_kind_types('internal') );

    my %shard = ( shard => 'all' );
    if ( my $client_ids = $params->{product_client_ids} ) {
        if (!@$client_ids) {
            return { campaigns => [], warnings => [ iget('У Вас нет доступа ни к каким продуктам') ] };
        }
        $sql_where{'c.ClientID'} = $client_ids;
        %shard = ( ClientID => $client_ids );
    }

    # логика такая: если здесь массив, а не undef, то это поиск по продуктам;
    # и тогда, если в каком-то из них ничего не нашлось, можно показать ссылку на его грид,
    # чтобы первую кампанию завести, например
    my $all_product_client_ids = undef;

    if ( exists $CAMPAIGNS_ID_FIELD_NAME_BY_CRITERIA{$criteria_type} ) {
        my $ids = _extract_numbers($criteria);

        my ( @valid_ids, @invalid_ids );
        for my $id (@$ids) {
            if ( is_valid_id($id) ) {
                push @valid_ids, $id;
            } else {
                push @invalid_ids, $id;
            }
        }

        if (@invalid_ids) {
            push @warnings, iget( "Это не номера: %s", join( ', ', @invalid_ids) );
            if (!@valid_ids) {
                return { campaigns => [], warnings => \@warnings };
            }

            $ids = \@valid_ids;
        }

        my $id_field_name = $CAMPAIGNS_ID_FIELD_NAME_BY_CRITERIA{$criteria_type};
        $sql_where{"c.$id_field_name"} = $ids;
        %shard = ( $id_field_name => $ids );
    } elsif ( $criteria_type eq 'campaign_name' ) {
        my @cid_lookup_where = _construct_sql_condition_for_fulltext_search( $criteria, 'c.name', $params->{exact_match} );

        # JOIN используется для селективности, потому что индекса по c.type нет
        $sql_where{'c.cid'} = get_one_column_sql( PPC(%shard),[
            'SELECT c.cid',
            'FROM campaigns_internal AS cint',
            'INNER JOIN campaigns AS c ON cint.cid = c.cid',
            WHERE => \@cid_lookup_where,
        ]);
    } elsif ( $criteria_type eq 'places' ) {
        my $place_ids = _extract_numbers($criteria);

        my ( @valid_place_ids, @invalid_place_ids );
        for my $place_id (@$place_ids) {
            if ( is_valid_id($place_id) ) {
                push @valid_place_ids, $place_id;
            } else {
                push @invalid_place_ids, $place_id;
            }
        }

        if (@invalid_place_ids) {
            push @warnings, iget( "Это не номера: %s", join( ', ', @invalid_place_ids) );
            if (!@valid_place_ids) {
                return { campaigns => [], warnings => \@warnings };
            }

            $place_ids = \@valid_place_ids;
        }

        $sql_where{'c.cid'} = get_one_column_sql( PPC(%shard), [
            'SELECT cid FROM campaigns_internal',
            WHERE => { place_id__int => $place_ids }
        ]);
    } elsif ( $criteria_type eq 'products' ) {
        my $product_names = _extract_numbers($criteria);
        my %product_search_where = ( product_name => $product_names );
        if ( my $client_ids = $params->{product_client_ids} ) {
            $product_search_where{ClientID} = $client_ids;
        }
        my $product_name_to_client_id = get_hash_sql( PPC(%shard), [
            'SELECT product_name, ClientID',
            'FROM internal_ad_products',
            WHERE => \%product_search_where,
        ]);

        if ( my @missing_product_names = grep { ! exists $product_name_to_client_id->{$_} } @$product_names ) {
            push @warnings, iget( 'Не найдены продукты: %s', join( ', ', @missing_product_names ) );

            unless (%$product_name_to_client_id) {
                return { campaigns => [], warnings => \@warnings };
            }
        }

        # стоит сохранить порядок, поэтому здесь map { ... } grep { ... } @$product_names
        $sql_where{'c.ClientID'} = $all_product_client_ids = [
            map { $product_name_to_client_id->{$_} }
            grep { exists $product_name_to_client_id->{$_} }
            @$product_names
        ];
    } else {
        error( iget( 'Неизвестный тип критерия: %s', $criteria_type ) );
    }

    if ( !exists $CAMPAIGNS_ID_FIELD_NAME_BY_CRITERIA{$criteria_type} && !$params->{include_inactive_campaigns} ) {
        %sql_where = ( %sql_where, %ACTIVE_CAMPAIGNS_FILTER );
    }

    my %search_opts = (
        join => 'INNER JOIN campaigns_internal cint ON c.cid = cint.cid',
        convert_yes_no_fields => 1,
        remove_nds => 1,
        add_discount_bonus => 1,
        shard => \%shard,
        total_with_wallet => 1,
        order_by => 'c.cid',
    );

    my $result = get_user_camps_by_sql( \%sql_where, \%search_opts );
    my $campaigns = $result->{campaigns};

    my $cids = [ map { $_->{cid} } @$campaigns ];
    my $extra_data = get_hashes_hash_sql( PPC( cid => $cids ), [
        'SELECT c.cid, pr.product_name, cint.place_id',
        'FROM campaigns AS c',
        'INNER JOIN internal_ad_products AS pr ON c.ClientID = pr.ClientID',
        'INNER JOIN campaigns_internal AS cint ON c.cid = cint.cid',
        WHERE => { 'c.cid' => $cids },
    ]);

    my $is_camp_deletable = mass_is_camp_deletable_by_hash($campaigns);

    # убираем из $campaigns такие кампании, у которых у клиентов нет записей в internal_ad_products
    # (исторические клиенты с фичей)
    @$campaigns = grep { exists $extra_data->{ $_->{cid} } } @$campaigns;

    # попатчим status.text в качестве решения по месту; DIRECT-103299
    # после правильного получения трат из БК (DIRECT-92961) надо подумать, чтобы отпилить;
    # ну или, может быть, вообще вся страница уедет в новый интерфейс
    for my $campaign (@$campaigns) {
        if ( $campaign->{archived} ne 'Yes' &&
            $campaign->{statusModerate} eq 'Yes' && $campaign->{statusPostModerate} ne 'No' &&
            $campaign->{statusShow} eq 'Yes' &&
            ( $campaign->{sum} - $campaign->{sum_spent} < $Currencies::EPSILON ) )
        {
            if ( $campaign->{OrderID} ) {
                $campaign->{status}->{text} = iget('Идут показы');
            } else {
                $campaign->{status}->{text} = iget('Идут показы. Идет активизация');
            }
        }
    }

    my $product_client_id_access = $params->{product_client_id_access};
    for my $campaign (@$campaigns) {
        my $cid = $campaign->{cid};
        my $extra_data_for_campaign = $extra_data->{$cid};
        my $place_id = $extra_data_for_campaign->{place_id};
        $campaign->{product_name} = $extra_data_for_campaign->{product_name};
        $campaign->{place_id} = $place_id;
        $campaign->{is_deletable} = $is_camp_deletable->{$cid};

        my $product_client_id = $campaign->{ClientID};

        $campaign->{actions} = _calculate_campaign_actions($campaign,
            readonly_role => $params->{readonly_role},
            product_client_id => $product_client_id,
            product_client_id_access => $product_client_id_access,
        );
    }

    my $info_for_empty_products = undef;
    if ($all_product_client_ids) {
        my %non_empty_products = map { $_ => 1 } uniq( map { $_->{ClientID} } @$campaigns );

        if ( my @empty_product_client_ids = grep { !$non_empty_products{$_} } @$all_product_client_ids ) {
            my %map_of_info_for_empty_products = map { $_->{ClientID} => $_ } @{ get_all_sql( PPC( ClientID => \@empty_product_client_ids ), [
                'SELECT u.uid, u.ClientID, u.login, iap.product_name',
                'FROM users u',
                'INNER JOIN internal_ad_products iap ON u.ClientID = iap.ClientID',
                WHERE => { 'u.ClientID' => SHARD_IDS },
                'ORDER BY product_name'
            ] ) };

            $info_for_empty_products = [ map { $map_of_info_for_empty_products{$_} } @empty_product_client_ids ];
        }
    }

    return { campaigns => $campaigns, warnings => \@warnings, info_for_empty_products => $info_for_empty_products };
}

sub _calculate_campaign_actions {
    my ( $campaign, %params ) = @_;

    if ( $params{readonly_role} ) {
        return [];
    }

    my $product_client_id_access = $params{product_client_id_access};
    my $product_client_id = $params{product_client_id};

    if ( defined $product_client_id_access ) {
        if (! exists $product_client_id_access->{$product_client_id} ) {
            return [];
        }

        my $access = $product_client_id_access->{$product_client_id};
        if ( $access->{type} ne 'internal_ad_publisher' ) {
            return [];
        }

        if ( my @place_ids = @{ $access->{place_ids} } ) {
            if ( none { $_ eq $campaign->{place_id} } @place_ids ) {
                return [];
            }
        }
    }

    my $actions = [];

    if ( $campaign->{archived} eq 'No' && $campaign->{statusShow} ne 'Yes' ) {
        push @$actions, { action => 'resume', name => iget('Включить') };
    }

    if ( $campaign->{archived} eq 'No' && $campaign->{statusShow} eq 'Yes' ) {
        push @$actions, { action => 'stop', name => iget('Остановить') };
    }

    if ( $campaign->{archived} eq 'No' && $campaign->{total} < $Currencies::EPSILON && $campaign->{camp_stopped} ) {
        push @$actions, { action => 'archive', name => iget('Архивировать') };
    }

    if ( $campaign->{archived} eq 'Yes' ) {
        push @$actions, { action => 'unarchive', name => iget('Разархивировать') };
    }

    if ( $campaign->{is_deletable} ) {
        push @$actions, { action => 'delete', name => iget('Удалить') };
    }

    return $actions;
}

sub _search_adgroups {
    my ( $criteria_type, $criteria, $params ) = @_;

    unless ( $criteria_type ) {
        error( iget('Не задано условие поиска') );
    }

    my @warnings;
    my @sql_where = ();

    if ( my $client_ids = $params->{product_client_ids} ) {
        if (!@$client_ids) {
            return { adgroups => [], warnings => [ iget('У Вас нет доступа ни к каким продуктам') ] };
        }
        push @sql_where, 'c.ClientID' => $client_ids;
    }

    if ( $criteria_type eq 'pids' ) {
        my $pids = _extract_numbers($criteria);

        my ( @valid_pids, @invalid_pids );
        for my $pid (@$pids) {
            if ( is_valid_id($pid) ) {
                push @valid_pids, $pid;
            } else {
                push @invalid_pids, $pid;
            }
        }

        if (@invalid_pids) {
            push @warnings, iget( "Это не номера: %s", join( ', ', @invalid_pids) );
            if (!@valid_pids) {
                return { adgroups => [], warnings => \@warnings };
            }

            $pids = \@valid_pids;
        }

        push @sql_where, 'pid__int' => $pids;
    } elsif ( $criteria_type eq 'group_name' ) {
        push @sql_where, _construct_sql_condition_for_fulltext_search( $criteria, 'group_name', $params->{exact_match} );
    } else {
        error( iget( 'Неизвестный тип критерия: %s', $criteria_type ) );
    }

    if ( $criteria_type ne 'pids' && !$params->{include_inactive_campaigns} ) {
        push @sql_where, %ACTIVE_CAMPAIGNS_FILTER;
    }

    my $adgroup_data = get_all_sql( PPC( shard => 'all' ), [
        'SELECT p.pid, p.group_name, p.cid, c.ClientID, c.name AS campaign_name, cint.place_id, u.login, iap.product_name',
        'FROM phrases AS p',
        'INNER JOIN campaigns AS c ON p.cid = c.cid',
        'INNER JOIN campaigns_internal AS cint ON c.cid = cint.cid',
        'INNER JOIN users AS u ON c.ClientID = u.ClientID',
        'INNER JOIN internal_ad_products AS iap ON u.ClientID = iap.ClientID',
        WHERE => \@sql_where,
    ]);

    return { adgroups => $adgroup_data, warnings => \@warnings };
}

sub _search_ads {
    my ( $criteria_type, $criteria, $params ) = @_;

    unless ( $criteria_type ) {
        error( iget('Не задано условие поиска') );
    }

    my @warnings;
    my @sql_where = ();

    if ( my $client_ids = $params->{product_client_ids} ) {
        if (!@$client_ids) {
            return { ads => [], warnings => [ iget('У Вас нет доступа ни к каким продуктам') ] };
        }
        push @sql_where, 'c.ClientID' => $client_ids;
    }

    if ( exists $ADS_ID_FILTER_BY_CRITERIA{$criteria_type} ) {
        my $ids = _extract_numbers($criteria);

        my ( @valid_ids, @invalid_ids );
        for my $id (@$ids) {
            if ( is_valid_id($id) ) {
                push @valid_ids, $id;
            } else {
                push @invalid_ids, $id;
            }
        }

        if (@invalid_ids) {
            push @warnings, iget( "Это не номера: %s", join( ', ', @invalid_ids) );
            if (!@valid_ids) {
                return { campaigns => [], warnings => \@warnings };
            }

            $ids = \@valid_ids;
        }

        push @sql_where, $ADS_ID_FILTER_BY_CRITERIA{$criteria_type} => $ids;
    } elsif ( $criteria_type eq 'count_link' ) {
        my $decoding_result = _decode_bs_count_link($criteria);
        if ( my $error = $decoding_result->{error} ) {
            return { ads => [], warnings => [ iget( 'Ошибка при раскодировании count-ссылки: %s', $error ) ] };
        }

        my $bid = $decoding_result->{bid};
        push @sql_where, 'b.bid__int' => $bid;
    } elsif ( $criteria_type eq 'all_ad_fields' ) {
        my $exact_match = $params->{exact_match};
        push @sql_where, _OR => [
            _construct_sql_condition_for_fulltext_search( $criteria, 'bint.template_variables', $exact_match ),
            _construct_sql_condition_for_fulltext_search( $criteria, 'bint.description', $exact_match ),
        ];
    } else {
        error( iget( 'Неизвестный тип критерия: %s', $criteria_type ) );
    }

    if ( !exists $ADS_ID_FILTER_BY_CRITERIA{$criteria_type} && $criteria_type ne 'count_link' ) {
        unless ( $params->{include_inactive_campaigns} ) {
            push @sql_where, %ACTIVE_CAMPAIGNS_FILTER;
        }

        unless ( $params->{include_inactive_ads} ) {
            push @sql_where, %ACTIVE_ADS_FILTER;
        }
    }

    my $ad_data = get_all_sql( PPC( shard => 'all' ), [
        'SELECT b.bid, b.BannerID, bint.description, p.pid, p.group_name, p.cid,',
        'c.ClientID, c.name as campaign_name, cint.place_id, u.login,',
        'iap.product_name',
        'FROM banners AS b',
        'INNER JOIN banners_internal AS bint ON b.bid = bint.bid',
        'INNER JOIN phrases AS p ON b.pid = p.pid',
        'INNER JOIN campaigns AS c ON p.cid = c.cid',
        'INNER JOIN campaigns_internal AS cint ON c.cid = cint.cid',
        'INNER JOIN users AS u ON c.ClientID = u.ClientID',
        'INNER JOIN internal_ad_products AS iap ON u.ClientID = iap.ClientID',
        WHERE => \@sql_where,
    ]);

    return { ads => $ad_data, warnings => \@warnings };
}

sub _extract_numbers {
    my ($str) = @_;

    return [ split /[ ,;]+/, $str =~ s/^\s+|\s+$//gr ];
}

sub _decode_bs_count_link {
    my ($link) = @_;

    # ручка БК не умеет адреса с query string, так что её надо оторвать вместе со знаком вопроса
    my $uri = URI->new($link);
    $uri->query(undef);
    my $bs_link = $uri->as_string;

    my $payload = encode_json( { link => $bs_link } );
    my $reqnum = Yandex::HTTP::HTTP_FETCH_REQUEST_ID;
    my $bs_results = http_parallel_request( POST => { $reqnum => { url => "http://$Settings::BSRANK_HOST/decode_count", body => $payload } },
        timeout => 20,
        headers => { 'Content-type' => 'application/json' },
    );
    my $bs_result = $bs_results->{$reqnum};
    die "no bs result" unless $bs_result;

    unless ( $bs_result->{is_success} ) {
        my $error_msg;
        try {
            my $content = decode_json( $bs_result->{content} );
            die "no error in bs response" unless exists $content->{error} && $content->{error} ne '';
            $error_msg = $content->{error};
        } catch {
            warn "cannot format error message: $_";
            $error_msg = join( ' / ', $bs_result->{headers}->{Status}, $bs_result->{headers}->{Reason}, $bs_result->{content} );
        };
        return { error => $error_msg };
    }

    my $content = decode_json( $bs_result->{content} );
    die "no bid in bs response" unless exists $content->{ExportID} && $content->{ExportID};
    return { bid => $content->{ExportID} };
}

1;
