package DoCmdReports;

# $Id$

=head1 NAME

    DoCmdReports

=head1 DESCRIPTION

    Функции для управления очередью офлайновых отчётов
    Pdf-отчёты
    Отчёты по Broadmatch
    оффлайновые отчёты для менеджеров/тимлидеров/...

=cut

use strict;
use warnings;

use base qw/DoCmd::Base/;

use List::Util qw/min max/;
use List::MoreUtils qw/uniq none all any/;
use URI::Escape qw/uri_escape_utf8 uri_unescape/;
use Date::Calc;
use Date::Calc qw(
    Localtime
    Today
    Delta_Days
    Add_Delta_YM
    Mktime
    Localtime
    Add_Delta_YMD
    Week_of_Year
    Monday_of_Week
    check_date
    );

use Yandex::HashUtils;
use Yandex::DateTime;
use Yandex::I18n;
use Settings;
use Tools;
use Direct::ResponseHelper;
use Direct::Storage;
use Common qw(:globals :subs);
use Yandex::DBTools;
use Yandex::DBShards;
use RBACDirect;
use RBAC2::DirectChecks;
use PrimitivesIds;
use Client;

use Campaign::Types;
use Reports::Offline;
use Reports::Offline::Postview qw//;
use Reports::ClientPotential;
use Reports::Queue;
use Reports::Checks;
use Currencies;
use GeoTools;

use utf8;

# Order a new report

sub cmd_addPdfReport :Cmd(addPdfReport)
    :Description('заказ PDF отчёта')
    :CheckCSRF
    :Rbac(Code => rbac_cmd_pdf_report)
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   vars   c/};
    my %FORM = %{$_[0]{FORM}};

    $vars->{uid} = $login_rights->{client_chief_uid};
    $vars->{UID} = $UID;

    $vars->{group} = $FORM{grouping} || 'day';
    $vars->{lang} = Yandex::I18n::current_lang();
    $vars->{lang} = 'en' if $vars->{lang} =~ /^tr$/;

    hash_copy $vars, \%FORM, qw/cids type date_to date_from/;

    for my $key (qw/date_to date_from/) {
        next unless defined $vars->{$key};
        $vars->{$key} =~ s/^(\d{2})\-(\d{2})\-(\d{4})$/$3-$2-$1/g;
    }

    if ($vars->{type} eq 'video') {
        $vars->{ClientID} = $c->client_client_id;
        $vars->{goals} = $FORM{goals};
        $vars->{campaign_type} = $FORM{campaign_type};
        $vars->{banner_type} = $FORM{banner_type};
        $vars->{group_by} = $FORM{grouping};
        $vars->{report_type} = $FORM{report_type};
    } elsif ($vars->{type} eq $Reports::Offline::Postview::POSTVIEW_REPORT_TYPE) {
        if (!Client::ClientFeatures::has_postview_conversions_report($c->client_client_id)) {
            return respond_json($r, {errors => [iget('Нет прав для получения отчета заданного типа!')]});
        }
    }

    # В запросе отчета по смарт-баннерам для суперов могут быть переданы дополнительные опции
    # + для этом случае в отчет всегда добавляются данные об OrderID
    if ($vars->{type} eq 'performance' && $login_rights->{role} eq 'super') {
        $vars->{add_order_id} = 1;
        $vars->{add_vendor} = $FORM{add_vendor} ? 1 : 0;
        $vars->{add_categories} = $FORM{add_categories} ? 1 : 0;
    }

    # Саппортам разрешаем заказ отчётов только по ДРФ и динамическим кампаниям
    if ($login_rights->{role} eq 'support' && ($vars->{type} // '') !~ /^(?:bm|dynamic)$/) {
        return respond_json($r, {errors => [iget('Нет прав для выполнения данной операции!')]});
    }

    if (my @errors = Reports::Offline::validate_report_params($vars)) {
        return respond_json($r, {errors => \@errors});
    }

    Reports::Offline::place_report_order($vars);

    return respond_json($r, {ok => 1});
}

# Get a report by unique link
sub cmd_loadPdfReport :Cmd(loadPdfReport)
    :Description('выгрузка PDF отчёта')
    :Rbac(Code => rbac_cmd_pdf_report)
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   vars/};
    my %FORM = %{$_[0]{FORM}};

    my $rid = $FORM{rid};
    my $type = $FORM{type} || 'pdf';

    return error(iget('Указан неверный номер отчёта')) if !$rid;
    return error(iget('Указан неверный тип отчёта'))  if !$Reports::Offline::REPORT_CLASS{$type};

    my ($mime_type, $report_file) = Reports::Offline::get_report_data($login_rights->{client_chief_uid}, $type, $rid);
    return error(iget('Отчета с такими параметрами не существует'))  if !$report_file;

    return respond_data($r, $report_file, $mime_type);
}



sub cmd_repairPdfReport :Cmd(repairPdfReport)
    :Description('перезапуск PDF отчёта')
    :CheckCSRF
    :Rbac(Code => rbac_cmd_pdf_report)
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   vars/};
    my %FORM = %{$_[0]{FORM}};

    my $type = $FORM{type} || 'pdf';
    error(iget("Неподдерживаемый тип отчёта"))  if !$Reports::Offline::REPORT_CLASS{$type};

    Reports::Offline::requeue_report_order($login_rights->{client_chief_uid}, $type => $FORM{rid});

    if ($FORM{retpath}) {
        return redirect($r, $FORM{retpath});
    } else {
        return redirect($r, "$SCRIPT?cmd=newPdfReport&".join('&', map { $_."=". uri_escape_utf8( $vars->{$_}||$FORM{$_} ) } grep {defined $vars->{$_}||$FORM{$_} } qw/ulogin/ ).'#list');
    }
}


sub cmd_newPdfReport :Cmd(newPdfReport)
    :Description('заказ PDF отчёта')
    :Rbac(Code => rbac_cmd_pdf_report)
    :PredefineVars(qw/enable_cpm_deals_campaigns enable_content_promotion_video_campaigns enable_cpm_yndx_frontpage_campaigns/)
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   vars   c/};
    my %FORM = %{$_[0]{FORM}};

    my $client_id = get_clientid(uid => $uid);
    my $client_nds = get_client_NDS($client_id);
    my $client_discount = get_client_discount($client_id);

    # Показать пользователю только те кампании, по которым можно построить отчёт
    my $user_camps = get_user_camps($c->client_chief_uid,
        mediaType => get_camp_kind_types("pdf_report"),
        BS_active => 1,
        no_subjects => 1,
        client_nds => $client_nds,
        client_discount => $client_discount,
    );
    my @orders = sort {
        (($b->{is_active} // 0) != 0) <=> (($a->{is_active} // 0) != 0)
            || $a->{cid} <=> $b->{cid}
    } grep {
        # performance: хорошо бы делать не по вызову на кампанию, а сделать массовую версию на основе rbac_is_owner + rbac_check_allow_show_stat_camps
        !RBAC2::DirectChecks::rbac_cmd_showCampStat( $rbac, {UID => $UID, cid => $_->{cid}} ) &&
        !($login_rights->{role} eq 'media' && !(rbac_is_scampaign($rbac, $_->{cid}) || rbac_is_agencycampaign($rbac, $_->{cid})))
    } @{$user_camps->{campaigns} || []};

    $vars->{orders} = \@orders;

    $vars->{goals_list} = [ map { $_->{name} //= iget('Удаленная цель ID-%s', $_->{goal_id}); $_ } @{Stat::Tools::orders_goals(OrderID => [map { $_->{OrderID} } @orders])} ];

    $vars->{last_month_date} = sprintf "%04d-%02d-%02d", Date::Calc::Add_Delta_YM((Date::Calc::Today), 0,-1);

    for my $order (@{$vars->{orders}}) {
        Campaign::correct_sum_and_total($order);
        $order->{isAvailableForPostview} = _get_availability_for_postview_report($order);
    }

    if ($vars->{enable_cpm_deals_campaigns}){
        $vars->{new_deals_count} = Client::get_count_received_deals($c->login_rights->{ClientID});
    }

    $vars->{features}->{video_report} = Client::ClientFeatures::has_video_report_feature($c->login_rights->{ClientID});

    $vars->{features_enabled_for_client} //= {};
    hash_merge $vars->{features_enabled_for_client}, Client::ClientFeatures::get_features_enabled_for_client(
        $c->client_client_id, [qw/support_chat/]);

    hash_merge $vars->{features_enabled_for_client}, Client::ClientFeatures::get_features_enabled_for_client(
        $c->login_rights->{ClientID}, [qw/is_grid_enabled is_hide_old_show_camps is_show_dna_by_default/]);

    return respond_bem($r, $c->reqid, $vars, source => 'data3');
}

sub _get_availability_for_postview_report {
    my ($order) = @_;

    my $is_autobudget = $order->{strategy_name} =~ /autobudget/;
    my $has_goal_id = $order->{strategy_decoded}{goal_id} || @{$order->{meaningful_goals}};
    my $is_context_or_both = $order->{platform} eq 'both' || $order->{platform} eq 'context';

    return $is_autobudget && $has_goal_id && $is_context_or_both ? 1 : 0;
}

# Show the list of all reports
sub cmd_listPdfReport :Cmd(listPdfReport)
    :Description('список PDF отчётов')
    :Rbac(Code => rbac_cmd_pdf_report)
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   vars/};
    my %FORM = %{$_[0]{FORM}};

    $vars->{uid} = $login_rights->{client_chief_uid};
    $vars->{UID} = $UID;
    $vars->{FORM} = \%FORM;

    my @valid_types = grep {$_ ne 'offline'} keys %Reports::Offline::REPORT_CLASS;
    my $reports = Reports::Offline::get_reports_list($vars->{uid}, type => \@valid_types);

    # Уберем из списка отчеты перформанса для суперов, если это не супер
    my $is_super = ($login_rights->{role} eq 'super') ? 1 : 0;
    $reports = [grep {$is_super || !$_->{add_order_id}} @$reports];

    my @all_cids = uniq map { split /\s*,\s*/, $_->{cids} } @$reports;
    my $serviced_camps = get_hash_sql(PPC(cid => \@all_cids), ["SELECT cid FROM campaigns",
                                                                 WHERE => {_OR => { AgencyUID__gt => 0,
                                                                                    ManagerUID__gt => 0},
                                                                           cid => SHARD_IDS}]);
    # Drop bad reports
    # доступ к отчету есть:
    # если есть доступ к статистике всех кампаний из отчета по RBAC
    # И, если пользователь медиапланер, в отчете не ни одной самоходной кампании
    $reports = [grep { !( RBAC2::DirectChecks::rbac_cmd_showCampStat( $rbac, {UID => $UID, cid => $_->{cids}} )  ||
                          $login_rights->{role} eq 'media' && any { !exists $serviced_camps->{$_} } split(/\s*,\s*/, $_->{cids}) ) } @$reports];

    my $params          = hash_cut(\%FORM, qw/ulogin cmd bpp/);
    $vars->{url}        = $SCRIPT.'?'. join '&', map {$_.'='.uri_escape_utf8($params->{$_})} grep {$params->{$_}} keys %$params;

    # заполняет $vars переменными, связанными с отчетами
    populate_report_vars($vars, $reports);

    return respond_json($r, $vars);
}

# Remove the reportfrom order list and unlink the pdf if exists
sub cmd_delPdfReport :Cmd(delPdfReport)
    :Rbac(Code => rbac_cmd_pdf_report)
    :CheckCSRF
    :Description('удаление pdf-отчета')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   vars/};
    my %FORM = %{$_[0]{FORM}};

    my $rid = $FORM{rid};
    my $type = $FORM{type} || 'pdf';

    Reports::Offline::delete_report($login_rights->{client_chief_uid}, $type, $rid);

    if ($FORM{retpath}) {
        return redirect($r, $FORM{retpath});
    } else {
        return redirect($r, "$SCRIPT?cmd=newPdfReport&".join('&', map { $_."=". uri_escape_utf8( $vars->{$_}||$FORM{$_} ) } grep {defined $vars->{$_}||$FORM{$_} } qw/ulogin/ ).'#list');
    }
}

sub cmd_newClientPotentialReport :Cmd(newClientPotentialReport)
    :Description('заказ отчёта "Потенциал клиента"')
    :Rbac(Role => [super, superreader, media])
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   vars   c/};
    my %FORM = %{$_[0]{FORM}};

    $vars->{UID} = $UID;

    $vars->{currencies} = [get_currencies_list()];

    return respond_bem($r, $c->reqid, $vars, source => 'data3');
}

sub cmd_addClientPotentialReport :Cmd(addClientPotentialReport)
    :Description('обработка формы заказ отчёта "Потенциал клиента"')
    :Rbac(Role => [super, media])
    :CheckCSRF
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS vars/};
    my %FORM = %{$_[0]{FORM}};

    my $form_data = hash_merge( hash_cut(\%FORM, qw/ cids logins date_from date_to date_group client_name category_name report_date ext_phrases
                                                     geo geo_stat_flag ext_phrases_geo ext_phrases_geo_flag currency use_common_minus_words
                                                     remove_phrases_operators remove_ext_phrases_operators/),
                                {operator_uid => $UID, UID => $UID} );
    if ($FORM{use_turkish_advq}) {
        $form_data->{lang} = 'tr';
    }

    my @errors = Reports::ClientPotential::check_report_order($form_data);

    if ( @errors ) {
        return respond_json($r, {errors => \@errors});
    } else {
        Reports::ClientPotential::create_report(hash_merge {}, $form_data);

        return respond_json($r, {ok => 1});
    }
}

sub cmd_listClientPotentialReport :Cmd(listClientPotentialReport)
    :Description('список отчётов "Потенциал клиента"')
    :Rbac(Role => [super, superreader, media])
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   vars/};
    my %FORM = %{$_[0]{FORM}};

    my $page = int($FORM{page} // 0) || 1;
    my $page_size = 10;

    my $sort_sql = 'ORDER BY ' .(
                    ($FORM{sort} && $FORM{sort} =~ /^(id|create_time)$/)
                    ? sql_quote_identifier($FORM{sort}) . ($FORM{sort_reverse} ? ' DESC' : ' ASC')
                    : 'id DESC' );
    my $limit_sql = 'LIMIT ' . ($page-1)*$page_size . ", $page_size";

    my $reports_num = get_one_field_sql(PPC(uid => $UID), "SELECT COUNT(*)
                                                             FROM stat_reports
                                                            WHERE uid=? AND report_type = 'client_potential'",
                                                                  $UID) || 0;
    my $pages_num = int($reports_num / $page_size) + ($reports_num % $page_size ? 1 : 0);

    my $reports = get_all_sql(PPC(uid => $UID),
        "SELECT id, cids, date_from, date_to, status, IF(status = 'ready' AND report_data_parts_qty = 0, 'Yes', 'No') as status_no_data,
                ready_time, create_time, extra_opt_compressed, uid, operator_uid, report_data_format
           FROM stat_reports
          WHERE uid=? AND report_type = 'client_potential'
                $sort_sql $limit_sql",
                $UID );

    foreach my $rep ( @$reports) {
        Reports::Queue::_parse_extra_opt($rep);
        if ($rep->{status} eq 'ready') {
            $rep->{report_filename} = Reports::ClientPotential::get_report_filename($rep);
        }

        delete $rep->{extra_opt_compressed};
        delete $rep->{ext_phrases};
        delete $rep->{extra_opt}->{ext_phrases} if $rep->{extra_opt};
    }

    return respond_json($r, {page => $page,
                             pages_num => $pages_num,
                             reports => $reports});
}

sub cmd_repairClientPotentialReport :Cmd(repairClientPotentialReport)
    :Rbac(Role => [super, media])
    :CheckCSRF
    :Description('перезапуск генерации отчета "Потенциал клиента"')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS     vars/};
    my %FORM = %{$_[0]{FORM}};

    my $rid = $FORM{rid};

    my $queue = new Reports::Queue(type => 'client_potential');
    my $success = $queue->repair_report($UID, $rid);

    return respond_json($r, $success ? {ok => 1} : {errors => [iget('Неизвестная ошибка')]});
}

sub cmd_delClientPotentialReport :Cmd(delClientPotentialReport)
    :Rbac(Role => [super, media])
    :CheckCSRF
    :Description('удаление отчета "Потенциал клиента"')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS     vars/};
    my %FORM = %{$_[0]{FORM}};


    my $rid = $FORM{rid};
    my $queue = new Reports::Queue(type => 'client_potential');
    my $report = $queue->get_report_info($UID, $rid);

    my @errors = ();
    if ($report->{operator_uid} == $UID) {
        my $success = $queue->delete_report($UID, $report);
        push @errors, iget('Неизвестная ошибка') unless $success;
    } else {
        push @errors, iget('У вас нет прав на удаление отчета');
    }

    return respond_json($r, @errors ? {errors => \@errors} : {ok => 1});
}

sub cmd_loadClientPotentialReport :Cmd(loadClientPotentialReport)
    :Description('выгрузка отчёта "Потенциал клиента"')
    :Rbac(Role => [super, superreader, media])
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS vars/};
    my %FORM = %{$_[0]{FORM}};

    my $rid = $FORM{rid} or error "No report id specified\n";

    ($rid, my $report_uid) = get_one_line_array_sql(PPC(shard => 'all'), "
            SELECT id, uid
              FROM stat_reports
             WHERE id=? AND report_data_parts_qty > 0 AND status='ready' AND report_type = 'client_potential'
                ", $rid);

    my ($report_data, $report_data_format) = Reports::ClientPotential::get_report($report_uid, $rid)
            or error(iget('Отчета с такими параметрами не существует'));
    my $filename;
    if ($r->uri && $r->uri =~ /\/([^\/?]+)(\?|$)/) {
        $filename = $1;
    }

    return respond_data($r, $report_data, ":$report_data_format", $filename);
}


sub cmd_getCampsGeo :Cmd(getCampsGeo)
    :Description('получить объединенный гео по кампаниям')
    :Rbac(Role => [super, media, superreader])
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   vars   c/};
    my %FORM = %{$_[0]{FORM}};

    my $check_info = {operator_uid => $UID};
    my (@cids, @errors);
    if (exists $FORM{cids}) {
        my @cids_errors = Reports::Checks::check_cids($FORM{cids}, $check_info, is_required => 0, check_rights => 1, no_archive => 1 );
        if (@cids_errors) {
            push @errors, @cids_errors;
        } else {
            @cids = split /,/, $check_info->{cids};
        }
    } elsif ($FORM{logins}) {
        my @logins_errors = Reports::Checks::check_logins($FORM{logins}, $check_info, is_required => 0, check_rights => 1);
        if (@logins_errors) {
            push @errors, @logins_errors;
        } else {
            my @logins = split /,/, $check_info->{logins};
            my $login2uid = get_login2uid(login => \@logins);
            @cids = @{ get_one_column_sql(PPC(uid => [values %$login2uid]), ["select cid from campaigns", where => { uid => SHARD_IDS, statusEmpty => 'No'}]) };
        }
    }
    return respond_json($r, {errors => \@errors}) if @errors;

    my @geo = ();
    if (@cids) {
        my @phrases_geo = @{get_one_column_sql(PPC(cid => \@cids), ["select distinct geo from phrases",
                                                                      where => {cid => SHARD_IDS,
                                                                                geo__is_not_null => 1}])};
        push @geo, uniq map { GeoTools::substitute_temporary_geo($_) || 0 } @phrases_geo;
    } else {
        push @geo, 0;
    }

    #TODO: как появится поддержка в транке - заменить ru на api
    my $geo_union = get_targetings_union(\@geo, {tree => 'ru'});

    return respond_json($r, {geo => $geo_union, geoText => get_geo_names($geo_union)});
}

=head2 cmd_downloadCopyCampReport

    Скачивание отчётов по скопированным кампаниям для суперпользователей и менеджеров
    Скачать можно два типа отчетов - по скопированным баннерам и фразам.
    Заказываются такие отчёты в cmd_copyCamp, а формируются скриптом ppcCampCopyReportsQueue.pl

=cut

sub cmd_downloadCopyCampReport : Cmd(downloadCopyCampReport)
    :Rbac(Role => [super, superreader, manager], AllowDevelopers => 1)
    :Description('скачивание отчётов по скопированным кампаниям')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   vars   c/};
    my %FORM = %{$_[0]{FORM}};

    my $client_id =  $FORM{clid};
    my $report_name = $FORM{name};

    return error(iget('Некорректные параметры отчета')) unless ($client_id && $report_name);

    my $storage = Direct::Storage->new();
    my $info = $storage->get_file('camp_copy_report', ClientID => $client_id, filename => $report_name);

    return error(iget('Отчет не найден')) unless $info;

    my $data = $info->content();
    my $mds_report_name = $info->filename();

    return respond_data($r, $data, 'text/csv', "$mds_report_name.csv");
}

1;
