package DoCmdRetargeting;

=head1 NAME

DoCmdRetargeting - работа с условиями ретаргетинга
https://jira.yandex-team.ru/browse/DIRECT-15172

=cut

# $Id$

use Direct::Modern;

use base qw/DoCmd::Base/;

use List::MoreUtils qw/any all uniq part/;

use Settings;
use Direct::ResponseHelper;
use Direct::PredefineVars;
use RBACElementary;
use Yandex::DBTools;
use Primitives;
use Common qw/get_user_camps_name_only/;
use PrimitivesIds;
use Retargeting;
use User;
use Client;
use BSAuction;
use Yandex::I18n;
use Yandex::SendSMS;
use Yandex::IDN;
use Yandex::HashUtils;
use Yandex::Validate qw/is_valid_id/;

use Direct::Model::RetargetingCondition;
use Direct::RetargetingConditions;
use Direct::Validation::RetargetingConditions qw/
    validate_retargeting_conditions
    validate_retargeting_conditions_for_client
/;

=head2 ajaxGetGoalsForRetargeting

     Выдаем для логина все доступные цели для ретаргетинга
     за целями онлайново ходим в метрику

     Праметры:
        для клиента - без параметров, возвращаем все его цели
        для менеджера/агентства/супера/... - ulogin, возвращаем по заданному клиенту

     Возвращаем json-массив хешей целей:
     [
         {
             allow_to_use => 0,
             goal_domain => ,
             goal_id => 1893109,
             goal_name => "Шаг 1",
             goal_type => "goal"
         },
         {
             allow_to_use => 0,
             goal_domain => ,
             goal_id => 1893100,
             goal_name => "First Goal"
             goal_type => "segment"
         }
     ]

=cut

sub cmd_ajaxGetGoalsForRetargeting
    :Cmd(ajaxGetGoalsForRetargeting)
    :Rbac(Code => [rbac_cmd_by_owners])
    :Description('Возвращаем цели метрики для ретаргетинга')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c/};

    my $client_uids = rbac_get_client_uids_by_clientid($c->client_client_id);
    my $all_retargeting_conditions = Retargeting::get_retargeting_conditions(ClientID => $c->client_client_id);
    my $metrika_goals_for_uids = Retargeting::get_all_goals_by_uid($client_uids, [ values %$all_retargeting_conditions ]);

    return respond_json($r, $metrika_goals_for_uids);
}

=head2 ajaxGetRetCondWithGoals

Возвращаем список всех условий ретаргетинга с целями для клиента в виде:

    {
       "ret_cond_id - 1": {
           "condition_name": "...",
           "condition_desc": "...",
           "ClientID": "11111",
           "ret_cond_id": "ret_cond_id - 1",
           "is_accessible": 1,
           "condition": ... some data structure describing used goals and their relationships ...
       },
       ...
    }

=cut

sub cmd_ajaxGetRetCondWithGoals
    :Cmd(ajaxGetRetCondWithGoals)
    :Rbac(Code => [rbac_cmd_by_owners])
    :Description('Возвращаем список всех условий ретаргетинга c целями для клиента')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c) = @{$_[0]}{
        qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c/};

    my %FORM = %{$_[0]{FORM}};
    return respond_json($r, {result => 'error', error_text => 'ret_cond_id is invalid'}) unless is_valid_id($FORM{ret_cond_id});

    my $client_uids = rbac_get_client_uids_by_clientid($c->client_client_id);
    my $ret_cond = Retargeting::get_retargeting_conditions(ClientID => $c->client_client_id, ret_cond_id => [ $FORM{ret_cond_id} ], with_campaigns => 1)->{$FORM{ret_cond_id}};
    return respond_json($r, {}) unless %{$ret_cond // {}};

    my %goals = map { $_->{goal_id} => $_ } @{ Retargeting::get_all_goals_by_uid($client_uids, [ $ret_cond ]) };
    foreach my $goal (map { @{$_->{goals}} } @{$ret_cond->{condition}}) {
        hash_merge($goal, hash_cut($goals{$goal->{goal_id}}, qw/allow_to_use goal_domain goal_name goal_subtype uploading_source_id/));
    }
    $ret_cond->{is_used} = scalar @{(delete($ret_cond->{campaigns}) // [])} ? 1 : 0;

    return respond_json($r, $ret_cond);
}

# --------------------------------------------------------------------

=head2 ajaxSaveRetargetingCond

Сохраняем условие ретаргетинга

На входе в C<json_retargeting_condition>:

    {
      "ret_cond_id": 111,
      "condition_name": "...",
      "condition_desc": "...",
      "condition": [
          {
              "type": "<all|or|not>",
              "goals": [
                  {"goal_id": 349, "time": 30, "goal_type": "<goal|segment>"},
                  ...
              ]
          },
          ...
      ]
    }

=cut
sub cmd_ajaxSaveRetargetingCond
    :Cmd(ajaxSaveRetargetingCond)
    :Rbac(Code => [rbac_cmd_by_owners], ExceptRole => [media])
    :CheckCSRF
    :Description('Сохранение условий ретаргетинга')
    :RequireParam(json_retargeting_condition => 'ajaxSaveRetargetingCond')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c/};
    my %FORM = %{$_[0]{FORM}};

    my $user_ret_cond = $FORM{json_retargeting_condition};

    my $ex_ret_cond_by = Direct::RetargetingConditions->get_by(client_id => $c->client_client_id)->items_by;

    my $ret_cond;
    if (my $ex_ret_cond = $ex_ret_cond_by->{ $user_ret_cond->{ret_cond_id} || 0 }) {
        $ret_cond = $ex_ret_cond->clone(old => $ex_ret_cond);
    } else {
        $ret_cond = Direct::Model::RetargetingCondition->new(id => 0, client_id => $c->client_client_id);
    }

    eval {
        Retargeting::prepare_user_ret_cond($user_ret_cond);

        $ret_cond->condition_name($user_ret_cond->{condition_name} // '');
        $ret_cond->condition_desc($user_ret_cond->{condition_desc} // '');
        $ret_cond->condition($user_ret_cond->{condition});
        1;
    } or do {
        return respond_json($r, {result => 'error', error_type => 'invalid_data', error_text => iget("Ошибка в пользовательских данных")});
    };

    # Разделим условия ретаргетинга на две группы: новые/изменившиеся и не изменившиеся
    my ($checked_ret_conds, $remaining_ret_conds) = part { !$_->id || $_->is_changed ? 0 : 1 } ($ret_cond, (grep { $_->id != $ret_cond->id } values %$ex_ret_cond_by));

    my $vr = validate_retargeting_conditions_for_client($checked_ret_conds // [], $remaining_ret_conds);
    if (!$vr->is_valid) {
        $vr->process_objects_descriptions(Direct::RetargetingConditions->WEB_FIELD_NAMES);
        return respond_json($r, {result => 'error', error_type => 'error', error_text => $vr->get_error_descriptions->[0]});
    }

    my $logic = Direct::RetargetingConditions->new([$ret_cond]);
    if (!$ret_cond->id) { $logic->create() } else { $logic->update() };

    return respond_json($r, {result => 'ok', ret_cond_id => $ret_cond->id});
}

# --------------------------------------------------------------------

=head2 showRetargetingCond

    Показ и редактирование условий ретаргетинга

=cut

sub cmd_showRetargetingCond
    :Cmd(showRetargetingCond)
    :Rbac(Code => [rbac_cmd_by_owners])
    :Description('Показ условий ретаргетинга')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c, $cvars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c vars/};
    my %FORM = %{$_[0]{FORM}};

    my $vars = {};

    # Предустановленные переменные: Доступность Я.Аудиторий. На странице просмотра условий ретаргетинга аудитории всегда доступны.
    # Требуется выставление для интерфейса, так как у них код универсальный.
    $vars->{is_audience_enabled} = 1;

    $vars->{all_retargeting_conditions} = Retargeting::get_retargeting_conditions(ClientID => $c->client_client_id);
    my $bids_retargeting = Retargeting::get_group_retargeting(ret_cond_id => [keys %{ $vars->{all_retargeting_conditions} }],
                                                              ClientID => $c->client_client_id);

    my $campaigns = get_all_sql(PPC(ClientID => $c->client_client_id), ["
        SELECT p.pid, c.cid, c.name,
                MAX(b.BannerID) AS BannerID
        FROM phrases p
            JOIN campaigns c ON c.cid = p.cid
            JOIN banners b ON p.pid = b.pid
        ", WHERE => {
            'p.pid' => [keys %$bids_retargeting],
            'c.statusEmpty' => 'No'
        },
        "GROUP BY p.pid"
    ]);

    my %campaigns_by_pid;
    foreach my $c (@$campaigns) {
        push @{$campaigns_by_pid{$c->{pid}}}, $c;
    }

    my $ret_cond_camp_info = Retargeting::find_camps_used_ret_cond($c->client_client_id);
    my $ret_cond_cids;
    foreach my $ret_id (keys %$ret_cond_camp_info){
        my $arr = $ret_cond_camp_info->{$ret_id};
        for my $hash (@$arr){
            push @{$ret_cond_cids->{$ret_id}}, $hash->{cid};
        }
    }
    my %campaigns = map {
        $_->{cid} => $_
    } @{ get_user_camps_name_only($c->client_chief_uid, {cid => [uniq map { @$_ } values %$ret_cond_cids], add_archived => 1}) };

    # забираем клики/показы по условиям из БК
    my $groups_for_bs = [];
    for my $pid (keys %$bids_retargeting) {
        next unless $campaigns_by_pid{$pid};

        push @$groups_for_bs, map {{
            bid => $_->{bid},
            pid => $pid,
            BannerID => $_->{BannerID},
            image_BannerID => $_->{image_BannerID},
            phrases => [],
            retargetings => $bids_retargeting->{$pid}
        }} grep {$_->{BannerID}} @{$campaigns_by_pid{$pid}};
    }

    my %shows_clicks_by_ret_cond_id;
    if (@$groups_for_bs) {
        bs_get_phrases_statuses($groups_for_bs, {update_retargetings => 1});
        for my $group (@$groups_for_bs) {
            for my $ret_row (@{ $group->{retargetings} }) {
                $shows_clicks_by_ret_cond_id{ $ret_row->{ret_cond_id} }->{ctx_shows} += $ret_row->{ctx_shows} || 0;
                $shows_clicks_by_ret_cond_id{ $ret_row->{ret_cond_id} }->{ctx_clicks} += $ret_row->{ctx_clicks} || 0;
            }
        }
    }

    for my $ret_cond_id (keys %{ $vars->{all_retargeting_conditions} }) {
        if (exists $ret_cond_cids->{$ret_cond_id}) {
            my ($ctx_shows, $ctx_clicks) = (0, 0);
            if (exists $shows_clicks_by_ret_cond_id{$ret_cond_id}) {
                $ctx_shows = $shows_clicks_by_ret_cond_id{ $ret_cond_id }->{ctx_shows};
                $ctx_clicks = $shows_clicks_by_ret_cond_id{ $ret_cond_id }->{ctx_clicks};
            }
            my $ctx_ctr = $ctx_shows > 0 ? $ctx_clicks / $ctx_shows : 0;

            hash_merge $vars->{all_retargeting_conditions}->{$ret_cond_id}, {
                is_used_in_banner => 1
                , campaigns => [map { $campaigns{$_} } @{$ret_cond_cids->{$ret_cond_id}}],
                , ctx_shows => $ctx_shows
                , ctx_clicks => $ctx_clicks
                , ctx_ctr => sprintf("%0.2f", $ctx_ctr * 100)
            };
        }
    }

    $vars->{enable_cpm_deals_campaigns} = Direct::PredefineVars::_enable_cpm_deals_campaigns($c);
    if ($vars->{enable_cpm_deals_campaigns}){
        $vars->{new_deals_count} = Client::get_count_received_deals($c->login_rights->{ClientID});
    }
    $vars->{enable_content_promotion_video_campaigns} = Direct::PredefineVars::_enable_content_promotion_video_campaigns($c);
    $vars->{enable_cpm_yndx_frontpage_campaigns} = Direct::PredefineVars::_enable_cpm_yndx_frontpage_campaigns($c);

    $vars->{features_enabled_for_operator_all} = $cvars->{features_enabled_for_operator_all};
    $vars->{features_enabled_for_client_all} = $cvars->{features_enabled_for_client_all};
    
    $vars->{features_enabled_for_client} //= {};
    hash_merge $vars->{features_enabled_for_client}, Client::ClientFeatures::get_features_enabled_for_client(
        $c->client_client_id,
        [qw/support_chat b2b_balance_cart lal_segments_enabled retargeting_only_lal_enabled/]);

    hash_merge $vars->{features_enabled_for_client}, Client::ClientFeatures::get_features_enabled_for_client(
        $c->login_rights->{ClientID}, [qw/is_grid_enabled is_hide_old_show_camps is_show_dna_by_default/]);

    return respond_bem($r, $c->reqid, $vars, source => 'data3');
}

# --------------------------------------------------------------------

=head2 requestAccessToMetrikaCounters

    Запрос на доступ к счетчикам метрики

    https://jira.yandex-team.ru/browse/CONV-4701

=cut

sub cmd_requestAccessToMetrikaCounters
    :Cmd(requestAccessToMetrikaCounters)
    :Rbac(Code => [rbac_cmd_by_owners], ExceptRole => [media])
    :CheckCSRF
    :Captcha(Key => [UID], Freq => 1, Interval => 60, MaxReq => 5)
    :Description('Запрос на доступ к счетчикам метрики')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c/};
    my %FORM = %{$_[0]{FORM}};

    unless ($FORM{owner} || $FORM{goals} || $FORM{comment}) {
        # просто показ формы
        my $vars = {
            emails => [grep {defined $_ && length($_) > 0} map {$_->{email}} @{get_valid_emails($uid) || []}],
            sms_phone => $login_rights->{is_any_client} ? sms_check_user($uid, $c->user_ip) : undef
        };

        $vars->{enable_cpm_deals_campaigns} = Direct::PredefineVars::_enable_cpm_deals_campaigns($c);
        if ($vars->{enable_cpm_deals_campaigns}){
            $vars->{new_deals_count} = Client::get_count_received_deals($c->login_rights->{ClientID});
        }
        $vars->{enable_content_promotion_video_campaigns} = Direct::PredefineVars::_enable_content_promotion_video_campaigns($c);
        $vars->{enable_cpm_yndx_frontpage_campaigns} = Direct::PredefineVars::_enable_cpm_yndx_frontpage_campaigns($c);

        return respond_template($r, $template, 'request-access-to-metrika-counters/request-access-to-metrika-counters.tt2', $vars);
    } else {
        eval {
            die {error => iget('Не задан владелец счетчика')} unless defined $FORM{owner};
            die {error => iget('Логин владельца счетчика не найден')} unless get_uid_by_login($FORM{owner}, with_pdd => 1, with_social => 1);
            if (defined $FORM{send_to_email}) {
                die {error => iget('e-mail не указан')} unless defined $FORM{email};
                die {error => iget('неверно указан e-mail')} if ! Yandex::IDN::is_valid_email($FORM{email});
            }

            my @goals_or_sites;
            my $requests = [];

            if (defined $FORM{goals}) {
                @goals_or_sites = split /\s*,\s*/, $FORM{goals};
            }

            if (0 == scalar @goals_or_sites) {
                # запрос на весь логин
                $requests = [{
                    object_type => "delegate",
                    object_id => $FORM{owner},
                    owner_login => $FORM{owner},
                    comment => $FORM{comment},
                    send_email => $FORM{send_to_email} ? $FORM{email} : '',
                    send_sms => $FORM{send_to_phone} ? 1 : 0,
                    requestor_login => get_login(uid => $uid),
                    permission => "RO",
                    service_name => "direct",
                    lang => Yandex::I18n::current_lang(),
                }];
            } else {
                for my $goal_or_site (@goals_or_sites) {
                    push @$requests, {
                        object_type => $goal_or_site =~ /^\d+$/ ? 'counter' : 'site',
                        object_id => $goal_or_site,
                        owner_login => $FORM{owner},
                        comment => $FORM{comment},
                        send_email => $FORM{send_to_email} ? $FORM{email} : '',
                        send_sms => $FORM{send_to_phone} ? 1 : 0,
                        requestor_login => get_login(uid => $uid),
                        permission => "RO",
                        service_name => "direct",
                        lang => Yandex::I18n::current_lang(),
                    };
                }
            }

            my $result = Retargeting::request_access_to_metrika_counters($requests, $uid);
            my @errors = uniq (map {$_->{error_text}} grep {$_->{result} eq 'error'} @$result);
            die {error => join(", ", @errors)} if @errors;

        };

        if ($@) {
            if (ref($@) eq 'HASH') {
                return redirect($r, $SCRIPT, {cmd => 'requestAccessToMetrikaCounters'
                                     , error => 1
                                     , error_text => $@->{error}
                                     , ulogin => $FORM{ulogin}
                                     , csrf_token => $FORM{csrf_token}}
                                     , map {$_ => $FORM{$_}} qw/owner comment email goals send_to_email send_to_phone/
                        );
            } else {
                die $@;
            }
        } else {
            return redirect($r, $SCRIPT, {cmd => 'requestAccessToMetrikaCounters', success => 1, ulogin => $FORM{ulogin}, csrf_token => $FORM{csrf_token}});
        }
    }
}

1;
