##################################################
#
#  Direct.Yandex.ru
#
#  DoCmdStaff
#   контроллеры для работы с ролями и агенствами
#
#  $Id$
#
##################################################

=head1 NAME
    DoCmdStaff

=head1 DESCRIPTION

    контроллеры для работы с ролями и агенствами

=cut


package DoCmdStaff;

use warnings;
use strict;
use utf8;

use base qw/DoCmd::Base/;

use Yandex::Balance;
use Yandex::DBTools;
use Yandex::HashUtils;
use Yandex::I18n;
use Yandex::IDN qw(is_valid_email);
use Yandex::TimeCommon qw/human_date/;
use Yandex::Validate qw/is_valid_int is_valid_phone is_valid_id/;

use Settings;
use Primitives;
use PrimitivesIds;
use BalanceWrapper;
use Common qw(:globals :subs);
use Staff;
use EnvTools;
use Tools;
use Direct::ResponseHelper;
use Direct::Model::AgencyOptions;
use Direct::Model::Pixel;
use Direct::Model::Pixel::Utils;
use Direct::PredefineVars;
use RBACElementary;
use RBACDirect;
use Rbac qw/set_manager_supervisor/;
use RBAC2::DirectChecks;
use YandexOffice;
use Client;
use Client::CustomOptions;
use ServicedClient;
use User;
use UnitsFactory;
use APIUnits;
use GeoTools;
use geo_regions;
use List::MoreUtils qw/any uniq firstval/;
use Agency;
use Currencies;
use Currency::Format;
use API::Settings;
use BannerImages::Pool;
use DeviceTargeting qw/can_view_user_device_targeting_option/;
use XLSCampImport;
use API::ClientOptions;

use URI::Escape qw/uri_escape_utf8/;
use JSON qw/from_json/;

# ---------------------------------------------------------------------------------------------------------------------------
sub cmd_deleteUser :Cmd(deleteUser)
    :Description('удаление пользователя')
    :CheckCSRF
    :Rbac(Code => rbac_cmd_deleteUser)
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS/};
    my %FORM = %{$_[0]{FORM}};

    if ($FORM{role} !~ /^(super)?teamleader$/ && is_production() ) {                                                                                                           
        error("Используйте управлятор для удаления ролей");
    }

    my $vars;

    my $error_code = rbac_drop_user($rbac, $uid, $FORM{role});
    error(iget($RBAC2::DirectChecks::CMD_ERRORS{$error_code})) if $error_code;

    if (defined $FORM{role} && $FORM{role} eq 'manager') {
        my $affected_clients = get_one_column_sql(PPC(shard => 'all'), [
            'SELECT DISTINCT ClientID FROM campaigns', WHERE => {ManagerUID => $uid}
        ]);
        Direct::TurboLandings::mass_refresh_metrika_grants_for_all_client_counters($affected_clients);
        do_sql(PPC(shard => 'all'), "update campaigns set ManagerUID = null, statusBsSynced = 'No' where ManagerUID = ?", $uid);
    }
    if (defined $FORM{role} && $FORM{role} eq 'agency') {
        my $affected_clients = get_one_column_sql(PPC(shard => 'all'), [
            'SELECT DISTINCT ClientID FROM campaigns', WHERE => {AgencyUID => $uid}
        ]);
        Direct::TurboLandings::mass_refresh_metrika_grants_for_all_client_counters($affected_clients);

        do_sql(PPC(shard => 'all'), "update campaigns set AgencyUID = null, statusBsSynced = 'No' where AgencyUID = ?", $uid);
    }

    Client::update_role(get_clientid(uid => $uid), 'empty', undef);

    my $rcmd = $FORM{rcmd} ? $FORM{rcmd} : 'showStaff';
    return redirect($r, "$SCRIPT?cmd=$rcmd");
}

# ---------------------------------------------------------------------------------------------------------------------------
sub cmd_modifyUser :Cmd(modifyUser)
    :Rbac(Code => rbac_cmd_modifyUser, ExceptRole => [media], AllowDevelopers => 1)
    :CheckCSRF
    :Description('изменение параметров логина')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c, $cvars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c vars/};
    my %FORM = %{$_[0]{FORM}};

    if ( $UID != $uid ) {
        my $operator_perminfo = Rbac::get_perminfo( uid => $UID );
        my $client_perminfo = Rbac::get_perminfo( uid => $uid );
        if ( $operator_perminfo->{role} eq 'agency' &&
            $operator_perminfo->{rep_type} ne 'main' && $operator_perminfo->{rep_type} ne 'chief' &&
            $client_perminfo->{role} eq 'agency' )
        {
            error( iget('Нет прав для выполнения этой операции') );
        }
    }

    my $agency_options_select_fields = Direct::Model::AgencyOptions->get_db_columns('agency_options');
    # view data
    my $vars = get_one_line_sql(PPC(uid => $uid), "select
                                            u.uid,
                                            u.email,
                                            u.FIO,
                                            u.phone,
                                            u.createtime,
                                            u.ClientID,
                                            u.login,
                                            u.hidden,
                                            u.not_resident,
                                            u.statusBlocked,
                                            u.description,
                                            u.captcha_freq,
                                            u.allowed_ips,
                                            u.statusYandexAdv,
                                            u.showOnYandexOnly,
                                            u.lang as user_lang,

                                            uo.statusPostmoderate,
                                            uo.manager_office_id,
                                            uo.geo_id,
                                            uo.sendAgencyMcbLetters,
                                            uo.sendAgencyDirectLetters,
                                            uo.show_fa_teaser,
                                            uo.manager_use_crm,
                                            uo.agency_email_to_chief,
                                            uo.opts AS _opts_set,

                                            iu.domain_login,
                                            iu.manager_private_email,
                                            iu.is_developer,
                                            iu.is_super_manager,

                                            uc.is_captcha_amnested,
                                            uc.captcha_expires,

                                            uao.api_offer,
                                            uao.api_units_daily,
                                            uao.advq_queries_lim,
                                            uao.allow_create_subclients,
                                            uao.api_geo_allowed,
                                            uao.api_allow_old_versions,
                                            uao.excel_rows_limit,

                                            from_unixtime(createtime, '\%d.\%m.\%Y') as createtime_text,
                                            cl.allow_create_scamp_by_subclient,
                                            is_autobanned, autoban_date, IFNULL(autoban_bonus, 0) as autoban_bonus,
                                            clients_autoban.bids_count as clients_autoban_bids_count,
                                            clients_autoban.sum_total as clients_autoban_sum_total,

                                            clim.camp_count_limit,
                                            clim.unarc_camp_count_limit,
                                            clim.banner_count_limit,
                                            clim.feed_count_limit,
                                            clim.feed_max_file_size,
                                            clim.keyword_count_limit,
                                            clim.general_blacklist_size_limit,
                                            clim.video_blacklist_size_limit,

                                            cl.report_name,
                                            cl.country_region_id,
                                            clients_options.subregion_id,
                                            clients_options.can_manage_price_packages,
                                            clients_options.can_approve_price_packages,
                                            clients_options.default_disallowed_page_ids,
                                            clients_options.default_allowed_domains,
                                            clients_options.social_advertising,
                                            IF(cfmt.ClientID IS NOT NULL, 1, 0) AS force_multicurrency_teaser,
                                            cfmt.modify_convert_allowed,
                                            cl.is_favicon_blocked,
                                            uag.is_no_pay as agency_no_pay,
                                            uag.disallow_money_transfer as agency_disallow_money_transfer,
                                            uag.show_agency_contacts,
                                            uag.lim_rep_type as agency_lim_rep_type,
                                            uag.group_id as agency_group_id,
                                            FIND_IN_SET('no_display_hrefs', clients_options.client_flags)>0 as no_display_hrefs,
                                            FIND_IN_SET('not_convert_to_currency', clients_options.client_flags)>0 as not_convert_to_currency,
                                            FIND_IN_SET('suspend_video', clients_options.client_flags)>0 as suspend_video,
                                            FIND_IN_SET('can_copy_ctr', clients_options.client_flags)>0 as can_copy_ctr,
                                            FIND_IN_SET('cant_unblock', clients_options.client_flags)>0 as cant_unblock,

                                            $agency_options_select_fields
                                       from users u
                                       left join users_options uo using (uid)
                                       left join users_api_options uao using (uid)
                                       left join internal_users iu using (uid)
                                       left join users_captcha uc using (uid)
                                       left join clients cl on u.ClientID = cl.ClientID
                                       left join clients_autoban on u.ClientID = clients_autoban.ClientID
                                       left join client_limits clim on clim.ClientID = u.ClientID
                                       left join clients_to_force_multicurrency_teaser cfmt ON cfmt.ClientID = u.ClientID
                                       left join clients_options ON clients_options.ClientID = u.ClientID
                                       left join users_agency uag using (uid)
                                       left join agency_options ON agency_options.ClientID = IF(cl.role = 'agency', u.ClientID, NULL)
                                       where u.uid = ?", $uid);

    $vars->{uid} = $uid; # if not left join users_options

    $vars->{features_enabled_for_operator_all} = $cvars->{features_enabled_for_operator_all};
    $vars->{features_enabled_for_client_all} = $cvars->{features_enabled_for_client_all};

    my $client_options = API::ClientOptions::get($vars->{ClientID}, [qw/api_enabled api5_units_daily/]);
    $vars->{api_enabled} = $client_options->{api_enabled};
    my $api5_units = UnitsFactory::create(
        $vars->{ClientID},
        $client_options->{api5_units_daily}
    );

    $vars->{optimized_camps} = get_all_sql(PPC(uid => $uid), "select o.cid, count(o.request_id) as cnt from optimizing_campaign_requests o join campaigns c using(cid) where c.uid=? group by o.cid", $uid);

    $vars->{hidden_status} = 1 if $vars->{hidden} eq 'Yes';

    my $user_role = $vars->{user_role} = $login_rights->{client_role};

    $vars->{is_internal_user} = rbac_is_internal_user($rbac, uid => $uid);

    $vars->{user_is_any_client} = 1 if $user_role =~ /client$/;

    # users_options.opts expand
    if (my $opts = delete $vars->{_opts_set}) {
        $vars->{$_} = 1 for split /,/, $opts;
    }

    my $client_client_id;
    if ($vars->{user_is_any_client}) {
        my $user_options = get_user_options($uid);
        my $user_custom_options = get_user_custom_options($vars->{ClientID});
        
        $vars->{serviced} = ServicedClient::is_user_serviced($login_rights->{client_chief_uid});
        $vars->{allow_create_scamp_by_subclient} = $vars->{allow_create_scamp_by_subclient} && $vars->{allow_create_scamp_by_subclient} eq 'Yes' ? 1 : 0;
        $client_client_id = rbac_get_client_clientid_by_uid($uid) or die "ClientID not found for uid = $uid";
        $vars->{rbac_client_id} = $client_client_id;

        $vars->{has_agency_campaigns} = get_one_field_sql(PPC(uid => $uid), [ "select 1 from campaigns c", where => { 
            uid => $uid, 
            AgencyUID__gt => 0 
        }, "limit 1" ]);

        # не даем редактировать FIO/email/phone клиента если у него > 1 типа обслуживания или обслуживание завершенно
        if ($login_rights->{agency_control}) {
            my $agency_client_id = rbac_get_agency_clientid_by_uid( $UID) or die "ClientID not found for uid = $UID";
            $vars->{dont_edit_fio_email_by_agency} = rbac_agency_can_edit_user_data($rbac, $agency_client_id, [$client_client_id])->{$client_client_id} ? 0 : 1;
        }

        # Настройка таргетинга на устройства (у developer'ов ТОЛЬКО просмотр)
        if (can_view_user_device_targeting_option($c)) {
            $vars->{show_device_targeting} = 1;
            $vars->{allow_device_targeting} = $user_options->{can_have_device_targeting};
            # Раньше на логине менеджера была опция allow_device_targeting - означала право менеджера видеть/ставить таргетинг на устройства
        }

        # опция "Запретить перенос средств с клиента"
        hash_copy $vars, $user_custom_options, qw/disallow_money_transfer/;
    }

    if (defined $vars->{subregion_id}) {
        $vars->{city} = get_cityname_by_geoid($vars->{subregion_id});
    }

    my $agency_client_id;
    if ($user_role eq 'agency') {
        my $agency_vars = get_agency_client_vars($rbac, $uid);
        hash_merge $vars, $agency_vars;
        $agency_client_id = $vars->{agency_client_id};
        $vars->{rbac_client_id} = $agency_client_id;
        $vars->{additional_currencies} = get_agency_additional_currencies($agency_client_id, include_expired => 1);
        $vars->{agency_options} = Direct::Model::AgencyOptions->from_db_hash($vars, \{}, delete_keys => 1);
        $vars->{lim_rep_chiefs} = Agency::get_agency_lim_rep_chiefs($agency_client_id);
        if ($vars->{agency_rep_type} eq 'limited' && $vars->{agency_group_id}) {
            my @lim_rep_chief = grep {$_->{group_id} eq $vars->{agency_group_id}} @{$vars->{lim_rep_chiefs}};
            $vars->{lim_rep_chief_uid} = $lim_rep_chief[0]->{uid} if scalar @lim_rep_chief;
        }
        if ($vars->{agency_rep_type} eq 'limited' && $vars->{agency_lim_rep_type}) {
            $vars->{agency_rep_type} .= '_' . $vars->{agency_lim_rep_type};
        }
    }
    
    if ($user_role eq 'manager') {
        # для менеджера получаем список офисов
        $vars->{yandex_offices} = [get_yandex_offices()];
        $vars->{rbac_client_id} = $vars->{ClientID};
    }

    $vars->{AllowAgencyMassADVQ} = rbac_get_allow_agency_mass_advq($uid) if $user_role eq 'agency';
    $vars->{isSuperPlacer} = rbac_is_super_placer($rbac, $uid) if $user_role eq 'placer';
    $vars->{isSuperMedia} = rbac_is_super_media_planner($rbac, $uid) if $user_role eq 'media';
    $vars->{isGoodUser} = $vars->{statusPostmoderate} || 'No'; # from users_options table
    $vars->{ManagerOfAgencyUID} = $user_role eq 'agency' ? (rbac_get_manager_of_agency($rbac, $uid)||0) : 0;
    $vars->{allow_edit_description} = rbac_allow_edit_description($user_role, $login_rights);
    my $client_total_sums = $vars->{user_is_any_client} ? client_total_sums($vars->{ClientID}, type => 'text', currency => 'YND_FIXED') || {} : {};
    my $sum_on_client = $client_total_sums->{sum} || 0;
    $vars->{autoban_current_limit} = $Settings::AUTOBAN_UPPER_BOUND 
                                   + $vars->{autoban_bonus} 
                                   + $Settings::AUTOBAN_MONEY_COEF * $sum_on_client;

    $vars->{default_disallowed_page_ids} = join", ", @{from_json($vars->{default_disallowed_page_ids} // '[]')};
    $vars->{default_allowed_domains} = join", ", @{from_json(Encode::decode('UTF-8', $vars->{default_allowed_domains}) // '[]')};
   
    if (($login_rights->{super_control} || $login_rights->{support_control} || $login_rights->{superreader_control} || $login_rights->{limited_support_control}) 
            && $user_role =~ /^(?:agency|.*client)$/) {
        $vars->{balance_info} = eval {balance_get_info_about_rep($UID, $uid)};
        my $client = eval { get_client_by_uid($uid) };
        $vars->{balance_info}->{ClientName} = $client->{NAME} if $client;
    }

    for my $scheme (qw/API XLS/) {
        my $units = APIUnits->new({scheme => $scheme})->check_or_init_user_units($uid)->{$uid};
        $vars->{"${scheme}_units"} = $units->{units_rest};
        $vars->{"${scheme}_daily_units"} = $units->{daily_units};
    }

    # свойства субклиентов относительно его агентств (саппортам даем только смотреть)
    my $subclients_rights;
    if ($vars->{user_is_any_client}
        && ! $login_rights->{is_any_client}
        && ($login_rights->{ModifySubClient}
            || $login_rights->{support_control}
            || $login_rights->{limited_support_control}
           )
       )
    {
        my $operator_uid = ($login_rights->{support_control}) ? 1 : $UID; # саппортам для просмотра - показываем как для супера
        $subclients_rights = Agency::get_client_agency_options(operator_uid => $operator_uid, client_uids2_client_ids => { $uid => $client_client_id })->{$client_client_id};
        for my $agency_client_id (keys %$subclients_rights) {
            push @{ $vars->{client_by_agency_rights} }, $subclients_rights->{$agency_client_id};
        }
    }

    # может ли пользователь реально использовать api
    $vars->{can_use_api} = rbac_can_use_api($rbac, {uid => $uid, UID => $UID});
    $vars->{is_bad_passport_karma} = is_bad_passport_karma($uid);
    $vars->{methods} = \@API::Settings::LIMITABLE_METHODS;
    $vars = hash_merge $vars, query_all_api_special_options($vars->{rbac_client_id});

    # отдельным суперам даём возможность изменять страну/валюту
    $vars->{can_edit_country_currency} = (($login_rights->{super_control} && any {$UID == $_} @$Settings::CAN_EDIT_COUNTRY_CURRENCY_UIDS) ? 1 : 0);
    if ($vars->{can_edit_country_currency}) {
        $vars->{countries} = \@geo_regions::COUNTRY_REGIONS;
        $vars->{currencies} = [keys %Currencies::_CURRENCY_DESCRIPTION];
        $vars->{country_field_name} = "name_" . $vars->{user_lang};
    }

    # Суперам и тимлидерам даем возможность изменять город клиента
    $vars->{can_edit_subregion} = ((rbac_is_teamleader($rbac, $UID) || rbac_is_superteamleader($rbac, $UID) || $login_rights->{super_control}) ? 1 : 0);

    if (defined $FORM{save} && $FORM{save} eq 'yes' && !$login_rights->{superreader_control}) {

        error(iget("Нет прав для выполнения данной операции!")) if $login_rights->{superreader_control};
        error(iget("Нет прав для выполнения данной операции!")) if $login_rights->{limited_support_control};
    
        #Фрилансерам не даем редактировать настройки пользователя для сопровождаемых клиентов
        if ($uid != $UID && $login_rights->{is_freelancer} && RBACDirect::rbac_is_related_freelancer($rbac, $UID, $uid)) {
            error(iget("Редактирование данных клиента и его представителей запрещено"));
        }
        
        if (defined $FORM{api} && $FORM{api}) { # сохраняем настройки API пользователя
            $vars->{error} = save_edited_user_API_settings({
                uid => $uid,
                LOGIN_RIGHTS => $login_rights,
                FORM => $_[0]{FORM},
                CLIENT_ID => $vars->{rbac_client_id}
            });
            unless ($vars->{error}) {
                if ($FORM{retpath}) {
                    return redirect($r, $FORM{retpath});
                }
                elsif ($FORM{rcmd}) {
                    return redirect($r, $SCRIPT, {cmd => $FORM{rcmd}, ulogin => $FORM{rulogin}});
                }
                else {
                    return redirect($r, $SCRIPT, {cmd => 'modifyUser', ulogin => $FORM{ulogin}});
                }
            }
        } else { # сохраняем общие настройки пользователя
            # не даём изменить статус агенства на АБУ, если текущий статус не АБУ
            if ($user_role eq 'agency' && ($vars->{agency_status} // '') ne 'ABU' && ($FORM{agency_status} // '') eq 'ABU') {
                error('Невозможно создать агентство без участия в расчетах');
            }

            # пустые лимиты заменяем на 0
            # https://st.yandex-team.ru/DIRECT-50109#1453728656000
            for my $limit (qw/banner_count_limit 
                            feed_count_limit
                            keyword_count_limit
                            video_blacklist_size_limit
                            general_blacklist_size_limit/) {
                next if !exists $_[0]{FORM}->{$limit};
                $_[0]{FORM}->{$limit} ||= q{};
            }

            $vars->{error} = save_edited_user ( 
                        {
                            UID => $UID,
                            uid => $uid,
                            RBAC => $rbac,
                            LOGIN_RIGHTS => $login_rights,
                            VARS => $vars,
                            SUBCLIENTS_RIGHTS => $subclients_rights,
                            AGENCY_CLIENT_ID => $agency_client_id,
                            CLIENT_CLIENT_ID => $client_client_id,
                            FORM => $_[0]{FORM},
                        }
                    );
            if ( !$vars->{error} ) {

                set_show_agency_contacts_flag($uid, $FORM{show_agency_contacts} ? 1 : 0);
                if ($FORM{retpath}) {
                    return redirect($r, $FORM{retpath});
                }
                elsif ($FORM{rcmd}) {
                    return redirect($r, $SCRIPT, {cmd => $FORM{rcmd}, ulogin => $FORM{rulogin}});
                }
                else {
                    return redirect($r, $SCRIPT, {cmd => 'modifyUser', ulogin => $FORM{ulogin}});
                }
            } else {
                # если была ошибка, возвращаем в форму введенные пользователям значения прав
                foreach my $ag (@{$vars->{client_by_agency_rights}}) {
                    $ag->{$_} = $FORM{$_.'_'.$ag->{ClientID}} foreach qw/isSuperSubClient allowImportXLS allowTransferMoney/;
                }

                $vars->{country_region_id} = $FORM{country};
            }

        }
    }

    if ($vars->{ClientID} && any {$login_rights->{$_}} qw(support_control limited_support_control manager_control super_control superreader_control)) {
        my $client_currencies = get_client_currencies($vars->{ClientID});
        $vars->{work_currency} = $client_currencies->{work_currency};

        my $balance_clients = eval { balance_find_client({ ClientID => $vars->{ClientID} }) };
        if ($balance_clients && @$balance_clients) {
            my $balance_country = $balance_clients->[0]->{REGION_ID};
            if ($balance_country) {
                my $geo_name_field = GeoTools::get_geo_name_field();
                my $country_data = get_exact_geo_region($balance_country);
                $vars->{balance_country}->{localized_name} = $country_data->{$geo_name_field};
            }
            my $balance_currency = $balance_clients->[0]->{SERVICE_DATA}->{ $Settings::SERVICEID{direct} }->{CURRENCY};
            $vars->{balance_currency} = $balance_currency;
        } else {
            $vars->{no_balance_client_data} = 1;
        }
    }

    $vars->{rcmd} = $FORM{rcmd};
    $vars->{camp_count_limit} ||= $Settings::DEFAULT_CAMP_COUNT_LIMIT;
    $vars->{unarc_camp_count_limit} ||= $Settings::DEFAULT_UNARC_CAMP_COUNT_LIMIT;
    $vars->{default_banner_count_limit} = $Settings::DEFAULT_BANNER_COUNT_LIMIT;
    $vars->{default_keyword_count_limit} = $Settings::DEFAULT_KEYWORD_COUNT_LIMIT;
    $vars->{default_general_blacklist_size_limit} = $Settings::DEFAULT_GENERAL_BLACKLIST_SIZE_LIMIT;
    $vars->{default_video_blacklist_size_limit} = $Settings::DEFAULT_VIDEO_BLACKLIST_SIZE_LIMIT;
    $vars->{default_feed_count_limit} = $Settings::DEFAULT_FEED_COUNT_LIMIT;
    $vars->{default_feed_max_file_size} = $Settings::DEFAULT_FEED_MAX_FILE_SIZE;
    $vars->{default_excel_rows_limit} = $XLSCampImport::DEFAULT_EXCEL_ROWS_LIMIT;
    $vars->{available_languages} = [Yandex::I18n::default_lang(), sort(Yandex::I18n::get_other_langs())];
    $vars->{image_pool_limit} = $BannerImages::Pool::MINIMUM_IMAGES_IN_POOL if ! defined $vars->{image_pool_limit};
    if ($vars->{agency_options}) {
        $vars->{agency_options} = $vars->{agency_options}->to_template_hash();
    }

    $vars->{API5_units_limit} = $api5_units->limit;
    $vars->{API5_today_spent} = $api5_units->spent;

    # TODO: составить список действительно нужных полей и копировать только их
    $vars = hash_merge $vars, \%FORM;

    if (exists $vars->{country_region_id}) {
        my $country_region_id = delete $vars->{country_region_id};
        if ($country_region_id) {
            my $geo_name_field = GeoTools::get_geo_name_field();
            my $country_data = get_exact_geo_region($country_region_id);
            $vars->{country} = {
                region_id => $country_region_id,
                localized_name => $country_data->{$geo_name_field},
            };
        }
    }

    if ($vars->{ClientID} && ($login_rights->{role} eq 'super' || $login_rights->{role} eq 'support')) {
        # получаем текущие разрешенные клиенту пикселы охватного продукта
        $vars->{pixels} = {
            providers => $Direct::Model::Pixel::PERMISSIBLE_PROVIDERS,
            campaign_types => $Direct::Model::Pixel::PERMISSIBLE_CAMPAIGN_TYPES,
            criterion_types => $Direct::Model::Pixel::PERMISSIBLE_CRITERION_TYPES,
            permitted_providers => Direct::Model::Pixel::Utils::get_permitted_providers($vars->{ClientID})
        };
    }

    $vars->{enable_cpm_deals_campaigns} = Direct::PredefineVars::_enable_cpm_deals_campaigns($c);
    if ($vars->{enable_cpm_deals_campaigns}){
        $vars->{new_deals_count} = Client::get_count_received_deals($c->login_rights->{ClientID});
    }
    $vars->{enable_content_promotion_video_campaigns} = Direct::PredefineVars::_enable_content_promotion_video_campaigns($c);
    $vars->{enable_cpm_yndx_frontpage_campaigns} = Direct::PredefineVars::_enable_cpm_yndx_frontpage_campaigns($c);


    return respond_bem($r, $c->reqid, $vars, source=>'data3');
}

=head2 _create_internal_client

     Только для внутренних ролей:
     создать клиента в Балансе
     записать clients с ролями
     вернуть ClientID
=cut
sub _create_internal_client {
    my ($UID, $uid, $role, $subrole, $email) = @_;
    my $ClientID = get_clientid(uid => $uid);
    if ($ClientID) {
        Client::update_role($ClientID, $role, $subrole);
    } else {
        $ClientID = Client::create_client_in_balance($UID, $uid, role => $role, subrole => $subrole,
                initial_country => $geo_regions::RUS, initial_currency => 'RUB', email => $email);
        create_update_client({ client_data => { ClientID => $ClientID, role => $role, subrole => $subrole } });
    }
    return $ClientID;
}

# ---------------------------------------------------------------------------------------------------------------------------
sub cmd_createAgency :Cmd(createAgency)
    :Description('создание агентства')
    :CheckCSRF
    :Rbac(Role => [super, manager])
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c/};
    my %FORM = %{$_[0]{FORM}};

    my $vars = \%FORM;

    # эти данные используются в шаблоне (т.е. нужны там, где вызывается error_tmpl)
    my $currencies_by_country = Currencies::get_currencies_by_country_hash_for_agency(undef, undef, {force_allow_all => 1});
    $vars->{countries} = \@geo_regions::COUNTRY_REGIONS;
    $vars->{countries_currencies} = $currencies_by_country;
    $vars->{main_countries} = \%GeoTools::MAIN_COUNTRIES;

    # Если это back
    if ( $FORM{back} ) {
        return error_tmpl( $r, $template, 'addagency.html', $vars, '' );
    }

    # login - обязательное для заполнения поле
    if (! defined $FORM{login} || $FORM{login} =~ /^\s*$/ ) {
        $vars->{login_error} = 1;
        return error_tmpl( $r, $template, 'addagency.html', $vars, iget('не введен login') );
    }

    # Получаем uid
    $FORM{login} =~ s/\s//g;
    $uid = get_uid_by_login( $FORM{login} );
    if ( !$uid ) {
        $vars->{login_error} = 1;
        return error_tmpl( $r, $template, 'addagency.html', $vars, iget('такой логин не существует') );
    }
    
    #переопределяем набор с учётом ограничений
    $currencies_by_country = Currencies::get_currencies_by_country_hash_for_agency($uid, undef);
    $vars->{countries} = \@geo_regions::COUNTRY_REGIONS;
    $vars->{countries_currencies} = $currencies_by_country;
    $vars->{main_countries} = \%GeoTools::MAIN_COUNTRIES;

    # Проверяем, что у него нет кампаний
    if ( get_one_field_sql(PPC(uid => $uid), q/SELECT 1 FROM campaigns WHERE uid = ? AND statusEmpty = 'No' LIMIT 1/, $uid) ) {
        $vars->{login_error} = 1;
        return error_tmpl( $r, $template, 'addagency.html', $vars, iget('у этого пользователя есть непустые кампании') );
    }
    # Проверяем, что его нет в RBAC
    my $role = rbac_who_is($rbac, $uid);
    if ( $role !~ /^(client|empty)$/ ) {
        my $role_name = rbac_role_name( $role );
        if ( $role eq 'agency' ) {
            my $muid = rbac_get_manager_of_agency( $rbac, $uid );
            my $mlogin = get_login(uid =>  $muid );
            $role_name = iget("агентство менеджера %s", $mlogin);
        }
        $vars->{login_error} = 1;
        return error_tmpl( $r, $template, 'addagency.html', $vars, iget("пользователь %s уже зарегистрирован в Директ как %s", $FORM{login}, $role_name) );
    }

    # Если нет client_id, пытаемся его найти
    # этот же блок кода полностью продублирован чуть ниже. измениния нужно вносить и здесь и там!
    if ( $FORM{sbm_find} ) {
        # ищем по uid
        my $client = get_client_by_uid( $uid );
        if ( !defined $client ) {
            # Ошибка
            return error_tmpl( $r, $template, 'addagency.html', $vars, iget('ошибка поиска клиента') );
        } elsif ( $client->{CLIENT_ID} ) {
            if ( !$client->{IS_AGENCY} ) {
                return error_tmpl( $r, $template, 'addagency.html', $vars, iget("по указанному логину %s в билинге найден клиент %s (%s), не являющийся агентством", $FORM{login}, $client->{NAME}, $client->{CLIENT_ID}) );
            }
            if (@{get_uids(ClientID => $client->{CLIENT_ID})}) {
                # В директе уже есть логины с данным ClientID (чего этим хотят добиться - не ясно).
                # Результат дальнейших взаимодействия (в особенности страна/валюта у нас и биллинге)
                # слабо предсказуем, поэтому ничего не даем делать.
                $vars->{login_error} = 1;
                return error_tmpl( $r, $template, 'addagency.html', $vars, iget('В Директе уже есть логины, соответствующие указанному ClientID: %s.', $client->{CLIENT_ID}) );
            }
            # Копируем данные в форму
            $FORM{client_id} = $client->{CLIENT_ID};
            $FORM{enter_data} = {%FORM};
            if ($FORM{enter_data}{client_country}) {
                $FORM{enter_data}{country_name} = get_cityname_by_geoid($FORM{enter_data}{client_country});
            }

            # данные о клиенте в Балансе
            for my $field ( qw( name phone email url ) ) {
                $FORM{$field} = $client->{uc $field} || $FORM{$field};
            }
            my $firm_country_currency_data = BalanceWrapper::get_firm_country_currency($FORM{client_id}, AgencyID => $FORM{client_id}, currency_filter => 1);

            my @countries = uniq map {$_->{region_id}} @$firm_country_currency_data;
            my $agency_any_country_currency_cnt = Currencies::get_any_country_currency_cnt_for_agency();
            if (scalar(@countries) == 1) {
                # используем уже зафиксированную в Балансе страну
                $FORM{client_country} = $FORM{force_client_country} = $countries[0];
                if ($FORM{client_country}) {
                    $FORM{country_name} = $FORM{force_client_country_name} = get_cityname_by_geoid($FORM{client_country});
                }
            } else {
                if (scalar(@$firm_country_currency_data) == $agency_any_country_currency_cnt) {
                    # в Балансе страна и валюта не выбраны, используем указанную менеджером страну
                    $FORM{force_client_country} = $FORM{enter_data}{client_country};
                    if ($FORM{force_client_country}) {
                        $FORM{force_client_country_name} = get_cityname_by_geoid($FORM{force_client_country});
                    }
                    $FORM{client_country} = undef;
                }
            }

            my @currencies = uniq map {$_->{currency}} @$firm_country_currency_data;
            if (scalar(@currencies) == 1) {
                $FORM{currency} = $currencies[0];
            } else {
                # в Балансе страна и валюта не выбраны, используем указанную менеджером валюту
                # или выбрана страна и доступно несколько валют
                $FORM{force_currency} = $FORM{enter_data}{currency};
                $FORM{currency} = undef;
                # если указанная валюта не является допустимой - покажем ошибку при попытке создания
            }

            # Запрашиваем подтверждение
            $FORM{association_confirm} = 1;
            my $currency = $FORM{force_currency} || $FORM{currency};
            my $currency_name = ($currency) ? format_currency($currency) : iget('<не выбрана>');
            my $msg = iget("по указанному логину %s в билинге найдено агентство. будет применена страна: %s валюта: %s",
                $FORM{login}, $FORM{force_client_country_name} || $FORM{country_name}, $currency_name);
            return error_tmpl( $r, $template, 'addagency.html', $vars, $msg );
        }
    }

    $FORM{clients} = [];
    my $currency = $FORM{currency};
    # Если есть флажок, создаём агентство в балансе
    if ( !$FORM{client_id} && $FORM{sbm_create} ) {
        if ( !$FORM{name} || $FORM{name} =~ /^\s*$/
            || !$FORM{phone} || $FORM{phone} =~ /^\s*$/
            || !$FORM{email} || $FORM{email} =~ /^\s*$/
            || !$FORM{url} || $FORM{url} =~ /^\s*$/
        ) {
            return error_tmpl( $r, $template, 'addagency.html', $vars, iget('не все поля заполнены') );
        } elsif (!is_valid_email($FORM{email})) {
            return error_tmpl($r, $template, 'addagency.html', $vars, iget('неправильно указан email'));
        } elsif (!is_valid_phone($FORM{phone})) {
            return error_tmpl($r, $template, 'addagency.html', $vars, iget('неправильно указан телефон'));
        } elsif (!$FORM{client_country} || !is_valid_int($FORM{client_country}, 0)) {
            return error_tmpl( $r, $template, 'addagency.html', $vars, iget('не указана страна') );
        } elsif (!$FORM{currency}) {
            return error_tmpl( $r, $template, 'addagency.html', $vars, iget('не указана валюта') );
        } else {
            $vars->{from} = 'createAgency';
            $vars->{client_uid} = $uid;

            my $country_region_id = $FORM{client_country};
            die "invalid country region_id: $country_region_id" unless Client::is_valid_client_country($country_region_id);
            unless (any {$country_region_id == $_->{region_id}} @{$vars->{countries}}) {
                return error_tmpl( $r, $template, 'addagency.html', $vars, iget('указанная страна недоступна для выбора') )
            }
            $vars->{country_region_id} = $country_region_id;

            my $possible_currencies = $currencies_by_country->{$country_region_id} || $currencies_by_country->{other};
            unless (any {$currency eq $_} @$possible_currencies) {
                return error_tmpl( $r, $template, 'addagency.html', $vars, iget('указанная валюта недоступна  для выбора') )
            }
            $vars->{currency} = $currency;

            return error_tmpl( $r, $template, 'addagency.html', $vars, iget('ошибка создания клиента') ) if create_client_id( $vars, $UID );
            # записываем новый ClientID в том числе и в %FORM, т.к. $vars — ссылка на тот же самый хеш
            $vars->{client_id} = $vars->{new_clientID};
        }
    } elsif ( $FORM{sbm_find} ) {
        # Ищем по остальным полям
        for my $field ( qw( name phone email url ) ) {
            next if !$FORM{$field};
            my $hash = { from => 'createAgency', $field => $FORM{$field} };
            return error_tmpl( $r, $template, 'addagency.html', $vars, iget('ошибка при поиске клиента') ) if get_client_id_list( $hash );
            if ( $hash->{client} ) {
                push @{$FORM{clients}}, @{$hash->{client}};
            }
        }
        # Чистим от дубликатов
        my %dup;
        @{$FORM{clients}} = grep { ! $dup{ $_->{CLIENT_ID} }++ } @{$FORM{clients}};
    }

    # Если есть client_id - можно сохранять
    if ( $FORM{client_id} && $FORM{client_id} =~ /^\d+$/
        && ( $FORM{sbm_create} || $FORM{sbm_assoc} )
    ) {
        # проверяем может ли пользователь исправлять данные у Агентства и geo_id агентства         
        my $geo_id_ok_flag = 1;
        if ( $login_rights->{super_control} || $login_rights->{manager_control} ) {            
            if(! defined $FORM{geo_id} || $FORM{geo_id} !~ /^\d+$/){
                $geo_id_ok_flag = 0;
            }
        }

        # проверка данных
        if ( $FORM{sbm_create}
            && (
                !$FORM{name} || $FORM{name} =~ /^\s*$/
                || !$FORM{phone} || $FORM{phone} =~ /^\s*$/
                || !$FORM{email} || $FORM{email} =~ /^\s*$/
                || !$FORM{url} || $FORM{url} =~ /^\s*$/
                || !$geo_id_ok_flag
                )
        ) {
            return error_tmpl( $r, $template, 'addagency.html', $vars, iget('не все поля заполнены') );
        } elsif (!is_valid_email($FORM{email})) {
            return error_tmpl($r, $template, 'addagency.html', $vars, iget('неправильно указан email'));
        } elsif (!$FORM{client_country} || !is_valid_int($FORM{client_country}, 0)) {
            return error_tmpl( $r, $template, 'addagency.html', $vars, iget('не указана страна') );
        } elsif (!$FORM{currency} || !is_valid_currency($FORM{currency})) {
            return error_tmpl( $r, $template, 'addagency.html', $vars, iget('не указана валюта') );
        } elsif (@{get_uids(ClientID => $FORM{client_id})}) {
            error(iget('В Директе уже есть логины, соответствующие указанному ClientID: %s. Обратитесь, пожалуйста, в службу поддержки по адресу account@direct.yandex.ru.', $FORM{client_id}) );
        }

        # https://jira.yandex-team.ru/browse/DIRECT-30313?focusedCommentId=5210893&page=com.atlassian.jira.plugin.system.issuetabpanels%3Acomment-tabpanel#comment-5210893
        # Кнопку "Создать" могли нажать вместо "Найти", ожидая страницу подтверждения привязки
        # Заполнили не все поля - нашли точное соответствие по логину в биллинге - нажали "подтвердить" - попали на страницу поиска, дозаполнили - нажали "создать"
        # Нужно еще раз попробовать поискать - и если точное соответствие (по логину) есть - показывать подтверждение.
        # Правильно когда-нибудь это место переделать на разные контроллеры - сначала поиск, потом или создание или подтверждение, без возможности изменить данные
        if ($FORM{sbm_create}) {
            # ужасная копипаста из ветки $FORM{sbm_find} выше -------------------------------------------
            # ищем по uid
            my $client = get_client_by_uid( $uid );
            if ( !defined $client ) {
                # Ошибка
                return error_tmpl( $r, $template, 'addagency.html', $vars, iget('ошибка поиска клиента') );
            } elsif ( $client->{CLIENT_ID} ) {
                if ( !$client->{IS_AGENCY} ) {
                    return error_tmpl( $r, $template, 'addagency.html', $vars, iget("по указанному логину %s в билинге найден клиент %s (%s), не являющийся агентством", $FORM{login}, $client->{NAME}, $client->{CLIENT_ID}) );
                }
                if (@{get_uids(ClientID => $client->{CLIENT_ID})}) {
                    # В директе уже есть логины с данным ClientID (чего этим хотят добиться - не ясно).
                    # Результат дальнейших взаимодействия (в особенности страна/валюта у нас и биллинге)
                    # слабо предсказуем, поэтому ничего не даем делать.
                    $vars->{login_error} = 1;
                    return error_tmpl( $r, $template, 'addagency.html', $vars, iget('В Директе уже есть логины, соответствующие указанному ClientID: %s.', $client->{CLIENT_ID}) );
                }
                # Копируем данные в форму
                $FORM{client_id} = $client->{CLIENT_ID};
                $FORM{enter_data} = {%FORM};
                if ($FORM{enter_data}{client_country}) {
                    $FORM{enter_data}{country_name} = get_cityname_by_geoid($FORM{enter_data}{client_country});
                }

                # данные о клиенте в Балансе
                for my $field ( qw( name phone email url ) ) {
                    $FORM{$field} = $client->{uc $field} || $FORM{$field};
                }
                my $firm_country_currency_data = BalanceWrapper::get_firm_country_currency($FORM{client_id}, AgencyID => $FORM{client_id}, currency_filter => 1);

                my @countries = uniq map {$_->{region_id}} @$firm_country_currency_data;
                my $agency_any_country_currency_cnt = Currencies::get_any_country_currency_cnt_for_agency();
                if (scalar(@countries) == 1) {
                    # используем уже зафиксированную в Балансе страну
                    $FORM{client_country} = $FORM{force_client_country} = $countries[0];
                    if ($FORM{client_country}) {
                        $FORM{country_name} = $FORM{force_client_country_name} = get_cityname_by_geoid($FORM{client_country});
                    }
                } else {
                    if (scalar(@$firm_country_currency_data) == $agency_any_country_currency_cnt) {
                        # в Балансе страна и валюта не выбраны, используем указанную менеджером страну
                        $FORM{force_client_country} = $FORM{enter_data}{client_country};
                        if ($FORM{force_client_country}) {
                            $FORM{force_client_country_name} = get_cityname_by_geoid($FORM{force_client_country});
                        }
                        $FORM{client_country} = undef;
                    }
                }

                my @currencies = uniq map {$_->{currency}} @$firm_country_currency_data;
                if (scalar(@currencies) == 1) {
                    $FORM{currency} = $currencies[0];
                } else {
                    # в Балансе страна и валюта не выбраны, используем указанную менеджером валюту
                    # или выбрана страна и доступно несколько валют
                    $FORM{force_currency} = $FORM{enter_data}{currency};
                    $FORM{currency} = undef;
                    # если указанная валюта не является допустимой - покажем ошибку при попытке создания
                }

                # Запрашиваем подтверждение
                $FORM{association_confirm} = 1;
                my $currency = $FORM{force_currency} || $FORM{currency};
                my $currency_name = ($currency) ? format_currency($currency) : iget('<не выбрана>');
                my $msg = iget("по указанному логину %s в билинге найдено агентство. будет применена страна: %s валюта: %s",
                    $FORM{login}, $FORM{force_client_country_name} || $FORM{country_name}, $currency_name);
                return error_tmpl( $r, $template, 'addagency.html', $vars, $msg );
            # конец копипасты ---------------------------------------------------------------------------------------------
            } else {
                # с таким логином в биллинге никого нет - можно создавать
            }
        }

        # для существующих клиентов корректируем доступные страны/валюты с учётом данных Баланса
        my $firm_country_currency_data = BalanceWrapper::get_firm_country_currency($FORM{client_id}, AgencyID => $FORM{client_id}, currency_filter => 1);
        my $allowed_country_currency_data = Common::get_client_allowed_country_currency(
            firm_country_currency_data => $firm_country_currency_data,
            is_direct => $c->is_direct,
            is_for_agency => 1,
            uid => $uid,
        );
        $currencies_by_country = $allowed_country_currency_data->{countries_currencies};

        my $country_region_id = $FORM{client_country};
        die "invalid country region_id: $country_region_id" unless Client::is_valid_client_country($country_region_id);
        unless (any {$country_region_id == $_->{region_id}} @{$allowed_country_currency_data->{countries}}) {
            error(iget('Несоответствие страны в Балансе и Директе. Обратитесь, пожалуйста, в службу поддержки по адресу account@direct.yandex.ru.'));
        }
        $vars->{country_region_id} = $country_region_id;

        my $possible_currencies = $currencies_by_country->{$country_region_id} || $currencies_by_country->{other};
        unless (any {$currency eq $_} @$possible_currencies) {
            error(iget('Несоответствие стран и валют в Балансе и Директе. Обратитесь, пожалуйста, в службу поддержки по адресу account@direct.yandex.ru.'));
        }
        $vars->{work_currency} = $currency;

        # Делаем привязку в балансе
        if ( !$FORM{sbm_assoc} ) {
            if ( !create_client_id_association($uid, $FORM{client_id}, $UID) ) {
                return error_tmpl( $r, $template, 'addagency.html', $vars, iget('ошибка привязки пользователя в балансе') );
            }
        }

        if ($FORM{agency_status} && $FORM{agency_status} eq 'ABU') {
            error('Невозможно создать агентство без участия в расчетах');
        };

        my $agency_status = $FORM{agency_status} && $FORM{agency_status} =~ /^(?:SA|AA|HY|WC|NR)$/ ? $FORM{agency_status} : undef;

        my $success = eval {
            Agency::create_agency($rbac, $UID, $c->is_direct, {
                uid => $uid,
                name => $FORM{name},
                phone => $FORM{phone},
                email => $FORM{email},
                url => $FORM{url},
                client_id => $FORM{client_id},
                country_region_id => $country_region_id,
                subregion_id => $FORM{geo_id},
                currency => $currency,
                agency_status => $agency_status,
                show_agency_contacts => $FORM{show_agency_contacts},
            });
            return 1;
        };
        if (!$success) {
            my $err = $@;

            if (ref $err eq 'HASH' && $err->{code}) {
                if ($err->{code} eq $Agency::ERROR_CREATE_AGENCY_RBAC_ERROR) {
                    return error_tmpl( $r, $template, 'addagency.html', $vars, iget('ошибка создания агентства в RBAC') );
                }
            }
            die $err;
        }

        # Всё ok, редиректим на страницу с клиентами
        return redirect($r, "$SCRIPT?cmd=showManagerMyClients");
    } else {
        return error_tmpl( $r, $template, 'addagency.html', $vars, @{$FORM{clients}} ? '' : iget('агентство с такими параметрами в базе не найдено') );
    }
}

# ---------------------------------------------------------------------------------------------------------------------------
sub cmd_createTeamleader :Cmd(createTeamleader)
    :Description('создание тимлидера')
    :CheckCSRF
    :Rbac(Code => rbac_cmd_createTeamleader)
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS/};
    my %FORM = %{$_[0]{FORM}};

    my $vars = \%FORM;
    $vars->{msg} = $rights;
    $uid = get_uid_by_login($FORM{login});
    $vars->{error} = check_create_internal_user({
                                                  data => {
                                                                form => \%FORM,
                                                                uid => $uid,
                                                          },
                                                  checks => {
                                                                filled_login => 1,
                                                                login_exists => 1,
                                                                is_manager => 1,
                                                            }
                                                });
    if ($vars->{error}) {
        return respond_template($r, $template, 'addhuman.html', $vars);
    }

    # create in rbac
    if (rbac_create_teamleader($rbac, $uid)) {
        $vars->{error} = iget('не удалось создать тимлидера');
        return respond_template($r, $template, 'addhuman.html', $vars);
    }

    _create_internal_client($UID, $uid, 'manager', 'teamleader', $FORM{email});

    # add to my group
    if ($login_rights->{is_superteamleader}) {
        my $errorcode = rbac_add_team_to_superteamleader($rbac, $uid, $UID);
        error('не удалось создать привязать тимлидера') if $errorcode;
        set_manager_supervisor(get_clientid(uid => $uid), get_clientid(uid => $UID));
    }
    
    return redirect($r, "$SCRIPT?cmd=manageTeamleaders");
}

# ---------------------------------------------------------------------------------------------------------------------------
sub cmd_createSuperTeamleader :Cmd(createSuperTeamleader)
    :Description('создание супертимлидера')
    :CheckCSRF
    :Rbac(Code => rbac_cmd_createTeamleader)
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS/};
    my %FORM = %{$_[0]{FORM}};

    my $vars = \%FORM;
    $vars->{msg} = $rights;
    $uid = get_uid_by_login($FORM{login});
    $vars->{error} = check_create_internal_user({
                                                  data => {
                                                                form => \%FORM,
                                                                uid => $uid,
                                                          },
                                                  checks => {
                                                                filled_login => 1,
                                                                login_exists => 1,
                                                                is_manager => 1,
                                                            }
                                                });
    if ($vars->{error}) {
        return respond_template($r, $template, 'addhuman.html', $vars);
    }

    # create in rbac
    if (rbac_create_superteamleader($rbac, $uid)) {
        $vars->{error} = iget('не удалось создать super тимлидера');
        return respond_template($r, $template, 'addhuman.html', $vars);
    }

    _create_internal_client($UID, $uid, 'manager', 'teamleader', $FORM{email});

    return redirect($r, "$SCRIPT?cmd=manageSTeamleaders");
}

# ---------------------------------------------------------------------------------------------------------------------------

=head2 cmd_showStaff

  show direct staff

=cut

sub cmd_showStaff :Cmd(showStaff)
    :Description('просмотр внутренних ролей')
    :Rbac(Perm => UsersAdministration, AllowDevelopers => 1)
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS/};
    my %FORM = %{$_[0]{FORM}};

    my ($vars, $sql_in);

    if ($FORM{agencies}) {

        my $ag_uids = rbac_get_all_agencies($rbac) || [];
        my $agencies_data = get_users_data($ag_uids, [qw/uid email valid LastChange fio phone sendNews sendWarn createtime
                                                        ClientID login hidden sendAccNews not_resident statusArch statusBlocked
                                                        description lang captcha_freq allowed_ips statusYandexAdv showOnYandexOnly/]);
        # В RBAC есть агентства, отсутствующие в users. отсекаем таких.
        $vars->{agencies} = [grep { $_->{uid} } values %$agencies_data];

        my $manager_uid_by_agency_uid = RBACDirect::rbac_mass_get_manager_of_agencies($rbac, [map { $_->{uid} } @{ $vars->{agencies} }]);
        # Менеджеры агентств тоже не все находятся.
        my $managers_data = get_users_data([uniq grep { defined $_ } values %$manager_uid_by_agency_uid], [qw/login fio/]);

        for my $agency_info (@{ $vars->{agencies} }) {
            $agency_info->{createtime_text} = human_date($agency_info->{createtime});
            if (my $manager_uid = $manager_uid_by_agency_uid->{ $agency_info->{uid} }) {
                $agency_info->{manager_login} = $managers_data->{$manager_uid}->{login};
                $agency_info->{manager_fio} = $managers_data->{$manager_uid}->{fio};
            } else {
                $agency_info->{manager_login} = undef;
                $agency_info->{manager_fio} = undef;
            }
        }

    } else {

        my $staff = rbac_get_staff($rbac);
        my %all_super_placers = map {$_ => 1} @{ $staff->{superplacers} };

        # get users info from db by uids
        my @all_staff_uids = map { @{$_} } values %$staff;
        my $users_info = get_users_data(\@all_staff_uids, [qw/uid email valid LastChange fio phone sendNews sendWarn
                                                            createtime ClientID login hidden sendAccNews not_resident
                                                            statusArch statusBlocked description lang captcha_freq
                                                            allowed_ips statusYandexAdv showOnYandexOnly
                                                            domain_login manager_private_email is_developer is_super_manager/]);
        for my $user_info (values %$users_info) {
            $user_info->{createtime_text} = human_date($user_info->{createtime});
        }


        # generate staff data hash
        while (my ($staff_role_name, $staff_uids) = each %$staff) {
            next if $staff_role_name eq 'superplacers';

            for my $user_uid (@$staff_uids) {
                my $user_info = $users_info->{$user_uid};

                # if user not found in direct db ('users' table)
                unless ($user_info) {
                    my $passport_info = get_info_by_uid_passport($user_uid);
                    $user_info = {uid => $user_uid
                                  , login => $passport_info->{login}
                                  , fio => $passport_info->{fio}
                                  , email => ($passport_info->{email} || '-')
                                  , createtime_text => '-'
                                 };
                }

                $user_info->{super_placer} = 1 if $all_super_placers{$user_uid};

                push @{$vars->{$staff_role_name}}, $user_info;
            }
        }
    }

    if (defined $FORM{error} && $FORM{error} eq '1') {
        $vars->{error} = 'user not found';
    }

    if (defined $FORM{login_or_uid}) {

        $vars->{login_or_uid} = $FORM{login_or_uid};

        if ($FORM{login_or_uid} =~ /^\d+$/) {
            $uid = $FORM{login_or_uid};
        } else {
            $uid = get_uid_by_login($FORM{login_or_uid});
        }

        if (! defined $uid || $uid == 0 || get_one_field_sql(PPC(uid => $uid), "select count(*) from users where uid = ?", $uid) != 1) {
            $vars->{error} = 'пользователь не найден';
        } else {
            my $login = uri_escape_utf8(get_login(uid => $uid));
            return redirect($r, "$SCRIPT?cmd=modifyUser&rcmd=showStaff&ulogin=$login");
        }
    }

    if ($FORM{raw_format}) {
        $r->content_type('text/plain');

        for my $role (qw/superusers managers placers mediaplanners supports superreaders/) {
            for my $row (@{$vars->{$role}}) {
                my @fields = ($row->{login});
                if ($FORM{raw_format} eq 'with_roles') {
                    my $str_role = $role;
                    $str_role =~ s/s$//;
                    push @fields, $str_role;
                }
                push @fields, $row->{email}, $row->{FIO}, $row->{createtime_text}, $row->{statusBlocked};
                print join("\t", map {defined $_ ? $_ : ''} @fields) . "\n";
            }
        }
        return;

    } else {
        if ($FORM{agencies}) {
            return respond_template($r, $template, 'staff_agencies.html', $vars);
        } else {
            return respond_template($r, $template, 'staff.html', $vars);
        }
    }
}

# ---------------------------------------------------------------------------------------------------------------------------

=head2 cmd_manageTeamleaders

  manage (super)teamleaders
  add, delete, move managers between teamleaders
  for manageSTeamleaders and manageTeamleaders cmds

=cut

sub cmd_manageTeamleaders :Cmd(manageTeamleaders, manageSTeamleaders)
    :Rbac(Cmd => manageTeamleaders, Perm => ManageTeamleaders)
    :Rbac(Cmd => manageSTeamleaders, Perm => ManageSTeamleaders)
    :CheckCSRF
    :Description('управление группами менеджеров')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS/};
    my %FORM = %{$_[0]{FORM}};

    my $vars;

    # get FROM data
    my %old_tuids;
    for (keys %FORM) {
        $old_tuids{$1} = $FORM{$_} if m/^cb_(\d+)$/;
    }
    my $error_no;
    my $is_superteam = $FORM{cmd} eq 'manageSTeamleaders' ? 1 : 0;

    # submit pressed -----------------------------------------------------------
    if (defined $FORM{bt_submit}) {
        if (defined $FORM{new_tuid} && keys %old_tuids) {
            my $supervisor_client_id = $FORM{new_tuid} ne 'none' ? get_clientid(uid => $FORM{new_tuid}) : undef;
            error("No ClientID for supervisor uid=$FORM{new_tuid}") unless ($supervisor_client_id || $FORM{new_tuid} eq 'none');

            my $client_id_by_uid = get_key2clientid(uid => [keys %old_tuids]);

            # apply changes
            for my $muid (keys %old_tuids) {
                my $manager_client_id = $client_id_by_uid->{ $muid };
                error("No ClientID for manager uid=$muid") unless $manager_client_id;

                my $apply_changes;
                my $errorcode;
                if ($old_tuids{$muid} ne 'none' && $FORM{new_tuid} ne 'none') {
                    # 1. move between teamleaders
                    if ($is_superteam) {
                        $errorcode = rbac_move_team_between_superteamleaders($rbac, $muid, $old_tuids{$muid}, $FORM{new_tuid});
                    } else {
                        $errorcode = rbac_move_manager_between_teamleaders($rbac, $muid, $old_tuids{$muid}, $FORM{new_tuid});
                    }

                    $error_no = 2 if $errorcode; # "перенести не удалось"
                    $apply_changes = 1;
                } elsif ($old_tuids{$muid} eq 'none') {
                    # 2. add to teamleader
                    if ($is_superteam) {
                        $errorcode = rbac_add_team_to_superteamleader($rbac, $muid, $FORM{new_tuid});
                    } else {
                        $errorcode = rbac_add_manager_to_teamleader($rbac, $muid, $FORM{new_tuid});
                    }

                    $error_no = 2 if $errorcode; # "перенести не удалось"
                    $apply_changes = 1;
                } elsif ($FORM{new_tuid} eq 'none') {
                    # 3. remove from teamleader
                    if ($is_superteam) {
                        $errorcode = rbac_delete_team_from_superteamleader($rbac, $muid, $old_tuids{$muid});
                    } else {
                        $errorcode = rbac_delete_manager_from_teamleader($rbac, $muid, $old_tuids{$muid});
                    }

                    $error_no = 2 if $errorcode; # "перенести не удалось"
                    $apply_changes = 1;
                } elsif ($old_tuids{$muid} eq 'none' && $FORM{new_tuid} eq 'none') {
                    $errorcode = 1;
                    $error_no = 1; # "Исходные параметры не верны"
                }
                
                if (!$errorcode && $apply_changes) {
                    set_manager_supervisor($manager_client_id, $supervisor_client_id);
                }
            }
            # /apply changes
        } else {
            $error_no = 1; # "Исходные параметры не верны"
        }

        return redirect($r, "$SCRIPT?cmd="
                                   . ($is_superteam ? 'manageSTeamleaders' : 'manageTeamleaders')
                                   . (defined $error_no ? "&error_no=$error_no" : "")
                                   . ($FORM{show_hidden} ? '&show_hidden=1' : '')
                      );
    }

    # get all data -----------------------------------------------------------
    my $raw_data = $is_superteam ? rbac_get_superteamleaders_data($rbac) : rbac_get_teamleaders_data($rbac, ($login_rights->{is_superteamleader} ? $UID : 0));

    my $users_data = get_users_data([grep { is_valid_id($_) } keys(%$raw_data), map { @$_ } values(%$raw_data)],
                                    [qw/uid email valid LastChange fio phone sendNews sendWarn createtime ClientID
                                        login hidden sendAccNews not_resident statusArch statusBlocked description
                                        lang captcha_freq allowed_ips statusYandexAdv showOnYandexOnly/]);
    for my $tl_uid (keys %$raw_data) {
        if (defined $users_data->{$tl_uid}) {
            $vars->{user}->{$tl_uid} = $users_data->{$tl_uid};
            $vars->{user}->{$tl_uid}->{createtime_text} = human_date($users_data->{$tl_uid}->{createtime});
        }
        for my $muid (@{ $raw_data->{$tl_uid} }) {
            if (defined $users_data->{$muid}) {
                $vars->{user}->{$muid} = $users_data->{$muid};
                $vars->{user}->{$muid}->{createtime_text} = human_date($users_data->{$muid}->{createtime});
            }

            # get managers count in teams (for show expand icon for superteams)
            if ($is_superteam) {
                my $managers = rbac_get_managers_of_teamleader_with_idm($rbac, $muid) || [];
                $vars->{managers_count_in_team}{$muid} = scalar @$managers;
            }
        }

        # sorting
        $vars->{teams}{$tl_uid} = [sort {uc($vars->{user}{$a}{fio}) cmp uc($vars->{user}{$b}{fio})}
                                   grep {defined $vars->{user}{$_}{login} && ! exists $raw_data->{$_}} # exist in DB and this manager is not teamleader
                                   @{ $raw_data->{$tl_uid} }
                                  ];
    }

    # sorting teamleaders
    $vars->{teams_sort_by_fio} = [sort {uc($vars->{user}{$a}{fio}) cmp uc($vars->{user}{$b}{fio})} grep {/^\d+$/} keys %{ $vars->{teams} }];
    push @{ $vars->{teams_sort_by_fio} }, 'none';
    # /get all data

    return respond_template($r, $template, 'manage_teamleaders.html', $vars);
}

sub cmd_ajaxModifyClientPixels :Cmd(ajaxModifyClientPixels)
    :Description('разрешение/запрещение дополнительных пикселей для клиента')
    :CheckCSRF
    :Rbac(Code => rbac_cmd_internal_user_only, Role => [super, support])
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c/};

    my %FORM = %{$_[0]{FORM}};

    my $provider = $FORM{provider};
    my $campaign_type = $FORM{campaign_type};
    my $criterion_type = $FORM{criterion_type} // 'none';
    my $result;
    if ($provider && $campaign_type && $login_rights->{client_role} eq 'client') {
        if ($FORM{action} eq 'add') {
            if (($campaign_type eq 'cpm_banner' || $campaign_type eq 'cpm_yndx_frontpage' || $campaign_type eq 'cpm_price') && $criterion_type eq 'none' || $campaign_type eq 'cpm_deals' && $criterion_type ne 'none') {
                my $is_yandex_page = $criterion_type eq 'yandex' ? 1 : 0;
                my $has_private_criterion = $criterion_type eq 'private' ? 1 : 0;
                if (Direct::Model::Pixel->new(provider => $provider)->need_to_check_provider_permission($campaign_type, $is_yandex_page, $has_private_criterion)
                    && $provider ne 'omi') {
                    $result = do_sql(PPC(uid => $uid), 'INSERT IGNORE INTO client_pixel_providers (ClientID,provider,campaign_type,criterion_type) VALUES (?,?,?,?)', $c->client_client_id, $provider, $campaign_type, $criterion_type);
                }
            }
        } elsif ($FORM{action} eq 'delete') {
            do_sql(PPC(uid => $uid), 'DELETE FROM client_pixel_providers WHERE ClientID = ? AND provider = ? AND campaign_type = ? AND criterion_type = ?', $c->client_client_id, $provider, $campaign_type, $criterion_type);
            $result = 1;
        }
    }

    return respond_json($r, { status => $result ? 'ok' : 'failed' });
}

sub cmd_ajaxAcceptAssessorOffer :Cmd(ajaxAcceptAssessorOffer)
    :Description('Принять оферту на обслуживание асессорами')
    :CheckCSRF
    :Rbac(Role => [client])
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c/};

    die "Cant get client_id in AcceptAssessorOffer" unless $c->client_client_id;
    set_user_custom_options($c->client_client_id, {assessor_offer_accepted => 1});
}

1;
