package DoCmdWallet;

=head1 NAME

DoCmdWallet - работа с "единым счетом"
https://jira.yandex-team.ru/browse/DIRECT-19914

=cut

# $Id$

use warnings;
use strict;

use base qw/DoCmd::Base/;

use Settings;
use Common qw/:globals :subs/;
use Client;
use Campaign;
use Currency::Pseudo;
use Property;
use Tools;
use HttpTools;
use RBACElementary;
use RBACDirect;
use User qw/get_user_options/;
use Wallet;
use WalletUtils;
use WalletLock;
use WalletUtils;
use Direct::Wallets;
use BalanceQueue;
use Yandex::Balance::Simple qw/balance_simple_create_binding
                               balance_simple_list_payment_methods
                               balance_simple_do_binding/;
use Yandex::Balance qw/balance_get_client_persons/;

use Yandex::I18n;
use Yandex::HashUtils;
use Yandex::IDN qw/is_valid_email/;
use Direct::ResponseHelper;
use Direct::PredefineVars;
use Direct::Validation::UserData qw/check_user_data/;
use Direct::Validation::Wallets;
use DoCmd::Checks;
use User qw/get_user_data/;

use utf8;
use List::MoreUtils qw/none/;
use List::Util qw/any max min first/;

# Можно привязать не более пяти банковских карт
use constant MAX_CARDS_COUNT => 5;
use constant OVERWRITED_RECOMMENDATION_SUM_MIN_RUB => 10000;

# --------------------------------------------------------------------

=head2 _isPaymenAllowed

    Проверка что клиент не социальный, а если социальный что ему разрешена оплата

=cut

sub _is_payment_allowed {
    my ($client_id) = @_;

    my $is_social = Client::ClientFeatures::social_advertising_feature($client_id);
    return !$is_social || Client::ClientFeatures::social_advertising_payable_feature($client_id);
}

=head2 _calc_recurring_payment_sum_offer

    Вычислить предлагаемую сумму повторного платежа (DIRECT-142057)

=cut

sub _calc_recurring_payment_sum_offer {
    my ($client_wallet, $options) = @_;

    if (defined $client_wallet->{wallet_camp} && $client_wallet->{wallet_camp}->{day_budget}->{sum} > 0) {
        return $client_wallet->{wallet_camp}->{day_budget}->{sum} * 7;
    }

    # Тут кампании, привязанные к кошельку. Архивные отфильтрованы ранее.
    my $camps = $client_wallet->{text_camps};
    my @alive_camps = grep {$_->{statusShow} ne 'No' && !is_campaign_finished(cid => $_->{cid}, finish_time => $_->{finish_time})} @$camps;

    # Ищем максимальное значение недельного ограничения.
    my $max_week_budget = max(map { $_->{autobudget_sum} } @alive_camps);
    if (defined $max_week_budget) {
        return $max_week_budget;
    }

    return $options->{payment_sum};
}

=head2 clientWallet

     Страница общего счета клиента

=cut

sub cmd_clientWallet
    :Cmd(clientWallet, clientWalletPayment)
    :Rbac(Code => [rbac_cmd_by_owners], ExceptRole => [media, placer], AllowDevelopers => 1)
    :Description('Страница общего счета клиента')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c, $cvars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c vars/};
    my %FORM = %{$_[0]{FORM}};

    my $vars = {};

    if ($login_rights->{client_role} !~ /client$/) {
        error(iget("Общий счет может быть только у клиентов"));
    }

    if (!_is_payment_allowed( $c->client_client_id )) {
        error(iget("Для социальной рекламы оплата недоступна"));
    }

    my $AgencyID = 0;
    if (defined $FORM{AgencyID}) {
        $AgencyID = rbac_is_agency_client_id_of_client($rbac, $FORM{AgencyID}, $uid);
        error(iget("Агентство указано неверно")) unless $AgencyID;
    }

    my $ulogin = $FORM{ulogin};

    if (!defined $ulogin) {
        my $user_info = get_user_data($uid, [qw/login/]);
        $ulogin = $user_info->{login};
    }

    my $uatraits = HttpTools::get_uatraits(http_get_header($r, 'User-Agent'));
    my $has_mobile_version = $uatraits->{isMobile} && !$uatraits->{isTablet};

    if ($has_mobile_version && defined $FORM{showpopup}) {
        return redirect($r, '/wizard/campaigns', {ulogin => $ulogin, showPayment => 1, paySum => $FORM{topay}});
    }

    my $client_currencies = get_client_currencies($c->client_client_id);
    hash_merge($vars, $client_currencies);
    my $client_wallets = Wallet::get_wallets_by_uids([{c => $c, agency_client_id => $AgencyID, client_currency => $client_currencies->{work_currency}}]
                                                     , with_bonus => 1
                                                     , with_outlay => 1
                                                     , with_emails_sms => 1
                                                     , with_camp_stop_daily_budget_stats => 1
                                                    );

    my $client_wallet = (grep {$_->{agency_client_id} == $AgencyID} @$client_wallets)[0];

    $vars->{is_mcc_control} = ($login_rights->{is_mcc} && $login_rights->{mcc_client_ids}{$c->client_client_id}) ? 1 : 0;

    if ($AgencyID) {
        $vars->{wallet}->{agencies} = {$AgencyID => $client_wallet->{wallet_camp}};
    } else {
        $vars->{wallet}->{self} = $client_wallet->{wallet_camp};
    }

    my $is_owners = rbac_check_allow_edit_camps($rbac, $UID, [map {$_->{cid}} @{$client_wallet->{text_camps}}]);
    my $allow_transfer_money = $login_rights->{is_any_client} && $AgencyID
                               ? {map {$_ => 1} @{rbac_get_allow_transfer_money_agencies($rbac, $c->client_client_id)}}
                               : {};

    $vars->{pay_method} = $client_wallet->{wallet_camp}->{wallet_cid}
                          ? check_method_pay($client_wallet->{wallet_camp}->{wallet_cid})
                          : '';

    $vars->{MAX_MONEY_WARNING_VALUE} = $Settings::MAX_MONEY_WARNING_VALUE;
    my $client_nds = get_client_NDS($c->client_client_id);
    $vars->{client_nds} = $client_nds;
    if ($client_currencies->{work_currency} eq 'YND_FIXED') {
        $vars->{pseudo_currency} = get_pseudo_currency(hostname => $r->hostname);
        $vars->{pseudo_currency}->{name} = iget($vars->{pseudo_currency}->{name});
    }
    $vars->{is_wallet_locked} = WalletLock::is_wallet_locked($c->client_client_id);
    $vars->{is_offer_accepted} = Common::is_offer_accepted($uid);
    $vars->{is_moderation_offer_enabled_for_dna} = Client::ClientFeatures::has_moderation_offer_enabled_for_dna($c->client_client_id);
    $vars->{is_promocode_hint_enabled} = Client::ClientFeatures::is_promocode_hint_enabled($c->login_rights->{ClientID});

    if ($client_wallet->{wallet_cid} && Direct::Wallets->can_client_use_autopay($rbac, $uid)) {
        my $wallet = Direct::Wallets->get_by(campaign_id => $client_wallet->{wallet_cid})->items->[0];
        if (defined $wallet) {
            if (!$wallet->has_autopay) {
                $wallet->autopay(Direct::Model::Autopay->new(id=>$wallet->id, user_id=>$uid, currency=>$client_currencies->{work_currency}));
            } else {
                $wallet->autopay->currency($client_currencies->{work_currency});
            }

            if ($client_wallet->{wallet_camp}->{allow_pay}) {
                my $autopay_settings = {};
                hash_merge $autopay_settings, $wallet->autopay->to_template_hash($wallet);
                my $get_list_payment_methods = Direct::Wallets->get_list_payment_methods($rbac, $uid, http_remote_ip($r));

                if ($get_list_payment_methods->{error}) {
                    $autopay_settings->{list_payment_methods_error} = $get_list_payment_methods->{error};
                } else {
                    hash_merge $autopay_settings, $get_list_payment_methods;
                }

                my $autopay = {autopay_settings => $autopay_settings,
                               available_paymethod_types => Direct::Wallets->get_available_autopay_paymethod_types($client_currencies->{work_currency}),
                               max_cards_count => MAX_CARDS_COUNT,
                };

                $autopay->{read_only} = (none {$c->login_rights->{"role"} eq $_} qw/client super/) ? 1 : 0;
                $autopay->{allow_bind_unbind} = ($c->login_rights->{"role"} eq "client") ? 1 : 0;

                $vars->{autopay} = $autopay;
            }
        }
    }
    my $client_data = get_client_data($c->client_client_id, [qw/country_region_id work_currency agency_client_id non_resident/]);
    hash_copy $vars, $client_data, qw(country_region_id);

    for my $agency (keys %{$vars->{wallet}->{agencies}}) {
        Campaign::correct_sum_and_total($vars->{wallet}->{agencies}->{$agency});
    }

    $vars->{enable_cpm_deals_campaigns} = Direct::PredefineVars::_enable_cpm_deals_campaigns($c);
    if ($vars->{enable_cpm_deals_campaigns}){
        $vars->{new_deals_count} = Client::get_count_received_deals($c->login_rights->{ClientID});
    }
    $vars->{enable_content_promotion_video_campaigns} = Direct::PredefineVars::_enable_content_promotion_video_campaigns($c);
    $vars->{enable_cpm_yndx_frontpage_campaigns} = Direct::PredefineVars::_enable_cpm_yndx_frontpage_campaigns($c);

    $vars->{features_enabled_for_operator_all} = $cvars->{features_enabled_for_operator_all};
    $vars->{features_enabled_for_client_all} = $cvars->{features_enabled_for_client_all};
    $vars->{features_enabled_for_client} //= {};
    hash_merge $vars->{features_enabled_for_client}, Client::ClientFeatures::get_features_enabled_for_client(
        $c->client_client_id,
        [qw/new_cash_payment_enabled pay_with_cash_disabled support_chat b2b_balance_cart enabled_change_offer_for_clients_from_turkey/]);

    hash_merge $vars->{features_enabled_for_client}, Client::ClientFeatures::get_features_enabled_for_client(
        $c->login_rights->{ClientID}, [qw/is_grid_enabled is_hide_old_show_camps is_show_dna_by_default facelift_disabled_for_dna/]);

    hash_merge $vars, WalletUtils::get_auto_overdraft_params($c->client_client_id, $vars->{wallet}->{self}, $client_currencies->{work_currency}, $client_nds);

    if (WalletUtils::is_new_payment_workflow_enabled($c->client_client_id, $client_data)) {

        $vars->{new_payment_workflow_enabled} = 1;

        my $autooverdraft_active = 0;
        if ($vars->{autooverdraft_params} && $vars->{autooverdraft_params}->{auto_overdraft_lim}) {
            $autooverdraft_active = ($vars->{autooverdraft_params}->{auto_overdraft_lim} > 0);
        }

        hash_merge $vars, WalletUtils::get_new_payment_workflow_values($uid, $c->client_client_id, $client_wallet->{wallet_cid},
            http_remote_ip($r), $vars->{autooverdraft_enabled}, $autooverdraft_active);
    }

    my $options = get_user_options($uid);
    if (exists $options->{payment_sum}) {
        # Повторный платеж
        if (Client::ClientFeatures::change_recurring_payment_sum_offer($c->client_client_id)) {
            $vars->{payment_sum_offer} = _calc_recurring_payment_sum_offer($client_wallet, $options);
        }
    } else {
        # Первый платеж
        if (Client::ClientFeatures::change_first_payment_min_sum_recommendation($c->client_client_id)
            && defined $client_wallet->{wallet_camp}
            && $client_wallet->{wallet_camp}->{currency} eq 'RUB')
        {
            $vars->{OVERWRITED_RECOMMENDATION_SUM_MIN} = OVERWRITED_RECOMMENDATION_SUM_MIN_RUB;
        }
    }

    $vars->{customMinPay} = calc_min_pay_sum($c->client_client_id, $client_currencies->{work_currency});
    if (Property->new('NEW_CARD_PAY_LIMIT')->get(60)
        && $client_currencies->{work_currency} eq 'RUB') {
        # Переопределяем на фронте RecommendWarnSumm по проперте
        # Включаем в DIRECT-149354
        # После включения выпиливаем в DIRECT-149770
        $vars->{customRecommendWarnSumm} = 249999;
    }

    $vars->{isCashbackAvailable} = Client::has_cashbacks_available($c, $client_data);

    if ($FORM{tt}) {
        return respond_template($r, $template, 'client-wallet/client-wallet.tt2', $vars);
    } else {
        return respond_bem($r, $c->reqid, $vars, source=>"data3");
    }
}

# --------------------------------------------------------------------

=head2 enableWallet

     Подключение общего счета

=cut

sub cmd_enableWallet
    :Cmd(enableWallet)
    :Rbac(Code => [rbac_cmd_by_owners], ExceptRole => [media, superreader, placer, limited_support])
    :CheckCSRF
    :Description('Подключение общего счета')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c/};
    my %FORM = %{$_[0]{FORM}};

    my $vars = {};
    my $AgencyID;

    if (!_is_payment_allowed( $c->client_client_id )) {
        error(iget("Для социальной рекламы оплата недоступна"));
    }

    if (defined $FORM{AgencyID}) {
        $AgencyID = rbac_is_agency_client_id_of_client($rbac, $FORM{AgencyID}, $uid);
        error(iget("Агентство указано неверно")) unless $AgencyID;
    }

    my $agency_uid = $c->login_rights->{agency_control}
                     ? $UID
                     : $AgencyID
                       ? rbac_get_chief_rep_of_agency($AgencyID)
                       : 0;

    my $client_currencies = get_client_currencies($c->client_client_id, uid => $uid);
    my $result = Wallet::enable_wallet($c, $client_currencies->{work_currency}, $agency_uid);

    my $wallet_cid;
    if ($result->{error}) {
        error($result->{error});
    } elsif ($result->{wallet_cid}) {
        $wallet_cid = $result->{wallet_cid};
    } else {
        die "enable_wallet: invalid wallet_cid";
    }

    BalanceQueue::add_to_balance_info_queue($UID, 'cid', $wallet_cid, BalanceQueue::PRIORITY_CAMP_ON_ENABLE_WALLET);

    return redirect($r, $SCRIPT, {cmd => 'clientWallet'
                                  , uid_url => $FORM{uid_url}
                                  , ($AgencyID ? (AgencyID => $AgencyID) : ())
                                 });
}

# --------------------------------------------------------------------

=head2 ajaxSaveWalletSettings

     Сохранение параметров уведомлений на общий счет
       email
       порог уровня остатка средств на кампании
       SMS типы и время

=cut

sub cmd_ajaxSaveWalletSettings
    :Cmd(ajaxSaveWalletSettings)
    :Rbac(Code => [rbac_cmd_by_owners_of_cid_through_uid], ExceptRole => [media, superreader, placer, limited_support])
    :RequireParam(cid => 'Cid')
    :CheckCSRF
    :Description('Сохранение параметров уведомлений на общий счет')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c/};
    my %FORM = %{$_[0]{FORM}};

    my $new_settings = {};

    if (defined $FORM{money_warning_value}) {
        my $money_warning_value = $FORM{money_warning_value};
        return respond_json($r, {error => iget('Задано неверное значение порога')})
            unless $money_warning_value =~ /^[0-9]+$/
                   && $money_warning_value > 0
                   && $money_warning_value <= $Settings::MAX_MONEY_WARNING_VALUE;

        $new_settings->{money_warning_value} = $money_warning_value;
    }

    if (defined $FORM{sms_time_hour_from} || defined $FORM{sms_time_hour_to} || defined $FORM{sms_time_min_from} || defined $FORM{sms_time_min_to}) {
        $new_settings->{sms_time} = sms_time2string($FORM{sms_time_hour_from}, $FORM{sms_time_min_from}, $FORM{sms_time_hour_to}, $FORM{sms_time_min_to});
    }

    if (defined $FORM{notify_order_money_in_sms} || defined $FORM{active_orders_money_out_sms} || defined $FORM{notify_cashback_in_sms}) {
        my @sms_flags;
        push @sms_flags, 'notify_order_money_in_sms' if $FORM{notify_order_money_in_sms};
        push @sms_flags, ('active_orders_money_out_sms', 'active_orders_money_warning_sms') if $FORM{active_orders_money_out_sms};
        push @sms_flags, 'notify_cashback_in_sms' if $FORM{notify_cashback_in_sms};
        $new_settings->{sms_flags} = join(',', @sms_flags);
    }

    if (defined $FORM{email}) {
        return respond_json($r, {error => iget("Неверно указан e-mail")}) if ! is_valid_email($FORM{email});
        $new_settings->{email} = $FORM{email};
    }

    $FORM{paused_by_day_budget_email} = delete $FORM{email_notify_paused_by_day_budget};
    my %new_opts = (
            map { defined($FORM{$_}) ? ($_ => $FORM{$_}) : () } qw/
                    paused_by_day_budget_email
                    paused_by_day_budget_sms
                /
        );

    my $result = Wallet::update_wallet_sms_email_settings($FORM{cid}, $new_settings, %new_opts);
    return respond_json($r, {error => iget('Не заданы параметры уведомлений')}) unless $result;

    return respond_json($r, {result => 'ok'});
}

=head2 cmd_autopaySettings

    Достает для страницы включения и редактирования автопополнения счета.

=cut

sub cmd_autopaySettings
    :Cmd(autopaySettings)
    :Rbac(Code => [rbac_cmd_by_owners], ExceptRole => [media, placer], AllowDevelopers => 1)
    :Description('Страница настройки автопополнения счета клиента')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c, $cvars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c vars/};

    error(iget("Вам недоступна функция автопополнения счета.")) unless Direct::Wallets->can_client_use_autopay($rbac, $uid);

    if (!_is_payment_allowed( $c->client_client_id )) {
        error(iget("Для социальной рекламы оплата недоступна"));
    }

    my $autopay_settings = {};

    my ($wallet, $client_currency) = Direct::Wallets->get_actual_client_wallet($c);

    # Если так и не нашли общий кошелек, значит бросаем ошибку.
    error(iget("Общий счет еще не настроен.")) if !defined $wallet;

    error(iget("Вам недоступна функция автопополнения счета.")) if !$wallet->allow_pay;

    if (!$wallet->has_autopay) {
        $wallet->autopay(Direct::Model::Autopay->new(id=>$wallet->id, user_id=>$uid, currency=>$client_currency));
    }
    hash_merge $autopay_settings, $wallet->autopay->to_template_hash($wallet);
    my $list_payment_methods = Direct::Wallets->get_list_payment_methods($rbac, $uid, http_remote_ip($r));

    return respond_bem($r, $c->reqid, $list_payment_methods, source=>"data3") if defined $list_payment_methods->{error};

    if ($wallet->is_active_autopay) {
        my $allowed_card = first { $_->{paymethod_id} eq $wallet->autopay->paymethod_id } @{$list_payment_methods->{cards}};
        $list_payment_methods->{cards}  = [ $allowed_card // () ];
    } else {
        $list_payment_methods->{cards} = [];
        delete $autopay_settings->{paymethod_id};
    }
    hash_merge $autopay_settings, $list_payment_methods;

    my $vars = {autopay_settings => $autopay_settings,
                available_paymethod_types => Direct::Wallets->get_available_autopay_paymethod_types($client_currency),
                max_cards_count => MAX_CARDS_COUNT,
    };

    $vars->{features_enabled_for_operator_all} = $cvars->{features_enabled_for_operator_all};
    $vars->{features_enabled_for_client_all} = $cvars->{features_enabled_for_client_all};

    $vars->{is_offer_accepted} = Common::is_offer_accepted($uid);
    $vars->{is_moderation_offer_enabled_for_dna} = Client::ClientFeatures::has_moderation_offer_enabled_for_dna($c->client_client_id);

    $vars->{read_only} = (none {$c->login_rights->{"role"} eq $_} qw/client super/) ? 1 : 0;
    $vars->{allow_bind_unbind} = ($c->login_rights->{"role"} eq "client") ? 1 : 0;
    $vars->{client_nds} = get_client_NDS($c->client_client_id);

    hash_merge $vars, WalletUtils::get_autopay_suggest_sums();

    my $persons = balance_get_client_persons($c->client_client_id);
    $vars->{ur_persons} = [ grep { $_->{type} eq 'ur' && !$_->{hidden} } @$persons ];

    $vars->{features_enabled_for_client} = {};

    if (Client::ClientFeatures::has_new_payment_workflow_enabled_feature($c->client_client_id)) {
        $vars->{features_enabled_for_client}->{new_payment_workflow_enabled} = 1;
    }

    hash_merge $vars->{features_enabled_for_client}, Client::ClientFeatures::get_features_enabled_for_client(
        $c->client_client_id,
        [qw/support_chat b2b_balance_cart enabled_change_offer_for_clients_from_turkey facelift_disabled_for_dna/]);

    return respond_bem($r, $c->reqid, $vars, source=>"data3");
}

=head2 cmd_ajaxSaveAutopaySettings

    Сохранение настроек автопополнения платежа.
    Аяксовый контроллер.
    Возвращает {ok=>1} в случае успеха,
               {error => "текст ошибки"} в случае проблем.

=cut

sub cmd_ajaxSaveAutopaySettings
    :Cmd(ajaxSaveAutopaySettings)
    :Rbac(Code => [rbac_cmd_by_owners,rbac_cmd_pay], Role => [client, super])
    :CheckCSRF
    :Description('Сохранение настройки автопополнения на общий счет клиента')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c/};
    my %FORM = %{$_[0]{FORM}};

    unless (Direct::Wallets->can_client_use_autopay($rbac, $uid)) {
        error(iget("Вам недоступна функция автопополнения счета."));
    }

    if (!_is_payment_allowed( $c->client_client_id )) {
        error(iget("Для социальной рекламы оплата недоступна"));
    }

    unless (defined $FORM{json_autopay}) {
        return respond_json($r, {error => iget("Неверные входные данные автопополнения счета")});
    }

    my ($wallet, $currency) = Direct::Wallets->get_actual_client_wallet($c);
    unless (defined $wallet) {
        error(iget("Общий счет еще не настроен."));
    }

    if (!$wallet->allow_pay) {
        error(iget("Вам недоступна функция автопополнения счета."));
    }

    my $autopay_values = $FORM{json_autopay};
    $autopay_values->{wallet_cid} = $wallet->id;
    $autopay_values->{user_id} = $uid;
    my $pre_autopay_validation_result = check_user_data(
            $autopay_values, \%DoCmd::Checks::AUTOPAY_SETTINGS
    );

    # Если есть ошибки пред-валидации, то выведем их в STDERR (для диагностики), а клиенту отдадим стандартное сообщение
    if (!$pre_autopay_validation_result->is_valid) {
        warn join("\n", @{$pre_autopay_validation_result->get_error_descriptions});
        return respond_json($r, {error => iget("Неверные входные данные автопополнения счета")});
    }

    if ($autopay_values->{autopay_mode}) {
        if ($autopay_values->{autopay_mode} ne 'none' &&
            !Direct::Wallets->is_user_paymethod_id($rbac, $uid, $autopay_values->{paymethod_type}, $autopay_values->{paymethod_id}, http_remote_ip($r))
           )
        {
            return respond_json($r, {error => iget("Карта или кошелек принадлежат другому представителю. Выберите для оплаты вашу карту или кошелек.")})
        }
        my $autopay_model = Direct::Model::Autopay->new(
            id => $autopay_values->{wallet_cid},
            %{hash_cut $autopay_values, qw/user_id paymethod_type paymethod_id remaining_sum payment_sum/},
            currency => $currency
        );
        $wallet->autopay_mode($autopay_values->{autopay_mode});

        if ($wallet->is_active_autopay()) {
            my $person_id_to_set;

            if ($autopay_values->{person_id} && $autopay_values->{ur_person}) {
                my $persons = balance_get_client_persons($c->client_client_id);
                my $found_person_num = scalar( grep { $_->{type} eq 'ur' && !$_->{hidden} && $_->{id} eq $autopay_values->{person_id} } @$persons );
                if ($found_person_num > 0) {
                    $person_id_to_set = $autopay_values->{person_id};
                } else {
                    return respond_json($r, {error => iget("Выбранный плательщик недоступен")});
                }
            } else {
                $person_id_to_set = WalletUtils::get_or_create_ph_person($UID, $autopay_values->{wallet_cid});
            }
            $autopay_model->person_id($person_id_to_set);

            $wallet->autopay($autopay_model);
            my $vr = Direct::Validation::Wallets::validate_autopay_sum_limits($wallet);
            unless ($vr->is_valid) {
                return respond_json($r, {error => $vr->get_first_error_description});
            }
        }

        Direct::Model::Wallet::Manager->new(items => [$wallet])->update();
    }

    return respond_json($r, {result => 'ok'});

}

=head2 cmd_ajaxGetBindingForm

    Взятие из Баланса ссылки на форму для привязки карты

=cut

sub cmd_ajaxGetBindingForm
    :Cmd(ajaxGetBindingForm)
    :CheckCSRF
    :Rbac(Code => [rbac_cmd_pay], Role => [client], AllowDevelopers => 0)
    :Description('Взятие из Баланса ссылки на форму для привязки карты')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c/};
    my %FORM = %{$_[0]{FORM}};

    error(iget("Вам недоступна функция автопополнения счета.")) unless Direct::Wallets->can_client_use_autopay($rbac, $uid);

    if (!_is_payment_allowed( $c->client_client_id )) {
        error(iget("Для социальной рекламы оплата недоступна"));
    }

    # Берем токен Баланса. С этого момент начинается отсчет 20 минут на привязку карты.
    my $lang = Yandex::I18n::current_lang();

    # Для украинского языка должны отсылать uk а не ua
    # https://wiki.yandex-team.ru/Balance/Simple/xmlrpc/#imenovannyeparametry36
    if ($lang eq 'ua') {
        $lang = 'uk';
    }
    my $csrf_token = $FORM{csrf_token};
    die "invalid csrf token '$csrf_token'" unless $csrf_token =~ /^[a-zA-Z0-9_\-]+$/;
    my $create_binding_response = balance_simple_create_binding(
                                    {
                                        uid => $uid,
                                        return_path=>"$SCRIPT?cmd=ajaxBindingPaymentResponse&status={status}&payment_method_id={payment_method_id}&csrf_token=".$csrf_token,
                                        lang => $lang,
                                    });

    my %redirect_params = (cmd => 'ajaxBindingPaymentError');
    return redirect($r, $SCRIPT, {%redirect_params, error => iget("Не удалось привязать карту. Повторите попытку через 20 минут.")}) if $create_binding_response->{status_code} && $create_binding_response->{status_code} eq 'too_many_active_bindings';

    return redirect($r, $SCRIPT, {%redirect_params, error => iget('Проблемы с созданием токена.')}) unless $create_binding_response->{status} eq 'success' && defined $create_binding_response->{purchase_token};

    # Берем ссылку на форму Баланса.
    my $do_binding_response = balance_simple_do_binding(hash_cut $create_binding_response, qw/purchase_token/);

    return redirect($r, $SCRIPT, {%redirect_params, error => iget('Проблемы с созданием токена.')}) unless $do_binding_response->{status} eq 'success' && defined $do_binding_response->{binding_form};


    return redirect($r, $do_binding_response->{binding_form}->{_TARGET},
                    hash_cut $do_binding_response->{binding_form}, qw/purchase_token/);

}

=head2 cmd_ajaxBindingPaymentError

    отображение ошибок из cmd=ajaxGetBindingForm

=cut

sub cmd_ajaxBindingPaymentError
    :Cmd(ajaxBindingPaymentError)
    :CheckCSRF
    :Rbac(Code => [rbac_cmd_pay], Role => [client], AllowDevelopers => 0)
    :Description('отображение ошибок из cmd=ajaxGetBindingForm')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c/};
    my %FORM = %{$_[0]{FORM}};

    return respond_bem($r, $c->reqid, {error => $FORM{error}, error_code => $FORM{error_code}}, source=>"data3");
}

=head2 cmd_ajaxBindingPaymentResponse

    вызывается из баланса после привязки карты. Если привязка прошла успешно, то загружается смисок методов платещей,
    если неуспешно - то редиректит на повторную попытку привязки (cmd=ajaxGetBindingForm)

=cut

sub cmd_ajaxBindingPaymentResponse
    :Cmd(ajaxBindingPaymentResponse)
    :CheckCSRF
    :Rbac(Code => [rbac_cmd_pay], Role => [client], AllowDevelopers => 0)
    :Description('Вызывается как ответ на привязку карты в Балансе')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c) = @{$_[0]}{
        qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c/};
    my %FORM = %{$_[0]{FORM}};

    error(iget("Вам недоступна функция автопополнения счета.")) unless Direct::Wallets->can_client_use_autopay($rbac, $uid);

    my $list_payment_methods = Direct::Wallets->get_list_payment_methods($rbac, $uid, http_remote_ip($r));

    my $status = $FORM{status};
    my $payment_method_id = $FORM{payment_method_id};
    my $allowed_binding_interval = time() - 300;

    for (@{$list_payment_methods->{cards}}) {
        if ($_->{binding_ts} > $allowed_binding_interval) {
            $payment_method_id = $_->{paymethod_id};
            $allowed_binding_interval = $_->{binding_ts};
            $status = 'success';
        }
    }

    if (defined $status && $status eq 'success') {
        my $allowed_card = first { $_->{paymethod_id} eq $payment_method_id } @{$list_payment_methods->{cards}};
        $list_payment_methods->{cards}  = [$allowed_card];

        return respond_bem($r, $c->reqid, {list_payment_methods => $list_payment_methods}, source=>"data3");
    } else {
        if (@{$list_payment_methods->{cards} || []} >= MAX_CARDS_COUNT){
            return redirect($r, $SCRIPT, {
                cmd => 'ajaxBindingPaymentError',
                error => iget("Достигнут лимит привязанных карт."),
                error_code => 'TOO_MANY_CARDS',
            });
        } else {
            $list_payment_methods->{cards} = [];
            return respond_bem($r, $c->reqid, {list_payment_methods => $list_payment_methods}, source=>"data3");
        }
    }
}

=head2 cmd_ajaxResumeAutopay

    Возобновить автоплатежи клиента

=cut

sub cmd_ajaxResumeAutopay
    :Cmd(ajaxResumeAutopay)
    :CheckCSRF
    :Rbac(Code => [rbac_cmd_pay], Role => [client, super], AllowDevelopers => 0)
    :Description('Возобновить автоплатежи')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c/};
    my %FORM = %{$_[0]{FORM}};

    if (!_is_payment_allowed( $c->client_client_id )) {
        error(iget("Для социальной рекламы оплата недоступна"));
    }

    error(iget("Вам недоступна функция автопополнения счета.")) unless Direct::Wallets->can_client_use_autopay($rbac, $uid);

    my ($wallet) = Direct::Wallets->get_actual_client_wallet($c);

    if ($wallet->id == $FORM{wallet_cid} && $wallet->has_autopay) {
        $wallet->autopay->tries_num(0);
        Direct::Model::Wallet::Manager->new(items => [$wallet])->update();
    }

    return respond_json($r, {result => 'ok'});
}

=head2 cmd_ajaxUnbindCard

    Отвязать карту клиента

=cut
sub cmd_ajaxUnbindCard
    :Cmd(ajaxUnbindCard)
    :Rbac(Code => [rbac_cmd_pay], Role => [client], AllowDevelopers => 0)
    :CheckCSRF
    :Description('Отвязать карту')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c/};
    my %FORM = %{$_[0]{FORM}};

    error(iget("Вам недоступна функция автопополнения счета.")) unless Direct::Wallets->can_client_use_autopay($rbac, $uid);

    return respond_json($r, {error => iget("Данная карта не числится за пользователем")}) unless Direct::Wallets->is_user_paymethod_id($rbac, $uid, $FORM{paymethod_type}, $FORM{paymethod_id}, http_remote_ip($r));

    my ($session_id) = get_cookie($r, 'Session_id');
    return respond_json($r, {error =>  iget("Не найдены данные пользователя")}) if !$session_id;

    my $unbind = Direct::Wallets->unbind_card($uid, $FORM{paymethod_id}, http_remote_ip($r), $session_id, yandex_domain($r));

    return respond_json($r, ($unbind) ? {error => iget("Не удалось отвязать карту.")} : {result => 'ok'}) ;
}

1;


