#######################################################################
#
#  Direct.Yandex.ru
#
#  DoCmdXls
#  XLS import|export related commands to be executed from main.pl
#
#  $Id$
#
#######################################################################

=head1 NAME

DoCmdXls - execute XLS import|export related commands from main.pl

=head1 DESCRIPTION

execute XLS related commands

=cut

package DoCmdXls;

use warnings;
use strict;
use utf8;

use base qw/DoCmd::Base/;

use Encode;
use Date::Calc;

use Yandex::HashUtils;
use Yandex::ListUtils qw/xsort/;
use Yandex::ScalarUtils;
use Yandex::Compress;
use Yandex::Trace;
use Yandex::I18n;
use HashingTools;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::DBQueue;

use Settings;
use Moderate::Quick;
use StoredVars;
use Primitives;
use PrimitivesIds;
use Common qw(:globals :subs);
use CampAutoPrice::Common;
use VCards;
use Tools;
use HttpTools;
use Direct::ResponseHelper;
use Direct::PredefineVars;
use RBACDirect;
use JSON;
use JavaIntapi::CreateMobileApp;
use List::Util qw/min max sum/;
use LockObject;
use XLSCampImport;
use Campaign;
use Campaign::Types;
use APIUnits;
use Client;
use BannersCommon;
use User;
use Mediaplan;
use Phrase;
use XLSMediaplan;
use DoCmdXls::History;
use BannerImages::Queue;
use GeoTools;
use Retargeting qw//;
use Promocodes qw/get_promocode_domains/;
use Geo qw/geo_changes_to_string get_common_geo_for_camp/;

use Models::AdGroup;
use Models::Banner;
use Models::Campaign;

use List::MoreUtils qw/uniq any all part none/;
use File::Temp qw/tempfile/;
use File::Slurp qw/write_file/;

use Try::Tiny qw/try catch finally/;

# 65к - ограничение формата - 50 строк на заголовок
use constant MAX_XLS_ROWS => 2**16-50;

use constant ORDER_ID_OFFSET => 100_000_000;

sub cmd_exportCampXLS :Cmd(exportCampXLS)
    :ParallelLimit(Num => 3, Key => [UID, FORM.cid])
    :Description('экспорт кампании в xls')
    :Rbac(Code => rbac_cmd_exportCampXls)
    :RequireParam(cid => 'Cid')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   vars   c/};
    my %FORM = %{$_[0]{FORM}};

    if (!$login_rights->{superreader_control}) {
        if (my $lock = new LockObject({object_type=>"campaign", object_id=>$FORM{cid}})->load()) {
            if ($FORM{release_camp_lock}) {
                $lock->delete();
            } else {
                # непорядок это, делать вызов DoCmd::чего-то-там...
                return DoCmd::cmd_retryLater(@_);
            }
        }
    }

    if (defined $FORM{hash}) {
        my $url = make_xls_history_camp_url(type => 'export', md5_hex => $FORM{hash}, cid=>$FORM{cid});
        error(iget("can't find saved campaign")) unless $url;
        return respond_accel_redirect($r, $url);
    }

    # Считаем количество фраз (не рубрик) в неархивных баннерах.
    my $phrases_count = Campaign::count_phrases_campaign($FORM{cid}, skip_arch => $FORM{skip_arch})->{$FORM{cid}};
    if ($phrases_count > XLSCampImport::get_max_rows_limit($login_rights->{client_chief_uid})) {
        error(iget('Количество фраз в кампании превышает максимальный размер файла.'));
    }
    my $format = $FORM{xls_format} && $FORM{xls_format} eq 'xlsx' ? 'xlsx' : 'xls';
    if ($format eq 'xls' && $phrases_count >= MAX_XLS_ROWS) {
        error(iget('Количество строк в excel файле превышает максимальный размер XLS-файла. Выберите формат выгрузки XLSX.'));
    }

    my $xls_camp = get_camp_snapshot($login_rights->{client_chief_uid}, $FORM{cid}, {
        skip_arch => $FORM{skip_arch},
        host => http_server_host($r)
    });
    my $xls = camp_snapshot2excel($xls_camp,
        format => $format,
        host => http_server_host($r),
        ClientID => $c->client_client_id,
        uid => $login_rights->{client_chief_uid}, # TODO addgroup: Для автогенерации имён групп в выгрузке, на время переходного периода - чтобы клиенту руками не пришлось их проставлять.
        show_callouts => 1,
    );

    if (!$login_rights->{superreader_control} && $FORM{lock_on_export}) {
        my $half_md5_hash = half_md5_hash(md5_hex_utf8($xls));
        my $lock = new LockObject({object_type=>"campaign", object_id=>$FORM{cid}, half_md5_hash=>$half_md5_hash, duration=>7*24*60*60})->save();
        $vars->{lock_id} = $lock->{id};
    }

    save_xls_camp_to_history(cid => $FORM{cid}, type => 'export', data => $xls, filetype => $format) if !$login_rights->{superreader_control};

    return respond_data($r, $xls, ":$format", $FORM{cid} . ($format eq 'xlsx' ? ".xlsx" : ".xls"));
}

sub cmd_getImportedCampXls :Cmd(getImportedCampXls)
    :Description('список загруженных xls файлов')
    :Rbac(Code => rbac_cmd_exportCampXls)
    :RequireParam(cid => 'Cid')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS  vars/};
    my %FORM = %{$_[0]{FORM}};

    my $url = make_xls_history_camp_url(type => 'import', md5_hex => $FORM{hash}, cid=>$FORM{cid});
    error("No such file") unless $url;
    return respond_accel_redirect($r, $url);
}

sub cmd_dropImportedCampXls :Cmd(dropImportedCampXls)
    :Description('удаление загруженного xls файла')
    :Rbac(Code => rbac_cmd_importCampXls)
    :RequireParam(cid => 'Cid')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS  vars/};
    my %FORM = %{$_[0]{FORM}};

    delete_xls_history( md5_hex => $FORM{hash}, cid => $FORM{cid});

    return redirect($r, "$SCRIPT?cmd=showExportedXlsList", hash_cut \%FORM, qw/ulogin cid tab/);
}

sub cmd_getImportedCampXlsReport :Cmd(getImportedCampXlsReport)
    :Description('отчет об отложенной загрузке изображений')
    :Rbac(Code => rbac_cmd_exportCampXls)
    :RequireParam(cid => 'Cid')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS  vars    c/};
    my %FORM = %{$_[0]{FORM}};

    unless(check_xls_history_by_id(xls_id  => $FORM{xls_id}, cid => $FORM{cid})){
         error("No such file");
    }

    my $queue = Yandex::DBQueue->new(PPC(cid => $FORM{cid}), 'banner_images');
    my $job_ids = get_hashes_hash_sql(PPC(cid => $FORM{cid}), [
        "select job_id, bid from banner_images_process_queue_bid",
        where => {
            xls_id => $FORM{xls_id},
        },
    ]);

    my $jobs =  $queue->find_jobs(
        ( %$job_ids
            ? ( job_id => [keys $job_ids] )
            : ( ClientID => $c->client_client_id)
        ),
        status__not_in => [qw/New Grabbed Finished/]
    );
    $jobs = [ grep { $job_ids->{$_->job_id} || $_->args->{xls_id} == $FORM{xls_id} } @$jobs ];
    unless (@$jobs) {
         error("No such file");
    }
    my $img_data = [
        xsort { $_->{job_id}, $_->{bid} }
        map {
            {
                bid => $job_ids->{$_->job_id}->{bid},
                url => $_->args->{url},
                error => $_->result->{error},
            }
        }
        @$jobs ];

    my $xls_format = {
        sheetname=> iget("Ошибки"),
        set_column => [
            {col1 => 0, count => 1, width => 15},
            {col1 => 1, count => 1, width => 35},
            {col1 => 2, count => 1, width => 80},
        ],
    };

    my @data = ();
    push @data, (
            ( map {[undef]} 1..5),
            [{data=>iget('Отчет об ошибках при загрузке изображений'), format=>{bold=>1, size => 14}}],
            [undef, iget('№ заказа:'), {data => $FORM{cid}, as_text => 1} ],
        );

    push @data, [
            {data=> iget('Номер объявления'), global_format=>{border=>2}} ,
            iget('Ссылка'),
            iget('Статус загрузки'),
        ];
    foreach my $img (@$img_data) {
        push @data, [ $img->{bid}, $img->{url}, iget($img->{error}) ];
    }


    my $xls = Yandex::ReportsXLS->new()->array2excel2scalar(
        [\@data],
        [$xls_format],
    );

    return respond_data($r, $xls, ':xls', $FORM{cid} . '_errors.xls');
}


#
#   Upload XLS file, parse and save in DB for thurther procesing
#   returns json-formated page with following data:
#    - saved data half_md5_hash
#    - cid, if uploaded file has valid sign
#    - error_list (parse errors (except errors in geo) + (banner|phrase) validation errors)
#    - geo_errors
#

sub cmd_preImportCampXLS :Cmd(preImportCampXLS)
    :Description('загрузка рекламных материалов из xls')
    :Rbac(Code => rbac_cmd_importCampXls)
    :CheckCSRF
    :NoCaptcha
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   vars   c/};
    my %FORM = %{$_[0]{FORM}};

    my $import_format = $FORM{import_format} && $FORM{import_format} eq 'csv' ? 'csv' : 'xls';

    my $client_id = get_clientid(uid => $uid);
    my $client_currencies = get_client_currencies($client_id, allow_initial_currency => 1, uid => $uid);
    my $currency = $client_currencies->{work_currency};

    $vars->{errors} = [];
    try {
        _throw (
            iget('Файл не найден. Укажите файл с данными'),
            'no file-data uploaded'
        ) unless defined $FORM{xls};

        my $fh = $FORM{xls};
        my $options = {
            currency => $currency,
        };

        my $filename = "$fh";
        my $is_xls_extension = ($filename =~ /(xls|xlsx)$/i) ? 1 : 0;
        my $is_csv_extension = ($filename =~ /csv$/i) ? 1 : 0;

        if ($import_format eq 'csv') {
            _throw( iget('Неправильный формат файла. Допускается использование файлов только форматов .xls/.xlsx и .csv.'))
                unless ( $is_xls_extension || $is_csv_extension );

            if ($filename =~ /\.(xlsx?)$/i) {
                $options->{csv_file_format} = $1;
                binmode $fh;
            } else {
                $options->{csv_file_format} = 'csv';
                binmode $fh, ':encoding(UCS-2LE)';
            }
        } else {
            _throw( iget('Неправильный формат файла. Допускается использование файлов только формата .xls/.xlsx.') )
                unless ( $is_xls_extension );
            $import_format = ($fh =~ /\.(xlsx?)$/i)[0];
            binmode $fh;
        }
        _throw(
            iget('Размер %s-файла не должен превышать %s байт', uc($import_format), $Settings::MAX_XLS_SIZE),
            'XLS-file is too big'
        ) if ( (stat($fh))[7] > $Settings::MAX_XLS_SIZE );

        my $xlsx_filename;
        eval {
            read($fh, $vars->{xls}, (stat($fh))[7]);
            if ($import_format eq 'xlsx' || ($import_format eq 'csv' && $options->{csv_file_format} eq 'xlsx')) {
                (undef, $xlsx_filename) = tempfile(undef, UNLINK => 1, OPEN => 0);
                write_file($xlsx_filename, {binmode => ':raw'}, $vars->{xls});
            }
        };
        _throw( iget('Неверный формат файла'), $@ ) if ($@);

        do {
            my $buff = $vars->{xls};
            Encode::_utf8_off($buff);
            my $compressed_data = deflate($buff);
            _throw ( iget('Размер %s-файла слишком велик', uc($import_format) ) ) if length($compressed_data) > $Settings::MAX_COMPRESSED_XLS_SIZE;
        };

        $options->{allow_empty_geo} = 0;
        $options->{geo} = GeoTools::modify_translocal_region_before_save($FORM{geo}, {ClientID => $c->client_client_id});
        $options->{split_strategy} = $FORM{split_strategy};
        $options->{uid} = $c->client_chief_uid;
        $options->{ClientID} = $c->client_client_id;

        hash_merge $vars, XLSCampImport::preimport_xls($import_format => $xlsx_filename || $vars->{xls}, $options, $client_id);

        unlink $xlsx_filename or warn "XLSX temporary file can't remove: $!" if $xlsx_filename;

        _throw() if scalar @{$vars->{parse_errors}||[]} || scalar @{$vars->{href_errors}} || scalar @{$vars->{new_errors}||[]};

        # if no errors found, save uploaded camp

        $filename =~ m/\.(\w+)$/;
        $vars->{history_row_id} = save_xls_camp_to_history(not_save_metadata => 1, data => $vars->{xls}, filetype => $1, ClientID => $client_id);
        my $stash = hash_cut $vars, qw/history_row_id xls_camp csv_camps xls header_keys has_empty_geo geo geo_errors href_errors/;
        $stash->{import_format} = $import_format;
        my $svars = new StoredVars({
                uid  => $login_rights->{client_chief_uid},
                data => $stash,
                prefer_string => 1
            })->save() or error(iget('Не удалось сохранить данные сессии'));
        $vars->{svars_name} = $svars->{name};

    }
    catch {
        my $error = shift;
        unless ( ref $error ) {
            warn "ERROR: $error\n";
            push @{$vars->{errors}}, iget('Внутренняя ошибка при обработке файла. Попробуйте разбить его на несколько частей или повторить попытку загрузки позже.');
        }
        else {
            warn "ERROR: $error->{log}\n" if defined $error->{log};
            push @{$vars->{errors}}, $error->{message} if defined $error->{message};
        }
    }
    finally {
        # get the cid of root campaign even if there are errors in XLS
        if ($vars->{xls_camp}->{cid}) {
            my $ex_cid = $vars->{xls_camp}->{cid};
            my $ex_campaign = get_camp_info($ex_cid, $login_rights->{client_chief_uid}, short => 1) // {};

            if ($ex_campaign->{cid} && $ex_campaign->{statusEmpty} eq 'No') {
                if ($ex_campaign->{mediaType} ne $vars->{xls_camp}->{mediaType}) {
                    push @{$vars->{camp_comments}}, iget("Нельзя импортировать файл в кампанию № %s: Невозможно изменить тип кампании.", $ex_cid);
                } elsif ($ex_campaign->{archived} eq 'Yes') {
                    push @{$vars->{camp_comments}}, iget("Нельзя импортировать файл в архивную кампанию № %s", $ex_cid);
                } elsif (!camp_kind_in(type => $ex_campaign->{mediaType}, 'xls') || $ex_campaign->{source} eq 'zen') {
                    push @{$vars->{camp_comments}}, iget("Нельзя импортировать файл в кампанию № %s", $ex_cid);
                } elsif ($ex_campaign->{source} eq 'uac' || $ex_campaign->{source} eq 'widget') {
                    push @{$vars->{camp_comments}}, iget("Нельзя импортировать файл в кампанию № %s, созданную из Мастера кампаний", $ex_cid);
                } else {
                    $vars->{cid} = $ex_cid;
                    $vars->{camp_name} = $ex_campaign->{name};
                }
            } else {
                push @{$vars->{camp_comments}}, iget("Кампания № %s не существует", $ex_cid);
            }
        }
    };

    unless ($FORM{json}) {
        # по фирме определяем какую оферту показать
        $vars->{client_firm_id} = Client::get_client_firm_id($client_id);

        return respond_template($r, $template, 'pre_import_camp_xls.html', $vars);
    }

    my $parse_warnings_for_exists_camp = $vars->{parse_warnings_for_exists_camp};
    $vars->{snapshot_errors} = [];
    if ($vars->{cid}) {
        my ($snapshot_warnings, $snapshot_errors) = get_snapshot_warnings_and_errors_for_exists_camp($login_rights->{client_chief_uid}, $vars->{xls_camp}->{cid}, $vars->{xls_camp});
        hash_merge $parse_warnings_for_exists_camp, $snapshot_warnings;
        push @{$vars->{snapshot_errors}}, @{$snapshot_errors};
    } else {
        hash_merge $parse_warnings_for_exists_camp, compare_currencies($vars->{xls_camp}->{currency}, $currency);
    }
    my ($only_mobiles) = XLSCampImport::are_uniform_banners($vars->{xls_camp});
    my @parse_warnings = @{$vars->{parse_warnings} || []};

    if ($only_mobiles) {
        push @parse_warnings, iget('Обратите внимание: в файле есть группы только с мобильными объявлениями, в этом случае на десктопе будет показано одно из мобильных объявлений группы');
    }

    my $json_vars = hash_merge {
            geo_errors  => [],
            errors  => [ uniq map { @{$vars->{$_}||[]} } qw/errors parse_errors new_errors href_errors snapshot_errors/ ],
            warnings => [ uniq(@parse_warnings, @{$vars->{camp_comments} // []}) ],
            parse_warnings_for_exists_camp => $parse_warnings_for_exists_camp,
            from_cid => $vars->{xls_camp}->{cid},
            mediaType => $vars->{xls_camp}->{campaign_type},
        }, hash_cut $vars, qw/svars_name cid camp_name has_empty_geo geo_errors camp_comments elder_loads has_oversized_banners/;

    if ($import_format eq 'csv') {
        # если в загруженном csv несколько кампаний, то выдаем список из имен кампаний для выбора и ворнинги если есть ошибки
        $json_vars->{available_csv_campaigns} = [map {{camp_name => $_->{camp_name}
                                                     , camp_number => $_->{camp_number}
                                                     , parse_warnings => $_->{parse_warnings}
                                                     , parse_errors => $_->{parse_errors}
                                                     }} @{$vars->{csv_camps}}
                                                ];
    }
    my $json_data = to_json($json_vars);
    my $callback = $FORM{callback};

    # При загрузке через iframe сериализуемся через respond_jsonp
    return $callback ? respond_jsonp($r, $json_data, $callback) : respond_json($r, $json_data);
}

sub _throw {
    my ($message, $log_message) =  @_;

    return die { message => $message, log => $log_message };
}

sub _get_or_create_client_mobile_app_id {
    my ($client_id, $adgroups_store_content_href, $tracker_url) =  @_;
    my $client_mobile_app_id = MobileApps::get_mobile_app_id_by_store_href($client_id, $adgroups_store_content_href);
    unless ($client_mobile_app_id) {
        my $entity = {storeHref => $adgroups_store_content_href, trackerUrl => $tracker_url};
        my $response = JavaIntapi::CreateMobileApp->new( client_id => $client_id, entities => [$entity] )->call;
        my $client_mobile_app = $response->[0];
        if (!$client_mobile_app->{success}) {
            die to_json($client_mobile_app) . "\n";
        }
        $client_mobile_app_id = $client_mobile_app->{mobileAppId};
    }
    return $client_mobile_app_id;
}

sub _get_banner_tracker_url {
    my ($adgroups) =  @_;
    for my $adgroup (@$adgroups) {
        if ($adgroup->{banners} ) {
            for my $banner (@{$adgroup->{banners}}) {
                return $banner->{mobile_href} if $banner->{mobile_href};
            }
        }
    }
    return
}

#
#   Upload xls-file and/or read parsed file from stored session
#   gets id of stored session in URL params
#
#   depending on selected option validates the uploaded campaign_snapshot for
#   1) creating new camp
#   2) merging with existent campaing
#
#   redirects to confirmation page
#   with changes listing or page with merge error list
#
sub cmd_importCampXLS :Cmd(importCampXLS)
    :Description('загрузка рекламных материалов из xls')
    :CheckCSRF
    :Rbac(Code => rbac_cmd_importCampXls,  ExceptRole => [limited_support])
#    :Lock(NoRetry => 1)    # internal campaign_lock processing
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   vars   c/};
    my %FORM = %{$_[0]{FORM}};

    if ($FORM{svars_name}) { #pre_import scheme is used

        my $svars = new StoredVars({name => $FORM{svars_name}, uid => $login_rights->{client_chief_uid}, prefer_string => 1})->load()
            or error(iget("Сохранённого объекта с указанным номером не существует"));

        hash_merge $vars, hash_cut $svars->data, qw/sign xls_camp csv_camps history_row_id xls header_keys has_empty_geo geo geo_errors href_errors import_format/;
    } else {
        error("old scheme parsing the uploaded XLS-file is no longer supported");
    }

    error(iget('Недопустимое значение внутреннего параметра')) if ($FORM{destination_camp}||='') !~ m/^(new|old|other)$/;
    error(iget('Не указан номер кампании')) if $FORM{destination_camp} !~ m/new/ and !$FORM{cid};
    error(iget('Пользовательское соглашение не принято')) if (defined $FORM{oferta} and $FORM{oferta} eq 'decline');

    my $camp_snapshot;

    if ($vars->{import_format} && $vars->{import_format} eq 'csv') {
        if (scalar(@{ $vars->{csv_camps} }) > 1 && ! exists $FORM{choosed_csv_camp}) {
            error(iget('Не выбрана кампания для загрузки'));
        } elsif (scalar(@{ $vars->{csv_camps} }) > 1) {
            ($camp_snapshot) = grep {$_->{camp_number} == $FORM{choosed_csv_camp}} @{$vars->{csv_camps}};
            error(iget('Не выбрана кампания для загрузки')) unless $camp_snapshot;
        } else {
            $camp_snapshot = $vars->{csv_camps}->[0]; # иначе выбираем 1-ую и единственную кампанию
        }
    } else {
        # загрузка из xls
        $camp_snapshot = $vars->{xls_camp};
        $vars->{is_group_format} = $camp_snapshot->{is_group_format};
    }

    # check if the campaign locked
    if ( $FORM{destination_camp} eq 'old' and defined $FORM{cid}
            and my $lock = new LockObject({object_type=>"campaign", object_id=>$FORM{cid}})->load()
            ) {

        #   if the camp is locked with named lock, and the name doesn't math the uploaded xls-file sign
        #   or an unnamed lock is present and no "release_camp_lock" option forced
        if ((!$lock->{half_md5_hash} or $lock->{half_md5_hash} ne ($vars->{sign}||0)) and !$FORM{release_camp_lock}) {
            $vars->{NoRetry} =1 if !$FORM{svars_name}; # if file is loaded, don't let retry
            return DoCmd::cmd_retryLater(@_);
        } else {
            new LockObject({object_type=>"campaign", object_id=>$FORM{cid}})->delete();
        }
    }
    if ($FORM{set_common_geo}) {
        my $new_common_geo;
        #$FORM{geo} - старый формат, гео приходит строкой, $FORM{json_geo_changes} - список регионов приходит структурой
        my $geo = $FORM{geo};
        if (!defined $geo && $FORM{json_geo_changes}){
            my $regions = $FORM{json_geo_changes};
            $regions = from_json($regions) unless ref $regions;

            $new_common_geo = geo_changes_to_string($regions, undef, $c->client_client_id);
            error(iget('Не заданы регионы показа')) unless $new_common_geo;
        }
        #Если задано пустое гео - преобразование превратит его в 0, не делаем
        elsif(defined $geo) {
            $geo = GeoTools::modify_translocal_region_before_save($geo, {ClientID => $c->client_client_id});
            $new_common_geo = GeoTools::refine_geoid($geo, undef, {ClientID => $c->client_client_id});
        }

        my $error = XLSCampImport::validate_xls_geo(
            new_common_geo => $new_common_geo,
            has_empty_geo  => $vars->{has_empty_geo},
            geo_errors     => $vars->{geo_errors},
            geo            => $vars->{geo}
        );
        error($error) if $error;

        XLSCampImport::set_geo_to_groups($camp_snapshot, $vars->{geo}, $new_common_geo // '0');
    }

    # принудительно выставляем баннерам с украинским, казахским, турецким текстом соответствующий регион
    # если выбран не такой (для случая с csv, когда регион выбирается)
    if ($vars->{import_format} && $vars->{import_format} eq 'csv') {
        my %banners_by_lang = ();
        foreach my $group (@{$camp_snapshot->{groups}}) {
            # Тут получается ерунда: мы для каждого баннера узнаем регион, чтобы узнать его мы достаем из БД опять все объявления и проверяем.
            # check_geo_restrictions может принимать ссылку на массив баннеров, но надо придумать как быть с подсчетом количество групп
            # с измененным регионом, его надо считать.
            for my $banner (@{ $group->{banners} }) {
                unless (Models::Banner::check_geo_restrictions($banner, {}, geo => $group->{geo}, ClientID => $c->client_client_id)) {
                    if (my $geo_by_text = Models::Banner::get_geo_by_banner_text($group->{banners}->[0])) {
                        XLSCampImport::set_geo_to_groups({groups => [$group]}, $geo_by_text->{geo});
                        $banners_by_lang{$geo_by_text->{lang}} ++;
                    }
                }
            }
        }
        if (keys %banners_by_lang) {
            my %forced_geo_message = (uk => iget_noop('Количество объявлений, которым будет установлен регион Украина: %s'),
                                      kk => iget_noop('Количество объявлений, которым будет установлен регион Казахстан: %s'),
                                      tr => iget_noop('Количество объявлений, которым будет установлен регион Турция: %s'));
            $vars->{forced_geo_messages} = [];
            foreach my $lang (sort { $a cmp $b} keys %banners_by_lang) {
                push @{$vars->{forced_geo_messages}}, iget($forced_geo_message{$lang}, $banners_by_lang{$lang});
            }
        }
    }
    error(join "\n", @{$vars->{href_errors}}) if scalar @{$vars->{href_errors}};

    my $client_id = get_clientid(uid => $uid);
    my $is_tycoon_organizations_feature_enabled = Client::ClientFeatures::has_tycoon_organizations_feature($c->client_client_id);

    my $currency;

    my $is_mobile_content_camp = $camp_snapshot->{mediaType} eq 'mobile_content' ? 1 : 0;
    my ($adgroups_store_hrefs, $adgroups_store_href, $tracker_url);
    if ($is_mobile_content_camp) {
        my $group_filter = $FORM{destination_camp} eq 'old' ? 'without_pid' : 'all';
        $adgroups_store_hrefs = get_adgroups_store_content_hrefs( $camp_snapshot->{groups}, $group_filter );
        $adgroups_store_href = $adgroups_store_hrefs->[0];
        $tracker_url = _get_banner_tracker_url($camp_snapshot->{groups});
    }

    if ($FORM{destination_camp} eq 'new') {
        my $client_currencies = get_client_currencies($client_id, allow_initial_currency => 1, uid => $c->client_chief_uid);
        $currency = $client_currencies->{work_currency};

        if ($is_mobile_content_camp) {
            my $rmp_enabled_error = validate_rmp_store_hrefs($adgroups_store_hrefs);
            if ($rmp_enabled_error) {
                push @{$vars->{errors}}, $rmp_enabled_error;
            } else {
                $camp_snapshot->{mobile_app_id} = _get_or_create_client_mobile_app_id($c->client_client_id, $adgroups_store_href, $tracker_url);
            }
        }
    } else {
        my $existing_camp_info = get_camp_info($FORM{cid}, $c->client_chief_uid, with_strategy => 1, without_multipliers => 1) || {};
        $currency = $existing_camp_info->{currency};

        if ($existing_camp_info->{cid} && $existing_camp_info->{statusEmpty} eq 'No') {
            error(iget("Невозможно изменить тип кампании № %s", $FORM{cid}))
                if $existing_camp_info->{mediaType} ne $camp_snapshot->{mediaType};

            error(iget("Нельзя импортировать файл в кампанию № %s", $FORM{cid}))
                if (($existing_camp_info->{source} eq 'zen') || ($existing_camp_info->{source} eq 'uac') || ($existing_camp_info->{source} eq 'widget'));

            if ($is_mobile_content_camp) {
                my $rmp_enabled_error;
                if ($existing_camp_info->{mobile_app_id}) {
                    my $store_href = MobileApps::get_store_href( $c->client_client_id, $existing_camp_info->{mobile_app_id});
                    $rmp_enabled_error = validate_rmp_store_hrefs($adgroups_store_hrefs, $store_href);
                } else {
                    $camp_snapshot->{mobile_app_id} = _get_or_create_client_mobile_app_id($c->client_client_id, $adgroups_store_href, $tracker_url);
                }
                push @{$vars->{errors}}, $rmp_enabled_error if $rmp_enabled_error;
            }

            $vars->{campaign} = $existing_camp_info;
        } else {
            error(iget("Кампания № %s не существует", $FORM{cid}));
        }
    }

    #merge changes
    if ( $FORM{destination_camp} ne 'old' ) { # new + other
        # if new camp - do dumn merging .. i.e. copy the imported camp
        $vars->{new_camp} = $camp_snapshot;
        my $status_moderate = $FORM{send_to_moderation} ? 'Ready' : 'New';
        foreach (@{$camp_snapshot->{groups}}) {
            $_->{statusModerate} = $status_moderate;
            $_->{statusModerate} = $status_moderate for @{$_->{banners}};
        }
        XLSCampImport::_clear_ids_in_groups($vars->{new_camp}->{groups}, dont_clear_group_name=>1);
        fill_empty_price_for_snapshot($vars->{new_camp}, currency => $currency);

        my $create_phrase = sum(0, map { get_banner_phrases_count($_) } @{$camp_snapshot->{groups}});
        my $create_relevance_matches = sum(0, map { get_banner_relevance_matches_count($_) } @{$camp_snapshot->{groups}});
        my $create_retargetings = sum(0, map { get_banner_retargetings_count($_) } @{$camp_snapshot->{groups}});
        # calc objects count
        $vars->{changes} = {
            create_group    => scalar(@{$camp_snapshot->{groups}}),
            create_banner   => sum(0, map {scalar @{$_->{banners}}} @{$camp_snapshot->{groups}}),
            create_phrase   => $create_phrase + $create_relevance_matches + $create_retargetings,
        };

        $vars->{new_urls} = Models::AdGroup::get_urls($camp_snapshot->{groups});

        my $strategy;
        # При импорте в другую существующую кампанию - валидируем общие минус-слова
        if ($FORM{destination_camp} eq 'other' && $FORM{cid}) {
            my $x = XLSCampImport::validate_group_camp_minus_words($vars->{new_camp}, cid => $FORM{cid}, use_line_numbers => 1);
            push @{$vars->{merge_warnings}}, @{$x->{warnings}};
            $strategy = campaign_strategy($FORM{cid});
            if (!$FORM{send_to_moderation}) {
                foreach (@{$vars->{new_camp}->{groups}}) {
                    $_->{save_as_draft} = 1;
                    $_->{save_as_draft} = 1 for @{$_->{banners}};
                }
            }
        } else { # $FORM{destination_camp} eq 'new'
            $strategy = {
              is_autobudget => '',
              is_net_stop => '',
              is_search_stop => '',
              name => '',
              net => {name => 'default'},
              search => {name => 'default'}
            };
            $vars->{new_camp}->{statusModerate} = $FORM{send_to_moderation} ? 'Ready' : 'New';
        }

        for (@{$vars->{new_camp}->{groups}}) {
            $_->{strategy} = $strategy;
            $_->{currency} = $currency;
        }
    } else {
        my $camp_strategy;
        if ($FORM{cid}) {
            $camp_strategy = $vars->{campaign}->{strategy};
            error(iget("Кампания № %s не существует", $FORM{cid})) if !$vars->{campaign}{cid};
            error(iget("Нельзя импортировать файл в архивную кампанию № %s", $FORM{cid})) if $vars->{campaign}{archived} eq 'Yes';
            error(iget("Нельзя импортировать файл в медийную кампанию № %s", $FORM{cid})) unless camp_kind_in(cid => $FORM{cid}, 'xls');;
        }

        # real merging
        $vars->{curr_camp} = get_camp_snapshot($login_rights->{client_chief_uid}, $FORM{cid}, {
            host => http_server_host($r),
            pass_empty_groups => 1,
            skip_arch => 1,
            ClientID => $c->client_client_id,
        });
        my $merge_options = {
            strategy => $vars->{curr_camp}->{strategy},
            search_strategy => $vars->{curr_camp}->{search_strategy},
            context_strategy => $vars->{curr_camp}->{context_strategy},
            dont_change_prices => $FORM{dont_change_prices},
            apply_changes_minus_words => $FORM{changes_minus_words} && $FORM{changes_minus_words} eq 'change' ? 1 : 0,
            host => http_server_host($r),
            currency => $currency,
            send_new_banners_to_moderation => $FORM{send_to_moderation} ? 1 : 0,
            is_internal_user => $login_rights->{is_internal_user}, # временно, при подсчете баллов, для клиентов, не учитывать уточнения
            (map {+("remove_${_}" => ($FORM{$_} || '') eq 'change')} qw/lost_banners lost_phrases lost_groups lost_retargetings/),
            ClientID => $c->client_client_id,
        };
        $merge_options->{remove_lost_groups} = $merge_options->{remove_lost_banners} unless exists $FORM{lost_groups};
        $merge_options->{remove_lost_retargetings} = $merge_options->{remove_lost_retargetings} ? 1 : $merge_options->{remove_lost_phrases};
        $merge_options->{ignore_image} = !$vars->{header_keys}->{image_url};
        $merge_options->{ignore_display_href} = !$vars->{header_keys}->{display_href};

        ($vars->{new_camp}, $vars->{merge_errors}, $vars->{merge_warnings}, $vars->{changes}, $vars->{stop_banners})
            = merge_camp_snapshot($vars->{curr_camp}, $camp_snapshot, $merge_options);

        for my $group  (@{$vars->{new_camp}->{groups}}) {
            $group->{strategy} = $camp_strategy;
            $group->{currency} = $currency;
            next if none { $group->{$_} } qw/_added _changed _has_changes_in_phrases
                                        _has_changes_in_banners
                                        _has_changes_in_retargetings/;
        }

        $vars->{common_contact_info} = get_common_contactinfo_for_camp($vars->{curr_camp}{cid}, {dont_skip_arch => 1});
        $vars->{will_common_contact_info} = check_common_contact_info_for_snapshot($login_rights->{client_chief_uid}, $vars->{curr_camp}{cid}, $vars->{new_camp});

        # Для сохранения ЕКИ скопируем старую визитку
        if ($vars->{common_contact_info} && $vars->{will_common_contact_info}) {
            my @ignore_fields = qw/metro org_details_id geo_id address_id cid uid/;
            my $old_hash = VCards::vcard_hash($vars->{common_contact_info}, ignore_fields => \@ignore_fields);
            my $new_hash = VCards::vcard_hash($vars->{new_camp}->{contact_info}, ignore_fields => \@ignore_fields);

            $vars->{new_camp}->{contact_info} = $vars->{common_contact_info}  if $old_hash eq $new_hash;
        }

        my @common_geo = get_common_geo_for_camp($vars->{curr_camp}{cid});
        $vars->{has_common_geo} = scalar(@common_geo) ? 1 : 0;
        $vars->{will_common_geo} = check_common_geo_for_snapshot($vars->{new_camp});
    }
    $vars->{new_camp}->{cid} = $FORM{cid};

    $vars->{errors} ||= [];

    my @bids;
    for (@{$vars->{new_camp}->{groups}}) {
        push @bids, map { $_->{bid} ? $_->{bid} : () } @{$_->{banners}};
    }
    my $exists_banners_type = BannersCommon::get_banners_type(text => \@bids);
    my %exists_retargetings;
    for my $group ($vars->{curr_camp} ? @{$vars->{curr_camp}->{groups}} : ()) {
        $exists_retargetings{$group->{pid}} = { map { $_->{ret_id} => 1 } @{$group->{retargetings}} };
    }
    my $exists_ret_conds = Retargeting::get_retargeting_conditions(ClientID => $client_id);

    my $bid2type = {};
    if (@bids) {
        $bid2type = get_hash_sql(PPC(bid => \@bids), [
                "select bid, if(banner_type = 'image_ad', 'image_ad', 'text') as ad_type from banners",
                where => { bid => SHARD_IDS }
        ]);
    }
    my @has_changed_type;

    for my $group (@{$vars->{new_camp}->{groups}}) {

        # проверяем только группы с измененниями
        next if none { $group->{$_} } qw/_added _changed _has_changes_in_phrases
                                        _has_changes_in_banners
                                        _has_changes_in_retargetings
                                        _has_changes_in_callouts
                                        /;

        for my $banner (@{$group->{banners}}) {
            $banner->{has_href} = !!$banner->{href};
            if ($banner->{contact_info} && $banner->{contact_info} eq '+') {
                $banner->{vcard} = $vars->{new_camp}->{contact_info};
                $banner->{has_vcard} = 1;
            }

            if ($banner->{bid} && $bid2type->{$banner->{bid}} && $banner->{ad_type} ne $bid2type->{$banner->{bid}}) {
                push @has_changed_type, $banner;
            }
        }

        my $group_errors = {};
        if (($group->{adgroup_type} || 'base') eq 'base') {
            $group_errors = validate_group($group, {
                ClientID                                => $c->client_client_id,
                exists_banners_type                     => $exists_banners_type,
                exists_ret_conds                        => $exists_ret_conds,
                exists_group_retargetings               => $group->{pid} ? $exists_retargetings{$group->{pid}} : {},
                for_xls                                 => 1,
                is_tycoon_organizations_feature_enabled => $is_tycoon_organizations_feature_enabled,
            });
        }


        next unless keys %$group_errors;

        push @{$vars->{errors}}, Direct::XLS::Validation->decorate_errors($group => $group_errors->{group_name}) if $group_errors->{group_name};
        for (qw/group_name phrases retargetings common geo/) {
           my $errors = $group_errors->{$_} // next;
           push @{$vars->{errors}}, map {Direct::XLS::Validation->decorate_error($group => $_)} ref $errors eq 'ARRAY' ? @$errors : ($errors);
        }

        for my $banner ($group_errors->{banners} ? @{$group->{banners}} : ()) {
            if ($banner->{errors}) {
                push @{$vars->{errors}}, map {Direct::XLS::Validation->decorate_error($banner => $_)} map {@$_} values %{$banner->{errors}};
            }
        }
    }

    if (@has_changed_type){
        push @{$vars->{errors}}, map {Direct::XLS::Validation->decorate_error($_ => iget("Нельзя изменять тип баннера"))} @has_changed_type;
    }

    my $split_strategy = $FORM{split_strategy};
    $split_strategy = 'always_new_banner' unless $split_strategy && any { $split_strategy eq $_ } qw/always_new_banner shared_new_banner/;
    my ($added_groups, $added_banners) = split_oversized_groups($split_strategy => $vars->{new_camp}->{groups}, client_id => $c->client_client_id);
    $vars->{changes}->{create_group} = ($vars->{changes}->{create_group} || 0) + $added_groups;
    $vars->{changes}->{create_banner} = ($vars->{changes}->{create_banner} || 0) + $added_banners;

    # отдельно проверяем превышение длины фраз при изменении существующих фраз
    # оторвать?
    my $affected_group;
    if (has_oversized_banners($vars->{new_camp}->{groups}, client_id => $c->client_client_id, get_affected_group_as => \$affected_group)) {
        error(Direct::XLS::Validation->decorate_error($affected_group => iget('Количество фраз в одном из объявлений превышает допустимый максимум')));
    }

    $vars->{new_camp}->{uid} = $login_rights->{client_chief_uid};

    push @{$vars->{errors}}, Models::Campaign::validate_campaign($vars->{new_camp});

    if ( $FORM{destination_camp} eq 'new' ) {
        my $camp_limit_error = check_add_client_campaigns_limits(uid => $uid);
        push @{$vars->{errors}}, $camp_limit_error if $camp_limit_error;
    } else {
        # Если добавление не в новую кампанию, то на всякий случай снова валидируем, вдруг количество меток превышено.
        my $error = validate_tags_snapshot($FORM{cid}, $vars->{new_camp}->{groups});
        push @{$vars->{errors}}, $error if $error;
    }

    # Domain availability check
    push @{$vars->{errors}}, @{check_domain_availability($camp_snapshot->{groups}, $vars->{curr_camp}{groups})};

    # check if the user has enough units;
    my $uhost = new APIUnits({scheme=>'XLS'});
    foreach my $type (grep {!/group/} keys %{$vars->{changes}}) {
        $uhost->reserve_units_by_type($login_rights->{client_chief_uid}, $type, $vars->{changes}->{$type});
    }
    $vars->{reserved_units} = $uhost->get_reserved_units($login_rights->{client_chief_uid});

    unless ($uhost->have_enough_user_units($login_rights->{client_chief_uid}, allow_monitor_fault => 1) ) {
        # sub have_enough_user_units() changes the $user_units
        $vars->{user_units} = $uhost->check_or_init_user_units($login_rights->{client_chief_uid})->{$login_rights->{client_chief_uid}}{units_rest};

        push @{$vars->{errors}}, iget("Для выполнения операции требуется %s баллов при текущем балансе %s, попробуйте уменьшить количество изменяемых объектов или повторите операцию позднее.", $vars->{reserved_units}, $vars->{user_units});
    }
    # sub have_enough_units() changes the $user_units
    $vars->{user_units} = $uhost->check_or_init_user_units($login_rights->{client_chief_uid})->{$login_rights->{client_chief_uid}}{units_rest};

    # проверяем параметры для установки цен на всю новую кампанию
    my $camp_auto_price_params;
    if ($vars->{import_format} && $vars->{import_format} eq 'csv') {
        $camp_auto_price_params = {
            max_price            => $FORM{max_price}
            , price_base         => CampAutoPrice::Common::convert_base_place_from_templates($FORM{price_base})
            , position_ctr_correction => $FORM{position_ctr_correction}
            , use_position_ctr_correction => 1
            , proc_base          => $FORM{proc_base}
            , proc               => $FORM{proc}
            , platform           => 'search'
            , update_phrases     => 1
            , change_all_banners => 1 # менять цены на баннерах-черновиках тоже
            , currency           => $currency
            , for                => 'front'
            , mobile_content     => $is_mobile_content_camp
        };
        my $camp_auto_price_params_errors = Common::validate_camp_auto_price_params($camp_auto_price_params);
        push @{$vars->{errors}}, uniq values(%$camp_auto_price_params_errors) if $camp_auto_price_params_errors;
    }

    # if no errors encountered...
    if ( !scalar(@{$vars->{parse_errors}||[]}) and !scalar(@{$vars->{new_errors}||[]}) and !scalar(@{$vars->{merge_errors}||[]}) and !scalar(@{$vars->{errors}||[]}) ) {
        # make a sessin and locke the camp if needed

        $vars->{statusModerate} = ($FORM{destination_camp} ne 'new') ? get_camp_info($FORM{cid}, undef, short => 1)->{statusModerate} : 'New';

        my $lock;
        if ( $FORM{destination_camp} eq 'old' ) {
            $lock = new LockObject({object_type=>'campaign', object_id=>$FORM{cid}})->save();
        }

        my $svars = new StoredVars( {
                data => (hash_merge {
                    history_row_id => $vars->{history_row_id},
                    new_camp    => $vars->{new_camp},
                    destination_camp   => $FORM{destination_camp},
                    reserved_units  => $uhost->get_reserved_units($login_rights->{client_chief_uid}),
                    changes         => $vars->{changes},
                    # Будем ли проверять баннеры мгновенной модерацией в процессе загрузки xls
                    perform_quick_moderation  => $FORM{send_to_moderation} && $vars->{statusModerate} eq 'New',
                    import_format => $vars->{import_format},
                    camp_auto_price_params => $camp_auto_price_params,
                    stop_banners => $vars->{stop_banners},
                    header_keys => $vars->{header_keys}
                },
                $lock ? {lock_id => $lock->{id}} : {},
                ),
                uid => $login_rights->{client_chief_uid},
                cid => $FORM{cid},
                prefer_string => 1
            })->save()
            or error(iget("Не удалось сохранить данные сессии"));

        $vars->{svars_name} = $svars->{name};
        # ... continue with confirmation page
    } else {
        # the errors are going to be printed on the "non-confirmation" page with single option "go back"
    }

    # $vars->{root_camp} = $vars->{root_camp}->{data} if $vars->{root_camp};
    $vars->{cid} = $FORM{cid};

    # по фирме определяем какую оферту показать
    $vars->{client_firm_id} = Client::get_client_firm_id($client_id);

        if ($FORM{cid}) {
            $vars->{has_promocode_domains} = scalar @{get_promocode_domains($FORM{cid})} > 0;
        } else {
            $vars->{has_promocode_domains} = any {
                @{get_promocode_domains($_->{wallet_cid})}
            } @{get_all_wallet_camps(client_client_id => $c->client_client_id)};
        }

    return respond_bem($r, $c->reqid, $vars, source => 'data3');
}

sub cmd_confirmSaveCampXLS :Cmd(confirmSaveCampXLS)
    :Description('загрузка рекламных материалов из xls')
    :CheckCSRF
    :Rbac(Code => rbac_cmd_importCampXls)
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   vars   c/};
    my %FORM = %{$_[0]{FORM}};

    my $client_chief_uid = $login_rights->{client_chief_uid};
    my $client_id = get_clientid(uid => $client_chief_uid);
    my $svars = new StoredVars({name => $FORM{svars_name}, cid => $FORM{cid} || 0, uid => $client_chief_uid, prefer_string => 1})->load()
        or error(iget("Сессия импорта не существует"));
    if (  $svars->lock() <= 0 ) {
        error(iget("Сессия импорта файла заблокирована на другой странице подтверждения"));
    }

    my $cid = $FORM{cid};
    my $strategy_id;
    my $mediaType = $FORM{mediaType} || $svars->data->{new_camp}->{mediaType} || 'text';

    if ( $FORM{confirm}) {
        if (!$cid || $svars->data->{destination_camp} eq 'new') {
            $cid = get_new_id('cid', uid => $uid);
            $strategy_id = ($mediaType ne 'wallet' && $mediaType ne 'billing_aggregate')
                ? Client::ClientFeatures::has_get_strategy_id_from_shard_inc_strategy_id_enabled($client_id)
                    ? get_new_id('strategy_id', ClientID => $client_id)
                    : ($cid + ORDER_ID_OFFSET)
                : 0;
        }
        my $xls_id = history_preimport_to_import(cid => $cid, id => $svars->data->{history_row_id});

        my $agency_uid;
        if (defined $FORM{for_agency}) {
            $agency_uid = get_uid_by_login($FORM{for_agency});
            error("агентство не найдено") unless $agency_uid;
        } elsif ($login_rights->{agency_control}) {
            $agency_uid = $UID;
        }

        if ($login_rights->{manager_control} && ! defined $agency_uid) {
            $vars->{ManagerUID} = $UID;
        } elsif ($login_rights->{agency_control} || defined $agency_uid) {
            $vars->{AgencyUID} = $agency_uid;
        }

        my $uhost = new APIUnits({scheme=>'XLS'});
        my $changes = $svars->data->{changes};
        foreach my $type (grep {!/group/} keys %$changes) {
            $uhost->reserve_units_by_type($login_rights->{client_chief_uid}, $type, $changes->{$type});
        }

        my $reserved_units = $uhost->get_reserved_units($login_rights->{client_chief_uid});

        # die if the units calculation result differs in cases before and after confirmation
        die sprintf(
            "XLS operations costs changed during the confirmation process! %s -> %s",
            $svars->data->{reserved_units},
            $reserved_units
        ) unless $svars->data->{reserved_units} == $reserved_units;

        unless ($uhost->have_enough_user_units($login_rights->{client_chief_uid}, allow_monitor_fault => 1)) {
            my $user_units = $uhost->check_or_init_user_units($login_rights->{client_chief_uid})->{$login_rights->{client_chief_uid}}{units_rest};
            error(iget("Для выполнения операции требуется %s баллов при текущем балансе %s, попробуйте уменьшить количество изменяемых объектов или повторите операцию позднее.", $reserved_units, $user_units));
        }

        # списываем баллы до импорта кампании во избежании гонок при импорте больших кампаний
        # иначе можно параллельно запустить импорт нескольких файлов и использовать больше баллов, чем есть
        # логгирование для новых кампаний производим после окончания импорта, чтобы записать cid
        $uhost->update_units();
        if ($svars->data->{destination_camp} ne 'new') {
            $uhost->log_reserved_units($login_rights->{client_chief_uid},
                hash_copy({cid => $FORM{cid}}, $vars, qw/ManagerUID AgencyUID/)
            );
        }

        # пытаемся вернуть баллы при ошибке загрузки
        my $cancel = sub {
            my $uhost2 = APIUnits->new({scheme=>'XLS'});
            $uhost2->reserve_units($login_rights->{client_chief_uid}, -$reserved_units);
            $uhost2->update_units();
            error(@_);
        };

        $vars->{statusModerate} = ($svars->data->{destination_camp} ne 'new') ? get_camp_info($cid, undef, short => 1)->{statusModerate} : 'New';

        my $user = get_user_info($client_chief_uid) || {};
        $user->{fio}   ||= $r->pnotes('user_fio');
        $user->{email} ||= $r->pnotes('user_email');

        my $ignore_image = $svars->data->{header_keys}->{image_url} ? 0 : 1;
        my $campaign;
        if ( $svars->data->{destination_camp} eq 'new') {

            if ($agency_uid) {
                my $agency_id = get_clientid(uid => $agency_uid);
                my $agency_relations = get_agency_client_relations($agency_id, $c->client_client_id);
                if ($agency_relations->{$c->client_client_id}->{agency_unbind}) {
                    $cancel->(iget("Невозможно создать новую кампанию для клиента, обслуживание которого завершено."));
                }
            }

            my $errors;
            ($campaign, $cid, $errors) = XLSCampImport::load_xls_to_new_camp(
                user => $user,
                UID => $UID,
                uid => $uid,
                login_rights => $login_rights,
                agency_uid => $agency_uid,
                xls_id => $xls_id,
                cid => $cid,
                strategy_id => $strategy_id,

                new_camp => $svars->data->{new_camp},
                auto_price_params => $svars->data->{camp_auto_price_params},
                camp_mediaType => $mediaType,

                context => $c,
                rbac => $rbac,

                ignore_image => $ignore_image,
            );
            $cancel->(map {iget($_)} @$errors) if $errors;

            $FORM{cid} = $cid;

        } elsif ($svars->data->{destination_camp} eq 'other') {

            $cancel->(iget("Не указан номер кампании или кампания не существует")) unless $FORM{cid} && camp_kind_in(cid => $FORM{cid}, 'xls');
            my $errors;
            ($campaign, $errors) = XLSCampImport::load_xls_to_other_camp(
                to_campaign => $cid,
                UID => $UID,
                user => $user,
                xls_id => $xls_id,
                login_rights => $login_rights,
                new_camp => $svars->data->{new_camp},
                ignore_image => $ignore_image,
            );

            $cancel->( map {iget($_)} @$errors ) if $errors;

        } elsif ($svars->data->{destination_camp} eq 'old'){

            $cancel->(iget("Не указан номер кампании или кампания не существует")) unless $FORM{cid} && camp_kind_in(cid => $FORM{cid}, 'xls');

            if (!new LockObject({object_type=>"campaign", object_id=>$FORM{cid}, id=>$svars->data->{lock_id}})->load()) {
                $svars->delete();
                $cancel->(iget("Кампания разблокирована до принятия изменений. Попробуйте повторить операцию импорта снова."));
            }

            my $errors;
            ($campaign, $errors) = XLSCampImport::load_xls_to_old_camp(
                to_campaign => $cid,
                UID => $UID,
                user => $user,
                login_rights => $login_rights,
                xls_id => $xls_id,

                new_camp => $svars->data->{new_camp},
                stop_banners => ref $svars->data->{stop_banners} eq 'ARRAY' ? $svars->data->{stop_banners} : [],
                ignore_image => $ignore_image,
            );

            $cancel->( map {iget($_)} @$errors ) if $errors && @$errors;
        } else {
            die "bad 'destination_camp' parametr value: $svars->data->{destination_camp}\n";
        }

        # авто-менеджера вычисляем заранее, чтобы правильно решить с мгновенной модерацией
        my $auto_manager_uid = $svars->data->{destination_camp} eq 'new' &&
            get_one_field_sql(PPC(uid => $client_chief_uid), q{
                SELECT c.ManagerUID
                FROM campaigns c
                WHERE c.uid = ? AND c.archived = 'No' AND c.ManagerUID > 0 AND c.type = 'text'
                GROUP BY c.ManagerUID
                ORDER BY count(c.cid) DESC
                LIMIT 1
                },
                $client_chief_uid,
            );

        if ($svars->data->{perform_quick_moderation}) {

            my @new_bids = map {map {$_->{bid}} grep {$_->{is_new}} @{$_->{banners}}} @{$campaign->{groups}};

            if (@new_bids) {

                $client_id = get_clientid(uid => $client_chief_uid);
                my $client    = get_client_data($client_id, [ 'country_region_id' ]) || {};

                # так как добавляются только ключи верхнего уровня - используем одноуровневое копирование
                # полагаемся на то что quick_moderate не изменяет своих аргументов
                my $camp_for_quick_moderate = { %$campaign };
                # $client->{country_region_id}, ManagerUID, AgencyUID, currency, OfferServicing и UserNotResident
                # используются для проверки "можно ли применять мгновенную модерацию"
                hash_copy($camp_for_quick_moderate, $svars->data->{new_camp}, [qw/currency/]);
                hash_copy($camp_for_quick_moderate, $vars, [qw/AgencyUID/]);
                $camp_for_quick_moderate->{ManagerUID} = $auto_manager_uid;
                $camp_for_quick_moderate->{OfferServicing} = rbac_get_camp_wait_servicing($rbac, $cid);
                $camp_for_quick_moderate->{UserNotResident} = get_one_user_field($client_chief_uid, 'not_resident');

                $camp_for_quick_moderate->{uid} = $client_chief_uid;
                $camp_for_quick_moderate->{cid} = $cid;

                # Send to moderation
                my $status_post_moderate = 'No';

                my $profile = Yandex::Trace::new_profile('confirmSaveCampXLS', tags => 'quick');
                my ($quick_modresult) = quick_moderate($camp_for_quick_moderate, $client);
                undef $profile;

                $profile = Yandex::Trace::new_profile('confirmSaveCampXLS', tags => 'order');
                order_camp($r, $client_chief_uid, $cid, $status_post_moderate, {bids => \@new_bids});
                undef $profile;

                # принимаем кампанию для возможности оплаты до результата проверки
                if (defined $quick_modresult) {
                    preliminary_moderate_campaign($cid, $quick_modresult);
                }
            }
        }
        my $import_format = $svars->data->{import_format};
        my $have_image_jobs = ($import_format eq 'xls' || $import_format eq 'xlsx') && @{$campaign->{imq_jobs}};
        if ($have_image_jobs) {
            my $shard = get_shard(cid => $cid);
            # при загрузке xls (в XLSCampaign::apply_camp_group_snapshot) задания на скачивание добавляются в очередь
            # и затем обрабатываются в скрите ppcProcessImageQueue.pl
            BannerImages::Queue::update_xls_id($shard, $xls_id, $campaign->{imq_jobs});
        }

        if ($svars->data->{destination_camp} eq 'new') {
            # Автоматическая отправка кампании на сервисирование на менеджера с наибольшим числом сервисируемых кампаний у клиента
            if ($auto_manager_uid) {
                Campaign::send_camp_to_service($rbac, $cid, $client_chief_uid, $auto_manager_uid, force => 1);
            }

            # Логгируем списание баллов для новой кампании после того, как у нас появился её cid
            # само списание уже произведено до начала импорта
            $uhost->log_reserved_units($login_rights->{client_chief_uid},
                hash_copy({cid => $cid}, $vars, qw/ManagerUID AgencyUID/)
            );
        }

        # remove used data
        $svars->delete();

        if ( $svars->data->{destination_camp} eq 'old') {
            new LockObject({object_type=>"campaign", object_id=>$cid})->delete();
        }

        return redirect($r, "$SCRIPT?cmd=importCampSuccess", hash_merge {
            ( make_new_camp => ($svars->data->{destination_camp} eq 'new') ? 1 : 0),
            ( has_imagead_jobs => $have_image_jobs ? 1 : 0 ),
        }, hash_cut \%FORM, qw/ulogin cid for_agency import_format/);
    }
    else {
        # !$FORM{confirm}

        # remove used data
        $svars->delete();

        if ( $svars->data->{destination_camp} eq 'old') {
            new LockObject({object_type=>"campaign", object_id=>$cid})->delete();
        }
        my $to_tab = $FORM{import_format} && $FORM{import_format} eq 'csv' ? "import_csv" : "import";
        return redirect($r, "$SCRIPT?cmd=showExportedXlsList", hash_merge {tab => $to_tab}, hash_cut \%FORM, qw/ulogin cid for_agency/);
    }
}

=head2 cmd_importCampToMediaplanXLS

        Второй шаг при загрузке медиаплана из XLS файла.
        В медиаплан не подгружаются параметры(Param1 и Param2) и теги.

=cut
sub cmd_importCampToMediaplanXLS :Cmd(importCampToMediaplanXLS)
    :Rbac(Code => [rbac_cmd_importCampXls, rbac_cmd_createMediaplan], Role => [super, media])
    :RequireParam(cid => 'Cid')
    :CheckCSRF
    :Description('Второй шаг при загрузке медиаплана из XLS файла.')
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS  vars    c/};
    my %FORM = %{$_[0]{FORM}};

    error(iget("Сессия импорта не существует")) unless ($FORM{svars_name});
    my $cid = $FORM{cid};
    my $svars = new StoredVars({name => $FORM{svars_name}, uid => $login_rights->{client_chief_uid}, prefer_string => 1})->load()
        or error(iget("Сохранённого объекта с указанным номером не существует"));

    my $xls_camp = $svars->data->{xls_camp};

    for my $g (@{$xls_camp->{groups}}) {
        $g->{banners} = [ grep { $_->{ad_type} ne 'image_ad' } @{$g->{banners}} ];
    }

    error(iget("Загрузить файл с группами в медиаплан невозможно.")) if any {@{$_->{banners}} > 1} @{$xls_camp->{groups}};

    my @only_relevance_match_groups_ids = map {$_->{pid}} grep { Models::AdGroup::has_only_relevance_match($_) } @{$xls_camp->{groups}};
    if (@only_relevance_match_groups_ids) {
        error(iget('Группы объявлений (№ %s) только с условием показа ---autotargeting не могут быть добавлены в медиаплан.', join ',', @only_relevance_match_groups_ids));
    }

    my $split_strategy = $FORM{split_strategy};
    $split_strategy = 'always_new_banner' unless $split_strategy && any { $split_strategy eq $_ } qw/always_new_banner shared_new_banner/;
    split_oversized_groups($split_strategy => $xls_camp->{groups}, client_id => $c->client_client_id);

    my ($mediaplan, $curr_camp) = XLSMediaplan::convert_to_mediaplan(
        $xls_camp,
        $FORM{cid},
        client_chief_uid => $c->client_chief_uid,
        uid => $uid
    );

    my $new_common_geo;
    if ($FORM{set_common_geo}) {
        #$FORM{geo} - старый формат, гео приходит строкой, $FORM{json_geo_changes} - список регионов приходит структурой
        my $geo = $FORM{geo};
        if (!defined $geo && $FORM{json_geo_changes}){
            my $regions = $FORM{json_geo_changes};
            $regions = from_json($regions) unless ref $regions;

            $new_common_geo = geo_changes_to_string($regions, undef, $c->client_client_id);
            error(iget('Не заданы регионы показа')) unless $new_common_geo;
        }
        #Если задано пустое гео - преобразование превратит его в 0, не делаем
        elsif (defined $geo) {
            $geo = GeoTools::modify_translocal_region_before_save($geo, {ClientID => $c->client_client_id});
            $new_common_geo = GeoTools::refine_geoid($geo, undef, {ClientID => $c->client_client_id});
        }
        my @errors = XLSMediaplan::validate_xls_mediaplan($mediaplan, $curr_camp,
            new_common_geo => $new_common_geo,
            has_empty_geo  => $svars->data->{has_empty_geo},
            geo_errors     => $svars->data->{geo_errors},
            geo            => $svars->data->{geo},
            ClientID       => $c->client_client_id,
        );
        error(join "\n", @errors) if @errors;
    }

    XLSMediaplan::save($mediaplan,
        client_chief_uid => $c->client_chief_uid,
        UID => $UID,
        geo => $svars->data->{geo},
        form_geo => $new_common_geo,
        rbac => $rbac,
        login_rights => $login_rights
    );

    return redirect($r, $SCRIPT,
        hash_merge {
            cmd => 'importCampSuccess',
            make_new_camp => ($svars->data->{destination_camp} eq 'new') ? 1 : 0,
            to_mediaplan  => 1,
        }, hash_cut \%FORM, qw/ulogin cid for_agency import_format/
    );
}

sub cmd_dropCampXLS :Cmd(dropCampXLS)
    :Description('удаление выгруженной кампании в формате xls')
    :RequireParam(cid => 'Cid')
    :Rbac(Code => rbac_cmd_importCampXls)
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS  vars/};
    my %FORM = %{$_[0]{FORM}};

    if(defined $FORM{hash}) {
        delete_xls_history( md5_hex => $FORM{hash}, cid => $FORM{cid});
    } else {
        error("nothing to delete")
    }

    return redirect($r, "$SCRIPT?cmd=showExportedXlsList", hash_cut \%FORM, qw/ulogin cid tab/);
}

sub cmd_showExportedXlsList :Cmd(showExportedXlsList)
    :ParallelLimit(Num => 3, Key => [UID, uid])
    :Description('список выгруженных кампаний в xls')
    :Rbac(Code => rbac_cmd_exportCampXls)
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $vars, $c) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS  vars    c/};
    my %FORM = %{$_[0]{FORM}};

    $vars->{cid} = $FORM{cid};

    my $cids = rbac_get_allow_xls_import_camp_list($rbac, $UID, $uid);
    if(@$cids){
        my $history = load_xls_history(cid => $cids);
        push(@{$vars->{exported_xls_list}}, grep { $_->{type} eq 'export' } @$history);
        push(@{$vars->{imported_xls_list}}, grep { $_->{type} eq 'import' } @$history);
    }

    my $client_id = get_clientid(uid => $uid);

    add_xls_image_processing_info($vars->{imported_xls_list}, $client_id);

    my $client_nds = get_client_NDS($client_id);
    my $client_discount = get_client_discount($client_id);

    my $options = {rbac => $rbac, owned_by_uid => $UID, cids => $cids, mediaType => 'text', client_nds => $client_nds, client_discount => $client_discount};
    $vars->{camps_name_only} = get_user_camps_name_only($c->client_chief_uid, $options);

    my $phrases_qty = Campaign::count_phrases_campaign($cids);
    $_->{is_xls_allow} = ($phrases_qty->{$$_{cid}} || 0) <= MAX_XLS_ROWS foreach @{$vars->{camps_name_only}};

    my $agencies_for_client = rbac_get_agencies_for_create_camp_of_client($rbac, $UID, $uid, 'for_import_xls');
    if (@$agencies_for_client) {
        $vars->{for_agencies} = get_all_sql(PPC(uid => $agencies_for_client),
                                                 ["select login, IFNULL(clients.name, users.FIO) as agency_name
                                                   from users
                                                   left join clients using(ClientID)",
                                                   where => {uid => SHARD_IDS}
                                                 ]);
    }

    if ( rbac_has_agency($rbac, $uid) ) {
        # есть свобода создавать самоходные/сервисируемые кампании у субклиента
        $vars->{allow_create_scamp_by_subclient} = get_allow_create_scamp_by_subclient($uid);
    }

    my $uhost = new APIUnits({scheme=>'XLS'});
    $vars->{user_units} = $uhost->check_or_init_user_units($login_rights->{client_chief_uid})->{$login_rights->{client_chief_uid}}{units_rest};

    my $tab = $FORM{tab} || 'export';
    if ( $tab eq 'import' || $tab eq 'import_csv') {
        $vars->{camp_limit_error} = check_add_client_campaigns_limits(uid => $uid);
    }
    $vars->{XLS_EXPIRES_DAYS} = $Settings::XLS_EXPIRES_DAYS;

    if ($tab eq 'import_csv') {
        my $client_currencies = get_client_currencies($client_id, allow_initial_currency => 1, uid => $c->client_chief_uid);
        hash_copy $vars, $client_currencies, qw/work_currency/;
    }

    $vars->{enable_cpm_deals_campaigns} = Direct::PredefineVars::_enable_cpm_deals_campaigns($c);
    if ($vars->{enable_cpm_deals_campaigns}){
        $vars->{new_deals_count} = Client::get_count_received_deals($c->login_rights->{ClientID});
    }
    $vars->{enable_content_promotion_video_campaigns} = Direct::PredefineVars::_enable_content_promotion_video_campaigns($c);
    $vars->{enable_cpm_yndx_frontpage_campaigns} = Direct::PredefineVars::_enable_cpm_yndx_frontpage_campaigns($c);

    $vars->{features_enabled_for_client} //= {}; 
    hash_merge $vars->{features_enabled_for_client}, Client::ClientFeatures::get_features_enabled_for_client(
        $c->client_client_id, [qw/support_chat/]);

    hash_merge $vars->{features_enabled_for_client}, Client::ClientFeatures::get_features_enabled_for_client(
        $c->login_rights->{ClientID}, [qw/is_grid_enabled is_hide_old_show_camps is_show_dna_by_default/]);

    return respond_bem($r, $c->reqid, $vars, source => 'data3');
}

sub cmd_importCampSuccess :Cmd(importCampSuccess)
    :Description('страница успеха загузки кампании из xls')
    :Rbac(Code => rbac_cmd_importCampXls)
{
    my ($r, $SCRIPT, $template, $UID, $uid, $rbac, $rights, $login_rights, $c, $vars) = @{$_[0]}{
      qw/R   SCRIPT   TEMPLATE   UID   uid   RBAC   RIGHTS   LOGIN_RIGHTS   c   vars/};
    my %FORM = %{$_[0]{FORM}};

    # формируем ссылку для нового файла
    if($FORM{cid} && $FORM{make_new_camp}){
        my $cid = $FORM{cid};
        $vars->{xls_file_data} = load_xls_history( cid => [$cid] )->[0];
    }

    # проверка баннеров в кампании на лицензируемые тематики, для загрузки документов модерации
    # удалена в DIRECT-70199

    return respond_bem($r, $c->reqid, $vars, source => 'data3');
}

# ---------------------------------------------------------------------------------------------------------------------------

1;

