=head1 NAME

DoCmdXls::History

=head1 DESCRIPTION

subroutines for working with export/import xls files

=cut

package DoCmdXls::History;
use Direct::Modern;

use Carp qw/croak/;

use Settings;
use HashingTools;
use Yandex::DBTools;
use Yandex::DBShards;
use Direct::Storage;
use PrimitivesIds qw/get_clientid/;
use Yandex::Validate;
use List::MoreUtils qw/none/;

our @POSSIBLE_TYPES = qw(export import);

use base qw/Exporter/;
our @EXPORT = qw/
    check_xls_history_by_id
    delete_xls_history
    history_preimport_to_import
    load_xls_history
    make_xls_history_camp_url
    save_xls_camp_to_history
/;


=head2 history_preimport_to_import(cid => $cid, id => $id)

    receives:
        cid: campaigns id
        id: id of row in xls_history table

    returns: 
=cut
sub history_preimport_to_import
{
    my %params = @_;

    if(!$params{id}){
        croak 'id is mandatory parameter';
    }

    if(!$params{cid}){
        croak 'cid is mandatory parameter';
    }

    my $filename = $params{id};
    my $md5_hex = $filename;
    $md5_hex =~ s/_.*$//;
    my $id = do_insert_into_table(PPC(cid=>$params{cid}), 'xls_history', {type => 'import', cid => $params{cid}, md5_hex => $md5_hex, filename => $filename});

    return $id;

}

=head2 save_xls_camp_to_history( cid => $cid, data => $xls_file, type => $type)

    receives:
        cid - campaign id
        data - scalar wich contains xls file
        type - possible types are 'import', 'export'
        not_save_metadata - boolean, if true just save file into filestorage without history metadata
        ClientID - client id

    returns: id of saved file
=cut
sub save_xls_camp_to_history
{
    my %params = @_;

    my $dbh = $params{cid} ? PPC(cid => $params{cid})
              : $params{ClientID} ? PPC(ClientID => $params{ClientID})
              : croak "sharding key missing";

    if(!$params{data}){
        croak 'data is required';
    } elsif(!$params{not_save_metadata} && (!$params{type} || !$params{cid})){
        croak 'not enough parameters';
    }

    if($params{type} && none { $params{type} eq $_ } @POSSIBLE_TYPES){
        croak 'unknown type';
    }

    my $extension = $params{filetype} || 'xls';

    my $md5_hex = md5_hex_utf8($params{data});

    my $id;
    unless($params{not_save_metadata}){
        $id = get_one_field_sql($dbh, ["select id from xls_history", where => {type => $params{type}, cid => $params{cid}, md5_hex => $md5_hex}]);
        return $id if $id;
    }
    
    my $client_id = $params{ClientID} // get_clientid(cid => $params{cid});

    my $filename = $md5_hex . '_' . time . ".$extension";
    # сохраняем в хранилище
    my $storage = Direct::Storage->new();
    $storage->save('xls_history', $params{data}, ClientID => $client_id, filename => $filename);
    eval {
        # сохраняем метаинформацию
        if ($params{cid}) {
            $id = do_insert_into_table($dbh, 'xls_history', {type => $params{type}, cid => $params{cid}, md5_hex => $md5_hex, filename => $filename});
        }
    };
    if ($@) {
        warn $@;
        # если не удалось сохранить метаинформацию, то удаляем файл из хранилища
        $storage->delete_file('xls_history', ClientID => $client_id, filename => $filename);
    }

    # если метаинформация сохранена, то возвращаем id строки из xls_history
    # иначе возвращаем имя файла, который состоит из md5_hex и времени сохранения
    return $id || $filename;
}

=head2 make_xls_history_camp_url(cid => $cid, md5_hex => $md5, type => $type)

    receives:
        cid - campaign id
        md5_hex - xls file md5 hex hash
        type - row type, possible types see in @POSSIBLE_TYPES

    returns:
        uri for internal redirect, for example: /xls.history/1234_5678.xls


=cut
sub make_xls_history_camp_url
{
    my %params = @_;

    if(!$params{cid} || !$params{md5_hex} || !$params{type}){
        croak 'not enough parameters';
    }

    if(none { $params{type} eq $_ } @POSSIBLE_TYPES){
        croak 'unknown type';
    } elsif( not is_valid_id($params{cid}) ){
        croak 'cid is not valid';
    } elsif( not _is_valid_md5_hex($params{md5_hex}) ){
        croak 'md5_hex is not valid';
    }

    my $filename = get_one_field_sql(PPC(cid => $params{cid}),
        ["select filename from xls_history", where => {md5_hex => $params{md5_hex}, type => $params{type}, cid => SHARD_IDS}]);

    return undef unless $filename;
    
    my $client_id = get_clientid(cid => $params{cid});

    my $url = "/storage/$client_id/xls_history/$filename";

    return $url;
}

=head2 load_xls_history( cid => [qw(123 456 789)] )

    receives:
        cid - arrayref to list of campaign ids

    returns:
        all import/export history
        [
          {
            'cid' => '1916666',
            'filename' => '19166661397804778',
            'logdate' => '2014-04-18 11:06:20',
            'md5_hex' => '83352afc2609bca34015f7553f0a6ef9',
            'type' => 'export'
          },
          {
            'cid' => '4306341',
            'filename' => '43063411397804785',
            'logdate' => '2014-04-18 11:06:27',
            'md5_hex' => 'b5d5ec153e68587fdee52b939d824457',
            'type' => 'export'
          },
        ];
=cut
sub load_xls_history
{
    my %params = @_;

    my $data = get_all_sql(PPC(cid=>$params{cid}), ["select id, id as xls_id, cid, md5_hex, filename, type, logdate from xls_history", where => {cid => SHARD_IDS}]);
    for my $row (@$data){
        $row->{filename} =~ /\.(\w+)$/;
        $row->{file_for_url} = sprintf "%s.%s", $row->{cid}, $1;
    }

    return $data;
}

=head2 delete_xls_history( md5_hex => $id, cid => $cid);

    receives:
        cid - campaign id
        md5_hex - xls file md5 hex hash

    returns:
        how much rows were deleted, "0E0" otherwise

=cut
sub delete_xls_history
{
    my %params = @_;

    unless($params{cid} && $params{md5_hex}){
        croak 'not enough parameters';
    }

    do_delete_from_table(PPC(cid=>$params{cid}), 'xls_history', where => {md5_hex => $params{md5_hex}, cid => $params{cid}});
}

sub _is_valid_md5_hex
{
    my $md5_hex = shift;

    my $result = $md5_hex =~ /[a-f0-9]{32}/ ? 1 : 0;
    return $result;
}


=head2 check_xls_history(xls_id => $id, cid => $cid);

    receives:
        cid - campaign id

        xls_id - pk in xls_history table

    returns:
        1 if row is exists, 0 otherwise
=cut
sub check_xls_history_by_id
{
    my %params = @_;

    unless($params{cid} && $params{xls_id}){
        croak 'not enough parameters';
    }

    my $result = get_one_field_sql(PPC(cid=>$params{cid}), ["select 1 from xls_history",  where => {id => $params{xls_id}}]) ? 1 : 0;

    return $result;
}

1;
