package EventLog;

# $Id$

=head1 NAME

    EventLog

=head1 DESCRIPTION

    Пакет для работы с логом важных событий по кампании/баннеру/фразе/клиенту.
    Данные из event-лога используется для формирования PUSH-уведомлений для мобильного Директа.
    Данные хранятся в таблице eventlog в ppc.

=cut

use strict;
use warnings;
use utf8;

use Try::Tiny;
use List::MoreUtils qw/any all uniq/;
use JSON;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::HashUtils;
use Yandex::ScalarUtils;
use Yandex::Validate qw/is_valid_int/;
use Yandex::TimeCommon qw/check_mysql_date unix2mysql/;
use Yandex::I18n;

use Settings;
use TextTools;

use parent qw/Exporter/;
our @EXPORT_OK = qw/
    log_event
    log_several_events
    get_events
    get_type2slug
    get_events_text_descr
    is_event_for_object
    push_notifications_enrich_payload
    %EVENTS
    @API_EVENTS
    %CLEAR_EVENTS_AFTER_DAYS
    INFORMATIONAL_WITH_LINK_TYPE
/;

use constant INFORMATIONAL_WITH_LINK_TYPE => 20;

=head2 EVENTS

    Перечень событий, которые могут храниться в таблице eventlog. Имеют структуру:
    slug => {
        type => # идентификатор типа события
        name => # человекопонятное наименование события
        object => client|campaign|banner|phrase|client # тип объекта, с которым асcоциировано событие
        params => [qw/param1 param2/], # наименования дополнительных параметров, которые сопровождают событие
        hidden => 1|0 не показывать в методе чтения EventLog для пользователя (GetEventsLog)
    }

=cut

#Warning: при добавлении нового события, не забыть про API
our %EVENTS = (
    money_out                   => {
        type   => 1,
        name   => iget_noop('На кампании закончились деньги'),
        object => 'campaign',
    },
    money_warning               => {
        type   => 2,
        name   => iget_noop('На кампании израсходовано 80% средств'),
        object => 'campaign',
        params => [ qw/sum_rest currency/ ],
    },
    money_in                    => {
        # пишется из java
        type   => 3,
        name   => iget_noop('На кампанию поступили деньги'),
        object => 'campaign',
        params => [ qw/sum_payed currency/ ],
    },
    camp_finished               => {
        # пишется из java
        type   => 5,
        name   => iget_noop('Кампания остановлена, так как наступила дата окончания показов'),
        object => 'campaign',
        params => [ qw/finish_date/ ],
    },
    warn_place                  => {
        type   => 7,
        name   => iget_noop('Фраза вытеснена со своей позиции'),
        object => 'phrase',
        params => [ qw/old_place/ ],
    },
    banner_moderated            => {
        type   => 8,
        name   => iget_noop('Получен результат модерации баннера'),
        object => 'banner',
        params => [ qw/results is_edited_by_moderator/ ],
    },
    paused_by_day_budget        => {
        type   => 9,
        name   => iget_noop('Кампания остановилась по дневному бюджету'),
        object => 'campaign',
        params => [ qw/bs_stop_time/ ],
    },
    campaign_copied             => {
        type   => 10,
        name   => iget_noop('Кампания скопирована'),
        object => 'campaign',
        params => [ qw/from_cid flags manager_uid agency_uid/ ],
    },
    currency_convert_finished   => {
        type   => 11,
        name   => iget_noop('Завершён переход в реальную валюту'),
        object => 'client',
        params => [ qw/new_currency convert_type/ ],
    },
    retargeting_goals_check     => {
        type   => 12,
        name   => iget_noop('Изменилась доступность цели из условия подбора аудитории для клиента'),
        object => 'client',
        params => [ 'retargetings' ],
    },
    # -- WALLET
    money_out_wallet            => {
        type   => 13,
        name   => iget_noop('На общем счете закончились деньги'),
        object => 'campaign',
    },
    money_warning_wallet        => {
        type   => 14,
        name   => iget_noop('На общем счете израсходовано 80% средств'),
        object => 'campaign',
        params => [ qw/sum_rest currency/ ],
    },
    money_in_wallet             => {
        # пишется из java
        type   => 15,
        name   => iget_noop('Общий счет пополнен'),
        object => 'campaign',
        params => [ qw/sum_payed currency/ ],
    },
    paused_by_day_budget_wallet => {
        type   => 16,
        name   => iget_noop('Кампании остановлены по дневному бюджету'),
        object => 'campaign',
        params => [ qw/bs_stop_time/ ],
    },
    money_out_wallet_with_ao    => {
        type   => 17,
        name   => iget_noop('На общем счёте закончились деньги, и был достигнут порог отключения'),
        object => 'campaign',
    },
    strategy_data_sum_changed   => {
        type    => 18,
        name    => iget_noop('Поменялось ограничение бюджета стратегии'),
        object  => 'campaign',
        params  => [ qw/budget_before budget_after/ ],
        hidden  => 1,
    },
    daily_budget_sum_changed   => {
        type    => 19,
        name    => iget_noop('Поменялось ограничение дневного бюджета'),
        object  => 'campaign',
        params  => [ qw/budget_before budget_after/ ],
        hidden  => 1,
    },
    custom_message_with_link   => {
        # пишется из java
        type    => INFORMATIONAL_WITH_LINK_TYPE, # 20
        name    => iget_noop('Информационное сообщение'),
        object  => 'client',
        params  => [ qw/link text/ ],
        hidden  => 1,
    },
);

our @API_EVENTS = map { $_->{type} } grep { !($_->{hidden}) } values %EventLog::EVENTS;

=head2 is_event_for_object
    Принимает:
        $event_class название класса событий например campaign_copied или money_in
        $object название объекта из %EVENTS например 'campaign'
    Возвращает:
        1 если данное событие происходит с данным объектом, 0 если нет
=cut

sub is_event_for_object ($$){
    my ($event_class, $object) = @_;
    return ($EVENTS{$event_class} && $EVENTS{$event_class}->{object} eq $object) ? 1 : 0;
}


=pod

    Параметр results в событии banner_moderated это ссылка на хеш с ключами:
        global => accepted|declined|declined_partly    # глобальный результат модерации для баннера (принят|отклонён|частично отклонён)
        text => Yes|No    # результат модерации текста объявления
        phrases => Yes|No    # результат модерации фраз и рубрик
        contactinfo => Yes|No    # результат модерации контактной информации
        sitelinks_set => Yes|No    # результат модерации сайтлинков
        geo_exception => ['kz', 'by', 'all', ...]    # регионы, в которых не будет показываться объявление
    Эти данные не вынесены на верхний уровень параметров, т.к. могут присутствовать не все из них.

=cut

=head2 %CLEAR_EVENTS_AFTER_DAYS

    В хеше %CLEAR_EVENTS_AFTER_DAYS храним данные о том, сколько дней нужно хранить события каждого типа в логе.
    Ключами в нём являются текстовые описания событий из %EVENTS, а значениям -- количество дней, за который
    надо хранить события данного типа. Если вместо срока хранения указать undef, то удаляться такие не будут.
    В хеше должны обязательно присутствовать все события из %EVENTS (проверяется юнит-тестом).

        %CLEAR_EVENTS_AFTER_DAYS = (
            money_in => 30,
            warn_place => 7,
        );

=cut

our %CLEAR_EVENTS_AFTER_DAYS = (
    money_warning               => 30,
    money_in                    => undef,
    money_out                   => undef,
    camp_finished               => 30,
    warn_place                  => 7,
    banner_moderated            => 30,
    paused_by_day_budget        => 30,
    campaign_copied             => undef,
    retargeting_goals_check     => 30,
    currency_convert_finished   => undef,
    money_in_wallet             => undef,
    money_out_wallet            => undef,
    money_warning_wallet        => 30,
    paused_by_day_budget_wallet => 30,
    money_out_wallet_with_ao    => 30,
    strategy_data_sum_changed   => 30,
    daily_budget_sum_changed    => 30,
    custom_message_with_link    => 7,
);

=head2 log_event

    Записывает событие с переданными параметрами в event-лог.
    Принимает именованные параметры:
        slug -- один из ключей хеша %EVENTS; определяет тип произошедшего события; обязательный параметр
        ClientID -- идентификатор клиента в балансе; обязательный параметр
        cid -- № кампании, с которой связано событие; может отсутствовать
        bid -- № баннера, с которым связано событие; может отсутствовать; если указан bid, должен быть указан и cid
        bids_id -- идентификатор фразы (из таблицы bids); может отсутствовать; если указан bids_id, должны быть указаны bid и cid
        params -- ссылка на хеш, описывающий дополнительные параметры, сопровождающие событие
    При указании неверных параметров -- умирает.
    При ошибках во время записи данных в БД, отправляет alert и НЕ умирает.

    log_event(slug => 'money_warning', ClientID => $client_id, cid => $cid, params => {param1 => $param1, param2 => $param2});

=cut

sub log_event {
    my (%event) = @_;
    log_several_events([\%event]);
}

=head2 log_several_events

    Записывает несколько событий в event-лог.
    Принимает ссылку на список ссылок на хеши. Каждый хеш описывает одно событие, данные в нём аналогичны данным, принимаемым log_event.
    При указании неверных параметров -- умирает.
    При ошибках во время записи данных в БД, отправляет alert и НЕ умирает.

    $event1 = {slug => 'money_warning', ClientID => $client_id, cid => $cid, params => {sum_rest => 123}};
    $event2 = {slug => 'money_out', ClientID => $client_id, cid => $cid};
    log_several_events([$event1, $event2]);

=cut

sub log_several_events {
    my ($data) = @_;

    die 'список событий не передан или передан неверно' unless $data && ref($data) eq 'ARRAY' && @$data;
    my @data_to_insert;
    my $eventtime = unix2mysql(time);
    for my $event(@$data) {
        my $err = _validate_event($event);
        die $err if defined $err;

        my $row = hash_cut $event, qw/ClientID cid bid bids_id/;
        $row->{$_} ||= 0 for qw/ClientID cid bid bids_id/; # эти поля NOT NULL
        $row->{type} = $EVENTS{$event->{slug}}->{type};
        if ($event->{params}) {
            $row->{params} = encode_json $event->{params};
        }
        push @data_to_insert, [$eventtime, map {$row->{$_}} qw(ClientID cid bid bids_id type params)];
    }
    foreach_shard ClientID => \@data_to_insert, by => sub {$_->[1]}, sub {
        my ($shard, $chunk) = @_;
        do_mass_insert_sql(PPC(shard => $shard), 'INSERT INTO eventlog (eventtime, ClientID, cid, bid, bids_id, type, params) VALUES %s', $chunk);
    };
}

=head2 get_events

    Возвращает список событий из eventlog'а.
    Принимает именованные параметры:
        ClientID -- идентификатор клиента из Баланса; может быть ссылкой на массив идентификаторов; обязательный параметр
        date_from -- дата (и опционально время) начиная С которой (включительно) необходимо выбрать события; в MySQL-совместимом формате; обязательный параметр
        date_to -- дата (и опционально время) ПО которую (включительно) необходимо выбрать события; в MySQL-совместимом формате; обязательный параметр
        only_last_event -- если истина, то будет выбрана только последняя запись о каждом типе события по каждому объекту; необязательный параметр
        conditions -- ссылка на хеш с дополнительными условиями отбора событий в DBTools-совместимом формате; необязательный параметр
        limit -- опционально
        offset — опционально, работает только тогда, когда указан limit
    Возвращает ссылку на массив со ссылками на хеши, описывающие события.

    $events = get_events(ClientID => $clientid, date_from => '2011-11-01', date_to => '2011-11-15', only_last_event => 1, conditions => {'cid' => $cid});
    $events = [
        {
            id => 123, # идентификатор события
            eventtime => '2011-11-17 00:13:14', # время, в которое произошло событие
            slug => 'camp_finished', # короткое обозначение событие (соответствует ключам %EVENTS)
            type => 5, # числовой тип события (посмотреть возможные значения можно в %EVENTS)
            ClientID => 456789, # идентификатор клиента из Баланса, с объектом которого произошо событие
            cid => 234567, # номер кампании, с которой произошло событие
            bid => 3456789, # номер объявления, с которым произошло событие
            bids_id => 45678900, # идентификатор фразы из таблицы bids
            params => { # ссылка на хеш с параметрами события; список параметров можно посмотреть в %EVENTS
                finish_date => '2011-11-17',
            },
        },
    ];

=cut

sub get_events {
    my (%O) = @_;

    my $clientids = ref($O{ClientID}) eq 'ARRAY' ? $O{ClientID} : [$O{ClientID}];
    die 'ClientID не указан или указан неверно: ' . join ', ', map {str $_} @$clientids unless $clientids && @$clientids && all { is_valid_int($_, 0) } @$clientids;
    die 'date_from не указан или указан неверно: ' . str($O{date_from}) unless $O{date_from} && check_mysql_date($O{date_from});
    die 'date_to не указан или указан неверно: ' . str($O{date_to}) unless $O{date_to} && check_mysql_date($O{date_to});

    my @sql_request = ('SELECT e.id, e.eventtime, e.type, e.ClientID, e.cid, e.bid, e.bids_id, e.params FROM eventlog e');
    my %sql_where_conditions;

    $sql_where_conditions{'ClientID'} = SHARD_IDS;
    $sql_where_conditions{'eventtime__ge'} = $O{date_from};
    $sql_where_conditions{'eventtime__le'} = $O{date_to};
    if ($O{conditions} && ref($O{conditions}) eq 'HASH') {
        hash_merge \%sql_where_conditions, $O{conditions};
    }
    if ($O{only_last_event}) {
        push @sql_request, 'INNER JOIN ( SELECT max(id) AS max_id FROM eventlog', (WHERE => \%sql_where_conditions), 'GROUP BY ClientID, type, cid, bid, bids_id) maxt ON maxt.max_id = e.id';
    }

    push @sql_request, (where => \%sql_where_conditions);

    # для избежания разности между естественной сортировкой по времени
    # и случайной сортировкой, когда сортировка по времени сбивается
    # дополнительным под-запросом из-за only_last_event
    # включаем явную сортировку DIRECT-28851
    push @sql_request, "ORDER BY e.eventtime, e.id";

    if($O{limit} && $O{limit} =~ /^\d+$/) {
        push @sql_request, (limit => $O{limit});
        if($O{offset} && $O{offset} =~ /^\d+$/) {
            push @sql_request, (offset => $O{offset});
        }
    }

    my $events = get_all_sql(PPC(ClientID => $clientids), \@sql_request);
    if ($events && ref($events) eq 'ARRAY' && @$events) {
        my $eventtype2slug = get_type2slug();
        for my $event(@$events) {
            $event->{slug} = $eventtype2slug->{$event->{type}};
            $event->{params} = from_json($event->{params}) if $event->{params};
        }
    }

    return $events;
}

=head2 get_type2slug

    Возвращяет ссылку на хеш для перехода от числового типа события к его короткому названию (slug).
    $type2slug = get_type2slug();
    $slug = $type2slug->{5}; # $slug == 'camp_finished'

=cut

sub get_type2slug {
    return {map {$EVENTS{$_}->{type} => $_} keys %EVENTS};
}

=head2 _validate_event

    Функция проверки параметров события.
    На вход принимает ссылку на хеш, описывающую событие. В нём должны быть ключи:
        slug -- проверяется наличие и существование такового в $EVENTS
        ClientID -- проверяется наличие и что это число
        cid -- при наличии проверяется, что это число
        bid -- при наличии проверяется, что это число и что задан cid
        bids_id -- при наличии проверяется, что это число и что заданы cid и bid
        params -- проверяется наличие всех параметров (ключей), значения никак не проверяются
    Возвращает текст ошибки или undef. Тексты не обёрнуты в iget, т.к. предполагается, что наружу они использоваться не будут.

=cut

sub _validate_event {
    my ($event) = @_;

    return 'slug не указан или указан неверно: ' . str($event->{slug}) unless defined $event->{slug} && exists $EVENTS{$event->{slug}};

    return 'ClientID не указан или указан неверно: ' . str($event->{ClientID}) unless is_valid_int($event->{ClientID});

    my $event_attribs = $EVENTS{$event->{slug}};

    if (defined $event->{cid} || defined $event->{bid} || defined $event->{bids_id} || any {$event_attribs->{object} eq $_} qw/campaign banner phrase/) {
        return 'cid не указан или указан неверно: ' . str($event->{cid}) unless defined $event->{cid} && is_valid_int($event->{cid});
    }
    if (defined $event->{bid} || defined $event->{bids_id} || any {$event_attribs->{object} eq $_} qw/banner phrase/) {
        return 'bid не указан или  указан неверно: ' . str($event->{bid}) unless is_valid_int($event->{bid});
    }
    if (defined $event->{bids_id} || $event_attribs->{object} eq 'phrase') {
        return 'bids_id не указан или указан неверно: ' . str($event->{bids_id}) unless is_valid_int($event->{bids_id});
    }

    if ($event_attribs->{params}) {
        return 'не указаны или неверно указаны параметры события' unless $event->{params} && ref($event->{params}) eq 'HASH';
        for my $param_name(@{$event_attribs->{params}}) {
            return "отсутствует параметр $param_name" unless exists $event->{params}->{$param_name};
        }
        for my $event_param_name(keys %{$event->{params}}) {
            return "лишний параметр $event_param_name" unless any {$event_param_name eq $_} @{$event_attribs->{params}};
        }
    }

    return undef;
}

=head2 get_events_text_descr

    Возвращает хэш, event_id => {descr => описание объекта, camp_descr => описание кампании объекта},
    На вход ждет массив хэшей событий, среди которых есть id события, а так же cid, bid и bids_id

=cut

sub get_events_text_descr {

    my $events = shift;

    my $TYPE2SLUG = get_type2slug();
    $_->{type_object} = $EVENTS{$TYPE2SLUG->{$_->{type}}}{object} for @$events;

    my $bids = [uniq map {$_->{bid}} grep {$_->{type_object} eq 'banner'} @$events];

    my $banners_descr;
    if (@$bids) {
        $banners_descr = get_hashes_hash_sql(PPC(bid => $bids), ['select bid, cid, title from banners', where => {bid => SHARD_IDS}]);
    }

    foreach my $bid (@$bids) {
        $banners_descr->{$bid}{title} = html2string($banners_descr->{$bid}{title});
    }
    my @phrase_events = grep {$_->{type_object} eq 'phrase'} @$events;

    my $phrases_descr = {};

    if (@phrase_events) {
        foreach_shard ClientID => \@phrase_events, by => sub {$_->{ClientID}}, sub {
            my ($shard, $chunk) = @_;
            my @bids_ids = map {$_->{bids_id}} @$chunk;
            hash_merge $phrases_descr, get_hashes_hash_sql(PPC(shard => $shard),
                    ['select id, cid, phrase from bids', where => {id => \@bids_ids}]);
        };

        foreach my $descr (values %$phrases_descr) {
            $descr->{phrase} =~ s/\s\-.+$//;

            my $phrase_old = $descr->{phrase};

            $descr->{phrase} = substr($descr->{phrase}, 0, 42);
            if ($descr->{phrase} ne $phrase_old) {
                substr($descr->{phrase}, 39, 3, '...');
            }
        }
    }
    # идентификаторы кампаний из событий на кампанию
    my @cids = map {$_->{cid}} grep {$EVENTS{$TYPE2SLUG->{$_->{type}}}{object} eq 'campaign'} @$events;
    # идентификаторы кампаний из событий на баннер
    push @cids, map {$_->{cid}} values %$banners_descr;
    # идентификаторы кампаний из событий на фразу
    push @cids, map {$_->{cid}} values %$phrases_descr;

    @cids = uniq grep {defined $_} @cids;

    my $camps_descr;
    if ( scalar @cids ) {
        $camps_descr = get_hashes_hash_sql(PPC(cid => \@cids), ['select cid, name from campaigns', where => {cid => SHARD_IDS}]);
    }

    foreach my $cid (@cids) {
        my $camp_name_old = $camps_descr->{$cid}{name} // '';

        $camps_descr->{$cid}{name} = substr($camps_descr->{$cid}{name} // '', 0, 40);
        if ($camps_descr->{$cid}{name} ne $camp_name_old) {
            $camps_descr->{$cid}{name} .= '...';
        }
    }

    my $result;
    foreach my $event (@$events) {
        if ($event->{type_object} eq 'campaign') {
            $result->{$event->{id}}{descr} = $camps_descr->{$event->{cid}}{name};
        } elsif ($event->{type_object} eq 'banner') {
            $result->{$event->{id}}{descr} = $banners_descr->{$event->{bid}}{title};
        } elsif ($event->{type_object} eq 'phrase') {
            $result->{$event->{id}}{descr} = $phrases_descr->{$event->{bids_id}}{phrase} || 'удалена';
        } elsif ($event->{type} == INFORMATIONAL_WITH_LINK_TYPE) {
            if ($event->{params}) {
                $event->{params} = from_json($event->{params});
                # требуем полной валидности события
                if ($event->{params}->{text} && $event->{params}->{link}) {
                    $result->{$event->{id}}->{descr} = $event->{params}->{text};
                    $result->{$event->{id}}->{link} = $event->{params}->{link};
                }
            }
        }
        $result->{$event->{id}}{camp_descr} = $camps_descr->{$event->{cid}}{name};
    }
    return $result;
}

# префикс, превращающий link в пушах в валидно выглядющую ссылку
use constant LINK_PREFIX => 'https://direct.yandex.ru/';

my %push_notifications_enrich_data = (
    money_out_wallet => {
        link => 'account/payment'
    },
    money_out => {
        link => 'campaign/{CID}',
        category => 'PAYMENT_AND_CAMPAIGN'
    },
    money_warning_wallet => {
        link => 'account',
        category => 'PAYMENT_AND_STATS'
    },
    money_warning => {
        link => 'campaign/{CID}',
        category => 'PAYMENT_AND_STATS'
    },
    paused_by_day_budget_wallet => {
        link => 'account/dayBudget'
    },
    paused_by_day_budget => {
        link => 'campaign/{CID}/dayBudget'
    },
    money_in_wallet => {
        link => 'account',
    },
    money_in => {
        link => 'campaign/{CID}'
    },
    warn_place => {
        link => 'campaign/{CID}/changePrice'
    },
    banner_moderated => {
        link => 'campaign/{CID}/banner/{BID}'
    },
    banner_moderated_multi => {
        link => 'campaign/{CID}'
    },
    banner_moderated_multicamp => {
        link => 'campaigns'
    },
);

=head2 push_notifications_enrich_payload

Добавить в payload дополнительные ключи и значения в зависимости от типа события и от различных id в $first_event
Вернуть undef или ссылку после неудачной попытки параметризации.

=cut

sub push_notifications_enrich_payload {
    my ($payload, $msg_type, $first_event) = @_;
    return undef unless exists $push_notifications_enrich_data{$msg_type};

    if (exists $push_notifications_enrich_data{$msg_type}->{category}) {
        $payload->{category} = $push_notifications_enrich_data{$msg_type}->{category};
    }
    if (exists $push_notifications_enrich_data{$msg_type}->{link}) {
        my $link = $push_notifications_enrich_data{$msg_type}->{link};
        if ($first_event->{cid}) {
            $link =~ s/{CID}/$first_event->{cid}/;
        }
        if ($first_event->{bid}) {
            $link =~ s/{BID}/$first_event->{bid}/;
        }
        if ($link !~ /[{}]/) {
            $payload->{link} = LINK_PREFIX . $link;
        } else {
            return $link;
        }
    }
    return undef;
}

1;
