package Experiments;

# $Id$

use Direct::Modern;

use JSON;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::I18n qw/iget/;
use Yandex::TimeCommon;
use Yandex::DateTime;
use Yandex::HashUtils qw/hash_merge hash_cut/;

use Campaign::Types;

use Settings;



=head2 create_experiment

Создание записей об эксперименте

Структура данных эксперимента:

    primary_cid
    secondary_cid
    percent
    date_from
    date_to

Возвращает experiment_id

=cut

sub create_experiment {
    my ($client_id, $experiment) = @_;

    my $exp_id;

    $exp_id = get_new_id('experiment_id');
    do_in_transaction {
        do_insert_into_table(PPC(ClientID => $client_id), experiments => {
                experiment_id => $exp_id,
                ClientID => $client_id,
                percent => $experiment->{percent},
                date_from => $experiment->{date_from},
                date_to => $experiment->{date_to},
                extra => _rebuild_extra({}, create_time => unix2human(time)),
            });

        my @ce_records = (
            [ $exp_id, primary => $experiment->{primary_cid} ],
            [ $exp_id, secondary => $experiment->{secondary_cid} ],
        );
        do_mass_insert_sql(PPC(ClientID => $client_id),
            'INSERT INTO campaigns_experiments (experiment_id, role, cid) VALUES %s',
            \@ce_records,
        );
    };

    return $exp_id;
}



=head2 validate_experiment

Проверка данных эксперимента

Возвращает ошибку или пустое значение в случае успеха

=cut

sub validate_experiment {
    my ($uid, $experiment, %opt) = @_;

    my $primary_cid = $experiment->{primary_cid};
    my $secondary_cid = $experiment->{secondary_cid};

    my $user_camp = get_hashes_hash_sql(PPC(uid => $uid), [
            'SELECT cid, IFNULL(currency, "YND_FIXED") as currency, wallet_cid, experiment_id
            FROM campaigns c
            LEFT JOIN campaigns_experiments ce USING(cid)',
            WHERE => {
                uid => $uid,
                type => get_camp_kind_types('web_edit_base'),
                cid => [$primary_cid, $secondary_cid],
            },
        ]);

    for my $cid ($primary_cid, $secondary_cid) {
        my $camp = $user_camp->{$cid};
        return iget("Неверный id кампании %s", $cid)  if !$camp;
        return iget("Кампания %s не валютная", $cid)  if $camp->{currency} eq 'YND_FIXED';
        return iget("Кампания %s не подключена к общему счёту", $cid)  if !$camp->{wallet_cid};
        return iget("Кампания %s уже участвует в эксперименте", $cid)  if $camp->{experiment_id};
    }

    my $date_from = str_round_day($experiment->{date_from});
    my $date_to = str_round_day($experiment->{date_to});
    return iget("Дата начала не может быть раньше завтра")  if $date_from lt str_round_day(tomorrow());
    return iget("Дата завершения не может быть раньше даты начала")  if $date_from gt $date_to;
    return iget("Продолжительность не должна превышать 90 дней")  if scalar get_distinct_dates($date_from, $date_to) > 90;

    return iget("Недопустимый процент аудитории")  if $experiment->{percent} < 1 || $experiment->{percent} > 99;

    return;
}




=head2 start_experiment

Запуск эксперимента

=cut

sub start_experiment {
    my ($client_id, $experiment_id) = @_;

    do_in_transaction {
        my $experiment = get_experiment($client_id, $experiment_id);
        croak "Invalid experiment id"  if !$experiment;
        croak "Inappropriate status to start"  if $experiment->{status} ne 'New';

        do_update_table(PPC(ClientID => $client_id), campaigns => {statusBsSynced => 'No'},
            where => {cid => [$experiment->{primary_cid}, $experiment->{secondary_cid}]},
        );
        do_update_table(PPC(ClientID => $client_id), experiments => {
                status => 'Started',
                extra => _rebuild_extra($experiment, start_time => unix2human(time)),
            },
            where => {experiment_id => $experiment_id},
        );
    };

    return;
}



=head2 stop_experiment

Остановка эксперимента

=cut

sub stop_experiment {
    my ($client_id, $experiment_id) = @_;

    do_in_transaction {
        my $experiment = get_experiment($client_id, $experiment_id);
        croak "Invalid experiment id"  if !$experiment;
        croak "Experiment already stopped"  if $experiment->{status} eq 'Stopped';

        do_update_table(PPC(ClientID => $client_id), experiments => {
                status => 'Stopped',
                extra => _rebuild_extra($experiment, stop_time => unix2human(time)),
            },
            where => {experiment_id => $experiment_id},
        );
        do_delete_from_table(PPC(ClientID => $client_id), 'campaigns_experiments', where => {experiment_id => $experiment_id});
        do_update_table(PPC(ClientID => $client_id), campaigns => {statusBsSynced => 'No'},
            where => {cid => [$experiment->{primary_cid}, $experiment->{secondary_cid}]},
        );
    };

    return;
}

=head2 stop_experiment

Остановка эксперимента при удалении одной из участвующих в нем кампаний

=cut

sub stop_experiment_on_delete_campaign {
    my ($client_id, $cid) = @_;

    my $experiments = _query_experiments(PPC(ClientID => $client_id), {
        _OR => {
            'pce.cid' => $cid,
            'sce.cid' => $cid,
        },
        'e.ClientID' => $client_id,
        status__ne => 'Stopped',
    });

    for my $experiment (@$experiments) {
        stop_experiment($client_id, $experiment->{experiment_id});
    }
}


=head2 get_user_experiments

Возвращает список экспериментов клиента в виде хеша { $experiment_id => {}, ... }

Опции:

    experiment_id

=cut

sub get_user_experiments {
    my ($client_id, %opt) = @_;

    my $where = {'e.ClientID' => $client_id};
    $where->{"e.experiment_id"} = $opt{experiment_id}  if $opt{experiment_id};

    my $experiments = _query_experiments(PPC(ClientID => $client_id), $where);
    my %experiment_by_id = map {($_->{experiment_id} => $_)} @$experiments;

    return \%experiment_by_id;
}


sub get_experiment {
    my ($client_id, $experiment_id) = @_;

    my $experiments = _query_experiments(PPC(ClientID => $client_id), {
            'e.ClientID' => $client_id,
            'e.experiment_id' => $experiment_id,
        },
        for_update => 1,
    );
    return $experiments->[0];
}


sub _query_experiments {
    my ($shard, $sql_where, %opt) = @_;

    my $experiments = get_all_sql($shard, [
            'SELECT e.experiment_id, e.ClientID, e.status, e.percent, e.date_from, e.date_to, e.extra,
                pce.cid as primary_cid, pc.name as primary_name,
                sce.cid as secondary_cid, sc.name as secondary_name
            FROM experiments e
            LEFT JOIN campaigns_experiments pce ON pce.experiment_id = e.experiment_id AND pce.role="primary"
            LEFT JOIN campaigns pc ON pc.cid = pce.cid
            LEFT JOIN campaigns_experiments sce ON sce.experiment_id = e.experiment_id AND sce.role="secondary"
            LEFT JOIN campaigns sc ON sc.cid = sce.cid',
            WHERE => $sql_where,
            ($opt{for_update} ? 'FOR UPDATE' : ()),
        ]);

    for my $experiment (@$experiments) {
        hash_merge $experiment, from_json(delete $experiment->{extra} || "{}");
    }

    return $experiments;
}


sub _rebuild_extra {
    my ($experiment, %new_values) = @_;

    state $extra_fields = [qw/
        create_time
        start_time
        stop_time
        primary_cid
        secondary_cid
    /];

    my $new_extra = hash_merge hash_cut($experiment, @$extra_fields), \%new_values;
    
    return to_json $new_extra;
}

1;
