package Export::AgencyCheckForMetrika;

=head1 NAME

    Export::AgencyCheckForMetrica

=head1 DESCRIPTION

    Проверка прав представителя агентства на создание счётчика для Метрики
    http://wiki.yandex-team.ru/JandexMetrika/KartochkaProekta/Techinfo/DirectAgencies

=cut

use strict;
use warnings;

use Settings;
use RBAC2::Extended;
use RBACElementary;
use RBACDirect;

use utf8;

=head2 handler

    Приходят cgi параметры
      uid - пользователь, авторизовавшийся в Метрике
      client_uid - uid клиента
      agency_uid - uid агентства

    Проверяем:
      - является ли uid представителем agency_uid
      - является ли client_uid клиентом agency_uid

    Возвращаем хэш, сериализованный в JSON:
      {
          result: 1 | 0,    // результат проверки 
          limited: 1 | 0,   // является ли представитель ограниченным 
          message: '',      // если result=0 - описание проблемы
      }

=cut
sub handler {
    my ($r, $multiform) = @_;

    # получаем, проверяем параметры
    my %param = %$multiform;
    for my $key (qw/uid agency_uid client_uid/) {
        if (!defined $param{$key} || $param{$key} !~ /^\d+$/ || !$param{$key}) {
            return export_result(500, {result => 0, message => "Incorrect param $key"});
        }
    }
    my ($uid, $agency_uid, $client_uid) = @param{qw/uid agency_uid client_uid/};

    # инициализация RBAC
    my $rbac = RBAC2::Extended->get_singleton($uid);

    # проверяем, является ли uid агентством
    my $rbac_info = rbac_who_is_detailed($rbac, $uid);
    if ($rbac_info->{role} ne 'agency') {
        return export_result(200, {result => 0, message => "$uid is not agency in direct"});
    }
    # проверяем, является ли uid представителем agency_uid
    my $agency_chief_uid = $rbac_info->{is_agency_chief} ? $uid : rbac_get_chief_rep_of_agency_rep($uid);
    if ($agency_chief_uid != $agency_uid) {
        return export_result(200, {result => 0, message => "$agency_uid is not chief agency for $uid"});
    }

    # проверяем, является ли агентство ограниченным
    my $limited = 0;
    if ($agency_chief_uid != $uid && $rbac_info->{is_agency_limited}) {
        $limited = 1;
    }

    # проверяем, является ли client_uid клиентов агентства uid
    if (!rbac_is_owner($rbac, $uid, $client_uid)) {
        return export_result(200, {result => 0, message => "$client_uid is not subclient of agency $uid"});        
    }

    return export_result(200, {result => 1, limited => $limited, message => "All ok - $agency_uid is chief of $uid and $client_uid is subclient of $uid"});
}

sub export_result($$)
{
    my ($code, $ret) = @_;
    return {json => $ret, code => $code};
}

1;
