package Export::UserRole;

=head1 NAME

    Export::UserRole

=head1 DESCRIPTION

    Получить роль пользователя и информацию о пользователе по uid

=cut

use strict;
use warnings;

use XML::LibXML qw//;

use Yandex::HashUtils;

use Settings;
use RBAC2::Extended;
use RBACElementary;
use RBACDirect;
use Yandex::DBTools;
use PrimitivesIds;
use User;
use Client;

use Stat::OrderStatDay;

use utf8;

=head2 handler

    Ппараметры формы
      uid - id пользователя
      format - формат вывода, допускаются форматы: json (по-умолчанию), xml

    Возвращаем хэш, сериализованный в JSON:
      {
          role: 'agency', роль пользователя 
          email: 'user@email',
          account_age: $age_in_seconds,
          is_easy: 1/0,
          avg_3month: 0.0 # Среднемесячная сумма расходов за последние 3 месяца
          avg_1month: 0.0 # Среднемесячная сумма расходов за последний месяц
          client_id: привязка к клиенту из rbac
          first_show_date: 'YYYY-MM-DD' # дата первой открутки
      }
    или в xml
    <?...?>
    <user>
        <role>agency</role>
        ... те же поля что и выше
    </user>

    В случае ошибки - 500

=cut
sub handler {
    my ($r, $multiform) = @_;

    # получаем параметры
    my %param = %$multiform;

    # проверяем параметры
    my $uid = $param{uid};
    if (!defined $uid || $uid !~ /^\d+$/) {
        return export_result(500, {result => 0, message => "Incorrect param uid"}, $param{format});
    }

    # инициализация RBAC
    my $rbac = RBAC2::Extended->get_singleton($uid);

    my $role = rbac_who_is($rbac, $uid);
    my $uid_info = {role => $role};

    my @fields = qw/createtime email/;
    if ($role eq 'manager') {
        push @fields, 'domain_login';
    }
    my $uid_row = get_user_data($uid, \@fields);
    $uid_info->{account_age} = time - $uid_row->{createtime}; 
    $uid_info->{email} = $uid_row->{email};

    if ($role eq 'client') {
        $uid_info->{client_id} = rbac_get_client_clientid_by_uid($uid) or die "failed to get client client_id by uid $uid";
        merge_client_info($uid_info, $uid);
    } elsif ($role eq 'agency') {
        $uid_info->{client_id} = rbac_get_agency_clientid_by_uid( $uid) or die "failed to get agency client_id by uid $uid";
        merge_agency_info($uid_info, $uid, $rbac);
    } elsif ($role eq 'manager') {
        $uid_info->{domain_login} = $uid_row->{domain_login};
    }

    return export_result(200, $uid_info, $param{format});
}

sub export_result($$$)
{
    my ($code, $ret, $format) = @_;
    if ($format && $format eq 'xml') {
        my $document = XML::LibXML::Document->new('1.0', 'UTF-8');
        my $root = XML::LibXML::Element->new('user');
        $document->setDocumentElement($root);
        for my $key (keys %$ret) {
            $root->appendTextChild($key, $ret->{$key});
        }

        return {text => $document->toString(), code => $code, content_type => 'text/xml; charset=utf-8'};
    } else {
        return {json => $ret, code => $code}; 
    }
}

sub merge_client_info
{
    my $uid_info = shift;
    my $uid = shift;

    my $oids = get_orderids(ClientID => $uid_info->{client_id});
    my $currency = get_client_currencies($uid_info->{client_id})->{work_currency};
    if (@$oids) {
        my $orders_info = Stat::OrderStatDay::get_orders_info($uid_info->{client_id}, $oids, $currency);
        hash_merge $uid_info, hash_cut($orders_info, qw/first_show_date avg_1month avg_3month/);
    } else {
        $uid_info->{avg_1month} = $uid_info->{avg_3month} = 0;
    }
    $uid_info->{is_easy} = 0; # for backward compatibility
}

sub merge_agency_info
{
    my $uid_info = shift;
    my $uid = shift;
    my $rbac = shift;

    $uid_info->{rep_type} = rbac_get_agency_rep_type($uid);
    my $agency_chief_uid = rbac_get_chief_rep_of_agency_rep($uid);
    $uid_info->{chief_agency_login} = get_login(uid => $agency_chief_uid);
    if (my $manager_of_agency_uid = rbac_get_manager_of_agency($rbac, $uid, 'text')) {
        hash_merge $uid_info, get_one_line_sql(PPC(uid => $manager_of_agency_uid), 
            "select fio as manager_fio, login as manager_login, email as manager_email  
            from users 
            where uid = ?", 
            $manager_of_agency_uid
        );
    }
}

1;
