package Export::Widget;

=pod

    $Id$
    экспорт для виджета в JSON
    DIRECT-7105

    отдается JSON вида:
    {
    sum_rest: 123123.04,
    active_camps_num: 3,
    overdraft: {
        debt: 12331.34, // сколько клиент уже должен
        overdraft_rest: 412.32, // размер доверительного платежа, который может внести клиент
        pay_date: '2010-05-30', // дата платежа
        }
    camps_list: [
           {cid: 1231, name: 'тестовая кампания'},
           {cid: 1231, name: 'тестовая кампания'},
           {cid: 1231, name: 'тестовая кампания'},
    ],
    camps_info: {
        1231: {
           cid: 1231,
           name: 'asdfasdf',
           status: 'идут показы',
           clicks_today: 123,
           sum_rest: 42.2134321,
        }
    }

=cut

use Direct::Modern;

use Plack::Builder;
use Plack::UTF8Request;

use Yandex::I18n;
use Settings;
use Yandex::Blackbox;
use Yandex::HashUtils qw/hash_merge/;
use Yandex::DBTools;
use TextTools;
use HttpTools;
use Direct::ResponseHelper;
use RBACElementary;
use RBAC2::Extended;
use Client;
use Campaign;
use Campaign::Types;
use Currency::Format;
use Currencies;
use Stat::OrderStatDay;

use JSON;
use XML::LibXML qw//;
use List::Util qw/sum/;
use List::MoreUtils qw/any/;

sub handler {
    my $r = shift;

    Yandex::I18n::init_i18n($Yandex::I18n::DEFAULT_LANG);

    my $form = multiform2directform($r->parameters, undef_empty_str => 1);
    my $format = $form->{format} || 'json';

    # если существует стоп-файл - ничего не делаем, отдыхаем
    if (-f $Settings::WEB_STOP_FLAG) {
        return export_error($r, 200, "Internal server error", $format);
    }

    if ($format ne 'xml') {
        my $yandexuid = $r->cookies->{yandexuid};
        if (!defined $form->{yandexuid} || $form->{yandexuid} ne $yandexuid) {
            return export_error($r, 200, "Invalid session", $format);
        }
    }

    # получем куки с сессией
    my $session_id = $r->cookies->{Session_id} || return export_error($r, 200, "No session cookie", $format);

    # проверка авторизации в ЧЯ
    my $bb_res = eval { bb_sessionid($session_id, $r->address, 'direct.yandex.ru'
                              , [$BB_SUID_DIRECT]) };
    return export_error($r, 200, "Internal server error", $format) if $@;
    return export_error($r, 200, "Invalid session", $format) if !$bb_res->{valid};
    my $uid = $bb_res->{uid};

    # делаем выборку из базы только если человек заходил в Директ
    my $result = {};
    if (!$bb_res->{dbfield}->{$BB_SUID_DIRECT}) {
        $result->{no_campaigns} = 1;
    } else {
        hash_merge $result, _get_widget_info(
                                $uid,
                                cids => [$form->{cid} ? split(/\D+/, $form->{cid}) : ()],
                                bar => $form->{bar},
                            );
    }

    # отвечаем
    return _format_output($result, $format, $r);
}

#-------------------------------------------------------------------------------
sub export_error {
    my ($r, $code, $text, $format) = @_;
    my $response = _format_output({error => $text}, $format, $r);
    $response->[0] = $code;
    return $response;
}

#-------------------------------------------------------------------------------
sub _format_output {
    my ($output, $format, $r) = @_;
    if ($format eq 'xml') {
        my $document = XML::LibXML::Document->new('1.0', 'UTF-8');
        my $root = XML::LibXML::Element->new('root');
        if (my $error = $output->{error}) {
            $root->appendTextChild('error', $error);
        } else {
            foreach my $toplevel_attr (qw/sum_rest active_camps_num no_campaigns agency currency currency_code/) {
                if (defined $output->{$toplevel_attr}) {
                    $root->appendTextChild($toplevel_attr, $output->{$toplevel_attr});
                }
            }
            if ($output->{camps_list}) {
                my $camps_list_node = XML::LibXML::Element->new('camps_list');
                foreach my $camp (@{ $output->{camps_list} }) {
                    my $camp_node = XML::LibXML::Element->new('camp');
                    foreach my $attr (keys %$camp) {
                        $camp_node->setAttribute($attr,  $camp->{$attr});
                    }
                    $camps_list_node->appendChild($camp_node);
                }
                $root->appendChild($camps_list_node);
            }
            if ($output->{camps_info}) {
                my $camps_info_node = XML::LibXML::Element->new('camps_info');
                foreach my $camp (@{ $output->{camps_info} }) {
                    my $camp_node = XML::LibXML::Element->new('camp');
                    foreach my $attr (keys %$camp) {
                        $camp_node->setAttribute($attr,  $camp->{$attr});
                    }
                    $camps_info_node->appendChild($camp_node);
                }
                $root->appendChild($camps_info_node);
            }
            if ($output->{overdraft}) {
                my $overdraft_node = XML::LibXML::Element->new('overdraft');
                foreach my $attr (keys %{ $output->{overdraft} }) {
                    $overdraft_node->setAttribute($attr,  $output->{overdraft}->{$attr});
                }
                $root->appendChild($overdraft_node);
            }
        }
        $document->setDocumentElement($root);
        respond_text($r, $document->toString(), "text/xml; charset=utf-8");
    } elsif ($format eq 'json') {
        # TODO respond_json. Сначала понять, что делать с charset=utf-8 -- напр., устанавливать всегда?
        respond_text($r, to_json($output), "application/x-javascript; charset=utf-8");
    } else {
        respond_text($r, 'Unknown format', "text/plain; charset=utf-8");
    }
}

#-------------------------------------------------------------------------------
sub _get_widget_info {
    my ($uid, %O) = @_;
    my $all_campaigns = get_all_sql(PPC(uid => $uid), ["
            SELECT c.cid, c.name, c.OrderID, c.archived, c.type as mediaType,
                   c.sum + IF(c.wallet_cid, wc.sum, 0) as sum,
                   c.sum_spent as sum_spent,
                   c.sum - c.sum_spent as total,
                   IF(c.wallet_cid, wc.sum - wc.sum_spent, 0) as wallet_total,
                   IF(c.wallet_cid, wc.sum_to_pay, c.sum_to_pay) as sum_to_pay,
                   c.wallet_cid,
                   DATE_FORMAT(c.start_time, '%Y%m%d000000') start_time,
                   c.statusShow, c.statusActive, c.statusBsSynced, c.statusModerate,
                   c.timeTarget, c.timezone_id,
                   o.statusPostModerate, o.stopTime,
                   IFNULL(cl.work_currency, 'YND_FIXED') as currency,
                   u.ClientID,
                   c.wallet_cid
            FROM users u
            JOIN users u2 ON u2.ClientID = u.ClientID
            LEFT JOIN clients cl ON u.ClientID = cl.ClientID
            JOIN campaigns c ON c.uid = u2.uid AND IFNULL(cl.work_currency, 'YND_FIXED') = IFNULL(c.currency, 'YND_FIXED')
            LEFT JOIN camp_options o ON c.cid = o.cid
            LEFT JOIN campaigns wc on wc.cid = c.wallet_cid",
            where => {'u.uid' => $uid,
                'c.type' => get_camp_kind_types('with_currency'),
                'c.statusEmpty' => 'No',
                'u.ClientID__gt' => 0
              }]);

    my $result = {};
    if (@$all_campaigns) {
        my $client_id = $all_campaigns->[0]->{ClientID};
        my $client_nds = get_client_NDS($client_id);
        my $client_discount = get_client_discount($client_id);
        campaign_remove_nds_and_add_bonus($_, client_nds => $client_nds, client_discount => $client_discount) for @$all_campaigns;

        my @campaigns_to_show = grep {$_->{archived} eq 'No' && $_->{sum} > $Currencies::EPSILON && $_->{mediaType} eq 'text'} @$all_campaigns;
        my $sum_rest = sum grep {$_ > $Currencies::EPSILON} map {$_->{total}} @campaigns_to_show;
        my %all_wallet_cids = map {$_->{wallet_cid} => 1} grep {$_->{wallet_cid} > 0} @campaigns_to_show;

        if (%all_wallet_cids) {
            my %wallets = map {$_->{cid} => $_} grep {$_->{mediaType} eq 'wallet'} @$all_campaigns;
            for my $camp (@$all_campaigns) {
                if ($camp->{total} < -$Currencies::EPSILON && exists($wallets{$camp->{wallet_cid}})) {
                    $wallets{$camp->{wallet_cid}}->{total} += $camp->{total};
                }
            }
            $sum_rest += sum map {$wallets{$_}->{total}} keys %all_wallet_cids;
        }

        my $active_camps_num = scalar grep {
            ( $_->{sum} - $_->{sum_spent} > $Currencies::EPSILON )
                && Campaign::get_camp_status_info($_)->{timetarget}
        } @campaigns_to_show;

        my $client_currencies = get_client_currencies($client_id);
        my $overdraft_info = get_overdraft_info($client_id, client_discount => $client_discount, client_nds => $client_nds, client_currencies => $client_currencies);
        my $overdraft = {
            pay_date       => $overdraft_info->{nextPayDateText},
            debt           => round2s($overdraft_info->{debt}),
            overdraft_rest => round2s($overdraft_info->{overdraft_rest}),
        };

        my $camps_list  = [
            sort { $a->{name} cmp $b->{name} }
            map { { cid => $_->{cid}, name => $_->{name} } }
            @campaigns_to_show
            ];

        my %camps_by_id = map { $_->{cid} => $_ } @campaigns_to_show;

        # учитываем минусы с кампаний под счетом на кампании с ОС
        for my $row (@campaigns_to_show) {
            if ($row->{wallet_cid} > 0) {
                $row->{wallet_total} += sum 0,
                                        map {$_->{total}}
                                        grep {$_->{wallet_cid} == $row->{wallet_cid}
                                              && $_->{cid} != $row->{cid}
                                              && $_->{total} < 0
                                             }
                                        @$all_campaigns;
            }
        }

        my $camps_info = [
            map { _get_campaign_widget_info($camps_by_id{$_}) }
            grep {$camps_by_id{$_}}
            @{$O{cids} || []}
            ];

        hash_merge $result, {
            sum_rest         => Campaign::round_campaign_sum_field($sum_rest),
            active_camps_num => $active_camps_num,
            camps_list       => $camps_list,
            camps_info       => $camps_info,
            currency         => iget(format_currency($all_campaigns->[0]->{currency})),
            currency_code    => $all_campaigns->[0]->{currency},
        };
        if (any { $overdraft->{$_} && $overdraft->{$_} > 0 } qw/debt overdraft_rest/) {
            $result->{overdraft} = $overdraft;
        }
    } else {
        $result->{no_campaigns} = 1;
        if ($O{bar}) {
            my $rbac = RBAC2::Extended->get_singleton($uid);
            if (rbac_who_is($rbac, $uid) eq 'agency') {
                $result->{agency} = 1;
            }
        }
    }
    return $result;
}

#-------------------------------------------------------------------------------
sub _get_campaign_widget_info {
    my $campaign_data = shift;

    my $clicks_today = Stat::OrderStatDay::order_clicks_today($campaign_data->{OrderID});
    my $status_info = Campaign::get_camp_status_info($campaign_data);
    my $status;
    if ( $status_info->{timetarget} ) {
        if ( $status_info->{timetarget} eq iget('Идут показы') ) {
            $status = 'Идут показы';
        } else {
            $status = 'Отключена';
        }
    } elsif ( $status_info->{money_finished} ) {
        $status = 'Нет средств';
    } elsif ( $status_info->{stopped} ) {
        $status = 'Остановлена';
    } elsif ( $status_info->{sum_to_pay} ) {
        $status = 'Ждёт оплаты';
    } elsif ( $status_info->{moderate} ) {
        my %moderate_status = (
            New   => iget("Черновик"),
            Yes   => iget("Принята"),
            No    => iget("Отклонена"),
            Ready => iget("На модерации"),
            Sent  => iget("На модерации"),
        );
        $status = $moderate_status{ $status_info->{moderate} };
    }

    return {
        cid          => $campaign_data->{cid},
        clicks_today => int($clicks_today),
        name         => $campaign_data->{name},
        status       => $status,
        sum_rest     => Campaign::round_campaign_sum_field($campaign_data->{total} + $campaign_data->{wallet_total}),
    };
}

1;
