package FakeAdminTools;

# $Id$

=pod

    Модуль для работы с обновлениями данных по кампаниям/баннерам на тестовых средах

=cut

use strict;
use warnings;
use utf8;

use Settings;
use Moderate::Settings;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::Validate qw/is_valid_int is_valid_float/;
use Yandex::TimeCommon;
use Yandex::Log;
use Yandex::Balance;
use Yandex::I18n;
use Yandex::Shell qw/yash_system yash_qx yash_quote/;
use Yandex::ListUtils qw/xisect chunks xminus/;
use Yandex::HTTP qw/http_fetch/;
use Yandex::TVM2;

use User;
use Client;
use Client::CustomOptions;
use Campaign;
use Fake;
use Stat::OrderStatDay;
use APIUnits;
use Primitives;
use PrimitivesIds;
use Currency::Rate;
use Yandex::HashUtils;
use RBAC2::Extended;
use RBACElementary;
use RBACDirect;
use API::Errors;
use GeoTools ();
use ShardingTools;
use EnvTools;
use Stat::Const;
use SOAP::Lite;

use List::MoreUtils qw/uniq none all any/;
use JSON;
use Models::Campaign;
use Models::AdGroup;
use Models::CampaignOperations;
use API::ClientOptions;

use Direct::Model::Campaign;
use Direct::Model::AdGroupDynamic;
use Direct::Model::AdGroupDynamic::Manager;
use Direct::Model::BannerDynamic;
use Direct::Model::BannerDynamic::Manager;
use Direct::Model::DynamicCondition;
use Direct::Model::DynamicCondition::Manager;

use Direct::Validation::AdGroupsDynamic qw/validate_add_dynamic_adgroups/;
use Direct::Validation::BannersDynamic qw/validate_add_dynamic_banners/;
use Direct::Validation::DynamicConditions qw/validate_dynamic_conditions_for_adgroup/;

use Direct::AdGroups2;
use Direct::Bids::BidRelevanceMatch;
use Direct::DynamicConditions;
use Direct::Banners::Dynamic;

use Client::ConvertToRealMoney;
use Client::ConvertToRealMoneyTasks;

our @fields = qw/
    OrderID
    type
    sum
    sum_units
    sum_spent
    sum_to_pay
    sum_last
    sum_spent_units
    statusModerate
    statusShow
    statusActive
    statusBsSynced
    archived
    shows
    clicks
    lastShowTime
    finish_time
    statusMail
    paid_by_certificate
    statusNoPay
    currency
    currencyConverted
    ProductID
    LastChange
    AgencyID
    AgencyUID
    ContextLimit
    ContextPriceCoef
    strategy_name
    strategy_data
/;

our @camp_opt_fields = qw/
    statusPostModerate
    day_budget_daily_change_count
    day_budget_stop_time
    day_budget_last_change
    day_budget_notification_status
    sms_time
    sms_flags
    auto_optimize_request
    mediaplan_status
    manual_autobudget_sum
    camp_description
    strategy
/;

our @set_fields = qw/
    sms_flags
/;

our @banner_fields = qw/
    BannerID
    statusModerate
    statusPostModerate
    vcard_id
    phoneflag
    sitelinks_set_id
    statusSitelinksModerate
    statusShow
    statusActive
    statusBsSynced
    type
    LastChange
    flags
/;

our @phrases_fields = qw/
    group_name
    geo
    PriorityID
    statusModerate
    LastChange
    statusBsSynced
    statusAutobudgetShow
    forecastDate
    statusShowsForecast
    statusPostModerate
/;

our @group_params_fields = qw/
    has_phraseid_href
/;

our @image_fields = qw/
    statusModerate
    image_id
    PriorityID
    BannerID
/;

our @imagead_fields = qw/
image_id
statusModerate
/;

our @creative_fields = qw/
banner_creative_id
statusModerate
/;

our @bids_fields = qw/
    id
    pid
    PhraseID 
    phrase
    statusModerate
    statusBsSynced
    place
    optimizeTry
    autobudgetPriority
    showsForecast
/;

our @display_href_fields = qw/
    statusModerate
/;

# поля из clients_custom_options
my @clients_custom_options_fields = qw/
    disallow_money_transfer
/;
my %clients_custom_options_fields = map { $_ => 1 } @clients_custom_options_fields;

my @clients_api_special_options_fields = qw/
    image_pool_limit
    advq_report_queue_priority
    api_reports_limit
/;
my %clients_api_special_options_fields = map { $_ => 1 } @clients_api_special_options_fields;

my %clients_force_currency_convert_fields = map { +"force_currency_convert_" . $_ => 1 } qw(
    accepted_at
);

=head2 update_retargeting_goals (ConditionID)

    Обновляет is_accessible в ppc.retargeting_goals по данным из метрики

=cut

sub update_retargeting_goals($) {
    my $condition_id = shift;

    my $rbac = RBAC2::Extended->get_singleton(1);

    # Т.к. метрика отдает цели только по ClientID 
    # получаем его по condition_id
    my $client_id = get_one_field_sql(PPC(ret_cond_id => $condition_id), [
        "SELECT ClientID from retargeting_conditions",
        where => { ret_cond_id => $condition_id }
    ]);

    # получаем uid по ClientID
    my $client_uids = rbac_get_client_uids_by_clientid($client_id);
    my $actual_goals =
    {
        map {$_ => 1}
        @{Retargeting::get_metrika_goal_ids_by_uid($client_id)}
    };
    
    my $current_goals = get_all_sql(PPC(ret_cond_id => $condition_id), [
        "SELECT goal_id, is_accessible FROM retargeting_goals",
        where => { ret_cond_id => $condition_id }
    ]);

    my (@accessible, @inaccessible, %unchanged);
    foreach my $rg (@$current_goals) {
        my $id = $rg->{goal_id};
        my $accessible = (exists $actual_goals->{$id}) ? 1 : 0;
        if( $rg->{is_accessible} && !$accessible ) {
            push @inaccessible, $id;
        } elsif ( !$rg->{is_accessible} && $accessible ) {
            push @accessible, $id;
        } else {
            $unchanged{$id} = $rg->{'is_accessible'};
        }
    }

    Retargeting::update_condition_goals_accessibility(
        1, $condition_id, \@accessible) if @accessible;

    Retargeting::update_condition_goals_accessibility(
        0, $condition_id, \@inaccessible) if @inaccessible;

    return {
        ClientID => $client_id,
        ClientUIDs => $client_uids,
        GoalsBecameAccessible => \@accessible,
        GoalsBecameInaccessible => \@inaccessible,
        GoalsUnchanged => \%unchanged
    };
}

=head2 get_client_params(uid, \@fields)

    Выводит данные по клиенту
        для того чтобы получить значение параметра,
        нужно явно передать его название в параметре fields

=cut

sub get_client_params($$)
{
    my ($login, $fields) = @_;
    
    my $uid = get_uid_by_login($login);
    my $all_client_fields = ['non_resident_client'];
    foreach my $table (keys %Client::CLIENT_TABLES) {
        foreach my $field (@{$Client::CLIENT_TABLES{$table}{fields}}) {
            push @$all_client_fields, $field;
        }
    }
    unless (@$fields){
        push @$fields,
             @{$User::USER_TABLES{users}{fields}},
             @{$User::USER_TABLES{users_options}{fields}},
             @{$User::USER_TABLES{users_api_options}{fields}},
             @clients_custom_options_fields,
             @clients_api_special_options_fields,
             (keys %clients_force_currency_convert_fields),
             @$all_client_fields,
             keys %Client::CLIENT_LIMIT_FIELDS,
             'API_units',
             'AllowEditCampaigns',
             'non_resident_client',
    } else {
        push @$fields, 'ClientID';
    }
    my %other_fields_methods = (API_units => \&get_API_units,
                                AllowEditCampaigns => \&is_super_subclient,
                                );

    my @users_custom_options = grep { $clients_custom_options_fields{$_} } @$fields;
    my @users_api_custom_options = grep { $clients_api_special_options_fields{$_} } @$fields;
    my @force_currency_convert_fields = grep { $clients_force_currency_convert_fields{$_} } @$fields;

    $fields = [ grep { !$clients_custom_options_fields{$_} } @$fields ];

    my $user_data = get_user_data($uid, $fields);

    $user_data->{uid} = $uid;
    my @client_fields = @{xisect($fields, $all_client_fields)};
    my @client_limit_fields = grep {exists($Client::CLIENT_LIMIT_FIELDS{$_})} @$fields;
    my %client_api_options_fields = ('api_enabled' => 1);
    my @client_api_options_fields = grep {exists($client_api_options_fields{$_})} @$fields;

    if (scalar @client_api_options_fields) {
        my $client_options = API::ClientOptions::get($user_data->{ClientID}, \@client_api_options_fields);
        hash_merge $user_data, $client_options;
    }

    my ($client_data, $client_limit_data);
    if (scalar @client_fields) {
        my %rename_fields = ('non_resident_client' => 'non_resident');
        $client_data = get_client_data($user_data->{ClientID},[map {$rename_fields{$_} || $_} @client_fields]);
        while (my ($resp_field, $get_field) = each %rename_fields) {
            next unless exists $client_data->{$get_field};
            $client_data->{$resp_field} = delete $client_data->{$get_field};
        }
        my @boolean_fields = qw(can_use_day_budget non_resident_client);
        for my $field (@boolean_fields) {
            next unless exists $client_data->{$field};
            $client_data->{$field} = ($client_data->{$field} // 0) == 0 ? 'No':'Yes';
        }
    }
    if (scalar @client_limit_fields) {
        $client_limit_data = get_client_limits($user_data->{ClientID});
    }
    my %limit_data = map {$_ => $client_limit_data->{$_}} @client_limit_fields;
    hash_merge $user_data, $client_data, \%limit_data;

    my @other_fields = grep {exists($other_fields_methods{$_})} @$fields;
    for my $field (@other_fields){
        $user_data->{$field} = $other_fields_methods{$field}->($uid);
    }
    hash_copy $user_data, get_user_custom_options($user_data->{ClientID}), @users_custom_options;
    hash_copy $user_data, query_all_api_special_options($user_data->{ClientID}), @users_api_custom_options;

    if (@force_currency_convert_fields) {
        my $fields_sql = join(',', map { s/^force_currency_convert_//; sql_quote_identifier($_) } @force_currency_convert_fields);
        my $fcc_data = get_one_line_sql(PPC(ClientID => $user_data->{ClientID}), ["SELECT $fields_sql FROM force_currency_convert", WHERE => {ClientID => SHARD_IDS}]);
        for my $fcc_key (keys %$fcc_data) {
            $user_data->{"force_currency_convert_$fcc_key"} = $fcc_data->{$fcc_key};
        }
    }

    return $user_data;
}

=head2 make_subclient($uid, $is_super_subclient)

=cut

sub make_subclient($$){

    my ($client_uid, $is_super_subclient) = @_;
    my $client_client_id = get_clientid(uid => $client_uid);
    my $rbac = RBAC2::Extended->get_singleton(1);
    my $agency_data = get_agency_clients_relations_data([$client_client_id]);
    return unless @$agency_data;
    my $agency_client_id = shift(@$agency_data)->{agency_client_id};
    return unless $agency_client_id;
    if ($is_super_subclient == 1) {
        rbac_make_super_subclient($rbac, $agency_client_id, $client_uid);
    } elsif ($is_super_subclient == 0) {
        rbac_make_simple_subclient($rbac, $agency_client_id, $client_uid);
    } else {return;}
    return 1;
}

=head2 is_super_subclient($uid)

=cut

sub is_super_subclient($){

    my $client_uid = shift;
    my $client_client_id = get_clientid(uid => $client_uid);
    my $rbac = RBAC2::Extended->get_singleton(1);
    my $agency_data = get_agency_clients_relations_data([$client_client_id]);
    return unless scalar @$agency_data;
    my $agency_client_id = shift(@$agency_data)->{agency_client_id};
    return unless $agency_client_id;
    my $is_supersubclient = rbac_is_supersubclient($rbac, $agency_client_id, $client_client_id);
    return $is_supersubclient || 0;
}



=head2 get_user_shard($login)

    Возвращает шард клиента

=cut

sub get_user_shard($$){

    my ($login, $ClientID) = @_;
    $ClientID ||= get_clientid(login => $login);
    return get_error_object('BadLogin') unless $ClientID;
        
    my $shard = get_shard(ClientID => $ClientID);
    return $shard;
}

=head2 reshard_user($login, $shard)

    Переносит клиента в указанный шард

=cut

sub reshard_user($$$){
    my ($login, $ClientID, $shard) = @_;
    my $result;
    $ClientID ||= get_clientid(login => $login);
    return get_error_object('BadLogin') unless $ClientID;

    my $old_shard = get_shard(ClientID => $ClientID);
    return get_error_object("BadRequest", "Пользователь $login уже в шарде $shard")
        if $old_shard == $shard;
    my @available_shards = ppc_shards();
    return get_error_object("BadRequest", "Шарда $shard у нас нет")
        if none { $_ == $shard } @available_shards;

    my $reshard_sub = sub {
        eval {
            yash_system("$Settings::ROOT/protected/ppcReSharder.pl", '--client-id' => $ClientID);
            1;
        };
    };

    my $max_tries = 2;
    for my $try (1..$max_tries) {
        my $active_tasks_cnt = get_one_field_sql(PPCDICT, 
                                                 ["SELECT count(*) from reshard_queue", 
                                                  WHERE => {status__not_in => ['done', 'error'], ClientID => $ClientID}
                                                 ]);
        last unless $active_tasks_cnt;
        if ($try < $max_tries && is_beta()) {
            $reshard_sub->();
        } else {
            return get_error_object("BadRequest", "Перенос ClientID=$ClientID уже запланирован");
        }
    }

    my $id = do_insert_into_table(PPCDICT, "reshard_queue", {
                ClientID => $ClientID,
                status => 'new',
                old_shard => $old_shard,
                new_shard => $shard,
                wanted_start_time__dont_quote => 'now()',
                         });

    if (is_beta()) {
        # на бетах запускаем скрипт сами
        if (!$reshard_sub->()) {
            return get_error_object(500, "ppcReSharder.pl run failed: $@");
        }
    }

    # на остальных средах ждём, пока отработает фоновый процесс
    my $border_time = time + 600;
    while(time <= $border_time) {
        my $status = get_one_field_sql(PPCDICT, "SELECT status FROM reshard_queue WHERE id = ?", $id);
        return 1 if $status eq 'done';
        return 0 if $status eq 'error';
        last if is_beta();
    }
    return 0;
}


sub get_API_units($){
    my $uid = shift;
    my $uhost = new APIUnits({ scheme=>"API" });
    my $units_data = $uhost->check_or_init_user_units($uid); 
    return $units_data->{$uid}{units_rest}
}

=head2 save_client_params(uid, \%DATA)

    Обновляет данные по клиенту
        позволяет изменить любые поля из моделей User и Client
        в том числе и создать нового пользователя,
        если передать uid пользователя, которого нет в Директе
        обновляются только те параметры которые были явно указаны

=cut

sub save_client_params($$)
{
    my ($login, $data) = @_;
    
    my %user_custom_options = map { exists $data->{$_} ? ($_ => delete $data->{$_}) : () } @clients_custom_options_fields;
    my %user_api_custom_options = map { exists $data->{$_} ? ($_ => delete $data->{$_}) : () } @clients_api_special_options_fields;
    my %force_currency_convert_options = map { exists $data->{$_} ? ($_ => delete $data->{$_}) : () } keys %clients_force_currency_convert_fields;

    my $uid = get_uid_by_login($login);
    $data->{ClientID} ||= get_clientid(uid => $uid);
    my $ClientID = $data->{ClientID};

    my ($client_data, $client_limit_data);
    if (exists $data->{api_enabled}) {
        my $api_enabled = $data->{api_enabled} =~ /(Yes|No|Default)/ ? $data->{api_enabled} : 'Default';
        API::ClientOptions::add($data->{ClientID}, {api_enabled => $api_enabled});
    }

    my $all_client_fields = [];
    foreach my $table (keys %Client::CLIENT_TABLES) {
        foreach my $field (@{$Client::CLIENT_TABLES{$table}{fields}}) {
            push @$all_client_fields, $field;
        }
    }
    
    my @client_fields = @{xisect([keys %$data], $all_client_fields)};
    $client_data = hash_copy({}, $data, @client_fields);

    create_update_client({client_data =>$client_data}) if scalar keys %$client_data;
    
    my @client_limit_fields = grep {exists($Client::CLIENT_LIMIT_FIELDS{$_})} keys %$data;
    $client_limit_data = hash_copy({}, $data, @client_limit_fields);
    set_client_limits($data->{ClientID}, $client_limit_data) if scalar keys %$client_limit_data;
    create_update_user($uid, $data);
    if (exists $data->{API_units}){
        save_user_units($uid,$data->{API_units});
    }
    if (exists $data->{AllowEditCampaigns}){
        make_subclient($uid,$data->{AllowEditCampaigns});
    }
    
    if (%user_custom_options) {
        set_user_custom_options($data->{ClientID}, \%user_custom_options);
    }
    if (%user_api_custom_options) {
        for my $option (keys %user_api_custom_options) {
            set_special_user_option($data->{ClientID}, $option, $user_api_custom_options{$option})
        }
    }
    if (%force_currency_convert_options) {
        if ($force_currency_convert_options{force_currency_convert_accepted_at}) {
            do_insert_into_table(PPC(ClientID => $ClientID), 'force_currency_convert', {ClientID => $ClientID, accepted_at => $force_currency_convert_options{force_currency_convert_accepted_at}}, on_duplicate_key_update => 1, key => 'ClientID');
        } else {
            do_delete_from_table(PPC(ClientID => $ClientID), 'force_currency_convert', where => {ClientID => SHARD_IDS});
        }
    }
    return 1;
}

sub save_user_units($$){
    my ($uid, $balance) = @_;
    my $uhost = new APIUnits({ scheme=>"API" });
    my $units_data = $uhost->check_or_init_user_units($uid);
    my $units = $units_data->{$uid}{daily_units} - $balance;
    if ($units < 0) {
        $units = 0
    }
    do_update_table(PPC(uid => $uid), 'api_users_units_consumption', {units => $units, 'date__dont_quote' => 'NOW()'},
                    where => {uid => $uid, scheme => 'API'});
}

=head2 save_camp_data(cid, options, %DATA)

    Обновляет данные по кампании
    При указании $options->{only_specified} - обновляет только указанные в %DATA параметры
        , иначе обновляет все (не переданные считает обнулением)

=cut

sub save_camp_data
{
    my ($cid, $options, %DATA) = @_;

    my $shard = get_shard(cid => $cid);
    return 0 unless $shard;
    my $not_empty = get_one_field_sql(PPC(shard => $shard), ['select 1 from campaigns', where => {'cid' => $cid, 'statusEmpty' => 'no'}]);
    return 0 unless $not_empty;

    if ($DATA{wallet_cid}) {
        do_update_table(PPC(shard => $shard), 'wallet_campaigns'
                           , {onoff_date => $DATA{wallet_onoff_date}}
                           , where => {wallet_cid => $DATA{wallet_cid}}
                       ) if $DATA{wallet_onoff_date};

        delete $DATA{wallet_cid}; # включать счет через модификацию нельзя
    }

    for my $field (@set_fields) {
        $DATA{$field} = join ',', grep {defined} map {/^${field}_(.+)$/ ? $1 : undef} keys %DATA;
    }
    
    my (@fields_for_update, @camp_opt_fields_for_update);
    if ($options->{only_specified}) {
        @fields_for_update = grep {exists $DATA{$_}} @fields;
        @camp_opt_fields_for_update = grep {exists $DATA{$_}} @camp_opt_fields;
    } else {
        @fields_for_update = @fields;
        @camp_opt_fields_for_update = @camp_opt_fields
    }

    my $lastShowTime;
    if ($DATA{lastShowTimePlusDays} && $DATA{lastShowTimePlusDays} =~ /^[-+]?\d+$/) {
        $lastShowTime = "DATE_ADD(lastShowTime, INTERVAL " . sql_quote($DATA{lastShowTimePlusDays}) . " DAY)";
    }
    my $ClientID = get_clientid(cid => $cid);
    if (is_valid_int($DATA{OrderID}, 1)){
        save_shard(OrderID => $DATA{OrderID}, ClientID => $ClientID);
    }
    # если нужно что-то в кампании изменить
    if (scalar @fields_for_update) {
        my %params = map {$_ => $DATA{$_}} @fields_for_update;

        $params{lastShowTime__dont_quote} = $lastShowTime if $lastShowTime;
        do_update_table(PPC(shard => $shard), 'campaigns'
                , \%params
                , where => {cid => $cid});
    }
    do_update_table(PPC(shard => $shard), 'camp_options'
        , {map { $_ => $DATA{$_} } @camp_opt_fields_for_update}
        , where => {cid => $cid});

    if (defined $DATA{wallet_total_sum}) {
        do_update_table(PPC(shard => $shard), 'wallet_campaigns'
            , {total_sum => $DATA{wallet_total_sum}}
            , where => {wallet_cid => $cid});
    }

    if ($DATA{reset_optimizing}) {
        do_delete_from_table(PPC(shard => $shard), 'optimizing_campaign_requests'
                , where => {cid => $cid});
    }

    return 1;
}

=head2 add_fake_metrika_data_to_camp(cid)

    Добавляет фейковые данные в ТГО и смарт-кампании, необходимые для подключения конверсионных стратегий
    ТГО:
        - Средняя цена конверсии;
        - Средняя рентабельность инвестиций (ROI);
        - Недельный бюджет с настройкой "получать максимальную конверсию по цели"
    Смарт:
        - Оптимизация количества конверсий;
        - Средняя рентабельность инвестиций (ROI)

=cut

sub add_fake_metrika_data_to_camp
{
    my($cid, $OrderID, $type) = @_;
    
    my $is_metrika_goal_exists = get_one_field_sql(PPC(cid => $cid)
                                , ['select 1 from camp_metrika_goals'
                                , where => {'cid' => $cid, 'goal_id__is_not_null' => 1, 'goals_count__gt' => 0}
                                , "limit 1"]);
    if (!$is_metrika_goal_exists) {
        do_insert_into_table(PPC(cid => $cid)
                            , 'camp_metrika_goals'
                            , {cid => $cid, stat_date => today(), goal_id => 15, goals_count => 10});
    }

    if ($type eq 'performance') {
        my $is_info_in_campaigns_performance = get_one_field_sql(PPC(cid => $cid)
                                , ['select 1 from campaigns_performance'
                                , where => {'cid' => $cid}]);

        if (!$is_info_in_campaigns_performance) {
            do_insert_into_table(PPC(cid => $cid)
                                , 'campaigns_performance'
                                , {cid => $cid});
        }
    }
    return 1;
}

=head2 get_camp_data(cid)

    Получает атрибуты кампании

=cut

sub get_camp_data($)
{
    my $cid = shift;
    my $vars = get_one_line_sql(PPC(cid => $cid), ["
                    SELECT c.cid, c.OrderID, u.login, c.uid
                         ".join("", map {", c.$_"} @fields)."
                         ".join("", map {", co.$_"} @camp_opt_fields)."
                         , r.request_id
                         , c.AgencyID
                         , c.AgencyUID
                         , c.wallet_cid
                         , IF(ca.cid IS NOT NULL, 1, 0) AS camp_activization_queued
                         , c.sum_balance
                         , wc.is_sum_aggregated
                         , cms.chips_spent AS cms_chips_spent
                         , cms.chips_cost AS cms_chips_cost
                      FROM campaigns c
                           JOIN users u on u.uid = c.uid
                           LEFT JOIN optimizing_campaign_requests r on r.cid = c.cid
                           LEFT JOIN camp_options co ON c.cid = co.cid
                           LEFT JOIN camp_activization ca ON c.cid = ca.cid
                           LEFT JOIN wallet_campaigns wc ON wc.wallet_cid = IF(c.type = 'wallet', c.cid, c.wallet_cid)
                           LEFT JOIN campaigns_multicurrency_sums cms ON cms.cid = c.cid
                  " , where => {'c.cid' => $cid, 'c.statusEmpty' => 'no'}]);
    if ($vars->{OrderID} && $vars->{OrderID} > 0) {
        $vars->{spent_today} = Stat::OrderStatDay::get_order_spent_today($vars->{OrderID}, with_nds => 1) || 0;
    }

    if (defined $vars->{type} && 
        any { $vars->{type} eq $_ } qw/text wallet/) {
        my $currency = $vars->{currency} || 'YND_FIXED';
        my $wallet = get_wallet_camp($vars->{uid}, $vars->{AgencyID}, $currency);
        if ($vars->{type} eq 'text') {
            $vars->{wallet_onoff_date} = $wallet->{onoff_date};
            $vars->{wallet_is_enabled} = $wallet->{is_enabled};
            $vars->{wallet_cid} = $wallet->{wallet_cid};
        }
        if ($vars->{type} eq 'wallet') {
            $vars->{wallet_total_sum} = $wallet->{total_sum};    
        }
    }
    $vars->{AgencyLogin} = get_login(uid => $vars->{AgencyUID}) if exists $vars->{cid};

    my $sum_from_balance;
    if ($vars->{type} eq 'mcb') {
        $sum_from_balance = $vars->{sum_units};
    } else {
        if ($vars->{is_sum_aggregated} // 'No' eq 'Yes') {
            $sum_from_balance = $vars->{sum_balance};
        } else {
            $sum_from_balance = $vars->{sum};
        }
    }
    $vars->{sum_from_balance} = $sum_from_balance;

    if ($vars->{type} ne 'wallet' && defined($vars->{cms_chips_spent})) {
        $vars->{has_chips_cost} = 1;
    } else {
        $vars->{has_chips_cost} = 0;
    }

    return $vars;
}


=head2 balance_notification(cid, %DATA)

    Эмулирует нотификацию о приходе денег от Баланса

=cut

sub balance_notification
{
    my ($cid, %DATA) = @_;

    my $camp = get_camp_info($cid, undef, short => 1);

    my $new_tid = ($camp->{balance_tid} // 0) + 1;

    my ($service_id, $sum, $sum_real_money);
    if ($DATA{mcb_campaign}) {
        $service_id = $Settings::SERVICEID{bayan};
        $sum = $DATA{sum_from_balance};
        $sum_real_money = convert_currency($sum, $camp->{currency}, 'RUB', with_nds => 1);
    } else {
        $service_id = $Settings::SERVICEID{direct};
        $sum = convert_currency($DATA{sum_from_balance}, $DATA{Currency} || $camp->{currency}, $camp->{currency}, with_nds => 1);
        $sum_real_money = convert_currency($DATA{sum_from_balance}, $camp->{currency}, ($camp->{currency} eq 'YND_FIXED' ? 'RUB' : $camp->{currency}), with_nds => 1);
    }

    my %notification_data = (
        ServiceID => $service_id,
        ServiceOrderID => $cid,
        Tid => $new_tid,
        ConsumeQty => $sum,
        ConsumeMoneyQty => $sum_real_money,
        ProductCurrency => $DATA{Currency} || $camp->{currency},
        Certificate => $DATA{balance_paid_by_certificate} ? 1 : 0,
    );
    # TODO: для сконвертированных без копирования надо уметь CompletionFixedMoneyQty и CompletionFixedQty
    if ($DATA{has_chips_cost}) {
        $notification_data{CompletionFixedMoneyQty} = $DATA{cms_chips_cost};
        $notification_data{CompletionFixedQty} = $DATA{cms_chips_spent};
    }

    my $total_sum_cur;
    if ($camp->{type} eq 'wallet' || $camp->{wallet_cid}) {
        my $wallet_cid = $camp->{type} eq 'wallet' ? $camp->{cid} : $camp->{wallet_cid};
        my $is_sum_aggregated = get_one_field_sql(PPC(cid => $wallet_cid), "select is_sum_aggregated FROM wallet_campaigns WHERE wallet_cid = ?", $wallet_cid);
        my $sum_balance_field_sql = $is_sum_aggregated eq 'Yes' ? 'sum_balance' : 'sum';

        $total_sum_cur = get_one_field_sql(PPC(cid => $cid), "select sum(`$sum_balance_field_sql`) from campaigns
                                                               where cid <> ? 
                                                                 and IFNULL(currency, 'YND_FIXED') = ?
                                                                 and (cid = ? OR wallet_cid = ?)
                                                                 and uid = ?",
                                                                     $cid, $camp->{currency}, $wallet_cid, $wallet_cid, $camp->{uid});
        $total_sum_cur += $camp->{currency} eq 'YND_FIXED' ? $notification_data{ConsumeQty} : $notification_data{ConsumeMoneyQty};

        if ($camp->{type} eq 'wallet') {
            my $total_balance_tid = get_one_field_sql(PPC(cid => $cid), "select total_balance_tid from wallet_campaigns where wallet_cid = ?", $wallet_cid) // 0;
            $notification_data{Tid} = $total_balance_tid+1 if $total_balance_tid >= $notification_data{Tid};

            $notification_data{TotalConsumeQty} = $total_sum_cur;
        }
    }

    if ($camp->{type} eq 'internal_autobudget') {
        $notification_data{ServiceID} = 67;
        # внутренняя реклама всегда оплачивается сертификатами
        $notification_data{Certificate} = 1;
    }

    my $ticket = eval { Yandex::TVM2::get_ticket($Settings::TVM2_APP_ID{intapi}) } or die "Cannot get ticket for $Settings::TVM2_APP_ID{intapi}: $@";
    my $content = http_fetch(POST => "$Settings::DIRECT_JAVA_INTAPI_URL/BalanceClient/NotifyOrder2"
                             , to_json([\%notification_data])
                             , timeout => 20
                             , headers => {'Content-type' => 'application/json', 'X-Ya-Service-Ticket' => $ticket }
                             );
    my @result = @{ decode_json($content) };
    if (!$result[0] && $camp->{wallet_cid} && defined $total_sum_cur) {
        # если нотификация успешно применена и он была по кампании под кошельком
        # меняем в БД общую сумму по кошельку и кампаниям под ним
        do_insert_into_table(PPC(cid => $camp->{wallet_cid}), 'wallet_campaigns', {wallet_cid => $camp->{wallet_cid},
                                                                                   total_sum => $total_sum_cur}, 
                                                              on_duplicate_key_update => 1, key => 'wallet_cid');

    }

    return @result;
}

=head2 save_group_data(pid, options, %DATA)

    Обновляет данные по группе
    При указании $options->{specified_only} - обновляет только указанные в %DATA параметры
        , иначе обновляет все (не переданные считает обнуление)

=cut

sub save_group_data
{
    my ($pid, $options, %DATA) = @_;
    my @phrases_fields_for_update = grep {!$options->{only_specified} || exists $DATA{$_} } @FakeAdminTools::phrases_fields;
    my %phrases_params = map {$_ => $DATA{$_}} @phrases_fields_for_update;
    if (!$phrases_params{LastChange}) {
        $phrases_params{LastChange__dont_quote} = 'LastChange';
    }

    my $shard = get_shard(pid => $pid);

    do_update_table(PPC(shard => $shard), 'phrases'
            , \%phrases_params
            , where => {pid => $pid});
    
    if (($DATA{has_phraseid_href} // '') eq 'Yes') {
        do_insert_into_table(PPC(shard => $shard), 'group_params', {
            pid => $pid, 
            has_phraseid_href => ($DATA{has_phraseid_href} // '') eq 'Yes' ? 1 : 0,
        }, on_duplicate_key_update => 1, key => ['pid']);
    } else {
        do_sql(PPC(shard => $shard), "UPDATE group_params SET has_phraseid_href = 0 WHERE pid = ?", $pid);
    }
    my $cid = get_cid(pid => $pid);
    Models::CampaignOperations::db_update_campaign_statusModerate($cid);
    return 1;
}

=head2 get_group_data(pid)

    Получает атрибуты группы

=cut

sub get_group_data($)
{
    my $pid = shift;
    my $data = get_one_line_sql(PPC(pid => $pid), "
                        SELECT p.pid, p.cid, u.login,
                            IF(gp.has_phraseid_href = 1, 'Yes', 'No') AS has_phraseid_href,
                            p.bid as group_master_bid,
                            amc.mobile_content_id
                            ".join("", map {", p.$_" } @phrases_fields)."
                          FROM phrases p
                                JOIN campaigns c ON c.cid = p.cid
                                JOIN users u on u.uid = c.uid
                                LEFT JOIN group_params gp ON p.pid = gp.pid
                                LEFT JOIN adgroups_mobile_content amc ON p.adgroup_type = 'mobile_content' AND amc.pid = p.pid
                         WHERE p.pid = ?
                        ", $pid);
    return $data;
}

=head2 save_banner_data(bid, options, %DATA)

    Обновляет данные по объявлению
    При указании $options->{specified_only} - обновляет только указанные в %DATA параметры
        , иначе обновляет все (не переданные считает обнуление)
=cut

sub save_banner_data
{
    my ($bid, $options, %DATA) = @_;

    # если нужно что-то в баннере изменить
    my @banner_fields_for_update = grep {!$options->{only_specified} || exists $DATA{$_} } @FakeAdminTools::banner_fields;
    my %banner_params = map {$_ => $DATA{$_}} @banner_fields_for_update;
    if (!$banner_params{LastChange}) {
        $banner_params{LastChange__dont_quote} = 'LastChange';
    }
    my $ClientID = get_clientid(bid => $bid);
    my $shard = get_shard(bid => $bid);
    do_update_table(PPC(shard => $shard), 'banners'
            , \%banner_params
            , where => {bid => $bid});
    
    do_update_table(PPC(shard => $shard), 'banner_images', 
        {
            date_added__dont_quote => 'date_added',

            map {
                $_ => $DATA{"image_$_"}
            } grep {
                !$options->{only_specified} || exists $DATA{"image_$_"}
            } @FakeAdminTools::image_fields
        },
        where => { bid => $bid }
    );

    my %image_ad_data = map { $_ => $DATA{"imagead_$_"} } grep { !$options->{only_specified} || exists $DATA{"imagead_$_"} } @FakeAdminTools::imagead_fields;
    if (%image_ad_data) {
        do_update_table(PPC(shard => $shard), 'images', \%image_ad_data, where => { bid => $bid });
    }

    my %creative_data = map { $_ => $DATA{"creative_$_"} } grep { !$options->{only_specified} || exists $DATA{"creative_$_"} } @FakeAdminTools::creative_fields;
    if (%creative_data) {
        do_update_table(PPC(shard => $shard), 'banners_performance', \%creative_data, where => { bid => $bid });
    }

    do_update_table(PPC(shard => $shard), 'banner_display_hrefs', 
        {
            bid__dont_quote => 'bid',

            map {
                $_ => $DATA{"display_href_$_"}
            } grep {
                !$options->{only_specified} || exists $DATA{"display_href_$_"}
            } @FakeAdminTools::display_href_fields
        },
        where => { bid => $bid }
    );
    my $cid = get_cid(bid => $bid);
    Models::CampaignOperations::db_update_campaign_statusModerate($cid);
    return 1;
}

=head2 get_banner_data(bid)

    Получает атрибуты объявления

=cut

sub get_banner_data($)
{
    my $bid = shift;
    my $data = get_one_line_sql(PPC(bid => $bid), "
                        SELECT b.bid, b.cid, u.login, b.pid
                             ".join("", map {", b.$_"} @banner_fields)."
                             ".join("", map {", bim.$_ as image_$_"} @image_fields)."
                             ".join("", map {", bdh.$_ as display_href_$_"} @display_href_fields)."
                             ".join("", map {", images.$_ as imagead_$_"} @imagead_fields)."
                             ".join("", map {", banners_performance.$_ as creative_$_"} @creative_fields)."
                          FROM banners b
                                JOIN campaigns c ON c.cid = b.cid
                                JOIN users u on u.uid = c.uid
                                LEFT JOIN banner_images bim on b.bid = bim.bid
                                LEFT JOIN banner_display_hrefs bdh on b.bid = bdh.bid
                                LEFT JOIN images on images.bid = b.bid
                                LEFT JOIN banners_performance on banners_performance.bid = b.bid
                         WHERE b.bid = ?
                        ", $bid);
    return $data;
}

=head2 get_phrases_data(bid)

    Получает атрибуты всех фраз объявления

=cut

sub get_phrases_data($;$)
{
    my ($id, $id_type) = @_;
    $id_type ||= 'pid';
    my %shard;
    if ($id_type eq 'pid') {
        %shard = (pid => [$id]);
    } elsif ($id_type eq 'id'){
        %shard = (shard => 'all');
    }
    my $where = {$id_type => $id};
    my $phrases = get_all_sql(PPC(%shard), [select => join(',', @bids_fields), from => 'bids', where => $where]);
    for my $ph (@{$phrases}){
        fake_up($ph->{context_stop_flag}, type => 'context_stop_flag', id => $ph->{PhraseID});
    }
    return $phrases;
}


=head2 save_phrases_data(phrases)

    Обновляет атрибуты всех фраз объявления

=cut

sub save_phrases_data($)
{
    my $phrases = shift;
    my $pid = (ref $phrases eq 'HASH') ? $phrases->{pid} : $phrases->[0]{pid};
    my $result = _create_structure($phrases);
    for my $id (keys %$result){
        _update_phrase($result->{$id}, $pid);
    }
    if (all { $result->{$_}{statusModerate} eq 'No' } keys %$result) {
        # если все фразы отклонены
        do_sql(PPC(pid => $pid), "UPDATE phrases
            SET statusModerate = 'No', statusPostModerate = IF(statusPostModerate = 'Yes', 'Rejected', 'No')
            WHERE pid =?", $pid);
    }
    return 1;
}

=head2 delete_phrases($pid, $ids)

    Удаление фраз($ids) из группы объявлений($pid)
    
        $pid - номер группы
        $ids - массив id фраз 

=cut

sub delete_phrases {
    
    my ($pid, $ids) = @_;

    return do_delete_from_table(PPC(pid => $pid), 'bids', where => {id => $ids});
}

=head2 _create_structure(phrases)

    Из данных пришедших с веб-интерфейса и из API создает универсальную структуру

=cut

sub _create_structure($)
{
    my $phrases = shift;
    my ($id, @list, %hash);
    if (ref $phrases eq 'HASH'){
        for my $key (keys %$phrases){
            if ($key =~ /^bid_(.+?)-(\d+)$/){
                my ($attr,$id) = ($1, $2);
                $hash{$id}->{id} ||= $id;
                $hash{$id}->{$attr} = $phrases->{$key};
            }
        }
    } elsif (ref $phrases eq 'ARRAY') {
        for my $phrase (@$phrases){
            $hash{$phrase->{id}} = $phrase;
        }
    }
    return \%hash;
}

=head2 _update_phrase(phrase)

    Проверяет правильность новых атрибутов, если все хорошо записывает в базу

=cut

sub _update_phrase($$) {
    my ($phrase, $pid) = @_;
    my $id = $phrase->{id};

    my $update_bids = {
        statusModerate => $phrase->{statusModerate} || 'New', 
        statusBsSynced => $phrase->{statusBsSynced} || 'Yes' 
    };
    if (defined $phrase->{place} && any { $phrase->{place} == $_ } values %PlacePrice::PLACES) {
        $update_bids->{place} = $phrase->{place};
    }
    for my $field (qw/PhraseID optimizeTry autobudgetPriority showsForecast/) {
        if (is_valid_int($phrase->{$field}, 0)) {
            $update_bids->{$field} = $phrase->{$field};
        } 
    }
    for my $field (qw/price price_context/) {
        if (is_valid_float($phrase->{$field}, 0)) {
            $update_bids->{$field} = $phrase->{$field};
        }
    }
    my %shard;
    if ($pid) {
      %shard = (pid => $pid);
    } else{
       %shard = (shard => 'all');
    }
    $update_bids->{warn} = defined $phrase->{warn} && $phrase->{warn} eq 'Yes' ? 'Yes' : 'No';
    if (defined $phrase->{phrase}){
        $update_bids->{phrase} = $phrase->{phrase};
    }

    do_update_table(PPC(%shard), 'bids', $update_bids, where => {id => $id});
    return 1;
}

=head2 get_relevance_matches(pid)

    Получить автотаргетинг группы

=cut

sub get_relevance_matches($) {
    my $pid = shift;

    my $relevance_matches = Direct::Bids::BidRelevanceMatch->get_by(adgroup_id => $pid, with_deleted => 1);
    return [ map { $_->to_template_hash  } @{$relevance_matches->items} ];
}

=head2 delete_relevance_matches (pid, [bid_ids])

    Удалить (отметить удаленным) автотаргетинг

=cut

sub delete_relevance_matches($$) {
    my ($pid, $bids_ids) = @_;

    my $adgroup = Direct::AdGroups2->get_by(adgroup_id => $pid, extended => 1)->items->[0];
    my $relevance_matches_for_delete = Direct::Bids::BidRelevanceMatch->get_by(bid_id => $bids_ids, pid => $pid);
    $_->adgroup($adgroup) foreach @{$relevance_matches_for_delete->items};

    $relevance_matches_for_delete->delete();

    return scalar(@{$relevance_matches_for_delete->items});
}

=head2 moderate_result_notification(cid, rbac)

    Эмулирует нотификацию о результате модерации для указанной кампании

=cut

sub moderate_result_notification
{
    my ($cid, $rbac) = @_;
    my $shard = get_shard(cid => $cid);

    yash_system("$Settings::ROOT/protected/ppcSendMailMaster.pl", "--shard-id", $shard, "--cid", $cid, "--once");
    yash_system("$Settings::ROOT/protected/ppcSendMailNew.pl", "--shard-id", $shard, "--cid", $cid);

    return 1;
}

=head2 fake_balance_notification_nds

    Имитация нотификации баланса о НДС для клиента

=cut

sub fake_balance_notification_nds
{
    my @clients_nds_data = @_;
    my @client_ids = uniq map {$_->[0]} @clients_nds_data;

    do_mass_insert_sql(PPC(ClientID => \@client_ids), 'INSERT INTO client_nds (ClientID, nds, date_from, date_to) VALUES %s ON DUPLICATE KEY UPDATE nds = VALUES(nds), date_to = VALUES(date_to), date_from = VALUES(date_from)', \@clients_nds_data);

    return 1;
}

=head2 sync_campaigns_sums_to_balance

    Синхронизирует суммы зачислено/истрачено на кампаниях с Балансом

    sync_campaigns_sums_to_balance(\@cids);

=cut

sub sync_campaigns_sums_to_balance {
    my ($cids) = @_;

    my $balance_info;
    for my $cids_chunk (chunks $cids, 50) {
        my $balance_info_chunk = balance_get_orders_info($cids_chunk);
        push @$balance_info, $balance_info_chunk;
    }
    if ($balance_info && @$balance_info) {
        my (%cid2sum, %cid2sum_spent, @currency_sums_to_insert, %wallet_totals, %wallet2cids, %wallet_total_chip_costs);
        my $camps_info = get_hashes_hash_sql(PPC(cid => $cids), ['SELECT cid, type, currencyConverted, IFNULL(currency, "YND_FIXED") AS currency FROM campaigns', WHERE => {cid => SHARD_IDS}]);
        my @orders = map {@$_} @$balance_info;
        for my $order (@orders) {
            next unless $order && ref($order) eq 'HASH';
            my $cid = $order->{ServiceOrderID};
            my $chips_cost = 0;
            if ($camps_info && $camps_info->{$cid} && $camps_info->{$cid}{currency} eq 'RUB' && $camps_info->{$cid}{currencyConverted} eq 'Yes') {
                $cid2sum{$cid} = $order->{ConsumeMoneyQty} // 0;
                $cid2sum_spent{$cid} = $order->{CompletionMoneyQty} // 0;
                if ($camps_info->{$cid}{type} ne 'wallet') {
                    $chips_cost = ($order->{CompletionFixedMoneyQty} || $order->{CompletionMoneyQty}) // 0;
                    my $chips_spent = $order->{CompletionFixedQty} // 0;
                    push @currency_sums_to_insert, [$cid, $order->{ConsumeMoneyQty} // 0, $chips_cost, $chips_spent, 0];
                } else {
                    push @currency_sums_to_insert, [$cid, $order->{ConsumeMoneyQty} // 0, 0, 0, 0];
                }
            } else {
                $cid2sum{$cid} = $order->{consume_qty} // 0;
                $cid2sum_spent{$cid} = $order->{completion_qty} // 0;
                push @currency_sums_to_insert, [$cid, $order->{ConsumeMoneyQty} // 0, 0, 0, 0];
            }

            if ($order->{GroupServiceOrderID}) {
                my $wallet_cid = $order->{GroupServiceOrderID};
                $wallet_totals{$wallet_cid} += $cid2sum{$cid};
                $wallet_total_chip_costs{$wallet_cid} += $chips_cost;
                push @{$wallet2cids{$wallet_cid}}, $cid;
            } else {
                push @{$wallet2cids{0}}, $cid;
            }
        }
        my @wallet_cids = keys %wallet_totals;
        for my $wallet_cid (@wallet_cids) {
            $wallet_totals{$wallet_cid} += $cid2sum{$wallet_cid};
            $cid2sum_spent{$wallet_cid} = 0;
        }
        my $wallet_sum_aggregated = get_hash_sql(PPC(cid => \@wallet_cids), ["
            SELECT wallet_cid, is_sum_aggregated
            FROM wallet_campaigns",
            WHERE => [
                'wallet_cid' => SHARD_IDS,
            ]
        ]);

        my $sum_case = sql_case('cid', \%cid2sum, default => 0);
        my $sum_spent_case = sql_case('cid', \%cid2sum_spent, default => 0);
        do_in_transaction {
            for my $wallet_cid (keys %wallet2cids) {
                my $cids_under_wallet = xminus $wallet2cids{$wallet_cid}, \@wallet_cids;
                next unless @$cids_under_wallet;

                my $sum_sql;
                my $sum_balance_sql;
                my $total_chips_cost;
                my $wallet_total = $wallet_totals{$wallet_cid} // 0;
                if (($wallet_sum_aggregated->{$wallet_cid} // 'No') eq 'Yes') {
                    $sum_sql = sql_case('cid', {$wallet_cid => $wallet_total}, default => 0);
                    $sum_balance_sql = $sum_case;
                    $total_chips_cost = $wallet_total_chip_costs{$wallet_cid} // 0;
                } else {
                    $sum_sql = $sum_case;
                    $sum_balance_sql = '0';
                    $total_chips_cost = 0;
                }
                push @$cids_under_wallet, $wallet_cid;
                do_update_table(PPC(cid => $cids_under_wallet), 'campaigns'
                    , {sum__dont_quote => $sum_sql, sum_spent__dont_quote => $sum_spent_case, sum_balance__dont_quote => $sum_balance_sql, balance_tid => 0}
                    , where => {cid => $cids_under_wallet}
                );
                if ($wallet_cid != 0) {
                    do_update_table(PPC(cid => $wallet_cid), 'wallet_campaigns'
                        , {total_sum => $wallet_total, total_chips_cost => $total_chips_cost}
                        , where => {wallet_cid => $wallet_cid}
                    );
                }
            }

            if (@currency_sums_to_insert) {
                foreach_shard cid => \@currency_sums_to_insert, by => sub {$_->[0]}, sub {
                    my ($shard, $currency_sums_to_insert_to_shard) = @_;
                    do_mass_insert_sql(PPC(shard => $shard), '
                        INSERT INTO campaigns_multicurrency_sums (cid, sum, chips_cost, chips_spent, balance_tid)
                        VALUES %s
                        ON DUPLICATE KEY UPDATE
                            sum = VALUES(sum)
                            , chips_cost = VALUES(chips_cost)
                            , chips_spent = VALUES(chips_spent)
                            , balance_tid = VALUES(balance_tid)'
                        , $currency_sums_to_insert_to_shard);
                };
            }
        }
    }
}

=head2 create_dynamic_adgroup

    принимает:
    cid
    uid
    domain
    geo
    dynamic ([{condition => [{},{},{}], price => , price_context => }])
    banner_bodies ([body1, body2,...])
    p_statusModerate
    p_statusPostModerate
    b_statusModerate
    b_statusPostModerate

    Примерный правильный порядок сохранения динамической группы:
    ppcdict.domains_dict
    ppc.domains
    ppc.phrases
    ppc.adgroups_dynamic
    ppc.banners
    ppc.dynamic_conditions
    ppc.bids_dynamic

=cut
sub create_dynamic_adgroup {
    my %data = @_;

    my $cid = $data{cid};
    my $uid = $data{uid};
    my $ClientID = get_clientid(cid => $cid);

    my $campaign = Models::Campaign::get_user_camp_gr($uid, $cid, {no_groups => 1, without_multipliers => 1});
    die "cannot find campaign $cid with uid $uid" if !defined $campaign;

    die 'no dynamic data' unless $data{dynamic} && @{$data{dynamic}};
    die 'no banners bodies' unless $data{banner_bodies} && @{$data{banner_bodies}};
    die 'need dynamic campaign' unless $campaign->{type} eq 'dynamic';

    $data{geo} //= '0';
    my $geoflag;
    if (GeoTools::validate_geo($data{geo})) {
        die "invalid geo: $data{geo}";
    } else {
        $data{geo} = GeoTools::refine_geoid($data{geo}, \$geoflag, {ClientID => $ClientID});
    }

    my $adgroup = Direct::Model::AdGroupDynamic->new(
        id => get_new_id('pid', ClientID => $ClientID),
        client_id => $ClientID,
        campaign_id => $cid,
        has_show_conditions => 0, banners_count => 0, dyn_conds_count => 0,
        adgroup_name => "test dynamic adgroup #" . int(rand(1000)),
        main_domain => $data{domain},
        geo => $data{geo},
        status_moderate => $data{p_statusModerate} // 'Yes',
        status_post_moderate => $data{p_statusPostModerate} // 'Yes',
    );

    my @banners;
    for my $body (@{ $data{banner_bodies} }) {
        push @banners, Direct::Model::BannerDynamic->new(
            id => get_new_id('bid', ClientID => $ClientID),
            client_id => $ClientID,
            campaign_id => $cid,
            adgroup_id => $adgroup->id,
            body => $body,
            status_show => 'Yes',
            status_moderate => $data{b_statusModerate} // 'Yes',
            status_post_moderate => $data{b_statusPostModerate} // 'Yes',
            geoflag => $geoflag,
        );
    }

    my @dyn_conds;
    my $rule_type = $adgroup->get_class_for_condition_rules();
    for my $target (@{ $data{dynamic} }) {
        push @dyn_conds, Direct::Model::DynamicCondition->new(
            id => get_new_id('dyn_id'),
            adgroup_id => $adgroup->id,
            condition_name => "test dynamic condition #" . int(rand(1000)),
            condition => [map { $rule_type->new(%$_) } @{$target->{condition}}],
            price => $target->{price} || 1.5,
            price_context => $target->{price_context} || 1.6,
            autobudget_priority => (1,3,5)[int(rand(3))],
            is_suspended => 0,
        );
    }

    # Валидация
    my $adgroups_vr = validate_add_dynamic_adgroups(
        [$adgroup],
        Direct::Model::Campaign->from_db_hash({
            adgroups_count => get_one_field_sql(PPC(cid => $cid), "SELECT count(pid) FROM phrases WHERE cid = ?", $cid),
            adgroups_limit => Client::get_client_limits($ClientID)->{banner_count_limit},
        }, \{}, with => 'AdGroupsCount'),
    );

    my $dyn_conds_vr = validate_dynamic_conditions_for_adgroup(\@dyn_conds, [], $campaign);
    $dyn_conds_vr->process_objects_descriptions(Direct::DynamicConditions->WEB_FIELD_NAMES);

    my $banners_vr = validate_add_dynamic_banners(\@banners, $adgroup);
    $banners_vr->process_objects_descriptions(Direct::Banners::Dynamic->WEB_FIELD_NAMES);

    my @errors = (@{$adgroups_vr->get_error_descriptions}, @{$dyn_conds_vr->get_error_descriptions}, @{$banners_vr->get_error_descriptions});
    die join("\n", @errors) if @errors;

    # Запись в базу
    do_in_transaction {
        Direct::Model::AdGroupDynamic::Manager->new(items => [$adgroup])->create();
        Direct::Model::BannerDynamic::Manager->new(items => \@banners)->create();
        Direct::Model::DynamicCondition::Manager->new(items => \@dyn_conds)->create();
    };

    return $adgroup->id;
}

sub create_adgroups {
    
    my ($cid, $groups) = @_;

    my $campaign = {cid => $cid, strategy => {name => '', search => {}, net => {}}};

    foreach my $group (@$groups) {
        $group->{banners} = [];
        Models::AdGroup::save_group($campaign, $group, 
            ignore_tags => 1,
            pass_phrases => 1,
            ClientID => get_clientid(cid => $cid),
            UID => get_uid(cid => $cid),
            where_from => "fake_web",
        );     
    }
}

sub update_clients_discount {
    my ($clientid2discount) = @_;

    my @clients_discount_data;
    while (my($clientid, $discount) = each %$clientid2discount) {
        if (ref $discount eq 'ARRAY'){
            push @clients_discount_data, map { [$clientid, $_->{discount}, $_->{date_from}, $_->{date_to},] } grep { $_->{discount} > 0 } @$discount; 
        } else {
            next if $discount == 0; # нулевые скидки в базу не записываем
            push @clients_discount_data, [
                $clientid,
                $discount,
                $BEGIN_OF_TIME_FOR_STAT,
                $Settings::END_OF_TIME,
            ];
        }
    }
    my @discount_client_ids = keys %$clientid2discount;
    do_delete_from_table(PPC(ClientID => \@discount_client_ids), 'client_discounts', where => {ClientID => SHARD_IDS});
    if (@clients_discount_data) {
        foreach_shard ClientID => \@clients_discount_data, by => sub {$_->[0]}, sub {
            my ($shard, $chunk) = @_;
            do_mass_insert_sql(PPC(shard => $shard), '
                                INSERT INTO client_discounts (ClientID, discount, date_from, date_to)
                                       VALUES %s
                                       ON DUPLICATE KEY UPDATE discount = VALUES(discount), date_to = VALUES(date_to), date_from = VALUES(date_from)
                            ', $chunk
            );
        };
    }
}

=head2 convert_client_currency

    $convert_ok = convert_client_currency($client_id);

=cut

sub convert_client_currency {
    my ($client_id) = @_;

    my $log = Yandex::Log->new(
        date_suf => "%Y%m%d",
        log_file_name => "convert_client_currency.log",
        auto_rotate => 1,
        lock => 1,
    );
    my $msg_prefix_guard = $log->msg_prefix_guard("[$client_id]");

    {
        no warnings 'once';
        $Client::ConvertToRealMoneyTasks::log = $log;
    }

    do_update_table(PPC(ClientID => $client_id), 'currency_convert_queue', {start_convert_at__dont_quote => 'NOW() + INTERVAL 1 MINUTE'}, where => {ClientID => $client_id, start_convert_at__gt__dont_quote => 'NOW()'});

    # делаем не более 200 итераций, в нормальном случае по last выйдем раньше
    my $convert_ok = 0;
    for(1..200) {
        my $tasks_to_add = get_all_sql(PPC(ClientID => $client_id), ['
            SELECT ClientID, uid, convert_type, state, new_currency, country_region_id, email, start_convert_at
                 , balance_convert_finished
            FROM currency_convert_queue
            WHERE', { state__ne => 'DONE', ClientID => SHARD_IDS }]);
        $log->out($tasks_to_add);

        if ($tasks_to_add && @$tasks_to_add) {
            
            $log->out('Processing ' . scalar(@$tasks_to_add) . ' tasks');
            for my $task (@$tasks_to_add) {
                if ($task->{state} eq 'OVERDRAFT_WAITING') {
                    # 100 итераций пролетают много быстрее, чем успевает доехать нотификация из Баланса
                    # поэтому при ожидании овердрафта добавляем принудительные паузы
                    sleep 2;
                } elsif (mysql2unix($task->{start_convert_at}) > time()) {
                    # при ожидании начала конвертации тоже добавляем паузы
                    sleep 2;
                } elsif (!$task->{balance_convert_finished} && $task->{state} ne 'NEW') {
                    $log->out('Skipping task execution because of Balance-side convert is not finished yet');
                    sleep 4;
                    next;
                }

                $task->{action} = Client::ConvertToRealMoney::get_next_convert_task_name($task->{convert_type}, $task->{state});
                if (!$task->{action}) {
                    $log->die("Cannot determine which action should be run for task in state $task->{state}");
                }
                
                $log->out("Running job for action '$task->{action}' to convert currency for ClientID $task->{ClientID}");
                eval { no warnings 'once'; $Client::ConvertToRealMoneyTasks::ACTION_MAP{ $task->{action} }->($task); 1 } or $log->warn($@);
            }
        } else {
            $log->out('No tasks found');
            $convert_ok = 1;
            last;
        }
    }

    if (!$convert_ok) {
        $log->out("Client is still not converted. Giving up.");
    }

    return $convert_ok;
}

=head2 set_camp_day_budget_limit_stop_time($order_id, $datetime)

    Фейковая нотификация об остановке кампании/общего счета по дневному бюджету
    set_camp_day_budget_limit_stop_time(12345, '2017-06-27 18:17:32');

=cut

sub set_camp_day_budget_limit_stop_time {
    my ($order_id, $datetime) = @_;

    $datetime =~ s/\D+//g;
    unless (length($datetime) == 14) {
        die "invalid date format. Need to have 14 digits";
    }
    unless ($order_id && $order_id =~ /^\d+$/) {
        die "invalid order_id";
    }

    SOAP::Lite->proxy($Settings::DIRECT_INTAPI_SOAP_URL)->uri("ServiceSOAP")->setCampDayBudgetLimitStopTime({$order_id => $datetime});
}

=head2 try_disable_wallet($wallet_cid)

    Пробует отвязать от ОС привязанную к нему кампанию. Если получилось, значит общий счет отключаемый.
    Если получили ошибку только при попытке отвязать кампанию от ОС, значит неотключаемый.

    Возвращает {success => 'текст упеха'}, если удалось отключить ОС (ОС потом включается обратно)
    Если отключить не удалось, или произошла какая другая ошибка, возвращает {error => 'текст ошибки'}

=cut
sub try_disable_wallet {
    my ($wallet_cid) = @_;

    my $OPERATOR_UID = 1;
    my $rbac = RBAC2::Extended->get_singleton($OPERATOR_UID);

    my $camp_cid = get_one_field_sql(PPC(cid => $wallet_cid), [
        "SELECT c.cid
        FROM campaigns wc
            JOIN campaigns c ON c.ClientID = wc.ClientID AND c.wallet_cid = wc.cid",
        WHERE => [
            'wc.cid' => $wallet_cid,
        ],
        "LIMIT 1"
    ]);

    if (!$camp_cid) {
        return {error => "Под общим счетом нет ни одной кампании"};
    }

    my $bind_balance_res = create_campaigns_balance($rbac, $OPERATOR_UID, [$camp_cid]);
    unless ($bind_balance_res && $bind_balance_res->{balance_res}) {
        return {error => "Не получается отправить кампанию $camp_cid в Баланс"};
    }

    my $result = {error => "Неизвестная ошибка"};
    eval {
        do_in_transaction sub {
            do_update_table(PPC(cid => $camp_cid),
                "campaigns",
                {wallet_cid => 0},
                where => [
                    'cid' => $camp_cid,
                ]
            );

            my $balance_res = create_campaigns_balance($rbac, $OPERATOR_UID, [$camp_cid]);

            if ($balance_res && $balance_res->{balance_res}) {
                $result = {success => "Общий счет может быть отключен"};
            } elsif ($balance_res && $balance_res->{balance_res} == 0) {
                $result = {error => "Нельзя отключить общий счет, он неотключаемый"}
            }

            die "no commit\n";
        };
    };

    my $restore_balance_res = create_campaigns_balance($rbac, $OPERATOR_UID, [$camp_cid]);
    unless ($restore_balance_res && $restore_balance_res->{balance_res}) {
        $result = {error => "Не получается перепривязать кампанию $camp_cid обратно к ОС в Балансе"};
    }

    return $result;
}

=head2 send_campaign_to_balance

    Переотправляет кампанию в баланс.
    Используется в автотестах для противодействия переналивке базы баланса&
    Принимает номер кампании, возвращает ответ баланса "as is"

=cut

sub send_campaign_to_balance {
    my ($cid) = @_;

    return create_campaigns_balance(undef, 0, [$cid]);
}

1;

