package FilterSchema;

use Direct::Modern;

use Carp qw/croak carp/;
use Try::Tiny qw/try catch/;
use Cache::SizeAwareMemoryCache;

use Settings;
use File::Slurp qw/read_file/;
use JSON::XS;
use JSON::Pointer;
use Scalar::Util qw/blessed/;
use Yandex::Clone qw/yclone/;
use Yandex::ListUtils qw/xisect/;
use Yandex::I18n;
use Data::Dumper qw//;
use List::MoreUtils qw/any all/;
use HashingTools qw/url_hash_utf8/;
use Direct::ValidationResult;
use Direct::Validation::HRef qw//;
use Direct::Errors::Messages;

use Mouse;
 
#Каталог со схемами фильтров
our $SCHEMA_ROOT //= "$Settings::ROOT/data/json_schemas";

#Настройки Cache::SizeAwareMemoryCache
our $FILTER_SCHEMA_CACHE_OPTIONS //= {
    namespace => 'FilterSchema',
    max_size => 1_073_741_824, #1Mb
    default_expires_in => 3600,  
};

#Ноды, для значений которых требуется перевод
our $IGET_NODES //= { map { $_ => 1} qw/error frontendError errorOnWrongType frontendErrorOnWrongType/ };

my $CACHED_JSON = Cache::SizeAwareMemoryCache->new($FILTER_SCHEMA_CACHE_OPTIONS);


#Рабочая json-схема
has 'schema' => (
    is => 'ro',
    isa => 'HashRef'
);

#Скомпилированная схема (раскрыты типы в нодах перечисленных в "explain")
has 'compiled_schema' => (
    is => 'ro',
    isa => 'HashRef',
    lazy_build => 1
);
#Включение трассировки хода проверки на STDERR
has 'debug' => (
    is => 'rw',
    isa => 'Bool',
    default => '0',
);

#Аварийно завершаться при первой найденной ошибке
has 'die_on_error' => (
    is => 'rw',
    isa => 'Bool',
    default => '0',
);

has '_base_types' => (
    is => 'ro',
    isa => 'HashRef',
    lazy_build => 1
);

has '_constraints' => (
    is => 'ro',
    isa => 'HashRef',
    lazy_build => 1
);

has '_error_types' => (
    is => 'ro',
    isa => 'HashRef',
    lazy_build => 1
);

has '_cache' => (
    is => 'ro',
    isa => 'HashRef',
    default => sub { {} },
);

has '_shared' => (
    is => 'rw',
    isa => 'HashRef',
    default => sub { {} },
);

has 'validation_result' => (
    is => 'rw',
    isa => 'Direct::ValidationResult',
    lazy_build => 1,
);

has '_default_error' => (
    is => 'ro',
    isa => 'Str',
    default => iget_noop('Неверные входные данные')
);

has '_default_error_type' => (
    is => 'ro',
    isa => 'Str',
    default => 'invalidFormat'
);

has '_internal_error' => (
    is => 'rw',
    isa => 'Maybe[ArrayRef]',
    default => sub{[]},
);

has '_state' => (
    is => 'rw',
    isa => 'HashRef',
    default => sub { {} },
);


around BUILDARGS => sub {
    my ($orig, $class, %args) = @_;

    my $filter_type = delete $args{filter_type};
    my $file = delete $args{file};
    my $is_file = 0;
    if (defined $file) {
        $filter_type = $file;
        $is_file = 1;
    }
    
    croak 'filter_type required' unless $filter_type || $args{schema};

    $args{schema} = $class->get_schema_by_filter_type($filter_type, $is_file => $is_file) if $filter_type;
    
    return $class->$orig(%args);
};

=head2 get_schema_by_filter_type($class, $filter_type)
    Для заданного типа фильтра считывает конфиг из json-файла и возвращает результат парсинга json
    Usage:
    my $schema = FilterSchema->get_schema_by_filter_type('ftype');
    my $validator = FilterSchema->new(schema => $schema);
=cut

sub get_schema_by_filter_type {
    my ($class, $filter_type, $is_file) = @_;

    croak 'Invalid character at filter name: allowed "a-z", "0-9", "_"' if !$is_file && $filter_type =~/\W/;

    my $schema = $CACHED_JSON->get($filter_type);
    return yclone($schema) if defined $schema;
    
    my $file_path = $is_file ? $filter_type : sprintf("%s/%s.json", $SCHEMA_ROOT, $filter_type);
    if (-f $file_path) {
        my $content = read_file($file_path) || croak 'Can`t read file '.$file_path;
        $schema = JSON::XS->new->relaxed(1)->utf8(1)->decode($content);
    } else {
        $schema = {};
    }

    $CACHED_JSON->set($filter_type => $schema);
    return $schema;
}

=head2 check($self, $data)
    Выполняет валидацию переданной структуры данных по текущей схеме
    
    Возвращает Direct::ValidationResult с результатами проверки
    
=cut

sub check {
    my ($self, $data) = @_;

    try {
        my ($error, $error_type) = $self->_check_node($data => '/', $self->_get_node_by_path($self->schema =>'/checkedStruct'));
        $self->validation_result->add_generic($self->_mk_vr_error($error, $error_type)) if $error;
    }
    catch {
        my ($exception) = @_;

            if (blessed $exception eq blessed $self) {
                #Мы поймали свое исключение
                $self->validation_result->add_generic(error_CantPerform(@{$exception->_internal_error}));
            }
            else{
                #Внешнее исключение, отправим его дальше
                croak $exception;
            }
    };
    return $self->validation_result;
}

=head2 as_json($self, $schema_type, $pretty)
    Возвращает заданную схему в виде json.
    Если для $schema_type передано значение 'compiled' - возвращается сериализованная скомпилированная схема,
    иначе возвращается рабочая схема.
    
    Параметр $pretty задает будет или нет отдаваемый json отформатирован в удобном для чтения виде.
    
=cut

sub as_json {
    my ($self, $schema_type, $pretty) = @_;
    my $desired_schema = defined $schema_type && $schema_type eq 'compiled' ? 'compiled_schema' : 'schema';
    $pretty = defined $pretty ? 1 : 0;
    
    return JSON::XS->new->pretty($pretty)->utf8(1)->encode($self->$desired_schema)
}

=head2 reset()
    Приводит текущий экземпляр валидатора в состояние пригодное для проверки новой структуры данных -
    сбрасывается ValidationResult и данные, используемые для подстановки в тексты ошибок.
    Usage:
        my $check1 = $validator->check($data);
        my $check2 = $validator->reset->check($another_data);
=cut

sub reset {
    my ($self) = @_;
    $self->_shared ({});
    $self->_state ({});
    $self->validation_result( $self->_build_validation_result);
    
    return $self;
}

=head2 iget_nodes()
    Возвращает хеш, ключами которого являются имена нод, к значениям которых применяется iget
=cut

sub iget_nodes { return $IGET_NODES }

=head2 walk_by_error_texts($schema, $handler_ref)
    Обходит все ноды переданной схемы.
    Если имя ноды совпадает с одним из имен нод, определяющих тексты ошибок,
    вызывается функция, ссылка на которую передана как $handler_ref.
    Handler-у передается единственный аргумент - ссылка на содержимое ноды.
=cut

sub walk_by_error_texts {
    my ($self, $schema, $handler) = @_;
    
    $schema = $self->schema unless ref $schema;
    
    my $error_nodes = $self->iget_nodes;
    
    if (ref $schema eq 'HASH') {
        #нужно сделать в обобщенном виде
        if (exists $schema->{translations} && ref $schema->{translations} && exists $schema->{translations}->{rule}) {
            my $translations = $schema->{translations}->{rule};
            &$handler(\$translations->{$_}) foreach keys %$translations;
        }

        while (my ($k, $v) = each %$schema ) { 
            if (ref $v) {
                $self->walk_by_error_texts( $v, $handler );
                next;
            }
            next unless exists $error_nodes->{$k};
            &$handler(\$schema->{$k});
        }
    }   
    elsif( ref $schema eq 'ARRAY' ) { 
        foreach my $v (@$schema) {
            #Скаляры в массиве не могут быть интересующими нас нодами - пропускаем
            next unless ref $v;
            $self->walk_by_error_texts($v, $handler);
        }
    }   
    
    return;
}

=head2 iget_schema($schema)
    
    Возвращает схему с локализованными текстами ошибок.
    
    Если не передано параметров - локализуется копия рабочей схемы,
    если передана строка 'compiled' - локализуется скомпилированная схема,
    если передана ссылка на структуру - буду локализоваться ноды этой структуры.

=cut

sub iget_schema {
    my ($self, $in) = @_;
    
    if (! keys %{$self->schema}) {
        return {};
    }
    
    my $schema = $in // yclone $self->schema;

    $schema = $self->compiled_schema if !ref $schema && $schema eq 'compiled';
    
    $self->walk_by_error_texts( $schema => sub { my ($err_ref) = @_; $$err_ref = iget($$err_ref) if defined $$err_ref } );
    
    return $schema;
}

sub _check_node {
    my ($self, $checked_struct, $path, $subschema) = @_;

    $self->_log('---------: check node: %s', { path => $path});    
    $subschema = $self->_explain_ref( $subschema ) unless ref $subschema;
    $self->_throw(sprintf('Invalid set of constraints: %s', Data::Dumper->new([$subschema], ['subschema'])->Dump))
            unless ref $subschema eq 'HASH';
    my $node = $self->_get_node_by_path($checked_struct => $path);

    #Определяем тип
    my $type_definition;
    if (exists $subschema->{typeByCondition}) {
        $self->_throw('Mixing type and typeByCondition not allowed') if exists $subschema->{type};
        
        $type_definition = $self->_get_type_by_condition($subschema->{typeByCondition}, $subschema->{default}, $checked_struct => $path);
    }
    else {
        $type_definition = $subschema->{type};
    }
    
    my ($type, $constraints, $extra_fields) = $self->_explain_type_for_subschema($type_definition, $subschema);
    
    my ($error, $error_type);

    $self->_log('check by type: %s %s %s', {type => $type}, {extra_fields => $extra_fields}, {node_data => $node});  
    ($error, $error_type) = $self->_check_by_type($type->{base} => $type->{parameters}, $extra_fields, $node);

    return $error if $error;
    $self->_log('check by constraints: %s %s %s', {type => $type->{base}}, {constraints => $constraints}, {node_data => $node});
    ($error, $error_type) = $self->_check_by_constraints($type->{base}, $constraints, $node);

    return ($error, $error_type);

}

sub _get_current_vr {
    my ($self) = @_;
    
    return $self->_state->{validation_result} // $self->validation_result;
}

sub _next_index {
    my ($self) = @_;
    
    $self->_state->{index} //= -1;
    my $index = ++$self->_state->{index};
    $self->_state->{validation_result} = $self->validation_result->get_nested_vr_by_index($index) // $self->validation_result->next;
    
    return;    
}

sub _reset_index {
    my ($self) = @_;
    undef $self->_state->{index};
    undef $self->_state->{validation_result};
    return;
}

sub _mk_vr_error {
    my ($self, $error, $error_type) = @_;
    $error_type //= $self->_default_error_type;
    my $vr_error = $self->_error_types->{$error_type} // $self->_error_types->{$self->_default_error_type};
    
    return &$vr_error( $self->_fill_error(iget($error)) );
}

sub _log {
    my ($self, $msg, @data) = @_;
    return unless $self->debug;
    {
        no warnings 'once';
        local $SIG{__WARN__} = $Carp::Always::OLD_SIG{__WARN__};
        my $_dump = sub {
            if (ref $_[0]) {
                return Data::Dumper->new([$_[0]],[])->Dump;
            }
            return $_[0];
        };
        warn sprintf( $msg, map { $_dump->($_) } @data), "\n";
    }
    return;
}

sub _explain_type_for_subschema {
    my ($self, $type_definition, $subschema) = @_;
    
    my ($type, $constraints, $extra_fields) = $self->_resolve_type($type_definition);
    
    $constraints = $self->_merge_constraints($constraints, $subschema->{constraints} || []);
    foreach (@$constraints){
        warn (sprintf('Constraint %s not allowed for type %s', $_->{name}, $type->{base}) )
            unless $self->_type_has_constraint($type->{base} => $_);
    }
    $extra_fields = $self->_merge($extra_fields, $subschema);
    
    return ($type, $constraints, $extra_fields);
}

sub _check_by_type {
    my ($self, $type, $type_parameters, $extra_fields, $node) = @_;

    my $checker = $self->_base_types->{$type}->{checker} || $self->_throw(sprintf('Checker doesn`t exists for type: %s', $type));
    my $error_text = $extra_fields->{errorOnWrongType};
    my $custom_error_type = $extra_fields->{errorType};

    return (
                $self->$checker($type_parameters, $node, $extra_fields, $error_text),
                $custom_error_type,
           );
}

sub _check_by_constraints {
    my ($self, $type, $constraints, $node) = @_;

    my ($error, $error_type);
    $self->_throw(sprintf('List of costraints must be ARRAY: %s', $constraints)) if ref $constraints ne 'ARRAY';
    foreach my $constr (@$constraints) {
        $self->_throw(sprintf('Invalid definition of constraint: %s', Data::Dump->new([$constr], ['constraint'])->Dump))
            unless ref $constr eq 'HASH' && $constr->{name};
        $self->_throw(sprintf('Constraint %s not allowed for type %s', $constr->{name}, $type) )
            unless $self->_type_has_constraint($type => $constr);

        my $error_text = $constr->{error};
        my $custom_error_type = $constr->{errorType};
        my $constraint = $self->_constraints->{ $constr->{name} } || $self->_throw(sprintf('Checker doesn`t exists for constraint: %s', $constr->{name}));
        my $method = $constraint->{method};

        ($error, $error_type) = $self->$method( $constr->{parameters}, $node, $error_text);
        $error_type = $custom_error_type if defined $custom_error_type;
        
        next unless $error;
        last if $constraint->{break} || $constr->{break};
        
        $self->_set_error($error, $error_type);
        undef $error;
    }

    return unless $error;
    return ($error, $error_type);
}

sub _type_has_constraint {
    my ($self, $type, $constraint) = @_;
    
    return any { $_ eq $constraint->{name} } @{$self->_base_types->{$type}->{avalible_constraints} };
}

sub _get_node_by_path {
    my ($self, $data, $path) = @_;

    return $data if ($path eq '/');
    
    return JSON::Pointer->get($data => $path);
}

sub _get_type_by_condition{
    my ($self, $conditions, $default, $data, $path) = @_;
    
    my $type_definition;
    $self->_throw(sprintf('List of conditions for typeByCondition must be ARRAY, but %s given', $conditions)) unless ref $conditions eq 'ARRAY';
    #Т.к. определяем тип для текушей ноды, в условиях могут фигурировать только данные родительской ноды
    my ($parent_path, $last_node) = $self->_extract_parent_node_path($path);

    $self->_throw(sprintf('Node %s hasn`t parrent, can`t use typeByCondition', $path)) unless $parent_path;

    my $parent_node = $self->_get_node_by_path($data => $parent_path);
    foreach my $c_rec (@$conditions){
        my $check = $c_rec->{condition}; 
        $self->_throw(sprintf('Invalid condition: %s ', Data::Dumper->new([$check], ['condition'])->Dump) ) unless $check;

        next unless $self->_check_condition($check, $parent_node);
        ##Потенциально опасное место.
        #Т.к. $type_definition клонируется при раскрытии типа в _resolve_type, здесь можно обойтись без клонирования,
        #но только до тех пор, пока мы уверены, что при дальнейшем merge-е исходная структура из typeByCondition не испортится.
        $type_definition = $c_rec;

        last;
    }
    $type_definition //= $default;
    $self->_throw(sprintf('Can`t define type: %s ', Data::Dumper->new([$conditions, $parent_node], ['conditions', 'data'])->Dump))
        unless $type_definition;

    return $type_definition;
}

sub _check_condition {
    my ($self, $check, $parent_node) = @_;
    #Пока поддерживаем только конструкцию field => xxx  in => [....]
    my $field_name = $check->{field};
    my $possible_values = $check->{in};
    $self->_throw(sprintf('Invalid format of condition: %s ', Data::Dumper->new([$check], ['condition'])->Dump)) 
        unless $field_name && $possible_values && ref $possible_values eq 'ARRAY';
    
    return unless ref $parent_node eq 'HASH';
    return exists $parent_node->{$field_name} && any {$_ eq $parent_node->{$field_name}} @$possible_values;
}

sub _resolve_type {
    my ($self, $type_definition) = @_;
    return (undef, [], {} ) unless defined $type_definition;

    my $type_name;
    my $type_struct = {};
    my %parent;
    my $name_or_jpointer;

    unless ( ref $type_definition ) {
        return @{ $self->_cache->{$type_definition} } if exists $self->_cache->{$type_definition};
        ($type_name) = $type_definition =~ qr|([^/]+)$|;
        $name_or_jpointer = $type_definition;
        $type_definition = $self->_explain_ref($type_definition);
    }
    # Ссылка могла раскрыться в структуру
    if (ref $type_definition) {
        #Клонируем структуру, чтобы не портить схему
        $type_struct = yclone $type_definition;
        #Если определение типа - это структура, возможны два варианта:
        # - структура содержит ключевое слово "type" - это определение типа
        # - структура содержит один ключ, отличный от type - это параметризованный тип
        #Если ни то и ни другое - просто неверное определение типа
        $self->_throw('Invalid structure for definition of type') unless ref $type_struct eq 'HASH';
        if ( exists $type_struct->{type} ){
            @parent{qw/type constraints extra_fields/} = $self->_resolve_type(delete $type_struct->{type});
            $type_name = delete $type_struct->{typeName} if exists $type_struct->{typeName};
        }
        elsif( scalar keys %$type_struct == 1){
            my ($ptype, $parameters) = %$type_struct;
            delete $type_struct->{$ptype};
            @parent{qw/type constraints extra_fields/} = $self->_define_base_type_by_name( $ptype, $parameters);
        }else {
            $self->_throw(sprintf('Invalid definition of type: %s ', Data::Dumper->new([$type_struct], ['type'])->Dump))
        }
    }
    else {
        @parent{qw/type constraints extra_fields/} = $self->_define_base_type_by_name($type_definition);
    }

    my $type = { 
        base => $parent{type}->{base},
        name => $type_name || $parent{type}->{name},
    };
    $type->{parameters} = $parent{type}->{parameters} if exists $parent{type}->{parameters};

    my $constraints = $self->_merge_constraints($parent{constraints}, delete $type_struct->{constraints});
    $type_struct = $self->_merge($parent{extra_fields}, $type_struct);
    undef $type_struct unless keys %$type_struct;

    my $result = [$type, $constraints, $type_struct];
    #Результаты для базовых типов и типов, задаваемых по ссылке кешируем, чтобы не собирать по многу раз одни и те же cтруктуры
    $self->_cache->{$type_definition} = $result if $name_or_jpointer;

    return @$result;
}

sub _merge_constraints {
    my ($self, $left, $right) = @_;
    $_ //= [] foreach ($left, $right);
    
    my $merged;
    $self->_throw(sprintf('Merged list of constraints must be ARRAY: %s <- %s', $left, $right)) if ref $left ne 'ARRAY' || ref $right ne 'ARRAY';
    
    #Констрейнты смешиваем по имени - при совпадении имени используем конструкцию из правого списка в той же позиции,
    #уникальные из правого списка добавляем в конец.
    #Считаем, что два констрейнта с одним и тем же именем не могут одновременно присутствовать в списке
    my $i = 0;
    my $r_idx;
    my $drop = {};
    foreach my $r_constr (@$right){
        $self->_throw(sprintf('Invalid structure for right constraint: %s', Data::Dumper->new([$r_constr], ['constraint'])->Dump))
            unless ref $r_constr eq 'HASH' && (exists $r_constr->{name} || exists $r_constr->{drop});
        $self->_throw(sprintf('Constraints at one level must be unique by name: %s', Data::Dumper->new([$r_constr], ['constraint'])->Dump))
            if exists $r_idx->{ $r_constr->{name} // '' };

        if (defined $r_constr->{drop}) {
            $drop->{$r_constr->{drop}} = 1;
        }
        else {
            $r_idx->{ $r_constr->{name} } = $i;
        }
        $i++;
    }
    #Пропускаем констрейнты, для которых указано drop
    $merged =  [ grep {!exists $drop->{$_->{name}}} @$left ];
    
    my $cloned_right = yclone($right);
    foreach my $i (0 .. $#{$merged}){
        my $l_constr = $left->[$i]; # при проверках используем значения из исходных структур
        $self->_throw(sprintf('Invalid structure for left constraint: %s', Data::Dumper->new([$l_constr], ['constraint'])->Dump))
            unless ref $l_constr eq 'HASH' && exists $l_constr->{name};
        if (exists $r_idx->{$l_constr->{name} }) {
            my $right_id = delete $r_idx->{$l_constr->{name}};
            next if $left->[$i] eq $right->[$right_id]; #если значения совпадают (например, получены по одной ссылке) - ничего не меняем
            $merged->[$i] = $cloned_right->[$right_id];
        }
    }
    #Теперь добавляем в конец то, что есть только в $right
    push @$merged, (map { $cloned_right->[$_]} sort {$a <=> $b} values %$r_idx);

    return $merged;
}

sub _merge {
    my ($self, $left, $right) = @_;

    return undef unless defined $left || defined $right;
    return yclone($right) unless defined $left;
    return yclone($left) unless defined $right;
    
    my $merged;
    $self->_throw(sprintf('Can`t merge structures with different type: %s и %s', $left, $right)) unless ref $left eq ref $right;
    
    $merged = ref $left eq 'ARRAY' ? $self->_merge_arrays($left, $right) : $self->_merge_hashes($left, $right);
    
    return $merged;
}

sub _merge_hashes{
    my ($self, $left, $right) = @_;
    my $merged;
    $self->_throw(sprintf('Merged structure isn`t HASH: %s <- %s', $left, $right)) if ref $left ne 'HASH' || ref $right ne 'HASH';

    $merged = yclone($left);
    foreach my $key ( keys %$right) {
        #На эквивалентность проверяем элементы исходных структур
        if ( exists $left->{$key} ) { 
            next if $left->{$key} eq $right->{$key};
            $self->_throw(sprintf('Can`t merge hash elements with different type: %s и %s', $left->{$key}, $right->{$key})) unless ref $left->{$key} eq ref $right->{$key};
            if ( ref $merged->{$key} eq 'HASH') {
                $merged->{$key} = $self->_merge_hashes($left->{$key}, $right->{$key});
            }
            elsif ( ref $left->{$key} eq 'ARRAY' ) {
                my $method = $key ne 'constraints' ? '_merge_arrays' : '_merge_constraints';
                $merged->{$key} = $self->$method($left->{$key}, $right->{$key});
            }
            else {
                $merged->{$key} = $right->{$key}
            }
        }
        else { 
            $merged->{$key} = $right->{$key};
        }
    }
    return $merged;
}

sub _merge_arrays {
    my ($self, $left, $right) = @_;

    $self->_throw(sprintf('Merged structure isn`t ARRAY: %s <- %s', $left, $right)) if ref $left ne 'ARRAY' || ref $right ne 'ARRAY';
    #Массивы не объединяем, всегда используем правый.
    #При более сложном смешивании нужно знать, что именно за контент смешивается, если хотим получить на выходе адекватную структуру.
    my $merged = $right;

    return $merged
}


sub _define_base_type_by_name {
    my ($self, $type_name, $parameters ) = @_;
    my ($type, $constraints, $extra_fields);
    
    my $base_type = $self->_base_types->{$type_name} ||
        $self->_throw(sprintf('Unhandled base type: %s, allowed: [ %s ]', $type_name, (join ', ', keys %{$self->_base_types})));

    $type = { base => $type_name, $parameters ? ( parameters => $parameters ) : () };
    #Набор ограничений для данного базового типа, которые нужно дополнительно отобразить в схеме
    $constraints = $base_type->{ default_constraints } || [];
    #Дополнительная структура, которую нужно добавить в схему при раскрытии этого типа
    $extra_fields = $base_type->{ extra_fields } || {};

    return $type, $constraints, $extra_fields;
}

sub _explain_ref {
    my ($self, $ref) = @_;

    return $ref unless defined $ref && $ref =~qr|^#(/.+)$|;
    
    my $path = $1;
    my $node = $self->_get_node_by_path($self->schema, $path) 
        || $self->_throw(sprintf('Scheme contains the reference to non existing element: %s', $path));

    return $node;
}

sub _explain_type_by_condition{
    my ($self, $schema, $node_path) = @_;
    
    my $type_by_condition_path = $node_path.'/'.'typeByCondition';
    my $types = $self->_get_node_by_path($schema => $type_by_condition_path);
    $types = $self->_explain_ref($types) unless ref $types;
    $self->_throw(sprintf('Invalid definition of typeByCondition: %s', Data::Dumper->new([$types], ['typeByCondition'])->Dump)) unless ref $types eq 'ARRAY';;

    #Заменяем в схеме определение массива условных типов раскрытым определением
    $self->_set_node($schema => $type_by_condition_path, $types);
    #Раскрываем типы в каждом определении
    foreach my $i ( 0 .. $#{$types} ) {
        $self->_explain_node($schema => $type_by_condition_path.'/'.$i);
    }
    #Раскрываем определение дефолтного типа
    my $node = $self->_get_node_by_path($schema => $node_path);
    $self->_explain_node($schema => $node_path.'/'.'default') if exists $node->{default};
    
    return;
}

sub _explain_fields{
    my ($self, $schema, $fields_path) = @_;
        
    my $fields = $self->_get_node_by_path($schema => $fields_path);
    $fields = $self->_explain_ref($fields) unless ref $fields;
    $self->_throw(sprintf('Invalid definition for set of fields: %s', Data::Dumper->new([$fields], [$fields_path])->Dump)) unless ref $fields eq 'HASH';

    #Заменяем в схеме определение типов полей объекта раскрытым определением
    $self->_set_node($schema => $fields_path, $fields);
    #Раскрываем типы в определении каждого поля
    foreach my $f (keys %$fields) {
        $self->_explain_node($schema => $fields_path.'/'.$f);
    }
        
    return;
}

sub _explain_node {
    my ($self, $schema, $path) = @_;

    my $node = $self->_get_node_by_path($schema => $path);
    $self->_throw(sprintf('Element doesn`t exist: %s', $path)) unless defined $node;

    $node = $self->_explain_ref($node) unless ref $node;
    if ($node->{typeByCondition}) {
        $self->_throw(sprintf('%s - mixing type and typeByCondition not allowed', $path)) if exists $node->{type} && $node->{type} ne 'skip';
        delete $node->{type} if exists $node->{type};
        $self->_explain_type_by_condition($schema => $path);        
    }
    else {
        my ($type, $constraints, $extra_fields) = $self->_explain_type_for_subschema($node->{type}, $node);
        
        my $compiled_node = $extra_fields;
        $compiled_node->{typeName} = $type->{name} // $type->{base};
        $compiled_node->{type} = defined $type->{parameters} ? {$type->{base} => $type->{parameters}} : $type->{base};
        $compiled_node->{constraints} = $constraints if $constraints && @$constraints;
        
        $self->_set_node($schema => $path, $compiled_node);
        if ($compiled_node->{type} eq 'array') {
            #Если базовый тип - массив, развернем определение его элементов
            $self->_explain_node($schema => $path.'/'.'items') if exists $compiled_node->{items};
        }
        elsif ($compiled_node->{type} eq 'object') {
            #Если базовый тип - объект, развернем определение полей
            $self->_explain_fields($schema => $path.'/'.'fields') if exists $compiled_node->{fields};
        }
    }
    
    return;
}

sub _set_node {
    my ($self, $schema, $path, $node_data) = @_;

    my ($parent_path, $last_node) = $self->_extract_parent_node_path($path);
    my $parent =  $self->_get_node_by_path($schema => $parent_path);
    if (ref $parent eq 'HASH') {  $parent->{$last_node} = $node_data; }
    else{ $parent->[$last_node] = $node_data;}
    
    return;
}

sub _build_compiled_schema {
	my ($self) = @_;

	my $schema = yclone($self->schema);
    my $compile = delete $schema->{explain} // ['#/checkedStruct'];
    $self->_throw(sprintf('Invalid format for list of explainable nodes: %s',Data::Dumper->new([$compile], ['explain'])->Dump) ) unless ref $compile eq 'ARRAY';
    
    foreach my $ref (@$compile) {
        my ($path, $is_multiple) = $ref =~ qr|^\#(.+?)(/\*)?$|;
        $self->_throw(sprintf('Invalid reference: %s', $ref) ) unless defined $path;
        
        if ($is_multiple) {
            #Если путь заканчивается на * - раскрываем все дочерние ноды 
            $self->_explain_fields($schema => $path);
        }
        else{
            $self->_explain_node ($schema => $path);
        }
    }

    return $schema;    
}

sub _extract_parent_node_path{
    my ($self, $path ) = @_;
    my ($parent_path, $last_node ) = $path =~ qr|^(/.*?)/?((?<=/)[^/]+)?/?$|;
    $self->_throw(sprintf('Can`t define parrent node for path: %s', $path)) unless defined $parent_path && defined $last_node;

    return ($parent_path, $last_node);
}


sub _throw {
    my $self = shift;

    carp @_;
    $self->_internal_error(\@_);
    $self->_state({});
    return die $self;
}

sub _fill_error {
    my ($self, $error) = @_;
    my $shared=$self->{_shared};
    $error =~ s/\[%\s*(\w+)\s*%\]/$shared->{$1}/mg;
    $self->_throw($error) if $self->die_on_error;
    return $error;
}

sub _build_validation_result { return Direct::ValidationResult->new() }

sub _build__base_types {
    my @STRING_CONSTRAINTS = qw/maxLength minLength patternMatch containsDisallowedLetters/;
    return {
        number  => { checker => '_check_number', avalible_constraints => [@STRING_CONSTRAINTS, qw/maxValue minValue/] },
        integer => { checker => '_check_integer', avalible_constraints => [@STRING_CONSTRAINTS, qw/maxValue minValue/] },
        boolean => { checker => '_check_boolean', avalible_constraints => [@STRING_CONSTRAINTS]},
        enum    => { checker => '_check_enum', avalible_constraints => [@STRING_CONSTRAINTS]},
        string  => { checker => '_check_string', avalible_constraints => [@STRING_CONSTRAINTS, qw/correctUrl withoutSpaces/]},
        url     => { checker => '_check_url', avalible_constraints => [@STRING_CONSTRAINTS, qw/correctUrl withoutSpaces/]},
        object  => { checker => '_check_object' },
        array   => { checker => '_check_array',  avalible_constraints => [qw/itemsCountMax itemsCountMin itemAnyOf itemsDenyAnotherWhenContainsAny itemsUnique/]},
        range   => { checker => '_check_range', avalible_constraints => [@STRING_CONSTRAINTS, qw/leftLessRight boundaryMinValue boundaryMaxValue boundaryPatternMatch/] },

        string_or_number  => { checker => '_check_string_or_number', avalible_constraints => [@STRING_CONSTRAINTS, qw/withoutSpaces maxValue minValue/]},
        always_wrong => { checker => '_check_always_wrong', avalible_constraints => [] },
    }
}

sub _build__constraints {
    return {
        maxValue  => { method => '_num_maxValue' },
        minValue  => { method => '_num_minValue' }, 
        maxLength => { method => '_str_maxLength' },
        minLength => { method => '_str_minLength' },
        containsDisallowedLetters => { method => '_str_containsDisallowedLetters' },
        withoutSpaces => { method => '_str_withoutSpaces' },
        patternMatch  => { method => '_str_patternMatch' },
        correctUrl    => { method => '_str_correctUrl' },
        itemsCountMax => { method => '_arr_itemsCountMax', break => 1},
        itemsCountMin => { method => '_arr_itemsCountMin', break => 1},
        itemAnyOf     => { method => '_arr_itemAnyOf' },
        itemsUnique   => { method => '_arr_itemsUnique' },
        itemsDenyAnotherWhenContainsAny => { method => '_arr_itemsDenyAnotherWhenContainsAny', break => 1 },
        leftLessRight => { method => '_rng_leftLessRight' },
        boundaryMinValue => { method => '_rng_boundaryMinValue' },
        boundaryMaxValue => { method => '_rng_boundaryMaxValue' },
        boundaryPatternMatch => { method => '_rng_boundaryPatternMatch' },
    }
}

sub _build__error_types {
    return {
        invalidFormat => \&error_InvalidFormat,
        invalidField  => \&error_InvalidField,
        maxLength     => \&error_MaxLength,
        invalidChars  => \&error_InvalidChars,
        limitExceeded => \&error_LimitExceeded,
        inconsistentState => \&error_InconsistentState,
        itemsDuplicated => \&error_Duplicated,
    }
}

sub _build__pointer {
    return JSON::Pointer->new();
}
#--------------------------------------------------------------------------------------------

sub _get_internal_type{
    my ($self, $value) = @_;
    my $flags = B::svref_2object(\$value)->FLAGS;
    if (( $flags & B::SVp_IOK ) == B::SVp_IOK) {
        return "integer";
    }
    elsif (( $flags & B::SVp_NOK ) == B::SVp_NOK ) {
        return "number";
    }
    elsif (( $flags & B::SVp_POK ) == B::SVp_POK) {
        return "string";
    }    
    return 'other';
}


#------ Checkers
sub _check_number {
    my ($self, $parameters, $node, $extra_fields, $error_text ) = @_;

    return if length($node // '') == 0;

    my $type = $self->_get_internal_type($node);
    
    return if any {$type eq $_} qw/number integer/;

    return $error_text // $self->_default_error;
}

sub _check_integer {
    my ($self, $parameters, $node, $extra_fields, $error_text ) = @_;
  
    return if length($node // '') == 0 || $self->_get_internal_type($node) eq 'integer';
   
    return $error_text // $self->_default_error;
}

sub _check_boolean {
    my ($self, $parameters, $node, $extra_fields, $error_text ) = @_;
    
    return if JSON::XS::is_bool($node);

    return $error_text // $self->_default_error;

}

sub _check_enum {
    my ($self, $parameters, $node, $extra_fields, $error_text ) = @_;
    
    $self->_throw(sprintf('Invalid definition for list of values: %s', Data::Dumper->new([$parameters], ['enum'])->Dump))
        unless ref $parameters eq 'ARRAY';
    
    return if !ref $node && ( length($node) == 0 || any {$_ eq $node} @$parameters );
    return $error_text // $self->_default_error;
}

sub _check_string {
    my ($self, $parameters, $node, $extra_fields, $error_text ) = @_;

    return if length($node // '') == 0 || $self->_get_internal_type($node) eq 'string';
    
    return $error_text // $self->_default_error;
}

sub _check_string_or_number {
    my ($self, $parameters, $node, $extra_fields, $error_text ) = @_;

    return if length($node // '') == 0 || $self->_get_internal_type($node) ne 'other';
    
    return $error_text // $self->_default_error;
}

sub _check_always_wrong {
    my ($self, $parameters, $node, $extra_fields, $error_text ) = @_;  
    return $error_text // $self->_default_error;
}

sub _check_url {
    my ($self, $parameters, $node, $extra_fields, $error_text ) = @_;

    my $url_domain_errors = Direct::Validation::HRef::validate_href($node);
    return unless $url_domain_errors && @$url_domain_errors;
    return $error_text // $self->_default_error;
}

sub _check_object {
    my ($self, $parameters, $node, $extra_fields, $error_text) = @_;
    
    my $error = $error_text // $self->_default_error;
    my $error_type;
    return $error unless ref $node eq 'HASH';

    my $fields_definition = $extra_fields->{ fields };
    return unless $fields_definition;
    $self->_throw(sprintf('Invalid definition for set of fields: %s', Data::Dumper->new([$fields_definition], ['fields'])->Dump))
        unless ref $fields_definition eq 'HASH';
    
    my $allowed_optional = $extra_fields->{allowOptionalFields} // 0;
    
    
    my $dispensable_fields = $extra_fields->{dispensableFields} // [];
    $self->_throw(sprintf('Invalid definition for list of dispensable fields: %s', Data::Dumper->new([$dispensable_fields], ['dispensableFields'])->Dump))
        unless ref $dispensable_fields eq 'ARRAY';

    my %dispensable_fields_idx = map { $_ => 1 } @$dispensable_fields;
    
    #Проверяем, что поля структуры полностью совпадают с полями, определенными в схеме
    my @required_fields = grep {! exists $dispensable_fields_idx{$_} } keys %$fields_definition;
    return $error if $#{xisect([keys %$node], \@required_fields ) } != $#required_fields;
    return $error if !$allowed_optional && scalar (grep {! exists $dispensable_fields_idx{$_} } keys %$node) != @required_fields;

    #Теперь проверяем сами поля,
    #ошибка будет определяться по результатам их проверки, общую ошибку для однозначности сразу сбросим
    undef $error;
    
    #Если в _state->{field} не определен - значит выше нас объектов нет, будем переключать _state->{field}
    my $can_switch_field = defined $self->_state->{field} ? 0 : 1;
    foreach my $field ( keys %$fields_definition){
        next if $dispensable_fields_idx{$field} && !exists $node->{$field};
        $self->_state->{field} = $field if $can_switch_field;
        ($error, $error_type) = $self->_check_node($node, '/'.$field, $fields_definition->{$field});
        next unless $error;
       
        $self->_set_error($error, $error_type);
        undef $error;
    }
    undef $self->_state->{field} if $can_switch_field;
    
    return;
}

sub _check_array {
    my ($self, $parameters, $node, $extra_fields, $error_text) = @_;

    $node = [$node] if !ref $node && $extra_fields->{flexible};
    return $error_text // $self->_default_error unless ref $node eq 'ARRAY';
    
    my $item_definition = $extra_fields->{ items };
    return unless $item_definition;
 
    my $item_id_alias = $extra_fields->{itemIdAlias};
    
    $self->_throw(sprintf('Invalid definition for array elements: %s', Data::Dumper->new([$item_definition], ['items'])->Dump))
        unless ref $item_definition eq 'HASH';
    
    my ($error, $has_error, $error_type);

    #Если в _state нет индекса - значит мы на верхнем уровне и будем писать туда свой
    my $can_switch_index = defined $self->_state->{index} || defined $self->_state->{field} ? 0 : 1;
    foreach my $i ( 0..$#{$node} ){
        $self->_next_index if $can_switch_index;
        #Если в схеме был указан алиас для id - сохраняем его в _shared, чтобы затем использовать при подстановке в тексты ошибок
        $self->_shared->{$item_id_alias} = $i+1 if defined $item_id_alias;
        
        ($error, $error_type) = $self->_check_node($node, '/'.$i, $item_definition);
       
        $self->_set_error($error, $error_type) if $error;
    }
    $self->_reset_index if $can_switch_index;

    return;
}

sub _check_range{
    my ($self, $parameters, $node, $extra_fields, $error_text ) = @_;
    
    if (!ref $node && ( length($node // '') > 0 )){
        my ($left, $right) = $self->_rng__extract_values($node);
        return if defined $left || defined $right;
    }
    return $error_text // $self->_default_error;
}

#------ Constraints
sub _num_maxValue {
    my ($self, $maxValue, $node, $error_text) = @_;
    $self->_throw(sprintf('Invalid maxValue: %s', $maxValue )) if $self->_check_number(undef, $maxValue);
    return if $node <= $maxValue;
    return (($error_text // $self->_default_error), 'invalidField');
}

sub _num_minValue {
    my ($self, $minValue, $node, $error_text) = @_;

    $self->_throw(sprintf('Invalid minValue: %s', $minValue )) if $self->_check_number(undef, $minValue);
    return if $node >= $minValue;
    return (($error_text // $self->_default_error), 'invalidField');
}

sub _str_maxLength {
    my ($self, $maxLength, $node, $error_text) = @_;
    $self->_throw(sprintf('Invalid maxLength: %s', $maxLength )) if $self->_check_number(undef, $maxLength);
    return if length($node // '' ) <= $maxLength;
    return (($error_text // $self->_default_error), 'maxLength');
}

sub _str_minLength {
    my ($self, $minLength, $node, $error_text) = @_;
    $self->_throw(sprintf('Invalid minLength: %s', $minLength )) if $self->_check_number(undef, $minLength);
    return if length($node // '' ) >= $minLength;
    return (($error_text // $self->_default_error), 'invalidField');
}

sub _str_withoutSpaces {
    my ($self, $parameters, $node, $error_text) = @_;
    return if ( length($node // '') == 0 || $node =~ /^\S+$/ );
    return (($error_text // $self->_default_error), 'invalidField');
}

sub _str_patternMatch {
    my ($self, $pattern, $node, $error_text) = @_;
    return if ( length($node // '') == 0 || $node =~ qr|$pattern| );
    return (($error_text // $self->_default_error), 'invalidField');
}

sub _str_containsDisallowedLetters {
    my ($self, undef, $node, $error_text) = @_;
    
    return (($error_text // $self->_default_error), 'invalidChars') if ( length($node // '') && $node =~ qr/[^\Q${Settings::ALLOW_BANNER_LETTERS_STR}|\E]/i );
    return;
}

sub _str_correctUrl {
    my ($self, undef, $node, $error_text) = @_;
    
    return $self->_check_url(undef,  $node, {}, $error_text);
}


sub _arr_itemsDenyAnotherWhenContainsAny {
    my ($self, $parameters, $node, $error_text) = @_;

    my @affected = grep {($_->{type} //'') eq 'any'} @$node;
    return if !@affected || @$node == 1;

    return (($error_text // $self->_default_error), 'inconsistentState');
}

sub _arr_itemsUnique {
    my ($self, $parameters, $node, $error_text) = @_;
    
    #Уникальность может считаться для разных наборов полей, поэтому данный constraint может принимать несколько наборов параметров
    $parameters = [$parameters] unless ref $parameters eq 'ARRAY';
    foreach my $prms (@$parameters) {
        my $affected;
        my $error = $prms->{error};
        if ($prms->{for}) {
            my $filter = $prms->{for};
            my @fields = keys %$filter;
            $affected = [ grep { my $item = $_; all { ($item->{$_} // '') eq $filter->{$_}} @fields } @$node ];
        }
        $affected //= $node;
        my $index = {};
        my @keys = @{$prms->{by}};
        foreach ( @$affected ) {
            my $hash = $self->_get_hash($_ => \@keys);

            return (($error // $error_text // $self->_default_error), 'itemsDuplicated') if $index->{$hash};
    
            $index->{$hash}++;
        }
    }
    return
}

sub _get_hash {
    my ($self, $data, $keys) = @_;
    
    my $string;
    foreach my $k (@$keys) {
        if (!defined $data->{$k}) {
            $string .=$k;
        }
        elsif(!ref $data->{$k}){
            $string .= $k.$data->{$k}
        }
        elsif(ref $data->{$k} eq 'ARRAY'){
            $string .= $k.join '',( sort {$a cmp $b} @{$data->{$k}} );
        }
        elsif(ref $data->{$k} eq 'HASH'){
             $string .= $k.join '',@{$data->{$k}}{ sort {$a cmp $b} keys %{$data->{$k}} };
        }
        else{
            $self->_throw(sprintf 'Unknown data type: %s', Data::Dumper->new([$data->{$k}], ['field'])->Dump);
        }
    }
    return url_hash_utf8($string);
}

sub _arr_itemsCountMax {
    my ($self, $itemsCountMax, $node, $error_text) = @_;
    $self->_throw(sprintf('Invalid itemsCountMax: %s', $itemsCountMax )) if $self->_check_integer(undef, $itemsCountMax);

    $node = [$node] unless ref $node;
    return if @$node <= $itemsCountMax;
    return (($error_text // $self->_default_error), 'limitExceeded');
}

sub _arr_itemsCountMin {
    my ($self, $itemsCountMin, $node, $error_text) = @_;
    $self->_throw(sprintf('Invalid itemsCountMin: %s', $itemsCountMin )) if $self->_check_integer(undef, $itemsCountMin);
    
    $node = [$node] unless ref $node;
    return if @$node >= $itemsCountMin;
    return ($error_text // $self->_default_error, 'limitExceeded');
}

sub _arr_itemAnyOf {
    my ($self, $parameters, $node, $error_text) = @_;

    $node = [$node] unless ref $node;
    $self->_throw(sprintf('Invalid parameters for constraint itemAnyOf: %s', Data::Dumper->new([$parameters], ['parameters'])->Dump))
        unless ref $parameters eq 'HASH' && exists $parameters->{variants} && defined $parameters->{mapBy};
    
    my $item_id_alias = $parameters->{itemIdAlias};
    
    my $variants = $parameters->{variants};
    $variants = $self->_explain_ref($variants) unless ref $variants;
    $self->_throw(sprintf('Invalid set of restrictions at itemAnyOf: %s', Data::Dumper->new([$variants], ['variants'])->Dump))
        unless ref $variants eq 'HASH' && scalar keys %$variants;
    
    my $field_name = $parameters->{mapBy};
    my ($error, $has_error, $error_type);

    my $i = 0;
    #Если в _state нет индекса - значит мы на верхнем уровне и будем писать туда свой
    my $can_switch_index = defined $self->_state->{index} || defined $self->_state->{field} ? 0 : 1;
    
    foreach my $item ( @$node ){
        $self->_next_index if $can_switch_index;

        #Если в схеме был указан алиас для id - сохраняем его в _shared, чтобы затем использовать при подстановке в тексты ошибок
        $self->_shared->{$item_id_alias} = $i + 1 if defined $item_id_alias;

        unless (exists $item->{$field_name}) {
            $error = $error_text // $self->_default_error;
        }
        else {
            my $subschema = $variants->{ $item->{$field_name} };

            unless ( defined $subschema ) {
                #Если для данного значения поля дополнительные ограничения не заданы и allowOnlyDefined:false - считаем его валидным,
                #т.к. оно уже прошло проверку по типу
                next unless $parameters->{allowOnlyDefined};
                $error = $error_text // $self->_default_error;
            }
            else {
                $self->_throw(sprintf('Invalid set of restrictions for value %s, field %s at itemAnyOf: %s', $item->{$field_name}, $field_name, Data::Dumper->new([$subschema], ['subschema'])->Dump))
                    unless ref $subschema;
            
                ($error, $error_type) = $self->_check_node($node, '/'.$i, $subschema);
            }

            if ( $error ) {
                $has_error = 1;
                $self->_set_error($error, $error_type);
                undef $error;
            }
        }
        $i++;
    }
    $self->_reset_index if $can_switch_index;
    
    return;
}

sub _set_error {
    my ($self, $error, $error_type) = @_;
    
    if ( defined $self->_state->{field} ){
        $self->_get_current_vr->add( $self->_state->{field} => $self->_mk_vr_error($error, $error_type));
    }
    else{
        $self->_get_current_vr->add_generic($self->_mk_vr_error($error, $error_type));
    }

    return;
}

sub _rng__extract_values {
    #Вспомогательный метод
    my ($self, $range) = @_;
    my @values = $range =~ /^(\d+(?:\.\d+)?)?-(\d+(?:\.\d+)?)?$/;

    return @values;
}

sub _rng_leftLessRight {
    my ($self, $parameters, $node, $error_text) = @_;
    my ($left, $right) = $self->_rng__extract_values($node);

    return if !defined $left || !defined $right;
    return if $left < $right;
    return (($error_text // $self->_default_error), 'invalidField');
}

sub _rng_boundaryMinValue {
    my ($self, $rangeMinValue, $node, $error_text) = @_;

    $self->_throw(sprintf('Invalid rangeMinValue: %s', $rangeMinValue )) if $self->_check_number(undef, $rangeMinValue);
    my ($left, $right) = $self->_rng__extract_values($node);
    
    return if (!defined $left || $left >= $rangeMinValue) && (!defined $right || $right ) >= $rangeMinValue;
    return (($error_text // $self->_default_error), 'invalidField');
}

sub _rng_boundaryMaxValue {
    my ($self, $rangeMaxValue, $node, $error_text) = @_;
    $self->_throw(sprintf('Invalid rangeMaxValue: %s', $rangeMaxValue )) if $self->_check_number(undef, $rangeMaxValue);
    my ($left, $right) = $self->_rng__extract_values($node);
    
    return if (!defined $left || $left <= $rangeMaxValue ) && (!defined $right || $right <= $rangeMaxValue);
    return (($error_text // $self->_default_error), 'invalidField');
}

sub _rng_boundaryPatternMatch {
    my ($self, $pattern, $node, $error_text) = @_;

    my ($left, $right) = $self->_rng__extract_values($node);

    return if (!defined $left || $left =~ qr|$pattern| ) && (!defined $right ||  $right =~ qr|$pattern|);
    return (($error_text // $self->_default_error), 'invalidField');
}

#--------------------------------------------------------------------------------------------

__PACKAGE__->meta->make_immutable;

1;
