package Forecast;
## no critic (TestingAndDebugging::RequireUseWarnings)

# $Id$
# подсчёт прогноза бюджета кампании
# Автор: Сергей Журавлёв

use strict;
use vars qw(@ISA @EXPORT @EXPORT_OK);
require Exporter;

use Data::Dumper;
use HashingTools;
use TextTools;
use List::MoreUtils qw/any uniq/;
use List::Util qw/sum/;
use Yandex::Clone qw/yclone/;

use Yandex::I18n;

use Settings;
use Yandex::DBTools;
use Yandex::Trace;
use GeoTools;
use Currencies;
use Direct::Validation::Keywords qw/validate_forecast_keywords/;
use Suggestions;
use Yandex::HashUtils;
use Yandex::ListUtils qw/chunks/;
use ADVQ6;
use BS::TrafaretAuction;
use Pokazometer;
use MinusWords;
use PhraseText;
use PlacePrice qw/PREMIUM_PLACE_COUNT GUARANTEE_PLACE_COUNT/;

use utf8;

# what advq we should use
# possibilities: undef, 'bs', 'old'
# undef - default behaviour: bs for clients, old for other

# set the version for version checking
@ISA         = qw(Exporter);
@EXPORT_OK = @EXPORT = qw(
        forecast_shows
        forecast_calc
        forecast_calc_new
        forecast_get_camp
        forecast_calc_camp
        get_geo_k
);

my $ADVQ = 'http://direct.advq.yandex.ru/advq';
my $CTR_COEFF_FOR_GUARANTEE_PLACES = [1.637883592, 1.215571687, 1.075717721, 1];
my $CTR_COEFF_FOR_PREMIUM_PLACES = [1.32999876, 1.150062297, 1, 1];
my $CTR_COEFF_FOR_SITELINKS = 1.2;

our $MAX_PHRASES_NUM ||= 1000;

=head2 test_phrases

    test and validate phrases for forecast

    Параметры позиционные:
        $phrases -- ссылка на массив фраз

    Возвращаемое значение
      Если нашлась ошибка: 
        { 
            error => '...'     # список сообщений об ошибках через точку с запятой
        }
      Если фразы нормальные: 
        {
            phrases => '...'   # список фраз через запятую 
        }

    TODO и ошибки, и фразы возвращать массивами, а не склеенной строкой

=cut

sub test_phrases
{
    my ($phrases) = @_;

    my @result;

    push @result, iget('Не заданы ключевые фразы') if !@$phrases;
    push @result, iget('Превышено допустимое количество фраз') if @$phrases > $MAX_PHRASES_NUM;

    my $validation_result = validate_forecast_keywords($phrases);
    push @result, @{$validation_result->one_error_description_by_objects} unless $validation_result->is_valid; 

    my $error_mess = join ";\n", map {html2string($_)} @result;
    if ($error_mess) {
        return {error => $error_mess};
    } else {
        return {phrases => join(',', @$phrases)};
    }
}



=head prepare_phrases_for_forecast

    Готовит фразы для подсчета прогноза: из списка через запятую/перевод строки делает аккуратный массив фраз.

    Параметры позиционные 
        $str -- (неаккуратный) список фраз через запятую/перевод строки, с лишними пробелами, повторяющимися минусами и т.п.
                Также может содержать оператор "(|)"

    Возвращаемое значение 
        ссылка на массив фраз без лишних пробелов, со склеенными минусами и раскрытыми скобками


    TODO Дать имя получше, и использовать не только в прогнозе

=cut

sub prepare_phrases_for_forecast
{
    my ($str) = @_;

    $str ||= '';

    # DIRECTINCIDENTS-122 квадратные скобки тяжелы в обработке в advq, временно выкидываем 
    $str =~ s/[\[\]]/ /g;

    $str =~ s/\s*[\,\n][\n\,\s*]*/,/g;
    $str =~ s/^[\s\,]+|[\s\,]+$//g;
    $str =~ s/-\s+/ -/g;
    $str =~ s/-+/-/g;
    $str =~ s/\s+/ /g;

    smartstrip($str);

    $str =~ s/(,[,\s]*)/,/gs;
    $str =~ s/[,\s]+$//g;
    $str =~ s/\- +/-/g;
    $str =~ s/\s*"\s*/"/g;
    $str =~ s/!\s+/!/g;
    $str =~ s/\+\s+/+/g; 

    process_phrase_brackets($str);

    return [ uniq split ",", $str ];
}


=head forecast_shows

   Посчитать прогнозируемое число показов для заданных фраз и категорий

   forecast_shows([$block])

   $block = {
       geo=>"0,-3",
       Phrases=[
           {
               phrase  => "windows",
           },
           {
               phrase  => "apple",
           },
       ],
       camp_minus_words => "-word -word",
       banner_minus_words => "-word -word",
       predefined_shows    => 0, # true if month shows are defined for each element of Phrases
   }

=cut
sub forecast_shows {

    my ( $blocks, %O ) = @_;

    my $profile = Yandex::Trace::new_profile('forecast:forecast_shows', obj_num => scalar @$blocks);

    for my $block ( @$blocks ) {

        my $geo = $block->{geo} || 0;

        # Нормализуем
        for my $ph ( @{$block->{Phrases}} ) {
            my $props = get_phrase_props($ph->{phrase}) || next;
            hash_merge $ph, hash_cut $props, qw/norm_phrase/;
        }

        #   Не пересчитывать прогноз показов если показы уже вычислены заранее
        $block->{advq_result} = 'ok';
        unless ( $block->{predefined_shows} ) {
            # Считаем статистику по фразам с advq
            my @phrases = @{$block->{Phrases}};
            get_advq_shows_of_phrases(\@phrases, $block, $geo, period => $O{period}, lang => $O{lang}, devices => $O{devices});
        }

        # коэфицент временного таргетинга        
        my $tt_koef = 1;
        # Проставляем нули. Если нужно - учитываем временной таргетинг
        foreach my $ph ( @{$block->{Phrases}}) {
            my $i_koef = 1;

            $ph->{shows} ||= 0;
            $ph->{shows} = int( $tt_koef * $ph->{shows} * $i_koef + 0.5 );
        }

        #print STDERR "  2 >>:".Dumper($block)."\n";
    }
    return '';
}


=head2 forecast_calc

 Функция производит подсчет прогноза цены, показов и пр. для фраз в баннерах. 

 Параметры позиционные: 
     $blocks -- ссылка на массив баннеров

 Параметры именованные: 
     period => { year => 1 } # или { month => <1 .. 12>} или { quarter => <1..4>}
               необязательный. 
               Если не указан -- прогноз считается по статистике за последние 30 дней. 
               Если указан -- прогноз корректируется на указанный квартал/месяц
               Поправочные коэффициенты для периодов рассчитываются в advq_get_time_coef
             Для рубрик сезонность не учитывается (пока неясно, как), 
             для квартала/года просто умножаем на 3|12.

    die_if_nobsdata => 1 -- падать если недоступны торги БК

       каждый баннер - хэш с параметрами, который должен иметь по возможности по максимуму заполнеными следующие поля:
         - OrderID
         - geo 
         - filter_domain
         - phone
         - fairAuction
         - title
         - body
         - currency        -- валюта; обязательно к заполнению
         - Phrases - массив фраз
           каждая фраза - хэш с полями:
             - phrase - текст фразы или номер категории каталога
             - PriorityID
             - PhraseID
             - phraseIdHistory

  Чем больше заполнено полей, тем точнее прогноз.

  Если прогнозы различаются для одного баннера, проверьте заполненность полей.
  
  %O
    lang - для какого языка считать прогноз (tr - для турецкого)

=cut

sub forecast_calc {
    #   Эта функция меняет структуру массива $block->{Phrases}, добавляя в него данные о категориях
    #   изменённые данные о фразах - в $block->{phrases}

    my ( $blocks, %O ) = @_;
    delete $O{period} if defined $O{period} && $O{period}->{week};
    my %campaigns;
    for my $block ( @$blocks ) {
        # printf STDERR "GEO: $block->{geo} / %s\n", join(" / ", map {$_->{phrase}} @{ $block->{Phrases} });
        #print STDERR "  1 >>:".Dumper($block)."\n";

        if ($block->{cid} && !exists $campaigns{$$block{cid}}) {
            my $words = MinusWords::get_words(type => 'minus', cid => $block->{cid});
            $campaigns{$$block{cid}} = @$words ? MinusWords::polish_minus_words_array($words) : undef;
        }

        $block->{camp_minus_words} = $campaigns{$$block{cid}} if $block->{cid} && $campaigns{$$block{cid}};
        forecast_shows([$block], period => $O{period}, lang => $O{lang});

        my $currency = $block->{currency};
        die 'no currency given' unless $currency;
        my $min_price_constant = get_currency_constant($currency, 'MIN_PRICE');
        my $max_price_constant = get_currency_constant($currency, 'MAX_PRICE');

        # Считаем цены в крутилке
        foreach my $ph ( @{$block->{Phrases}} ) {
            $block->{OrderID} ||= $ph->{orderid};
            ensure_phrase_have_props($ph);

            $ph->{price} = $max_price_constant;
            $ph->{phr} = $ph->{phrase};
        }

        my $all_ctr_data = PhraseText::mass_get_ctr_data([map {$_->{norm_hash}} @{$block->{Phrases}}]);
        foreach my $ph ( @{$block->{Phrases}} ) {
            my $ctr_data = $all_ctr_data->{$ph->{norm_hash}};
            my ($ctr, $p_ctr) = (0, 0);
            if (ref($ctr_data) eq 'HASH') {
                $ctr = $ctr_data->{ctr} || 0;
                $p_ctr = $ctr_data->{p_ctr} || 0;
            }
            $ph->{stored_ctr} = $ctr;
            $ph->{stored_p_ctr} = $p_ctr;
        }

        # !!! del-rubrics: тут phrases из строки становятся массивом
        $block->{Phrases} = $block->{phrases} = [ @{$block->{Phrases}} ];
        $block->{is_bs_rarely_loaded} = 0;
    }
    trafaret_auction($blocks, { calc_rotation_factor => 1, dont_use_trafaret => 1, forecast_without_banners => $O{forecast_without_banners} } );

    for my $block (@$blocks) {
        if (any {$_->{nobsdata}} @{$block->{phrases}}) {
            if ($O{die_if_nobsdata}) {
                die "forecast_calc() -> trafaret_auction() failed" ;
            } else {
                $block->{not_full_bsdata} = 1;
                last;
            }
        }
    }

    for my $block ( @$blocks ) {
        my $currency = $block->{currency};
        die 'no currency given' unless $currency;
        my $min_price_constant = get_currency_constant($currency, 'MIN_PRICE');
        for my $ph (@{$block->{Phrases}}) {
            # Если есть взвешенные  цены - проставляем их
            foreach(0..$PlacePrice::PREMIUM_PLACE_COUNT-1) {
                hash_merge $ph->{premium}->[$_], $ph->{weighted}->{premium}->[$_];
            }
            foreach(0..$PlacePrice::GUARANTEE_PLACE_COUNT-1) {
                hash_merge $ph->{guarantee}->[$_], $ph->{weighted}->{guarantee}->[$_];
            }
                        
            # Вычисляем CTR
            my $ctr;
            if ( $ph->{real_rshows} && $ph->{real_rshows} >= 100 ) {
                $ctr = ( $ph->{real_rclicks} || 0.5 ) / $ph->{real_rshows};
            } else {
                $ctr = $ph->{stored_ctr} || 0;
                unless ( $ctr > 0.001 && $ph->{shows} > 100 && $ph->{shows} * $ctr > 1 ) {
                    $ctr = $ph->{ectr};
                }
            }
            my $p_ctr;
            if ( $ph->{real_pshows} && $ph->{real_pshows} >= 100 ) {
                $p_ctr = ( $ph->{real_pclicks} || 0.5 ) / $ph->{real_pshows};
            } else {
                $p_ctr = $ph->{stored_p_ctr} || 0;
                unless ($p_ctr > 0.001 && $ph->{shows} > 100 && $ph->{shows} * $p_ctr > 1) {
                    $p_ctr = $ph->{pectr};
                }
            }
            my $clicks = int( $ph->{shows} * $ctr + 0.5 );
            my $p_clicks = int( $ph->{shows} * $p_ctr + 0.5 );
            # cool trick, заново пересчитываем ctr
            $ctr = $ph->{shows} ? $clicks * 100 / $ph->{shows} : 0;
            $p_ctr = $ph->{shows} ? $p_clicks * 100 / $ph->{shows} : 0;

            $ph->{ctr} = $ctr;
            $ph->{p_ctr} = $p_ctr;

            $ph->{premium_clicks} = $p_clicks;
            $ph->{clicks} = $clicks;
            #Разбираем по позициям:
            # 1СР, 2CP, 3CP
            my $need_use_ctr_coef = $ph->{premium}->[0]->{amnesty_price} != $ph->{premium}->[-1]->{amnesty_price};
            foreach (0..$PlacePrice::PREMIUM_PLACE_COUNT - 1) {
                my $px = $ph->{premium}->[$_];
                $px->{ctr} = $p_ctr *  ( $need_use_ctr_coef ? $CTR_COEFF_FOR_PREMIUM_PLACES->[$_] : 1);
                # модель сайтлинков: ctr *= 1.2
                $px->{ctr} *= ( $block->{consider_sitelinks_ctr} ? $CTR_COEFF_FOR_SITELINKS : 1);
                $px->{clicks} = int($ph->{shows} * $px->{ctr} / 100 + 0.5);
            }

            # 1ГП, 2ГП, 3ГП, 4ГП: 
            # fp_ctr второй раз пересчитывается с округленными кликами,
            # чтобы его значение совпадало с тем которое пользователь может самостоятельно получить
            # по формуле используя клики которые мы показываем в отчете
            foreach (0 .. $PlacePrice::GUARANTEE_PLACE_COUNT - 1) {
                my $px = $ph->{guarantee}->[$_];
                my $ctr_draft = $ctr * $CTR_COEFF_FOR_GUARANTEE_PLACES->[$_];
                # модель сайтлинков: ctr *= 1.2
                $ctr_draft *= ( $block->{consider_sitelinks_ctr} ? $CTR_COEFF_FOR_SITELINKS : 1);
                $px->{clicks} = int($ph->{shows} * $ctr_draft / 100 + 0.5);
                $px->{ctr} = $ph->{shows} ? $px->{clicks} * 100 / $ph->{shows} : 0;
            }
            $ph->{first_place_clicks} = $ph->{guarantee}->[0]->{clicks};
            $ph->{first_place_ctr} = $ph->{shows} ? $ph->{first_place_clicks} * 100 / $ph->{shows} : 0;

            # Считаем суммы
            $block->{shows} += $ph->{shows};
            $block->{clicks} += $ph->{clicks};
            # TODO DIRECT-67003 прокидывать тип группы == mcbanner, для фильтрации позиций показа
            foreach my $pos (@{PlacePrice::get_usedto_show_places()}) {
                my $position = PlacePrice::get_data_by_place($ph, $pos);
                my $price = $min_price_constant;
                my $clicks = $position->{clicks};
                if ($block->{max_bid} && $block->{max_bid} < $position->{amnesty_price} / 1e6) {
                    foreach ( reverse(@{PlacePrice::get_usedto_show_places()}) ){
                        my $pl = PlacePrice::get_data_by_place($ph, $_);
                        if ($pl->{amnesty_price} <= $block->{max_bid}) {
                            $price = $pl->{amnesty_price};
                            $clicks = $pl->{clicks};
                        }
                    }
                } else {
                    $price = $position->{amnesty_price};
                }
                $position->{calculated_price}  = $price;
                $position->{calculated_clicks} = $clicks;
            }
        }

        foreach my $i (0..$PlacePrice::PREMIUM_PLACE_COUNT-1) {
            $block->{premium}->[$i] = {price => sum(map {$_->{premium}->[$i]->{calculated_price} * 
                                                         $_->{premium}->[$i]->{calculated_clicks}
                                                        } @{$block->{Phrases}}),
                                       clicks => sum(map {$_->{premium}->[$i]->{calculated_clicks}} @{$block->{Phrases}})
                                      };
        }
        foreach my $i (0..$PlacePrice::GUARANTEE_PLACE_COUNT-1) {
            $block->{guarantee}->[$i] = {price => sum(map {$_->{guarantee}->[$i]->{calculated_price} * 
                                                         $_->{guarantee}->[$i]->{calculated_clicks}
                                                        } @{$block->{Phrases}}),
                                         clicks => sum(map {$_->{guarantee}->[$i]->{calculated_clicks}} @{$block->{Phrases}})
                                        };
        }
    }

    # Если требуется -- запрашиваем данные Показометра
    if ( $O{calc_context_forecast} ){
        # если блок один - разбиваем его на несколько, распределяя фразы пачками (данные показометр добавляет только во фразы)
        # после получения результатов - сворачиваем разбитые блоки обратно в один
        my $single_block_is_splited = 0;
        if (@$blocks == 1) {
            $single_block_is_splited = 1;
            my $phrases = $blocks->[0]->{phrases};
            delete $blocks->[0]->{$_} for qw/phrases Phrases/;
            my $block_sample = $blocks->[0];
            for my $phrases_chunk (chunks $phrases, 100) {
                unless (exists $blocks->[0]->{phrases}) {
                    # первый (исходный) блок еще не наполнен фразами
                    $blocks->[0]->{Phrases} = $blocks->[0]->{phrases} = $phrases_chunk;
                } else {
                    my $block = yclone($block_sample);
                    $block->{Phrases} =$block->{phrases} = $phrases_chunk;
                    push @$blocks, $block;
                }
            }
        }
        get_pokazometer_data($blocks, get_all_phrases => 1, period => $O{period}, consider_minus_words => 1);
        get_pokazometer_data($blocks, get_all_phrases => 1, period => $O{period}, net => 'search', pokazometer_field => 'pokazometer_data_search', consider_minus_words => 1);
        get_pokazometer_data($blocks, get_all_phrases => 1, period => $O{period}, net => 'context', pokazometer_field => 'pokazometer_data_context', consider_minus_words => 1);

        if ($single_block_is_splited) {
            for my $i (1 .. $#{$blocks}) {
                push @{$blocks->[0]->{phrases}}, @{$blocks->[$i]->{phrases}};
                delete $blocks->[$i]->{$_} for qw/phrases Phrases/;
            }
            @$blocks = ($blocks->[0]);
        }
        
        for my $block (@$blocks){
            last unless defined $O{period};
            my $chrono_table = ADVQ6::advq_get_time_coef(geo => $block->{geo}, phrases => [ map { $_->{phrase} } @{$block->{Phrases}} ], period => $O{period});
            for my $ph ( @{$block->{Phrases}} ){ 
                for my $pokazometer_field (qw/pokazometer_data pokazometer_data_search pokazometer_data_context /){
                    correct_pokazometer_data_with_chrono( $ph->{$pokazometer_field}, $chrono_table );
                }
            }
        }
    }

    return @$blocks;
}

=head2 forecast_calc_new

    Функция производит подсчет прогноза ставок, показов, кликов и бюджета для фраз в баннерах.

    Основные результаты по каждой фразе пишутся в shows (кол-во прогнозируемых
    хитов по advq) и positions (прогнозируемые бюджеты в зависимости от
    позиции).

    %O
        filter_duplicate_md5
        фильтрация дубликатов по md5, предназначено для get_new_advanced_forecast
        (т.к. новый прогнозатор учитывает пересечение хитов между фразами, то после
        расчёта бюджета фильтровать уже поздно, а на странице вычисления бюджетов
        md5 используется в качестве ключа и дубликаты нужно убирать)

        distribution
        расчёт распределения бюджета по ставкам, а не позициям

        positions
        позиции по которым нужно возвращать данные

        with_nds
        возвращать ставки с НДС или вычесть

=cut

sub forecast_calc_new {
    # Эта функция меняет структуру массива $block->{Phrases}, добавляя в него
    # данные о категориях, изменённые данные о фразах - в $block->{phrases}
    my ( $blocks, %O ) = @_;
    my %campaigns;
    for my $block ( @$blocks ) {
        if ($block->{cid} && !exists $campaigns{$$block{cid}}) {
            my $words = MinusWords::get_words(type => 'minus', cid => $block->{cid});
            $campaigns{$$block{cid}} = @$words ? MinusWords::polish_minus_words_array($words) : undef;
        }

        $block->{camp_minus_words} = $campaigns{$$block{cid}} if $block->{cid} && $campaigns{$$block{cid}};

        # Считаем показы advq, у этого есть два предназначения:
        # - посчитать коэффициенты поправок на период
        # - фразы сортируются по кол-ву показов
        forecast_shows([$block], period => $O{period}, lang => $O{lang}, devices => $O{devices});

        my $currency = $block->{currency};
        die 'no currency given' unless $currency;
        $block->{advq_min_bid} = get_currency_constant($currency, 'MIN_PRICE');
        $block->{advq_max_bid} = get_currency_constant($currency, 'MAX_PRICE');

        my %used_md5;
        my $phrases = [];
        foreach my $ph ( @{$block->{Phrases}} ) {
            $block->{OrderID} ||= $ph->{orderid};
            ensure_phrase_have_props($ph);
            if ($O{filter_duplicate_md5}) {
                # Убираем дубликаты по md5, т.к. новый прогнозатор учитывает пересечения
                # и наличие дубликатов существенно меняет прогноз, кроме того это
                # избавляет от необходимости фильтровать по md5 в результатах для
                # интерфейса.
                next if $used_md5{$ph->{md5}}++;
            }
            push @$phrases, $ph;
        }

        my $all_ctr_data = PhraseText::mass_get_ctr_data([map {$_->{norm_hash}} @$phrases]);
        foreach my $ph (@$phrases) {
            my $ctr_data = $all_ctr_data->{$ph->{norm_hash}};
            my ($ctr, $p_ctr) = (0, 0);
            if (ref($ctr_data) eq 'HASH') {
                $ctr = $ctr_data->{ctr} || 0;
                $p_ctr = $ctr_data->{p_ctr} || 0;
            }
            $ph->{stored_ctr} = $ctr;
            $ph->{stored_p_ctr} = $p_ctr;
        }

        # Сохраняем в блоке изменённый набор фраз
        $block->{Phrases} = $block->{phrases} = $phrases;
    }

    if ($O{distribution}) {
        ADVQ6::advq_get_phrases_budget_dist_forecast_multi(
            $blocks, lang => $O{lang}, devices => $O{devices}, with_nds => $O{with_nds}
        );
    } else {
        ADVQ6::advq_get_phrases_multibudget_forecast_multi(
            $blocks, lang => $O{lang}, devices => $O{devices}, positions => $O{positions}, with_nds => $O{with_nds},
        );
    }

    return @$blocks;
}

=head2 correct_pokazometer_data_with_chrono

    Получает Показометровые данные (от одной фразы) и хроно-таблицу (полученную из ADVQ).
    Умножает все клики и показы на хроно-коэффициент.

=cut
sub correct_pokazometer_data_with_chrono
{
    my ( $pokazometer_data, $chrono_table ) = @_;
    my $profile = Yandex::Trace::new_profile('forecast:correct_pokazometer_data_with_chrono');

    my $c = $chrono_table->{$pokazometer_data->{query}}->{coef}; 
    $_->{cnt} *= $c for @{$pokazometer_data->{shows_list}};
    $_->{cnt} *= $c for @{$pokazometer_data->{clicks_list}};
    $_->{clicks} *= $c for @{$pokazometer_data->{complete_list}};
    $_->{shows} *= $c for @{$pokazometer_data->{complete_list}};

    $pokazometer_data->{$_} *= $c for qw/clicks_cnt shows_cnt/; 

    return;
}

=head2 clear_block_forecast

    секция "some calcs and clears" из cmd_calcForecast 

=cut
sub clear_block_forecast
{
    my ($block) = @_;

    # some calcs and clears
    my $result;
    my %H;
    for my $row (@{$block->{Phrases}}) {
        my $cleared_row = {
            md5 => $row->{md5},
            sign => md5_hex_utf8(($row->{phrase} ||  ''). $Settings::SECRET_PHRASE),
        };
        hash_copy $cleared_row, $row, qw/
            phrase
            guarantee
            premium
            shows
            category_name
            category_url
            currency
        /;
        foreach (@{$cleared_row->{premium}},@{$cleared_row->{guarantee}}) {
            $_->{sum} = $_->{clicks} * $_->{amnesty_price};
        }
        
        push @$result, $cleared_row unless $H{$cleared_row->{md5}};
        $H{$cleared_row->{md5}} = 1;
    }

    return $result;
}

=head2 forecast_get_camp

    достать подробные данные для прогноза (цены/показы/ctr) по кампании (подробно по объявлениям и фразам)
    Параметры: 
    $cid, $max_bid 
    $options -- хеш дополнительных (необязательных) параметров: 
        $options->{get_all_categories} -- флаг (0 или не-0), если взведен, то обрабатываются все рубрики 
            (по умолчанию пропускаем рубрики первого и второго уровня - там может быть медийка);
        $options->{banners_filter} -- какие объявления отбирать для вычисления прогноза
            'forecast_easy' -- для прогноза на странице чайниковской кампании (=> все, кроме остановленных и отклоненных)
            'active_only' (по умолчанию) -- только активные объявления (для Мастера Цен, для ppcAutobudgetForecast.pl и т.п.)
        $options->{die_if_nobsdata} -- падать если недоступны торги БК, нужно для того чтобы правильно рассчитывать прогноз бюджета

    Результат: 
    ссылка на хеш с подробной информацией по фразам из объявлений
        {
            3855682 => {
                Phrases => [
                    {
                        Clicks     => undef,
                        Ctr        => 0.00,
                        PhraseID   => 4370387,
                        PriorityID => 1625190,
                        Shows      => 370436,
                        ... 
                    },
                    ...
                ],
                advq_result => ok,
                advq_timeout => 300,
                categories => [],
                clicks => 9089,
                filter_domain => 'www.yandex.ru',
                geo => '225,166',
                currency => 'YND_FIXED',
                ...
            },
            ...
        }

=cut
sub forecast_get_camp {
    my ( $cid, $max_bid, $options ) = @_;
    my $profile = Yandex::Trace::new_profile('forecast:forecast_get_camp');
    $max_bid ||= 20;
    # получаем из базы данные по кампании
    my $sth = exec_sql( PPC(cid => $cid), "SELECT STRAIGHT_JOIN b.bid, p.geo, bi.phrase
                                    , bi.id, p.PriorityID, b.BannerID, bi.PhraseID, c.orderid
                                    , bph.phraseIdHistory
                                    , bi.price as real_price
                                    , c.timeTarget, c.timezone_id
                                    , ifnull(fd.filter_domain, b.domain) filter_domain
                                    , vc.phone, b.title, b.body
                                    , p.statusShowsForecast, bi.showsForecast as shows
                                    , IFNULL(c.currency, 'YND_FIXED') as currency
                                    , p.pid
                                    , FIND_IN_SET('no_extended_geotargeting', c.opts)>0 as no_extended_geotargeting
                                 FROM campaigns c 
                                      join phrases p on p.cid = c.cid
                                      join banners b on p.pid = b.pid
                                      left join vcards vc on vc.vcard_id = b.vcard_id
                                      join bids bi on bi.pid = p.pid
                                      left join bids_phraseid_history bph on bph.cid = bi.cid and bph.id = bi.id
                                      left join bs_auction_stat auct on auct.pid = p.pid and auct.PhraseID = bi.PhraseID 
                                      left join filter_domain fd on fd.domain = b.domain
                                WHERE c.cid = ?
                                  AND b.statusArch = 'No'
                                  AND (
                                      b.statusActive = 'Yes'
                                      OR (b.statusModerate <> 'No' AND b.statusShow = 'Yes' AND p.statusModerate <> 'No' AND bi.statusModerate <> 'No')
                                  )
                                  AND IFNULL(auct.rank, 1) > 0", $cid );
    my %BANNERS;
    while( my $row = $sth->fetchrow_hashref ) {
        next if $options->{exceptional_bid} && $row->{bid} == $options->{exceptional_bid};

        my $banner = $BANNERS{$row->{bid}} ||= { phrases => [],
                                               , max_bid => $max_bid
                                           };
        hash_copy $banner, $options, qw/allow_emergency_shows_forecast/;
        hash_copy $banner, $row, qw/geo timeTarget title body filter_domain phone currency bid pid BannerID no_extended_geotargeting/;

        $banner->{advq_timeout} = 300;
        $row->{phrase} =~ s/!/+/g;
        push @{ $banner->{Phrases} }, $row;
        if ($row->{statusShowsForecast} eq 'Processed') {
            $banner->{predefined_shows} = 1;
        }
    }
    # считаем прогноз
    forecast_calc([values %BANNERS], calc_context_forecast => $options->{calc_context_forecast}
                                         , die_if_nobsdata => $options->{die_if_nobsdata}
                 );

    if ( $ENV{DEBUG_FORECAST} ) {
        print STDERR Dumper \%BANNERS;
    }
    return \%BANNERS;

}

# Посчитать прогноз для кампании
sub forecast_calc_camp {

    my ($cid, $max_bid, $options) = @_;

    $options->{calc_context_forecast} = 1 if $options->{for_net};
    my $banners = forecast_get_camp($cid, $max_bid, $options);

    my %res;
    if ($options->{for_net}) {
        my ($sum, $clicks) = (0, 0);
        foreach (values %$banners) {
            foreach (@{$_->{Phrases}}) {
                
                my $context = $_->{pokazometer_data_context}->{complete_list}->[-1];
                next unless $context;
                $context->{cost} /= 1E6;
                next if $context->{cost} > $max_bid;
                
                $sum += $context->{cost} * $context->{clicks};
                $clicks += $context->{clicks};
            }
        }
        %res = (sum => $sum, clicks => $clicks);
    } else {
        foreach my $i (0..$PlacePrice::PREMIUM_PLACE_COUNT-1) {
            $res{premium}->[$i] = {price  => sum(map {$_->{premium}->[$i]->{price}} values %$banners) // 0,
                                   clicks => sum(map {$_->{premium}->[$i]->{clicks}} values %$banners) // 0,
                                  };
        }
        foreach my $i (0..$PlacePrice::GUARANTEE_PLACE_COUNT-1) {
            $res{guarantee}->[$i] = {price  => sum(map {$_->{guarantee}->[$i]->{price}} values %$banners) // 0,
                                     clicks => sum(map {$_->{guarantee}->[$i]->{clicks}} values %$banners) // 0,
                                    };
        }

        $res{shows}= sum(map {$_->{shows}} values %$banners);

    }
    
    return \%res;
}



#-----------------------------------------------------------

=head2 get_advq_shows_of_phrases

    get_advq_shows_of_phrases($phrases_arr_ref, $block, $geo, %O);

    $phrases_arr_ref:
          [
            {
              'norm_phrase' => 'qwerty',
              'phrase' => 'qwerty',
            },
            {
              'norm_phrase' => 'asd',
              'phrase' => 'asd',
            }
          ]

    added 'shows'

    %O = (
        minus_words =>
        period =>
        lang =>
        devices =>
    )

=cut

sub get_advq_shows_of_phrases
{
    my ($phrases, $block, $geo, %O) = @_;
    return if ref($phrases) ne 'ARRAY' || ! @$phrases;
    my $profile = Yandex::Trace::new_profile('advq:get_advq_shows_of_phrases', obj_num => scalar @$phrases);

    my @potential_minus_words;
    for my $mwords (grep { defined $_ && @$_ } ($O{minus_words}, $block->{minus_words}, $block->{banner_minus_words}, $block->{campaign_minus_words}, $block->{camp_minus_words})) {
        push @potential_minus_words, @$mwords;
    }
    $block->{advq_result} = ADVQ6::advq_get_phrases_shows( $phrases, $geo, timeout => $block->{advq_timeout} || 30, 
                                                                           period  => $O{period},
                                                                           lang => $O{lang},
                                                                           devices => $O{devices},
                                                                           minus_words => \@potential_minus_words,
                                                         );
    
    $_->{shows} = delete $_->{showsForecast} foreach @$phrases;
    $_->{shows_time_coef} = delete $_->{showsTimeCoef} foreach @$phrases;

    # get shows_forecast from Direct DB
    if($block->{allow_emergency_shows_forecast} && $block->{advq_result} ne 'real') {   
        # in case of 'nonreal' ADVQ response - for ADVQ3
        emergency_get_shows_of_phrases($phrases, $geo)
    }
}

=head2 get_geo_k("1,-213")

    По строке, содержащей гео-таргетинг получить отношение показов серпа,
    попадающих в таргетинг ко всем показам
    
=cut
{
my %CACHE;
sub get_geo_k {
    my $geo = shift;
    if (!defined $CACHE{$geo}) {
        my $ch = join(",", get_geo_children($geo, {tree => 'ua'}));
        $CACHE{$geo} = get_one_field_sql( PPCDICT, ["
                            SELECT sum(coef) 
                              FROM serp_geo_stat gs
                                   JOIN geo_translation gt on gt.region = gs.region
                            ", WHERE => {'gt.direct_region' => [get_geo_children($geo, {tree => 'ua'})]}
                              ]);
        $CACHE{$geo} = 0 if !$CACHE{$geo} || $CACHE{$geo} < 0.00001;
    }
    return $CACHE{$geo};
}
}


=head2 emergency_get_shows_of_phrases

    аварийный прогноз показов: показы из подсказочных таблиц, география из get_geo_k

    Нужен, например, для Ускоренного интерфейса. Там без прогноза даже объявления не посмотреть.

=cut
sub emergency_get_shows_of_phrases
{
    my ($phrases, $geo) = @_;
    my $profile = Yandex::Trace::new_profile('forecast:emergency_get_shows_of_phrases');
    $geo ||= 0;
    #print STDERR "emergency_get_shows_of_phrases\n";

    get_phrases_info_from_suggest($phrases, {shows_only => 1});
    my $geo_k = get_geo_k($geo);
    $_->{shows} *= $geo_k for @$phrases;

    return;
}

1;
