package Forecast::Autobudget;
## no critic (TestingAndDebugging::RequireUseWarnings)

# $Id$

=head1 NAME
    
    Forecast::Autobudget

=head1 DESCRIPTION

    Прогноз работы автобюджета

    Применения, хронологически: 
      * Мастер Цен (для менеджеров, как-то не очень успешно...)
      * Баблометр (moneymeter, слайдер, бегунок) в Легком интерфейсе
      * Легкие стратегии -- расчеты бюджетов

=head1 SYNOPSIS
    
    use Forecast::Autobudget;

    # ...создание Легкого объявления...
    # Если объявление -- не первое, то добавляем информацию для Баблометра (по всем остальным объявлениям в кампании)
    my $opts = {
        cid => $cid, 
        purpose => 'slider',
        banners_filter => 'forecast_easy',
        mark_trans_except_bid => 1,
        exceptional_bid => 0,
        currency => 'RUB',
    };
    my $transitions_for_slider = get_moneymeter_data($opts);
    $vars->{transitions_for_slider} = to_json($transitions_for_slider);

=cut


# TODO 
# объединить получение прогноза и расчет ctr'ов из get_moneymeter_data 
# исправить расчет ctr'ов и покрытий по ставке
# отнести пересчет в рубли в prepare_phrases_prices_ctrs
# 

use strict;
use vars qw(@ISA @EXPORT @EXPORT_OK);
require Exporter;

use Settings;
use Forecast;
use AutoBroker;
use HashingTools;
use Yandex::ExecuteJS;
use Lang::Unglue;
use Phrase;
use Currencies;
use Currency::Rate;
use Campaign;
use MinusWordsTools;
use MinusWords;

use Encode;
use List::Util qw/max min sum/;
use POSIX qw/ceil/;

use Yandex::HashUtils;
use Yandex::ListUtils;
use Yandex::Trace;

use utf8;

# таймайт в баблометре при запросе advq
our $MONEYMETER_ADVQ_TIMEOUT = 10;

# set the version for version checking
@ISA         = qw(Exporter);
@EXPORT_OK = @EXPORT = qw( 
    get_moneymeter_data 
    get_new_advanced_forecast
    calc_autobudget_clicks_by_avg_price
);


# Прогноз кликов в сети ограничиваем сверху как клики на Поиске * коэфф. 
# убрать ограничение на Показометр -- выставить здесь огромное значение, например, 1000
# см. также ./js/js-perl/AdvancedForcast-p.js, MAX_YAN_TO_YANDEX_CLICKS_RATIO
our $MAX_YAN_TO_YANDEX_CLICKS_RATIO = 1.2;


# По фразам собираем "переходы", отдельно для каждой фразы, в правильном порядке. 
# !!! Описать параметры и результат.
sub fc_get_raw_transitions{
    my ($phrases, %O) = @_;
    my $profile = Yandex::Trace::new_profile('autobudget:fc_get_raw_transitions', obj_num => scalar keys %$phrases);
    $O{rec_pos} ||= "std";

    my @transitions;
    my $raw_transitions = {};

    while(my ($key, $p) = each(%$phrases)) {
        #Позицию, на которой сначала находятся все доступные показы, можно указывать вручную         
        #если ни на какую позицию показы не приписаны -- записываем все на 'none'
        #
        #Проверяем, записаны ли на какую-то позицию показы. 
        #Если все доступные показы ($p->{total_shows}) уже расписаны по позициям -- все хорошо, ничего менять не надо. 
        #Если не все показы расписаны, то создаем позицию 'none' (не-показы), и записываем все нераспределенные показы на нее.
        my @positions = keys %{$p->{positions}};
        $p->{positions}->{$_}->{shows} ||= 0 for (@positions);
        my $init_shows = 0;
        $init_shows += $p->{positions}->{$_}->{shows} for (@positions);
        if ($init_shows < $p->{total_shows}) {
            if (! defined $p->{positions}->{none} ) {
                $p->{positions}->{none} = {
                    ctr   => 0, 
                    price => 0,
                    shows => 0
                }
            }
            $p->{positions}->{none}->{shows} += $p->{total_shows} - $init_shows;
        }
        #Конец начального распределения показов.

        #Получаем новый список позиций (с начала блока там могла появиться новая позиция 'none')
        @positions = keys %{$p->{positions}};
        my %from_to;
        for my $from (@positions) {
            my ($to, $mcost) = (undef, undef);
            for my $pos (@positions) {
                next if $pos eq $from;
                my ($ctr1, $ctr2, $pr1, $pr2) = ($p->{positions}->{$from}->{ctr}, $p->{positions}->{$pos}->{ctr}, 
                                                 $p->{positions}->{$from}->{price}, $p->{positions}->{$pos}->{price});
                next if $ctr2 <= $ctr1;
                my $cost = ($ctr2*$pr2 - $ctr1*$pr1) / ($ctr2 - $ctr1);
                ($to, $mcost) = ($pos, $cost) if ( !defined($mcost) || $cost < $mcost );
            }
            next unless defined $to;
            $from_to{$from} = {
                md5      => $p->{md5},
                phrase   => $p,
                from     => $from,
                to       => $to,
                cost     => $mcost,
                fctr     => $p->{positions}->{$from}->{ctr},
                tctr     => $p->{positions}->{$to}->{ctr},
                to_price => $p->{positions}->{$to}->{price}
            };
        }
        # Отбираем "нужные" переходы, т.е. те, по которым можно пройти, начав с позиции, у которой есть какие-то показы.
        my @phrase_transitions;
        my %TRANS;
        for my $pos (@positions) {
            if ($p->{positions}->{$pos}->{shows}){
                my $f = $pos;
                while (my $tr = $from_to{$f}) {
                    $TRANS{$f} = 1;
                    $f = $tr->{to};
                }
            }
        }
        push @phrase_transitions, $from_to{$_} for (keys %TRANS);
        @phrase_transitions = sort { $a->{to_price} <=> $b->{to_price} } @phrase_transitions;
        # Добавляем к фразе информацию о максимальных расходах и расходах в гарантии - !!! отнести в prepare_phrases_prices_ctrs
        if( scalar(@phrase_transitions) ){
            my $max_pos = $phrase_transitions[-1]->{to};
            my $rec_pos = defined $p->{positions}->{$O{rec_pos}} ? $O{rec_pos} : (keys %{$p->{positions}})[-1];
            $p->{max_exps}   = $p->{positions}->{$max_pos}->{price} * 
                                 $p->{positions}->{$max_pos}->{ctr} * 
                                 $p->{total_shows} / 100;
            $p->{rec_pos_exps} = $p->{positions}->{$rec_pos}->{price} * 
                                 $p->{positions}->{$rec_pos}->{ctr} * 
                                 $p->{total_shows} / 100;
        } else {
            $p->{max_exps} = 0;
            $p->{rec_pos_exps} = 0;
        }

        $raw_transitions->{$p->{md5}} = {
            phrase      => $p,
            transitions => \@phrase_transitions
        };
    }

    return $raw_transitions;
}


=head2 order_transitions

    исключительно внутренняя функция, извне модуля не вызывать

    собираем из $raw_transitions одни упорядоченный массив переходов

    пред-условие: в $raw_transitions переходы по каждой фразе уже упорядочены

=cut
sub order_transitions
{
    my ($raw_transitions) = @_;

    my @otr; # ordered transitions

    for my $tr (map {$_->{transitions}} values %$raw_transitions ){
        # { идем по @ordered_transitions и находим место для $tr->[$k] } пока не обработали весь $tr
        my ($i, $k) = (0, 0); # $i -- позиция в $otr, $k -- в $tr 

        while( $k <= scalar @$tr - 1) {
            # в результате $i -- позиция, на к-рую надо вставить $k-й переход из $tr. Т.е. для 8 и (1, 4, 12) д.б. $i = 2.
            while( $i <= scalar @otr - 1 && $tr->[$k]->{cost} > $otr[$i]->{cost} ) { 
                $i++;
            } 

            splice(@otr, $i, 0, $tr->[$k]);
            $k++;
        }
    }

    return \@otr;
}




=head2 get_moneymeter_data 
    получить данные для прогноза работы Автобюджета (=для Баблометра, =для продвинутого Мастера Цен)

    На входе бывает: 
        $cid -- тогда нужна информация по всей сохраненной кампании
            опция: какие объявления учитывать (для Мастера Цен -- только активные, для чайников -- условие мягче)
        набор фраз (просто тексты) -- тогда нужен прогноз только по этим фразам 
            к просто текстам нужно обязательно указывать $options->{currency}

    На выходе бывают нужны: 
        "переходы" одним целым массивом, правильно упорядоченным, с подробной информацией по позициям и ставкам -- для Мастера Цен (сумма/клики => ставки)
        "переходы" одним целым массивом, правильно упорядоченным, без подробной информации по позициям и ставкам -- собственно для прогноза работы Автобюджета (сумма => клики)
        "переходы" без особо подробной информации по фразам, но отдельно по каждой фразе -- для бегунка-Баблометра (сумма => клики)
            (набор фраз может интенсивно меняться, поэтому хочется не гонять все фразы вместе, а запрашивать только новые)

        +"рекомендуемый" бюджет (для целого набора фраз или для каждой фразы в отдельности). Вопрос: всегда ли рек. бюджет считается по одной и той же формуле?

    в любом случае структура на выходе желательна без внутренних ссылок -- чтобы легко было сделать to_json

    В случае какой-либо неудачи возващать пустой хеш
    
    Основной пункт -- получение "сырых" переходов, fc_get_raw_transitions($phrases) 
    Соответственно, до этого происходит подготовка данных, а после -- форматирование результатов


    Параметры: 
    $options -- хеш содержательных параметров: 
        $options->{purpose} -- обязательный; где будет использоваться прогноз. Главное, что зависит от purpose -- формат результата. Возм. значения: (priceoptimizer|slider|advanced_forecast)
            priceoptimizer -- Мастер Цен
            slider -- бегунок на странице редактирования чайниковского объявления и в чайниковской кампании
            advanced_forecast -- для Суперпрогнозатора
        $options->{cid} -- cid кампании, по которой брать прогноз;
        conv_unit_rate -- курс у.е. к псевдовалюте (игнорируется для валют, отличных от YND_FIXED)
        $options->{phrases} -- ссылка на массив фраз ["слон", "бегемот", "носорог"], по которым нужен расчет;
            если указан cid -- берем данные по кампании целиком 
            если указаны phrases -- данные по фразам 
            если указано и то, и другое -- получится объединение кампании и набора новых фраз
        $options->{phrases_geo} -- регион для фраз ($options->{phrases}). Если не указан, считается 0 (Все);
        $options->{banners_filter} -- Если указан cid, то banners_filter передается в forecast_get_camp (=> определяет, какие именно объявления из кампании брать в расчет)
            Допустимые значения: (forecast_easy|active_only)
            Значение по умолчанию -- 'active_only'
        $options->{use_priorities} -- архаический флаг. "Использовать ли приоритеты фраз". Нигде пока толком не используется.
        $options->{mark_trans_except_bid} -- флаг. Если выставлен И формат результата -- 'slider', то будут помечены переходы, не относящиеся к текущему объявлению
        $options->{without_categories_for_except_bid} -- флаг. Если выставлен И формат результата -- 'slider', то для текущего объявления не будут учитвываться категории (надо для редактирования Легких объявлений)    
            текущее объявление определяется номером $options->{exceptional_bid}
        $options->{exceptional_bid} -- номер текущего объявления. См. $options->{mark_trans_except_bid}
        period -- 'week' или 'month', по умолчанию 'week'. На такой срок будет рассчитан прогноз показов
        currency -- валюта, в которой принимать и отдавать денежные значения; обязательное поле, если не указан cid
        lang -- язык, пока используется только для домена .com.tr - "tr"

    Результат: в зависимости от $options->{purpose}. Подробные описания см. в конце модуля, "про форматы выходных данных"

    результат для purpose == slider:
    {
      'data_distributed' => {...} -- прогноз по фразам из нового прогнозатора для стратегии "Поток новых посетителей"
      'low_ctr_keys' => ['da3324...', 'dfa123...', ...] -- id (MD5) фраз
      'phrases' => {...},         -- прогноз по фразам из старого прогнозатора для стратегий "быстрое привлечение"/"распределенный недельный бюджет"
      'weekday_factor' => '0.17'  -- множитель для перевода недельных кликов в дневные ("в будний день") - для стратегий "быстрое привлечение"/"распределенный недельный бюджет"
    }

=cut

sub get_moneymeter_data
{
    my ($options) = @_;

    # === Разбираем параметры ==
    my $O = {};
    $O->{$_} = $options->{$_} || '' for qw/purpose
                                           cid
                                           use_priorities phrases
                                           banners_filter
                                           mark_trans_except_bid
                                           exceptional_bid
                                           without_categories_for_except_bid
                                           phrases_geo
                                           conv_unit_rate
                                           consider_sitelinks_ctr
                                           minus_words
                                           unglue
                                           fixate_stopwords
                                           forecast_camp_data_cache
                                          /;

    $O->{exceptional_bid} = 0 unless defined $O->{exceptional_bid};
    $O->{period} = $options->{period} || 'week';
    $O->{advanced_period} = $options->{advanced_period} if exists $options->{advanced_period};
    delete $O->{advanced_period} if exists $options->{advanced_period}->{month} && $options->{advanced_period}->{month} == 0;
    $O->{categories} = $options->{categories} || [];
    return {} unless $O->{purpose} =~/^(priceoptimizer|slider|advanced_forecast)$/;

    $O->{start_pos} = $O->{purpose} eq 'priceoptimizer' ? 'cent1' : 'none';
    $O->{phrases} = [] unless ref $O->{phrases} eq 'ARRAY';
    $O->{banners_filter} = 'active_only' unless $O->{banners_filter} =~ /(forecast_easy|active_only)/;
    $O->{lang} = $options->{lang} if exists $options->{lang};

    my $advq_timeout = $options->{advq_timeout} || $MONEYMETER_ADVQ_TIMEOUT;

    # === Берем данные прогноза ==
    # если есть целая кампания...
    my $forecast_data = [];
    my ($forecast_camp_data, $currency, $no_extended_geotargeting);
    if ($O->{cid}) {
        $forecast_camp_data =
            $O->{forecast_camp_data_cache}
            ? $O->{forecast_camp_data_cache}
            : forecast_get_camp($O->{cid}, undef, {
                get_all_categories => 1, 
                banners_filter => $O->{banners_filter},
                allow_emergency_shows_forecast => 1,
                advq_timeout => $advq_timeout,
                calc_context_forecast => ($O->{purpose} eq 'advanced_forecast' ? 1 : 0) ,
                exceptional_bid => $O->{exceptional_bid},
              });
        $forecast_data = [values %$forecast_camp_data];
        if ($forecast_camp_data && %$forecast_camp_data) {
            # если выбрался хоть один баннер, то forecast_get_camp вернёт в нём валюту кампании
            # в пределах кампании валюта должна быть одинаковой, поэтому используем её из первого баннера
            my @bids = keys %$forecast_camp_data;
            my $first_bid = $bids[0];
            $currency = $forecast_camp_data->{$first_bid}->{currency};
            $no_extended_geotargeting = $forecast_camp_data->{$first_bid}->{no_extended_geotargeting};
        } else {
            # если баннеры не выбрались, достаём валюту из кампании. не уверен, что она потом используется.
            my $camp_info = get_camp_info($O->{cid}, undef, short => 1);
            $currency = $camp_info->{currency};
            $no_extended_geotargeting = $camp_info->{opts} =~ /\bno_extended_geotargeting\b/;
        }
    }

    $currency ||= $options->{currency};
    die 'no currency found' unless $currency;

    $O->{currency} = $currency;
    if ($currency eq 'YND_FIXED') {
        die "get_moneymeter_data -- conv_unit_rate needed" unless $O->{conv_unit_rate}; # Можно было бы по умолчанию считать в у.е., но, кажется, это было бы хуже (нет _частого_ использования Баблометра в у.е.)
    } else {
        $O->{conv_unit_rate} = 1;
    }

    # ... и если есть набор новых фраз
    if ( @{$O->{phrases}} > 0 || @{$O->{categories}} > 0 ){
        my $ph_block={ 
            Phrases => [], 
            geo     => $O->{phrases_geo},
            advq_timeout => $advq_timeout,
            allow_emergency_shows_forecast => 1,
            fairAuction => 1,
            consider_sitelinks_ctr => ($O->{consider_sitelinks_ctr} =~ /^(true|1)$/) ? 1 : 0,
            currency => $currency,
            no_extended_geotargeting => $no_extended_geotargeting,
        };

        fixate_stopwords_lite($O->{phrases}) if $O->{fixate_stopwords};
        my $unglued = $O->{unglue} ? unglue_phrases_lite($O->{phrases}) : {};

        my $id = 0;
        for my $phr (@{$O->{phrases}}){
            my $phrase_before_unglue = $phr;
            my $phrase_unglued = $unglued->{phrases}->{$phr}->{phrase_unglued} || $phr;
            my $unglued = $unglued->{phrases}->{$phr}->{unglued_suffix} ? 1 : 0;

            # "Общие" минус-слова: сначала подменяем фразы (приклеиваем все минус-слова, не содержащиеся в исходной фразе)
            my @minus_words_to_add;
            unless ( $phrase_unglued =~ /^"/ ) {
                # Проверяем с помощью validate_key_phrase только минус-слова, так как оно не работает с минус-фразами
                my (@mphrases, @mwords);
                for my $minus_obj (@{$O->{minus_words} || []}) {
                    if ($minus_obj =~ /\s+/) {
                        push @mphrases, $minus_obj;
                    } else {
                        push @mwords, $minus_obj;
                    }
                }
                my $bad_words = MinusWords::key_words_with_minus_words_intersection(key_words => [$phrase_unglued], minus_words => $O->{minus_words} || [])->{minus_words};
                # при валидации теряется (!) для точных минус слов
                push @$bad_words, map { '!'.$_ } @$bad_words;
                my $common_minus_words = xminus(\@mwords, $bad_words);
                @minus_words_to_add = (@$common_minus_words, @mphrases);
            }
            my $common_minus_words_suffix = @minus_words_to_add 
                ? ' ' . MinusWordsTools::minus_words_array2str_with_brackets_and_minus(\@minus_words_to_add) : '';

            push @{$ph_block->{Phrases}}, {
                phrase => $phrase_unglued . $common_minus_words_suffix, 
                minus_words => \@minus_words_to_add,
                phrase_wo_common_minus_words => $phrase_unglued, 
                phrase_before_unglue => $phrase_before_unglue,
                unglued => $unglued,
                phrase_id_for_html => $id++, 
            };
        }

        my %forecast_calc_params = (
            calc_context_forecast => ($O->{purpose} eq 'advanced_forecast' ? 1 : 0),
            forecast_without_banners => 1,
        );
        $forecast_calc_params{period} = $O->{advanced_period} if exists $O->{advanced_period};
        $forecast_calc_params{lang} = $O->{lang} if exists $O->{lang};
        forecast_calc( [ $ph_block ], %forecast_calc_params );
        push @$forecast_data, $ph_block;
    }

    # "Общие" минус-слова: после forecast_calc возвращаем исходные фразы. 
    # Прогнозы показов/ctr -- по полностью отминусованным фразам.
    # Заодно пересчитываем md5 (а действительно ли мешают неправильные md5?)
    if($O->{purpose} eq 'advanced_forecast'){
        for my $block ( @$forecast_data ){
            for my $ph ( @{$block->{phrases}} ){
                $ph->{phrase} = $ph->{phrase_wo_common_minus_words};
                $ph->{md5} = md5_hex_utf8($ph->{norm_phrase});
            }
        }
    }

    # === Собираем фразы, позиции, цены и ctr'ы в нужном виде ==
    if ($O->{purpose} eq 'advanced_forecast' ){
        $O->{low_ctr_value} = 0.5;
        $O->{low_ctr_mark_only} = 1;
    }
    my $phrases = prepare_phrases_prices_ctrs($forecast_data, $O);

    # === Самое главное: рассчитываем переходы ==
    my $raw_transitions = fc_get_raw_transitions($phrases, rec_pos => ( $O->{purpose} eq 'advanced_forecast' ? 'first_place' : '' )); 

    # === Форматируем результат ==
    my $res = {};
    # формат №1
    $res = prepare_transitions_for_priceoptimizer($raw_transitions, $phrases) if $O->{purpose} eq 'priceoptimizer';
    # формат №2
    if ($O->{purpose} eq 'slider') {
        $res = prepare_transitions_for_slider($raw_transitions, $O);

        my $options_for_advanced_forecast = {%$options};
        $options_for_advanced_forecast->{purpose} = 'advanced_forecast';
        $options_for_advanced_forecast->{forecast_camp_data_cache} = $forecast_camp_data;
        delete $options_for_advanced_forecast->{advanced_period};
        my $advanced_forecast = get_moneymeter_data($options_for_advanced_forecast);
        $res->{data_distributed} = $advanced_forecast->{data_distributed};
        $res->{low_ctr_keys} = $advanced_forecast->{low_ctr_keys};
    }

    # формат "Суперпрогнозатор"
    if ( $O->{purpose} eq 'advanced_forecast' ){
        my $context_stat = prepare_context_stat($forecast_data);

        my $cleared_forecast = [ map { Forecast::clear_block_forecast($_) } @$forecast_data ];

        $res = prepare_transitions_for_advanced_forecast($raw_transitions, $phrases, $context_stat, $cleared_forecast, $currency);
    }

    $res->{currency} = $currency;

    return $res;
}

=head2 get_new_advanced_forecast

    Получение данных для нового прогнозатора
    
    Только для использования в ajaxDataForNewBudgetForecast
    
    Результаты очень похожи на get_moneymeter_data с purpose => advanced_forecast,
    с некоторыми отличиями:
    
    * Не считаются данные для слайдера баблометра (data_distributed)
    * У каждой фразы в data_by_positions есть positions, где по каждой позиции
      (P11, P12, P13, P21, P24) лежат данные бюджета.
    * Пока поддерживается только YND_FIXED

=cut

sub get_new_advanced_forecast
{
    my ($options) = @_;
    my $profile = Yandex::Trace::new_profile('forecast:get_new_advanced_forecast');

    # === Разбираем параметры ==
    my $O = {};
    $O->{$_} = $options->{$_} || '' for qw/cid
                                           phrases
                                           phrases_geo
                                           phrases_bids
                                           conv_unit_rate
                                           consider_sitelinks_ctr
                                           minus_words
                                           unglue
                                           fixate_stopwords
                                          /;

    $O->{period} = $options->{period} || 'month';
    $O->{advanced_period} = $options->{advanced_period} if exists $options->{advanced_period};
    delete $O->{advanced_period} if exists $options->{advanced_period}->{month} && $options->{advanced_period}->{month} == 0;
    $O->{advanced_period} = {week => 1} unless exists $O->{advanced_period} || $O->{period} ne 'week';
    $O->{categories} = $options->{categories} || [];

    $O->{phrases} = [] unless ref $O->{phrases} eq 'ARRAY';
    $O->{phrases_bids} = [] unless ref $O->{phrases_bids} eq 'ARRAY';
    $O->{lang} = $options->{lang} if exists $options->{lang};
    $O->{distribution} = $options->{distribution} if exists $options->{distribution};

    $O->{minus_words} = [] unless ref $O->{minus_words} eq 'ARRAY';

    my $advq_timeout = $options->{advq_timeout} || $MONEYMETER_ADVQ_TIMEOUT;

    die 'budget for campaigns is not supported yet' if $O->{cid};
    my $currency = $options->{currency};
    die 'no currency found' unless $currency;
    my $forecast_data = [];

    $O->{currency} = $currency;
    if ($currency eq 'YND_FIXED') {
        die "get_moneymeter_data -- conv_unit_rate needed" unless $O->{conv_unit_rate}; # Можно было бы по умолчанию считать в у.е., но, кажется, это было бы хуже (нет _частого_ использования Баблометра в у.е.)
    } else {
        $O->{conv_unit_rate} = 1;
    }

    # ... и если есть набор новых фраз
    my %phrase2key;
    my %key2phrase;
    my @low_ctr_keys;
    my @unglued_keys;
    my @data_by_positions;
    my %data_distributed = (phrases => [], index2key => {});
    my $low_ctr = 0.5;
    if (@{$O->{phrases}} > 0 || @{$O->{categories}} > 0) {
        my $block = {
            Phrases => [],
            geo     => $O->{phrases_geo},
            advq_timeout => $advq_timeout,
            allow_emergency_shows_forecast => 1,
            fairAuction => 1,
            consider_sitelinks_ctr => ($O->{consider_sitelinks_ctr} =~ /^(true|1)$/) ? 1 : 0,
            currency => $currency,
        };

        fixate_stopwords_lite($O->{phrases}) if $O->{fixate_stopwords};
        my $unglued = $O->{unglue} ? unglue_phrases_lite($O->{phrases}) : {};

        my $id = 0;
        for my $phr (@{$O->{phrases}}) {
            my $phrase_before_unglue = $phr;
            my $phrase_unglued = $unglued->{phrases}->{$phr}->{phrase_unglued} || $phr;
            my $unglued = $unglued->{phrases}->{$phr}->{unglued_suffix} ? 1 : 0;

            # "Общие" минус-слова: сначала подменяем фразы (приклеиваем все минус-слова, не содержащиеся в исходной фразе)
            my @minus_words_to_add;
            if ($phrase_unglued !~ /^"/) {
                my $invalid_minus_words = MinusWords::key_words_with_minus_words_intersection(key_words => [$phrase_unglued], minus_words => $O->{minus_words} || [])->{minus_words};
                if (@$invalid_minus_words) {
                    my %words = map {$_ => 1} @$invalid_minus_words;
                    @minus_words_to_add = grep {! exists $words{$_}} @{$O->{minus_words}};
                } else {
                    @minus_words_to_add = @{$O->{minus_words}};
                }
            }

            my $common_minus_words_suffix = @minus_words_to_add ? ' ' . MinusWordsTools::minus_words_array2str_with_brackets_and_minus(\@minus_words_to_add) : '';
            my $advq_bid = $O->{phrases_bids}->[$id];
            push @{$block->{Phrases}}, {
                phrase => $phrase_unglued . $common_minus_words_suffix,
                minus_words => \@minus_words_to_add,
                phrase_wo_common_minus_words => $phrase_unglued, 
                phrase_before_unglue => $phrase_before_unglue,
                unglued => $unglued,
                phrase_id_for_html => $id++, 
                advq_bid => $advq_bid,
            };
        }

        my %forecast_calc_params = (
            calc_context_forecast => 1,
            filter_duplicate_md5 => 1,
            devices => $options->{devices},
        );
        $forecast_calc_params{distribution} = $O->{distribution} if exists $O->{distribution};
        $forecast_calc_params{period} = $O->{advanced_period} if exists $O->{advanced_period};
        $forecast_calc_params{lang} = $O->{lang} if exists $O->{lang};
        forecast_calc_new( [ $block ], %forecast_calc_params );

        die $block->{error}{message} if exists $block->{error};

        my $ph_indexes = 0;
        for my $ph (@{$block->{phrases}}) {
            my $ph_index = $ph_indexes++;
            # "Общие" минус-слова: после forecast_calc возвращаем исходные фразы. 
            # Прогнозы показов/ctr -- по полностью отминусованным фразам.
            $ph->{phrase} = $ph->{phrase_wo_common_minus_words};
            my $key = $ph->{md5};
            $key2phrase{$key} = $ph->{phrase};
            $phrase2key{$ph->{phrase}} = $key;
            if ($O->{distribution}) {
                push @{$data_distributed{phrases}}, {
                    md5 => $ph->{md5},
                    sign => md5_hex_utf8(($ph->{phrase} ||  ''). $Settings::SECRET_PHRASE),
                    distribution => $ph->{distribution},
                };
                $data_distributed{index2key}->{$ph_index} = $key;
            } else {
                # Рассчитываем максимальный ctr, который получается у фразы
                my $max_ctr = 0;
                for my $pos (values $ph->{positions}) {
                    # Исправляем ctr на основе целых кликов и показов
                    if ($pos->{shows} > 0) {
                        $pos->{ctr} = $pos->{clicks} * 100 / $pos->{shows};
                    } else {
                        $pos->{ctr} = 0;
                    }
                    # Исправляем бюджет на основе целых кликов
                    $pos->{bid} = ceil(1e6 * Currencies::round_price_to_currency_step($pos->{bid} / 1e6, $currency, up => 1));
                    if ($pos->{clicks} > 0) {
                        # Округляем цену клика до правильного шага цены
                        my $click_price = ceil(1e6 * Currencies::round_price_to_currency_step($pos->{budget} / $pos->{clicks} / 1e6, $currency, up => 1));
                        $pos->{budget} = $click_price * $pos->{clicks};
                    } else {
                        $pos->{budget} = 0;
                    }
                    if ($max_ctr < $pos->{ctr}) {
                        $max_ctr = $pos->{ctr};
                    }
                }
                if ($max_ctr < $low_ctr) {
                    # Помечаем фразы с низким ctr
                    $ph->{low_ctr} = 1;
                }
                # Готовим результирующие данные
                push @low_ctr_keys, $key if $ph->{low_ctr};
                push @unglued_keys, $key if $ph->{unglued};

                # явно указываем позиции входа в спецразмещение и гарантию
                my $premium_entry_pos = PlacePrice::get_premium_entry_place(forecast_style => 1);
                my $guarantee_entry_pos = PlacePrice::get_guarantee_entry_place(forecast_style => 1);
                if (!$ph->{positions}{premium} && $ph->{positions}{$premium_entry_pos}) {
                    $ph->{positions}{premium} = $ph->{positions}{$premium_entry_pos};
                }
                if (!$ph->{positions}{std} && $ph->{positions}{$guarantee_entry_pos}) {
                    $ph->{positions}{std} = $ph->{positions}{$guarantee_entry_pos};
                }
                push @data_by_positions, {
                    md5 => $ph->{md5},
                    sign => md5_hex_utf8(($ph->{phrase} // ''). $Settings::SECRET_PHRASE),
                    shows => $ph->{shows},
                    positions => $ph->{positions},
                };
            }
        }
    }

    # В случае advanced_forecast_new отдаём позиции как есть
    return {
        key2phrase => \%key2phrase,
        phrase2key => \%phrase2key,
        low_ctr_keys => \@low_ctr_keys,
        unglued_keys => \@unglued_keys,
        data_by_positions => \@data_by_positions,
        data_distributed => \%data_distributed,
        currency => $currency,
    };
}


#====================================================================================================================

sub prepare_context_stat_for_one_phrase
{
    my ($stat) = @_;

    my $ph_stat = hash_cut $stat, qw/clicks_cnt shows_cnt complete_list/;
        for my $point ( @{$ph_stat->{complete_list}} ){
            $point->{cost} = $point->{cost} / 1e6;
        }

    return $ph_stat;
}

=head2 

=cut 
sub prepare_context_stat
{
    my ($forecast_blocks) = @_;

    my $profile = Yandex::Trace::new_profile('forecast:prepare_context_stat', obj_num => scalar @$forecast_blocks);

    my $stat = {};
    for my $block (@$forecast_blocks){
        for my $ph (@{$block->{Phrases}}){
            $stat->{search}->{$ph->{md5}} = prepare_context_stat_for_one_phrase($ph->{pokazometer_data_search});
            $stat->{context}->{$ph->{md5}} = prepare_context_stat_for_one_phrase($ph->{pokazometer_data_context});
            $stat->{both}->{$ph->{md5}} = prepare_context_stat_for_one_phrase($ph->{pokazometer_data});
        }
    }

    return $stat;
}


=head2 prepare_phrases_prices_ctrs

    исключительно внутренняя функция, для вызова только из get_moneymeter_data

=cut
sub prepare_phrases_prices_ctrs
{
    my ($forecast_blocks, $O) = @_;

    my $phrases = {};
    # дополнительные поправочные коэффициенты для периодов. 
    # Если период уже корректно считается в forecast_calc -- здесь должна быть 1
    my $shows_coef_for_period = { week => 7/30, month => 1, year => 1, quarter => 1 }->{ $O->{period} } or die "unknown period $O->{period}"; 
 
    my %USED_MD5;
    for my $block (@$forecast_blocks){ 
        my $currency = $block->{currency};
        die 'no currency given' unless $currency;
        my $min_price_constant = get_currency_constant($currency, 'MIN_PRICE');
        my $auction_step_constant = get_currency_constant($currency, 'AUCTION_STEP');

        # фразы сортируем по возрастанию прогнозируемого количества показов (из соображений что так косвенно учитываются минус-слова)
        # ctr у фраз отличающихся только минус словами - одинаковый, потому для однозначности используем показы
        for my $p (sort { $a->{md5} eq $b->{md5} ? $a->{shows} <=> $b->{shows} : 0 } @{$block->{phrases}}) {
            next if defined $p->{rank} && $p->{rank} < 2; # Отключенные за низкий CTR
            next if $USED_MD5{$p->{md5}}++;
            my $phrase_id = "val_".($p->{bid}||'')."_".($p->{id}||'');
            my $phr = $phrases->{$p->{md5}."_".$phrase_id} = {
                positions   => {},
                phrase_id   => $phrase_id,
                total_shows => int ($p->{shows}*$shows_coef_for_period),
                bid         => $p->{bid}||0,
                min_price   => $p->{min_price} ? $p->{min_price} : $min_price_constant,
                md5         => $p->{md5},
                unglued      => $p->{unglued},
            };
            $phr->{rot_koef} = $p->{rotation_factor} || 1; # кажется, с использованием AutoBroker::calc_price коэф. ротации стал не нужен. Убедиться и удалить!

            hash_merge($phr, hash_cut($p, qw/phrase/));

            # == Примечание о смысле ctr ==
            # Для этого алгоритма  ctr -- не отношение кликов к _показам данного объявления_, 
            # а кликов к _количеству поиков данной фразы_.
            # Пример: слово "слон" искали 1000 раз, в т.ч. 100 раз было показано объявление, и произошло 10 кликов,
            # считаем ctr = 10 / 1000  = 1%
            # => если вероятность показа < 1 -- в алгоритме это учитывается как пропорциональное уменьшение ctr

            $phr->{positions} = {};
         
            # составляем хеш про все-все-все позиции
            my $low_ctr_value = $O->{low_ctr_value} || 0.6;
            if( $p->{ctr} >= $low_ctr_value || $O->{low_ctr_mark_only} ) { 
                # фразы "на грани отключения" -- не учитываем совсем, если не выставлен флаг "только пометить"
                if ( $p->{ctr} < $low_ctr_value && $O->{low_ctr_mark_only} ){
                    $phr->{low_ctr} = 1;
                }

                # добавляем позицию cent1, потому что: 
                #   исторически сложилось, что алгоритм завязан на ее существование (хотя м.б., это надо исправить)
                #   это более-менее правильно: пользователь в любом случае может выставить минимальную ставку
                $phr->{positions}->{cent1} = _calc_pos($p, $phr->{min_price}, $currency);
                $phr->{positions}->{cent1}->{shows} = $phr->{total_shows} if $O->{start_pos} eq 'cent1'; 

                # остальные позиции
                $phr->{positions}->{first_premium} = { 
                    #PREMIUM1
                    ctr   => $p->{premium}->[0]->{ctr},
                    price => $p->{premium}->[0]->{amnesty_price},
                };
                $phr->{positions}->{second_premium} = { 
                    #PREMIUM2
                    ctr   => $p->{premium}->[1]->{ctr},
                    price => $p->{premium}->[1]->{amnesty_price},
                };                
                $phr->{positions}->{premium} = { 
                    #PREMIUM3
                    ctr   => $p->{premium}->[-1]->{ctr},
                    price => $p->{premium}->[-1]->{amnesty_price},
                };
                $phr->{positions}->{first_place} = { 
                    #GUARANTEE1
                    ctr   => $p->{guarantee}->[0]->{ctr},
                    price => $p->{guarantee}->[0]->{amnesty_price},
                };
                $phr->{positions}->{std} = { 
                    #GUARANTEE4
                    ctr   => $p->{guarantee}->[-1]->{ctr},
                    price => $p->{guarantee}->[-1]->{amnesty_price},
                };
            }
            # составляем рекомендации по каждой фразе: min/max позиции, на к-рых ее стоит показывать
            if (scalar keys %{$phr->{positions}} <=1) {
                # "на грани отключения" -- не учитываем
                $phr->{positions_rec}->{left} = {
                    coverage => 0,
                    ctr => 0,
                    price => 0,
                };
            } elsif (exists $phr->{positions}->{"std"}) {
                # Фразы без негарантированных показов, но с гарантированными
                $phr->{positions_rec}->{left} = _calc_pos($p, $min_price_constant + $auction_step_constant, $currency);
            } else {
                # Остальные, то есть странные какие-то фразы
                $phr->{positions_rec}->{left} = _calc_pos($p, $min_price_constant + $auction_step_constant, $currency);
            }
            $phr->{positions_rec}->{right} = _calc_pos($p, convert_currency(10, 'YND_FIXED', $currency), $currency);

            # бюджеты для разных стратегий
            # high (Лучшие)
            if (exists $phr->{positions}->{premium} ) {
                $phr->{positions_rec}->{high} = {};
                hash_merge($phr->{positions_rec}->{high}, $phr->{positions}->{premium});
            } else {
                $phr->{positions_rec}->{high} = _calc_pos($p, convert_currency(1.0, 'YND_FIXED', $currency), $currency);
            }

            # middle (Средние)
            if (exists $phr->{positions}->{std} ) {
                $phr->{positions_rec}->{middle} = {};
                hash_merge($phr->{positions_rec}->{middle}, $phr->{positions}->{std});
            } else {
                $phr->{positions_rec}->{high} = _calc_pos($p, convert_currency(0.5, 'YND_FIXED', $currency), $currency);
            }

            # low (Ниже среднего)
            #my $low_price = $_->[int(scalar(@$_)*5/10)] for $competitors_prices;
            $phr->{positions_rec}->{low} = _calc_pos($p, $min_price_constant, $currency);

        }
    }

    return $phrases; 
}

=head3 _calc_pos

    Вычисляем ctr для "одноцентовой позиции"

    $pos_data = _calc_pos($phrase, $min_price, $currency);

=cut

sub _calc_pos
{
    my ($phrase, $min_price, $currency) = @_;
    my $prices = hash_merge AutoBroker::parse_prices($phrase);
    $prices->{price} = $min_price;
    $prices->{currency} = $currency;
    my $pos = calc_price($prices);
    return {
        ctr   => $phrase->{ctr}*$pos->{broker_coverage}, 
        price => $pos->{broker_price}, 
        coverage => $pos->{broker_coverage},
    };
}

=head3 _get_money_bind_functions

=cut

sub _get_money_bind_functions {
    my $currency_consts = Currencies::get_consts_for_js();

    return {
        get_currency => sub {
            my ($currency) = @_;
            my $currency_constants = $currency_consts->{$currency};
            utf8::encode($_) for grep {defined} values %$currency_constants;
            return $currency_constants;
        },
    };
}



#====================================================================================================================

=head2 calc_autobudget_clicks_by_avg_price($cid)

    сколько кликов получим при средней цене кликов

    Параметры:
        $cid
        $avg_price
        $which - площадка для которой расчитывается автобюджет (total|context|yandex|broadmatch)

    Результат:
        число кликов

=cut

sub calc_autobudget_clicks_by_avg_price {
    
    my ($cid, $avg_price, $which) = @_;
    
    $which = $which || 'total';

    my $conv_unit_rate = 1;
    my $moneymeter = get_moneymeter_data({
        cid => $cid,
        purpose => 'advanced_forecast',
        # валюту не указываем, будет браться из кампании $cid
        conv_unit_rate => $conv_unit_rate,
        consider_sitelinks_ctr => 1,
    });
    my $currency = $moneymeter->{currency};

    # убираем лишние данные, иначе исполнение js падает
    delete @{$moneymeter}{qw/data_by_positions low_ctr_keys key2phrase/};

    my $forecast = call_js($Settings::JS_PERL_DIR.'/AdvancedForcast-p.js', "calc_advanced_forecast_p",
        [$moneymeter, {
            type => 'cpc',
            place => $which,
            cpc => $avg_price,
        }, {
            broadmatch => 0,
            currency => $currency,
            $which eq 'context'
                ? (context => 1, context_sum_coef => 100)
                : (context => 0, context_sum_coef => -1)
        },
        $currency],
        dont_js_write => 1,
        bind_functions => _get_money_bind_functions(),
    );

    die 'calc_advanced_forecast_p failed' unless $forecast->{$which}->{clicks};
    return $forecast->{$which}->{clicks};
}

#====================================================================================================================
sub get_phrases_for_html
{
    my ($phrases) = @_;

    my $res;
    for my $p (values %$phrases){
        my $h = $res->{$p->{phrase_id}} = hash_cut($p, qw/min_price/);
        $h->{shows} = $p->{total_shows};
        $h->{rot_shows} = $p->{rot_koef}*$p->{total_shows};        
    }

    return $res;
}


=head2 prepare_transitions_for_priceoptimizer

    исключительно внутренняя функция, для вызова только из get_moneymeter_data

=cut
sub prepare_transitions_for_priceoptimizer
{
    my ($raw_transitions, $phrases) = @_;

    my $profile = Yandex::Trace::new_profile('forecast:prepare_transitions_for_priceoptimizer', obj_num => scalar keys %$phrases);

    my $ordered_transitions = order_transitions($raw_transitions); 

    $_->{total_shows} = $_->{phrase}->{total_shows} for @$ordered_transitions;
    $_->{phrase_id} = $_->{phrase}->{phrase_id} for @$ordered_transitions;

    # Считаем, сколько кликов и расходов будет в ротации
    my ($rot_clicks, $rot_exps, $rot_shows) = (0, 0, 0);
    while (my ($key, $p) = each %$phrases) {
        next unless exists $p->{positions}->{cent1};
        my $cl = $p->{total_shows}*$p->{positions}->{cent1}->{ctr}/100;
        $rot_clicks += $cl;
        $rot_exps += $cl * $p->{positions}->{cent1}->{price};
        $rot_shows += $p->{total_shows}*$p->{rot_koef};
    }

    # Считаем максимальное количество кликов и расходов
    my ($max_clicks, $max_exps) = (0, 0);
    while (my ($key, $p) = each %$phrases) {
        next unless exists $p->{positions}->{cent1};
        my $best_pos = "cent1";
        map {$best_pos = $_ if $p->{positions}->{$_}->{ctr} > $p->{positions}->{$best_pos}->{ctr}} keys %{$p->{positions}};
        my $cl = $p->{total_shows}*$p->{positions}->{$best_pos}->{ctr}/100;
        $max_clicks += $cl;
        $max_exps += $cl * $p->{positions}->{$best_pos}->{price};
    }

    my $shows = {};
    $shows->{$_->{phrase_id}}->{total}=$_->{total_shows} for values %$phrases;
    $shows->{$_->{phrase_id}}->{rotation}=$_->{total_shows}*$_->{rot_koef} for values %$phrases;

    my $phrases_for_html = get_phrases_for_html($phrases);

    delete $_->{phrase} for @$ordered_transitions;
    my $res = { 
        phrases    => $phrases_for_html,
        trans      => $ordered_transitions,
        rot_clicks => int($rot_clicks),
        rot_exps   => $rot_exps,
        rot_shows  => int($rot_shows),
        max_clicks => int($max_clicks),
        max_exps   => int($max_exps),
    };

    return $res;
}


#====================================================================================================================

=head2 prepare_transitions_for_slider

    исключительно внутренняя функция, для вызова только из get_moneymeter_data

    Принимает ссылку на хеш с дополнительными параметрами:
        currency — в какой валюте исходные и результирующие цифры
        conv_unit_rate — курс условной единицы
                         если currency == 'YND_FIXED', то суммы вернутся в псевдовалюте с указанным курсом к у.е.
        exceptional_bid
        mark_trans_except_bid

    TODO: Отнести пересчёт в псевдовалюту в prepare_phrases_prices_ctrs!

=cut

sub prepare_transitions_for_slider
{
    my ($raw_transitions, $O) = @_;
    my $profile = Yandex::Trace::new_profile('forecast:prepare_transitions_for_slider', obj_num => scalar keys %$raw_transitions);

    $O ||= {};
    my $currency = $O->{currency};
    die 'no currency given' unless $currency;
    die "prepare_transitions_for_slider -- conv_unit_rate needed" if $O->{currency} eq 'YND_FIXED' && !$O->{conv_unit_rate};

    my $res_phrases = [];
    while( my ($md5, $raw_trans) = each %$raw_transitions) {
        #next if scalar @{$raw_trans->{transitions}} <= 0;
        my $phr = { md5    => $md5, 
                    phrase => $raw_trans->{phrase}->{phrase}};

        # Добавляем если надо флажок, что фраза не относится к текущему объявлению
        my $exceptional_bid = $O->{exceptional_bid} || 0;
        $phr->{rest_camp} = 1 if $O->{mark_trans_except_bid}
                              && $raw_trans->{phrase}->{bid}
                              && $raw_trans->{phrase}->{bid} != $exceptional_bid;
        if(scalar @{$raw_trans->{transitions}} <= 0){
            $phr->{max_exps} = 0;
            $phr->{rec_budget} = 0;
            $phr->{transitions} = []; 
            push @$res_phrases, $phr;
            next;
        }
        
        $phr->{max_exps} = $raw_trans->{phrase}->{max_exps};
        $phr->{rec_budget} = $raw_trans->{phrase}->{rec_pos_exps}*1.1;


        my $p = $raw_trans->{phrase};
        $phr->{left_lim_exps} = $p->{positions_rec}->{left}->{price} * 
                                 $p->{positions_rec}->{left}->{ctr} * 
                                 $p->{total_shows} / 100;
        $phr->{right_lim_exps} = $p->{positions_rec}->{right}->{price} * 
                                 $p->{positions_rec}->{right}->{ctr} * 
                                 $p->{total_shows} / 100;
        for (qw/high middle low/){
            $phr->{"exps_$_"} = $p->{positions_rec}->{$_}->{price} * 
                                     $p->{positions_rec}->{$_}->{ctr} * 
                                     $p->{total_shows} / 100;
        }


        $phr->{transitions} = []; 
        for (@{$raw_trans->{transitions}}) {
            my $t = {add_clicks => ($_->{tctr} - $_->{fctr}) * $raw_trans->{phrase}->{total_shows} / 100, 
                     cost       => $_->{cost},
                     md5        => $phr->{md5},
                    };
            $t->{rest_camp} = $phr->{rest_camp} if exists $phr->{rest_camp};
            push @{$phr->{transitions}}, $t; 
        }
        $phr->{shows} = $raw_trans->{phrase}->{total_shows};
        push @$res_phrases, $phr;
    }
    # Переводим суммы в рубли
    for my $p (@$res_phrases) {
        $p->{$_} ||= 0 for qw/right_lim_exps left_lim_exps exps_high exps_middle exps_low/;
        next if $currency ne 'YND_FIXED';
        $p->{$_} *= $O->{conv_unit_rate} for qw/rec_budget max_exps right_lim_exps left_lim_exps exps_high exps_middle exps_low/;
        exists $p->{"exps_percent_$_"} ? $p->{"exps_percent_$_"} *= $O->{conv_unit_rate} : 1 for 1 .. 9;
        $_->{cost} *= $O->{conv_unit_rate} for @{$p->{transitions}};
    }

    my $res = {
        weekday_factor => 0.17,       # типичный коэффициент для рабочего дня в Автобюджете 2.0 
        phrases     => $res_phrases,
    };

    return $res;
}


=head2 prepare_transitions_for_advanced_forecast

    исключительно внутренняя функция, для вызова только из get_moneymeter_data

=cut

sub prepare_transitions_for_advanced_forecast
{
    my ($raw_transitions, $phrases, $context_stat, $cleared_forecast, $currency) = @_;

    my $profile = Yandex::Trace::new_profile('prepare_transitions_for_advanced_forecast');

    die 'no currency given' unless $currency;
    my $min_price_constant = get_currency_constant($currency, 'MIN_PRICE');

    my $sums_clicks_shows = {};

    my %phrase2key;
    my %key2phrase;
    for my $key (keys %$raw_transitions){
        my $phrase = $raw_transitions->{$key}->{phrase}->{phrase} ;
        $phrase2key{$phrase} = $key if defined $phrase;
        $key2phrase{$key} = $phrase; # push ?
    }
    my @low_ctr_keys = map { $raw_transitions->{$_}->{phrase}->{low_ctr} ? $_ : () } keys %$raw_transitions;
    my @unglued_keys = map { $raw_transitions->{$_}->{phrase}->{unglued} ? $_ : () } keys %$raw_transitions;

    # Убираем из Показометровых данных слишком дорогие клики
    # Особенность: дорогой хвост отбрасываем не по большой _ставке_, 
    # а по дорогим _добавочным_ кликам (по сравнению с предыдущей "ступенькой")
    for my $type (qw/search context both/){
        for my $cs (values %{$context_stat->{$type}}){
            my $new_complete_list = [];
            my ($clicks, $cost) = (0,0);
            for my $p (@{$cs->{complete_list}}){
                my $no_new_clicks = $p->{clicks} > 0 && $clicks > 0 && $p->{clicks} == $clicks;
                my $too_expensive = $p->{clicks} > $clicks && ( $p->{cost}*$p->{clicks} - $clicks*$cost) / ($p->{clicks} - $clicks) > 20;
                last if $no_new_clicks || $too_expensive;
                push @$new_complete_list, $p;
                ($clicks, $cost) = ($p->{clicks},$p->{cost});
            }
            $cs->{complete_list} = $new_complete_list;
        }
    }
    
    my @segments;
    for my $key (keys %$raw_transitions){
        my $shows = $raw_transitions->{$key}->{phrase}->{total_shows};

        my $clicks = 0;
        my $segment_number = 0; # внутри фразы
        for my $tr ( @{$raw_transitions->{$key}->{transitions}} ) {
            my $delta_clicks = int(($tr->{tctr} - $tr->{fctr}) * $shows / 100);
            next unless $delta_clicks > 0;

            my $delta_shows = $clicks == 0 ? $shows : 0;
            my $delta_sum = $delta_clicks*$tr->{cost};

            my $segment = {
                segment_number => $segment_number,
                phrase_id => $key,
                delta_sum => $delta_sum, 
                delta_clicks => $delta_clicks, 
                delta_shows => $delta_shows, 
                click_cost => $tr->{cost},
                next_position => $tr->{to}, # "следующая" позиция, т.е. та, на котороую переводим показы
            };

            push @segments, $segment;

            $clicks += $delta_clicks;
            $segment_number++;
        }
    }
    
    @segments = sort { $a->{phrase_id} eq $b->{phrase_id} ? 
                       $a->{segment_number} <=> $b->{segment_number}  : 
                       $a->{click_cost} <=> $b->{click_cost}
                     } @segments;

    my %total = (
        yandex => {
            sum    => 0,
            clicks => 0,
            shows  => 0,
        }
    );
    for my $s (@segments) {
        $s->{yandex} = {
            delta_sum    => $s->{delta_sum},
            delta_clicks => $s->{delta_clicks},
            delta_shows  => $s->{delta_shows},
        };

        $total{yandex}->{$_} += $s->{"delta_$_"} for qw/sum clicks shows/;

        # расходы по отдельным фразам
        $total{phrases}{$s->{phrase_id}}{clicks} += $s->{delta_clicks};
        $total{phrases}{$s->{phrase_id}}{shows} += $s->{delta_shows};
        $total{phrases}{$s->{phrase_id}}{sum} += $s->{delta_sum};
        $total{phrases}{$s->{phrase_id}}{cpc} = $total{phrases}{$s->{phrase_id}}{clicks} > 0 ? $total{phrases}{$s->{phrase_id}}{sum} / $total{phrases}{$s->{phrase_id}}{clicks} : 0;

    }

    my $data_by_positions = prepare_data_by_positions( $cleared_forecast, $context_stat );
    my $res = {
        phrase2key => \%phrase2key,
        key2phrase => \%key2phrase,
        data_distributed => \@segments,
        low_ctr_keys => \@low_ctr_keys,
        unglued_keys => \@unglued_keys,
        data_by_positions => $data_by_positions,
    };

    $res->{rec_budget} = int(1.1*(sum(map {$_->{phrase}->{rec_pos_exps}} values %$raw_transitions)||0));

    return $res;
}


sub prepare_data_by_positions
{
    my ($cleared_forecast, $context_stat) = @_;

    my @result;

    for my $phr ( map { @{$_ || []} } @$cleared_forecast){
        my $h = hash_merge {}, $phr;
        delete $h->{phrase};
        my %place_map = ('first_premium' => $PlacePrice::PLACES{PREMIUM1},
                         'second_premium' => $PlacePrice::PLACES{PREMIUM2},
                         'premium' => PlacePrice::get_premium_entry_place(),
                         'first_place' => $PlacePrice::PLACES{GUARANTEE1},
                         'std' => PlacePrice::get_guarantee_entry_place(),
                        );
        my $values = {}; #из-за пересечения имен (premium), приходится сначала заполнять отдельных хеш, и только потом его подтягивать
        foreach my $place (keys %place_map) {
            my $row = PlacePrice::get_data_by_place($h, $place_map{$place});
            $values->{$place} = {};
            $values->{$place}->{yandex} = hash_cut $row, qw/clicks sum ctr bid_price amnesty_price/;
        }
        hash_merge $h, $values;

        push @result, $h;
    }

    return \@result;
}

#====================================================================================================================
#====================================================================================================================



=head2 про форматы выходных данных

=head3 Для ajaxGetTransitions результат нужен вот такой (хронологически формат №1)
    $VAR1 = {
          'max_clicks' => 51671,
          'max_exps' => 76313,
          'phrases' => {
                         'val_3855682_39380059' => {
                                                   'min_price' => '0.01',
                                                   'rot_shows' => 386352,
                                                   'shows' => 386352
                                                 },
                         'val_3855682_39380060' => {
                                                   'min_price' => '0.01',
                                                   'rot_shows' => 174417,
                                                   'shows' => 174417
                                                 },
                         ...
                       },
          'rot_clicks' => 3828,
          'rot_exps' => '38.2855658451906',
          'rot_shows' => 990724,
          'trans' => [
                       {
                         'cost' => '0.895429229035131',
                         'fctr' => '0.266761506233936',
                         'from' => 'cent1',
                         'md5' => 'a608b9c44912c72db6855ad555397470',
                         'phrase_id' => 'val_3855682_39380062',
                         'tctr' => '6.66676755982334',
                         'to' => 'premium',
                         'to_price' => '0.86',
                         'total_shows' => 92507
                       },
                       {
                         'cost' => '1.06022475686126',
                         'fctr' => '0.0310876826035108',
                         'from' => 'cent1',
                         'md5' => '1f4473547e71e1fb4fb470db5b4fb8ad',
                         'phrase_id' => 'val_3855688_39380076',
                         'tctr' => '0.54212014469174',
                         'to' => 'nonguarantee',
                         'to_price' => '1.00',
                         'total_shows' => 29543
                       },
                       ...
                     ]
        };


=head3 Для прогноза на странице чайниковской кампании (формат №3)
    $VAR1 = {
          'arr' => [
                     {
                       'clicks' => 0,
                       'sum' => 0
                     },
                     {
                       'clicks' => 3831,
                       'sum' => 38
                     },
                     ...
                   ],
          'max_clicks' => 51668,
          'max_exps' => 76319
        };


=head3 Формат "сырых" переходов (raw_transitions) на Поиске
    $VAR1 = {
  '0490b305539f9a2d4fb47a454c3a0dda' => {
                                          'phrase' => {
                                                        'max_exps' => '993.136542875543',
                                                        'md5' => '0490b305539f9a2d4fb47a454c3a0dda',
                                                        'phrase' => 'nec',
                                                        'phrase_id' => 'val_3855682_39380061',
                                                        'positions' => {
                                                                         'cent1' => {
                                                                                      'ctr' => '0.400270810284186',
                                                                                      'price' => '0.01',
                                                                                      'shows' => 0
                                                                                    },
                                                                         'first_place' => {
                                                                                            'ctr' => '0.550372364140755',
                                                                                            'price' => '1.01',
                                                                                            'shows' => 0
                                                                                          },
                                                                         'none' => {
                                                                                     'ctr' => 0,
                                                                                     'price' => 0,
                                                                                     'shows' => 14130
                                                                                   },
                                                                         'nonguarantee' => {
                                                                                             'ctr' => '0.400270810284186',
                                                                                             'price' => '1.00',
                                                                                             'shows' => 0
                                                                                           },
                                                                         'premium' => {
                                                                                        'ctr' => '4.42048250466487',
                                                                                        'price' => '1.59',
                                                                                        'shows' => 0
                                                                                      },
                                                                         'std' => {
                                                                                  'ctr' => '0.500338512855232',
                                                                                  'price' => '1.01',
                                                                                  'shows' => 0
                                                                                }
                                                                       },
                                                        'rec_pos_exps' => '71.4048101851087',
                                                        'rot_koef' => '1',
                                                        'total_shows' => 14130
                                                      },
                                          'transitions' => [
                                                             {
                                                               'cost' => '0.01',
                                                               'fctr' => 0,
                                                               'from' => 'none',
                                                               'md5' => '0490b305539f9a2d4fb47a454c3a0dda',
                                                               'phrase' => $VAR1->{'0490b305539f9a2d4fb47a454c3a0dda'}{'phrase'},
                                                               'tctr' => '0.400270810284186',
                                                               'to' => 'cent1',
                                                               'to_price' => '0.01'
                                                             },
                                                             {
                                                               'cost' => '1.74731208412059',
                                                               'fctr' => '0.400270810284186',
                                                               'from' => 'cent1',
                                                               'md5' => '0490b305539f9a2d4fb47a454c3a0dda',
                                                               'phrase' => $VAR1->{'0490b305539f9a2d4fb47a454c3a0dda'}{'phrase'},
                                                               'tctr' => '4.42048250466487',
                                                               'to' => 'premium',
                                                               'to_price' => '1.59'
                                                             }
                                                           ]
                                        },
  '2109cebe4b6728bad380b79a861f184e' => ...
}


=cut

1;
