use strict;
use warnings;
use utf8;

package Forecast::Budget;


=head1 NAME
    
    Forecast::Budget

=head1 DESCRIPTION

    Расчет бюджета рекламной кампании для выгрузки.
    Бюджет доступен двух видов: бюджет по позициям, распределенный бюджет.

=cut

use Yandex::ReportsXLS;
use GeoTools;
use Currency::Rate;
use Currencies;

use Settings;

use List::MoreUtils qw(uniq);
use Yandex::I18n;
use POSIX qw/strftime/;

use XLSCampImport;
use HttpTools;
use TextTools qw/round2s/;
use MinusWords;

=head2 new(%param)

    Конструктор

=cut

sub new {
    my ($class, %option) = @_;

    $option{format} = {
        head => {
            size => 8, bold => 1, valign => 'vcenter',
            align => 'center', border => 1, fg_color => 22,
            text_wrap => 1, pattern => 1
        },
        sheet_name => {size => 12, bold => 1, align => 'left', valign => 'vcenter', border => 0},
        sheet_params => {size => 8, bold => 1, align => 'left', valign => 'vcenter', border => 0},
        simple => {
            size => 8, align => 'center', valign => 'vcenter',
            border => 1, num_format => 1, text_wrap => 1
        },
        simple_without_wrap => {
            size => 8, align => 'center', valign => 'vcenter', border => 1, num_format => 1
        },
        price => {size => 8, align => 'center', border => 1, num_format => '0.00', text_wrap => 1, valign => 'vcenter'},
        simple_silver => {
            size => 8, align => 'center', valign => 'vcenter',
            border => 1, num_format => 1, text_wrap => 1, color => 'silver'
        },
        price_silver => {
            size => 8, align => 'center', border => 1,
            num_format => '0.00', text_wrap => 1, valign => 'vcenter', color => 'silver'
        },
        total_simple_left => {
            size => 10, align => 'left', border => 1, bold => 1,
            text_wrap => 1, valign => 'vcenter', num_format => 1, italic => 1
        },
        total_simple => {
            size => 10, align => 'center', border => 1, bold => 1,
            text_wrap => 1, valign => 'vcenter', num_format => 1, italic => 1
        },
        total_price => {
            size => 10, align => 'center', border => 1, bold => 1,
            text_wrap => 1, valign => 'vcenter', num_format => '0.00', italic => 1
        },
        phrase => {size => 8, align => 'left', border => 1, valign => 'vcenter', num_format => 1},
        simple_left => {size => 8, align => 'left', border => 1, text_wrap => 1, valign => 'vcenter', num_format => 1},
#        without border
        simple_wb => {size => 8, align => 'left', border => 0, border_color => 'white', text_wrap => 1, valign => 'vcenter'},
        simple_bold_wb => {size => 8, align => 'left', border => 0, border_color => 'white', text_wrap => 1, valign => 'vcenter', bold => 1},
        selected => {
            size => 8, align => 'center', border => 1, num_format => 1, bg_color => 62, valign => 'vcenter'
        }, 
        selected_price => {
            size => 8, align => 'center', border => 1, num_format => '0.00', bg_color => 62, valign => 'vcenter'
        },
        ($option{format} ? %{$option{format}} : ())
    };
    
    $option{format}->{simple_bold} = {%{$option{format}->{simple}}, bold => 1};
    $option{format}->{simple_left_bold} = {%{$option{format}->{simple_left}}, bold => 1};
    $option{format}->{price_bold} = {%{$option{format}->{price}}, bold => 1};
    
    @option{qw/plan word/} = ({}, {});
    bless {%option}, ref $class || $class;
}

=pod

    Здесь были функции, которые нигде не использовались и не поддерживали мультивалютность:
    - calc_by_positions (Расчет бюджета по позициям)
    - calc_allocation (Расчет распределенного бюджета)
    - estimate

=cut

sub _regions {
    
    my $self = shift;
        
    unless ($self->{_regions}) {
        my $draft = $self->{draft};
        $self->{_regions} = iget('Регионы показа: ') . join '; ', uniq map {get_geo_names($$_{geo}, ', ')} $draft->{arr} ? @{$draft->{arr}} : ($draft) 
    }
    return $self->{_regions};
}

sub _params {
    
    my $self = shift;
        
    unless ($self->{_params}) {
        my $draft = $self->{draft};
        my %currency = $self->_currency;
        $self->{_params} = iget('Бюджет: %.2f %s, Ср. цена клика: %.2f %s, Количество кликов: %i',
            $draft->{restriction}->{sum}, $currency{name},
            $draft->{restriction}->{cpc}, $currency{name}, $draft->{restriction}->{clicks})
    }
    return $self->{_params};
}

use constant MONTH => {
    1 => iget_noop('январь'),
    2 => iget_noop('февраль'),
    3 => iget_noop('март'),
    4 => iget_noop('апрель'),
    5 => iget_noop('май'),
    6 => iget_noop('июнь'),
    7 => iget_noop('июль'),
    8 => iget_noop('август'),
    9 => iget_noop('сентябрь'),
    10 => iget_noop('октябрь'),
    11 => iget_noop('ноябрь'),
    12 => iget_noop('декабрь'),
};
use constant QUARTER => {
    1 => iget_noop('1 квартал'),
    2 => iget_noop('2 квартал'),
    3 => iget_noop('3 квартал'),
    4 => iget_noop('4 квартал')
};

# Виртуальные номера строк на вкладке "План", для облегчения добавления новых строк
use constant PLAN_ROW_LOGO => 0;
use constant PLAN_ROW_TITLE => 1;
use constant PLAN_ROW_PERIOD => 2;
use constant PLAN_ROW_DEVICES => 3;
use constant PLAN_ROW_CURRENCY => 4;
use constant PLAN_ROW_SETTINGS => 5;
use constant PLAN_ROW_REGIONS => 6;
use constant PLAN_ROW_HEADER1 => 7;
use constant PLAN_ROW_HEADER2 => 8;

# Виртуальные номера строк на вкладке "Фразы", для облегчения добавления новых строк
use constant WORD_ROW_LOGO => 0;
use constant WORD_ROW_TITLE => 1;
use constant WORD_ROW_SUBTITLE => 2;
use constant WORD_ROW_PERIOD => 3;
use constant WORD_ROW_SUBPERIOD => 4;
use constant WORD_ROW_DEVICES => 5;
use constant WORD_ROW_CURRENCY => 6;
use constant WORD_ROW_SETTINGS => 7;
use constant WORD_ROW_REGIONS => 8;
use constant WORD_ROW_HEADER => 9;

# Виртуальные номера строк распределённого бюджета, для облегчения добавления новых строк
use constant ALLOCATED_ROW_LOGO => 0;
use constant ALLOCATED_ROW_TITLE => 1;
use constant ALLOCATED_ROW_SUBTITLE => 2;
use constant ALLOCATED_ROW_PARAMS => 3;
use constant ALLOCATED_ROW_PERIOD => 4;
use constant ALLOCATED_ROW_SUBPERIOD => 5;
use constant ALLOCATED_ROW_DEVICES => 6;
use constant ALLOCATED_ROW_CURRENCY => 7;
use constant ALLOCATED_ROW_SETTINGS => 8;
use constant ALLOCATED_ROW_REGIONS => 9;
use constant ALLOCATED_ROW_HEADER => 10;

sub _period {

    my $self = shift;
        
    unless (defined $self->{_period}) {
        my $draft = $self->{draft};
        $self->{_period} = [$draft->{period} eq 'month' 
            ? (iget('месяц'), iget('месяц'), $$draft{period_num} ? iget(MONTH->{$$draft{period_num}}) : undef)
            : ($draft->{period} eq 'week'
                ? (iget('неделя'), iget('неделю'), undef)
                : ($draft->{period} eq 'quarter'
                    ? (iget('квартал'), iget('квартал'), iget(QUARTER->{$$draft{period_num}}))
                    : (iget('год'), iget('год'), undef)))];
    }    
    return $self->{_period};
}

# коэффициенты для рассчета количества показов в день/неделю/месяц от общего числа показов
# в зависимости от расчетного периода
sub _period_coeff {
 
    my $self = shift;

    return $self->{draft}->{period} eq 'month'
        ? (day => 30, week => 30, week_mul => 7)
        : ($self->{draft}->{period} eq 'week'
            ? (day => 7, week => 1, month => 7, month_mul => 30)
            : ($self->{draft}->{period} eq 'quarter' ? (day => 90, month => 3) : (day => 365, month => 12)));
}

sub _settings {
    
    my $self = shift;
    
    unless (defined $self->{_settings}) {

        my $settings = $self->{draft}->{settings};
        if (ref $settings && ref $settings eq 'HASH') {
            $self->{draft}->{period} = $settings->{periodType};
            $self->{draft}->{period_num} = $settings->{periodValue};
            $self->_period;
        } else {
            $self->{_settings} = '';
        }
    }
    return $self->{_settings};
}

sub _get_logo {
    my $logo_image = (Yandex::I18n::current_lang() eq 'ru') ? 'logo-big-txt.bmp' : 'yandex_eng_logo.png';
    return  "$Settings::ROOT/data/t/$logo_image";
}

sub plan {
    
    my ($self) = @_;
    
    my ($draft, %format) = ($self->{draft}, %{$self->{format}});
    my ($cnt, $total_row) = (0, $self->{word}->{last_row} + 2);
    my $get_positions_in_order = get_positions_in_order();
    
    my @plan_periods;
    my @show_columns = (0..3);
    foreach (['show_day', iget('в день'), 4], ['show_week', iget('в неделю'), 5], ['show_month', iget('в месяц'), 6]) {
        next unless $draft->{$$_[0]};
        push @plan_periods, $$_[1];
        push @show_columns, $$_[2];        
    }
    push @plan_periods, iget('всего'); 
    
    my @merge;
    if ($draft->{show_ctr}) {
        push @show_columns, 7..12;
        push @merge, {row1 => 'row7', col1 => 'ctr_col', row_count => 1, col_count => 0};
    }
    else {
        push @show_columns, 7, 9..12;
    }

    my %related = %{$self->{word}->{related_columns}};
    my $i = 0;
    my @show_rows;
    my $last_phrase_pos = $self->{word}->{last_phrase_pos};

    my %place_texts = (
        select => iget_noop('выбранного объёма трафика'),
        (map { $_ => iget('объёма трафика %s', $PlacePrice::POSITION_CTR_CORRECTION_BY_PLACE{$_}) } (
            $PlacePrice::PLACES{PREMIUM1}, 
            $PlacePrice::PLACES{PREMIUM2}, 
            PlacePrice::get_premium_entry_place(), 
            $PlacePrice::PLACES{GUARANTEE1}, 
            PlacePrice::get_guarantee_entry_place(), 
        ))
    );
        
    foreach (@$get_positions_in_order) {
        push @show_rows, {
            pos => iget($place_texts{$_}),
            shows => $related{$_}->{shows},
            col1 => $related{$_}->{clicks},
            col2 => $related{$_}->{sum},
            first => $self->{word}->{first_phrase_pos},
            last => $last_phrase_pos,            
        } if $draft->{show_place}->{$_};
        $i++;
    }
    my %currency = $self->_currency;
    my @ignore = $self->_ignore;
    my @statics = (    
        iget('       Средние цены за клик указаны по состоянию на %s и могут быть изменены без предварительного уведомления.', strftime('%d.%m.%Y', localtime)),
        iget('       Цены указаны без учета НДС.'),
        $self->_statics,
        iget('**     Имейте в виду, что реальный бюджет может отличаться от прогнозируемого, т.к. он подсчитан на основе анализа ставок конкурентов и CTR их кампаний, а эти параметры могут изменяться в процессе работы вашей рекламной кампании.') 
            . iget(' Кроме этого, в прогнозе бюджета не учитываются ') . join(iget(' и '), @ignore),
    ); 
    
    my %coeff = $self->_period_coeff;
    my $period = $self->_period->[0];

    my @head_rows = (PLAN_ROW_LOGO, PLAN_ROW_TITLE, PLAN_ROW_PERIOD, PLAN_ROW_REGIONS, PLAN_ROW_HEADER1, PLAN_ROW_HEADER2);
    my @param_rows = (PLAN_ROW_LOGO, PLAN_ROW_TITLE, PLAN_ROW_PERIOD, PLAN_ROW_REGIONS);
    if (defined $draft->{is_mobile}) {
        push @head_rows, PLAN_ROW_DEVICES;
        push @param_rows, PLAN_ROW_DEVICES;
    }
    if (defined $draft->{currency}) {
        push @head_rows, PLAN_ROW_CURRENCY;
        push @param_rows, PLAN_ROW_CURRENCY;
    }
    if ($self->_settings) {
        push @head_rows, PLAN_ROW_SETTINGS;
        push @param_rows, PLAN_ROW_SETTINGS;
    }
    @head_rows = sort {$a <=> $b} @head_rows;
    @param_rows = sort {$a <=> $b} @param_rows;
    
    push @merge, (
        ({row1 => 'row' . PLAN_ROW_LOGO(), col1 => 0, row_count => 0, col_count => $#show_columns},
        {row1 => 'row' . PLAN_ROW_TITLE(), col1 => 0, row_count => 0, col_count => $#show_columns},
        {row1 => 'row' . PLAN_ROW_PERIOD(), col1 => 0, row_count => 0, col_count => $#show_columns},
        {row1 => 'row' . PLAN_ROW_DEVICES(), col1 => 0, row_count => 0, col_count => $#show_columns},
        {row1 => 'row' . PLAN_ROW_CURRENCY(), col1 => 0, row_count => 0, col_count => $#show_columns},
        {row1 => 'row' . PLAN_ROW_SETTINGS(), col1 => 0, row_count => 0, col_count => $#show_columns},
        {row1 => 'row' . PLAN_ROW_REGIONS(), col1 => 0, row_count => 0, col_count => $#show_columns})[@param_rows],
        {row1 => 'row' . PLAN_ROW_HEADER1(), col1 => 0, row_count => 1, col_count => 1},    
        {row1 => 'row' . PLAN_ROW_HEADER1(), col1 => 2, row_count => 1, col_count => 0},    
        {row1 => 'row' . PLAN_ROW_HEADER1(), col1 => 3, row_count => 1, col_count => 0},    
        {row1 => 'row' . PLAN_ROW_HEADER1(), col1 => 4, row_count => 0, col_count => $#plan_periods},
        {row1 => 'row' . PLAN_ROW_HEADER1(), col1 => 'clicks_col', row_count => 1, col_count => 0},
        {row1 => 'row' . PLAN_ROW_HEADER1(), col1 => 'cpc_col', row_count => 1, col_count => 1},
        {row1 => 'row' . PLAN_ROW_HEADER1(), col1 => 'budget_col', row_count => 1, col_count => 0},
    );
    
    my ($phrase, $selected) = (iget('Фразы'), iget('выбранных'));
    my $data = [
        ([
            {image => {filename => _get_logo()}, save_row => 'row' . PLAN_ROW_LOGO()}
        ],
        [{
            data => iget('План рекламной кампании ') . ($draft->{campaign_name} ? qq["$$draft{campaign_name}"] : iget('на Яндекс.Директе')),
            global_format => $format{sheet_name},
            save_row => 'row' . PLAN_ROW_TITLE()
        }],
        [{
            data => iget('Срок кампании: ') . $self->_period->[0], 
            global_format => $format{sheet_params},
            save_row => 'row' . PLAN_ROW_PERIOD()
        }],
        [{
            data => iget('Площадки') . ': ' . ($draft->{is_mobile} ? iget('только мобильные') : iget('все')),
            global_format => $format{sheet_params},
            save_row => 'row' . PLAN_ROW_DEVICES()
        }],
        [{
            data => iget('Валюта: ') . $currency{full_name},
            global_forma => $format{sheet_params},
            save_row => 'row' . PLAN_ROW_CURRENCY()
        }],
        [{data => $self->_settings, save_row => 'row' . PLAN_ROW_SETTINGS()}],
        [{data => $self->_regions, save_row => 'row' . PLAN_ROW_REGIONS()}],
        [map {  
            !ref $_
                ? {data => $_, global_format => undef, format => $format{head}}
                : {%$_, format => $format{head}}
        } iget('Места размещения рекламных материалов'), undef, iget('Рекламный носитель'),
            iget('Время размещения'), iget('Показы рекламных материалов (прогноз)'), (undef) x $#plan_periods,
            ($draft->{show_ctr} ? {data => iget('Средний CTR,%%'), save_col => 'ctr_col'} : ()),
            {data => iget('Переходы'), save_col => 'clicks_col'},
            {data => iget('Средняя цена за клик, %s', $currency{name}), save_col => 'cpc_col'}, 
            undef,
            {data => iget('Стоимость за кампанию**, %s', $currency{name}), save_col => 'budget_col', save_row => 'row' . PLAN_ROW_HEADER1()}
        ], 
        [
            ({data => undef, format => $format{head}, save_row => 'row' . PLAN_ROW_HEADER2()}) x 4,
            (map {{data => $_, format => $format{head}}} @plan_periods),
            ({data => undef, format => $format{head}}) x ($draft->{show_ctr} ? 5 : 4)
        ])[@head_rows],
        (map {
            my $pos = $_->{pos};
            my $fpostfix = $draft->{show_select} && $_->{pos} !~ /$selected/ ? '_silver' : '';
            $cnt++;
            [(
                {data => iget('Директ'), global_format => $format{$fpostfix ? "simple$fpostfix" : 'simple_bold'}},
                (map {{data => $_, global_format => $format{"simple$fpostfix"}}} iget('По фразам*'), 
                    iget('текстовый блок %s+%s символ', $MAX_TITLE_LENGTH, $MAX_BODY_LENGTH), iget('по мере расхода бюджета, примерно %s', $period)),
#                day
                sprintf('=ROUND(%s!%s%i*%i/%i, 0)', $phrase, Yandex::ReportsXLS::_colnum2letter($_->{shows}), $total_row, $coeff{day_mul} || 1, $coeff{day}),
#                week
                sprintf('=ROUND(%s!%s%i*%i/%i, 0)', $phrase, Yandex::ReportsXLS::_colnum2letter($_->{shows}), $total_row, $coeff{week_mul} || 1, $coeff{week}),
#                month
                sprintf('=ROUND(%s!%s%i*%i/%i, 0)', $phrase, Yandex::ReportsXLS::_colnum2letter($_->{shows}), $total_row, $coeff{month_mul} || 1, $coeff{month}),
#                total
                {formula => sprintf('=%s!%s%i', $phrase, Yandex::ReportsXLS::_colnum2letter($_->{shows}), $total_row), global_format => undef, format => $format{"simple$fpostfix"}, save_cell => "t$cnt"},
                {
                    # формула c SUM(лист!ячейка) не работает
                    formula => sprintf('=IF({t%i}<>0,{c%i}/{t%i}*100,"-")', $cnt, $cnt, $cnt), 
                    format => $format{"price$fpostfix"}
                },
                {formula => sprintf('=%s!%s%i', $phrase, Yandex::ReportsXLS::_colnum2letter($_->{col1}), $total_row), save_cell => "c$cnt", format => $format{"simple$fpostfix"}},
                {formula => sprintf('=IF({c%i}<>0,{b%i}/{c%i},"-")', $cnt, $cnt, $cnt), format => $format{"price$fpostfix"}},                
                {data => $pos, format => $format{"simple$fpostfix"}},
                {
                    formula => sprintf('=%s!%s%i', $phrase, Yandex::ReportsXLS::_colnum2letter($_->{col2}), $total_row),
                    format => $format{$fpostfix ? "price$fpostfix" : 'price_bold'}, 
                    save_cell => "b$cnt"
                },
            )[@show_columns]]
        } @show_rows),
        [{data => undef, global_format => undef}],
        [{data => $statics[0], format => $format{simple_bold_wb}}],
        (map {[{data => $_, format => $format{simple_wb}}]} @statics[1..@statics-1])
    ];

    $cnt += @head_rows + 1;
    return ($data, {
        sheetname => iget('План'),
        hide_gridlines => 2,
        merge_cells => [
            @merge,
            map {{row1 => $cnt + $_, col1 => 0, row_count => 0, col_count => $#show_columns}} (0..$#statics)
        ],
        set_column => [
            {col1 => 0, count => 0, width => 12},
            {col1 => 1, count => 0, width => 9},
            {col1 => 2, count => 1, width => 13},
            {col1 => 4, count => 1, width => 13},
            {col1 => $#show_columns-3, count => 0, width => 10},
            {col1 => $#show_columns-2, count => 0, width => 9},
            {col1 => $#show_columns-1, count => 0, width => 14},
            {col1 => $#show_columns, count => 0, width => 12},
        ],        
        set_row => [
            ({row => 'row' . PLAN_ROW_LOGO(), height => 65},
            {row => 'row' . PLAN_ROW_TITLE(), height => 20},
            {row => 'row' . PLAN_ROW_PERIOD(), height => 20},
            {row => 'row' . PLAN_ROW_DEVICES(), height => 20},
            {row => 'row' . PLAN_ROW_CURRENCY(), height => 20},
            {row => 'row' . PLAN_ROW_SETTINGS(), height => 20},
            {row => 'row' . PLAN_ROW_REGIONS(), height => 20},
            {row => 'row' . PLAN_ROW_HEADER1(), height => 43},
            {row => 'row' . PLAN_ROW_HEADER2(), height => 25})[@head_rows],
            (map {{row => $_, height => 33}} (scalar(@head_rows) .. $cnt-2)),
            map {
                {row => $cnt + $_, height => length($statics[$_]) > 170 ? 10 + 15 * int(length($statics[$_])/170) : 17}
            } (0 .. @statics-1)
        ],
    })
}

sub _ignore {
    
    my $self = shift;
    
    my @ignore = (iget('настройки временного таргетинга.'));
    unshift @ignore, iget('показы объявлений в сети (Рекламная сеть Яндекса и внешние сети)') unless $self->{draft}->{settings}->{context_sum_coef};
    
    return @ignore;
}

my %POSITIONRU = (
    select => iget_noop('выбранный объём трафика'),
    (map { $_ => $PlacePrice::POSITION_CTR_CORRECTION_BY_PLACE{$_} } (
        $PlacePrice::PLACES{PREMIUM1}, 
        $PlacePrice::PLACES{PREMIUM2}, 
        $PlacePrice::PLACES{PREMIUM3}, 
        PlacePrice::get_premium_entry_place(),
        $PlacePrice::PLACES{GUARANTEE1}, 
        PlacePrice::get_guarantee_entry_place(),
    ))
);

sub _get_position_headers {
    my ($self, $period, $currency) = @_;

    my $positions_texts = { 
        'select' => { 
            shows_per_period  => iget('Примерное количество показов в %s (при выбранном объёме трафика)*', $period), # 3
            clicks_per_period => iget('Примерное количество переходов в %s (при выбранном объёме трафика)*', $period), # 9
            ctr_forecast      => iget('Прогноз CTR (при выбранном объёме трафика)'), # 15
            avg_bid_price     => iget('Средняя установленная ставка (при выбранном объёме трафика), %s', $currency), # 21
            avg_amnesty_price => iget('Средняя списываемая цена клика (при выбранном объёме трафика), %s', $currency), # 22
            avg_budget        => iget('Примерный бюджет, %s (при выбранном объёме трафика)', $currency), # 33
        },
    };
    my @traffic_volume_places = ($PlacePrice::PLACES{PREMIUM1}, $PlacePrice::PLACES{PREMIUM2}, PlacePrice::get_premium_entry_place(), 
         $PlacePrice::PLACES{GUARANTEE1}, PlacePrice::get_guarantee_entry_place());

    foreach my $place (@traffic_volume_places) {
        my $traffic_volume = $PlacePrice::POSITION_CTR_CORRECTION_BY_PLACE{ $place }; 
        $positions_texts->{ $place } = { 
            shows_per_period  => iget('Примерное количество показов в %s (при объёме трафика %s)*', $period, $traffic_volume), # 4
            clicks_per_period => iget('Примерное количество переходов в %s (при объёме трафика %s)*', $period, $traffic_volume), # 10
            ctr_forecast      => iget('Прогноз CTR (при объёме трафика %s)', $traffic_volume), # 16
            avg_bid_price     => iget('Примерная установленная ставка (при объёме трафика %s), %s', $traffic_volume, $currency), # 23
            avg_amnesty_price => iget('Примерная списываемая цена клика (при объёме трафика %s), %s', $traffic_volume, $currency), # 24
            avg_budget        => iget('Примерный бюджет, %s (при объёме трафика %s)', $currency, $traffic_volume), # 34
        };
    }
    my @places = ('select', @traffic_volume_places);

    my @headers;
    foreach my $field (qw/shows_per_period clicks_per_period ctr_forecast/) {
        push @headers, map { $positions_texts->{$_}->{ $field } } @places;
    } 
    push @headers, map { ($positions_texts->{$_}->{avg_bid_price}, $positions_texts->{$_}->{avg_amnesty_price}) } @places;
    push @headers, map { $positions_texts->{$_}->{avg_budget} } @places;
    return @headers;
}

sub word {
    
    my ($self) = @_;
    
    my ($cnt, $draft, %format) = (0, $self->{draft}, %{$self->{format}});

    my $get_positions_in_order = get_positions_in_order();
    
    my $saved_cols = {};
    my @show_columns = (0,1);
    push @show_columns, 2 if $draft->{show_select};
    if (!$draft->{separate_shows}) {
        # Для всех позиций используется общая колонка показов
        push @show_columns, 2;
        foreach my $place (@$get_positions_in_order) {
            if ($draft->{show_place}->{$place}) {
                $saved_cols->{$place}->{shows} = $draft->{show_select} ? 2 : 1;
            }
        }
    }
    foreach (3,9,15,21,33) {
        # На самом деле эти цифры не используются, они здесь больше для того, чтобы легче было понимать что это
        # Это номер колонок с которых начинается другой тип данных. То есть:
        # с 3й колонки начинаются раздельные показы
        # с 9й  - клики
        # с 15й - CTR
        # с 21й - цены, которые между собой перемежаются
        # с 33й - бюджеты
        # Вся проблема в том, чтобы найти когда будут цены и правильно оставить только нужные колонки
        my $col = $_;
        foreach my $place (@$get_positions_in_order) {
            next if $_ == 3 && !$draft->{separate_shows};
            next if $_ == 15 && !$draft->{show_ctr};
            if ($draft->{show_place}->{$place}) {
                push @show_columns, $col++;
                push @show_columns, $col++ if $_ == 21;
                # Запоминаем номера колонок в clicks - клики, в sum - бюджет.
                $saved_cols->{$place}->{shows} = $#show_columns if ($_ == 3);
                $saved_cols->{$place}->{clicks} = $#show_columns if ($_ == 9);
                $saved_cols->{$place}->{sum} = $#show_columns if ($_ == 33);
            } else {
                $col++;
                $col++ if $_ == 21;
            }
        }
    }
    @show_columns = sort {$a <=> $b} @show_columns;
    my $last_col = @show_columns - 1;

    my @ignore = $self->_ignore;
    my %currency = $self->_currency;

    my @statics = (
        iget('       Средние цены за клик указаны по состоянию на %s и могут быть изменены без предварительного уведомления.', strftime('%d.%m.%Y', localtime)),
        iget('       Цены указаны без учета НДС.'),
        $self->_statics,
        iget('**     Имейте в виду, что реальный бюджет может отличаться от прогнозируемого, т.к. он подсчитан на основе анализа ставок конкурентов и CTR их кампаний, а эти параметры могут изменяться в процессе работы вашей рекламной кампании.') . 
            iget(' Кроме этого, в прогнозе бюджета не учитываются ') . join(iget(' и '), @ignore),            
        iget('       Реклама по фразе будет показываться во всех случаях, когда в запросе присутствует эта фраза. Так, реклама по слову "мебель" будет показана и тому, кто спросил "мебель", и тому, кто спросил "каталог мебели".'),
        iget('       "Минус-фразы" и "минус-слова" используются для дополнительного уточнения слов (словосочетаний). Так, реклама по условию "шкаф -жарочный" будет показана по запросам "продажа шкафов", "шкаф-купе", но не будет показана по запросу "жарочный шкаф".'),
        iget('       В случае, если ставка окажется недостаточной для показа на поиске, реальное количество показов и бюджет по соответствующим словам (словосочетаниям) может оказаться существенно меньше прогнозируемого.')
    );

    my $period = $self->_period;    
    my @show_rows = (WORD_ROW_LOGO, WORD_ROW_TITLE, WORD_ROW_SUBTITLE, WORD_ROW_PERIOD, WORD_ROW_REGIONS, WORD_ROW_HEADER);
    my @param_rows = (WORD_ROW_LOGO, WORD_ROW_TITLE, WORD_ROW_SUBTITLE, WORD_ROW_PERIOD, WORD_ROW_REGIONS);
    if ($period->[2]) {
        push @show_rows, WORD_ROW_SUBPERIOD;
        push @param_rows, WORD_ROW_SUBPERIOD;
    }
    if (defined $draft->{is_mobile}) {
        push @show_rows, WORD_ROW_DEVICES;
        push @param_rows, WORD_ROW_DEVICES;
    }
    if (defined $draft->{currency}) {
        push @show_rows, WORD_ROW_CURRENCY;
        push @param_rows, WORD_ROW_CURRENCY;
    }
    if ($self->_settings) {
        push @show_rows, WORD_ROW_SETTINGS;
        push @param_rows, WORD_ROW_SETTINGS;
    }
    @show_rows = sort {$a <=> $b} @show_rows;
    @param_rows = sort {$a <=> $b} @param_rows;
    
    my @position_headers = $self->_get_position_headers($period->[1], $currency{name});

    $format{total_price_silver} = {
        size => 10, align => 'center', border => 1,
        text_wrap => 1, valign => 'vcenter', num_format => '0.00', italic => 1,
        color => 'silver'
    };
    $format{simple_selected} = {%{$format{simple}}, bg_color => 62};
    $format{price_selected} = {%{$format{price}}, bg_color => 62};
    my $context = $draft->{context};
    my $data = [
        ([
            {image => {filename => _get_logo()}, save_row => 'row' . WORD_ROW_LOGO()}
        ],
        [{
            data => iget('План рекламной кампании ') . ($draft->{campaign_name} ? qq["$$draft{campaign_name}"] : iget('на Яндекс.Директе')), 
            format => $format{sheet_name},
            save_row => 'row' . WORD_ROW_TITLE(),
        }],
        [{data => iget('(Предложение ключевых фраз)'), global_format => $format{sheet_params}, save_row => 'row' . WORD_ROW_SUBTITLE()}],
        [{data => iget('Срок кампании: %s', $period->[0]), save_row => 'row' . WORD_ROW_PERIOD()}],
        [{data => iget('Прогноз бюджета на: %s', $period->[2]), save_row => 'row' . WORD_ROW_SUBPERIOD()}],
        [{data => iget('Площадки') . ': ' . ($draft->{is_mobile} ? iget('только мобильные') : iget('все')), save_row => 'row' . WORD_ROW_DEVICES()}],
        [{data => iget('Валюта: %s', $currency{full_name}), save_row => 'row' . WORD_ROW_CURRENCY()}],
        [{data => $self->_settings, save_row => 'row' . WORD_ROW_SETTINGS()}],
        [{data => $self->_regions, save_row => 'row' . WORD_ROW_REGIONS()}],
        [(map {{data => $_, global_format => undef, format => $format{head}, save_row => 'row' . WORD_ROW_HEADER()}} (
            iget('Предложенные фразы'), # 0 
            iget('Примерное количество запросов'), # 1
            iget('Объём трафика'),  # 2
            @position_headers,
        ))[@show_columns]] ) [@show_rows],
        (map { ## no critic (ProhibitComplexMappings)
            my $record;
            if (exists $POSITIONRU{$$_{position}}) {

                $cnt++;
                my $row = $_;
                my $prefix = $$_{position};

                my %phrase = ();
                foreach my $pos (@$get_positions_in_order) {
                    next if $pos eq 'select';
                    $row->{$pos}->{$_} = round2s($row->{$pos}->{$_} / 1e6) foreach (qw/sum bid_price amnesty_price/);
                }

                foreach my $position (@$get_positions_in_order) {
                    $phrase{$position} = { map { my $selected = ($position eq 'select') ? '_selected' : '';
                                                $_ => { data => ($position eq 'select') ? $row->{$row->{position}}->{$_} : $row->{$position}->{$_},
                                                         format => /price|sum|ctr/ ? $format{"price$selected"} : $format{"simple$selected"},
                                                         save_cell => "$position${_}$cnt",
                                                        }
                                               } qw/shows clicks ctr bid_price amnesty_price sum/
                                         };
                }
                $phrase{shows} = { data => $row->{shows},
                                   format => $format{simple},
                                   save_cell => "shows$cnt",
                                 };

                $phrase{pos} = iget($POSITIONRU{$$_{position}});
                $record = [(
                    {data => $_->{phrase}, format => $format{phrase}},
                    $phrase{shows},
                    {data => $phrase{pos}, format => $format{simple_without_wrap}},
                    (map {$phrase{$_}->{shows}} @$get_positions_in_order),
                    (map {$phrase{$_}->{clicks}} @$get_positions_in_order),
                    (map {$phrase{$_}->{ctr}} @$get_positions_in_order),
                    (map {($phrase{$_}->{bid_price}, $phrase{$_}->{amnesty_price})} @$get_positions_in_order),
                    (map {$phrase{$_}->{sum}} @$get_positions_in_order),

                )[@show_columns]];
            }
            $record || ();
        } sort {
            lc($a->{phrase}|| '') cmp lc($b->{phrase}|| '')
        } @{$draft->{phrases}}),
        [(
            {data => iget('Итого с учетом выбранного объёма трафика**'), format => $format{total_simple_left}},
            {formula => sprintf('=SUM({%s1}:{%s%i})', 'shows', 'shows', $cnt), format => $format{/select/ ? 'total_simple' : 'simple'}},
            {data => undef, format => $format{simple}},
            (map {{formula => sprintf('=SUM({%s1}:{%s%i})', $_."shows", $_."shows", $cnt), format => $format{/select/ ? 'total_simple' : 'simple'}}} 
                @$get_positions_in_order),
            (map {{formula => sprintf('=SUM({%s1}:{%s%i})', $_."clicks", $_."clicks", $cnt), format => $format{/select/ ? 'total_simple' : 'simple'}}} 
                @$get_positions_in_order),
            ({data => undef, format => $format{simple}}) x (scalar(@$get_positions_in_order) * 3),
            (map {{formula => sprintf('=SUM({%ssum1}:{%ssum%i})', $_, $_ , $cnt), format => $format{/select/ ? 'total_price' : 'price'}}} 
                @$get_positions_in_order),
        )[@show_columns]],
        [{data => undef, global_format => undef}],        
        [{data => $statics[0], format => $format{simple_bold_wb}}],
        (map {[{data => $_, global_format => undef, format => $format{simple_wb}}]} @statics[1..@statics-1])
    ];
    
    my $last_row = $cnt + @show_rows - 1;
    $self->{word} = {
        last_row => $last_row,
        last_phrase_pos => @show_rows + $cnt,
        first_phrase_pos => @show_rows + 1,
        related_columns => $saved_cols,
    };
    
    $cnt += @show_rows + 2;
    return ($data, {
        sheetname => iget('Фразы'),
        sheetcolor => 'red',
        set_color => [
            [62, 225, 225, 225],
        ],
        set_row => [
            ({row => 'row' . WORD_ROW_LOGO(), height => 65},
            {row => 'row' . WORD_ROW_TITLE(), height => 20},
            {row => 'row' . WORD_ROW_SUBTITLE(), height => 20},
            {row => 'row' . WORD_ROW_PERIOD(), height => 20},
            {row => 'row' . WORD_ROW_SUBPERIOD(), height => 20},
            {row => 'row' . WORD_ROW_DEVICES(), height => 20},
            {row => 'row' . WORD_ROW_CURRENCY(), height => 20},
            {row => 'row' . WORD_ROW_SETTINGS(), height => 20},
            {row => 'row' . WORD_ROW_REGIONS(), height => 20},
            {row => 'row' . WORD_ROW_HEADER(), height => 77})[@show_rows],
            (map {
                {row => $cnt + $_, count => 0, height => length $statics[$_] > 130 ? 10 + 15 * int(length($statics[$_])/130) : 17}
            } (0 .. @statics-1))
        ],
        hide_gridlines => 2,
        print_orient => 'landscape',
        set_column => [
            {col1 => 0, count => 0, width => 40},
            ($draft->{show_select}
                ? ({col1 => 1, count => 0, width => 13}, {col1 => 2, col2 => $last_col, width => 12}) 
                : {col1 => 1, col2 => $last_col, width => 12}),
        ],
        merge_cells => [            
            (map {{row1 => "row$_", col1 => 0, row_count => 0, col_count => 19}} @param_rows),
            map {{row1 => $last_row + $_, col1 => 0, row_count => 0, col_count => 6}} (2..10)
        ]        
    });    
}

=head2 get_positions_in_order

    Возвращает список позиций, который может показываться в XLS.

=cut
sub get_positions_in_order {
    return ['select', 
            $PlacePrice::PLACES{PREMIUM1},
            $PlacePrice::PLACES{PREMIUM2},
            PlacePrice::get_premium_entry_place(),
            $PlacePrice::PLACES{GUARANTEE1},
            PlacePrice::get_guarantee_entry_place(),
           ];
}

sub minus_words {
    
    my ($self) = @_;
    
    ([
        [undef],
        [{data => iget('Единые минус-фразы'), format => $self->{format}->{head}}],
        map {
            [{data => $_, format => $self->{format}->{simple}}]
        } @{$self->{draft}->{minus_words}}
    ], {
        sheetname => iget('Единые минус-фразы'),
        hide_gridlines => 2,
        set_column => [
            {col1 => 0, count => 0, width => 20},
        ]
    })
}

sub _with_minus_words {
    my $draft = shift->{draft};
    ref $draft->{minus_words} && ref $draft->{minus_words} eq 'ARRAY';
}

sub _statics {

    my $self = shift;
    return (        
        $self->_with_minus_words
            ? iget('*      Число показов и переходов рассчитано на основании данных статистики за предыдущий период с учетом заданных единых минус-фраз (список приведен на вкладке Единые минус-фразы).')
            : iget('*      Число показов и переходов рассчитано на основании данных статистики за предыдущий период.')
    )
}

# Возвращает хеш с информацией о валюте денежных значений
#   name => имя валюты в которой записаны денежные значения
#   rate => курс валюты по отношению к фишкам
#   full_name => полное имя валюты
# Если валюта не указана, то возвращаются значения для у.е.
sub _currency {
    my $self = shift;
    my $currency = $self->{draft}->{currency} || 'YND_FIXED';
    my $name = get_currency_constant($currency, 'name');
    my $full_name = get_currency_constant($currency, 'full_name');
    my $rate = $self->{_currency_rate};
    if (!defined $rate) {
        if ($currency ne 'YND_FIXED') {
            # Делаем конвертацию через рубли, чтобы валюта считалась по текущему, а не фиксированному курсу
            $rate = convert_currency(convert_currency(1, 'YND_FIXED', 'RUB', with_nds => 1), 'RUB', $currency, with_nds => 1);
        } else {
            $rate = 1;
        }
        $self->{_currency_rate} = $rate;
    }
    return (name => $name, rate => $rate, full_name => $full_name);
}

sub _ifnull {shift || '-'}

sub allocated {
    
    my $self = shift;
  
    my ($draft, $period, $cnt) = ($self->{draft}, $self->_period, 0);
    my %format = %{$self->{format}};

    my %currency = $self->_currency;
    my @statics = (
        iget('       Средние цены за клик указаны по состоянию на %s и могут быть изменены без предварительного уведомления.', strftime('%d.%m.%Y', localtime)),
        iget('       Цены указаны без учета НДС.'),
        $self->_statics,
        iget('**     Имейте в виду, что реальный бюджет может отличаться от прогнозируемого, т.к. он подсчитан на основе анализа ставок конкурентов и CTR их кампаний, а эти параметры могут изменяться в процессе работы вашей рекламной кампании.') . 
            iget(' Кроме этого, в прогнозе бюджета не учитываются ') . join(iget(' и '), $self->_ignore)
    );

    my @show_rows = (ALLOCATED_ROW_LOGO, ALLOCATED_ROW_TITLE, ALLOCATED_ROW_SUBTITLE, ALLOCATED_ROW_PARAMS, ALLOCATED_ROW_PERIOD, ALLOCATED_ROW_REGIONS, ALLOCATED_ROW_HEADER);
    my @param_rows = (ALLOCATED_ROW_LOGO, ALLOCATED_ROW_TITLE, ALLOCATED_ROW_SUBTITLE, ALLOCATED_ROW_PARAMS, ALLOCATED_ROW_PERIOD, ALLOCATED_ROW_REGIONS);
    if ($period->[2]) {
        push @show_rows, ALLOCATED_ROW_SUBPERIOD;
        push @param_rows, ALLOCATED_ROW_SUBPERIOD;
    }
    if (defined $draft->{is_mobile}) {
        push @show_rows, ALLOCATED_ROW_DEVICES;
        push @param_rows, ALLOCATED_ROW_DEVICES;
    }
    if (defined $draft->{currency}) {
        push @show_rows, ALLOCATED_ROW_CURRENCY;
        push @param_rows, ALLOCATED_ROW_CURRENCY;
    }
    if ($self->_settings) {
        push @show_rows, ALLOCATED_ROW_SETTINGS;
        push @param_rows, ALLOCATED_ROW_SETTINGS;
    }
    @show_rows = sort {$a <=> $b} @show_rows;
    @param_rows = sort {$a <=> $b} @param_rows;
    $_->{positions} = [grep {$_} @{$_->{positions}}] foreach (@{$draft->{phrases}});
    my ($settings, $context) = @{$draft}{qw/settings context/};
    my $data = [
        ([
            {image => {filename => _get_logo()}, save_row => 'row' . ALLOCATED_ROW_LOGO()}
        ],
        [{
            data => iget('План рекламной кампании ') . ($draft->{campaign_name} ? qq["$$draft{campaign_name}"] : iget('на Яндекс.Директе')),
            format => $format{sheet_name},
            save_row => 'row' . ALLOCATED_ROW_TITLE()
        }],
        [{data => iget('(Предложение ключевых фраз)'), global_format => $format{sheet_params}, save_row => 'row' . ALLOCATED_ROW_SUBTITLE()}],
        [{data => iget('Расчет бюджета по параметрам: %s', $self->_params), save_row => 'row' . ALLOCATED_ROW_PARAMS()}],
        [{data => iget('Срок кампании: %s', $period->[0]), save_row => 'row' . ALLOCATED_ROW_PERIOD()}],
        [{data => iget('Прогноз бюджета на: %s', $period->[2]), save_row => 'row' . ALLOCATED_ROW_SUBPERIOD()}],
        [{data => iget('Площадки') . ': ' . ($draft->{is_mobile} ? iget('только мобильные') : iget('все')), save_row => 'row' . ALLOCATED_ROW_DEVICES()}],
        [{data => iget('Валюта: %s', $currency{full_name}), save_row => 'row' . ALLOCATED_ROW_CURRENCY()}],
        [{data => $self->_settings, save_row => 'row' . ALLOCATED_ROW_SETTINGS()}],
        [{data => $self->_regions, save_row => 'row' . ALLOCATED_ROW_REGIONS()}],
        [map {{data => $_, global_format => undef, format => $format{head}, save_row => 'row' . ALLOCATED_ROW_REGIONS()}} (
            iget('Предложенные фразы'), iget('Позиция'), 
            iget('Количество показов в %s* (прогноз)', $period->[1]),
            iget('Примерное количество переходов в %s (при заданных параметрах)*', $period->[1]),
            ($draft->{show_ctr} ? iget('Прогноз CTR (при заданных параметрах)') : ()),
            iget('Средняя цена клика (при заданных параметрах), %s', $currency{name}),
            iget('Примерный бюджет, %s (при заданных параметрах)', $currency{name})
        )])[@show_rows],
        (map {
            $cnt++;
            [
                {data => $_->{phrase}, format => $format{phrase}},
                {data => join(',', map {$POSITIONRU{$_}} @{$_->{positions}}), format => $format{simple_without_wrap}},
                {data => _ifnull($_->{shows}), format => $format{simple}, save_cell => "as$cnt"},
                {data => _ifnull($_->{clicks}), save_cell => "ac$cnt", format => $format{simple}},
                ($draft->{show_ctr} ? {formula => qq[=IF(AND(ISNUMBER({as$cnt}),{as$cnt} <> 0),{ac$cnt}/{as$cnt}*100,"-")], format => $format{price}} : ()),
                {data => _ifnull($_->{amnesty_price}), save_cell => "ap$cnt", format => $format{price}},
                {
                    formula => sprintf('=IF(ISNUMBER({ac%i}*{ap%i}),{ac%i}*{ap%i},0)', $cnt, $cnt, $cnt, $cnt),
                    save_cell => "ab$cnt", format => $format{price}
                }
            ]
        } @{$draft->{phrases}}),
        [
            {data => iget('Итого с учетом выбранных позиций**'), global_format => $format{total_simple_left}},
            undef,
            {formula => "=SUM({as1}:{as$cnt})", save_cell => 'as_total', global_format => undef, format => $format{simple}},
            {formula => "=SUM({ac1}:{ac$cnt})", save_cell => 'ac_total', format => $format{total_simple}},
            ($draft->{show_ctr} ? {formula => '=IF({as_total} <> 0,{ac_total}/{as_total}*100,"-")', format => $format{price}} : ()),
            {formula => '=IF({ac_total} <> 0,{ab_total}/{ac_total},"-")', format => $format{price}},
            {formula => "=SUM({ab1}:{ab$cnt})", save_cell => 'ab_total', format => $format{total_price}}
        ],
        [{data => undef, global_format => undef}],        
        [{data => $statics[0], format => $format{simple_bold_wb}}],
        (map {[{data => $_, global_format => undef, format => $format{simple_wb}}]} @statics[1..@statics-1])
    ];
    
    $cnt += @show_rows + 2;
    return ($data, {
        sheetname => iget('Фразы'),
        sheetcolor => 'red',
        hide_gridlines => 2,
        print_orient => 'landscape',
        set_row => [
            ({row => 'row' . ALLOCATED_ROW_LOGO(), height => 65},
            {row => 'row' . ALLOCATED_ROW_TITLE(), height => 20},
            {row => 'row' . ALLOCATED_ROW_SUBTITLE(), height => 20},
            {row => 'row' . ALLOCATED_ROW_PARAMS(), height => 20},
            {row => 'row' . ALLOCATED_ROW_PERIOD(), height => 20},
            {row => 'row' . ALLOCATED_ROW_SUBPERIOD(), height => 20},
            {row => 'row' . ALLOCATED_ROW_DEVICES(), height => 20},
            {row => 'row' . ALLOCATED_ROW_CURRENCY(), height => 20},
            {row => 'row' . ALLOCATED_ROW_SETTINGS(), height => 20},
            {row => 'row' . ALLOCATED_ROW_REGIONS(), height => 20},
            {row => 'row' . ALLOCATED_ROW_HEADER(), height => 77})[@show_rows],
            (map {
                {row => $cnt + $_, count => 0, height => length $statics[$_] > 170 ? 10 + 15 * int(length($statics[$_])/170) : 17}
            } (0..$#statics))
            
        ],
        merge_cells => [      
            (map {{row1 => "row$_", col1 => 0, row_count => 0, col_count => 6}} @param_rows),      
            map {{row1 => $cnt + $_, col1 => 0, row_count => 0, col_count => 6}} (0..$#statics)
        ],
        set_column => [
            {col1 => 0, count => 0, width => 40},
            {col1 => 1, count => 0, width => 23},
            {col1 => 2, col2 => 6, width => 12},
        ],
    })
}


=head2 by_positions($draft)

    Формирование данных для выгрузки бюджета по позициям. 

    Параметры:
        $draft - ссылка на хеш с данными по 
            minus_words - ссылка на масив с общими минус фразами
            phrases - ссылка на масив фраз/категорий
            settings - ссылка на хеш с параметрами расчета бюджета(период, учет РСЯ, автоматически добавленных фраз, ссылки для 1-ого спецразмещения)
    Возвращает:
        Объект Forecast::Budget 

=cut

sub by_positions {
    
    my ($self, $draft) = @_;
    
    $self->{draft} = $draft;
    
    $self->_settings;
    my ($sheet1, $format1) = $self->word;
    my ($sheet3, $format3) = $self->plan;
    my ($sheet2, $format2) = $self->_with_minus_words ? $self->minus_words : (undef, undef);
    
    $self->{sheets} = [$sheet1, $sheet2 || (), $sheet3];
    $self->{sheets_formats} = [$format1, $format2 || (), $format3];
    
    return $self;
}

=head2 allocation_budget($draft)

    Формирование данных для выгрузки распределенного бюджета. 

    Параметры:
        $draft - ссылка на хеш с данными по 
            minus_words - ссылка на масив с общими минус фразами
            phrases - ссылка на масив фраз/категорий
            settings - ссылка на хеш с параметрами расчета бюджета(период, учет РСЯ, автоматически добавленных фраз, ссылки для 1-ого спецразмещения)
            restriction - ссылка на хеш с ограничениями(по цене, по общему бюджеты, по количеству кликов) для прогноз
    Возвращает:
        Объект Forecast::Budget 

=cut

sub allocation_budget {
    
    my ($self, $draft) = @_;
    
    $self->{draft} = $draft;
    $self->_settings;
    
    my ($sheet1, $format1) = $self->allocated;
    my ($sheet2, $format2) = $self->_with_minus_words ? $self->minus_words : (undef, undef);

    $self->{sheets} = [$sheet1, $sheet2 || ()];
    $self->{sheets_formats} = [$format1, $format2 || ()];

    return $self;
}


=head2 as_xls()

    Выгрузка бюджета(распределенного или по позициям) в Excel. 
    Выгрузка в STDOUT.

=cut

sub as_xls {
    
    my ($self) = @_;
    my $xls_report = Yandex::ReportsXLS->new();
    return $xls_report->array2excel2scalar($self->{sheets}, $self->{sheets_formats});
}

sub as_xls_camp {
    my ($self, %opt) = @_;
    my $camp = $self->forecast2camp;
    my $xls = camp_snapshot2excel($camp, host => http_server_host($self->{r}), ClientID => $opt{ClientID});
    return $xls;
}

=head2 forecast2camp

Приготовить структуру $camp для отправки в XLSCampImport::camp_snapshot2excel

{
    banners => [
        {
            geo => $geo_id,
            phrases => [
                {
                    phr   => $phrase,
                    price => $cpc,
                },
            ],
        },
    ],
}

=cut

sub forecast2camp
{
    my ($self) = @_;

    my $draft = $self->{draft};
    my $camp = get_sample_camp_snapshot();

    # my %position_abbr = (
    #     premium       => 'p_',
    #     first_premium => 'fprem_',
    #     first_place   => 'fp_',
    #     std           => '',
    # );

    $camp->{currency} = $draft->{currency} || 'YND_FIXED';

    # Преобразуем минус-фразы из черновика в единые минус-фразы на кампанию
    $camp->{campaign_minus_words} = MinusWords::polish_minus_words_array($draft->{minus_words}) if $draft->{minus_words};
    $camp->{groups} = [{
        geo     => $draft->{geo},
        group_name => iget('Группа объявлений 1'),
        banners => [{
            title   => iget('Текст заголовка. Максимум %s символов', $MAX_TITLE_LENGTH),
            body    => iget('Текст объявления. Максимум %s символ', $MAX_BODY_LENGTH),
            bid     => '',
            href    => '',
            banner_type => 'desktop',
            real_banner_type => 'text',
        }],
        phrases => [ map {{
            phr    => $_->{phrase},
            phrase => $_->{phrase},
            price  => defined $_->{position} ? $_->{$_->{position}}->{bid_price} : $_->{bid_price}, # Прогноз, это хорошо, но какое значение сохранять в кампании? получается bid_price
#            price  => $_->{POSPREFIX->{$_->{position}}.'cpc'},
        }} @{$draft->{phrases}} ],
        tags    => []
    }];

    return $camp;
}

1;
