package ForecastXLS;
## no critic (TestingAndDebugging::RequireUseWarnings)

# $Id$
# Вывод результатов прогноза в формате XLS
# Автор: Сергей Журавлёв

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);
require Exporter;
use POSIX qw/strftime/;

use Settings;
use Currencies;
use Currency::Format;
use Currency::Pseudo;
use GeoTools;
use PlacePrice qw/get_data_by_place/;

use List::MoreUtils qw(uniq any);
use Yandex::I18n;
use Spreadsheet::WriteExcel;
use utf8;

# set the version for version checking
$VERSION     = 1.00;
@ISA         = qw(Exporter);
@EXPORT_OK = @EXPORT = qw(forecast_xls_new);

my %POSITIONS_TEXTS = (
    'select' => { name => iget("выбранного объёма трафика"),
                  clicks_per_month => iget("Примерное количество переходов в месяц (при выбранном объёме трафика)*"),
                  ctr_forecast => iget("Прогноз CTR (при выбранном объёме трафика)"),
                  avg_bid_price => iget("Средняя установленная ставка (при выбранном объёме трафика), %s"),
                  avg_amnesty_price => iget("Средняя списываемая цена клика (при выбранном объёме трафика), %s"),
                  avg_budget => iget("Примерный бюджет, %s (при выбранном объёме трафика)"),
                },
    (
        map { $_ => _get_texts_by_position($_) }
        ($PlacePrice::PLACES{PREMIUM1}, $PlacePrice::PLACES{PREMIUM2}, PlacePrice::get_premium_entry_place(), 
         $PlacePrice::PLACES{GUARANTEE1}, PlacePrice::get_guarantee_entry_place())
    )
    );

sub _get_texts_by_position {
    my ($place) = @_;
    my $position_ctr_correction = $PlacePrice::POSITION_CTR_CORRECTION_BY_PLACE{ $place }; 
     
    return {
        name => iget("объёма трафика %s", $position_ctr_correction),
        clicks_per_month => iget("Примерное количество переходов в месяц (при объёме трафика %s)*", $position_ctr_correction),
        ctr_forecast => iget("Прогноз CTR (при объёме трафика %s)", $position_ctr_correction),
        avg_bid_price => iget("Примерная установленная ставка (при объёме трафика %s)", $position_ctr_correction) . ", %s",
        avg_amnesty_price => iget("Примерная списываемая цена клика (при объёме трафика %s)", $position_ctr_correction) . ", %s",
        avg_budget => iget("Примерный бюджет, %s ") . iget("(при объёме трафика %s)", $position_ctr_correction),
    };
}

my %POSITIONS_CAPTIONS = (
        map { $_ => $PlacePrice::POSITION_CTR_CORRECTION_BY_PLACE{ $_ } }
        ($PlacePrice::PLACES{PREMIUM1}, $PlacePrice::PLACES{PREMIUM2},
         $PlacePrice::PLACES{PREMIUM3}, PlacePrice::get_premium_entry_place(),
         $PlacePrice::PLACES{GUARANTEE1}, PlacePrice::get_guarantee_entry_place())
    );

my $PLAN = iget("План");
my $WORDS = iget("Слова");
#...........................................................................................................
# Вывод прогноза в виде XLS (new)
#
#   Внимание, этой функцией делают не только прогноз бюджета, но и выгрузку Медиапланов в XLS.
#
sub forecast_xls_new($$$)
{
    my ($vars, $login_rights, $out) = @_;

    $Carp::Verbose = 1;

    # Создаём таблицу
    my $work = Spreadsheet::WriteExcel->new($out);
    $vars->{show_day} =    ($vars->{show_period} && $vars->{show_period} =~ /day/ ? 1 : 0);
    $vars->{show_week} =   ($vars->{show_period} && $vars->{show_period} =~ /week/ ? 1 : 0);
    $vars->{show_ctr} =    ($vars->{show_plan}   && $vars->{show_plan} =~ /ctr/ ? 1 : 0);

    my $valid_positions = {map {$_=>1} @{get_positions_in_order()}};
    $vars->{show_place} = {map {$_=>1} grep {$valid_positions->{$_}} split /\s*,\s*/, $vars->{show_plan}};

    my $planlist;
    if ($login_rights->{role} !~ /client|empty/) {
        $planlist = $work->addworksheet($PLAN);
    }
    my $directlist = $work->addworksheet($WORDS);

    my $ferrari = $work->set_custom_color(55, 255, 55, 55);
    $directlist->set_tab_color($ferrari);

    if ($login_rights->{role} !~ /client|empty/) {
        forecast_draw_plan_new($work, $planlist, $vars, $login_rights);
    }

    if ($login_rights->{role} =~ /client|empty/) {
        $vars->{show_select} = 1;
    }
    forecast_draw_words_new($work, $directlist, $vars, $login_rights);
}

sub show_plan_place_row {
    my (%params) = @_;
    my $row = $params{row};
    my $vars = $params{vars};
    my $list = $params{list};
    my $formats = $params{formats};
    my $words_result_row = $params{words_result_row};
    my $select_col = ($vars->{show_place}->{'select'}) ? 1 : 0;
    my $pseudo_currency = (defined($vars->{pseudo_currency})) ? { %{$vars->{pseudo_currency}} } : get_pseudo_currency( id => $vars->{pseudo_currency_id} );
    $pseudo_currency->{name} = iget($pseudo_currency->{name});
    my $rate = ($vars->{currency} eq "YND_FIXED") ? $pseudo_currency->{rate} : 1; 

    my $res_col  = xl(8 + $vars->{show_ctr} + $vars->{show_day} + $vars->{show_week});
    my $res_col2 = xl(5 + $vars->{show_ctr} + $vars->{show_day} + $vars->{show_week});
    my $res_col1_ctr = xl(6 + $vars->{show_day} + $vars->{show_week});
    my $res_col2_ctr = xl(4 + $vars->{show_day} + $vars->{show_week});

    my $col = 0;
    $list->write($row, $col++, iget("Директ"), $formats->{data_left_bold});
    $list->write($row, $col++, iget("По словам*"), $formats->{data_left});
    $list->write($row, $col++, iget("текстовый блок $MAX_TITLE_LENGTH+$MAX_BODY_LENGTH символов"), $formats->{data_left});
    $list->write($row, $col++, iget("по мере расхода бюджета, примерно месяц"), $formats->{data_left});
    if ($vars->{show_day}) {
        $list->write_formula($row, $col++, "=ROUND($WORDS!C$words_result_row / 30, 0)", $formats->{data});
    }
    if ($vars->{show_week}) {
        $list->write_formula($row, $col++, "=ROUND(7 * $WORDS!C$words_result_row / 30, 0)", $formats->{data});
    }
    $list->write_formula($row, $col++, "=$WORDS!C$words_result_row", $formats->{data});

    my $row_num = $row + 1;
    # ctr
    if ($vars->{show_ctr}) {
        $list->write_formula($row, $col++, "=IF($PLAN!$res_col2_ctr$row_num <> 0, $res_col1_ctr$row_num / $res_col2_ctr$row_num * 100, \"-\")", $formats->{data_price});
    }
    my $words_col = 0;
    foreach (@{get_positions_in_order()}) {
      last if ($params{place} eq $_);
      $words_col++ if $vars->{show_place}->{$_};
    }
    my $words_clicks_col = xl(3 + $words_col);

    # clicks
    $list->write_formula($row, $col++, "=$WORDS!$words_clicks_col$words_result_row", $formats->{data});

    # price
    $list->write_formula($row, $col++, "=IF($PLAN!$res_col2$row_num <> 0, $res_col$row_num / $res_col2$row_num / $rate, \"-\")", $formats->{data_price});
    # pos
    $list->write($row, $col++, $POSITIONS_TEXTS{$params{place}}->{name}, $formats->{data});

    # sums
    my $coll_for_sum = xl(3 + scalar(keys %{$vars->{show_place}}) * 4 + $words_col);
    $list->write_formula($row, $col++, "=$WORDS!$coll_for_sum$words_result_row", $formats->{data_sum});
    
}

=head2 get_positions_in_order

    Возвращает список позиций, который может показываться в XLS.

=cut
sub get_positions_in_order {
    # TODO DIRECT-67003 подумать нужно ли прокидывать тип группы == mcbanner, для фильтрации позиций показа
    return ['select', @{PlacePrice::get_usedto_show_places()}];
}

=head2 forecast_draw_plan_new

    Вывод страницы с планом

=cut 
sub forecast_draw_plan_new {
    my ($work, $list, $vars, $login_rights) = @_;
    my $is_no_mediaplan = !(!defined $vars->{isforecast} || !$vars->{isforecast});

    # для Оценки бюджета удобно передавать id валюты, а при выгрузке медиаплана в xls сразу же есть полный хеш $vars->{pseudo_currency}
    my $pseudo_currency = (defined($vars->{pseudo_currency})) ? { %{$vars->{pseudo_currency}} } : get_pseudo_currency( id => $vars->{pseudo_currency_id} );
    $pseudo_currency->{name} = iget($pseudo_currency->{name});
    my $use_nds = ($vars->{currency} eq "YND_FIXED") ? iget("с учетом НДС") : iget("без НДС");
    my $currency_name = Currencies::get_currency_constant($vars->{currency}, 'name');

    my $targeting = iget("Регионы показа: ") . join("; ", uniq(map { get_geo_names($_->{geo}, ', ') } ($vars->{arr} ? @{$vars->{arr}} : ($vars)) )); 
    my $cn = $vars->{campaign_name} ? "\"$vars->{campaign_name}\"" : '';

    # Ширина колонки "Показы"
    my $shows_width = 1;
    $shows_width++ if $vars->{show_period} && $vars->{show_period} =~ /day/;
    $shows_width++ if $vars->{show_period} && $vars->{show_period} =~ /week/;
    
    # Ширина таблицы
    my $tbl_width = 8 + $shows_width;
    my $ctr_width = $vars->{show_plan} && $vars->{show_plan} =~ /ctr/ ? 1 : 0;
    $tbl_width += $ctr_width;

    # Создаём форматы
    my $f_header = $work->addformat( size=>8, bold => 1, valign => 'vcenter', align => 'center', border => 1, fg_color => 22, text_wrap => 1, pattern => 1 );
    my $f_header_right = new_format( $work, $f_header, right => 2 );
    my $fbold = $work->addformat( size=>8, bold => 1 );
    my $fcnt = $work->addformat( size=>8, align => 'center', valign => 'vcenter' );
    $list->hide_gridlines( 2 );
    $list->set_landscape();

    if ( $login_rights->{agency_control} ) {
        $list->set_row(0, 0); # логотип
    } else {
        $list->insert_image('A1', "$Settings::ROOT/data/t/logo-big-txt.bmp");
        $list->set_row(0, 65); # логотип
    }

    # Закрепляем заголовок, устанавливаем ширину столбцов
    $list->set_column(0,0,12);
    $list->set_column(1,1,9);
    $list->set_column(2,3,13);

    $list->set_column(4,5, int(32/($tbl_width-9+1)) ); # Показы и ctr

    $list->set_column( $tbl_width-4, $tbl_width-4, 10 ); # переходы
    $list->set_column( $tbl_width-3, $tbl_width-3, 7 ); # цена за клик
    $list->set_column( $tbl_width-2, $tbl_width-2, 10 ); # цена за клик
    $list->set_column( $tbl_width-1, $tbl_width-1, 12 ); # стоимость

    $list->set_row(1, 20); # название кампании, ...
    $list->set_row(2, 20); # название кампании, ...
    $list->set_row(3, 20); # название кампании, ...
    $list->set_row(4, 43); # первая строка заголовка
    $list->set_row(5, 25); # вторая строка заголовка

    my $itogo_row = 6 + scalar(keys %{$vars->{show_place}});
    for my $r ( 6 .. $itogo_row-1 ) {
        $list->set_row($r, 33); # данные
    }

    $list->set_row($itogo_row, 33); # Итого

    # Рисуем рамку
    for my $col ( 0..$tbl_width-1 ) {
        $list->write_blank(1, $col, $work->addformat( top => 2  ));
    }
    $list->write_blank(1, $tbl_width-1, $work->addformat(top=>2, right=>2));
    $list->write_blank(2, $tbl_width-1, $work->addformat(right=>2));
    $list->write_blank(3, $tbl_width-1, $work->addformat(right=>2));
    $list->write(1, 0, iget('План рекламной кампании ') . ($cn ? $cn : iget("на Яндекс.Директе")), $work->addformat(size=>12, top=>2, bold=>1, left => 2));
    $list->merge_cells(1, 0, 1, $tbl_width-2);
    $list->write(2, 0, iget("срок кампании: 1 месяц"), $work->addformat(size=>8, bold=>1, left => 2));
    $list->merge_cells(2, 0, 2, $tbl_width-2);
    $list->write(3, 0, $targeting, $work->addformat(size => 8, bold => 1, left => 2, text_wrap => 1, valign => 'vtop'));
    $list->merge_cells(3, 0, 3, $tbl_width - 1);
    $list->set_row(3, 12 + length($targeting) / 15);

    # Рисуем заголовок
    for my $row (4 .. 5) {
        for my $col (0 .. $tbl_width - 1) {
            $list->write_blank($row, $col, $f_header);
        }
    }
    $list->write_blank(5, $tbl_width-1, $f_header_right);
    $list->write(4, 0, iget("Места размещения рекламных материалов**"), new_format($work, $f_header, left=>2));
    $list->write(5, 0, "", new_format($work, $f_header, left=>2));
    $list->merge_cells(4, 0, 5, 1);
    $list->write(4, 2, iget("Рекламный носитель"), $f_header);
    $list->merge_cells(4, 2, 5, 2);
    $list->write(4, 3, iget("Время размещения"), $f_header);
    $list->merge_cells(4, 3,  5, 3);

    $list->write(4, 4, iget("Показы рекламных материалов (прогноз)"), $f_header );
    $list->merge_cells(4, 4, 4, 4+$shows_width-1);
    my $sh_p = 0;
    if ( $vars->{show_period} && $vars->{show_period} =~ /day/ ) {
        $list->write(5, 4+$sh_p, iget("в день"), $f_header );
        $sh_p++;
    }
    if ( $vars->{show_period} && $vars->{show_period} =~ /week/ ) {
        $list->write(5, 4+$sh_p, iget("в неделю"), $f_header );
        $sh_p++;
    }
    $list->write(5, 4+$sh_p, iget("всего"), $f_header );

    if ( $vars->{show_plan} && $vars->{show_plan} =~ /ctr/ ) {
        $list->write(4, 4+$shows_width, iget("Средний CTR,%"), $f_header );
        $list->merge_cells(4, 4+$shows_width,  5, 4+$shows_width);
    }

    my $clicks_start = 4+$shows_width+$ctr_width;
    $list->write(4, $clicks_start, iget("Переходы"), $f_header);
    $list->merge_cells(4, $clicks_start, 5, $clicks_start);

    my $avg_click_price_header = sprintf("%s, %s (%s)", iget("Средняя цена за клик"), $currency_name, $use_nds);
    $list->write(4, $clicks_start + 1, ($vars->{currency} eq "YND_FIXED")
                                        ? sprintf("%s\n%s", $avg_click_price_header,
                                                         Currency::Format::conv_unit_explanation($vars->{currency}, $pseudo_currency->{id}))
                                        : $avg_click_price_header
                 , $f_header);
    $list->merge_cells(4, $clicks_start+1, 5, $clicks_start + 2);

    $list->write(4, $clicks_start + 3, sprintf("%s***, %s (%s)", iget("Стоимость за кампанию"), 
                                                 ($vars->{currency} eq "YND_FIXED") ? $pseudo_currency->{name} : $currency_name, 
                                                 $use_nds)
                 , $f_header_right);
    $list->merge_cells(4, $clicks_start+3, 5, $clicks_start + 3);

    # Строка с Итого на странице Слова ...............................................................
    my $formats = {
        "data" => $work->addformat(size=>8, valign => 'vcenter', align => 'center', text_wrap => 1, border => 1, color => ($vars->{show_select} ? 'silver' : 'black'))
    };
    $formats->{data_price} = new_format($work, $formats->{data}, num_format => '##0.00');
    $formats->{data_left} = new_format($work, $formats->{data}, align => 'left', bottom => 1);
    $formats->{data_left_bold} = new_format($work, $formats->{data}, align => 'left', left => 2, bold => 1);
    $formats->{data_sum} = new_format($work, $formats->{data}, num_format => 1);

    my ($row, $col) = (5, 0);
    my $words_result_row = 0;
    
    if($vars->{arr}){
        $words_result_row = 5 + 1 + ($vars->{geo} ? 1 : 0);    
        $words_result_row += 2 + $#{$_->{Phrases}} + 1 for @{$vars->{arr}};
    } else {
        $words_result_row = 7 + ($vars->{geo} ? 1 : 0) + $#{$vars->{Phrases}};
    }     
    
    foreach my $place (@{get_positions_in_order()}) {
        next unless $vars->{show_place}->{$place};
        $row++; 
        show_plan_place_row(place   => $place,
                            row     => $row,
                            list    => $list,
                            vars    => $vars,
                            formats => $formats,
                            words_result_row => $words_result_row);
    }

    # text messages
    my $f_notes = $work->addformat(size=>8, text_wrap => 1, align => 'top');
    my $f_notes_bold = new_format($work, $f_notes, bold => 1);
    my @text = (
        iget("Средние цены за клик указаны по состоянию на %s и могут быть изменены без предварительного уведомления.", strftime("%d.%m.%Y", localtime)),
        iget("*    Реклама будет показываться только тогда, когда в запросе пользователя будут присутствовать указанные рекламодателем слова или словосочетания.\n").
        iget("Предварительный список рекомендуемых ключевых фраз для показа рекламы можно посмотреть на странице \"Слова\" данного плана.\n"),
        iget("**   Объявления, размещенные через Яндекс.Директ, могут быть показаны на всех страницах выдачи результатов поиска Яндекса по выбранному вами запросу, а также на страницах в сети (Рекламная сеть Яндекса и внешние сети), на страницах результатов поиска по Каталогу Яндекса, на страницах результатов поиска по Яндекс.Адресам и по Блогам, на страницах просмотра всех объявлений Яндекс.Директа по ключевым словам."),
        iget("***  Имейте в виду, что реальный бюджет может отличаться от прогнозируемого, т.к. он подсчитан на основе анализа ставок конкурентов и CTR их кампаний, а эти параметры могут изменяться в процессе работы вашей рекламной кампании. Кроме этого, в прогнозе бюджета не учитываются показы объявлений в сети (Рекламная сеть Яндекса и внешние сети) и настройки временного таргетинга.")
    );

    for (my $j = 0; $j < @text; $j++) {
        $list->set_row( $itogo_row+1+$j, length( $text[$j] )>170 ? 8+15*int(length($text[$j])/170) : 17 );
        $list->write( $itogo_row+1+$j, 0, $text[$j], ($text[$j] =~ m/\d\d\.\d\d\.\d\d\d\d/ ? $f_notes_bold : $f_notes));
        $list->merge_cells( $itogo_row+1+$j, 0, $itogo_row+1+$j, $tbl_width-1 );
    }
}

=head2 forecast_draw_words_new
  
    Вывод страницы со словами (new)

=cut
sub forecast_draw_words_new
{
    my ($work, $list, $vars, $login_rights) = @_;

    my $row_num = 0;
    my $col;
    $list->set_landscape();
    $list->hide_gridlines(2);

    # для Оценки бюджета удобно передавать id валюты, а при выгрузке медиаплана в xls сразу же есть полный хеш $vars->{pseudo_currency}
    my $pseudo_currency = (defined($vars->{pseudo_currency})) ? { %{$vars->{pseudo_currency}} } : get_pseudo_currency( id => $vars->{pseudo_currency_id} );
    $pseudo_currency->{name} = iget($pseudo_currency->{name});
    my $rate = ($vars->{currency} eq 'YND_FIXED') ? $pseudo_currency->{rate} : 1; 
    my $currency_name = Currencies::get_currency_constant($vars->{currency}, 'name');

    # Рисуем заголовок
    $list->insert_image($row_num, 0, "$Settings::ROOT/data/t/logo-big-txt.bmp");
    $list->set_row($row_num++, 65); # логотип

    if ($login_rights->{role} =~ /client/) {
        $list->write($row_num, 0, iget("Оценка бюджета рекламной кампании на Яндекс.Директе"), $work->addformat(size=>12, top=>2, bold=>1, left => 2, right => 2));
    } else {
        my $campaign_name = $vars->{campaign_name} ? '"'.$vars->{campaign_name}.'"' : '';
        $list->write($row_num, 0, iget('План рекламной кампании ') . ($campaign_name ? $campaign_name : iget("на Яндекс.Директе")), $work->addformat(size=>12, top=>2, bold=>1, left => 2, right => 2));
    }
    my $table_width_cells = 2 + (scalar(keys %{$vars->{show_place}})) * 5;
    $list->merge_cells($row_num, 0, $row_num, $table_width_cells);

    # Рисуем рамку
    for my $col (1 .. $table_width_cells - 1) {
        $list->write_blank($row_num, $col, $work->addformat(top => 2));
    }
    $list->write_blank($row_num, $table_width_cells, $work->addformat(top=>2, right=>2));
    $list->set_row($row_num++, 20);

    if ($login_rights->{role} =~ /client/) {
        $list->write($row_num, 0, '', $work->addformat(size=>12, top=>0, bold=>1, left => 2, right => 2));
    } else {
        $list->write($row_num, 0, iget('(Предложение слов)'), $work->addformat(size=>12, top=>0, bold=>1, left => 2, right => 2));
    }
    $list->write_blank($row_num, $table_width_cells, $work->addformat(right => 2));
    $list->merge_cells($row_num, 0, $row_num, $table_width_cells);
    $list->set_row($row_num++, ($login_rights->{role} =~ /client/ ? 1 : 18));

    $list->write_blank($row_num, $table_width_cells - 1, $work->addformat(right => 2));
    $list->write($row_num, 0, iget('Срок кампании 1 месяц'), $work->addformat(size=>8, bold=>1, left => 2, right => 2));
    $list->write_blank($row_num, $table_width_cells, $work->addformat(right => 2));
    $list->merge_cells($row_num, 0, $row_num, $table_width_cells);
    $list->set_row($row_num, 18);
    $row_num++;

    if ($vars->{geo}) {
        my $geo_names = get_geo_names($vars->{geo}, ', ');  
        my $targeting = iget("Регионы показа: %s", $geo_names);
        $list->write($row_num, 0, $targeting, $work->addformat(size => 8, bold => 1, left => 2, text_wrap => 1, right => 2, valign => 'vtop'));
        $list->merge_cells($row_num, 0, $row_num, $table_width_cells);
        $list->set_row($row_num, 12 + int(length($targeting) / $table_width_cells / 2));
        $list->write_blank($row_num, $table_width_cells, $work->addformat(right => 2));
        $row_num++
    }

    # Заголовок
    my $f_header = $work->addformat(size=>8, bold => 1, valign => 'vcenter', align => 'center', border => 1, fg_color => 22, text_wrap => 1, pattern => 1);
    $list->set_row($row_num, 77);

    $col = 0;
    $list->set_column($col, $col, 38);
    $list->write($row_num, $col++, iget("Предложенные фразы"), $f_header);

    $list->set_column($col, $col, ($vars->{show_place}->{select} ? 8 : 0.1)); # show pos column if show_select only
    $list->write($row_num, $col++, iget("Объём трафика"), $f_header);
    $list->write($row_num, $col++, iget("Количество показов в месяц* (прогноз)"), $f_header);

    my @get_positions_in_order = @{get_positions_in_order()};
    my %col_nums;
    foreach (@get_positions_in_order) {
        if ($vars->{show_place}->{$_}) {
            $col_nums{"clicks_".$_} = xl($col);
            $list->write($row_num, $col++, $POSITIONS_TEXTS{$_}->{clicks_per_month}, $f_header);
        }
    }
    foreach (@get_positions_in_order) {
        if ($vars->{show_place}->{$_}) {
            $list->write($row_num, $col++, $POSITIONS_TEXTS{$_}->{ctr_forecast}, $f_header);
        }
    }

    my $currency_name_stars = $currency_name.(($vars->{currency} eq "YND_FIXED")?" ***":"");
    foreach (@get_positions_in_order) {
        if ($vars->{show_place}->{$_}) {
            $col_nums{"bid_price_".$_} = xl($col);
            $list->write($row_num, $col++, sprintf ($POSITIONS_TEXTS{$_}->{avg_bid_price}, $currency_name_stars), $f_header);
            $col_nums{"amnesty_price_".$_} = xl($col);
            $list->write($row_num, $col++, sprintf ($POSITIONS_TEXTS{$_}->{avg_amnesty_price}, $currency_name_stars), $f_header);
        }
    }

    my $pseudo_currency_name = ($vars->{currency} eq 'YND_FIXED') ? $pseudo_currency->{name} : $currency_name;
    foreach (@get_positions_in_order) {
        if ($vars->{show_place}->{$_}) {
            $list->write($row_num, $col++, sprintf ($POSITIONS_TEXTS{$_}->{avg_budget}, $pseudo_currency_name), $f_header);
        }
    }

    $list->set_column(2, 16, 12);
    $list->set_column(2, 17, 12);
    $list->set_column(2, 18, 12);
    $list->set_column(2, 19, 12);
    $row_num++;

    # main data
    my $my_grey = $work->set_custom_color(40, 225, 225, 225);
    my $f_data = $work->addformat(size=>8, align=>'center', border => 1, num_format => '0');
    my $f_data_sel = new_format($work, $f_data, bg_color => $my_grey);
    my $f_data_left = $work->addformat(size=>8, align=>'left', border => 1, num_format => '0');
    my $f_data_price = new_format($work, $f_data, num_format => '0.00');
    my $f_data_price_sel = new_format($work, $f_data_price, bg_color => $my_grey);
    my $f_data_round = new_format($work, $f_data, num_format => '0.00');
    my $f_data_round_sel = new_format($work, $f_data_round, bg_color => $my_grey);
    my $begin_row = $row_num + 1;

    my @media = $vars->{arr} ? @{$vars->{arr}} : ($vars);
    
    for my $phrase (@media){       
        if($vars->{arr}){
            $list->write( $row_num, 0, $phrase->{title} , new_format( $work, $f_data_left, bold => 1, size => 10, text_wrap => 1, border => 0) );
            $list->merge_cells( $row_num, 0, $row_num, 8 );                        
            $row_num++;            
            $list->write( $row_num, 0, $phrase->{body} , new_format( $work, $f_data_left, bold => 1, size => 10, text_wrap => 1, border => 0) );
            $list->merge_cells( $row_num, 0, $row_num, 8 );                        
            $row_num++;            
        }    

        # пересортировываем фразы по алфавиту
        @{ $phrase->{Phrases} } = sort {
            # рубрики всегда ниже
            return
                ($a->{category_name} ? 1 : 0) <=> ($b->{category_name} ? 1 : 0)
                || lc($a->{phrase}||$a->{category_name}||'') cmp lc($b->{phrase}||$b->{category_name}||'');
        } @{ $phrase->{Phrases} };

        for my $row (@{$phrase->{Phrases}}) {
            $row->{place} = PlacePrice::set_new_place_style($row->{place}) || PlacePrice::get_guarantee_entry_place();
            $col = 0;
            if ($row->{category_url}) {
                $list->write($row_num, $col++, iget("Рубрика: ") . $row->{category_name}, $f_data_left);
            } else {
                $list->write($row_num, $col++, $row->{phrase}, $f_data_left);
            }

            my ($sel_ctr, $bid_price, $amnesty_price, $pos_caption, $sel_clicks);
            my $place_data = PlacePrice::get_data_by_place($row, $row->{place});
            if ($place_data) {
                $sel_ctr = $place_data->{ctr};
                $sel_clicks = $place_data->{clicks};
                $bid_price = $place_data->{bid_price};
                $amnesty_price = $place_data->{amnesty_price};
            }
            $pos_caption = $POSITIONS_CAPTIONS{$row->{place}};

            $list->write($row_num, $col++, $pos_caption, $f_data_left);
            $list->write($row_num, $col++, $row->{shows}, $f_data);

             # clicks .................

             foreach my $place (@get_positions_in_order) {
                if ($place eq 'select' && $vars->{show_place}->{$place}) {
                    $list->write($row_num, $col++, $sel_clicks, $f_data);
                } elsif ($vars->{show_place}->{$place}) {
                    $list->write($row_num, $col++, get_data_by_place($row, $place)->{clicks}, ($row->{place} == $place && $vars->{show_place}->{select} ? $f_data_sel : $f_data));
                }
             }

             # ctr .................
             foreach my $place (@get_positions_in_order) {
                if ($place eq 'select' && $vars->{show_place}->{$place}) {
                    $list->write($row_num, $col++, $sel_ctr, $f_data_price);
                } elsif ($vars->{show_place}->{$place}) {
                    $list->write($row_num, $col++, get_data_by_place($row, $place)->{ctr}, ($row->{place} == $place && $vars->{show_place}->{select} ? $f_data_price_sel : $f_data_price));

                }
             }

             # bid/amnesty prices ..............
             foreach my $place (@get_positions_in_order) {
                if ($place eq 'select' && $vars->{show_place}->{$place}) {
                    $list->write($row_num, $col++, $bid_price / 1e6, $f_data_price);
                    $list->write($row_num, $col++, $amnesty_price / 1e6, $f_data_price);
                } elsif ($vars->{show_place}->{$place}) {
                    $list->write($row_num, $col++, get_data_by_place($row, $place)->{bid_price} / 1e6, ($row->{place} == $place && $vars->{show_place}->{select} ? $f_data_price_sel : $f_data_price));
                    $list->write($row_num, $col++, get_data_by_place($row, $place)->{amnesty_price} / 1e6, ($row->{place} == $place && $vars->{show_place}->{select} ? $f_data_price_sel : $f_data_price));
                }
             }

             my $row_num_tmp = $row_num + 1;
             # sum .................
             foreach my $place (@get_positions_in_order) {
                my $price_column_name = "amnesty_price_$place";
                if ($place eq 'select' && $vars->{show_place}->{$place}) {
                    $list->write_formula($row_num, $col++, "=".$col_nums{"clicks_$place"}."$row_num_tmp * ".$col_nums{$price_column_name}."$row_num_tmp * $rate", $f_data_round);
                } elsif ($vars->{show_place}->{$place}) {
                    $list->write_formula($row_num, $col++, "=".$col_nums{"clicks_$place"}."$row_num_tmp * ".$col_nums{$price_column_name}."$row_num_tmp * $rate", ($row->{place} == $place && $vars->{show_place}->{select} ? $f_data_round_sel : $f_data_round));

                }
             }

             $row_num++;
         }
    }

    # итого
    my $f_itogo = $work->addformat(right => 1, size=>10, bottom => 2, bold => 1, italic=>1, valign=>'vcenter', align=>'center');
    my $f_itogo_left = new_format($work, $f_itogo, align=>'left', left=>2);
    my $f_itogo_num = new_format($work, $f_itogo, num_format => '0');
    my $f_itogo_num_round = new_format($work, $f_itogo, num_format => '0.00');
    my $f_itogo_num_round_simple = new_format($work, $f_itogo, num_format => '0.00', bold => 0, italic=> 0);    
    my $f_itogo_num_silver = new_format($work, $f_itogo_num, color => 'silver', bold => 0);
    my $f_itogo_num_silver_round = new_format($work, $f_itogo_num, color => 'silver', bold => 0, num_format => '0.00');
    my $f_itogo_price = new_format($work, $f_itogo, num_format => '0.00');
    my $f_itogo_right = new_format($work, $f_itogo_price, right => 2);

    my $end_row = $row_num;
    $list->set_row($row_num, 20);
    $col = 0;
    $list->write($row_num, $col++, iget('Итого с учетом выбранного объёма трафика**'), $f_itogo_left);
    $list->write_blank($row_num, $col++, $f_itogo_price);

    # shows
    my $coll = xl($col);
    $list->write_formula($row_num, $col, "=SUM($coll$begin_row:$coll$end_row)", $f_itogo_num);
    $col++;

    # clicks
    foreach my $place (@get_positions_in_order) {
        $coll = xl($col);
        if ($place eq 'select' && $vars->{show_place}->{$place}) {
            $list->write_formula($row_num, $col++, "=SUM($coll$begin_row:$coll$end_row)", $f_itogo_num);
        } elsif ($vars->{show_place}->{$place}) {
            $list->write_formula($row_num, $col++, "=SUM($coll$begin_row:$coll$end_row)", $f_itogo_num_round_simple);
        }
    }


    # ctr
    foreach my $place (@get_positions_in_order) {
        $list->write_blank($row_num, $col++, $f_itogo_price) if ($vars->{show_place}->{$place});
    }

    # bid/amnesty price
    foreach my $place (@get_positions_in_order) {
        if ($vars->{show_place}->{$place}) {
            $list->write_blank($row_num, $col++, $f_itogo_price);
            $list->write_blank($row_num, $col++, $f_itogo_price);
        }
    }

    # sum
    foreach my $place (@get_positions_in_order) {
        $coll = xl($col);
        if ($place eq 'select' && $vars->{show_place}->{$place}) {
            $list->write_formula($row_num, $col++, "=SUM($coll$begin_row:$coll$end_row)", $f_itogo_num_round);
        } elsif ($vars->{show_place}->{$place}) {
            $list->write_formula($row_num, $col++, "=SUM($coll$begin_row:$coll$end_row)", ($vars->{show_place}->{select} ? $f_itogo_num_silver_round : $f_itogo_num_round));
        }
    }

    $row_num += 2;

    # примечания
    my $f_notes = $work->addformat(size=>8, text_wrap => 1);
    my $f_notes_bold = new_format($work, $f_notes, bold => 1);
    my @text = (
        iget("       Средние цены за клик указаны по состоянию на %s и могут быть изменены без предварительного уведомления.", strftime("%d.%m.%Y", localtime)),
        iget("*      Число показов и переходов указано по данным за предыдущий месяц."),
        iget("**     Имейте в виду, что реальный бюджет может отличаться от прогнозируемого, т.к. он подсчитан на основе анализа ставок конкурентов и CTR их кампаний, а эти параметры могут изменяться в процессе работы вашей рекламной кампании. Кроме этого, в прогнозе бюджета не учитываются показы объявлений в сети (Рекламная сеть Яндекса и внешние сети) и настройки временного таргетинга."),
        ($vars->{currency} eq "YND_FIXED") ? sprintf("***    %s", Currency::Format::conv_unit_explanation($vars->{currency}, $pseudo_currency->{id})) : (),      
        iget("       Реклама по слову (или словосочетанию) будет показываться во всех и только в тех случаях, когда в запросе присутствует это слово (или словосочетание). Так, реклама по слову \"мебель\" будет показана и тому, кто спросил \"мебель\", и тому, кто спросил \"каталог мебели\"."),
        iget("       \"Минус-фразы\" и \"минус-слова\" используются для дополнительного уточнения слов (словосочетаний). Так, реклама по условию \"шкаф -жарочный\" будет показана по запросам \"продажа шкафов\", \"шкаф-купе\", но не будет показана по запросу \"жарочный шкаф\"."),
        iget("       В случае, если ставка окажется недостаточной для показа на поиске, реальное количество показов и бюджет по соответствующим словам (словосочетаниям) может оказаться существенно меньше прогнозируемого.")
    );

    for (my $j = 0; $j < @text; $j++) {
        my $hh = 19;
        if (length($text[$j]) > 170 && length($text[$j]) <= 300) {
            $hh = 28;
        }
        if (length($text[$j]) > 300) {
            $hh = 45;
        }
        $list->set_row($row_num, $hh);
        $list->write($row_num, 0, $text[$j], ($text[$j] =~ m/\d\d\.\d\d\.\d\d\d\d/ ? $f_notes_bold : $f_notes));
        $list->merge_cells($row_num, 0, $row_num, 8);
        $row_num++;
    }
}

sub new_format {
    my ( $work, $fmt, %props ) = @_;
    my $f = $work->addformat();
    $f->copy( $fmt );
    $f->set_properties( %props );
    return $f;
}

=head2 xl

  по номеру получить букву

=cut
sub xl {
    my $num = shift;
    my $res = '';
    my $letterscount=26;
    my $letterstart = ord('A');
    while (int($num/$letterscount) > 0) {
        $res .= chr( $letterstart + int($num/$letterscount) -1 );
        $num = $num%$letterscount;
    }
    $res .= chr( $letterstart + $num );    
    return $res;
}


1;
