package Geo;

use Direct::Modern;

use Carp;
use List::MoreUtils qw/uniq none part any/;
use JSON qw/to_json/;

use Settings;
use Yandex::I18n;
use Yandex::DBTools;

use Models::AdGroup qw/get_groups/;

use Direct::Validation::BannersPerformance;

use Campaign::Types qw/camp_kind_in get_camp_supported_adgroup_types get_camp_supported_adgroup_types_with_geo/;
use Phrase qw/update_phrases/;
use ModerateChecks qw/check_moderate_region check_moderate_banner_for_regions/;
use MailNotification qw/mail_notification/;
use Tools qw/log_cmd/;
use Primitives qw/schedule_forecast content_lang_to_save/;
use GeoTools qw/is_targeting_in_region get_geo_names/;

use utf8;

use base qw/Exporter/;
our @EXPORT = qw/
    explain_common_geo
    set_common_geo
    geo_changes_to_string
    get_common_geo_for_camp
    get_union_geo_for_camp
    set_extended_geo_to_camp
    get_merged_geo_str
/;

our @EXPORT_OK = qw/

/;


=head2 explain_common_geo($cid, $geo, %options)

    Мержит указанное гео с гео групп кампании
    Возвращает ссылку на хеш со следующими полями
    {
        camp_type => тип кампании,
        geo_str => сформированная строка общего гео,
        changed_groups => ссылка на массив групп, для которых изменится гео, в поля групп добавлены $new_geo_str и $geoflag
        errors => ссылка на хеш с ошибками {pid1 => [ошибка1, ошибка2, ...], pid2 =>..., general => [ошибкаX, ...]}
    }
   Входные параметры: 
    $cid - номер кампании
    $geo - геотаргетинг
    %options
        client_chief_uid - uid основного представителя
        ClientID - пользователь
        merge_geo(merge|override) - режим применения общего гео

=cut

sub explain_common_geo {
    my ($cid, $geo_or_geo_changes, %options) = @_;
    
    my (@changed_groups, %errors);

    my $mode = $options{merge_geo} ? 'merge' : 'override';
    my $client_id = $options{ClientID};
    my $common_geoflag;
    my ($geo_str, $geo_changes);
    my $cache = {};
    if ($mode eq 'merge') {
        $geo_changes = $geo_or_geo_changes;
        $geo_str = _merge_geo(get_common_geo_for_camp($cid) // '', $geo_changes, $client_id, \$common_geoflag, $cache);
    }
    else {
        my $regions = ref $geo_or_geo_changes ?
             join( ', ', map { $geo_or_geo_changes->{$_}->{is_negative} ? -$_ : $_} sort {$a <=> $b} keys %$geo_or_geo_changes)
            :$geo_or_geo_changes;
        $geo_str = _merge_geo($regions, {}, $client_id, \$common_geoflag, $cache);
    }

    # geo должен быть либо null, либо comma-sep строка с цифрами (DIRECT-90253)
    if (defined $geo_str && $geo_str eq '') {
        $geo_str = undef;
    }

    my $camp_type = Campaign::Types::get_camp_type(cid => $cid);
    
    if ($camp_type eq 'mcb'){
        push @{$errors{general}}, iget('Не заданы регионы показа.') unless $geo_str;
    }
    else {
        # тип кампании получен раньше при вызове get_camp_type, поэтому в get_camp_supported_adgroup_types_with_geo не будет второго запроса в базу, а значение типа будет получено из кэша
        my $groups = get_groups({cid => $cid, adgroup_types=>get_camp_supported_adgroup_types_with_geo(cid => $cid)}, {only_creatives => 1});

        my @errors;
        foreach my $g (@$groups) {
            my $geoflag = $common_geoflag;
            my $new_geo_str = $mode ne 'merge' ? $geo_str : _merge_geo(
                                                                       GeoTools::modify_translocal_region_before_show($g->{geo} || '0', {ClientID=>$client_id}),
                                                                       $geo_changes, $client_id, \$geoflag, $cache
                                                                    );
            my @errors;
            if ($new_geo_str){
                next if ($g->{geo} // '' ) eq $new_geo_str;
                @errors = validate_adgroup_geo($g->{pid}, $new_geo_str, $camp_type, {ClientID => $client_id});
            }
            else{
                push @errors, '#empty_geo#';
            }

            if (@errors) {
                push @{$errors{$g->{pid}}}, @errors;
                next;
            }
            
            $g->{geoflag} = $geoflag;
            $g->{new_geo} = $new_geo_str;
            push @changed_groups, $g;
        }
    }

    return {
        camp_type => $camp_type,
        geo_str => $geo_str,
        @changed_groups ? (changed_groups => \@changed_groups) : (),
        keys %errors    ? (errors => \%errors) : (),
    }
}

=head2 set_common_geo($cid, $geo, %options)

    Установка общего региона на кампанию
    
    $cid - номер кампании
    $geo - геотаргетинг
    %options
        uid - uid пользователя
        ClientID - ClientID пользователя
        merge_geo(merge|override) - режим применения общего гео
        statusEmpty - statusEmpty кампании

=cut 

sub set_common_geo {
    my ($cid, $geo_or_geo_changes, %options) = @_;

    my $prepared = explain_common_geo($cid, $geo_or_geo_changes, %options);
    if ($prepared->{errors}){
        my $errors_by_pid = $prepared->{errors};
        my @errors;
        foreach my $pid (keys %$errors_by_pid){
            push @errors, map {
                    TextTools::process_text_template($_, empty_geo => iget("Применение изменений удаляет все регионы показа группы №%s.", $pid))
                } @{$errors_by_pid->{$pid}};
        }
        push @errors, iget("Сохранение невозможно.");
        return @errors;
    }

    my ($camp_type, $geo_str, $groups) = @$prepared{qw/camp_type geo_str changed_groups/};

    if ($camp_type eq 'mcb'){
        # for media campaigns
        do_update_table(PPC(cid => $cid), 'media_groups g LEFT JOIN media_banners b ON (b.mgid = g.mgid)',
                        {
                            'g.geo' => $geo_str,
                            'g.statusBsSynced' => 'No',
                            'b.statusBsSynced' => 'No',
                        },
                       where => {
                            _OR => {
                                'g.geo__ne' => $geo_str,
                                'g.geo__is_null' => 1,
                            },
                            'g.cid' => $cid,
                       }
        );
    }
    else {
        my %banners;

        my %update_by_pid;
        my $data_for_log_cmd = {};

        foreach my $g (@$groups) {
            my $geoflag = delete $g->{geoflag};
            my $new_geo_str = delete $g->{new_geo};
            
            push @{ $data_for_log_cmd->{$new_geo_str} }, $g->{pid};
            
            my $need_moderate = check_moderate_region($new_geo_str, $g->{geo});
            if ($g->{adgroup_type} ne 'performance') { # DIRECT-100514
                ## no critic (Freenode::DollarAB)
                foreach my $b (@{$g->{banners}}) {
                    # Проверяем надо ли отправлять баннеры в модерацию по признаку наличия флага forex
                    next if !$need_moderate && !check_moderate_banner_for_regions(BannerFlags::get_banner_flags_as_hash($b->{flags}), $new_geo_str, $g->{geo}, {ClientID => $options{ClientID}});

                    mail_notification('banner', 'b_geo', $b->{bid}, $g->{geo}, $new_geo_str, $options{client_chief_uid});
                    if ($need_moderate) {
                        # geo будет проставлен на группе ещё и отдельно, update_phrases хочет geo всё равно
                        my $status = update_phrases($g->{pid}, $cid, $g->{statusModerate},
                                                    $options{statusEmpty}, $options{uid}, $new_geo_str);
                        $banners{$b->{bid}} = $status if $status;
                    } else {
                        $banners{$b->{bid}} = 'Ready';
                    }
                }
            }
            $update_by_pid{$g->{pid}} = {
                'p.geo' => $new_geo_str,
                'b.geoflag' => $geoflag,
                'b.opts' => Yandex::DBTools::sql_set_mod('b.opts', {geoflag => $geoflag}),
            };
        }
        
        log_cmd({cmd => '_set_common_geo',
            uid => $options{uid}, cid => $cid,
            new_geo_data => to_json($data_for_log_cmd, utf8 => 1),
        });
        
        do_update_table(PPC(cid => $cid), 'banners', {
            statusModerate__dont_quote => sql_case(bid => \%banners, default__dont_quote => 'statusModerate'),
            statusPostModerate__dont_quote => "IF(statusPostModerate = 'Rejected', 'Rejected', 'No')"
        }, where => {bid => [keys %banners], statusModerate__ne => 'New'}) if keys %banners;
        Direct::Banners::delete_minus_geo(bid => [ keys %banners ], check_status => 1);
        
        # phrases.LastChange обновляется вместе с geo
        do_mass_update_sql(PPC(cid => $cid), "phrases p LEFT JOIN banners b USING (pid)",
            'p.pid' => \%update_by_pid,
            where => {'p.cid' => $cid},
            byfield_options => {
                'p.statusBsSynced' => {default => 'No'},
                'b.statusBsSynced' => {default => 'No'},
                'p.statusShowsForecast' => {default => 'New'},
                'b.LastChange' => {default__dont_quote => 'b.LastChange'},
                'b.opts' => {dont_quote_value => 1}
            }
        );
    }
    # for empty and mcb campaigns               
    do_update_table(PPC(cid => $cid), 'campaigns', {geo => $geo_str}, where => {cid => $cid}) if !($groups && @$groups);
    schedule_forecast($cid);
    
    return ();
}

sub _merge_geo {
    my ($geo_str, $changes, $client_id, $geoflag_ptr, $cache) = @_;

    #Если в кеше уже есть $geo_str - вернем результат для нее
    #Cчитаем, что при изменении $changes кеш сбросят снаружи
    if (exists $cache->{$geo_str}){
        if (defined $geoflag_ptr) {
            $$geoflag_ptr = $cache->{$geo_str}->[1];
        }
        return $cache->{$geo_str}->[0];
    }

    my @geo_ids = sort {$a <=> $b} uniq grep {/\S/} split( /\s*,\s*/, ($geo_str // '') );
    my $translocal_opts = {ClientID => $client_id};
    my $tree = GeoTools::get_translocal_georeg($translocal_opts);
    
    my $exclude = [];
    if ($changes && keys %$changes) {
        my %new_regions = %$changes;
        if (@geo_ids) {       
            #Уберем регионы, покрытые измененными регионами
            foreach my $region_id (keys %new_regions) {
                my $sign = $new_regions{$region_id}->{is_negative} ? -1 : 1;
                
                @geo_ids = grep { !(
                                    any {$region_id == $_} @{$tree->{abs($_)}->{parents}} 
                                )} @geo_ids;
            }
            #Оставим регионы, которые были в исходной строке и не были изменены, пары Регион -Регион уберем с обоих концов
            #минус-регионы в $changes представлены без минуса в id региона, но с {is_negative => 1} в значении
            # id_региона => {is_negative => 0} - плюс-регион,  id_региона => {is_negative => 1} - минус-регион
            my @_geo_ids;
            foreach (@geo_ids) {
                my $reg = abs($_);
                if (exists $new_regions{$reg}) {
                    #есть дубль или пара -+|+-
                    my $meta = $new_regions{$_};
                    #из новых удаляем в случаях ++, --, +-
                    delete $new_regions{$_} if ($_ >= 0 || $meta->{is_negative} );
                    #из существующих удаляем в случае +- (удаляются оба) и -+ (оставляем только новый)
                    next if ($meta->{is_negative} && $_ >= 0 || !$meta->{is_negative} && $_ < 0)
                }
                push @_geo_ids, $_;
            }
            @geo_ids = @_geo_ids;
        }
        #добавляем оставшиеся измененные плюс-регионы
        my ($include);
        ($exclude, $include) = part {$new_regions{$_}->{is_negative} == 1 ? 0 :1} sort {$a <=> $b} keys %new_regions;
        push @geo_ids, @{$include // []};
        #Если ничего не осталось - выдаем ошибку, внезапное разворачивание показов на "весь мир" клиенты не ожидают
        return unless (@geo_ids);
    }
   
    #Соберем в строку
    my %plus_regions  = map {$_ => $tree->{$_}} grep {$_ >= 0} @geo_ids;
    #Если в списке регионов у нас оказался 0 ("весь мир") - вернем ошибку 
    return if exists $plus_regions{0};
   
    my %minus_regions = map {$_ => $tree->{$_}} @{$exclude // []}, map {abs($_)} grep {$_ < 0} @geo_ids;
    
    #Оставляем только плюс-регионы, которые не покрыты более крупными плюс-регионами
    foreach my $reg (_sort_by_level( $tree, keys %plus_regions)) {
        my @super_regions =
                _sort_by_level( $tree,
                grep {exists $plus_regions{$_}} @{$plus_regions{$reg}->{parents}});
        next unless @super_regions;
        #Если покрывающие регионы есть - смотрим, нет ли минус-региона между регионом и самым мелким его родителем
        my $parent = $super_regions[0];
        my @minus = grep {
                 (any {$parent == $_} @{$minus_regions{$_}->{parents}}) && #минус-регион должен быти ниже родителя
                 !(any {$reg == $_} @{$minus_regions{$_}->{parents}}) #и выше региона
            } keys %minus_regions;
        delete $plus_regions{$reg} unless @minus; #если таких минус регионов нет - удаляем плюс-регион как поглощенный родителем
        
    }
    
    #Аналогично обрабатываем минус-регионы
    foreach my $reg (_sort_by_level( $tree, keys %minus_regions)) {
        unless ( any {exists $plus_regions{$_}} @{$minus_regions{$reg}->{parents}}){
            #убираяем "повисшие", т.е. не покрытые плюс-регионами минус-регионы
            delete $minus_regions{$reg};
            next;
        }
        
        my @super_regions =
                _sort_by_level( $tree,
                grep {exists $minus_regions{$_}} @{$minus_regions{$reg}->{parents}});
        next unless @super_regions;
        #Если покрывающие регионы есть - смотрим, нет ли плюс-региона между регионом и самым мелким его родителем
        my $parent = $super_regions[0];
        my @plus = grep {
                (any {$parent == $_} @{$plus_regions{$_}->{parents}}) && #плюс-регион должен быти ниже родителя
                !(any {$reg == $_} @{$plus_regions{$_}->{parents} }) #и выше региона
            } keys %plus_regions;
        delete $minus_regions{$reg} unless @plus; #если таких плюс-регионов нет - удаляем минус-регион как поглощенный родителем
        
    }
   
    my $merged_geo_str = join ',', sort {$a <=> $b} keys %plus_regions, map  {-$_} sort {$a <=> $b} keys %minus_regions;
    #Причешем результат
    if ($merged_geo_str gt '') {
        $merged_geo_str = GeoTools::modify_translocal_region_before_save($merged_geo_str, $translocal_opts);
        $merged_geo_str = GeoTools::refine_geoid($merged_geo_str, $geoflag_ptr, $translocal_opts);
    }
    

    #Сохраним в кеше - комбинаций регионов показа обычно меньше, чем групп
    $cache->{$geo_str} = [
         $merged_geo_str,
         defined $geoflag_ptr ? $$geoflag_ptr : undef,
    ];

    return $cache->{$geo_str}->[0];
}

sub _sort_by_level {
    my ($tree, @regions) = @_;
    return sort {$tree->{$b}->{level} <=> $tree->{$a}->{level} } @regions;
}

=head2 geo_changes_to_string($geo_changes, $geo_flag_ref, $client_id)

    Собирает по структуре измененных регионов приходящей с frontend'а в json_geo_changes
    строковое представление geo в сохраняемом в БД виде (применен GeoTools::modify_translocal_region_before_save)
    
    $geo_changes - структура измененных регионов вида
      {
        225 => {is_negative => 0},
        1   => {is_negative => 1},
        ...
        merge_geo => 0
      }
    $geo_flag_ref - ссылка на скаляр, в который запишется значение геофлага
    $client_id - идентификатор клиента для определения типа геодерева


    Возвращеет строку гео - список плюс и минус регионов через запятую
=cut 

sub geo_changes_to_string {
    my ($geo_changes, $geo_flag_ref, $client_id) = @_;
    
    my %regions = %$geo_changes;
    delete $regions{merge_geo};
    return undef unless keys %regions;
    
    return _merge_geo('', \%regions, $client_id, $geo_flag_ref, {});
}

=head2 @errors = validate_adgroup_geo($pid, $new_geo, $opt);

    Проверка, можно ли установить указанный таргетинг группе
    Именованные параметры:
        ClientID -- клиент, для использования из интерфейса
        tree -- использование транслокального дерева (для АПИ)
        content_lang -- использовать указанный язык баннеров

=cut

sub validate_adgroup_geo {
    my ($pid, $new_geo, $camp_type, $opt) = @_;
    return () if !$pid || !defined $new_geo;
    return () if (camp_kind_in(type => $camp_type, 'internal'));

    my $translocal_opt = $opt->{ClientID} ? {ClientID => $opt->{ClientID}} : {tree => $opt->{tree}};
    if ($camp_type eq 'performance') {
        my $creatives = get_all_sql(PPC(pid => $pid),
            ["SELECT pc.sum_geo
            FROM banners_performance bp
            JOIN perf_creatives pc USING (creative_id)",
            WHERE => {'bp.pid' => $pid},
            "GROUP BY pc.creative_id"]);

        my $adgroup = new Direct::Model::AdGroupPerformance->new(geo => $new_geo);
        for my $creative (@{Direct::Model::Creative->from_db_hash_multi($creatives)}) {
            my $geo_error = Direct::Validation::BannersPerformance::validate_creative_geo($creative, $adgroup, translocal_opts => $translocal_opt);
            if ($geo_error) {
                return iget('Геотаргетинг группы %s шире списка стран её cмарт-баннеров', $pid); 
            }
        }
    } else {
        my $content_lang = content_lang_to_save($opt->{content_lang});
        my %langs;

        if ($content_lang) {
            $langs{$content_lang} = 1;
        } else {
            # определяем язык объявлений
            my $banners = Direct::Banners->get_lang_banners(adgroup_id => $pid)->items // [];
            $langs{$_->detect_lang}++ foreach @$banners;
        }

        delete $langs{ru};
        if (keys(%langs) > 1) {
            return iget('Тексты некоторых объявлений написаны на украинском, казахском, турецком или узбекском языках, установить единый географический таргетинг невозможно.');
        } elsif ($langs{uk} && !is_targeting_in_region($new_geo, $geo_regions::UKR, $translocal_opt)) {
            return iget('Тексты некоторых объявлений написаны на украинском языке, единый географический таргетинг должен быть установлен на Украину.');
        } elsif ($langs{kk} && !is_targeting_in_region($new_geo, $geo_regions::KAZ, $translocal_opt)) {
            return iget("Вами установлен регион для показа объявлений: %s. Пожалуйста, установите один географический регион «Казахстан» (отключите все остальные регионы), поскольку текст объявления написан на казахском языке.", get_geo_names($new_geo));
        } elsif ($langs{tr} && !is_targeting_in_region($new_geo, $geo_regions::TR, $translocal_opt)) {
            return iget('Тексты некоторых объявлений написаны на турецком языке, единый географический таргетинг должен быть установлен на Турцию.');
        } elsif ($langs{be} && !is_targeting_in_region($new_geo, $geo_regions::BY, $translocal_opt)) {
            return iget('Тексты некоторых объявлений написаны на белорусском языке, единый географический таргетинг должен быть установлен на Беларусь.');
        } elsif ($langs{uz} && !is_targeting_in_region($new_geo, $geo_regions::UZB, $translocal_opt)) {
            return iget('Тексты некоторых объявлений написаны на узбекском языке, единый географический таргетинг должен быть установлен на Узбекистан.');
        } elsif ($langs{vie} && !is_targeting_in_region($new_geo, $geo_regions::ASIA, $translocal_opt)) {
            return iget('Тексты некоторых объявлений написаны на вьетнамском языке, единый географический таргетинг должен быть установлен на Азию.');
        }
    }
    return ();
}

=head2 get_merged_geo_str($cid, $geo, %options)

    Мержит указанное гео с гео групп кампании
    Возвращает строку содержащую объединенное гео

    Входные параметры: 
     $cid - номер кампании
     $geo - геотаргетинг
     %options
        ClientID - пользователь
        merge_geo(merge|override) - режим применения общего гео

=cut

sub get_merged_geo_str {
    my ($cid, $geo_or_geo_changes, %options) = @_;
    
    my $mode = $options{merge_geo} ? 'merge' : 'override';
    my $client_id = $options{ClientID};
    my $common_geoflag;
    my ($geo_str, $geo_changes);
    my $cache = {};

    if ($mode eq 'merge') {
        $geo_changes = $geo_or_geo_changes;
        my $union_geo = get_union_geo_for_camp($cid, $client_id)->{geo};
        $geo_str = _merge_geo($union_geo, $geo_changes, $client_id, \$common_geoflag, $cache);
    }
    else {
        my $regions = ref $geo_or_geo_changes ?
             join( ', ', map { $geo_or_geo_changes->{$_}->{is_negative} ? -$_ : $_} sort {$a <=> $b} keys %$geo_or_geo_changes)
            :$geo_or_geo_changes;
        $geo_str = _merge_geo($regions, {}, $client_id, \$common_geoflag, $cache);
    }
    return $geo_str;
}

=head2 get_common_geo_for_camp

    calculate default geo for banner from old banners
    my $common_geo = get_common_geo_for_camp($cid);
    if (defined $common_geo) {
        print STDERR "common geo = $common_geo\n";
    } else {
        print STDERR "common geo not set\n";
    }

=cut

sub get_common_geo_for_camp($)
{
    my ($cid ) = @_;
    return undef if !$cid;
    # тип кампании получен при вызове camp_kind_in, поэтому в get_camp_supported_adgroup_types_with_geo не будет второго запроса в базу, а значение типа будет получено из кэша
    my $all_banners_data = !camp_kind_in(cid => $cid, 'media') ?
        get_all_sql(PPC(cid => $cid), ['SELECT DISTINCT geo FROM phrases', WHERE => {cid => $cid, 'adgroup_type' => get_camp_supported_adgroup_types_with_geo(cid => $cid)}])
        : get_all_sql(PPC(cid => $cid), ['SELECT DISTINCT geo FROM media_groups', WHERE => {cid => $cid}]);
    if (defined $all_banners_data && ref($all_banners_data) eq 'ARRAY' && @$all_banners_data) {
        my %all_geo;
        for my $row (@$all_banners_data) {
            if (defined $row->{geo}) {
                my @sorted_regions = sort split(/,/, $row->{geo});
                $row->{geo} = join(',', @sorted_regions);
                $all_geo{$row->{geo}} = 1;
            }
        }

        if (scalar(keys %all_geo) == 1) {
            return GeoTools::substitute_temporary_geo((keys %all_geo)[0]);
        } else { # not common geo for camp
            return;
        }
    } else { # camp without banners
        my $camp_geo = get_one_field_sql(PPC(cid => $cid), "select geo from campaigns where cid = ?", $cid);
        return GeoTools::substitute_temporary_geo($camp_geo);
    }
}


=head2 get_union_geo_for_camp

        Вычисляет объединенный список регионов по всем группам кампании
        Входные параметры:
                $cid - идентификатор кампании
        Выходные данные:
                ссылка на hash с информацией о том, в для каких групп включено/выключено таргетировние на регион
                ключ - номер региона,
                значение - {all => 1}, если у всех групп кампании явным образом включено таргетирование на этот регион
                         - {partly => {kind => "enabled|disabled", adgroups_ids => [ ... ] }} - для каких груп таргетирование на этот регион
                         отличается от среднего по кампании:
                                kind => enabled - для групп из списка включено, для остальных выключено,
                                kind => disabled - для групп из списка выключено, для остальных включено
                        Из двух списков берется наиболее короткий.
                Пример:
                {
                        "225" => {all => 1},
                        "127" => {partly => {kind => "enabled", adgroup_ids => [ 111, 222, ... ]} },
                        "31"  => {partly => {kind => "disabled", adgroup_ids => [ 933, 922, ... ]} },
                        "-333" => {partly => {kind => "enabled", adgroup_ids => [ 111, 222, ... ]} },
                        ...
                }

=cut

sub get_union_geo_for_camp
{
    my ($cid, $client_id ) = @_;

    return undef if !$cid;
    return if camp_kind_in(cid => $cid, 'media');

    my (%result, $regions_str, %pid2group_name);

    # тип кампании получен раньше при вызове camp_kind_in, поэтому в get_camp_supported_adgroup_types_with_geo не будет второго запроса в базу, а значение типа будет получено из кэша
    my $all_groups_data = get_all_sql(PPC(cid => $cid), [qq/SELECT pid, group_name, geo FROM phrases/,
        WHERE => {cid => $cid, 'adgroup_type' => get_camp_supported_adgroup_types_with_geo(cid => $cid)},
        ORDER => 'BY pid']);
    if (@$all_groups_data) {
        my $translocal_opts = {ClientID => $client_id};

        my (%all_geo, @all_groups);
        for my $row (@$all_groups_data) {
            $pid2group_name{$row->{pid}} = $row->{group_name};
            # костыль для отсечени "повторных" плюс-регионов (типа 1,10717 = Моск. обл + Бронницы)
            # при этом Крым оставляем, потому что при записи в БД Россия+Украина становится Россия+Украина+Крым
            my $geo = $row->{geo}
                ? GeoTools::filter_useless_geo($row->{geo}, {tree => 'ua'}, preserve => $geo_regions::KRIM)
                : 'none';
            $geo =~ s/,$//; # remove last comma
            $all_geo{$geo} //= [];
            push @{$all_geo{$geo}}, $row->{pid};
            push @all_groups, $row->{pid};
        }

        my $top_regions;
        foreach my $empty_set (qw/none 0/, '') {
            next unless exists $all_geo{$empty_set};

            #Для групп, у которых не заданы регионы показа добавим "весь мир", развернутый в список регионов первого уровня
            $top_regions //= join ',', @{GeoTools::get_translocal_region(0, $translocal_opts)->{childs}};
            $top_regions = GeoTools::modify_translocal_region_before_save($top_regions, $translocal_opts);
            $all_geo{$top_regions} //= [];
            push @{$all_geo{$top_regions}}, @{delete $all_geo{$empty_set}};
        }

        my $translocal_type = GeoTools::get_translocal_type($translocal_opts);

        foreach my $geo_str (keys %all_geo) {
            my @regions = split /\s*,\s*/,  GeoTools::modify_translocal_region_before_show($geo_str, $translocal_opts);
            if ($translocal_type eq 'ru' && (any {$_ == $geo_regions::RUS} @regions)) {
                #Если у клиента с российским геодеровом под Крымом есть минус-регионы
                #Крым показывается как отдельный регион, modify_translocal_region_before_show его не убирает - уберем
                @regions = grep {$_ != $geo_regions::KRIM } @regions;
            }

            foreach (@regions) {
                $result{$_} //= [];
                push @{$result{$_}}, @{$all_geo{$geo_str}};
            }
        }
        my (@common_minus_regions);
        foreach my $region (keys %result) {
            if (@{$result{$region}} < @all_groups) {
                $result{$region} = {partly => {adgroup_ids => $result{$region}}};
            } else {
                $result{$region} = {all => 1};
                #минус-регионы общие для всех групп будут добавлены в общее гео
                push @common_minus_regions, $region if $region < 0;
            }
        }

        my @common_plus_regions = grep {$_ > 0} keys %result;
        #Для общего гео посчитаем объединение по плюс-регионам
        $regions_str = GeoTools::get_targetings_union(\@common_plus_regions, $translocal_opts);
        #добавим минус-регионы общие для всех групп
        $regions_str = join( ',', $regions_str, @common_minus_regions);
        #и причешем результат
        my $geoflag;
        $regions_str = GeoTools::refine_geoid(GeoTools::substitute_temporary_geo($regions_str), \$geoflag, $translocal_opts);
    } else { # camp without banners
        my $camp_geo = get_one_field_sql(PPC(cid => $cid), [qq/SELECT geo FROM campaigns/, WHERE => {cid => $cid}]);
        $camp_geo //= '';
        %result = map { ($_ => { all => 1}) } split /\s*,\s*/, $camp_geo;
        $regions_str = GeoTools::substitute_temporary_geo($camp_geo);
    }
    #перекинем данные минус-регионов в negative, чтобы фронтенд получил только положительные id
    my @minus_regions = grep { $_ < 0 } keys %result;
    foreach my $negative_region (@minus_regions){
        my $region = -$negative_region;
        $result{$region}->{negative} = delete $result{$negative_region};
    }

    return {geo => $regions_str, extended => \%result, pid2group_name => \%pid2group_name };
}

=head2 set_extended_geo_to_camp
        Получает на вход хеш кампании и идентификатор клиента
        Добавляет в хеш кампании union_geo, extended_geo, geo_multipliers_enabled, pid_to_group_name
=cut

sub set_extended_geo_to_camp {
    my ($camp, $client_id) = @_;

    my $cid = $camp->{cid};
    my $hierarchical_multipliers = $camp->{hierarchical_multipliers};

    if ($cid && $client_id) {
        my $geo_union = get_union_geo_for_camp($cid, $client_id);
        $camp->{union_geo} = $geo_union->{geo};
        $camp->{extended_geo} = $geo_union->{extended};
        $camp->{pid_to_group_name} = $geo_union->{pid2group_name};
    }
    my $geo_mlt = $hierarchical_multipliers->{geo_multiplier}->{regions} // [];

    #добавляем в extended_geo значения корректировок по регионам
    $camp->{extended_geo}->{$_->{region_id}}->{multiplier_pct} = $_->{multiplier_pct} foreach @$geo_mlt;
    $camp->{geo_multipliers_enabled} =  $hierarchical_multipliers->{geo_multiplier}->{is_enabled} ? 1 : 0;

    return $camp
}

1;

