##################################################
#
#  Direct.Yandex.ru
#
#  GeoTools
#
#  $Id$
#
##################################################

=head1 NAME

GeoTools

=head1 DESCRIPTION

  пока только определение geo по IP

=cut

package GeoTools;
## no critic (TestingAndDebugging::RequireUseStrict, TestingAndDebugging::RequireUseWarnings)

require Exporter;

our $VERSION = '0.01';
our @ISA = qw(Exporter);
our @EXPORT = qw(
    get_geobase_lookup
    get_geo_from_ip
    get_geo_projection
    get_geo_regions_data
    is_targeting_in_region
    is_targeting_include_region
    get_targetings_union
    get_geo_name_field
    get_geo_names
    get_geo_names_minus
    get_geo_numbers
    get_geo_children
    get_exact_geo_region
    get_cityname_by_geoid
    get_geoid_by_cityname
    get_direct_region_from_geo
    is_region_city
    get_geo_type_by_id
    refine_geoid
    validate_geo
    get_firm_by_country_hash
    moderation_countries
);

use warnings;
use strict;

use Carp;
use geobase5;
use List::Util qw/first/;
use List::MoreUtils qw/uniq none part any first_value/;

use Settings;
use Yandex::I18n;
use Yandex::Validate;
use geo_regions;
use EnvTools;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::HashUtils;
use Yandex::ListUtils qw/xminus nsort xminus xisect/;

use utf8;
use feature 'state';

=head1 VARIABLES

=cut

=head2 $GEOBASE_BIN

    Путь у бинарному файлу с геобазой

=cut

our $GEOBASE_BIN //= '/var/cache/geobase/geodata4.bin';

=head2 %GEO_TYPE_BY_ID

    Соответствия id типов регионов их текстовым названиям

=cut

our %GEO_TYPE_BY_ID = (
    1 => { text => iget('Континент'), token => 'Continent' },
    2 => { text => iget('Регион'), token => 'Region' },
    3 => { text => iget('Страна'), token => 'Country' },
    4 => { text => iget('Федеральный округ'), token => 'Administrative area' },
    5 => { text => iget('Субъект федерации'), token => 'Administrative area' },
    6 => { text => iget('Город'), token => 'City' },
    7 => { text => iget('Село'), token => 'Village' },
    8 => { text => iget('Район города'), token => 'City district' },
    10 => { text => iget('Район'), token => 'District' },
);

=head2 %MAIN_COUNTRIES

    Список "основных" стран их приоритетов.
    Используется для упрощения выбора страны клиентом (например, в списке основные страны стоят вверху).
    Со временем может перейти в динамический формируемый список (наиболее популярные среди клиентов/наиболее денежные и т.д.).

    Ключ в хеше -- id региона из геобазы. Значение -- числовой приоритет (чем меньше тем приоритетнее: выше в списке, более крупно и т.п.).
    Не стоит использовать приоритеты > 9, т.к. в TT используется сортировка в лексикографическом порядке.

=cut

our %MAIN_COUNTRIES = (
    $geo_regions::RUS => 1, # Россия
    $geo_regions::BY => 2, # Белоруссия
    $geo_regions::KAZ => 3, # Казахстан
    $geo_regions::TR => 4, # Турция
    $geo_regions::UZB => 5, # Узбекистан
);

our %ALLOWED_OVERSEAS_TERRITORIES = map { $_ => undef } (
    21534, # Ангилья
);


=head2 %SUBSTITUTE_GEO_ID

Список регионов для прозрачной подстановки (например, перед удалением)

    # Страны Балтии
    980 => [117, 179, 206],

=cut

# DIRECT-54448, можно очистить после миграции
our %SUBSTITUTE_GEO_ID = (
);

=begin comment

        # History

        # Юбилейный -> Королёв
        21620 => [20728],
        # Железнодорожный -> Балашиха
        21622 => [10716],

        # DIRECT-54448
        # Запад/Украина/СНГ/Евразия/Земля
        20524 => [20531, 20535, 20529, 20550, 20530, 20532, 20533, 20534],
        # Восток/Украина/СНГ/Евразия/Земля
        20525 => [20540, 20539, 20538, 20536, 20537],
        # Юг/Украина/СНГ/Евразия/Земля
        20526 => [20541, 20543, 20542],
        # Центр/Украина/СНГ/Евразия/Земля
        20527 => [20544, 20548, 20545, 20547, 20549, 20546],
        # Север/Украина/СНГ/Евразия/Земля
        20528 => [20551, 20552],
        # Страны Балтии
        980 => [117, 179, 206],
        # Ближний Восток/Азия/Евразия/Земля
        1004 => [1056, 181, 210],

=cut



=head1 FUNCTIONS

=cut

#======================================================================


{
my $_lookup;

=head2 get_geobase_lookup

  $geobase = get_geobase_lookup();

=cut

sub get_geobase_lookup
{
    return $_lookup  if defined $_lookup;
    
    $_lookup = geobase5::lookup->new($GEOBASE_BIN);
    return $_lookup;
}

=head2 get_geo_from_ip

  $geo = get_geo_from_ip('127.0.0.1');

=cut

sub get_geo_from_ip
{
    my $ip = shift;

    my $lookup = eval { get_geobase_lookup() }
        or do { warn $@; 0 };

    return $_lookup ? $_lookup->region_id($ip) : undef;
}

=head2 get_country_from_region

    получение страны по региону

=cut

sub get_country_from_region
{
    my $region = shift;

    if (!$region) {
        return undef;
    }

    my $lookup = eval { get_geobase_lookup() }
        or do { warn $@; 0 };

    return $lookup ? $lookup->find_country_id($region) : undef;
}
}



=head2 get_geo_projection($region, $opts) 

    $opts = {
        geo_list => [213, 45, 67],
        type => $geo_regions::CITY,    # или $geo_regions::COUNTRY
    }
    Получить "проекцию" региона $region 
        если задан $opts->{geo_list} -- проекция на этот список регионов  
        если задан $type -- проекция на "список всех регионов такого типа"
        если заданы оба -- проекция на пересечение списков

    Выбор транслокального дерева:
        $opts->{host => 'direct.yandex.ru'}
        $opts->{ClientID => 12345}
        $opts->{tree => 'ua'}

    $region -- только Директовский, т.е. из geo_regions.pm

    проекция региона на список [$r_1, $r_2, ...]: 
    если $region на каком-то уровне вложенности находится внутри $r_i, то результат -- $r_i 
    если таких $r_i несколько, то результат -- самый маленький (по включению) из них 
    если такого $r_i нет -- результат undef

=cut

sub get_geo_projection($$) {
    my ($region, $opts) = (@_);

    my $parents = [];

    if (exists $geo_regions::GEOREG{$region}) {
        my $translocal_opts = hash_cut($opts, qw/ClientID host tree/);
        $parents = get_translocal_region($region, $translocal_opts)->{parents}; 
    }

    my %base = map { $_ => undef } @{$opts->{geo_list} || []};
    for my $p ($region, reverse @{$parents||[]}){
        next if exists $opts->{geo_list} && ! exists $base{$p};
        next if exists $opts->{type} && ($geo_regions::GEOREG{$p}->{type}//0) != $opts->{type};
        return $p;
    }

    return undef;
}

=head2 is_targeting_in_region($targeting, $region, $opt)
    
    Проверить, содержатся ли все регионы таргетинга $targeting в указанном $region
    $opt -- клиент или домен для определения транслокального дерева регионов {host => 'direct.yandex.ru'} | {ClientID => 12345}
    Результат:
        булевское значение (1|0)

=cut    

sub is_targeting_in_region {
    my ($geo, $region, $opt) = @_;
    return 1 if !$region;
    return 0 if !$geo;
    
    my %regions = map {$_ => 1} grep {$_ > 0} split /[^\d-]+/, $region;
    
    for my $r (split /[^\d-]+/, $geo) {
        # пропускаем минус-регионы
        next if $r && $r < 0 || !exists $geo_regions::GEOREG{$r};
        # проверяем, что либо текущий регион, либо один из его предков равен указанному
        return 0 if !grep {defined $regions{$_}} $r, @{get_translocal_region($r, $opt)->{parents}};
    }
    return 1;
}

=head2 is_targeting_include_region($geo, $region, $opt)
    
    Проверить, содержатся ли указанный $region или один из вложенных в него регионов в регионах таргетинга $geo
    $opt -- клиент или домен для определения транслокального дерева регионов {host => 'direct.yandex.ru'} | {ClientID => 12345}
    Результат:
        булевское значение (1|0)

=cut    

sub is_targeting_include_region {
    my ($geo, $region, $opt) = @_;
    return 1 if !$geo || !$region;

    my %geo_plus  = map {$_ => 1} grep {$_ > 0} split /[^\d-]+/, $geo;
    my %geo_minus = map {abs($_) => 1} grep {$_ < 0} split /[^\d-]+/, $geo;

    my %regions = map {$_ => 1} grep {$_ > 0} split /[^\d-]+/, $region;

    # обрабатывается случай когда охват $geo >= $region
    foreach my $r (keys %regions) {
        return 1 if (!keys %geo_plus || grep { $geo_plus{$_} } ($r, @{get_translocal_region($r, $opt)->{parents}})) &&
                     !grep { $geo_minus{$_} } ($r, @{get_translocal_region($r, $opt)->{parents}});
    }

    # обрабатывается случай когда охват $geo < $region
    foreach my $r (keys %geo_plus) {
        return 1 if grep { $regions{$_} } $r, @{get_translocal_region($r, $opt)->{parents}};
    }
    
    return 0;
}


=head2 filter_useless_geo($geo_str, $opt, %O)

Убирает "ненужные" регионы из геотаргетинга
(то есть те, которые уже включены в борлее широкие)

Пример: 1,10717 (Моск. обл + Бронницы) -> 1

Опции:
  preserve => список регионов, которые не надо отфильтровывать

=cut

sub filter_useless_geo {
    my ($geo_str, $opt, %O) = @_;

    $opt //= {tree => 'ua'};
    my $tree = get_translocal_georeg($opt);

    my %preserve_geo = map {$_ => 1} (ref $O{preserve} ? @{$O{preserve}} : $O{preserve} || ());

    my @geo = grep {exists $tree->{abs $_}} split /,/ => $geo_str;
    my %geo_level = map {$_ => scalar @{$tree->{abs $_}->{parents}}} @geo;
    my @sorted_geo = sort {$geo_level{$a} <=> $geo_level{$b}} @geo;

    my %good_geo;
    for my $geo (@sorted_geo) {
        my $geo_id = abs $geo;
        my $reg = $tree->{$geo_id};
        my $known_parent = first {exists $good_geo{$_}} reverse @{$reg->{parents}};

        # skip hung minus-regions
        next if !defined $known_parent && $geo < 0;

        # skip regions with same-sign parents
        next if defined $known_parent && ($good_geo{$known_parent} < 0) == ($geo < 0);

        $good_geo{$geo_id} = $geo;
    }

    return join ',' => grep {exists $good_geo{abs $_} || $preserve_geo{abs $_}} @geo;
}


=head2 get_targetings_union([$geo1, $geo2 ...], $opt)

    формирует объединение геотаргетингов (т.е. если регион охватывается хотя бы одним таргетингом - он попадает в результат)
    учитывает минус-регионы

    Россия -Центр Москва && Россия -Москва => Россия
    Россия -Москва -Питер && Россия -Центр => Россия -Москва
    Россия -Москва -Питер && Центр => Россия -Питер
    Россия -Москва && Украина -Киев => Россия -Москва Украина -Киев

=cut

sub get_targetings_union {
    my ($geo_list, $opt) = @_;

    # преобразовываем все таргетинги в набор плюс-регионов (без минус-регионов)
    my %union_plus_list = ();
    foreach my $geo (@$geo_list) {    
        my $geo_converted_plus_list = convert_mixed_geo_to_plus_reg_list($geo, $opt);
        foreach (@$geo_converted_plus_list) {
            $union_plus_list{$_} = 1;
        }
    }

    # удаляем вложенности друг в друга
    foreach my $reg (keys %union_plus_list) {
        my $reg_id = $reg =~ s{^\!}{}r;
        delete $union_plus_list{$reg} if any { $union_plus_list{$_} } @{get_translocal_region($reg_id, $opt)->{parents}};
    }

    # преобразовываем полученный список плюс-регионов в смешанный таргетинг (плюс-регионы + минус-регионы)
    my $geo_union = convert_plus_reg_list_to_mixed_geo([sort keys %union_plus_list], $opt);

    return $geo_union;
}

=head2 convert_mixed_geo_to_plus_reg_list($geo, $opt)

    Конвертирует смешанный тергетинг (плюс-регионы + минус-регионы) в набор только плюс-регионов (модификатор "!" означает "только этот регион, без вложенных" )

=cut

sub convert_mixed_geo_to_plus_reg_list {
    my ($geo, $opt) = @_;

    $geo = refine_geoid($geo, undef, $opt);

    my %only_plus_list = ();
    my ($geo_plus, $geo_minus) = part {$_ >= 0 ? 0 : 1} split /[^\d-]+/, ($geo // '');
    my %geo_plus_regions_with_minus = %{ hash_merge({map { $_ => undef} @$geo_plus},
                                                    _minus_regions_for_plus_region($geo_minus, $geo_plus, $opt)) };

    # регионы, для которых в таргетинге есть родительские регионы, к самому ближнему родителю добавляем как минус-регионы
    foreach my $reg (keys %geo_plus_regions_with_minus) {
        my $parent_reg = get_geo_projection($reg, hash_merge({geo_list => [ grep { $_ != $reg } keys %geo_plus_regions_with_minus ]}, $opt));
        if (defined $parent_reg) {
            $geo_plus_regions_with_minus{$parent_reg} //= [];
            @{$geo_plus_regions_with_minus{$parent_reg}} = uniq @{$geo_plus_regions_with_minus{$parent_reg}}, $reg;
        }
    }

    # в списках минус-регионов удаляем вложенности друг в друга
    foreach my $minus_regions_list (grep { $_ } values %geo_plus_regions_with_minus) {
        my %cleaned_minus_regions = map { $_ => 1 } @$minus_regions_list;
        foreach my $reg (keys %cleaned_minus_regions) {
            delete $cleaned_minus_regions{$reg} if any { $cleaned_minus_regions{$_} } @{get_translocal_region($reg, $opt)->{parents}};
        }
        @$minus_regions_list = keys %cleaned_minus_regions;
    }

    foreach my $plus_reg (keys %geo_plus_regions_with_minus) {
        my %plus_reg_converted = ($plus_reg => 1);
        while ( @{$geo_plus_regions_with_minus{$plus_reg} // []} ) {
            my $minus_reg = $geo_plus_regions_with_minus{$plus_reg}->[0];
            if ($plus_reg_converted{$minus_reg}) {
                delete $plus_reg_converted{$minus_reg};
                shift @{ $geo_plus_regions_with_minus{$plus_reg} };
            } else {
                my $parent_reg = get_geo_projection($minus_reg, hash_merge({geo_list => [grep { ! m/^\!/ } keys %plus_reg_converted]}, $opt));
                my $is_expanded = 0;
                if (defined $parent_reg) {
                    my $childs = get_translocal_region($parent_reg, $opt)->{childs} // [];
                    if (@$childs) {
                        delete $plus_reg_converted{$parent_reg};
                        $plus_reg_converted{'!' . $parent_reg} = 1;
                        hash_merge \%plus_reg_converted, { map { $_ => 1 } @$childs };
                        $is_expanded = 1;
                    }
                } 
                shift @{ $geo_plus_regions_with_minus{$plus_reg} } unless $is_expanded;
            }
        }
        hash_merge \%only_plus_list, \%plus_reg_converted;
    }

    # удаляем вложенности друг в друга
    foreach my $reg (keys %only_plus_list) {
        my $reg_id = $reg =~ s{^\!}{}r;

        delete $only_plus_list{$reg} if any { $only_plus_list{$_} } @{get_translocal_region($reg_id, $opt)->{parents}};
    }

    return [keys %only_plus_list];
}

=head2 convert_plus_reg_list_to_mixed_geo($plus_reg_list, $opt)

    Конвертирует набор плюс-регионов в смешанный тергетинг (плюс-регионы + минус-регионы)
    (по принципу если есть не более половины детей одного родителя - берем перечень регионов, иначе родитель -исключения)

=cut

sub convert_plus_reg_list_to_mixed_geo {
    my ($plus_reg_list, $opt) = @_;

    my %plus_regions = map { $_ => 1 } @$plus_reg_list;

    # удаляем вложенности друг в друга
    foreach my $reg (keys %plus_regions) {
        my $reg_id = $reg =~ s{^\!}{}r;
        delete $plus_regions{$reg} if ($reg_id ne $reg && $plus_regions{$reg_id}) ||
                                      any { $plus_regions{$_} } @{get_translocal_region($reg_id, $opt)->{parents}};
    }

    my %result_minus_regions = ();
    my %result_plus_regions = ();
    while (keys %plus_regions) {
        # берем первый регион, у которого вместе с братьями из списка нет детей (в списке)
        my ($reg_parent, @reg_siblings, @all_reg_siblings);
        foreach my $reg (keys %plus_regions) {
            $reg =~ s/^\!//;

            $reg_parent = get_translocal_region($reg, $opt)->{parents}->[-1];
            if (defined $reg_parent) {
                @all_reg_siblings = @{get_translocal_region($reg_parent, $opt)->{childs} // []};
            } else {
                @all_reg_siblings = ($reg);
            }
            
            @reg_siblings = grep { $plus_regions{$_} } @all_reg_siblings;

            my @all_except_siblings_parents = uniq map { @{ get_translocal_region($_, $opt)->{parents} } } @{xminus([map { /(\d+)/ ? $1 : ()} keys %plus_regions], \@all_reg_siblings)};

            my $all_reg_siblings_no_childs = xminus \@all_reg_siblings, \@all_except_siblings_parents;
            #добавляем в список регионы заданные как "!регион" ("только регион")
            @reg_siblings = grep { $plus_regions{'!'.$_} } @all_reg_siblings unless @reg_siblings;
            last if scalar(@$all_reg_siblings_no_childs) == scalar(@all_reg_siblings);
            
            undef $reg_parent;
            @reg_siblings = ();
            @all_reg_siblings = ();
        }
        last unless @reg_siblings;

        if (!defined $reg_parent || !$plus_regions{'!'.$reg_parent}) {
            # берем перечень регионов
            foreach my $reg (@reg_siblings) {
                $result_plus_regions{$reg} = 1;
                delete $plus_regions{$reg} || delete $plus_regions{'!'.$reg}; #удаляем и "только регион"

                if ($plus_regions{'!'.$reg}) {
                    foreach my $child (@{get_translocal_region($reg, $opt)->{childs} // []}) {
                        $result_minus_regions{$child} = 1;
                    }
                }
            }
        } else {
            # берем родительский регион -дети не вошедшие в список
            delete $plus_regions{'!'.$reg_parent};
            foreach my $reg (@{xminus \@all_reg_siblings, \@reg_siblings}) {
                $result_minus_regions{$reg} = 1;
            }
            foreach my $reg (@reg_siblings) {
                if ($plus_regions{'!'.$reg}) {
                    foreach my $child (@{get_translocal_region($reg, $opt)->{childs} // []}) {
                        $result_minus_regions{$child} = 1;
                    }
                } else {
                    delete $plus_regions{$reg} || delete $plus_regions{'!'.$reg};
                }
            }
            $plus_regions{$reg_parent} = 1;
        }
    }

    unless ( keys %result_plus_regions ){
        #Если на входе был не пустой список плюс-регионов, а выход получили пустой - используем на выходе входной список, чтобы не разворачиваться на весь мир
        %result_plus_regions = map {/(\d+)/; $1 => 1} @$plus_reg_list;
    }


    return refine_geoid(join(',', @{_sort_regions_by_tree([keys %result_plus_regions], 0, $opt)}, map { "-$_" } sort keys %result_minus_regions), undef, $opt);
}

=head2 _sort_regions_by_tree($reg_list, $opt)

    Сортирует список регионов иерархически по дереву, например:
    Киев, Украина, Тула, Донецк, Весь мир, Россия => Весь мир, Россия, Тула, Украина, Донецк, Киев
    Дети одного родителя сортируются по region_id (для однозначности)

=cut

sub _sort_regions_by_tree {
    my ($reg_list, $tree_node_id, $opt, $sorted_reg_list, $cache) = @_;
    $cache //= {};
    $sorted_reg_list //= [];

    if (any { $_ == $tree_node_id } @$reg_list) {
        push @$sorted_reg_list, $tree_node_id;
    }

    $cache->{reg_list_parents} //= [uniq map { @{ get_translocal_region($_, $opt)->{parents} } } @$reg_list];
    my %reg_list_extended = map { $_ => 1 } (@{$cache->{reg_list_parents}}, @$reg_list);

    my @childs = nsort grep { $reg_list_extended{$_} } @{ get_translocal_region($tree_node_id, $opt)->{childs} // [] };
    foreach (@childs) {
        _sort_regions_by_tree($reg_list, $_, $opt, $sorted_reg_list, $cache);
    }
    return $sorted_reg_list;
}

=head2 get_geo_name_field($lang)

    По коду языка получить названия поля, в котором хранится название региона в geo_regions

=cut

{
my %aliases = (
    ru => 'name',
    ua => 'ua_name',
    en => 'ename',
    tr => 'tr_name'
);

sub get_geo_name_field {
    return $aliases{shift || Yandex::I18n::current_lang} || 'name';
}
}

# форматирование строки с названиями регионов в соотвествии с языком интерфейса
{
my %excepts = (
    ru => '(кроме: %s)',
    ua => '(окрім: %s)',
    en => '(except: %s)',
    tr => '(%s dahil değil)'
);
sub _format_geo_name {
    
    my ($lang, $regions, $sep) = @_;
    
    $sep = ', ' unless defined $sep;
    my $except = $excepts{$lang} || '%s';
    join $sep, map {
        @{$_->{excepts}} > 0
            ? sprintf('%s %s', $_->{name}, sprintf($except, join $sep, @{$_->{excepts}}))
            : $_->{name}
    } @$regions
}
}


=head2 get_geo_names

  Получить строку из названий регионов, разделённых сепаратором
  Делаем строку типа: Центр, Северо-Запад, Поволжье, Юг (кроме: Анапа, Сочи), Сибирь

=cut

sub get_geo_names
{
    my ( $geo, $sep ) = @_;

    my $lang = Yandex::I18n::current_lang();
    my $field = get_geo_name_field($lang);
    
    if (!defined $geo || $geo =~ /^\s*0?\s*$/) {
        $geo =  0;
    }

    my @regs = uniq grep { /-?\d/ && exists $geo_regions::GEOREG{abs($_)} }
               split( /\s*,\s*/, (defined $geo ? $geo : '') );

    # если регион 0 - формируем сложное название региона
    if (!grep {$_} @regs) {
        @regs = @{ $geo_regions::GEOREG{0}->{childs} };
    } elsif ($regs[0] < 0) {
        unshift @regs, 0;
    }

    # +0 нужен на случай, если попадется регион -0
    my @geo_names;
    push @geo_names, {
        name => $geo_regions::GEOREG{shift(@regs) + 0}->{$field},
        excepts => []
    };
    foreach my $reg (@regs) {
        if ($reg >= 0) {
            push @geo_names, {
                name => $geo_regions::GEOREG{$reg + 0}->{$field},
                excepts => []
            }
        } else {
            push @{$geo_names[-1]->{excepts}}, $geo_regions::GEOREG{abs $reg}->{$field};
        }
    }

    return _format_geo_name($lang, \@geo_names, $sep);
}

=head2 get_geo_names_minus

  упрощённый вариант перекодирования набора номеров регионов в названия со знаком "-" в качестве указателя исключаемого региона.

=cut

sub get_geo_names_minus {
    my ( $geo, %O ) = @_;

    my $lang = defined $O{lang} ? $O{lang} : Yandex::I18n::current_lang;
    my $field = get_geo_name_field($lang);

    return $geo_regions::GEOREG{0}->{$field} if !defined($geo) || $geo eq 0;
    my $sep = defined $O{separator} ? $O{separator} : ', ';
    
    my @regs = uniq grep { exists $geo_regions::GEOREG{abs($_)} }
               split( /\s*,\s*/, (defined $geo ? $geo : '') );
               
    return $geo_regions::GEOREG{0}->{$field} if !@regs;
    my @plus_regs = ();
    my $buf = '';
    my $except_flag = 0;
    
    return join( $sep, map {
            m/^(-?)/;
            $1 . ( ( exists $geo_regions::GEOREG_UNIQ_ALIAS{abs($_)} ? $geo_regions::GEOREG_UNIQ_ALIAS{abs($_)}->{$field} : undef )
                || $geo_regions::GEOREG{abs($_)}->{$field} )
        } @regs );
}

=head2 get_geo_numbers

  Получить номера из названий регионов

=cut

{ # get_geo_numbers closure
my %GEO_REVERSE;

sub get_geo_numbers {
    my ($geo_str, %O) = @_;

    my $lang = defined $O{lang} ? $O{lang} : Yandex::I18n::current_lang;
    my $field = get_geo_name_field($lang);
    
    if ( !$GEO_REVERSE{$lang} ) {
        while (my ($geo, $data) = each %geo_regions::GEOREG) {
            my $geo_name;
            if ( exists $geo_regions::GEOREG_UNIQ_ALIAS{$geo} ) {
                $geo_name = lc($geo_regions::GEOREG_UNIQ_ALIAS{$geo}->{$field});
            }
            $geo_name ||= lc($data->{$field});
            $geo_name =~ s/\s+//g;
            $GEO_REVERSE{$lang}{$geo_name} = $geo;
        }
    }
    return 0 if ! defined $geo_str || length($geo_str) == 0;
    $geo_str = lc($geo_str);
    $geo_str =~ s/\s+//g;
    return 0 if $geo_str eq 'все';

    my @geos;
    my %uniq_geo;
    for my $geo_one (split /\s*,\s*/, $geo_str) {

        my $is_minus_geo = $geo_one =~ s/^-// ? 1 : 0;

        next if exists $uniq_geo{$geo_one};
        next unless defined $GEO_REVERSE{$lang}{$geo_one};

        $uniq_geo{$geo_one} = 1;
        push @geos, ($is_minus_geo ? '-' : '') . $GEO_REVERSE{$lang}{$geo_one};
    }

    return join ',', @geos;
}

} # get_geo_numbers closure

=head2 get_geo_children

  Получить массив из детей, с учётом минус-регионов

=cut

sub get_geo_children
{
    my $profile = Yandex::Trace::new_profile('geo_tools:get_geo_children');
    my $geo = shift || 0;
    my $opt = shift;

    # дерево регионов обновляется вместе с рестартом апача, поэтому не заботимся о сбросе кеша
    state $cache = {};

    my $translocal_type = get_translocal_type($opt);
    if (exists $cache->{$translocal_type}->{$geo}) {
        return @{$cache->{$translocal_type}->{$geo}};
    }

    my @geo = grep {exists $geo_regions::GEOREG{abs($_)}} split(/\s*,\s*/, $geo);
    # Сортируем по уровню иерархии
    ## здесь у Freenode::DollarAB ложное срабатывание
    ## no critic (Freenode::DollarAB)
    my @regions = sort { scalar( @{ get_translocal_region(abs($a), $opt)->{parents} || [] } ) 
                         <=> scalar( @{ get_translocal_region(abs($b), $opt)->{parents} || [] } )
                         } @geo;
    my %res;
    for my $reg ( @regions ) {
        my $absreg = abs( $reg );
        # Получаем всех потомков текущего региона
        my @cur_childs = grep {
            grep { $_ eq $absreg }
            @{ get_translocal_region($_, $opt)->{parents} }
        } keys %{ get_translocal_georeg($opt) };

        # Заносим их в хэш (или удаляем, если это минус-регион)
        for( $absreg, @cur_childs ) {
            if ( $reg < 0 ) {
                delete $res{$_};
            } else {
                $res{$_} = 1;
            }
        }
    }
    $cache->{$translocal_type}->{$geo} = [ keys %res ];
    return keys %res;
}

=head2 get_exact_geo_region

    Получить данные по точному региону
    Параметры:
     - region_id
    Результат:
     - ссылка на хэш с полями 
         region_id
         parent_id
         type
         name / ename / ua_name
     - undef, если региона нет в базе

=cut
sub get_exact_geo_region {
    my $region_id = shift;
    return get_one_line_sql(PPCDICT, "
                            SELECT region_id, parent_id,
                                   type,
                                   name, ename, ua_name
                              FROM geo_regions
                             WHERE region_id = ?", $region_id);
}

=head2 get_cityname_by_geoid

    Возвращает название региона по его region_id
    Параметры:
     - region_id
    Результат:
     - название региона
     - undef, если региона нет в базе

=cut
sub get_cityname_by_geoid
{
    my $geo_id = shift;

    return get_one_field_sql(PPCDICT, "select name from geo_regions where region_id = ?", $geo_id);
}

=head2 get_geoid_by_cityname

    Находит точный номер региона по его названию
    Параметры:
     - название региона
    Результат:
     - region_id
     - название региона
     - 0, если регион не найден в базе

=cut
sub get_geoid_by_cityname
{
    my $city = shift || return 0;

    state $cache = {};
    return $cache->{$city} if exists $cache->{$city};

    my @name_fields = qw/name ua_name ename tr_name/;
    my $name_fields_sql = join ',', @name_fields;

    my $candidates = get_all_sql( PPCDICT, ["SELECT region_id, $name_fields_sql FROM geo_regions",
        where => { _OR => { map {  $_ => $city } @name_fields } }
    ]);

    my $city_lc = lc($city);
    for my $r (@$candidates) {
        if (any { lc($r->{$_}//'') eq $city_lc } @name_fields) {
            $cache->{$city} = $r->{region_id};
            last;
        }
    }

    return $cache->{$city} ||= 0;
}

=head2 get_direct_region_from_geo

    Получить регион с урезанной точностью по точному региону

=cut

sub get_direct_region_from_geo {
    my ($region_id) = @_;
    if (!$region_id) {
        return 0;
    } elsif (exists $geo_regions::GEOREG{$region_id}) {
        return $region_id;
    } else {
        my $lookup = GeoTools::get_geobase_lookup();
        my $region = geobase5::region->new;
        while(1) {
            $lookup->region_by_id($region_id, $region);
            $region_id = $region->parent_id;
            return 0 unless $region_id;
            return $region_id if exists $geo_regions::GEOREG{$region_id};
        }
    }
}

=head2 is_region_city

=cut

sub is_region_city {
    my $region_id = shift;

    return exists $geo_regions::GEOREG{$region_id} && $geo_regions::GEOREG{$region_id}{type} == 6;
}

=head2 get_geo_type_by_id

=cut

sub get_geo_type_by_id($;%) {
    my ($type, %O) = @_;
    return $O{get_token} ? $GEO_TYPE_BY_ID{$type}{token} : $GEO_TYPE_BY_ID{$type}{text} || '';
}


=head2 refine_geoid

 Нормализовать список id регионов, получить GeoFlag

    $geo_str = refine_geoid('213,-10650', \$geoflag, $opt);
    $geo_str = refine_geoid([213,-10650], \$geoflag, $opt);
    $geoflag => 1|0

=cut

sub refine_geoid {
    my ($idlist, $geoflag, $opt) = @_;

    return undef unless defined $idlist;

    my @geo_ids;
    if (ref($idlist) eq 'ARRAY') {
        @geo_ids = uniq @$idlist;
    } elsif(ref($idlist) eq '') {
        $idlist =~ s/\s+//g;
        @geo_ids = uniq grep {/\S/} split( /\s*,\s*/, $idlist );
    } else {
        die 'invalid idlist';
    }

    @geo_ids = grep {is_valid_int($_) && exists $geo_regions::GEOREG{abs($_)}} @geo_ids;

    my ($plus_regions, $minus_regions) = part {$_ >= 0 ? 0 : 1} @geo_ids;

    if (!$plus_regions || !@$plus_regions || (@$plus_regions == 1 && !$plus_regions->[0])) {
        $plus_regions = [0];
    }

    my %minus_regions_for_plus_region = %{ _minus_regions_for_plus_region($minus_regions, $plus_regions, $opt) };

    my @refined_geo_ids;
    for my $plus_region (@$plus_regions) {
        push @refined_geo_ids, $plus_region;
        if ($minus_regions_for_plus_region{$plus_region} && @{$minus_regions_for_plus_region{$plus_region}}) {
            push @refined_geo_ids, map {-$_} @{$minus_regions_for_plus_region{$plus_region}};
        }
    }

    # если 1 - то нужно показать регион, если 0 - то не надо
    if($geoflag) {
        $$geoflag = (none { !$geo_regions::GEOREG{abs($_)}->{geo_flag} } @refined_geo_ids) ? 1 : 0;
    }

    return join ',', @refined_geo_ids;
}

=head2 _minus_regions_for_plus_region

    Вспомогательная функция
    Делит минус-регионы по принадлежности к плюс-региону, возвращает ссылку на хеш

=cut

sub _minus_regions_for_plus_region {
    my ($minus_regions, $plus_regions, $opt) = @_;

    my %minus_regions_for_plus_region;
    if ($minus_regions && @$minus_regions) {
        for my $minus_region (map {abs} @$minus_regions) {
            my $plus_region = get_geo_projection($minus_region, {geo_list => $plus_regions, %$opt});
            if (defined $plus_region) {
                push @{$minus_regions_for_plus_region{$plus_region}}, $minus_region;
            } # минус-регионы, которые неоткуда вычитать, выкидываем
        }
    }
    return \%minus_regions_for_plus_region;
}

=head2 validate_geo

    Проверяет строку с идентификаторами регионов вида "255,-128,-42"
    Возвращает undef если строка корректна и текст ошибки в противном случае
    Вторым параметром принимает ссылку на хеш, в который в случае ошибки заносятся подробности.
    Возвращается первая найденная ошибка и её подробности. Порядок проверки не гарантируется.
    Минус-регионы в подробностях возвращаются положительными числами. Понять, что это был минус-регион
    можно в зависимости от ошибки.

    $error_text = validate_geo("255,-128,-42");
    $error_text = validate_geo("255,-128,-42", \%errors);
    $error_text => undef | "error abcd"
    %errors = (
        not_exists => $region_str, # Неверный или несуществующий регион %s
        duplicate => $region_id, # Регион %s повторяется несколько раз
        minus_only => 1, # Регионы не могут только исключаться
        includes => [$region_id_bigger, $region_id_smaller], # Регион %s уже включает в себя регион %s
        excludes => [$region_id_bigger, $region_id_smaller], # Регион %s полностью исключает регион %s
    );

=cut

sub validate_geo {
    my ($geo_str, $out_err, $opt) = @_;

    my @geo_ids;
    if (defined $geo_str) {
        $geo_str =~ s/\s+//g;
        @geo_ids = grep {/\S/} split( /\s*,\s*/, $geo_str );
    }

    @geo_ids = (0) unless @geo_ids;

    my %already_seen_geo_ids;
    for my $geo_id ( @geo_ids ) {
        if ( !is_valid_int($geo_id) || !exists $geo_regions::GEOREG{ abs($geo_id) } ) {
            $out_err->{not_exists} = $geo_id if $out_err;
            return iget('Неверный или несуществующий регион %s', $geo_id);
        }
        if ($already_seen_geo_ids{$geo_id}) {
            $out_err->{duplicate} = $geo_id if $out_err;
            return iget('Регион %s повторяется несколько раз', get_geo_names($geo_id));
        }
        $already_seen_geo_ids{$geo_id} = 1;
    }

    my ($plus_regions, $minus_regions) = part {$_ >= 0 && $_ !~ /^-/ ? 0 : 1} @geo_ids;
    $minus_regions = [map {abs($_)} @$minus_regions] if $minus_regions;
    $plus_regions = [map {abs($_)} @$plus_regions] if $plus_regions;

    if ($minus_regions && @$minus_regions && !($plus_regions && @$plus_regions)) {
        $out_err->{minus_only} = 1 if $out_err;
        return iget('Регионы не могут только исключаться');
    }

    # минус-регионы не должны один-в-один совпадать с плюс-регионами
    if ($plus_regions && @$plus_regions && $minus_regions && @$minus_regions) {
        for my $plus_region (@$plus_regions) {
            for my $minus_region (@$minus_regions) {
                if ($plus_region == $minus_region) {
                    $out_err->{excludes} = [ $minus_region, $plus_region ] if $out_err;
                    return iget("Регион %s полностью исключает регион %s", get_geo_names($minus_region), get_geo_names($plus_region));
                }
            }
        }
    }

    # каждому минус-региону должен соответствовать плюс-регион
    if ($minus_regions && @$minus_regions) {
        for my $minus_region (@$minus_regions) {

            my $excluded_from_this_plus_region = get_geo_projection($minus_region, {geo_list => $plus_regions, %$opt});
            if (!defined $excluded_from_this_plus_region) {
                $out_err->{nowhere_to_exclude} = $minus_region if $out_err;
                return iget("Исключаемый регион %s не содержится ни в одном из регионов показа", get_geo_names($minus_region));
            }
        }
    }

    return undef;
}

=head2 get_firm_by_country_hash

    Возвращает хеш соответствия балансового идентификатора фирмы идентификатору страны клиента
    Названия фирм хранятся в %Currencies::FIRM_NAME

    TODO: научиться учитывать флаг is_agency, так как из-за него фирмы могут быть разными

    $country2firm = get_firm_by_country_hash();
    $country2firm => {
        225 => 1,   # Россия
        187 => 2,   # Украина
        ...
    };

=cut

my $firm_by_country_hash;
sub get_firm_by_country_hash {
    if (!$firm_by_country_hash) {
        $firm_by_country_hash = get_hash_sql(PPCDICT, 'SELECT region_id, MAX(firm_id) AS firm_id FROM country_currencies GROUP BY region_id');
    }

    return $firm_by_country_hash;
}

=head2 get_geo_regions_data

    Возвращает массив хэшей с информацией о всех регионах с полями: RegionID RegionName ParentID RegionType
    Возвращается не транслокальное дерево с Крымом в корне,
    используется в методе API - GetRegions

    Параметры:
        lang:   язык названий, по умолчанию текущий
        fields: массив названий нужных полей, по умолчанию все, которые есть

=cut

sub get_geo_regions_data {
    my (%opts) = @_;

    my $fields = $opts{fields} || [ qw( RegionID RegionName ParentID RegionType ) ];
    my $lang   = $opts{lang}   || Yandex::I18n::current_lang();

    my @regions_data = ();

    my $geo_field = get_geo_name_field($lang);

    # дерево с Крымом в корне
    while( my( $geo_id, $v ) = each %geo_regions::GEOREG_API ) {
        my $region_type;
        if ( my $typeid = $v->{type} ) {
            $region_type = get_geo_type_by_id( $typeid, get_token => 1 );
        }

        my $region = {
                            RegionID    => $geo_id,
                            RegionName  => $v->{ $geo_field },
                            ParentID    => $v->{parents}->[-1],
                            RegionType  => $region_type,
                     };

        push @regions_data, $region;
    }

    # правила сортировки: "корневой" регион (ParentID = undef) в начале, потом по ParentID, потом по RegionID
    my $sort_cb = sub {
        ## на самом деле $sort_cb вызывается только из sort, так что здесь можно $a и $b
        ## no critic (Freenode::DollarAB)
        my ( $a, $b ) = @_;

        if ( ! defined $a->{ParentID} ) {
            return -1;
        }

        if ( ! defined $b->{ParentID} ) {
            return 1;
        }

        return $a->{ParentID} <=> $b->{ParentID} || $a->{RegionID} <=> $b->{RegionID};
    };

    @regions_data = sort { $sort_cb->( $a, $b ) } @regions_data;

    my @regions_out = map { hash_cut( $_, $fields ) } @regions_data;
    return \@regions_out;
}

=head2 get_country_names_by_id

    Получает названия страы по идентификатору её региона

    $country_data = get_country_names_by_id($country_region_id);
    $country_data => {
        region_id => '84',
        name_ru => 'США',
        name_ua => 'США',
        name_tr => 'Birleşik Devletler',
        name_en => 'United States',
    }

=cut

my %country_regionid2data;
sub get_country_names_by_id {
    my ($region_id) = @_;

    if (!%country_regionid2data) {
        %country_regionid2data = map {$_->{region_id} => $_} @geo_regions::COUNTRY_REGIONS;
    }

    return $country_regionid2data{$region_id};
}

# --------------------------------------------------------------------

=head2 get_translocal_type

Возвращаем тип транслокального дерева регионов клиента

$opt -- клиент или домен или определенное дерево для определения транслокального дерева регионов
    {ClientID => 12345}
    {host => 'direct.yandex.ru'}

    Предопределенные деревья:
    {tree => 'ru'}
    {tree => 'ua'}
    {tree => 'api'}

    Возвращает: 'ru' | 'ua' | 'api'

=cut

sub get_translocal_type($) {
    my $opt = shift // {};

    my $country_region_id;

    state $country_cache = {};

    if (defined $opt->{ClientID}) {
        my $ClientID = $opt->{ClientID};

        if ($ClientID == 0) {
            $country_region_id = 0;
        } elsif (defined $country_cache->{$ClientID} && time() - $country_cache->{$ClientID}->{time} < 3600) {
            # кешируем на час
            $country_region_id = $country_cache->{$ClientID}->{country_region_id};
        } else {
            $country_region_id = get_one_field_sql(PPC(ClientID => $ClientID), ["select country_region_id from clients", where => {ClientID => SHARD_IDS}]) || 0;
            $country_cache->{$ClientID} = {
                time => time(),
                country_region_id => $country_region_id
            };
        }
    } elsif ($opt->{host}) {
        my $tld = $opt->{host} =~ m/\.(\w+)$/ && $1 ? $1 : '';
        $country_region_id = {
            'ru' => $geo_regions::RUS
            , 'ua' => $geo_regions::UKR
            , 'kz' => $geo_regions::KAZ
            , 'by' => $geo_regions::BY
        }->{$tld} || 0;
    } elsif ($opt->{tree}) {
        die "get_translocal_type(): tree param is invalid" unless {'ru' => 1, 'ua' => 1, 'api' => 1}->{$opt->{tree}};
        return $opt->{tree};
    } else {
        if (is_beta()) {
            print STDERR Carp::longmess("get_translocal_georeg(): translocal options not found");
        }
        $country_region_id = 0;
    }

    return get_translocal_tree_type_by_country($country_region_id);
}

# --------------------------------------------------------------------

=head2 get_translocal_tree_type_by_country

По стране возвращаем тип транслокального дерева, пока только% "ru", "ua"

    my $tree_type = get_translocal_tree_type_by_country(187); # 'ua'
    my $tree_type = get_translocal_tree_type_by_country(213); # 'ru'

=cut

sub get_translocal_tree_type_by_country($) {
    my $country_region_id = shift // 0;

    return $country_region_id == $geo_regions::UKR
           ? 'ua'
           : 'ru';
}

# --------------------------------------------------------------------

=head2 get_translocal_georeg

    Возвращаем транслокальное дерево \%geo_regions::GEOREG в зависимости от страны клиента
    $opt -- клиент или домен или определенное дерево для определения транслокального дерева регионов
            {host => 'direct.yandex.ru'}
            {ClientID => 12345}
            {tree => 'ru'}
            {tree => 'ua'}

=cut

sub get_translocal_georeg($) {
    my $opt = shift;

    my $translocal_type = get_translocal_type($opt);

    if ($translocal_type eq 'ua') {
        return \%geo_regions::GEOREG;
    } elsif ($translocal_type eq 'ru') {
        return \%geo_regions::GEOREG_FOR_RU;
    } elsif ($translocal_type eq 'api') {
        return \%geo_regions::GEOREG_API;
    } else {
        die "get_translocal_georeg(): param is invalid";
    }
}

# --------------------------------------------------------------------

=head2 get_translocal_region

    Возвращаем транслокальный регион из дерева \%geo_regions::GEOREG по region_id
    $opt -- клиент или домен или определенное дерево для определения транслокального дерева регионов
            {host => 'direct.yandex.ru'}
            {ClientID => 12345}
            {tree => 'ru'}
            {tree => 'ua'}

    my $region = get_translocal_region($region_id, $opt);
    my $region_parents = get_translocal_region($region_id, $opt)->{parents};

=cut

sub get_translocal_region($$) {
    my $region_id = shift;
    my $opt = shift;

    return get_translocal_georeg($opt)->{$region_id};
}

# --------------------------------------------------------------------

=head2 modify_translocal_region_before_save($regions, $opt)

    Поддержка транслокальности, при сохранении таргетинга:

    - Рекламодатель из Украины:
        выбирает регион Украина или содержащий Украину - выставляем ему таргетинг "Украины + Крым"
            если сейчас нет явно Крыма - то для верстки отдаем "-977" - галка снята
            при сохранении в этом случае (регион Украина) "-977" убираем
        выбирает регион Россия или содержащий Россию - выставляем ему таргетинг "Россия"

    - Рекламодатель из России:
        выбирает регион Украина или содержащий Украину - выставляем ему таргетинг "Украина"
        выбирает регион Россия или содержащий Россию - выставляем ему таргетинг "Россия + Крым"

     https://st.yandex-team.ru/DIRECT-30368

     my $regions = modify_translocal_region_before_save($regions, {ClientID => 12345});

=cut

sub modify_translocal_region_before_save($$) {
    my ($regions, $opt) = @_;

    return undef unless defined $regions;

    my $regions_array = [split /\s*,\s*/, $regions];
    my $plus_regions_array = [grep {!m/^-/} @$regions_array];
    my $regions_hash = {map {$_ => 1} @$regions_array};

    my $translocal_type = get_translocal_type($opt);

    if ($translocal_type eq 'ua'
        && ! $regions_hash->{$geo_regions::KRIM}
        && ! $regions_hash->{"-$geo_regions::KRIM"}
        && ! $regions_hash->{"-$geo_regions::UKR"}        
        && (none {is_targeting_in_region($geo_regions::KRIM, $_, {tree => 'api'})} @$plus_regions_array) # по апи-дереву Крыма нет в регионах
        && (any {is_targeting_in_region($geo_regions::UKR, $_, $opt)} @$plus_regions_array)
       )
    {
        # транслокальное дерево для Украины
        push @$regions_array, $geo_regions::KRIM;
    } elsif ($translocal_type ne 'ua'
             && ! $regions_hash->{$geo_regions::KRIM}
             && ! $regions_hash->{"-$geo_regions::KRIM"}
             && (none {is_targeting_in_region($geo_regions::KRIM, $_, {tree => 'api'})} @$plus_regions_array) # по апи-дереву Крыма нет в регионах
             && (any {is_targeting_in_region($geo_regions::RUS, $_, $opt)} @$plus_regions_array)
            )
    {
        # для остального мира
        push @$regions_array, $geo_regions::KRIM;
    }

    # вычитание "Крыма" из любых регионов кроме "всего мира" не имеет смысла - убираем
    if (none {$_ == 0} @$plus_regions_array) {
        $regions_array = xminus($regions_array, ["-$geo_regions::KRIM"]);
    }

    return join(",", @$regions_array);
}

# --------------------------------------------------------------------

=head2 modify_translocal_region_before_show($regions, $opt)

    Поддержка транслокальности, при показе таргетинга:

    - Рекламодатель из Украины:
        если выбрана Украина и нет дополнительно Крыма (c "+" или "-"), то добавляем "-Крым"

    - Рекламодатель из России:
        если выбрана Россия и нет дополнительно Крыма (c "+" или "-"), то добавляем "-Крым"

    Нельзя вызывать более одного раза над одной строкой регонов (иначе получается: 225,977 -> 225 и 225 -> 225,-977)!

     https://st.yandex-team.ru/DIRECT-30368

     my $regions = modify_translocal_region_before_show($regions, {ClientID => 12345});

=cut

sub modify_translocal_region_before_show($$) {
    my ($regions, $opt) = @_;

    return undef unless defined $regions;

    my $regions_array = [split /\s*,\s*/, $regions];
    my $plus_regions_array = [grep {!m/^-/} @$regions_array];
    my $minus_regions_array = [map {s/^-//r} grep {m/^-/} @$regions_array];
    my $regions_hash = {map {$_ => 1} @$regions_array};

    my $translocal_type = get_translocal_type($opt);

    if ($translocal_type eq 'ua'
        && ! $regions_hash->{$geo_regions::KRIM}
        && ! $regions_hash->{"-$geo_regions::KRIM"}
        && (none {is_targeting_in_region($geo_regions::KRIM, $_, {tree => 'api'})} @$plus_regions_array) # по апи-дереву Крыма нет в регионах
        && (any {is_targeting_in_region($geo_regions::UKR, $_, $opt)} @$plus_regions_array)
       )
    {
        # транслокальное дерево для Украины
        push @$regions_array, "-$geo_regions::KRIM";
    } elsif ($translocal_type ne 'ua'
             && ! $regions_hash->{$geo_regions::KRIM}
             && ! $regions_hash->{"-$geo_regions::KRIM"}
             && (none {is_targeting_in_region($geo_regions::KRIM, $_, {tree => 'api'})} @$plus_regions_array) # по апи-дереву Крыма нет в регионах
             && (any {is_targeting_in_region($geo_regions::RUS, $_, $opt)} @$plus_regions_array)
            )
    {
        # для остального мира
        push @$regions_array, "-$geo_regions::KRIM";
    }

    # если есть регион который (по транслокальному дереву) уже должен содержать Крым
    # и нет минус регионов под Крымом
    # то явно его (Крым) не нужно показывать
    if ((any {$_ != $geo_regions::KRIM && is_targeting_in_region($geo_regions::KRIM, $_, $opt)} @$plus_regions_array)
        &&
        (none {$_ != $geo_regions::KRIM && is_targeting_in_region($_, $geo_regions::KRIM, $opt)} @$minus_regions_array)
       )
    {
        $regions_array = xminus($regions_array, [$geo_regions::KRIM]);
    }

    $regions_array = substitute_temporary_geo($regions_array);
    my $result_geo = refine_geoid($regions_array, undef, $opt);

    return $result_geo;
}

=head2 moderation_countries

Из строки геотаргетинга получае список стран, по правилам которых необходимо выполнять модерацию. Поведение
также зависит от выбранного геодерева (через $translocal_opts).

- Для объектов выше страны - перечисляет все входящие туда страны
- Для объектов ниже страны - добавляет в список ту страну, куда они входят
- Для объектов ниже страны, которые при этом не входят ни в одну страну (например, Крым в дереве API), перечисляет все известные страны.

    my $geo_str_with_countries_only = moderation_countries($geo_str, $translocal_opts);

=cut
sub moderation_countries {
    my ($geo, $translocal_opts) = @_;

    return undef unless defined $geo;
    return undef unless length($geo);

    my @regions_array = split /\s*,\s*/, $geo;
    my @plus_regions = grep {!m/^-/} @regions_array;

    my $georeg = get_translocal_georeg($translocal_opts);
    my $TYPE_COUNTRY = 3;

    my %countries;
    for my $plus_region (@plus_regions) {
        # Весь мир
        if ($plus_region == 0) {
            %countries = map { $_->{region_id} => 1 } @geo_regions::COUNTRY_REGIONS;
            last;
        }

        my $region_meta = $georeg->{$plus_region};
        next unless defined $region_meta;

        if ($region_meta->{type} eq $TYPE_COUNTRY) {
            $countries{$plus_region} = 1;
        } elsif ($region_meta->{type} < $TYPE_COUNTRY) {
            my @children = @{$region_meta->{childs}};
            while (@children) {
                my $current_child_id = pop @children;
                my $current_child = $georeg->{$current_child_id};
                if ($current_child->{type} == $TYPE_COUNTRY) {
                    $countries{$current_child_id} = 1;
                } elsif ($current_child->{type} < $TYPE_COUNTRY) {
                    push @children, @{$current_child->{childs}};
                }
            }
        } elsif ($region_meta->{type} > $TYPE_COUNTRY) {
            my $parent_country = first_value { ($georeg->{$_}{type} // 0) == $TYPE_COUNTRY } @{$region_meta->{parents}};
            if ($parent_country) {
                $countries{$parent_country} = 1;
            } else {
                my @all_countries = grep { !exists $GeoTools::ALLOWED_OVERSEAS_TERRITORIES{ $_->{region_id} } } @geo_regions::COUNTRY_REGIONS;
                %countries = map { $_->{region_id} => 1 } @all_countries;
                last;
            }
        }
    }
    return join ",", sort { $a <=> $b } keys %countries;
}


=head2 substitute_temporary_geo

Код для приведения старой версии гео-базы к новой, пока полностью не отработала миграция.
На входе список регионов, массив или строка через запятую. Результат в таком же формате.

Результат стоит прогнать через refine_geoid

=cut

sub substitute_temporary_geo {
    my ($geo_ids) = @_;
    return $geo_ids if !%SUBSTITUTE_GEO_ID || !$geo_ids;

    my $is_string_input = !ref $geo_ids;
    my @geo_ids = $is_string_input ? split /\s*,\s*/ => $geo_ids : @$geo_ids;

    my @subst_geos = grep {$SUBSTITUTE_GEO_ID{abs $_}} @geo_ids;
    return $geo_ids if !@subst_geos;

    # проверяем случай дубля плюс-региона - в этом случае просто удаляем
    state $path = +{
        map {my $r = get_geobase_lookup()->parents($_); shift @$r; ($_ => +{map {$_ => 1} @$r})}
        keys %SUBSTITUTE_GEO_ID
    };

    my %del_geo;
    for my $subst_geo (@subst_geos) {
        my $parents = $path->{$subst_geo};
        next if !$parents;
        next if none {$parents->{$_}} @geo_ids;
        $del_geo{$subst_geo} = 1;
    }

    my %skip = map {-$_ => 1} (
        @geo_ids,
        map {@{$SUBSTITUTE_GEO_ID{$_}}} grep {$_ > 0 && !$del_geo{$_}} @subst_geos,
    );

    my @result =
        uniq
        grep {!$skip{$_}}
        map {
            my $s = $_<0 ? -1 : 1;
            map {$s * $_} @{$SUBSTITUTE_GEO_ID{abs $_} || [abs $_]}
        }
        grep {!$del_geo{$_}}
        @geo_ids;

    # особый вырожденный случай: минус-регионы покрывают все плюс-регионы
    # записываем парента за минусом всех его подрегионов
    if (!@result) {
        my @parents =
            uniq
            map { get_direct_region_from_geo($_) }
            grep { $_ > 0 }
            @geo_ids;
        @result = map {($_, map {-$_} @{$geo_regions::GEOREG{$_}->{childs}})} @parents;
    }

    return $is_string_input
        ? join q{,} => @result
        : \@result;
}

=head2 exclude_region($geo, \@exclude, $opt)

    Добавить минус-регионы из exclude, если в geo есть их родительские регионы, то есть:
    exclude_region('Россия', ["Москва", "Санкт-Перербург"]) => 'Россия,-Москва,-Санкт-Перербург'
    exclude_region('Урал',   ["Москва", "Санкт-Перербург"]) => 'Урал'
    Значения - region_id, в примерах имена для наглядности
    Возвращает список:  ($new_geo, $excluded)
        $new_geo - новый гео
        $excluded - список стран, которые были фактически вычтены - ссылка на массив чисел

    $opt:
        ClientID => 12345 -- клиент для определения транслокального дерева регионов
        return_nonempty_geo => 1 -- если в результате вычитания получился пустой geo - вернуть не пустую строку,
            а строку вида "0,-123,-456"


=cut

sub exclude_region($$$)
{
    my $profile = Yandex::Trace::new_profile('geo_tools:exclude_region');
    my ($geo, $exclude, $opt) = @_;

    if (!defined $exclude || !@$exclude) {
        # если exclude пустой
        return wantarray ? ($geo, []) : $geo;
    }
    $geo ||= 0;
    my (@regions) = map { s/,$//; $_ } ($geo =~ /(\d+,?(?:-\d+,?)*)/g); # разбиваем на группы: "1,-2,-3,4,-5,6" => [ "1,-2,-3", "4,-5", "6" ]
    my @new_geo = ();
    my %used_minus = (); # минус-регион нужно использовать только 1 раз, и желательно - применить его к самому мелкому региноу:
                         # (Россия, Центр) - (МО) -- (Россия, Центр, -МО), но не (Россия,-МО,Центр)
                         # поэтому reverse: обходим регионы в обратном направлении, от мелких к крупным
    for my $reg (reverse @regions) {
        my ($plus_geo, $minus_geo) = part { /^-/?1:0 } split /,/, ($reg||0);
        $minus_geo ||= [];
        my @exclude = @$exclude;
        # если указан надрегион для исключения - убираем из исключений этот подрегион
        # например: (Россия, -Центр) минус (Москва, Спб)  -- Москву убираем из исключений, т.к. уже стоит -Центр
        if (@$minus_geo) {
            @exclude = @{xminus \@exclude, [ get_geo_children(join(',', map { abs } @$minus_geo), $opt) ]};
        }

        # Случай (Россия, Москва) минус (Москва) -- выкидываем Москву
        if (@exclude) {
            $plus_geo = xminus $plus_geo, [ get_geo_children(join(',', @exclude), $opt) ]; # если регион явно указан в гео - выкидываем
        }
            
        # не осталось +регионов - пропускаем эту группу
        unless (@$plus_geo) {
            next;
        }

        # Случай (Россия, Центр) минус (МосОбласть) -- (Россия, Центр, -МосОбласть) добавляем минус-регион
        push @$minus_geo, map { -$_ } @{ (xisect [ get_geo_children(join(',', @$plus_geo), $opt) ], \@exclude ) }; # если исключение - подрегион -- добавляем в минус
        $minus_geo = xminus $minus_geo, [ keys %used_minus ];
        $used_minus{$_}++ for @$minus_geo;
        push @new_geo, join ",", (@$plus_geo, uniq @$minus_geo);
    }
    my $new_geo = join ',', reverse @new_geo; # соединяем снова в обратном порядке: крупные регионы в начале
    $new_geo = join ',', (0, map { -$_ } @$exclude) if !$new_geo && $opt->{return_nonempty_geo};
    my @excluded = get_geo_diff($geo, $new_geo, $opt);
    return wantarray ? ($new_geo, \@excluded) : $new_geo;
}

=head2 get_disabled_geo

    Обертка над методом exclude_region
    Возвращает второй элемент из ответа exclude_region: $excluded - список стран, которые были фактически вычтены - ссылка на массив чисел

=cut

sub get_disabled_geo($$$)
{
    my ($group_geo, $banners_minus_geo, $client_id) = @_;
    my ($effective_geo, $disabled_geo) = exclude_region($group_geo, $banners_minus_geo, { ClientID => $client_id });
    return $disabled_geo;
}

=head2 get_geo_diff

Функция возвращает список geo, которые есть в $geoa, но нет в $geob
Список расширяется/сужается до уровня стран, напимер:
get_geo_diff("Минск,Москва", "Москва") = 'Беларусь'
get_geo_diff("Весь_Мир", "Весь_Мир,-Беларусь") = 'Беларусь'

=cut

sub get_geo_diff
{
    my ($geoa, $geob, $opt) = @_;
    my ($aplus, $aminus) = part { /^-/?1:0 } split /,/, $geoa;
    my ($bplus, $bminus) = part { /^-/?1:0 } split /,/, $geob;
    $aminus //= [];
    $bminus //= [];

    # регионы могут либо полностью исключиться, либо добавиться в минус
    my $gone = xminus $aplus, $bplus;
    my $minus = xminus $bminus, $aminus;

    my %opt = %$opt;
    $opt{type} = $geo_regions::COUNTRY;

    # Крым считаем отдельным регионом и не поднимаем до уровня страны. Только для украинских клиентов
    my $translocal_type = get_translocal_type($opt);
    return
        uniq
        grep { defined }
        map { $translocal_type eq 'ua' && $_ == $geo_regions::KRIM ? $_ : get_geo_projection($_,\%opt) }
        @$gone, map { abs($_) } @$minus;
}


1;

