package GuaranteeLetter;

# $Id$

=pod

=encoding utf-8

=head1 NAME

GuaranteeLetter - шаблонизатор гарантийных писем

=head1 DESCRIPTION

    Гарантийные письма. Создание, тематики.
    Про использование RTF документа как TT шаблона - http://wiki.yandex-team.ru/users/ppalex/RTFasTT

=cut

use strict;
use warnings;

use base 'Exporter';

our @EXPORT = qw/
    make_guarantee_letter_rtf
/;

use utf8;

use Settings;
use Template;
use Encode qw/encode/;
use List::MoreUtils qw/uniq/;

use BannersCommon qw/get_banners/;
use EnvTools;

use Yandex::DBTools;
use Yandex::HashUtils qw/hash_merge hash_cut/;
use Yandex::I18n;
use Yandex::Validate qw/is_valid_int/;

# Хеш соответствия id причины отклонения модерациии и возможных тематик гарантийного письма
my $DIAG_IDS = {
    1   => 1,       # Не соблюдена рекламная политика Яндекса" в пункте про благотворительность.
    2   => 'all',   # Не соответствует действующему законодательству
    43  => 'all',   # Требуется лицензия, сертификат или ГП
    102 => 4,       # Авторское право
};

# Хеш соответствия: фирма - суффикс ("регион") имени шаблона письма. Источник:
# https://jira.yandex-team.ru/browse/DIRECT-22438?focusedCommentId=3831718&page=com.atlassian.jira.plugin.system.issuetabpanels%3Acomment-tabpanel#comment-3831718
my $region_by_firm_id = {
    1  => 'ru',  # Яндекс
    2  => 'ua',  # Яндекс.Украина
    3  => 'ru',  # КазНет Медиа
    4  => 'us',  # Yandex Inc
    7  => 'us',  # Yandex Europe AG
    8  => 'tr',  # Yandex Turkey
    25 => 'kz',  # Яндекс.Казахстан
    27 => 'by',  # Яндекс.Беларусь
};

my $default_firm_id_by_region = { $geo_regions::RUS => 1,
                                  $geo_regions::UKR => 2,
                                  $geo_regions::TR => 8,
                                  $geo_regions::KAZ => 25, #Страна СНГ: Казахстан
                                  $geo_regions::BY  => 27, #Страна СНГ: Беларусь
                                  168 => 1, #Страна СНГ: Армения
                                  209 => 1, #Страна СНГ: Таджикистан
                                  29386 => 1, # Страна СНГ: Абхазия
                                  170 => 1, #Страна СНГ: Туркмения
                                  208 => 1, #Страна СНГ: Молдова
                                  207 => 1, #Страна СНГ: Кургизия
                                  167 => 1, #Страна СНГ: Азербайджан
                                  171 => 1, #Страна СНГ: Узбекистан
                                };

# Определяет, шаблоны какой фирмы использовать, если она не задана совсем
my $DEFAULT_FIRM_ID = 1;

# Определяет, какую тематику отдать в интерфейс, если что-то отдать нужно, а в шаблоне такой тематики нет.
our $DEFAULT_TOPIC_ID = 1;

# Путь, внутри которого лежат шаблоны RTF документов. По нему делаются проверки юнит-тестами
# ВАЖНО - все шаблоны должны иметь расширение .rtf
our $RTF_TEMPLATES_PATH ||= "$Settings::ROOT/data/t/docs/";
# Базовая часть имени файла
our $RTF_TEMPLATE_FILENAME ||= 'guarantee_letter';

# Максимальное количество подставляемых ключевых слов
my $MAXIMUM_KEYWORDS_COUNT = 10;


=head2 Тематики гарантийных писмем
    Отображаются в интерфейсе именно в этом порядке
    Описание ключей хеша:
        ID      - (ключи хеша) вшиты в шаблонный RTF. (проверяются юнит-тестами!)
        name    - отображаемое имя тематики (в интерфейсе)
        region  - в каких регионах данная тематика может быть использована
        dont_show - не показывать в общем списке в интерфейсе

=cut

my $topics = [
    {
        id      => 1,
        name => iget_noop('Общее'),
        region  => {ru => 1, ua => 1, tr => 1, us => 1, kz => 1, by => 1},
    }, { 
        id      => 10,
        name    => iget_noop('Электронные сигареты'),
        # данная тематика была удалена из шаблонов в DIRECT-27591
    }, {
        id      => 5,
        name    => iget_noop('Товары и услуги, не относящиеся к медицине'),
        region  => {ru => 1, ua => 1, tr => 0, us => 1, kz => 1, by => 1},
    }, {
        id      => 3,
        name    => iget_noop('Товары, законно введённые в оборот'),
        # данная тематика была удалена из шаблонов в DIRECT-38734
    }, {
        id      => 4,
        name    => iget_noop('Объекты интеллектуального права'),
        region  => {ru => 1, ua => 1, tr => 0, us => 1, kz => 1, by => 1},
        dont_show => 1,
    }, {
        id      => 7,
        name    => iget_noop('Патенты'),
        # данная тематика была удалена из шаблонов в DIRECT-38734
    }, {
        id      => 2,
        name    => iget_noop('Симуляторы игровых автоматов'),
        # данная тематика была удалена из шаблонов в DIRECT-38734
    }, {
        id      => 8,
        name    => iget_noop('Пиротехника'),
        region  => {ru => 1, ua => 1, tr => 0, us => 1, kz => 1, by => 1},
    }, {
        id      => 6,
        name    => iget_noop('Товары, конструктивно сходные с оружием'),
        region  => {ru => 1, ua => 1, tr => 0, us => 1, kz => 1, by => 1},
    }, {
        id      => 9,
        name    => iget_noop('Выездное техническое обслуживание автомобилей'),
        # данная тематика была удалена из шаблонов в DIRECT-38734
    }, {
        id      => 11,
        name    => iget_noop('Бинарные опционы'),
        region  => {ru => 1, ua => 0, tr => 0, us => 1, kz => 0, by => 0},
    }, {
        id      => 12,
        name    => iget_noop('Риэлторы'),
        region  => {ru => 0, ua => 0, tr => 0, us => 0, kz => 1, by => 1},
    },
];

my $topics_by_id;
$topics_by_id->{$_->{id}} = $_ for @$topics;

# Конфигурация шаблона
my %CONFIG = (
    PLUGIN_BASE => 'Yandex::Template::Plugin',
    ENCODING => 'utf8',
    EVAL_PERL => 0,
    INCLUDE_PATH => $RTF_TEMPLATES_PATH,
    COMPILE_EXT  => '.ttc',
    COMPILE_DIR  => "/tmp/tt-cache-GuaranteeLetter-$<",
    CACHE_TTL => is_beta() ? 1 : 3600,
    INTERPOLATE  => 0, # expand "$var" in plain text
    POST_CHOMP   => 0, # cleanup whitespace
);

my %encodings = (
    ru => 'windows-1251',
    ua => 'windows-1251',
    by => 'windows-1251',
    kz => 'windows-1251',
    us => 'cp1252',
    tr => 'windows-1254',
);

# Код, экранирующий некоторые символы должным образом
# Copy-paste (не всё) from RTF::Writer
{
    my @Escape = map { sprintf("\\'%02x", $_) } ( 0x00 .. 0xFF );
    foreach my $i ( 0x20 .. 0x7E ) {  $Escape[$i] = chr($i) }
    my @refinements = (
        "\\" => "\\'5c",
        "{"  => "\\'7b",
        "}"  => "\\'7d",

        "\cm"  => '',
        "\cj"  => '',
        "\n"   => "\n\\line ",
        # This bit of voodoo means that whichever of \cm | \cj isn't synonymous
        #  with \n, is aliased to empty-string, and whichever of them IS "\n",
        #  turns into the "\n\\line ".

        "\t"   => "\\tab ",     # Tabs (altho theoretically raw \t's might be okay)
        "\f"   => "\n\\page\n", # Formfeed
        "-"    => "\\_",        # Turn plaintext '-' into a non-breaking hyphen
                                #   I /think/ that's for the best.
        "\xA0" => "\\~",        # \xA0 is Latin-1/Unicode non-breaking space
        "\xAD" => "\\-",        # \xAD is Latin-1/Unicode soft (optional) hyphen
        '.' => "\\'2e",
        'F' => "\\'46",
    );

    my($char, $esc);
    while(@refinements) {
        ($char, $esc) = splice @refinements,0,2;
        $Escape[ord $char] = $esc;
    }

    sub _enc_char {
        my ($encoding, $char) = @_;
        return '\\\''.sprintf('%x', ord(encode($encoding, $char)));
    }

    sub _rtfesc {
        my ($encoding, $text) = @_;

        $text =~ s/([F\.\x00-\x1F\-\\\{\}\x7F-\xFF])/$Escape[ord$1]/g;  # ESCAPER
            # Escape \, {, }, -, control chars, and 7f-ff.
        $text =~ s/([^\x00-\xFF])/'\\uc1\\u'.((ord($1)<32768)?ord($1):(ord($1)-65536))._enc_char($encoding, $1)/eg;
            # фокус такой - записываем каждый символ в текст документа дважды: как юникодный
            # и в 8-битной заданной кодировке (для редакторов без поддержки юникода)

        return $text;
    }
}

# Собирает данные по баннеру для подстановки в шаблон письма:
#   номер кампании, ключевые слова, домен или номер телефона
# Возвращает хеш с данными для передачи в шаблон или undef, если баннер не нашелся
sub _get_data_from_banner {
    my $bid = shift;

    unless (is_valid_int($bid, 0)) {
        warn 'Valid bid must be specified';
        return undef;
    }

    my $vars = {};

    my ($banners, $banners_count) = get_banners(
        { bid => [$bid], adgroup_types => [qw/base mobile_content dynamic/] },
        { no_pokazometer_data => 1 },
    );

    return undef unless $banners_count;   # ничего не нашлось

    # Номер кампании
    $vars->{cid} = $banners->[0]->{cid};

    # Ключевые фразы
    my @keywords;
    foreach my $phrase (@{$banners->[0]->{phrases}}) {
        my $phr = $phrase->{phrase};
        $phr =~ s/\s-.*$//g;    # удаляем минус-слова
        $phr =~ s/[+"!]//g;     # удаляем операторы
        $phr =~ s/\]\[/ /g;     # заменяем на пробел, чтобы слова не слипались
        $phr =~ s/[\]\[]//g;    # удаляем оставшиеся операторы

        # В ключевиках попадаются большие буквы, в тексте смотрится некрасиво, но как отличать имена собственные?
        # Поэтому регистр не трогаем
        push @keywords, $phr;
        last if scalar @keywords >= $MAXIMUM_KEYWORDS_COUNT;
    }
    $vars->{phrases} = join ', ', @keywords;

    # Ссылка на сайт или телефон
    if ($banners->[0]->{adgroup_type} eq 'mobile_content') {
        $vars->{domain} = $banners->[0]->{store_content_href};
    } elsif ($banners->[0]->{domain} && $banners->[0]->{href}) {
        # На банере есть домен, подставляем его
        $vars->{domain} = $banners->[0]->{domain};
    } elsif ($banners->[0]->{phone}) {
        # Домена не оказалось, но есть телефон. Возьмем его
        $banners->[0]->{phone} =~ s/\#/ /gi;
        $banners->[0]->{phone} =~ s/\s+$//i;
        $vars->{phone} = $banners->[0]->{phone};
    } # else - ни домена ни телефона, подставлять нечего

    return $vars;
}

=head2 get_topics

    Возвращает ссылку на массив хешей с тематиками, которые будут отображены в интерфейсе
    Принимает позиционные параметры:
        diag_id - ID причины отклонения.
        letter_region - Регион, на который надо ориентироваться при выборе языка письма

=cut

sub get_topics {
    my $diag_id = shift || 0;
    my $letter_region = shift;

    # Если нет тематик писем для заданной причины отклонения - ничего показывать не будем.
    return undef unless exists $DIAG_IDS->{$diag_id};
    my $topicid = $DIAG_IDS->{$diag_id};

    return undef unless defined $letter_region;

    my $result = [];

    if ($topicid eq 'all') {
        # Отдаем весь список причин
        # В список попадают причины без флага dont_show и с флагом региона
        push @$result, map { hash_cut($_, qw/id name/)  } grep {!$_->{dont_show} && $_->{region}->{$letter_region}} @$topics;
    } else {
        # Отдаем только 1 причину. В интерфейсе будет сформирована сразу готовая ссылка
        if ($topics_by_id->{$topicid}->{region}->{$letter_region}) {
            # данная тематика доступна в заданном регионе
            push @$result, hash_cut($topics_by_id->{$topicid}, qw/id name/);
        } else {
            # Недоступна, используем общую (например авторское право - в Турции)
            push @$result, hash_cut($topics_by_id->{$DEFAULT_TOPIC_ID}, qw/id name/);
        }
    }

    return $result;
}

{
    my $template;

=head2 make_guarantee_letter_rtf

    Создает гарантийное письмо.
    Возвращает ссылку на скаляр, содержащий документ в формате RTF.

    Принимает следующие именованные параметры
        letter_region - Регион, на который надо ориентироваться при выборе языка пись
        topicid - id тематики. должен быть одним из ключей хеша $topics
        bid     - номер баннера, из которого будет заполнить некоторые сведения
        agency  - флаг, показывающий, что письмо предназначено для агентства

=cut

sub make_guarantee_letter_rtf {
    my %OPT = @_;

    if (ref($template) !~ /Template/) {
        $template = Template->new(\%CONFIG);
    }

    # Проверка доступности пользователю заданной тематики должна быть проведена "снаружи"
    my $vars = {
        topicid => $OPT{topicid},
        client_country_region_id => $OPT{client_country_region_id},
    };

    hash_merge($vars, _get_data_from_banner($OPT{bid}));

    # Часть текста различается  для агентств и рекламодателей
    $vars->{agency} = 1 if ($OPT{agency});

    my $doc;
    my $tmpl_file = sprintf ('%s_%s.rtf', $RTF_TEMPLATE_FILENAME, $OPT{letter_region});
    $template->context->define_filter( 'text', sub {_rtfesc($encodings{$OPT{letter_region}}, shift);} );
    $template->process($tmpl_file, $vars, \$doc) || die $template->error();

    return \$doc;
}

=head2 get_letter_region

    Выбирает какой язык(регион) ГП надо использовать для client_country_region_id
    Если для

=cut

sub get_letter_region {
    my ($client_country_region_id, $for_agency) = @_;

    return $region_by_firm_id->{$DEFAULT_FIRM_ID} unless $client_country_region_id;

    my $region_firm_ids = get_one_column_sql(PPCDICT, ['SELECT firm_id FROM country_currencies',
                            where => {region_id => $client_country_region_id,
                                      is_agency => $for_agency ? 1 : 0},
                            "GROUP by region_id, firm_id"]);

    my $firm_id = (scalar(@$region_firm_ids) == 1) ? $region_firm_ids->[0] : $default_firm_id_by_region->{$client_country_region_id} || $DEFAULT_FIRM_ID;

    return $region_by_firm_id->{$firm_id} || $region_by_firm_id->{$DEFAULT_FIRM_ID}; 
}

}


=head3 _get_topics_hash

    Возвращает ссылку на хеш с тематиками. Ключи хеша - id тематик. Используется в тестах.

=cut

sub _get_topics_hash {
    return $topics_by_id;
}

=head3 _get_regions

    Возвращает ссылку на массив с используемыми "регионами" (суффиксами) шаблонов. Используется в тестах

=cut

sub _get_regions {
    return [uniq values %$region_by_firm_id];
}

=head3 _get_encodings

    Возвращает ссылку на хеш с кодировками. Используется в тестах

=cut

sub _get_encodings {
    return \%encodings;
}

1;
