use warnings;
use strict;

=head1 NAME

HashingTools -- простые функции про хеш-суммы

=head1 DESCRIPTION

Простые функции для работы с хеш-суммами, специфичные для Директа. 

Не должен зависеть ни от чего в Директе (включая Settings), 
из внешнего допустимы только модули про хеш-суммы.

=cut


package HashingTools;

use feature 'state';
use base qw/Exporter/;

use Data::UUID;
use Digest::MD5 qw(md5 md5_hex md5_base64);
use Digest::SHA qw();
use Encode;
use MIME::Base64;


BEGIN {
    if (1 << 33 > 1 << 30) {
        *half_md5_hash = \&HashingTools::half_md5_hash_64;
    } else {
        require Math::Int64;
        *half_md5_hash = \&HashingTools::half_md5_hash_32;
    }
};


our @EXPORT = qw(
    half_md5hex_hash
    half_md5_hash
    md5_hex_utf8
    url_hash_utf8
    url_hash
    bs_md5int_utf8

    encode_64ya
    decode_64ya
    md5_base64ya
    base64_to_base64ya

    get_random_string
);
our @EXPORT_OK = qw(
    sha256_hex
    hmac_sha256_hex
);

sub md5_hex_utf8 {
    return md5_hex( Encode::encode('utf8', $_[0]) );
}

# Получить хэш от урла
# conv( substr( md5('sdafasdfasdf'), 1, 16 ), 16, 10 ) - SQL аналог

sub url_hash {
    return 0 if !defined $_[0];
    my $md5 = Digest::MD5::md5($_[0]);
    return half_md5_hash($md5);
}

# '000011f8f39b36e03c88b2a296d67b12' => 63116952593404464
sub half_md5hex_hash {
    return half_md5_hash(pack("H*", $_[0]));
}

sub url_hash_utf8 {
    my $text = shift;
    return url_hash(Encode::encode("utf8", $text));
}

sub half_md5_hash_32 {
    return 0 if !defined $_[0];
    my ($f, $s) = unpack "NN", $_[0];
    return ''.( ( Math::Int64::uint64($f) << 32 ) + $s );
}

sub half_md5_hash_64 {
    return 0 if !defined $_[0];
    my ($f, $s) = unpack "NN", $_[0];
    return ''.( ( $f << 32 ) + $s );
}

#   URI_encode_uniform version of encode_base64
sub encode_64ya{
    my $str = shift or return undef;
    $str = encode_base64($str,"");
    $str =~ tr!+/!\-_!;
    return $str;
}

#   URI_encode_uniform version of decode_base64
sub decode_64ya{
    my $str = shift or return undef;
    $str =~ tr!\-_!+/!;
    $str = decode_base64($str);
    return $str;
}

=head2 md5_base64ya

URI_encode_uniform version of md5_base64

=cut

sub md5_base64ya{
    return base64_to_base64ya( md5_base64(@_) );
}

=head2 base64_to_base64ya

convert a base64 string to URI_encode_uniform

=cut

sub base64_to_base64ya {
    my ($str) = @_;
    $str =~ tr!+/!\-_!;
    return $str;
}

=head2 is_valid_md5_base64ya($str)

=cut

sub is_valid_md5_base64ya {
    my ($str) = @_;
    return ( $str =~ m{^[A-Za-z0-9\-_]{22}$} );
}


=head2 bs_md5int_utf8($str)

	Тайная формула расчета числа из MD5 суммы, использующаяся для фраз в БК.
	Забрал из Yabs::Funcs::md5int (https://a.yandex-team.ru/arc/trunk/arcadia/yabs/basic_packages/yabs-base-modules/Yabs/Funcs.pm#L1408)
	На текущий момент нужна исключительно для МОЛ, в других местах лучше не использовать

=cut

sub bs_md5int_utf8 {
    my ($str) = @_;
    my @ar = unpack("N4", Digest::MD5::md5(Encode::encode("utf8", $str)));
    my $hig = $ar[1] ^ $ar[3];
    my $low = $ar[0] ^ $ar[2];
    return ($hig << 32 | $low);
}

=head2 get_random_string

    Генерирует случайную строку с префиксом (если передан)
        Параметры именованные:
        prefix - префикс случайной строки (default = '')
        alphabets - параметры выбора символов, из которых будет состоять строка, (default = wWds)
            w - маленькие латинские буквы
            W - большие латинские буквы
            d - цифры
            s - спецсимволы
        symbols - вместо alphabets можно указывать непосредственно символы
        length - итоговая длина строки (включая префикс) (default = 10)

=cut
sub get_random_string {
    state $called_srand;    # содержит информацию, вызывали ли мы srand для текущего процесса
    my %O = @_;

    # значения по умолчанию
    $O{length} ||= 10;
    $O{prefix} ||= '';
    $O{alphabets} ||= 'wWds';

    # собираем алфавит
    my $symbols = {
        'w' => 'abcdefghijklmnopqrstuvwxyz',
        'W' => 'ABCDEFGHIJKLMNOPQRSTUVWXYZ',
        'd' => '0123456789',
        's' => '!@#$%^&*()_+,.'
    };

    my $alphabet = $O{symbols} || join ('', @{$symbols}{split('', $O{alphabets})});

    # усекаем префикс, если его длина не позволяет сгенерировать 6 и более случайных символов
    if (length($O{prefix}) > $O{length} - 6) {
        $O{prefix} = substr($O{prefix}, 0, $O{length} - 6);
    }

    if (! defined $called_srand || $called_srand != $$) {
        srand();
        $called_srand = $$;
    }

    my $random = join '', map { substr($alphabet, int(rand(length($alphabet))), 1) } (1 .. ($O{length} - length($O{prefix})));

    return join '', $O{prefix}, $random;
}

=head2 generate_uuid()

    Сгенерировать UUID

=cut
sub generate_uuid {
    state $du;
    $du //= Data::UUID->new();
    return $du->create_str();    
}


=head2 sha256_hex(data)

    Вычислить SHA-256, вернуть строчное hex-представление

=cut
sub sha256_hex {
    return Digest::SHA::sha256_hex(@_);
}


=head2 hmac_sha256_hex(data, key)

    Вычислить HMAC_SHA256 хеш от data, с ключём key

=cut
sub hmac_sha256_hex($$) {
    my ($data, $key) = @_;
    return Digest::SHA::hmac_sha256_hex($data, $key);
}

=head2 generate_push_uuid

Сделать синтетический UUID установки приложения по пуш-токену. Размер не соответствует реальным UUID.
Входной параметр:
    токен
Выходной параметр:
    UUID

=cut

sub generate_push_uuid {
    my $token = shift;
    return (Digest::SHA::sha224_hex($token) . 'DIRE');
}

=head2 push_token_hash

Сделать преобразование аналогичное generate_push_uuid, только без хвоста «DIRE», зато с префиксом об алгоритме хеширования.
Входной параметр:
    токен
Выходной параметр:
    свёртка с префиксом алгоритма свёртки

=cut

sub push_token_hash {
    my $token = shift;
    return ('sha224:' . Digest::SHA::sha224_hex($token));
}

1;
