package HierarchicalMultipliers;

=head1 NAME

HierarchicalMultipliers - функции сохранения и загрузки иерархически перекрываемых опций на кампанию и группу

=head1 DESCRIPTION

Представление коэффициентов в виде хэша (в таком виде save_hierarchical_multipliers принимает на вход, а
get_hierarchical_multipliers загружает из БД). Любые ключи могут отсутствовать.

    {
        mobile_multiplier => {
            hierarchical_multiplier_id => XXX, # Справочно, не используется для сохранения
            last_change => 'YYYY-mm-dd HH:MM:SS', # Справочно.
            multiplier_pct => 237,
            os_type => 'ios', 'android', или undef для всех ОС
        },
        desktop_multiplier => {
            hierarchical_multiplier_id => XXX, # Справочно, не используется для сохранения
            last_change => 'YYYY-mm-dd HH:MM:SS', # Справочно.
            multiplier_pct => 111,
        },
        demography_multiplier => {
            is_enabled => 1,
            hierarchical_multiplier_id => XXX, # Справочно. Идентификатор набора соцдем условий
            last_change => 'YYYY-mm-dd HH:MM:SS', # Справочно.
            conditions => [
                {
                    age => '0-17,18-24,25-34,35-44,45-', # Строка с любыми из указанных диапазонов на выбор, или undef - все
                    gender => "male", # "female", или undef для "не важно"
                    multiplier_pct => 1435,
                    demography_multiplier_value_id => XXX, # Справочно. Идентификатор отдельного условия.
                    last_change => 'YYYY-mm-dd HH:MM:SS', # Справочно.
                },
                # .... до 10 непересекающихся условий
            ],
        },
        retargeting_multiplier => {
            is_enabled => 1,
            hierarchical_multiplier_id => XXX, # Справочно. Идентификатор всего набора корректировок по ретаргетингу
            last_change => 'YYYY-mm-dd HH:MM:SS', # Справочно.
            conditions => {  # Отображение ret_cond_id в значение
                154390 => {
                    multiplier_pct => 205,
                    retargeting_multiplier_value_id => XXX, # Справочно. Идентификатор отдельного условия.
                    last_change => 'YYYY-mm-dd HH:MM:SS', # Справочно.
                },
            },
        },
        geo_multiplier => {
            is_enabled => 1,
            hierarchical_multiplier_id => XXX, # Справочно. Идентификатор набора геокорректировок
            last_change => 'YYYY-mm-dd HH:MM:SS', # Справочно.
            regions => [
                {
        region_id => идентификатор георегиона
                    multiplier_pct => 1435,
                    geo_multiplier_value_id => XXX, # Справочно. Идентификатор отдельной корректировки
                    last_change => 'YYYY-mm-dd HH:MM:SS', # Справочно.
                },
                ....
            ],
        },
    };

=cut
use Direct::Modern;

use JSON;
use List::MoreUtils qw/each_array/;
use List::Util qw/min max sum reduce product/;
use Math::Round qw/round/;

use Yandex::I18n qw/iget/;
use Yandex::HashUtils;
use Yandex::DBTools;
use Yandex::ListUtils qw/chunks/;

use Settings;
use LogTools qw/log_hierarchical_multiplier/;
use Primitives;
use PrimitivesIds;
use Direct::Errors::Messages;
use Direct::Validation::HierarchicalMultipliers qw/
    validate_mobile_multiplier
    validate_desktop_multiplier
    validate_demography_multiplier_condition
    validate_demography_multiplier
    validate_retargeting_multiplier
    validate_geo_multiplier
    validate_multiplier_pct
    /;

use HierarchicalMultipliers::Base qw/insert_multiplier_set dispatch_by_type known_types/;

# Модули цепляются для авторегистрации, ничего лучше чтобы развязать кольцевую зависимость не придумалось.
use HierarchicalMultipliers::Demography qw();
use HierarchicalMultipliers::Inventory qw();
use HierarchicalMultipliers::BannerType qw();
use HierarchicalMultipliers::Retargeting qw();
use HierarchicalMultipliers::Mobile qw();
use HierarchicalMultipliers::Desktop qw();
use HierarchicalMultipliers::Video qw();
use HierarchicalMultipliers::Geo qw();
use HierarchicalMultipliers::AbSegment qw();
use HierarchicalMultipliers::PerformanceTgo qw();
use HierarchicalMultipliers::Weather qw();
use HierarchicalMultipliers::Expression qw();
use HierarchicalMultipliers::TrafaretPosition qw();
use HierarchicalMultipliers::SmartTV qw();
use HierarchicalMultipliers::Tablet qw();
use HierarchicalMultipliers::DesktopOnly qw();

use Campaign::Types qw/get_camp_type/;
use JavaIntapi::UpdateBidModifiers qw//;
use Client qw//;

use base 'Exporter';
our @EXPORT_OK = qw/save_hierarchical_multipliers
                    mass_get_hierarchical_multipliers
                    get_hierarchical_multipliers
                    copy_hierarchical_multipliers
                    delete_camp_hierarchical_multipliers
                    delete_camp_group_hierarchical_multipliers
                   /;

my @TYPE_CLUSTERS = (
    # эти корректировки рассматриваются как "корректировки на устройство" и
    # перекрывают друг друга если заданы на уровне и кампании и группы
    {desktop_multiplier => 'device_multiplier',  mobile_multiplier => 'device_multiplier'},
);

=head2 %JAVA_ADGROUP_COPY_TYPES

    Корретировки на группу, которые копируются через java intapi

=cut
my %JAVA_ADGROUP_COPY_TYPES = map {$_ => 1} (
    'smarttv_multiplier',
    'desktop_only_multiplier',
    'tablet_multiplier');


=head1 FUNCTIONS

=head2 save_hierarchical_multipliers

Сохраняет указанные наборы коэффициентов для кампании или группы. Операция выполняется атомарно в рамках
транзакции БД, указанные в вызове данные полностью заменяют те, что хранятся в БД.

    my $something_was_changed = save_hierarchical_multipliers(
        $cid, $pid_or_undef_if_saving_multipliers_for_campaign,
        {
            mobile_multiplier => 100,
            demography_multiplier => {
                ...
            },
            retargeting_multiplier => {
                ...
            }
        }
    );

Именованные параметры.
    dont_forward_to_java => не перенаправлять запрос в java-intapi, используется в старых юнит тестах.

Возвращает '' если не было сделано никаких изменений, или '1' если что-то в наборах коэффициентов изменилось.
Например, в ответ на это можно сбросить statusBsSynced.

=cut
sub save_hierarchical_multipliers {
    my ($cid, $pid_or_undef, $new_multipliers, %O) = @_;

    my $adgroup_type = defined($pid_or_undef) ? get_adgroup_type($cid, $pid_or_undef) : undef;
    if (defined($pid_or_undef) && $adgroup_type eq 'cpm_outdoor' && !$O{dont_forward_to_java}) {
        my %java_multipliers = (
            %$new_multipliers,
            campaign_id => $cid,
            adgroup_id => $pid_or_undef,
            campaign_type => get_camp_type(cid => $cid)
        );
        my $result = JavaIntapi::UpdateBidModifiers->new(%java_multipliers)->call();
        if (!$result->{success}) {
            die "failed to save hierarchical_multipliers via java-intapi: ".to_json($result);
        }
        return !!$result->{affectedResult};
    }

    my %log_data;
    do_in_transaction {
        my $old_multiplier_sets = get_hashes_hash_sql(
            PPC(cid => $cid), [
                "select type, hierarchical_multiplier_id, cid, pid, multiplier_pct, is_enabled, syntetic_key_hash from hierarchical_multipliers",
                where => _hierarchical_multiplier_where($cid, $pid_or_undef),
                "for update"
            ]
        );

        for my $multiplier_type (keys %$new_multipliers) {
            if (exists $old_multiplier_sets->{$multiplier_type}) {
                my $updated = _update_multiplier_set(
                    $cid, $pid_or_undef, $multiplier_type, $old_multiplier_sets->{$multiplier_type}, $new_multipliers->{$multiplier_type}
                );
                if ($updated) {
                    $log_data{$multiplier_type}{updated} = $updated;
                }
                delete $old_multiplier_sets->{$multiplier_type};
            } else {
                $log_data{$multiplier_type}{inserted} = insert_multiplier_set(
                    $cid, $pid_or_undef, $multiplier_type, $new_multipliers->{$multiplier_type}
                );
            }
        }

        if (%$old_multiplier_sets) {
            for my $mult (values %$old_multiplier_sets) {
                my $log_data = hash_cut $mult, qw/hierarchical_multiplier_id/;
                my $type_log_data = dispatch_by_type($mult->{type}, 'delete', $mult);
                $log_data->{values} = $type_log_data if $type_log_data;
                push @{$log_data{$mult->{type}}{deleted}}, $log_data;
            }
            do_delete_from_table(PPC(cid => $cid), "hierarchical_multipliers", where => _hierarchical_multiplier_where($cid, $pid_or_undef, type => [keys %$old_multiplier_sets]));
        }
        return;
    };

    if (%log_data) {
        $log_data{cid} = $cid;
        $log_data{pid} = $pid_or_undef if $pid_or_undef;
        log_hierarchical_multiplier(\%log_data);
    }
    return !!scalar(keys %log_data);
}

=head2 _update_multiplier_set

Обновляет условия в корретировках, вызывая для этого фукнкцию обнвовления от конкретного типа корректировки.

=cut
sub _update_multiplier_set {
    my ($cid, $pid_or_undef, $multiplier_type, $hierarchical_multiplier, $new_set) = @_;
    return dispatch_by_type($multiplier_type, 'update', $new_set, $hierarchical_multiplier);
}

=head2 get_hierarchical_multipliers

Загружает все наборы коэффициентов для кампании

    my $camp_hier_multiplier = get_hierarchical_multipliers($cid);

или отдельной группы

   my $adgroup_hier_multiplier = get_hierarchical_multipliers($cid, $pid);

=cut
sub get_hierarchical_multipliers {
    my ($cid, $pid) = @_;
    my %where = %{_hierarchical_multiplier_where($cid, $pid)};
    my $result = {};
    my $multipliers = get_all_sql(
        PPC(cid => $cid),
        ["select pid, cid, type, is_enabled, hierarchical_multiplier_id, last_change, multiplier_pct from hierarchical_multipliers", where => \%where]
    );

    for my $hierarchical_multiplier (@$multipliers) {
        $result->{$hierarchical_multiplier->{type}} = dispatch_by_type($hierarchical_multiplier->{type}, 'load', $hierarchical_multiplier);
    }
    return $result;
}


=head2 mass_get_hierarchical_multipliers

NB Надо оптимизировать так, чтобы для каждого отдельного типа выбирать за один sql-запрос.

Загружает все наборы коэффицинтов для указаного списка групп и/или кампаний. Недопустимо передавать
дублирующиеся данные на вход.

    my $result = mass_get_hierarchical_multipliers([
        { cid => 1243490 },
        { pid => 4930990 },
        { cid => 4640, pid => 6493400 }, # Явно передаём cid, не надо определять по pid
    ], %options);

где $result будет содержать записи в виде, описанном в начале модуля (и в том же порядке и количестве, что и
аргументы функции).

    [
        { .. data for campaign 1243490 },
        { .. data for pid 4930990 },
        { .. data for pid 6493400 },
    ]

%options:

- multiplier_type - выбирать только указанные типы коээфициентов, например 'retargeting_multiplier', или
  ['demography_multiplier', 'mobile_multiplier']
- all_groups - если этот флаг установлен, то для кампаний в дополнительном ключе 'groups' будут возвращены все
  группы из этой кампании, для которых заданы корректировки. Например:
      my $result = mass_get_hierarchical_multipliers([
        { cid => 651309 },
        { pid => 4930990 },
      ], all_groups => 1);
      # =>
      [
        { .. usual data for cid 651309 ..
          groups => { pid1 => { .. data for pid1 .. } , ... }, # additional non-standard field
        }
        { .. data for pid 4930990 ..}
      ]
- only_groups - аналогично all_groups, но данные самой кампании выбраны не будут
- heavy - при загрузке коэффициентов загружать всё-всё, в т.ч. добываемое тяжёлыми запросами.
- limit - остановить загрузку, если выбрано больше указанного количества корректировок.
          При этом в списочном контексте функция вернёт { is_partial_result => 1 } во втором элементе результата.

=cut
sub mass_get_hierarchical_multipliers {
    my ($requested_items, %opts) = @_;

    my $stats = {};
    my $loaded_count = 0;
    my @result = map { +{} } @$requested_items;
    my (%result_map, %type_filter);
    my @conditions;

    $type_filter{type} = $opts{multiplier_type} if $opts{multiplier_type};

    # Определяем недостающие cid'ы, запоминаем отображение cid/pid в место для сохранения результата.
    for  (my $pos = 0; $pos <= $#$requested_items; $pos++) {
        my %request = %{$requested_items->[$pos]};
        my $cid = delete $request{cid};
        if (!$cid) {
            die "Either cid or pid should be specified" unless $request{pid};
            $cid = get_cid(pid => $request{pid});
        }
        my $pid = delete $request{pid};
        die "Extra data in request: " . join(", ", keys %request) if %request;
        my $lookup_pid = $pid // 'campaign';
        # die "Duplicate data cid:$cid, pid:$lookup_pid" if exists $result_map{$cid}{$lookup_pid};
        $result_map{$cid}{$lookup_pid} = $result[$pos];

        if ($opts{only_groups} and $lookup_pid eq 'campaign') {
            push @conditions, {cid => $cid, pid__is_not_null => 1, %type_filter};
        } elsif ($opts{all_groups} and $lookup_pid eq 'campaign') {
            push @conditions, {cid => $cid, %type_filter};
        } else {
            push @conditions, _hierarchical_multiplier_where($cid, $pid, %type_filter);
        }
    }

    my @requests = sort { $a->{cid} <=> $b->{cid} } @conditions;
    my @multipliers;
    for my $chunk (chunks \@requests, 1000) {
        my (%cids, @where);
        for (@$chunk) {
            $cids{$_->{cid}} = undef;
            push @where, '_AND', $_;
        }
        my $chunk_mlt = get_all_sql(
            PPC(cid => [keys %cids]),
            ["select pid, cid, type, is_enabled, hierarchical_multiplier_id, last_change, multiplier_pct from hierarchical_multipliers",
             where => {_OR => \@where}]
        );
        push @multipliers, @$chunk_mlt;
    }

    my $add_groups_to_camp = $opts{all_groups} || $opts{only_groups};

    for my $multiplier (@multipliers) {
        my $lookup_pid = $multiplier->{pid} // 'campaign';
        my $loaded = dispatch_by_type($multiplier->{type}, 'load', $multiplier, %{hash_cut \%opts, qw/heavy/});
        if (exists $result_map{$multiplier->{cid}}{$lookup_pid}) {
            # Мы заинтересованы в результате напрямую
            $result_map{$multiplier->{cid}}{$lookup_pid}{$multiplier->{type}} = $loaded;
        }
        if ($add_groups_to_camp && $multiplier->{pid} && exists($result_map{$multiplier->{cid}}{'campaign'})) {
            # Для кампаний нам нужно грузить коэффициенты для всех её групп, т.к. установлена опция all_groups
            $result_map{$multiplier->{cid}}{campaign}{groups}{$multiplier->{pid}}{$multiplier->{type}} = $loaded;
        }
        if ($opts{limit}) {
            $loaded_count += dispatch_by_type($multiplier->{type}, 'calc_stats', $loaded)->{values_count};
            if ($loaded_count > $opts{limit}) {
                $stats->{is_partial_result} = 1;
                last;
            }
        }
    }
    return (\@result, $stats) if wantarray;
    return \@result;
}

=head2 copy_hierarchical_multipliers

Копирует указанные наборы корректировок (в т.ч. объекты, на которые корректировки ссылаются - условия
ретаргетинга и т.п.)

    copy_hierarchical_multipliers(
        $old_client_id, $new_client_id,
        [
            [{cid => $old_cid}, {cid => $new_cid}], # Копируем настройки со старой кампании на новую
            [{cid => $old_cid, pid => $old_pid_1}, {cid => $new_cid, pid => $new_cid_1}], # И со старой группы на новую.
        ]
    );

Старая и новая кампании должны обязательно принадлежать $old_client_id и $new_client_id соответственно (сами
ClientID могут и совпадать).

=cut

sub copy_hierarchical_multipliers {
    my ($old_client_id, $new_client_id, $copy_instructions) = @_;

    my $old_data = mass_get_hierarchical_multipliers([map { $_->[0] } @$copy_instructions]);
    my $iter = each_array @$copy_instructions, @$old_data;

    my $allowed_cpc_device = Client::ClientFeatures::has_cpc_device_modifiers_allowed_feature($new_client_id) ||
        Client::ClientFeatures::is_mobile_os_bid_modifier_enabled($new_client_id);
    while (my ($copy_instruction, $old_multipliers) = $iter->()) {
        my %new_multipliers;
        my %java_adgroup_multipliers;
        while (my($type, $multiplier) = each %$old_multipliers) {
            my $allowed_multiplier = 0;
            if ($type eq 'mobile_multiplier') {
                $allowed_multiplier = !defined $multiplier->{os_type}
                                       || defined $multiplier->{os_type}
                                          && $allowed_cpc_device;
            } elsif ($type eq 'desktop_multiplier') {
                $allowed_multiplier = $allowed_cpc_device;
            } else {
                $allowed_multiplier = 1;
            }

            if ($JAVA_ADGROUP_COPY_TYPES{$type}) {
                $java_adgroup_multipliers{$type} = $multiplier;
            }

            if ($allowed_multiplier) {
                dispatch_by_type($type, 'prepare_for_copy', $old_client_id, $new_client_id, $multiplier);
                $new_multipliers{$type} = $multiplier;
            }
        }
        my ($cid, $pid) = ($copy_instruction->[1]{cid}, $copy_instruction->[1]{pid});
        my $camp_type = get_camp_type(cid => $cid);
        if (!defined $pid) {
            my %java_multipliers = (
                %new_multipliers,
                campaign_id => $cid,
                campaign_type => $camp_type
            );
            JavaIntapi::UpdateBidModifiers->new(%java_multipliers)->call();
        } else {
            if (%java_adgroup_multipliers) {
                my %java_multipliers = (
                    %java_adgroup_multipliers,
                    campaign_id   => $cid,
                    campaign_type => $camp_type,
                    adgroup_id    => $pid
                );
                JavaIntapi::UpdateBidModifiers->new(%java_multipliers)->call();
            }
            save_hierarchical_multipliers($cid, $pid, $old_multipliers);
        }
    }

}

=head2 _hierarchical_multiplier_where

    Формирует условие WHERE для указанной кампании и/или группы. Везде указывать просто {pid => $pid_or_undef}
    нельзя, т.к. для кампании в этом месте нужна проверка IS NULL, а не равенство.

    Т.к. Yandex::DBTools::sql_condition({field => undef}) не превращается автоматически в 'field is null'.

=cut
sub _hierarchical_multiplier_where {
    my ($cid, $pid_or_undef, %rest_of_where) = @_;

    $rest_of_where{cid} = $cid;
    if (defined $pid_or_undef) {
        $rest_of_where{pid} = $pid_or_undef;
    } else {
        $rest_of_where{pid__is_null} = 1;
    }
    return \%rest_of_where;
}

=head2 delete_camp_hierarchical_multipliers

Удаляет данные коэффициентов для кампании и всех её групп - выполняется в рамках удаления кампании.

    delete_camp_hierarchical_multipliers($cid);

=cut
sub delete_camp_hierarchical_multipliers {
    my ($cid) = @_;
    for my $type (@{known_types()}) {
        dispatch_by_type($type, 'delete_camp_values', $cid, $type);
    }
}

=head2 delete_camp_group_hierarchical_multipliers

Удаляет данные коэффициентов для кампании и переданных групп — выполняется в рамках удаления групп.

    delete_camp_group_hierarchical_multipliers($cid, $pids);

=cut

sub delete_camp_group_hierarchical_multipliers {
    my ($cid, $pids) = @_;
    for my $type (@{known_types()}) {
        dispatch_by_type($type, 'delete_camp_group_values', $cid, $pids, $type);
    }
}

=head2 adjustment_bounds

    my $bounds = adjustment_bounds($group_mults, $camp_mults, $group);

где

    $bounds = { adjustments_lower_bound => 123, adjustments_upper_bound => 456 }

Вилка корректировок
Есть две разновидности предложений о корректировках:
А) "Для группы могут быть применены корректировки от X% до Y%"
В) "Для группы могут быть применены корректировки до Y%"
Где X - нижний порог, а Y - верхний порог вилки
Общие правила:
- Если выставлены коэффициенты одного типа (например, ретаргетинг) и на кампанию, и на группу, то при расчете учитываются коэффициенты на группу
- Если выставлены коэффициенты только на кампанию, то при расчете учитываются коэффициенты на кампанию
Расчеты порогов:
Нижний порог - произведение максимальных понижающих коэффициентов всех трех видов (пример: если указана моб. корректировка -50, соцдем -30 и ретаргетинг -10, получаем 5 * 38 1 =15 (или -1400%).
- Если понижающий коэффициент является единственным выбранным коэффициентом, то он становится верхним порогом (ситуация В)
- Если выставлен коэффициент 0, то нижнюю границу мы не показываем (ситуация В)
Верхний порог - произведение максимальных повышающих коэффициентов всех трех видов (пример: если указана моб. корректировка -50, соцдем +200 и ретаргетинг +300, получаем 3.0 * 4.0 = 12.0 (или +1100%).
- Ноль не является повышающим коэффициентом и не участвует при расчётах
- Если выставлена только одна повышающая корректировка, то она является верхним порогом

=cut
sub adjustment_bounds {
    my ($group_multipliers, $camp_multipliers, $opts) = @_;

    # опеределяем какие корректировки на группе перекрывают корректировки на кампании
    my %group_override_types;
    for my $gtype (keys %{$group_multipliers // {}}) {
        for my $cluster (@TYPE_CLUSTERS) {
            if (exists $cluster->{$gtype}) {
                @group_override_types{keys %{$cluster}} = ();
            }
        }
    }
    my $camps_without_groups = hash_kgrep {!exists $group_override_types{$_} } ($camp_multipliers // {});

    my %mults = (%{$camps_without_groups}, %{$group_multipliers //{}});
    my %min;
    my %max;

    while (my($type, $mult) = each %mults) {
        next if exists $mult->{is_enabled} && !$mult->{is_enabled};
        my $type_stat = dispatch_by_type($type, 'calc_stats', $mult, $opts);

        my $pseudo_type = $type;
        for my $cluster (@TYPE_CLUSTERS) {
            if (exists $cluster->{$type}) {
                $pseudo_type = $cluster->{$type};
                last;
            }
        }
        $min{$pseudo_type} = exists $min{$pseudo_type} ?
            min ($type_stat->{multiplier_pct_min}, $min{$pseudo_type})
            : $type_stat->{multiplier_pct_min};
        $max{$pseudo_type} = exists $max{$pseudo_type} ?
            max ($type_stat->{multiplier_pct_max}, $max{$pseudo_type})
            : $type_stat->{multiplier_pct_max};
    }
    my ($lower, $upper);
    if (%min) {
        $lower = round(100 * product map {$_/100} grep {defined $_ && $_< 100} values %min);
        $lower = min grep {defined $_ && $_>100} values %min  if $lower == 100;
        $lower = undef  if $lower && $lower == 100;

        $upper = round(100 * product map {$_/100} grep {defined $_ && $_>100} values %max);
        $upper = max grep {defined $_ && $_<100} values %max  if $upper == 100;
        $upper = undef  if $upper && $upper == 100;

        # если есть только одна граница - её пишем в upper
        ($upper, $lower) = ($lower, undef)  if !$upper || ($lower && $upper==$lower);
    }

    my $result = {
        adjustments_lower_bound => $lower,
        adjustments_upper_bound => $upper,
    };
    return $result;
}

=head2 get_types_allowed_on_camp

    Возвращает список типов корректировок, которые можно выставлять на кампаниях.

=cut
sub get_types_allowed_on_camp {
    return HierarchicalMultipliers::Base::types_allowed_on_camp();
}

=head2 get_expression_multiplier_types

    Возвращает список универсальных типов корректировок.

=cut
sub get_expression_multiplier_types {
    return HierarchicalMultipliers::Base::expression_types();
}

=head2 wrap_expression_multipliers

    см. HierarchicalMultipliers::Base::wrap_expression_multipliers

=cut
sub wrap_expression_multipliers {
    return HierarchicalMultipliers::Base::wrap_expression_multipliers(@_);
}

=head2 unwrap_expression_multipliers

    см. HierarchicalMultipliers::Base::unwrap_expression_multipliers

=cut
sub unwrap_expression_multipliers {
    return HierarchicalMultipliers::Base::unwrap_expression_multipliers(@_);
}

1;

