package HierarchicalMultipliers::AbSegment;

use Direct::Modern;

use List::Util qw/min max product/;
use Math::Round qw/round/;
use List::MoreUtils qw/any part all/;
use Yandex::DBTools;
use Yandex::HashUtils qw/hash_merge/;

use Settings;

use Direct::Model::AbSegmentCondition;
use Direct::AbSegmentConditions;
use JavaIntapi::GenerateObjectIds;

use PrimitivesIds qw/get_clientid/;

use Direct::Validation::HierarchicalMultipliers qw/
    validate_ab_segment_multiplier
    validate_ab_segment_multiplier_segments
    /;
use HierarchicalMultipliers::Base qw/register_type/;

BEGIN {
    register_type(ab_segment_multiplier    => {
            insert                => \&insert,
            update                => \&update,
            delete                => \&delete_set_values,
            load                  => \&load,
            prepare_for_copy      => sub { return undef; },
            calc_stats            => \&calc_stats,
            delete_camp_values    => \&delete_camp_values,
            delete_camp_group_values       => \&delete_camp_group_values,
            multiplier_set_can_be_disabled => 1,
        });
}

=head2 insert

Вставляет запись в 'hierarchical_multipliers'.
Вставляет записи в 'ab_segment_multiplier_values'.

=cut
sub insert {
    my ($data, $proposed_hierarchical_multiplier) = @_;
    $proposed_hierarchical_multiplier->{is_enabled} = $data->{is_enabled} ? 1 : 0;

    my $ab_segments = $data->{ab_segments};
    die 'Ab segments should be a array ref' unless ref($ab_segments) eq 'ARRAY';
    die 'Ab segments should be not empty' unless @$ab_segments;

    my $ids = JavaIntapi::GenerateObjectIds->new(object_type => 'multiplier',
            count =>scalar @$ab_segments)->call();
    my $client_id = get_clientid(cid => $proposed_hierarchical_multiplier->{cid});
    my $existing_ab_segments = Direct::AbSegmentConditions->get_by(client_id => $client_id)->items;
    my @insert_data;
    foreach my $ab_segment_data (@$ab_segments){
        next unless defined $ab_segment_data;
        my $ab_segment_ret_cond_id = _save_ab_segment_ret_cond($client_id,
            [{segment_id => $ab_segment_data->{segment_id}, section_id => $ab_segment_data->{section_id}}], $existing_ab_segments);
        push @insert_data, {
                ab_segment_multiplier_value_id => shift(@$ids),
                hierarchical_multiplier_id     => $proposed_hierarchical_multiplier->{hierarchical_multiplier_id},
                ab_segment_ret_cond_id         => $ab_segment_ret_cond_id,
                segment_id                     => $ab_segment_data->{segment_id},
                section_id                     => $ab_segment_data->{section_id},
                multiplier_pct                 => $ab_segment_data->{multiplier_pct},
            }
    }
    do_in_transaction {
            do_insert_into_table(PPC(cid => $proposed_hierarchical_multiplier->{cid}), "hierarchical_multipliers", $proposed_hierarchical_multiplier);
            do_mass_insert_sql(PPC(cid => $proposed_hierarchical_multiplier->{cid}), 'insert into ab_segment_multiplier_values'
                    . '(ab_segment_multiplier_value_id, hierarchical_multiplier_id, ab_segment_ret_cond_id, multiplier_pct)'
                    . ' values %s',
                [map { [@{$_}{qw/ab_segment_multiplier_value_id hierarchical_multiplier_id ab_segment_ret_cond_id multiplier_pct/}] } @insert_data]);
        };
    return @insert_data ? \@insert_data : undef;
}

=head2 update

Обновляет hierarchical_multipliers.is_enabled (если надо).
Обновляет/заменяет записи в ab_segment_multiplier_values.

=cut
sub update {
    my ($data, $hierarchical_multiplier) = @_;

    my $existing_values = get_hashes_hash_sql(
        PPC(cid => $hierarchical_multiplier->{cid}), [
            "select ab_segment_ret_cond_id, ab_segment_multiplier_value_id, multiplier_pct from ab_segment_multiplier_values",
            where => { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id} },
            "for update"
        ]
    );

    my $retargeting_conditions_by_id = Direct::AbSegmentConditions->get_by(ret_cond_id => [keys %$existing_values])->items_by('id');
    my %existing_values_map = map {$retargeting_conditions_by_id->{$_}->get_segments_ids->[0] => $existing_values->{$_}} keys %$existing_values;
    my $client_id = get_clientid(cid => $hierarchical_multiplier->{cid});

    my (%update_values, @insert_data);
    my %log_data = (inserted => [], updated => [], deleted => []);

    for my $value (@{$data->{ab_segments}}) {
        my $segment_id = $value->{segment_id};
        if (exists $existing_values_map{$segment_id}) {
            my $old_value = delete $existing_values_map{$segment_id};
            if ($old_value->{multiplier_pct} != $value->{multiplier_pct}) {
                $update_values{$old_value->{ab_segment_multiplier_value_id}} = $value->{multiplier_pct};
                push @{$log_data{updated}}, {
                        ab_segment_multiplier_value_id => $old_value->{ab_segment_multiplier_value_id},
                        ab_segment_ret_cond_id => $existing_values_map{$segment_id}->{ab_segment_ret_cond_id},
                        new_multiplier_pct => $value->{multiplier_pct}, old_multiplier_pct => $old_value->{multiplier_pct},
                    };
            }
        } else {
            push @insert_data, {
                    hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id},
                    segment_id => $segment_id,
                    section_id => $value->{section_id},
                    multiplier_pct => $value->{multiplier_pct},
                };
        }
    }

    #Оставшиеся записи удаляем
    my @delete_values = values %existing_values_map;
    if (@delete_values) {
        do_delete_from_table(PPC(cid => $hierarchical_multiplier->{cid}), 'ab_segment_multiplier_values',
            where => {ab_segment_multiplier_value_id => [map { $_->{ab_segment_multiplier_value_id} } @delete_values]});

        for my $value (@delete_values) {
            push $log_data{deleted}, {
                    ab_segment_multiplier_value_id => $value->{ab_segment_multiplier_value_id},
                    ab_segment_ret_cond_id => $value->{ab_segment_ret_cond_id},
                    old_multiplier_pct => $value->{multiplier_pct},
                };
        }
    }

    #Добавляем новые записи
    if (@insert_data) {
        my $ids = JavaIntapi::GenerateObjectIds->new(object_type => 'multiplier',
                count => scalar @insert_data)->call();
        for my $value (@insert_data) {
            $value->{ab_segment_multiplier_value_id} = shift(@$ids);
            $value->{ab_segment_ret_cond_id} = _save_ab_segment_ret_cond($client_id,
                [{segment_id =>$value->{segment_id}, section_id => $value->{section_id}}], [values %$retargeting_conditions_by_id]);
            push $log_data{inserted}, {
                    ab_segment_multiplier_value_id => $value->{ab_segment_multiplier_value_id},
                    ab_segment_ret_cond_id         => $value->{ab_segment_ret_cond_id},
                    segment_id                     => $value->{segment_id},
                    section_id                     => $value->{section_id},
                    new_multiplier_pct             => $value->{multiplier_pct},
                };
        }
        do_mass_insert_sql(PPC(cid => $hierarchical_multiplier->{cid}), 'INSERT INTO ab_segment_multiplier_values'
                . '(ab_segment_multiplier_value_id, hierarchical_multiplier_id, ab_segment_ret_cond_id, multiplier_pct)'
                . ' VALUES %s',
            [map { [@{$_}{qw/ab_segment_multiplier_value_id hierarchical_multiplier_id ab_segment_ret_cond_id multiplier_pct/}] } @insert_data]
        );
    }

    #Обновляем существующие
    if (%update_values) {
        do_update_table(PPC(cid => $hierarchical_multiplier->{cid}), 'ab_segment_multiplier_values',
            {
                multiplier_pct__dont_quote => sql_case('ab_segment_multiplier_value_id', \%update_values),
                last_change__dont_quote => 'NOW()',
            },
            where => { ab_segment_multiplier_value_id => [keys %update_values]});
    }

    #Обновляем last_change у изменившихся сетов, удаляем пустые
    if (@{$data->{ab_segments}}) {
        my $new_is_enabled = $data->{is_enabled} ? 1 : 0;
        my $is_enabled_has_changed = $hierarchical_multiplier->{is_enabled} != $new_is_enabled;
        my $need_update_last_change_for_whole_set = %update_values || @insert_data || @delete_values;
        if ($is_enabled_has_changed) {
            $log_data{is_enabled_change} = { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id}, is_enabled => $new_is_enabled };
        }
        if ($need_update_last_change_for_whole_set || $is_enabled_has_changed) {
            do_update_table(PPC(cid => $hierarchical_multiplier->{cid}), 'hierarchical_multipliers',
                {
                    is_enabled => $new_is_enabled,
                    last_change__dont_quote => 'NOW()',
                },
                where => {hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id}}
            );
        }
    } else {
        do_delete_from_table(PPC(cid => $hierarchical_multiplier->{cid}), 'hierarchical_multipliers',
            where => {hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id}});
        $log_data{deleted_set} = { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id} };
    }

    return unless %log_data;
    return \%log_data;
}

=head2 load

Возвращает:
    {
        is_enabled => XXX,
        last_change => XXX,
        hierarchical_multiplier_id => XXX,
        ab_segments => [
            {
                ab_segment_multiplier_value_id => XXX,
                section_id => XXX,
                segment_id => XXX,
                multiplier_pct => XXX,
            },
            ...
        ],
    }

=cut
sub load {
    my ($hierarchical_multiplier, %opts) = @_;
    my $ab_segment_multiplier_values = get_all_sql(
        PPC(cid => $hierarchical_multiplier->{cid}),
        [
            'SELECT multiplier_pct, ab_segment_ret_cond_id, ab_segment_multiplier_value_id FROM ab_segment_multiplier_values',
            WHERE => { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id} },
        ]
    );

    my @ret_cond_ids = map {$_->{ab_segment_ret_cond_id}} @$ab_segment_multiplier_values;
    my $retargeting_conditions_by_id = Direct::AbSegmentConditions->get_by(ret_cond_id => \@ret_cond_ids)->items_by('id');
    my $ab_segments = [];
    for my $multiplier_value (@$ab_segment_multiplier_values) {
        my $ret_cond = $retargeting_conditions_by_id->{$multiplier_value->{ab_segment_ret_cond_id}};
        delete $multiplier_value->{ab_segment_ret_cond_id};
        $multiplier_value->{multiplier_pct} = int $multiplier_value->{multiplier_pct};
        $multiplier_value->{ab_segment_multiplier_value_id} = int $multiplier_value->{ab_segment_multiplier_value_id};
        push @$ab_segments, hash_merge($multiplier_value,
                {segment_id => $ret_cond->get_segments_ids->[0], section_id => $ret_cond->get_section_ids->[0]});
    }

    my $result = {
                        is_enabled => int $hierarchical_multiplier->{is_enabled},
                        last_change => $hierarchical_multiplier->{last_change},
                        hierarchical_multiplier_id => int $hierarchical_multiplier->{hierarchical_multiplier_id},
                 };
    $result->{ab_segments} = $ab_segments;
    return $result;
}

=head2 calc_stats

Вычисляет часть статистики, используемой для формирования сводной в HierarchicalMultipliers::calc_stats
Корректировки из разных экспериментов перемножаются

=cut
sub calc_stats {
    my ($multiplier) = @_;

    my %segment_mult_by_section = ();
    for my $segment_mult (@{$multiplier->{ab_segments}}) {
        push @{$segment_mult_by_section{$segment_mult->{section_id}} ||=[]}, $segment_mult->{multiplier_pct};
    }

    my %min_values_by_section = map { $_ => min @{$segment_mult_by_section{$_}} } keys %segment_mult_by_section;
    my %max_values_by_section = map { $_ => max @{$segment_mult_by_section{$_}} } keys %segment_mult_by_section;
    my $pct_min = round(100 * product map {$_/100} grep {$_< 100} values %min_values_by_section);
    my $pct_max = round(100 * product map {$_/100} grep {$_> 100} values %max_values_by_section);
    my $result = {
        multiplier_pct_min => $pct_min,
        multiplier_pct_max => $pct_max,
        values_count       => scalar @{$multiplier->{ab_segments}},
    };

    $result->{adjustments_lower_bound} = $pct_min if $pct_min != 100;
    $result->{adjustments_upper_bound} = $pct_max if $pct_min > 100;
    return $result;
}

=head2 delete_camp_values

Удаляем все связанные с кампанией и её группами корректировки.

=cut
sub delete_camp_values {
    my ($cid) = @_;
    do_sql(PPC(cid => $cid), [
            "delete h, r from hierarchical_multipliers h join ab_segment_multiplier_values r using(hierarchical_multiplier_id)",
            where => { 'h.cid' => $cid, 'h.type' => 'ab_segment_multiplier' },
        ]);
}

=head2 delete_camp_group_values

Удаляем все связанные с кампанией и переданными группами корректировки.

=cut

sub delete_camp_group_values {
    my ($cid, $pids) = @_;
    do_sql(PPC(cid => $cid), [
            "delete h, r from hierarchical_multipliers h join ab_segment_multiplier_values r using(hierarchical_multiplier_id)",
            where => { 'h.cid' => $cid, 'h.pid' => $pids, 'h.type' => 'ab_segment_multiplier' },
        ]);
}

=head2 delete_set_values

Удаляем все ab_segment_multiplier_values для указанного набора корректировок.

=cut
sub delete_set_values {
    my ($set) = @_;
    my $deleted = get_all_sql(PPC(cid => $set->{cid}), [
            "select ab_segment_multiplier_value_id, ab_segment_ret_cond_id, multiplier_pct from ab_segment_multiplier_values",
            where => {hierarchical_multiplier_id => $set->{hierarchical_multiplier_id}}
        ]);
    do_delete_from_table(PPC(cid => $set->{cid}), 'ab_segment_multiplier_values', where => {ab_segment_multiplier_value_id => [map { $_->{ab_segment_multiplier_value_id} } @$deleted]});
    return $deleted;
}

=head2 _save_ab_segment_ret_cond

Сохранение корректировок аб-сегментов как ретаргетингов

=cut
sub _save_ab_segment_ret_cond {
    my ($client_id, $segments, $existing_segments) = @_;

    my %segments_by_id = map {$_->{segment_id} => 1} @$segments;
    my $segments_ctn = scalar keys %segments_by_id;
    for my $ret_cond (@$existing_segments) {
        my @cond_segments_ids = @{$ret_cond->get_segments_ids};
        if ($segments_ctn == scalar @cond_segments_ids && all {$segments_by_id{$_}} @cond_segments_ids) {
            return $ret_cond->id;
        }
    }

    my $ret_cond = Direct::Model::AbSegmentCondition->new(
        id         => 0,
        client_id  => $client_id,
        segments  => $segments,
    );
    Direct::AbSegmentConditions->new([$ret_cond])->create();

    return $ret_cond->id;
}
1;
