package HierarchicalMultipliers::Base;

=head1 DESCRIPTION

Реестр поддержимваемых типов корректировок + общие методы, используемые как кодом для работы с корректировками
в целом, так и модулями для отдельных типов корректировок. В отдельном модуле чтобы не было кольцевых
зависимостей.

=cut

use Direct::Modern;

use base qw(Exporter);
our @EXPORT_OK = qw/insert_multiplier_set dispatch_by_type register_type known_types/;

use Yandex::DBShards;

use HashingTools;

=head1 CONSTANTS

=head2 %KNOWN_TYPES

Возвращаемые функциями значения используются для записи в логи, а также для определения факта того, что
изменения действительно были - в этом случае могут быть выполены дополнительные действия, например сброс
statusBsSynced.

    {
        some_type => {
            insert => sub {
                # Should insert new corrections set into 'hierarchical_multipliers' and maybe into some child tables
                my ($some_type_multiplier_data, $proposed_hierarchical_multiplier) = @_;
                # update $proposed_hierarchical_multiplier values according to 'some_type' semantics
                do_insert_into_table(PPC(cid => $proposed_hierarchical_multiplier->{cid}), 'hierarchical_multipliers', $proposed_hierarchical_multiplier);
                # insert records into child tables if feasible
                return $some_summary_of_what_was_inserted_that_is_used_for_logging;
            },
            update => sub {
                # Should update data in child tables to resemble new information (i.e. add new, update existing, delete missing).
                # Whole set updates should also happen here (hierarchical_multipliers.is_enabled, hierarchical_multipliers.multiplier_pct, ...)
                my ($data, $hierarchical_multiplier) = @_;
                # ... do some magic here
                return $some_summary_of_changes_or_undef_if_no_changes_were_made;
            },
            delete => sub {
                # Should delete records from dependent tables (if any)
                my ($hierarchical_multiplier) = @_;
                return $some_summary_of_changes_or_undef_if_no_changes_were_made;
            },
            load => sub {
                my ($hierarchical_multiplier, %opts) = @_;
                # If $opts{heavy} is true, than we should load some optional data (dependent on type of multiplier)
                # E.g. for retargeting multiplier we could load information about metrika goals accessibility.
                # Load data from child tables (if any)
                return $some_hash_described_at_the_beginning_of_the_module;
            },
            prepare_for_copy => sub {
                # Copying multipliers between ClientID's
                # If multiplier set references some other objects, we should ensure their existence under new ClientID
                # And update their identifiers in data passed as arguments to this callback
                my ($old_client_id, $new_client_id, $hierarchical_multiplier) = @_;
                # ... do some magic (maybe updating $hierarchical_multiplier in process) ...
                return $some_summary_of_changes_or_undef_if_no_changes_were_made;
            },
            delete_camp_values => sub {
                my ($cid) = @_;
                # We should delete records from 'hierarchical_multipliers' and any other related tables, that
                # correspond to given campaign and all its groups.
            },
            delete_camp_group_values => sub {
                my ($cid, $pids) = @_;
                # We should delete records from 'hierarchical_multipliers' and any other related tables, that
                # correspond to given campaign and all given groups.
            },
            multiplier_set_can_be_disabled => 0 | 1, # Can be the whole set of modifers be disabled? It's here and not in validation metadata
                                                     # becausue it's independent of campaign type
        }
    }

=cut
my %KNOWN_TYPES = (
    # NB: Если будет добавляться ещё один составной тип, надо будет заодно выделить общий код из
    # _demography_multiplier_update и _retargeting_multiplier_update
);

=head2 @EXPRESSION_TYPES

    список типов универсальных корректировок

=cut
my @EXPRESSION_TYPES;

=head2 %JAVA_WRITE_ONLY_TYPES

    Некоторые корректировки только на кампании реализуются в java и не имеют кода в perl.
    Чтобы не падать, выносим их в исключения.
    Предполагается, что использования таких корректировок в perl не будет.

=cut
my %JAVA_WRITE_ONLY_TYPES = map {$_ => 1} (
    'banner_type_multiplier',
    'inventory_multiplier',
    'weather_multiplier',
    'trafaret_position_multiplier',
    'prisma_income_grade_multiplier',
    'smarttv_multiplier',
    'retargeting_filter',
    'desktop_only_multiplier',
    'tablet_multiplier');
=head1 FUNCTIONS

=head2 _syntetic_key_hash

Синтетический ключ для проверки на уникальность набора опций в привязке к кампании или группе.

    my $key = _syntetic_key_hash($cid, $adgroup_id_or_undef, $option_name);

=cut
sub _syntetic_key_hash {
    my ($cid, $pid_or_undef, $name) = @_;
    my $pid = $pid_or_undef // "campaign";

    return half_md5hex_hash(md5_hex_utf8("$name:$cid:$pid"));
}

=head2 insert_multiplier_set

Предзаполняет запись для 'hierarchical_multipliers' и передаёт её обработчику вставки для конкретного типа.

=cut
sub insert_multiplier_set {
    my ($cid, $pid_or_undef, $multiplier_type, $multiplier_set) = @_;
    die "Unknown multiplier type '$multiplier_type'" unless exists $KNOWN_TYPES{$multiplier_type};
    my $hierarchical_multiplier = {
        hierarchical_multiplier_id => get_new_id('hierarchical_multiplier_id'),
        syntetic_key_hash => _syntetic_key_hash($cid, $pid_or_undef, $multiplier_type),
        cid => $cid,
        pid => $pid_or_undef,
        type => $multiplier_type,
        # multiplier_pct and is_enabled can be overriden by 'insert' callback
        multiplier_pct => undef,
        is_enabled => 1,
    };
    return $KNOWN_TYPES{$multiplier_type}{insert}->($multiplier_set, $hierarchical_multiplier);
}

=head2 dispatch_by_type

Вызывает указанный callback для указанного типа коэффициента.

    my $cb_result = dispatch_by_type($multiplier_type, $method_name, @callback_args);

=cut
sub dispatch_by_type {
    my $type = shift;
    my $method = shift;

    die "Unknown type '$type'" unless exists $KNOWN_TYPES{$type};
    return undef if $method !~ /^(load|calc_stats)$/ && $JAVA_WRITE_ONLY_TYPES{$type};
    return $KNOWN_TYPES{$type}{$method}->(@_);
}

=head2 register_type

Should be used from BEGIN block.

=cut
sub register_type {
    my ($type, $config, %O) = @_;
    if (exists $KNOWN_TYPES{$type}) {
        die "hierarchical multiplier '$type' already registered!";
    }
    $KNOWN_TYPES{$type} = $config;
    if ($O{expression}) {
        push @EXPRESSION_TYPES, $type;
    }
}

=head2 known_types

Возвращает ARRAYREF с названиями известных типов корректировок.

=cut
sub known_types {
    return [keys %KNOWN_TYPES];
}

=head2 types_allowed_on_camp

    Возвращает список типов корректировок, которые можно выставлять на кампаниях.

=cut
sub types_allowed_on_camp {
    state $allowed_types = undef;
    if (!defined($allowed_types)) {
        $allowed_types = [qw/
                mobile_multiplier
                demography_multiplier
                retargeting_multiplier
                geo_multiplier
                video_multiplier
                performance_tgo_multiplier
                ab_segment_multiplier
                banner_type_multiplier
                inventory_multiplier
                desktop_multiplier
                trafaret_position_multiplier
            /,
            expression_types()
        ];
    }

    return @$allowed_types;
}

=head2 expression_types

    Возвращает список универсальных типов корректировок.

=cut
sub expression_types {
    return @EXPRESSION_TYPES;
}

=head2 wrap_expression_multipliers(\%multipliers_from_perl)

    Заворачивает универсальные корректировки в массив expression_multipliers.

    Возвращает ссылку на новый хеш, пригодный для отправки в java-intapi ручки.

    my %perl_mult = (geo_multiplier => {}, express_traffic_multiplier => {});
    my $java_mult = wrap_expression_multipliers(\%perl_mult);
    assert_eq($java_mult, {
        geo_multiplier => {},
        expression_multipliers => [{type => "express_traffic_multiplier", ...}]
    });

=cut
sub wrap_expression_multipliers {
    my ($multipliers_from_perl) = @_;
    if (ref($multipliers_from_perl) ne 'HASH') {
        return $multipliers_from_perl;
    }

    my %result = %$multipliers_from_perl;
    for my $type (expression_types()) {
        next unless $result{$type};
        my $multiplier = delete $result{$type};
        $multiplier->{type} = $type;
        push @{$result{expression_multipliers}}, $multiplier;
    }

    return \%result;
}

=head2 unwrap_expression_multipliers(\%multipliers_from_java)

    Разворачивает универсальные корректировки из java формата в перловый.
    Возвращает ссылку на новый хеш.

    my %java_mult = {
        geo_multiplier => {},
        expression_multipliers => [{type => "express_traffic_multiplier", ...}]
    };
    my $perl_mult = unwrap_expression_multipliers(\%java_mult);
    assert_eq($perl_mult, {geo_multiplier => {}, express_traffic_multiplier => {}});

=cut
sub unwrap_expression_multipliers {
    my ($multipliers_from_java) = @_;
    if (ref($multipliers_from_java) ne 'HASH') {
        return $multipliers_from_java;
    }

    my %result = %$multipliers_from_java;
    my $express_modifiers = delete $result{expression_multipliers};
    for my $express_mod (@$express_modifiers) {
        my $type = delete $express_mod->{type};
        $result{$type} = $express_mod;
    }

    return \%result;
}

1;
