package HierarchicalMultipliers::Demography;
use Direct::Modern;

use List::Util qw/min max/;
use Yandex::HashUtils;
use Yandex::DBTools;

use Settings;
use Direct::Validation::HierarchicalMultipliers qw/
    normalize_age 
    normalize_gender 
    validate_demography_multiplier_condition 
    validate_demography_multiplier
    /;
use HierarchicalMultipliers::Base qw/register_type/;
use JavaIntapi::GenerateObjectIds;

BEGIN {
    register_type(demography_multiplier => {
        insert                => \&insert,
        update                => \&update,
        delete                => \&delete_set_values,
        load                  => \&load,
        prepare_for_copy      => sub { return undef; },
        calc_stats            => \&calc_stats,
        delete_camp_values    => \&delete_camp_values,
        delete_camp_group_values       => \&delete_camp_group_values,
        multiplier_set_can_be_disabled => 1,
    });
}

=head2 insert

В соответствии с описанным в HierarchicalMultipliers API для добавления новых коэффициентов.

Вставляет запись в 'hierarchical_multipliers'.
Вставляет записи в 'demography_multiplier_values'.

=cut
sub insert {
    my ($data, $proposed_hierarchical_multiplier) = @_;

    $proposed_hierarchical_multiplier->{is_enabled} = $data->{is_enabled} ? 1 : 0;

    do_insert_into_table(PPC(cid => $proposed_hierarchical_multiplier->{cid}), "hierarchical_multipliers", $proposed_hierarchical_multiplier);

    die "Demography 'conditions' should be an array ref'" unless ref($data->{conditions}) eq 'ARRAY';
    my $ids = JavaIntapi::GenerateObjectIds->new(object_type => 'multiplier',
            count => scalar @{$data->{conditions}})->call();

    my @insert_data;
    for my $condition (@{$data->{conditions}}) {
        push @insert_data, {
            demography_multiplier_value_id => shift(@$ids),
            hierarchical_multiplier_id => $proposed_hierarchical_multiplier->{hierarchical_multiplier_id},
            gender => normalize_gender($condition->{gender}),
            age => normalize_age($condition->{age}),
            multiplier_pct => $condition->{multiplier_pct},
        };
    }
    do_mass_insert_sql(PPC(cid => $proposed_hierarchical_multiplier->{cid}), 'insert into demography_multiplier_values'
                           . '(demography_multiplier_value_id, hierarchical_multiplier_id, gender, age, multiplier_pct)'
                           . ' values %s',
                       [map { [@{$_}{qw(demography_multiplier_value_id hierarchical_multiplier_id gender age multiplier_pct)}] } @insert_data]);
    return @insert_data ? \@insert_data : undef;
}

=head2 load

В соответствии с описанным в HierarchicalMultipliers API для добавления новых коэффициентов.

Возвращает:
    {
        is_enabled => XXX,
        hierarchical_multiplier_id => XXX,
        last_change => XXX,
        conditions => [
            {
                demography_multiplier_value_id => XXX,
                age => XXX,
                gender => XXX,
                multiplier_pct => XXX,
                last_change => XXX,
            },
            ...
        ]
    }

=cut
sub load {
    my ($hierarchical_multiplier) = @_;
    my $conditions = get_all_sql(
        PPC(cid => $hierarchical_multiplier->{cid}),
        [
            "select age, gender, multiplier_pct, demography_multiplier_value_id, last_change from demography_multiplier_values",
            where => {hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id}},
        ]
    );

    my $result = hash_cut $hierarchical_multiplier, qw/is_enabled last_change hierarchical_multiplier_id/;
    $result->{conditions} = $conditions;
    return $result;
}

=head2 update

В соответствии с описанным в HierarchicalMultipliers API для добавления новых коэффициентов.

Обновляет hierarchical_multipliers.is_enabled (если надо).
Обновляет/заменяет записи в demography_multiplier_values.

=cut
sub update {
    my ($data, $hierarchical_multiplier) = @_;
    my $existing_values = get_all_sql(
        PPC(cid => $hierarchical_multiplier->{cid}), [
            "select demography_multiplier_value_id, gender, age, multiplier_pct from demography_multiplier_values",
            where => { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id} },
            "for update"
        ]
    );
    my %existing_values_map;
    for my $value (@$existing_values) {
        my $lookup_age = normalize_age($value->{age}) // 'all';
        my $lookup_gender = normalize_gender($value->{gender}) // 'all';
        $existing_values_map{$lookup_gender}{$lookup_age} = $value;
    }

    my (%update_values, @insert_data, %log_data);
    for my $value (@{$data->{conditions}}) {
        my $lookup_age = normalize_age($value->{age}) // 'all';
        my $lookup_gender = normalize_gender($value->{gender}) // 'all';
        if (exists $existing_values_map{$lookup_gender}{$lookup_age}) {
            my $old_value = delete $existing_values_map{$lookup_gender}{$lookup_age};
            if ($old_value->{multiplier_pct} != $value->{multiplier_pct}) {
                $update_values{$old_value->{demography_multiplier_value_id}} = $value->{multiplier_pct};
                push @{$log_data{updated}}, {
                    demography_multiplier_value_id => $old_value->{demography_multiplier_value_id},
                    age => $lookup_age, gender => $lookup_gender,
                    new_multiplier_pct => $value->{multiplier_pct}, old_multiplier_pct => $old_value->{multiplier_pct},
                };
            }
        } else {
            push @insert_data, {
                hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id},
                gender => normalize_gender($value->{gender}),
                age => normalize_age($value->{age}),
                multiplier_pct => $value->{multiplier_pct},
            };
        }
    }

    my @delete_values = map { values %$_ } values %existing_values_map;
    if (@delete_values) {
        do_delete_from_table(PPC(cid => $hierarchical_multiplier->{cid}), "demography_multiplier_values",
                             where => {demography_multiplier_value_id => [map { $_->{demography_multiplier_value_id} } @delete_values]});
        for my $value (@delete_values) {
            push @{$log_data{deleted}}, {
                demography_multiplier_value_id => $value->{demography_multiplier_value_id},
                age => $value->{age} // 'all', gender => $value->{gender} // 'all',
                old_multiplier_pct => $value->{multiplier_pct},
            };
        }
    }
    if (@insert_data) {
        my $ids = JavaIntapi::GenerateObjectIds->new(object_type => 'multiplier',
                count => scalar @insert_data)->call();
        for my $value (@insert_data) {
            $value->{demography_multiplier_value_id} = shift(@$ids);
            push @{$log_data{inserted}}, {
                demography_multiplier_value_id => $value->{demography_multiplier_value_id},
                age => $value->{age} // 'all', gender => $value->{gender} // 'all',
                new_multiplier_pct => $value->{multiplier_pct},
            };
        }
        do_mass_insert_sql(PPC(cid => $hierarchical_multiplier->{cid}), 'insert into demography_multiplier_values'
                               . '(demography_multiplier_value_id, hierarchical_multiplier_id, gender, age, multiplier_pct)'
                               . ' values %s',
                           [map { [@{$_}{qw(demography_multiplier_value_id hierarchical_multiplier_id gender age multiplier_pct)}] } @insert_data]);
    }
    if (%update_values) {
        do_update_table(PPC(cid => $hierarchical_multiplier->{cid}), "demography_multiplier_values",
                        {
                            multiplier_pct__dont_quote => sql_case('demography_multiplier_value_id', \%update_values),
                            last_change__dont_quote => 'NOW()',
                        },
                        where => { demography_multiplier_value_id => [keys %update_values]});
    }
    if (@{$data->{conditions}}) {
        my $new_is_enabled = $data->{is_enabled} ? 1 : 0;
        my $is_enabled_has_changed = $hierarchical_multiplier->{is_enabled} != $new_is_enabled;
        my $need_update_last_change_for_whole_set = %update_values || @insert_data || @delete_values;
        if ($is_enabled_has_changed) {
            $log_data{is_enabled_change} = { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id}, is_enabled => $new_is_enabled };
        }
        if ($need_update_last_change_for_whole_set || $is_enabled_has_changed) {
            do_update_table(PPC(cid => $hierarchical_multiplier->{cid}), 'hierarchical_multipliers',
                            {
                                is_enabled => $new_is_enabled,
                                last_change__dont_quote => 'NOW()',
                            },
                            where => {hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id}});
        }
    } else {
        do_delete_from_table(PPC(cid => $hierarchical_multiplier->{cid}), 'hierarchical_multipliers',
                             where => {hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id}});
        $log_data{deleted_set} = { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id} };
    }
    return unless %log_data;
    return \%log_data;
}

=head2 calc_stats

Вычисляет часть статистики, используемой для формирования сводной в HierarchicalMultipliers::calc_stats

=cut
sub calc_stats {
    my ($multiplier) = @_;
    my @values = map { $_->{multiplier_pct} } @{$multiplier->{conditions}};
    my $result = {
        multiplier_pct_min => min(@values),
        multiplier_pct_max => max(@values),
        values_count       => scalar @{$multiplier->{conditions}},
    };
    my $lower_bound = min(grep { $_ != 100 } @values);
    my $upper_bound = max(grep { $_ > 100 } @values);
    $result->{adjustments_lower_bound} = $lower_bound if defined $lower_bound;
    $result->{adjustments_upper_bound} = $upper_bound if defined $upper_bound;
    return $result;
}

=head2 delete_camp_values

Удаляем все связанные с кампанией и её группами корректировки.

=cut
sub delete_camp_values {
    my ($cid) = @_;
    do_sql(PPC(cid => $cid), [
        "delete h, r from hierarchical_multipliers h join demography_multiplier_values r using(hierarchical_multiplier_id)",
        where => { 'h.cid' => $cid, 'h.type' => 'demography_multiplier' },
    ]);
}

=head2 delete_camp_group_values

Удаляем все связанные с кампанией и переданными группами корректировки.

=cut

sub delete_camp_group_values {
    my ($cid, $pids) = @_;
    do_sql(PPC(cid => $cid), [
        "delete h, r from hierarchical_multipliers h join demography_multiplier_values r using(hierarchical_multiplier_id)",
        where => { 'h.cid' => $cid, 'h.pid' => $pids, 'h.type' => 'demography_multiplier' },
    ]);
}

=head2 delete_set_values

Удаляем все demography_multiplier_values для указанного набора корректировок.

=cut
sub delete_set_values {
    my ($set) = @_;
    my $deleted = get_all_sql(PPC(cid => $set->{cid}), [
        "select demography_multiplier_value_id, age, gender, multiplier_pct from demography_multiplier_values",
        where => {hierarchical_multiplier_id => $set->{hierarchical_multiplier_id}}
    ]);
    do_delete_from_table(PPC(cid => $set->{cid}), 'demography_multiplier_values', where => {demography_multiplier_value_id => [map { $_->{demography_multiplier_value_id} } @$deleted]});
    return $deleted;
}


1;
