package HierarchicalMultipliers::Expression;
use Direct::Modern;

use JSON qw/from_json to_json/;
use List::Util qw/min max/;

use Yandex::DBTools;
use Yandex::HashUtils;

use LogTools;
use Settings;
use HierarchicalMultipliers::Base;

BEGIN {
    for my $type (qw/
        express_traffic_multiplier
        prisma_income_grade_multiplier
    /) {
        HierarchicalMultipliers::Base::register_type($type, {
            load       => \&load,
            calc_stats => \&calc_stats,
            prepare_for_copy      => sub { return undef; },
            delete_camp_values => \&delete_camp_values,
            delete_camp_group_values => \&delete_camp_group_values,
            multiplier_set_can_be_disabled => 1,

            # эти методы ничего не делают, а только пишут ошибки в лог.
            insert     => \&insert,
            update     => \&update,
            delete     => \&delete,
        }, expression => 1);
    }
}

=head2 load

    Чтение набора универсальных корректировок.

=cut

sub load {
    my ($hierarchical_multiplier) = @_;
    my $adjustments = get_all_sql(
        PPC(cid => $hierarchical_multiplier->{cid}),
        [
            "select expression_multiplier_value_id, condition_json AS `condition`, multiplier_pct from expression_multiplier_values",
            where => { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id} },
        ]
    );

    my $result = hash_cut $hierarchical_multiplier, qw/is_enabled last_change hierarchical_multiplier_id/;
    foreach my $adj (@$adjustments) {
        $adj->{condition} = from_json($adj->{condition});
    }
    $result->{adjustments} = $adjustments;
    return $result;
}

=head2 calc_stats

    Вычисляет часть статистики, используемой для формирования сводной в HierarchicalMultipliers::calc_stats

=cut
sub calc_stats {
    my ($multiplier) = @_;

    my @values = map {$_->{multiplier_pct}} @{$multiplier->{adjustments}};
    my $result = {
        multiplier_pct_min => min(@values),
        multiplier_pct_max => max(@values),
        values_count       => scalar @{$multiplier->{adjustments}},
    };
    my $lower_bound = min(grep {$_ != 100} @values);
    my $upper_bound = max(grep {$_ > 100} @values);
    $result->{adjustments_lower_bound} = $lower_bound if defined $lower_bound;
    $result->{adjustments_upper_bound} = $upper_bound if defined $upper_bound;
    return $result;
}

=head2 delete_camp_values

    Удаление всех универсальных корректировок определённого типа из кампании и её групп.

=cut

sub delete_camp_values {
    my ($cid, $type) = @_;
    do_sql(PPC(cid => $cid), [
            "delete h, r from hierarchical_multipliers h left join expression_multiplier_values r using(hierarchical_multiplier_id)",
            where => { 'h.cid' => $cid, 'h.type' => $type },
        ]);
}

=head2 delete_camp_group_values

    Удаление всех универсальных корректировок определённого из группы.

=cut

sub delete_camp_group_values {
    my ($cid, $pids, $type) = @_;
    do_sql(PPC(cid => $cid), [
            "delete h, r from hierarchical_multipliers h left join expression_multiplier_values r using(hierarchical_multiplier_id)",
            where => { 'h.cid' => $cid, 'h.pid' => $pids, 'h.type' => $type },
        ]);
}

# вот эти три функции вызываются из save_hierarchical_multipliers, которая не поддерживается для
# универсальных корректировок

=head2 insert
=cut
sub insert {
    _report_unsupported_call('insert');
    return undef;
}

=head2 update
=cut
sub update {
    _report_unsupported_call('update');
    return undef;
}

=head2 delete
=cut
sub delete {
    _report_unsupported_call('delete');
    return undef;
}

sub _report_unsupported_call {
    my ($op) = @_;
    LogTools::log_messages('expression_multiplier_error', "called unsupported operation $op");
}

1;
