package HierarchicalMultipliers::Geo;
use Direct::Modern;

use List::Util qw/min max/;
use List::MoreUtils qw/any part/;
use GeoTools;
use Yandex::DBTools;

use Settings;
use Yandex::HashUtils qw//;

use geo_regions;

use PrimitivesIds qw/get_clientid/;

use Direct::Validation::HierarchicalMultipliers qw/
    validate_geo_multiplier
    validate_geo_multiplier_regions
    /;
use HierarchicalMultipliers::Base qw/register_type/;
use JavaIntapi::GenerateObjectIds;

BEGIN {
    register_type(geo_multiplier => {
        insert                => \&insert,
        update                => \&update,
        delete                => \&delete_set_values,
        load                  => \&load,
        prepare_for_copy      => sub { return undef; },
        calc_stats            => \&calc_stats,
        delete_camp_values    => \&delete_camp_values,
        delete_camp_group_values       => \&delete_camp_group_values,
        multiplier_set_can_be_disabled => 1,
    });
}

my %TRANSLOCAL_REGIONS = map { $_ => 1 } ($geo_regions::SNG, $geo_regions::UKR, $geo_regions::RUS, $geo_regions::KRIM);

sub _is_ukrainian_client {
    my ( $field, $field_value ) = @_;
   
    my $client_id = $field eq 'ClientID' ? $field_value : get_clientid(cid => $field_value);
    my $translocal_type = GeoTools::get_translocal_type({ClientID => $client_id});

    return $translocal_type eq 'ua' ? 1 : 0;
}

=head2 _find_translocal_regions
    Если в переданном наборе нет Крыма, и есть регионы, которые могут быть для него родительскими (Россия, Украина|СНГ)
    возвращает ссылку на хеш { region_id => значение корректировки, ...} по найденным потенциально родительским регионам.
    Если есть явно заданная корректировка для Крыма, либо ничего не найдено - возвращает undef

=cut

sub _find_translocal_regions {
    my ($regions) = @_;
    
    my %res = map {$_->{region_id} => $_->{multiplier_pct}} grep {$TRANSLOCAL_REGIONS{$_->{region_id}}} @$regions;
    
    return undef if $res{$geo_regions::KRIM};
    
    return %res ? \%res : undef;
}

sub _get_hidden_region {
    my ($cid, $regions) = @_;
    
    my $t_regions = _find_translocal_regions($regions);
    
    my $hidden_region;
    if ($t_regions) {
        my $super_region = _is_ukrainian_client(cid => $cid) ? $geo_regions::UKR : $geo_regions::RUS;
        #Для украинского клиента, если не задана корректировка для Украины, Крым может наследовать корректировку для СНГ
        $super_region = $geo_regions::SNG if $super_region == $geo_regions::UKR && !defined $t_regions->{$geo_regions::UKR};
        $hidden_region = {
            region_id   => $geo_regions::KRIM,
            is_hidden   => 1,
            multiplier_pct => $t_regions->{$super_region},
        } if defined $t_regions->{$super_region};
    }
    
    return $hidden_region;
}

=head2 insert

В соответствии с описанным в HierarchicalMultipliers API для добавления новых коэффициентов.

Вставляет запись в 'hierarchical_multipliers'.
Вставляет записи в 'geo_multiplier_values'.

=cut
sub insert {
    my ($data, $proposed_hierarchical_multiplier) = @_;

    $proposed_hierarchical_multiplier->{is_enabled} = $data->{is_enabled} ? 1 : 0;

    my $regions = $data->{regions};
    die 'Geo regions should be a array ref' unless ref($regions) eq 'ARRAY';
    die 'Geo regions should be not empty' unless @$regions;
    
    my $extra_region = _get_hidden_region($proposed_hierarchical_multiplier->{cid}, $regions);
    
    my $ids = JavaIntapi::GenerateObjectIds->new(
            object_type => 'multiplier',
            count => (scalar @$regions + ($extra_region ? 1 :0)))->call();
    my @insert_data;

    my %t_regions_idx = map {};
    foreach my $reg_data (@$regions, $extra_region){
        next unless defined $reg_data;
        push @insert_data, {
                geo_multiplier_value_id    => shift(@$ids),
                hierarchical_multiplier_id => $proposed_hierarchical_multiplier->{hierarchical_multiplier_id},
                region_id                  => $reg_data->{region_id},
                multiplier_pct             => $reg_data->{multiplier_pct},
                is_hidden                  => $reg_data->{is_hidden} // 0,
            }
    }
    do_in_transaction {
        do_insert_into_table(PPC(cid => $proposed_hierarchical_multiplier->{cid}), "hierarchical_multipliers", $proposed_hierarchical_multiplier);
        do_mass_insert_sql(PPC(cid => $proposed_hierarchical_multiplier->{cid}), 'insert into geo_multiplier_values'
                           . '(geo_multiplier_value_id, hierarchical_multiplier_id, region_id, multiplier_pct, is_hidden)'
                           . ' values %s',
                       [map { [@{$_}{qw/geo_multiplier_value_id hierarchical_multiplier_id region_id multiplier_pct is_hidden/}] } @insert_data]);
    };
    
    @insert_data = grep {!$_->{is_hidden}} @insert_data if $extra_region;
    
    return @insert_data ? \@insert_data : undef;
}

=head2 load

В соответствии с описанным в HierarchicalMultipliers API для добавления новых коэффициентов.

Возвращает:
    {
        is_enabled => XXX,
        hierarchical_multiplier_id => XXX,
        last_change => XXX,
        regions => [
                    {
                        geo_multiplier_value_id => XXX1,
                        region_id => RRR1,
                        multiplier_pct => NNN1,
                    },
                    {
                        geo_multiplier_value_id => XXX2,
                        region_id => RRR2,
                        ...
                    }...
                ],
    }

=cut

sub load {
    my ($hierarchical_multiplier, %options) = @_;
    
    my $regions = get_all_sql(
        PPC(cid => $hierarchical_multiplier->{cid}),
        [
            'SELECT multiplier_pct, region_id, geo_multiplier_value_id FROM geo_multiplier_values',
            WHERE => {
                hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id},
                is_hidden => 0,
            },
        ]
    );

    my $result = Yandex::HashUtils::hash_cut($hierarchical_multiplier, qw/is_enabled last_change hierarchical_multiplier_id/);
    $result->{regions} = $regions;
    
    return $result;
}

=head2 update

В соответствии с описанным в HierarchicalMultipliers API для добавления новых коэффициентов.

Обновляет hierarchical_multipliers.is_enabled (если надо).
Обновляет/заменяет записи в geo_multiplier_values.

=cut
sub update {
    my ($data, $hierarchical_multiplier) = @_;
    
    my $existing_values = get_all_sql(
        PPC(cid => $hierarchical_multiplier->{cid}), [
            'SELECT geo_multiplier_value_id, region_id, multiplier_pct, is_hidden FROM geo_multiplier_values',
            WHERE => { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id} },
            'FOR UPDATE'
        ]
    );

    my %existing_values_map;

    for my $value (@$existing_values) {
        $existing_values_map{$value->{region_id}} = $value;
    }
    
    my (%update_values, @insert_data);
    my %log_data = (inserted => [], updated => [], deleted => []);
    
    my $extra_region = _get_hidden_region($hierarchical_multiplier->{cid}, $data->{regions});
        
    for my $value (@{$data->{regions}}) {
        my $region_id = $value->{region_id};
        #Скрытую корректировку обновим отдельно, т.к. если для Крыма есть корректировка во входных данных -
        #значит она задана явно и скрытую корректровку нужно удалить
        if (exists $existing_values_map{$region_id} && !$existing_values_map{$region_id}->{is_hidden} ) {
            my $old_value = delete $existing_values_map{$region_id};
            if ($old_value->{multiplier_pct} != $value->{multiplier_pct}) {
                $update_values{$old_value->{geo_multiplier_value_id}} = $value->{multiplier_pct};
                push @{$log_data{updated}}, {
                    geo_multiplier_value_id => $old_value->{geo_multiplier_value_id},
                    region_id => $region_id,
                    new_multiplier_pct => $value->{multiplier_pct}, old_multiplier_pct => $old_value->{multiplier_pct},
                } unless $value->{is_hidden};
            }
        } else {
            push @insert_data, {
                hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id},
                region_id => $region_id,
                multiplier_pct => $value->{multiplier_pct},
            };
        }
    }
    
    if ($extra_region) {
        my $old = $existing_values_map{$extra_region->{region_id}};
        #Если есть скрытая корректировка - обновим ее, если была не скрытая для Крыма - добавим скрытую,
        #а явная удалится ниже по коду
        if ($old && $old->{is_hidden}) {
            $update_values{$old->{geo_multiplier_value_id}} = $extra_region->{multiplier_pct};
            delete $existing_values_map{$extra_region->{region_id}}
        }
        else {
             push @insert_data, {
                hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id},
                %$extra_region,
            };
        }
        
    }
    
    #Оставшиеся записи удаляем
    my @delete_values = values %existing_values_map;
    if (@delete_values) {
        do_delete_from_table(PPC(cid => $hierarchical_multiplier->{cid}), 'geo_multiplier_values',
            where => {geo_multiplier_value_id => [map { $_->{geo_multiplier_value_id} } @delete_values]});

        for my $value (@delete_values) {
            next if $value->{is_hidden};
            push $log_data{deleted}, {
                geo_multiplier_value_id => $value->{geo_multiplier_value_id},
                region_id => $value->{region_id},
                old_multiplier_pct => $value->{multiplier_pct},
            };
        }
    }
    
    #Добавляем новые записи
    if (@insert_data) {
        my $ids = JavaIntapi::GenerateObjectIds->new(object_type => 'multiplier',
                count => scalar @insert_data)->call();
        for my $value (@insert_data) {
            $value->{geo_multiplier_value_id} = shift(@$ids);
            push $log_data{inserted}, {
                geo_multiplier_value_id => $value->{geo_multiplier_value_id},
                region_id => $value->{region_id},
                new_multiplier_pct => $value->{multiplier_pct},
            } unless $value->{is_hidden};
        }
        do_mass_insert_sql(PPC(cid => $hierarchical_multiplier->{cid}), 'INSERT INTO geo_multiplier_values'
                               . '(geo_multiplier_value_id, hierarchical_multiplier_id, region_id, multiplier_pct)'
                               . ' VALUES %s',
                           [map { [@{$_}{qw/geo_multiplier_value_id hierarchical_multiplier_id region_id multiplier_pct/}] } @insert_data]
        );
    }
    
    #Обновляем существующие
    if (%update_values) {
        do_update_table(PPC(cid => $hierarchical_multiplier->{cid}), 'geo_multiplier_values',
                        {
                            multiplier_pct__dont_quote => sql_case('geo_multiplier_value_id', \%update_values),
                            last_change__dont_quote => 'NOW()',
                        },
                        where => { geo_multiplier_value_id => [keys %update_values]});
    }

    #Обновляем last_change у изменившихся сетов, удаляем пустые
    if (@{$data->{regions}}) {
        my $new_is_enabled = $data->{is_enabled} ? 1 : 0;
        my $is_enabled_has_changed = $hierarchical_multiplier->{is_enabled} != $new_is_enabled;
        my $need_update_last_change_for_whole_set = %update_values || @insert_data || @delete_values;
        if ($is_enabled_has_changed) {
            $log_data{is_enabled_change} = { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id}, is_enabled => $new_is_enabled };
        }
        if ($need_update_last_change_for_whole_set || $is_enabled_has_changed) {
            do_update_table(PPC(cid => $hierarchical_multiplier->{cid}), 'hierarchical_multipliers',
                            {
                                is_enabled => $new_is_enabled,
                                last_change__dont_quote => 'NOW()',
                            },
                            where => {hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id}}
            );
        }
    } else {
        do_delete_from_table(PPC(cid => $hierarchical_multiplier->{cid}), 'hierarchical_multipliers',
                             where => {hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id}});
        $log_data{deleted_set} = { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id} };
    }
    
    return unless %log_data;
    return \%log_data;
}

=head2 calc_stats

Вычисляет часть статистики, используемой для формирования сводной в HierarchicalMultipliers::calc_stats

=cut
sub calc_stats {
    my ($multiplier, $opts) = @_;
    
    my $translocal_opts = $opts->{ClientID} ? {ClientID => $opts->{ClientID}} : {tree => 'api'};
    my $geo = $opts->{group} ? $opts->{group}->{geo} // '0' : '0';
    
    my ($plus_regions, $minus_regions) = part { $_ >= 0 ? 0 : 1} split /\s*,\s*/, $geo;
    my $tree = GeoTools::get_translocal_georeg($translocal_opts);
    my @regions = sort {$tree->{$b}->{level} <=> $tree->{$a}->{level}} map {$_->{region_id}} @{$multiplier->{regions}};
    
    my %affected_regions;
    my %parents_idx;
    foreach my $group_region (@{$plus_regions // []}){
        foreach my $reg_id (@regions){
            my $reg_parents = $parents_idx{$reg_id} //= { map {$_ => 1} @{$tree->{$reg_id}->{parents}} };
            #Если регион, для которого задана корректировка покрывается регионом группы
            if (exists $reg_parents->{$group_region}) {
                #Оставляем только минус-регионы, покрывающие регион, для которого задана корректировка
                my @reg_minus = grep {exists $reg_parents->{abs($_)} || abs($_) == $reg_id} @{$minus_regions // []};

                #Если такие есть - ищем хотя бы один, лежащий не выше региона группы 
                my $reg_disabled = 0;
                foreach my $m_reg (@reg_minus){
                    next if any {abs($m_reg) == $_} (@{$tree->{$group_region}->{parents}});
                    $reg_disabled = 1;
                    last;
                }
                #Если таких минус-регионов нет - корректировка будет действовать
                $affected_regions{$reg_id} //= 1 unless $reg_disabled;
            } elsif (any {$reg_id == $_} ($group_region, @{$tree->{$group_region}->{parents}})) {
                #Если регион, для которого задана корректировка, покрывает регион группы или равен ему
                #используем корректировку и заканчиваем цикл.
                #Т.к. мы идем от регионов более низкого уровня к регионам более высокого -
                #первая найденная корректировка пришедшая "сверху" будет самой узкой
                $affected_regions{$reg_id} //= 1;
                last;
            }
        }
    }
 
    my @values = map {$_->{multiplier_pct}} grep { exists $affected_regions{$_->{region_id}} } @{$multiplier->{regions}};
    my $result = {
        multiplier_pct_min => min(@values),
        multiplier_pct_max => max(@values),
        values_count       => scalar @{$multiplier->{regions}},
    };
    my $lower_bound = min(grep { $_ != 100 } @values);
    my $upper_bound = max(grep { $_ > 100 } @values);
    $result->{adjustments_lower_bound} = $lower_bound if defined $lower_bound;
    $result->{adjustments_upper_bound} = $upper_bound if defined $upper_bound;
    return $result;
}

=head2 delete_camp_values

Удаляем все связанные с кампанией и её группами корректировки.

=cut
sub delete_camp_values {
    my ($cid) = @_;
    do_sql(PPC(cid => $cid), [
        'DELETE h, g FROM hierarchical_multipliers h JOIN geo_multiplier_values g ON (g.hierarchical_multiplier_id = h.hierarchical_multiplier_id )',
        WHERE => { 'h.cid' => $cid, 'h.type' => 'geo_multiplier' },
    ]);
}

=head2 delete_camp_group_values

Удаляем все связанные с кампанией и переданными группами корректировки.

=cut

sub delete_camp_group_values {
   my ($cid, $pids) = @_;
    do_sql(PPC(cid => $cid), [
        'DELETE h, g FROM hierarchical_multipliers h JOIN geo_multiplier_values g ON (g.hierarchical_multiplier_id = h.hierarchical_multiplier_id )',
        WHERE => { 'h.cid' => $cid, 'h.pid' => $pids, 'h.type' => 'geo_multiplier' },
    ]);
}

=head2 delete_set_values

Удаляем geo_multiplier_values для указанного набора корректировок.

=cut
sub delete_set_values {
    my ($set) = @_;
    my $deleted = get_all_sql(PPC(cid => $set->{cid}), [
        'SELECT geo_multiplier_value_id, region_id, multiplier_pct, is_hidden FROM geo_multiplier_values',
        WHERE => {hierarchical_multiplier_id => $set->{hierarchical_multiplier_id}}
    ]);
    do_delete_from_table(PPC(cid => $set->{cid}), 'geo_multiplier_values', where => {geo_multiplier_value_id => [map { $_->{geo_multiplier_value_id} } @$deleted]});
    return [grep {!$_->{is_hidden}} @$deleted];
}

1;
