package HierarchicalMultipliers::Inventory;
use Direct::Modern;

use List::Util qw/min max/;
use Yandex::DBTools;
use Yandex::HashUtils;

use Settings;
use HierarchicalMultipliers::Base qw/register_type/;

BEGIN {
    register_type(inventory_multiplier => {
        load                  => \&load,
        calc_stats            => \&calc_stats,
    });
}

=head2 load

В соответствии с описанным в HierarchicalMultipliers API для добавления новых коэффициентов.

Возвращает:
    {
        is_enabled => XXX,
        hierarchical_multiplier_id => XXX,
        last_change => XXX,
        conditions => [
            {
                multiplier_type => 'inpage',
                multiplier_pct => XXX,
            },
            ...
        ]
    }

=cut

sub load {
    my ($hierarchical_multiplier) = @_;
    my $conditions = get_all_sql(
        PPC(cid => $hierarchical_multiplier->{cid}),
        [
            "select inventory_type AS multiplier_type, multiplier_pct from inventory_multiplier_values",
            where => {hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id}},
        ]
    );

    my $result = hash_cut $hierarchical_multiplier, qw/is_enabled last_change hierarchical_multiplier_id/;
    $result->{conditions} = $conditions;
    return $result;
}


=head2 calc_stats

Вычисляет часть статистики, используемой для формирования сводной в HierarchicalMultipliers::calc_stats

=cut
sub calc_stats {
    my ($multiplier, $opts) = @_;

    return {values_count => 0} if $opts && $opts->{group}->{adgroup_type} ne 'cpm_video';

    my @values = map { $_->{multiplier_pct} } @{$multiplier->{conditions}};
    my $result = {
        multiplier_pct_min => min(@values),
        multiplier_pct_max => max(@values),
        values_count       => scalar @{$multiplier->{conditions}},
    };
    my $lower_bound = min(grep { $_ != 100 } @values);
    my $upper_bound = max(grep { $_ > 100 } @values);
    $result->{adjustments_lower_bound} = $lower_bound if defined $lower_bound;
    $result->{adjustments_upper_bound} = $upper_bound if defined $upper_bound;
    return $result;
}

1;
