package HierarchicalMultipliers::Mobile;

use Direct::Modern;

use List::Util qw/min max/;
use Yandex::HashUtils;
use Yandex::DBTools;
use Yandex::DBShards;

use Settings;

use HierarchicalMultipliers::Base qw/register_type/;
use HierarchicalMultipliers::MultiplierPct;
use Direct::Validation::HierarchicalMultipliers qw/normalize_os_type validate_mobile_multiplier/;

BEGIN {
    register_type(
        mobile_multiplier => {
            insert                         => \&insert,
            update                         => \&update,
            delete                         => \&delete_set_values,
            load                           => \&load,
            prepare_for_copy               => sub {return undef;},
            calc_stats                     => \&calc_stats,
            delete_camp_values             => \&delete_camp_values,
            delete_camp_group_values       => \&delete_camp_group_values,
            multiplier_set_can_be_disabled => 0,
        },
    );
}

=head2 insert

В соответствии с описанным в HierarchicalMultipliers API для добавления новых коэффициентов.

Вставляет запись в 'hierarchical_multipliers'.
Вставляет записи в 'mobile_multiplier_values'.

=cut

sub insert {
    my ($data, $proposed_hierarchical_multiplier) = @_;
    return unless defined $data->{multiplier_pct};
    my $has_os_type = defined $data->{os_type};
    $proposed_hierarchical_multiplier->{is_enabled} = 1;
    if (!$has_os_type) {
        $proposed_hierarchical_multiplier->{multiplier_pct} = $data->{multiplier_pct};
    } else {
        $proposed_hierarchical_multiplier->{multiplier_pct} = undef;
    }
    do_insert_into_table(PPC(cid => $proposed_hierarchical_multiplier->{cid}), "hierarchical_multipliers",
        $proposed_hierarchical_multiplier);
    if (!$has_os_type) {
        return {
            id                 => $proposed_hierarchical_multiplier->{hierarchical_multiplier_id},
            new_multiplier_pct => $data->{multiplier_pct},
        };
    }

    my $os_type = normalize_os_type($data->{os_type});
    die "Mobile 'os_type' should be correct. Incorrect value = ".$data->{os_type} unless $os_type;
    my $insert_data = {
        mobile_multiplier_value_id => get_new_id('hierarchical_multiplier_id'),
        hierarchical_multiplier_id => $proposed_hierarchical_multiplier->{hierarchical_multiplier_id},
        os_type                    => $os_type,
        multiplier_pct             => $data->{multiplier_pct},
    };

    do_insert_into_table(PPC(cid => $proposed_hierarchical_multiplier->{cid}), "mobile_multiplier_values",
        $insert_data);
    return $insert_data;
}

=head2 update

В соответствии с описанным в HierarchicalMultipliers API для добавления новых коэффициентов.

Обновляет коэффициент цены для мобильных в табличке 'hierarchical_multipliers'.
Обновляет/заменяет/удаляет записи в mobile_multiplier_values.

=cut

sub update {
    my ($data, $hierarchical_multiplier) = @_;

    my $existing_values = get_all_sql(
        PPC(cid => $hierarchical_multiplier->{cid}), [
            "select mobile_multiplier_value_id, os_type, multiplier_pct from mobile_multiplier_values",
            where => { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id} },
            "for update"
        ]
    );

    die "Inconsistent state - more than one mobile_multiplier_values" if scalar @$existing_values > 1;

    my $existing_value = @$existing_values[0];

    my %log_data;


    # Если корректировка удаляется

    if (!defined $data->{multiplier_pct}) {
        if ($existing_value) {
            do_delete_from_table(PPC(cid => $hierarchical_multiplier->{cid}), "mobile_multiplier_values",
                where => { mobile_multiplier_value_id => $existing_value->{mobile_multiplier_value_id} });
            $log_data{deleted} = {
                mobile_multiplier_value_id => $existing_value->{mobile_multiplier_value_id},
                os_type                    => $existing_value->{os_type},
                old_multiplier_pct         => $existing_value->{multiplier_pct},
            };
        }
        do_delete_from_table(PPC(cid => $hierarchical_multiplier->{cid}), 'hierarchical_multipliers',
            where => { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id} });
        $log_data{deleted_set} = { hierarchical_multiplier_id =>
            $hierarchical_multiplier->{hierarchical_multiplier_id} };

        return \%log_data;
    }

    # Если применяется корректировка для всех типов ОС

    if (!defined $data->{os_type}) {
        if ($existing_value) {
            do_delete_from_table(PPC(cid => $hierarchical_multiplier->{cid}), "mobile_multiplier_values",
                where => { mobile_multiplier_value_id => $existing_value->{mobile_multiplier_value_id} });
            do_delete_from_table(PPC(cid => $hierarchical_multiplier->{cid}), "hierarchical_multipliers",
                where => { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id} });
            $log_data{deleted} = {
                mobile_multiplier_value_id => $existing_value->{mobile_multiplier_value_id},
                os_type                    => $existing_value->{os_type},
                old_multiplier_pct         => $existing_value->{multiplier_pct},
                hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id},
            };
            $hierarchical_multiplier->{hierarchical_multiplier_id} = get_new_id('hierarchical_multiplier_id');
            $hierarchical_multiplier->{multiplier_pct} = $data->{multiplier_pct};
            do_insert_into_table(PPC(cid => $hierarchical_multiplier->{cid}), "hierarchical_multipliers", $hierarchical_multiplier);
            $log_data{inserted} = $hierarchical_multiplier;
            return \%log_data;
        }

        if (!$existing_value && $data->{multiplier_pct} != $hierarchical_multiplier->{multiplier_pct}) {
            do_update_table(PPC(cid => $hierarchical_multiplier->{cid}), 'hierarchical_multipliers',
                {
                    multiplier_pct          => $data->{multiplier_pct},
                    last_change__dont_quote => 'NOW()',
                },
                where => { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id} });
            $log_data{updated_set} = {
                hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id},
                new_multiplier_pct         => $data->{multiplier_pct},
                old_multiplier_pct         => $hierarchical_multiplier->{multiplier_pct},
            };
            return \%log_data;
        }
        return;
    }

    # Иначе - применяется корректировка для конкретной ОС

    my $os_type = normalize_os_type($data->{os_type});
    die "Mobile 'os_type' should be correct. Incorrect value = ".$data->{os_type} unless $os_type;

    if ($existing_value && $os_type eq $existing_value->{os_type}
        && $data->{multiplier_pct} == $existing_value->{multiplier_pct}) {
        return;
    }

    if ($existing_value && $os_type eq $existing_value->{os_type}
        && $data->{multiplier_pct} != $existing_value->{multiplier_pct}) {
        do_update_table(PPC(cid => $hierarchical_multiplier->{cid}), 'mobile_multiplier_values',
            {
                multiplier_pct          => $data->{multiplier_pct},
                last_change__dont_quote => 'NOW()',
            },
            where => { mobile_multiplier_value_id => $existing_value->{mobile_multiplier_value_id} });
        $log_data{updated} = {
            mobile_multiplier_value_id => $existing_value->{mobile_multiplier_value_id},
            os_type                    => $os_type,
            new_multiplier_pct         => $data->{multiplier_pct},
            old_multiplier_pct         => $existing_value->{multiplier_pct},
        };
        do_update_table(PPC(cid => $hierarchical_multiplier->{cid}), 'hierarchical_multipliers',
            {
                last_change__dont_quote => 'NOW()',
            },
            where => { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id} });
        return \%log_data;
    }

    if ($existing_value){
        do_delete_from_table(PPC(cid => $hierarchical_multiplier->{cid}), "mobile_multiplier_values",
            where => { mobile_multiplier_value_id => $existing_value->{mobile_multiplier_value_id} });
    }
    do_delete_from_table(PPC(cid => $hierarchical_multiplier->{cid}), "hierarchical_multipliers",
        where => { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id} });

    $hierarchical_multiplier->{hierarchical_multiplier_id} = get_new_id('hierarchical_multiplier_id');
    $hierarchical_multiplier->{multiplier_pct} = undef;
    do_insert_into_table(PPC(cid => $hierarchical_multiplier->{cid}), "hierarchical_multipliers", $hierarchical_multiplier);

    my $insert_data = {
        mobile_multiplier_value_id => get_new_id('hierarchical_multiplier_id'),
        hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id},
        os_type                    => $os_type,
        multiplier_pct             => $data->{multiplier_pct},
    };
    do_insert_into_table(PPC(cid => $hierarchical_multiplier->{cid}), "mobile_multiplier_values", $insert_data);
    $log_data{inserted} = $insert_data;
    return \%log_data;
}

=head2 load

В соответствии с описанным в HierarchicalMultipliers API для добавления новых коэффициентов.

Возврщает:
   {
      hierarchical_multiplier_id => XXX,
      last_change => XXX,
      multiplier_pct => XXX,
      os_type => XXX,
      mobile_multiplier_value_id => XXX,
   }

=cut

sub load {
    my ($hierarchical_multiplier) = @_;
    if (defined $hierarchical_multiplier->{multiplier_pct}) {
        return hash_cut $hierarchical_multiplier, qw/hierarchical_multiplier_id last_change multiplier_pct/;
    }

    my $existing_values = get_all_sql(
        PPC(cid => $hierarchical_multiplier->{cid}), [
            "select mobile_multiplier_value_id, os_type, multiplier_pct from mobile_multiplier_values",
            where => { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id} },
        ]
    );

    die "Inconsistent state - more than one mobile_multiplier_values" if scalar @$existing_values > 1;

    my $existing_value = @$existing_values[0];

    my $result = hash_merge ($existing_value,
        hash_cut $hierarchical_multiplier, qw/last_change hierarchical_multiplier_id/);
    return $result;
}

=head2 calc_stats

=cut

sub calc_stats {
    return HierarchicalMultipliers::MultiplierPct::calc_stats(@_);
}

=head2 delete_set_values

Удаляем все mobile_multiplier_values для указанного набора корректировок.

=cut

sub delete_set_values {
    my ($set) = @_;
    my $deleted = get_all_sql(PPC(cid => $set->{cid}), [
            "select mobile_multiplier_value_id, os_type, multiplier_pct from mobile_multiplier_values",
            where => { hierarchical_multiplier_id => $set->{hierarchical_multiplier_id} }
        ]);
    do_delete_from_table(PPC(cid => $set->{cid}), 'mobile_multiplier_values', where =>
        { mobile_multiplier_value_id => [ map {$_->{mobile_multiplier_value_id}} @$deleted ] });
    return $deleted;
}

=head2 delete_camp_values

Удаляем все связанные с кампанией и её группами корректировки.

=cut

sub delete_camp_values {
    my ($cid) = @_;
    do_sql(PPC(cid => $cid), [
            "delete h, r from hierarchical_multipliers h left join mobile_multiplier_values r using(hierarchical_multiplier_id)",
            where => { 'h.cid' => $cid, 'h.type' => 'mobile_multiplier' },
        ]);
}

=head2 delete_camp_group_values

Удаляем все связанные с кампанией и переданными группами корректировки.

=cut

sub delete_camp_group_values {
    my ($cid, $pids) = @_;
    do_sql(PPC(cid => $cid), [
            "delete h, r from hierarchical_multipliers h left join mobile_multiplier_values r using(hierarchical_multiplier_id)",
            where => { 'h.cid' => $cid, 'h.pid' => $pids, 'h.type' => 'mobile_multiplier' },
        ]);
}

1;
