package HierarchicalMultipliers::Retargeting;
use Direct::Modern;

use Yandex::DBTools;
use Yandex::HashUtils;
use List::Util qw/min max/;

use Settings;
use Direct::Validation::HierarchicalMultipliers qw/validate_retargeting_multiplier/;
use HierarchicalMultipliers::Base qw/register_type/;
use Retargeting qw//;
use Direct::Errors::Messages;
use JavaIntapi::GenerateObjectIds;

BEGIN {
    for my $type (qw/
        retargeting_multiplier
        retargeting_filter
    /) {
        register_type($type, {
            insert                => \&insert,
            update                => \&update,
            delete                => \&delete_set_values,
            load                  => \&load,
            prepare_for_copy      => \&prepare_for_copy,
            calc_stats            => \&calc_stats,
            delete_camp_values    => \&delete_camp_values,
            delete_camp_group_values       => \&delete_camp_group_values,
            multiplier_set_can_be_disabled => 1,
        });
    }
}

=head2 insert

Вставляет запись в 'hierarchical_multipliers'.
Вставляет записи в 'retargeting_multiplier_values'.

=cut
sub insert {
    my ($data, $proposed_hierarchical_multiplier) = @_;
    $proposed_hierarchical_multiplier->{is_enabled} = $data->{is_enabled} ? 1 : 0;

    do_insert_into_table(PPC(cid => $proposed_hierarchical_multiplier->{cid}), "hierarchical_multipliers", $proposed_hierarchical_multiplier);

    my $ids = JavaIntapi::GenerateObjectIds->new( object_type => 'multiplier',
            count => scalar keys %{$data->{conditions}})->call();
    my @insert_data;
    while (my ($ret_cond_id, $value) = each %{$data->{conditions}}) {
        push @insert_data, {
            retargeting_multiplier_value_id => shift(@$ids),
            hierarchical_multiplier_id => $proposed_hierarchical_multiplier->{hierarchical_multiplier_id},
            multiplier_pct => $value->{multiplier_pct},
            ret_cond_id => $ret_cond_id,
        };
    }
    do_mass_insert_sql(PPC(cid => $proposed_hierarchical_multiplier->{cid}), 'insert into retargeting_multiplier_values'
                           . '(retargeting_multiplier_value_id, hierarchical_multiplier_id, ret_cond_id, multiplier_pct)'
                           . ' values %s',
                       [map { [@{$_}{qw(retargeting_multiplier_value_id hierarchical_multiplier_id ret_cond_id multiplier_pct)}] } @insert_data]);
    return @insert_data ? \@insert_data : undef;
}

=head2 update

Обновляет hierarchical_multipliers.is_enabled (если надо).
Обновляет/заменяет записи в retargeting_multiplier_values.

=cut
sub update {
    my ($data, $hierarchical_multiplier) = @_;

    my $existing_values = get_hashes_hash_sql(
        PPC(cid => $hierarchical_multiplier->{cid}), [
            "select ret_cond_id, retargeting_multiplier_value_id, multiplier_pct from retargeting_multiplier_values",
            where => { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id} },
            "for update"
        ]
    );

    my (%update_values, @insert_data, %log_data);
    while (my($ret_cond_id, $value) = each %{$data->{conditions}}) {
        if (exists $existing_values->{$ret_cond_id}) {
            my $old_value = delete $existing_values->{$ret_cond_id};
            if ($old_value->{multiplier_pct} != $value->{multiplier_pct}) {
                $update_values{$old_value->{retargeting_multiplier_value_id}} = $value->{multiplier_pct};
                push @{$log_data{updated}}, {
                    retargeting_multiplier_value_id => $old_value->{retargeting_multiplier_value_id},
                    ret_cond_id => $ret_cond_id,
                    new_multiplier_pct => $value->{multiplier_pct}, old_multiplier_pct => $old_value->{multiplier_pct},
                };
            }
        } else {
            push @insert_data, {
                hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id},
                ret_cond_id => $ret_cond_id,
                multiplier_pct => $value->{multiplier_pct},
            };
        }
    }
    my @delete_values = values %$existing_values;
    if (@delete_values) {
        do_delete_from_table(PPC(cid => $hierarchical_multiplier->{cid}), "retargeting_multiplier_values",
                             where => {retargeting_multiplier_value_id => [map { $_->{retargeting_multiplier_value_id} } @delete_values]});
        for my $value (@delete_values) {
            push @{$log_data{deleted}}, {
                retargeting_multiplier_value_id => $value->{retargeting_multiplier_value_id},
                ret_cond_id => $value->{ret_cond_id},
                old_multiplier_pct => $value->{multiplier_pct},
            };
        }
    }
    if (@insert_data) {
        my $ids = JavaIntapi::GenerateObjectIds->new(object_type => 'multiplier',
                count => scalar @insert_data)->call();
        for my $value (@insert_data) {
            $value->{retargeting_multiplier_value_id} = shift(@$ids);
            push @{$log_data{inserted}}, {
                retargeting_multiplier_value_id => $value->{retargeting_multiplier_value_id},
                ret_cond_id => $value->{ret_cond_id},
                new_multiplier_pct => $value->{multiplier_pct},
            };
        }
        do_mass_insert_sql(PPC(cid => $hierarchical_multiplier->{cid}), 'insert into retargeting_multiplier_values'
                               . '(retargeting_multiplier_value_id, hierarchical_multiplier_id, ret_cond_id, multiplier_pct)'
                               . ' values %s',
                           [map { [@{$_}{qw(retargeting_multiplier_value_id hierarchical_multiplier_id ret_cond_id multiplier_pct)}] } @insert_data]);
    }
    if (%update_values) {
        do_update_table(PPC(cid => $hierarchical_multiplier->{cid}), "retargeting_multiplier_values",
                        {
                            multiplier_pct__dont_quote => sql_case('retargeting_multiplier_value_id', \%update_values),
                            last_change__dont_quote => 'NOW()',
                        },
                        where => { retargeting_multiplier_value_id => [keys %update_values]});
    }
    if (%{$data->{conditions}}) {
        my $new_is_enabled = $data->{is_enabled} ? 1 : 0;
        my $is_enabled_has_changed = $hierarchical_multiplier->{is_enabled} != $new_is_enabled;
        my $need_update_last_change_for_whole_set = %update_values || @insert_data || @delete_values;
        if ($is_enabled_has_changed) {
            $log_data{is_enabled_change} = { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id}, is_enabled => $new_is_enabled };
        }
        if ($need_update_last_change_for_whole_set || $is_enabled_has_changed) {
            do_update_table(PPC(cid => $hierarchical_multiplier->{cid}), 'hierarchical_multipliers',
                            {
                                is_enabled => $new_is_enabled,
                                last_change__dont_quote => 'NOW()',
                            },
                            where => {hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id}});
        }
    } else {
        do_delete_from_table(PPC(cid => $hierarchical_multiplier->{cid}), 'hierarchical_multipliers',
                             where => {hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id}});
        $log_data{deleted_set} = {hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id}};
    }
    return unless %log_data;
    return \%log_data;
}

=head2 load

Возвращает:
    {
        is_enabled => XXX,
        last_change => XXX,
        hierarchical_multiplier_id => XXX,
        conditions => {
            ret_cond_id_1 => {
                retargeting_multiplier_value_id => XXX,
                multiplier_pct => XXX,
                last_change => XXX,
            },
            ...
        },
    }

=cut
sub load {
    my ($hierarchical_multiplier, %opts) = @_;
    my $sql;
    if ($opts{heavy}) {
        $sql = [
            "select d.ret_cond_id
                  , d.multiplier_pct
                  , d.retargeting_multiplier_value_id
                  , d.last_change
                  , min(ifnull(g.is_accessible, '0')) as is_accessible
             from retargeting_multiplier_values d
             left join retargeting_goals g using (ret_cond_id)
             ",
            where => {hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id}},
            "group by d.retargeting_multiplier_value_id",
            ];
    } else {
        $sql = [
            "select ret_cond_id, multiplier_pct, retargeting_multiplier_value_id, last_change from retargeting_multiplier_values",
            where => {hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id}},
        ];
    }
    my $conditions = get_hashes_hash_sql(PPC(cid => $hierarchical_multiplier->{cid}), $sql);

    delete $_->{ret_cond_id} for values %$conditions;
    my $result = hash_cut $hierarchical_multiplier, qw/is_enabled last_change hierarchical_multiplier_id/;
    $result->{conditions} = $conditions;
    return $result;
}

=head2 prepare_for_copy

При необходимости создаёт новые условия ретаргетинга на новом ClientID и обновляет их идентификаторы в
переданной структуре данных.

=cut
sub prepare_for_copy {
    my ($old_client_id, $new_client_id, $retargeting_multiplier_data) = @_;
    return if $old_client_id == $new_client_id;
    my $old_to_new = Retargeting::copy_retargetings_between_clients(
        old_client_id => $old_client_id, new_client_id => $new_client_id,
        old_ret_cond_ids => [ keys %{$retargeting_multiplier_data->{conditions}}]
    );

    while (my ($old_ret_cond_id, $new_ret_cond_id) = each %$old_to_new) {
        $retargeting_multiplier_data->{conditions}{$new_ret_cond_id} = delete $retargeting_multiplier_data->{conditions}{$old_ret_cond_id};
    }
    return undef unless %$old_to_new;
    return {ret_cond_id_old_to_new => $old_to_new};
}

=head2 calc_stats

Вычисляет часть статистики, используемой для формирования сводной в HierarchicalMultipliers::calc_stats

=cut
sub calc_stats {
    my ($multiplier) = @_;
    my @values = map { $_->{multiplier_pct} } values %{$multiplier->{conditions}};
    my $result = {
        multiplier_pct_min => min(@values),
        multiplier_pct_max => max(@values),
        values_count       => scalar values %{$multiplier->{conditions}},
    };
    my $lower_bound = min(grep { $_ != 100 } @values);
    my $upper_bound = max(grep { $_ > 100 } @values);
    $result->{adjustments_lower_bound} = $lower_bound if defined $lower_bound;
    $result->{adjustments_upper_bound} = $upper_bound if defined $upper_bound;
    return $result;
}

=head2 delete_camp_values

Удаляем все связанные с кампанией и её группами корректировки.

=cut
sub delete_camp_values {
    my ($cid) = @_;
    do_sql(PPC(cid => $cid), [
        "delete h, r from hierarchical_multipliers h join retargeting_multiplier_values r using(hierarchical_multiplier_id)",
        where => { 'h.cid' => $cid, 'h.type' => 'retargeting_multiplier' },
    ]);
}

=head2 delete_camp_group_values

Удаляем все связанные с кампанией и переданными группами корректировки.

=cut

sub delete_camp_group_values {
   my ($cid, $pids) = @_;
    do_sql(PPC(cid => $cid), [
        "delete h, r from hierarchical_multipliers h join retargeting_multiplier_values r using(hierarchical_multiplier_id)",
        where => { 'h.cid' => $cid, 'h.pid' => $pids, 'h.type' => 'retargeting_multiplier' },
    ]);
}

=head2 delete_set_values

Удаляем все retargeting_multiplier_values для указанного набора корректировок.

=cut
sub delete_set_values {
    my ($set) = @_;
    my $deleted = get_all_sql(PPC(cid => $set->{cid}), [
        "select retargeting_multiplier_value_id, ret_cond_id, multiplier_pct from retargeting_multiplier_values",
        where => {hierarchical_multiplier_id => $set->{hierarchical_multiplier_id}}
    ]);
    do_delete_from_table(PPC(cid => $set->{cid}), 'retargeting_multiplier_values', where => {retargeting_multiplier_value_id => [map { $_->{retargeting_multiplier_value_id} } @$deleted]});
    return $deleted;
}

1;
