package HierarchicalMultipliers::Tablet;

use Direct::Modern;

use Yandex::HashUtils;
use Yandex::DBTools;

use Settings;

use HierarchicalMultipliers::Base qw/register_type/;
use HierarchicalMultipliers::MultiplierPct;

BEGIN {
    register_type(
        tablet_multiplier => {
            insert                => \&insert,
            update                => \&update,
            delete                => \&delete_set_values,
            load                  => \&load,
            prepare_for_copy      => sub { return undef; },
            calc_stats            => \&calc_stats,
            delete_camp_values    => \&delete_camp_values,
            delete_camp_group_values       => \&delete_camp_group_values,
            multiplier_set_can_be_disabled => 0,
        },
    );
}

=head2 insert

=cut

sub insert
{
    return HierarchicalMultipliers::MultiplierPct::insert(@_);
}

=head2 update

=cut

sub update
{
    return HierarchicalMultipliers::MultiplierPct::update(@_);
}

=head2 load

В соответствии с описанным в HierarchicalMultipliers API для добавления новых коэффициентов.

Возвращает:
   {
      hierarchical_multiplier_id => XXX,
      last_change => XXX,
      multiplier_pct => XXX,
      os_type => XXX,
      tablet_multiplier_value_id => XXX,
   }

=cut

sub load {
    my ($hierarchical_multiplier) = @_;
    if (defined $hierarchical_multiplier->{multiplier_pct}) {
        return HierarchicalMultipliers::MultiplierPct::load(@_);
    }

    my $existing_values = get_all_sql(
        PPC(cid => $hierarchical_multiplier->{cid}), [
        "select tablet_multiplier_value_id, os_type, multiplier_pct from tablet_multiplier_values",
        where => { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id} },
    ]
    );

    die "Inconsistent state - more than one tablet_multiplier_values" if scalar @$existing_values > 1;

    my $existing_value = @$existing_values[0];

    my $result = hash_merge ($existing_value,
        hash_cut $hierarchical_multiplier, qw/last_change hierarchical_multiplier_id/);
    return $result;
}

=head2 calc_stats

=cut

sub calc_stats
{
    return HierarchicalMultipliers::MultiplierPct::calc_stats(@_);
}

=head2 delete_set_values

=cut

sub delete_set_values
{
    return HierarchicalMultipliers::MultiplierPct::delete_set_values(@_);
}

=head2 delete_camp_values

=cut

sub delete_camp_values
{
    my ($cid) = @_;
    return HierarchicalMultipliers::MultiplierPct::delete_camp_values($cid, 'tablet_multiplier');
}

=head2 delete_camp_group_values

=cut

sub delete_camp_group_values
{
    my ($cid, $pids) = @_;
    return HierarchicalMultipliers::MultiplierPct::delete_camp_group_values($cid, $pids, 'tablet_multiplier');
}

1;
