package HierarchicalMultipliers::Weather;
use Direct::Modern;

use List::Util qw/min max/;
use List::MoreUtils qw/part/;
use Yandex::DBTools;
use JSON qw/from_json to_json/;
use Yandex::HashUtils;

use Settings;
use HierarchicalMultipliers::Base qw/register_type/;
use JavaIntapi::GenerateObjectIds;

BEGIN {
    register_type(weather_multiplier => {
        load       => \&load,
        calc_stats => \&calc_stats,
        insert     => \&insert,
    });
}

=head2 load

    В соответствии с описанным в HierarchicalMultipliers API для добавления новых коэффициентов.

    Возвращает:
    {
        is_enabled => XXX,
        hierarchical_multiplier_id => XXX,
        last_change => XXX,
        conditions => [
            {
                condition => <json condition>,
                multiplier_pct => XXX,
            },
            ...
        ]
    }

=cut

sub load {
    my ($hierarchical_multiplier) = @_;
    my $conditions = get_all_sql(
        PPC(cid => $hierarchical_multiplier->{cid}),
        [
            "select weather_multiplier_value_id, condition_json AS expression, multiplier_pct from weather_multiplier_values",
            where => { hierarchical_multiplier_id => $hierarchical_multiplier->{hierarchical_multiplier_id} },
        ]
    );

    my $result = hash_cut $hierarchical_multiplier, qw/is_enabled last_change hierarchical_multiplier_id/;
    foreach my $condition (@$conditions) {
        my $expression = from_json($condition->{expression});
        _sanitize_expression($expression);
        $condition->{expression} = $expression;
    }
    $result->{conditions} = $conditions;
    return $result;
}

=head2 _match

    Проверяет логическое условие на соответсвие шаблону,

    Принимает на вход:
      atom => атомарная логическая операция вида {parameter: <parameter>, operation: <operation>, value: <value> }
      pattern => шаблон соответствия:
        {
          parameter: <значение>
          operation: <значение>
          value:     <значение>
        }

    Возвращает 1 или 0

=cut

sub _match {
    my (%params) = @_;

    my $atom = $params{atom};
    my $pattern = $params{pattern};

    my @keys = grep {defined $params{pattern}->{$_}} qw/parameter operation value/;
    foreach my $key (@keys) {
        if ($atom->{$key} ne $pattern->{$key}) {
            return 0;
        }
    }
    return 1;
}

=head2 _sanitize_expression

    Прихорашиваем expression для чтения на фронте:
     - удаляем из условия "снег с дождем"(2) { parameter=>"prec_type", operation => "eq", value => 2 }
       этот тип осадков объединен с дождем(1) и дописывается на бекенде.

=cut

sub _sanitize_expression {
    my ($expression) = @_;

    foreach my $or_array (@$expression) {
        #удаляем "снег с дождем"
        if (scalar @$or_array > 1) { #встречается только вместе с дождем [ {prec_type eq 1}, {prec_type eq 2} ]
            my $pattern = { parameter => "prec_type", operation => "eq", value => 2 };
            my ($matched, $other) = part {  _match(atom => $_, pattern=>$pattern) ? 0 : 1} @$or_array;
            $or_array = $other if (defined $matched);
        }
    }
}

=head2 calc_stats

Вычисляет часть статистики, используемой для формирования сводной в HierarchicalMultipliers::calc_stats

=cut
sub calc_stats {
    my ($multiplier, $opts) = @_;

    return { value_count => 0 } if $opts && $opts->{group}->{adgroup_type} ne 'cpm_banner';

    my @values = map {$_->{multiplier_pct}} @{$multiplier->{conditions}};
    my $result = {
        multiplier_pct_min => min(@values),
        multiplier_pct_max => max(@values),
        values_count       => scalar @{$multiplier->{conditions}},
    };
    my $lower_bound = min(grep {$_ != 100} @values);
    my $upper_bound = max(grep {$_ > 100} @values);
    $result->{adjustments_lower_bound} = $lower_bound if defined $lower_bound;
    $result->{adjustments_upper_bound} = $upper_bound if defined $upper_bound;
    return $result;
}

=head2 insert

В соответствии с описанным в HierarchicalMultipliers API для добавления новых коэффициентов.

Вставляет запись в 'hierarchical_multipliers'.
Вставляет записи в 'weather_multiplier_values'.

=cut
sub insert {
    my ($data, $proposed_hierarchical_multiplier) = @_;

    $proposed_hierarchical_multiplier->{is_enabled} = $data->{is_enabled} ? 1 : 0;

    do_insert_into_table(PPC(cid => $proposed_hierarchical_multiplier->{cid}), "hierarchical_multipliers", $proposed_hierarchical_multiplier);

    die "Weather 'conditions' should be an array ref'" unless ref($data->{conditions}) eq 'ARRAY';
    my $ids = JavaIntapi::GenerateObjectIds->new(object_type => 'multiplier',
            count => scalar @{$data->{conditions}})->call();

    my @insert_data;
    for my $condition (@{$data->{conditions}}) {
        push @insert_data, {
            weather_multiplier_value_id => shift(@$ids),
            hierarchical_multiplier_id => $proposed_hierarchical_multiplier->{hierarchical_multiplier_id},
            condition_json => to_json($condition->{expression}),
            multiplier_pct => $condition->{multiplier_pct},
        };
    }
    do_mass_insert_sql(PPC(cid => $proposed_hierarchical_multiplier->{cid}), 'insert into weather_multiplier_values'
        . '(weather_multiplier_value_id, hierarchical_multiplier_id, condition_json, multiplier_pct)'
        . ' values %s',
        [map { [@{$_}{qw(weather_multiplier_value_id hierarchical_multiplier_id condition_json multiplier_pct)}] } @insert_data]);
    return @insert_data ? \@insert_data : undef;
}

1;
