package Holidays;

=head1 NAME

    Holidays

=head1 DESCRIPTION

    Модуль для поддержки локальных праздников и переносов выходных в странах присутствия

=cut

use Direct::Modern;

use Yandex::DBTools;
use Yandex::TimeCommon;
use Yandex::DateTime;

use Time::Local qw/timelocal/;
use List::MoreUtils qw/all none/;

use Settings;
use geo_regions;

use base qw/Exporter/;
our @EXPORT_OK = qw/
    is_weekend

    is_workday
    is_holiday

    is_great_holiday
    is_weekend_workday

    is_holiday_everywhere
    regions_with_holidays
    regions_with_weekend_workday
/;

# Пожалуйста, добавляйте константы в оба проекта. Список в Java:
# https://a.yandex-team.ru/arc/trunk/arcadia/direct/jobs/src/main/java/ru/yandex/direct/jobs/bannersystem/dataimport/HolidaySynchronizerJob.java
our @LOCAL_HOLIDAYS_SUPPORTED_REGIONS = (
    $geo_regions::RUS,
    $geo_regions::UKR,
    $geo_regions::KAZ,
    $geo_regions::BY,
    $geo_regions::TR,
);

# Праздничные дни
my %holidays_cache;
# Рабочие дни в выходные
my %workdays_cache;
my $UPDATED_KEY = '__updated';
# Время кеширования информации в секундах
my $CACHING_PERIOD = 60 * 60 * 24;

sub _has_valid_cache {
    my ($cache, $region_id, $year) = @_;

    return (exists $cache->{$region_id} &&
            exists $cache->{$region_id}->{$year} &&
            ($cache->{$region_id}->{$year}->{$UPDATED_KEY} // 0) + $CACHING_PERIOD > time ? 1 : 0);
}


sub _split_date {
    my ($date) = @_;

    my ($year, $m, $d);
    if ($date =~ /^(\d{4})-?(\d{2})-?(\d{2})/) {
        ($year, $m, $d) = ($1, $2, $3);
    } else {
        die "Invalid date $date";
    }

    return ($year, $m, $d);
}


sub _is_special_day {
    my ($date, $region_id, $type) = @_;
    my $cache = $type eq 'holiday' ? \%holidays_cache : \%workdays_cache;

    my ($year, $m, $d) = _split_date($date);
    if (!_has_valid_cache($cache, $region_id, $year)) {
        _populate_cache($year, [$region_id], $type);
    }
    return ($cache->{$region_id}->{$year}->{"$year-$m-$d"} ? 1 : 0);
}


sub _populate_cache {
    my ($year, $regions_to_fetch, $type) = @_;

    unless ($regions_to_fetch && @$regions_to_fetch) {
        $regions_to_fetch = [grep {!_has_valid_cache(\%holidays_cache, $_, $year) ||
                                   !_has_valid_cache(\%workdays_cache, $_, $year)} @LOCAL_HOLIDAYS_SUPPORTED_REGIONS];
    }
    if ($regions_to_fetch && @$regions_to_fetch) {
        my $where = {
            holiday_date__ge => "$year-01-01",
            holiday_date__le => "$year-12-31",
            region_id        => $regions_to_fetch,
        };
        $where->{type} = $type if defined $type;
        my $data = get_all_sql(PPCDICT, [
                "SELECT holiday_date, region_id, type
                   FROM great_holidays",
                WHERE => $where
            ]);

        # Проставляем дату обновления кеша
        my $now = time;
        for my $region_id (@$regions_to_fetch) {
            $holidays_cache{$region_id}->{$year}->{$UPDATED_KEY} = $now if (!$type || $type eq 'holiday');
            $workdays_cache{$region_id}->{$year}->{$UPDATED_KEY} = $now if (!$type || $type eq 'workday');
        }

        for my $h (@$data) {
            my $cache = ($h->{type} eq 'holiday' ? \%holidays_cache : \%workdays_cache);
            $cache->{$h->{region_id}}->{$year}->{$h->{holiday_date}} = 1;
        }
    }
}


=head3 is_great_holiday($datetime, $region_id)

    Праздник сегодня или нет в стране с $region_id?
    Для скриптов типа ppcAutobudgetForecast.pl, которые не хочется запускать по праздникам.

    Поддерживаются только страны присутствия. Для других стран - всегда false.
    Если регион не задан - берем Россию.
    Если не задана дата - берем текущую.
    Дата должна передаваться в виде строки формате 'YYYYMMDD' или 'YYYY-MM-DD'.

    $is_great_holiday = Holidays::is_great_holiday('20150504');

=cut


sub is_great_holiday {
    my ($date, $region_id) = @_;
    $date ||= today();
    $region_id ||= $geo_regions::RUS;

    return _is_special_day($date, $region_id, 'holiday');
}


=head3 is_weekend_workday($datetime, $region_id)

    Рыбочий выходной сегодня или нет в стране с $region_id?

    Поддерживаются только страны присутствия. Для других стран - всегда false.
    Если регион не задан - берем Россию.
    Если не задана дата - берем текущую.
    Дата должна передаваться в виде строки формате 'YYYYMMDD' или 'YYYY-MM-DD'.

=cut

sub is_weekend_workday {
    my ($date, $region_id) = @_;
    $date ||= today();
    $region_id ||= $geo_regions::RUS;

    return _is_special_day($date, $region_id, 'workday');
}


=head3 is_workday($datetime, $region_id)

    Рабочий сегодня день в стране с $region_id?
    Ответ дается с учетом празников и переносов.
    Поддерживаются только страны присутствия. Для других стран учитываются только обычные выходные дни недели.

    Если регион не задан - берем Россию.
    Если не задана дата - берем текущую.
    Дата должна передаваться в виде строки формате 'YYYYMMDD' или 'YYYY-MM-DD'.

=cut

sub is_workday {
    my ($date, $region_id) = @_;
    $date ||= today();
    $region_id ||= $geo_regions::RUS;

    my $result = (is_weekend($date) ?
        is_weekend_workday($date, $region_id) :
        !is_great_holiday($date, $region_id)
    );
    return $result;
}


=head3 is_holiday($datetime, $region_id)

    То же что и is_workday, но с точностью наоборот

    Если не задана дата - берем текущую.

=cut

sub is_holiday {
    my ($date, $region_id) = @_;
    $date ||= today();
    $region_id ||= $geo_regions::RUS;

    return (is_workday($date, $region_id) ? 0 : 1);
}


=head3 is_holiday_everywhere($date)

    Выяснить, является ли указанный день выходным во всех странах с учетом
    переносов в странах присутствия.

    Если не задана дата - берем текущую.
    Дата должна передаваться в виде строки формате 'YYYYMMDD' или 'YYYY-MM-DD'.

=cut

sub is_holiday_everywhere {
    my ($date) = @_;
    $date ||= today();

    return (is_weekend($date) &&
            (none {is_weekend_workday($date, $_)} @LOCAL_HOLIDAYS_SUPPORTED_REGIONS));
}


=head3 regions_with_holidays($date)

    Получить список всех стран присутствия (точнее, их region_id), в которых
    праздник в заданный день.

    Если не задана дата - берем текущую.
    Дата должна передаваться в виде строки формате 'YYYYMMDD' или 'YYYY-MM-DD'.

=cut

sub regions_with_holidays {
    my ($date) = @_;
    $date ||= today();

    return [grep { !is_workday($date, $_) } @LOCAL_HOLIDAYS_SUPPORTED_REGIONS];
}


=head3 regions_with_weekend_workday($date)

    Получить список всех стран присутствия (точнее, их region_id), в которых
    рабочий день в заданный выходной день.

    Если не задана дата - берем текущую.
    Дата должна передаваться в виде строки формате 'YYYYMMDD' или 'YYYY-MM-DD'.

=cut

sub regions_with_weekend_workday {
    my ($date) = @_;
    $date ||= today();

    return [grep { is_weekend_workday($date, $_) } @LOCAL_HOLIDAYS_SUPPORTED_REGIONS]
}


=head2 is_weekend($date)

    Определяет является ли указанный день субботой или воскресеньем

    $is_weekend = Holidays::is_weekend("20150503");

    Если не задана дата - берем текущую.
    Дата должна передаваться в виде строки формате 'YYYYMMDD' или 'YYYY-MM-DD'.

=cut

sub is_weekend {
    my ( $date ) = @_;
    $date ||= today();

    my ($year, $m, $d) = _split_date($date);
    my $tm = timelocal(0, 0, 0, $d, $m-1, $year-1900);
    my $wday = (localtime $tm)[6];
    return $wday == 0 || $wday == 6;
}

1;
