package HttpTools;

=head2 NAME

    HttpTools - набор функций для работы внутри веб-сервера

=cut

use Direct::Modern;

use parent 'Exporter';
our @EXPORT = qw(
    parse_form
    multiform2directform

    get_cookie
    get_all_cookies
    set_cookie
    http_server_host
    http_get_header
    http_server_scheme
    http_geo_region
    http_country_from_geobase
    http_geo_exact_region
    http_remote_ip

    yandex_domain
    get_passport_url
    get_passport_domain
    lang_auto_detect
);

# subs_placing: Tools не должен зависеть от Settings. Если есть функции, которым это почему-то точно надо -- отселить
use Settings;
# subs_placing: Tools, зависящий от DBTools -- подозрительно и неправильно. Функции, работающие с базой -- в Primitives или более высокоуровневые модули
use Yandex::I18n;
use GeoTools;
use POSIX qw(strftime ceil);
use CGI;

use Yandex::DateTime;
use Yandex::TimeCommon;
use Yandex::Trace;
use Yandex::URL qw/get_num_level_domain get_top_level_domain/;
use Yandex::UTF8CGI;

use CGI::Cookie;
use Encode;
use Hash::MultiValue;
use JSON;
use List::Util qw/max min sum first/;
use Time::Local qw(timelocal);
use URI::Escape qw/uri_escape_utf8/;
use cookiemy;
use langdetect;
use uatraits;

use IpTools qw//;
use HashingTools qw//;
use LogTools qw/log_messages/;

=head2 parse_form

This routine parses the variables passed in a form into %FORM: $FORM{'varname'}

=cut

sub parse_form {
    my %opt = @_;
    return %{ parse_multivalue_form(%opt)->[1] };
}

=head2 parse_multivalue_form

Превращает данные запроса в пару [$multivalue_form, $form], где $multivalue_form — это Hash::MultiValue, а $form — это
простой hashref с "упрощённой версией" данных: если в $multivalue_form для ключа больше одного значения, в $form
эти значения попадают склеенные через запятую (и про это пишется в лог); пустые строки в значениях в $form
превращаются в undef.

Если никаких множественных значений для ключа не ожидается, удобнее использовать parse_form, который возвращает
обычный список ключ-значение-ключ-значение.

=cut

sub parse_multivalue_form {
    my %opt = @_;

    my $profile = Yandex::Trace::new_profile('http_tools:parse_multivalue_form');

    my $multivalue_form = Hash::MultiValue->new;
    my $q = new Yandex::UTF8CGI( $opt{encoding} );

    my $encode = $opt{encoding}
        ? sub { my ($value) = @_; return $value if ref $value; return Encode::encode( $opt{encoding}, $value ); }
        : sub { $_[0] };

    for my $name ( $q->param() ) {
        for my $value ( $q->param($name) ) {
            $multivalue_form->add( $name, $encode->($value) );
        }
    }

    my %form;
    for my $name ( keys %$multivalue_form ) {
        my @values = $multivalue_form->get_all($name);

        if ( @values > 1 ) {
            # Записываем в лог случаи, когда получаем несколько значений параметра -- для оценки сложности перехода на использование Hash::MultiValue.
            eval {
                log_messages('parse_multiform', { cmd => $multivalue_form->{'cmd'}, param_name => $name });
            };
            warn $@ if $@;  # для того, чтобы хоть как-то отметить факт передачи нескольких значений в случае ошибки записи в лог

            $form{$name} = join ', ', grep { $_ ne '' } @values;

            next;
        }

        ## если код дошёл до сюда, в @values должен быть ровно один элемент
        $form{$name} = $values[0];

        # DIRECT-29024: Оставляем пустые значения в виде undef:
        # 1) В коде много проверок на `if (defined $FORM{...})`, которые предполагают, что данные в форме есть и они НЕ пустые
        # 2) Если передавать значение в качестве undef, то через exists мы можем убедиться что они есть, а через defined - что они есть и НЕ пустые
        # 3) Если передавать значение через '', то нам потребуется дополнительная проверка на НЕ пустоту
        if ( defined $form{$name} && $form{$name} eq '' ) {
            $form{$name} = undef;
        }
    }

    return [ $multivalue_form, \%form ];
}

=head2 parse_json_form($form)

Десериализация полей формы. Если имя параметра содержит префикс json_
считаем что данные сериализованы в JSON.

=cut

sub parse_json_form {

    my $form = shift;

    my %ret = %$form;
    for my $param (grep {/^json_/} keys %$form) {
        my $perl = eval {from_json $form->{$param}};
        die "Invalid json data in param $param: " . ($form->{$param} || '') if $@;
        $ret{$param} = $perl;
    }
    return %ret;
}

=head2 compose_json_form($form)

Cериализация полей формы. Если имя параметра содержит префикс json_
считаем что данные десериализованы в JSON.
Выполняет обратную parse_json_form операцию.

=cut

sub compose_json_form {

    my $form = shift;

    my %ret = %$form;
    for my $param (grep {/^json_/} keys %$form) {
        my $perl = eval {to_json $form->{$param}};
        die "Invalid json data in param $param: " . ($form->{$param} || '') if $@;
        $ret{$param} = $perl;
    }
    return %ret;
}

=head2 multiform2directform

    Принимает многозначную форму (Hash::MultiValue),
    переделывает ее в исторически Директовую (ссылка на простой хеш, не объект никакого пользовательского класса).

    Если у какого-то параметра несколько значений -- склеиваются через запятую.
    { cid => [111, 222, 333] }
    ==>
    { cid => '111,222,333' }

=cut
sub multiform2directform($%)
{
    my ($multiform, %opt) = @_;

    my $directform = $multiform->as_hashref_mixed;
    for my $k (keys %$directform){
        if (ref($directform->{$k}) eq 'ARRAY') {
            $directform->{$k} = join ', ', grep { $_ ne '' } @{$directform->{$k}};
        }
        # Опция undef_empty_str выставляется для вызовов multiform2directform там, где раньше использовалась parse_form, для имитации поведения последней (DIRECT-29024)
        if (defined($directform->{$k}) && $directform->{$k} eq '' && $opt{undef_empty_str}) {
            $directform->{$k} = undef;
        }
    }

    return $directform;
}


=head2 get_request

    Если загружен Apache2::RequestUtil - вернуть request, иначе undef

=cut
sub get_request {
    if ($INC{'Apache2/RequestUtil.pm'}) {
        return Apache2::RequestUtil->request;
    } else {
        return undef;
    }
}


=head2 get_cookie

  $value = get_cookie($r, $cokie_name);

=cut

sub get_cookie
{
    my ($r, $name) = @_;
    my $cookie;
    if ($r->isa('Plack::Request')) {
        $cookie = $r->cookies->{$name};
    } else {
        my $cookies = CGI::Cookie->fetch($r);
        if ( defined $cookies && defined $cookies->{$name} ) {
            $cookie = $cookies->{$name}->value;
        }
    }
    if (!defined $cookie) {
        return undef;
    } elsif (Encode::is_utf8($cookie)) {
        return $cookie;
    } else {
        return Encode::decode('utf8', $cookie);
    }
}

#======================================================================

=head2 get_all_cookies

  $cookies = get_all_cookies($r);

=cut

sub get_all_cookies
{
    my $r = shift;

    my $cookies;
    if ($r->isa('Plack::Request')) {
        $cookies = $r->cookies;
    } else {
        my $cookies_raw = CGI::Cookie->fetch($r);
        for my $name (keys %$cookies_raw) {
            $cookies->{$name} = $cookies_raw->{$name}->{value}->[0];
        }
    }

    for my $name (keys %$cookies) {
        if (! Encode::is_utf8($cookies->{$name})) {
            $cookies->{$name} = Encode::decode('utf8', $cookies->{$name});
            # защита от http response splitting
            if (defined $cookies->{$name}) {
                $cookies->{$name} =~ s/[\r\n\x{00}\N{U+c48dc48a}\N{U+0d0a}\N{U+e0b48a}]//g;
            }
        }
    }

    return $cookies;
}

#======================================================================

=head2 set_cookie

  $cookie_obj = set_cookie($name, $value, $expires);
  $r->headers_out->set('Set-Cookie' => $cookie_obj) if defined $cookie_obj;

  default expires == '+12M' - if $expires is undef

=cut

sub set_cookie
{
    my ($r, $name, $value, $expires, $domain) = @_;
    $value = Encode::encode('utf8', $value);
    $expires = '+12M' unless defined $expires;

    unless (defined $domain) {
        $domain = get_num_level_domain($r->hostname(), 3) || 'direct.yandex.ru';
    }

    my $cookie = new CGI::Cookie(-name    =>  $name,
                                 -value   =>  $value,
                                 -domain  =>  $domain,
                                 -expires =>  $expires);

    $r->headers_out->add('Set-Cookie' => $cookie);
    $r->err_headers_out->add('Set-Cookie' => $cookie);

    return $cookie;
}

#======================================================================


#======================================================================


# по реквесту получаем хост
sub http_server_host {
    my ($r, %O) = @_;
    if ($r->isa('Plack::Request')) {
        my $uri = $r->uri;
        return $uri->port == $uri->default_port || $O{no_port} ? $uri->host : $uri->host_port;
    } else {
        my $host = $r->headers_in->get('Host');
        if (!defined $host) {
            my $port = $r->get_server_port;
            $host = $r->hostname().(!$ENV{HTTPS} && $port == 80 || $ENV{HTTPS} && $port == 443 || $O{no_port} ? '' : ":$port");
        }
        return $host;
    }
}

# --------------------------------------------------------------------

=head2 http_get_header

по реквесту получаем http-заголовок

    $is_ajax = http_get_header($r, 'X-Requested-With') eq 'XMLHttpRequest';

=cut

sub http_get_header($$) {
    my $r = shift;
    my $header_name = shift;

    if ($r->isa('Plack::Request')) {
        return $r->headers->header($header_name) || '';
    } else {
        return $r->headers_in->get($header_name) || '';
    }
}

# определение протокола
sub http_server_scheme {
    my $r = shift;
    if ($r->isa('Plack::Request')) {
        return $r->uri->scheme;
    } else {
        return $r->subprocess_env('HTTPS') ? 'https' : 'http';
    }
}

# определение региона пользователя, округлённого до Директовских
sub http_geo_region {
    my ($r) = @_;
    my $region = http_geo_exact_region($r);
    return GeoTools::get_direct_region_from_geo($region);
}

=head2 guess_country_by_location

    определение страны, не зависит от геобазы Директа, с проверкой gdpr

=cut

sub guess_country_by_location {
    my ($r) = @_;

    # https://wiki.yandex-team.ru/users/4ernyakov/upravljator-kukami/#sostojanijakukigdpr
    my $is_gdpr_cookie = get_cookie($r, 'is_gdpr');
    my $gdpr_cookie = get_cookie($r, 'gdpr');
    my $can_guess_country_by_location = (defined $gdpr_cookie && ($gdpr_cookie == 0 || $gdpr_cookie == 3)) || (defined $is_gdpr_cookie && $is_gdpr_cookie == 0);

    return $can_guess_country_by_location ? http_country_from_geobase($r) : undef;
}

=head2 http_country_from_geobase

    определение страны, не зависит от геобазы Директа

=cut

sub http_country_from_geobase {
    my ($r) = @_;
    my $region = http_geo_exact_region($r);
    return GeoTools::get_country_from_region($region);
}

# определение точного региона пользователя
sub http_geo_exact_region {
    my ($r) = @_;
    # если есть кука - берём регион из неё
    my $region = get_cookie($r, 'yandex_gid');
    if (!$region || $region !~ /^\d+$/) {
        # если нет куки - из REMOTE_ADDR и X-Forwarded-For
        my $remote_ip = IpTools::get_first_non_private_ip(http_ips_list($r));
        $region = GeoTools::get_geo_from_ip($remote_ip) || 0;
    }
    return $region || 0;
}

# список ip адресов из X-Forwarded-For и REMOTE_ADDR
sub http_ips_list {
    my ($r) = @_;
    if ($r->isa('Plack::Request')) {
        my @ips = split /\s*,\s*/, $r->headers->header('X-Forwarded-For') || '';
        return (@ips, $r->address);
    } else {
        my @ips = split /\s*,\s*/, $r->headers_in->get('X-Forwarded-For') || '';
        return (@ips, http_remote_ip($r));
    }
}

# по объекту request вернуть REMOTE_ADDR
sub http_remote_ip {
    my ($r) = @_;
    if ($r->isa('Plack::Request')) {
        return $r->address;
    } else {
        # TODO(trusty): поменять на безусловное $r->useragent_ip
        if ($] ge 5.018) {
            return $r->useragent_ip;
        } else {
            return $r->connection->remote_ip;
        }
    }
}


#======================================================================


=head2 yandex_domain

  host = "direct.".yandex_domain($r);

  $r = Apache2::RequestRec или Plack::Request;

=cut
sub yandex_domain($)
{
    my ($r) = @_;

    my $hostname = $r->isa("Plack::Request") ? $r->uri->host : $r->hostname;

    my $yandex_domain = 'yandex.ru';
    $yandex_domain = $1 if $hostname =~ m/(yandex\.(?:ru|ua|com\.tr|com|kz|by))/;
    return $yandex_domain;
}

=head2 yandex_tld_domain

    Возвращает яндексовый домен верхнего уровня (TLD или com.tr)

    Параметры:
        $r - Apache2::RequestRec или Plack::Request;
    Результат:
        $tld - ru/ua/com/com.tr/kz/byb

=cut

sub yandex_tld_domain {
    my ($r) = @_;

    return yandex_domain($r) =~ s/^yandex\.//r;
}

#======================================================================

=head2 lang_auto_detect

    определение языка интерфейса
    http://doc.yandex-team.ru/lib/lang-detect/concepts/lang-detect-descr.xml
    используется:
     - настройки из tune.yandex.ru
     - текущий домен
     - текущий регион пользователя
     - язык из поля Accept-Language заголовка HTTP

    Yandex::I18n::init_i18n(lang_auto_detect($r));

=cut

sub lang_auto_detect($) {
    my $r = shift;

    my $langdetect = langdetect::lookup->new($Settings::LANG_DETECT_DATA_FILE);

    my $cookie_my_raw = get_cookie($r, 'my');
    my $cookie_my_lang = 0; # auto

    if ($cookie_my_raw) {
        my $parse_cookie_my = new cookiemy::Setup();
        $parse_cookie_my->parse($cookie_my_raw);
        $cookie_my_lang = $parse_cookie_my->find($Settings::LANG_DETECT_COOKIE_MY_LANG_NUM)->[1] || 0;
    }

    my $accept_language = $r->isa('Plack::Request') ? $r->headers->header('Accept-Language') : $ENV{HTTP_ACCEPT_LANGUAGE};
    my $yandex_domain = yandex_domain($r);
    my $lang_detect_params = {
        'geo'      => http_geo_exact_region($r),
        'domain'   => $yandex_domain,
        'cookie'   => $cookie_my_lang,
        'default'  => Yandex::I18n::default_lang,
        'language' => $accept_language,
    };

    # https://jira.yandex-team.ru/browse/DIRECT-18884
    # при изменении дополнительно нужно поменять верстку в футере (выбор языков) - data/block/i-footer/i-footer.tt2
    my $all_languages = [map {$_ eq 'ua' ? 'uk' : $_} Yandex::I18n::default_lang, Yandex::I18n::get_other_langs];
    my %allow_languages;
    if ($yandex_domain =~ /\.ru$/) {
        %allow_languages = map {$_ => 1} qw/ru en/;
    } elsif ($yandex_domain =~ /\.ua$/) {
        %allow_languages = map {$_ => 1} qw/ru uk/;
    } elsif ($yandex_domain =~ /\.by$/) {
        %allow_languages = map {$_ => 1} qw/ru en/;
    } elsif ($yandex_domain =~ /\.kz$/) {
        %allow_languages = map {$_ => 1} qw/ru en/;
    } elsif ($yandex_domain =~ /\.com$/) {
        %allow_languages = map {$_ => 1} qw/ru en/;
    } elsif ($yandex_domain =~ /\.com\.tr$/) {
        %allow_languages = map {$_ => 1} qw/tr en/;
    } else {
        %allow_languages = map {$_ => 1} @$all_languages;
    }
    $lang_detect_params->{filter} = [grep {$allow_languages{$_}} @$all_languages];

    my $lang = $langdetect->find($lang_detect_params) || Yandex::I18n::default_lang;

    $lang = ref($lang) eq 'HASH' && $lang->{lang} ? $lang->{lang} : Yandex::I18n::default_lang;
    $lang = 'ua' if $lang eq 'uk';
    return $lang;
}

#======================================================================

=head2 get_tune_secret_key

    сгенерировать ключ для изменения языка в tune.yandex.ru
    http://wiki.yandex-team.ru/AlekseyPonomarev/secret-key

=cut

sub get_tune_secret_key($$) {
    my ($r, $UID) = @_;

    my $sk;
    my $days = int(time / 86400);

    if ($UID) {
        $sk = "u" . lc(HashingTools::md5_hex_utf8($UID . '::' . $days));
    } else {
        my $cookie_yandexuid = get_cookie($r, 'yandexuid') || "";
        $sk = $cookie_yandexuid ? "y" . lc(HashingTools::md5_hex_utf8('0:' . $cookie_yandexuid . ':' . $days)) : '';
    }

    return $sk;
}

=head2 apache_add_cleanup

# установить apache cleanup handler для убийства слишком выросшего процесса

apache2: с Plack'ом переедет в какое-нибудь Middleware

require Apache делается прямо внутри функции для того, чтобы весь модуль при загрузке не тащил еще и Apache. И потому, что не хочется эту функцию срочно переносить в какой-нибудь другой модуль.

=cut
sub apache_add_cleanup
{
    my ($max_size, $reqid) = @_;
    require Apache2::RequestUtil;
    # $req_count -- счетчик обработанных внутри одного воркера запросов; объявляется не внутри cleanup-коллбека потому,
    # что при каждом запросе создается новый экземпляр коллбека (в новом объекте Apache2::RequestRec->pool()),
    # и переменные внутри коллбека живут только один запрос
    state $req_count = 0;
    Apache2::RequestUtil->request->pool->cleanup_register(
        sub {
            $req_count++;
            my $size = Yandex::ProcInfo::proc_memory();
            my $msize = int($size / 1024 / 1024);
            if ($size > $max_size) {
                print STDERR human_datetime()." Process [$$".($reqid ? "/$reqid" : "")."] too big: rss=${msize}Mb, req_count=$req_count, exit\n";
                Apache2::RequestUtil->request->child_terminate();
            }
        }
    );
}


#======================================================================

=head3 parse_date_form(form, options)

    Обрабатываем форму и получаем дату из станлартного календаря в необходимом виде

    options:
        fix_sort - первым вернуть меньшее значение, вторым - бОльшее
        set_default - выставлять ли умолчательные значения(по-умолчанию - 1)
=cut

# subs_placing: многоват логики и валидации, куда-нибудь бы отселить _validate_date, parse_date_form

sub parse_date_form
{
    my ($form, $options) = @_;

    my (%date_from, %date_to, @errors);

    # инициализируем хэш с датами
    foreach my $dn (qw/day month year/) {
        $date_from{$dn} = $form->{"from_$dn"} if defined $form->{"from_$dn"};
        $date_to{$dn} = $form->{"to_$dn"} if defined $form->{"to_$dn"};
    }

    my $validate_date = !_validate_date(\%date_from) || !_validate_date(\%date_to);
    if (!keys %date_from && !keys %date_to) {
        unless (defined $options->{set_default} && !$options->{set_default}) {
            ($date_from{day}
                , $date_from{month}
                , $date_from{year}) = split /\-/, strftime("%d-%m-%Y", localtime(time() - 3600 * 24 * 7)); # 7 days ago

            ($date_to{day}
                , $date_to{month}
                , $date_to{year}) = split /\-/, strftime("%d-%m-%Y", localtime(time()));
        } else {
            push @errors, "Неверно указан период!";
        }
    } elsif ($validate_date
                || !keys %date_from
                || !keys %date_to) {
        push @errors, "Неверно указан период!";
    }

    if (! @errors) {

        # если установлена опция сортировки возвращаемого результата
        if ($options->{fix_sort}) {
            my $d1 = mysql2unix(join "-", map {$date_from{$_}} qw/year month day/ );
            my $d2 = mysql2unix(join "-", map {$date_to{$_}} qw/year month day/ );

            if ($d2 < $d1) {
                return (\%date_to, \%date_from, \@errors);
            }
        }
    }

    return (\%date_from, \%date_to, \@errors);
}

sub _validate_date
{
    my $date_raw = shift;

    my $date = {};
    if (ref $date_raw eq 'HASH') {
        $date = $date_raw;
    } elsif (!ref $date_raw) {
        ($date->{year}, $date->{month}, $date->{day}) = split /\D+/, $date_raw;
    } else {
        return 1;
    }

    if (
        scalar(
            grep {/^\d+$/}
            map {$date->{$_} || ''}
            qw/day month year/) != 3) {

        return 0;
    }

    unless (
        $date->{day} >= 1 && $date->{day} <= 31 &&
        $date->{month} >= 1 && $date->{month} <= 12 &&
        $date->{year} >= 1970 && $date->{year} <= 2100
    ) {
        return 0;
    }

    return 1;
}


=head2 get_passport_url($r, $path, $args)

    Возвращает паспортный урл в соответствии с доменной зоной запроса
    $r - Apache::Request

=cut

sub get_passport_url {
    my ($r, $path, $args) = @_;

    $args = {} unless ref $args eq 'HASH';
    $args->{'origin'} = 'direct';
    $path ||= '';

    my $passport_domain = get_passport_domain($r);
    return "https://$passport_domain/$path" . (
        keys %$args
            ? '?' . join('&', map {uri_escape_utf8($_) . '=' . uri_escape_utf8($args->{$_})} keys %$args)
            : ''
    )
}

=head2 get_redirect_path_for_passport_verify_auth($FORM, $registered_main_uri)

    Возвращает урл для редиректа после подтверждения пароля в паспорте

=cut

sub get_redirect_path_for_passport_verify_auth
{
    my ($FORM, $registered_main_uri) = @_;

    my %redir_args = %$FORM;

    if ($FORM->{native_uri}) {
        # если есть native_uri - это один из фронтовых урлов
        # dna, daas и т.п.
        # в данном случае cmd удаляем, а native_uri берем ниже как relative_path
        delete $redir_args{native_uri};
        delete $redir_args{cmd};
    }

    return ($FORM->{native_uri} ? $FORM->{native_uri} : $registered_main_uri) . (
        %redir_args
            ? '?' . join('&', map {uri_escape_utf8($_) . '=' . uri_escape_utf8($redir_args{$_})} keys %redir_args)
            : ''
    );
}

=head2 get_passport_domain($r)

    Возвращает паспортный урл в соответствии с доменной зоной запроса
    $r - Apache::Request

=cut

sub get_passport_domain {
    my $r = shift;

    my $domain = yandex_domain($r);

    return "passport.$domain";
}

=head2 get_uatraits($ua_string)

    Возвращает хеш http://wiki.yandex-team.ru/EvgenijjVereshhagin/uatraits

=cut

{
my $ua_detector;

sub get_uatraits
{
    my $ua_string = shift;

    $ua_detector ||= uatraits->new("/usr/share/uatraits/browser.xml");
    return $ua_detector->detect($ua_string);
}
}

1;
