package Intapi;

# $Id$

=head1 NAME

    Intapi

=head1 DESCRIPTION

    "Внутреннее" API Директа

    Хеш %cmd:
    определяет доступные команды ("локейшены").
    Формат
        (<команда> => {описание команды})

    Описание команды -- хеш
    Обязательные поля
        module -- модуль, функция handler из которого отвечает за выполнение команды

    Допустимые поля
        allow_to -- acl.
            Если указан, то запрашивать команду разрешается только с хостов, перечисленных в списке.
            Можно указывать ip, можно имена хостов, можно имена хостов в виде регулярки qr/что-то/
            Маски сетей -- нет.
        tvm2_allow_* - список id TVM приложений которым можно запрашивать команду, если параметр не задан то TVM-заголовок не проверяется (пока)
            ids - в продакшене
            ids_in_testing - в тестинге
            ids_sandbox - в песочнице
            ids_sandbox_in_testing - в тестовой песочнице
        skip_tvm2_check* - ВАЖНО - пустой список означает "пропускать все".
        group -- группа (сервис). Для фильтрации/разбиения контроллеров по разным .psgi
        type -- Указывает, как надо вызывать функцию-обработчик
             для 'soap' -- выполняется SOAP-диспетчеризацию,
             по умолчанию -- просто вызывается функция handler
        no_warnings_on_require -- если флаг взведен, то при загрузке модуля будут отключены предупреждения
        custom_log -- не логировать запросы в ручку стандартными средствами
             (Plack::Middleware::IntapiAutoLogger, JsonrpcLogger); обработчик ручки может получить
             объект Yandex::Log из Intapi::request_logger($cmd) и записать в лог данные как нужно.
        custom_logger -- логировать стандартными средствами, но используя
            объект Yandex::Log указанный в custom_logger, его же бует возвращать Intapi::request_logger($cmd)
        log_response -- если выставлено, то в логи пишутся не только запросы и но и ответ
        sandbox_only -- если выставлено, то сервис будет работать только в песочнице


    Аргументы для обработчика команды (<Module>::handler)
    позиционные
    $r -- объект Plack::Request
    $multiform -- (многозначная) форма параметров (и GET, и POST).
          для перевода в Директовую форму есть Tools::multiform2directform


    Чтобы добавить новую команду intapi, надо:
      * модуль Intapi::... или Export::... с функцией handler
      * записать команду и модуль в хеш %cmd
      * записать команду в LocationMatch в etc/intapi/apache2/intapi.direct.yandex.ru.conf
      * (для тестирования на бете) записать команду в LocationMatch в etc/frontend/apache2/ppc.yandex.ru.conf
      * если для команды предусмотрен acl -- записать его в %cmd

=cut


use strict;
use warnings;
use feature 'state';

use Encode;
use SOAP::Lite;
use SOAP::Transport::HTTP;
use Readonly;
# use Carp;
# $SIG{__DIE__} = \&Carp::confess;

use Yandex::DBShards;
use Yandex::Shell;
use Yandex::Log;
use Yandex::SendMail qw/send_alert/;
use Yandex::I18n qw();
use Yandex::Trace qw/current_trace/;

use Yandex::SOAP::Transport::HTTP::Plack;
use Yandex::SOAP::UTF8Serializer;

use EnvTools;
use Rbac;
use RBACDirect;

use utf8;

Readonly our %GROUP_DISABLED_IN_PRODUCTION => (
    'secret-jsonrpc' => 1,
    'secret-plaintext' => 1,
);

our %LOG_SETTINGS;
%LOG_SETTINGS = ( no_log => 1, use_syslog => 1, syslog_prefix => 'INTAPI' ) if !%LOG_SETTINGS;

my @PPCDEVS = ( qr/^ppcdev\d{1,2}\.yandex\.ru$/ );

my $ACL_IN_TESTING = [
    @PPCDEVS,
    qw(
    ppctest-ts1-front.yandex.ru
    ppctest-ts2-front.yandex.ru
    ppctest-sandbox1-front.yandex.ru
    ppctest-sandbox2-front.yandex.ru
    ppctest-load-front.yandex.ru
    ppctest-load2-front.yandex.ru
    ppcdev-hardy.yandex.ru
    ppcdev.yandex.ru
    beta.geocontext.yandex.ru
    test.geocontext.yandex.ru
    beta-precise.geocontext.yandex.ru
    test-geocontext-precise.yandex.ru
    direct-mod-test.yandex.ru
    ppcmoddev.yandex.ru
    ppcmoddev-precise.yandex.ru
)];

my $TVM_DIRECT_SCRIPTS_TEST = 2000767;

my $TVM_DIRECT_WEB_PROD = 2000769;
my $TVM_DIRECT_WEB_TEST = 2000771;

my $TVM_DIRECT_AUTOTESTS = 2009913;
my $TVM_DIRECT_DEVELOPER = 2009921;

my @YACOADS = qw(
    yacoads1.yandex.ru
    yacoads01d.yandex.ru
    yacoads01e.yandex.ru
    yacoads01f.yandex.ru
    yacoads01h.yandex.ru
);

my @METRICA = qw(
    mtbot1.yandex.ru
    mtbot2.yandex.ru
    mtconv1.yandex.ru
    mtconv01f.yandex.ru
    mtconv01e.yandex.ru
    mtconv01i.yandex.ru
    api-metrika3.yandex.ru
    metrika-test.yandex.ru

    metrika2-1.yandex.ru
    metrika2-2.yandex.ru
    metrika2-3.yandex.ru
    metrika3-1.yandex.ru
    metrika3-2.yandex.ru

    metrika-ng1-1.yandex.ru
    metrika-ng1-2.yandex.ru
    metrika-ng2-1.yandex.ru
    metrika-ng2-2.yandex.ru
    metrika-ng3-1.yandex.ru
    metrika-ng3-2.yandex.ru

    mtback01e.yandex.ru
    mtback02e.yandex.ru
    mtback01i.yandex.ru
    mtback02i.yandex.ru
    mtback01d.yandex.ru
    mtback02d.yandex.ru

    mtcommon01t.yandex.ru
    mtcommon01-2.yandex.ru
    mtcommon01e.yandex.ru
    mtcommon01i.yandex.ru

    mtback01ep.yandex.ru
    metrika-ps.haze.yandex.ru
    networks:metrika.txt
);

my @BALANCE = qw(
    greed1e.yandex.ru
    greed2e.yandex.ru
    greed1f.yandex.ru
    greed2f.yandex.ru
    greed1g.yandex.ru
    greed2g.yandex.ru

    greed3g.yandex.ru
    greed3f.yandex.ru
    greed3e.yandex.ru
    greed1v.paysys.yandex.net
    greed2v.paysys.yandex.net
    greed3v.paysys.yandex.net
);

my @BSSOAP = (
    qr/^bssoap\d\d\w\.yandex\.ru$/,
    qr/^bssoap\d\d\w\.yabs\.yandex\.ru/,
    qr/^bssoap-pre\d\d\w\.yandex\.ru$/,
    qr/^bssoap-pre\d\d\w\.yabs.yandex\.ru$/,
);

my @BSHIVE = qw(
    zurom.bshive.yandex.net
    antilost.bshive.yandex.net
    antilost2.bshive.yandex.net
    abyss.precise.bshive.yandex.net
);

# хорошо бы сделать макросом _BSDEVNETS_, но надо уметь ipv6 и маски подсетей
my @BSDEV = (
    qw(
        bs-exchange01i.haze.yandex.net
        zurom.haze.yandex.net
    ),
    @BSHIVE,
);

my @BSINT = qw(
    bsint01e.yabs.yandex.ru
    bsint01f.yabs.yandex.ru
    bsint01i.yandex.ru
    bsint02e.yabs.yandex.ru
    bsint02f.yabs.yandex.ru
    bsint02i.yabs.yandex.ru
);

my @PPCBACK = qw(
    ppcback01i.yandex.ru
    ppcback01e.yandex.ru
    ppcback01f.yandex.ru
    sas2-1317-sas-ppc-direct-intapi-perl-32123.gencfg-c.yandex.net
    sas2-1324-sas-ppc-direct-intapi-perl-32123.gencfg-c.yandex.net
    sas2-1325-sas-ppc-direct-intapi-perl-32123.gencfg-c.yandex.net
    sas2-5078-sas-ppc-direct-intapi-perl-32123.gencfg-c.yandex.net
    sas2-5081-sas-ppc-direct-intapi-perl-32123.gencfg-c.yandex.net
    vla1-4273-vla-ppc-direct-intapi-perl-32114.gencfg-c.yandex.net
    vla1-5083-vla-ppc-direct-intapi-perl-32114.gencfg-c.yandex.net
    vla1-5093-vla-ppc-direct-intapi-perl-32114.gencfg-c.yandex.net
);

my @MODERATION = qw(
    ppcmod01e.yandex-team.ru
    ppcmod01g.yandex-team.ru
    ppcmod01e.yandex.ru
    ppcmod02e.yandex.ru
    ppcmod01g.yandex.ru
    ppcmod02g.yandex.ru
    ppcmod01i.yandex.ru
    ppcmod02i.yandex.ru
    modback01e.yandex.ru
    modback01g.yandex.ru
    modback01f.yandex.ru
    modback01h.yandex.ru
    modback01i.yandex.ru
    modback02i.yandex.ru
    modback02e.yandex.ru
    networks:directmod.txt
);

my @FRONTENDS = qw(
    ci01e.yandex.ru
    ci01f.yandex.ru
    ci01i.yandex.ru
    ci02e.yandex.ru
    ci02f.yandex.ru
    ci02i.yandex.ru
    ci03e.yandex.ru
    ci03f.yandex.ru
    ci03i.yandex.ru
    ci04e.yandex.ru
    ci04f.yandex.ru
    ci04i.yandex.ru
    networks:direct_frontends.txt
);

my @PPCSCRIPTS = qw(
    ppcscripts01e.yandex.ru
    ppcscripts01f.yandex.ru
    ppcscripts01g.yandex.ru
    ppcscripts01h.yandex.ru
    ppcscripts01i.yandex.ru
    ppcscripts02e.yandex.ru
    ppcscripts02f.yandex.ru
    ppcscripts02h.yandex.ru
    ppcscripts02i.yandex.ru
);

# временно отключаем ACL для тестовых и разработческих сред, т. к. существующий список включает далеко не всех, кто сейчас ходит в тестовый intapi
our %cmd = (
    CSRF => {
        module => 'Intapi::CSRF',
        type => 'jsonrpc',
        methods => ['get_csrf_token'],
        allow_to => $ACL_IN_TESTING,
    },
    CLegacyNews => {
        module => 'Intapi::CLegacy::News',
        type => 'jsonrpc',
        methods => ['get_news', 'get_news_by_id', 'get_news_regions_langs'],
    },
    CLegacyClientInfo => {
        module => 'Intapi::CLegacy::ClientInfo',
        type => 'jsonrpc',
        methods => ['user_info', 'get_surveys'],
    },
    CLegacyRBAC => {
        module => 'Intapi::CLegacy::RBAC',
        type => 'jsonrpc',
        methods => ['get_infoblock_rights'],
    },
    DirectEnv => {
        module => 'Intapi::DirectEnv',
        type => 'jsonrpc',
        methods => ['get_settings', 'get_setting_name'],
        allow_to => $ACL_IN_TESTING,
    },
    'file-storage' => {
        module => 'Intapi::FileStorage',
        allow_to => [
        @PPCBACK,
        @MODERATION
        ],
#        allow_to_in_testing => $ACL_IN_TESTING,
    },
    DirectConfiguration => {
        module => 'Intapi::DirectConfiguration',
    },
    idm => {
        module => 'Intapi::DostupUserManagement',
        tvm2_allow_ids => [
            2001600, # IDM: production
        ],
        tvm2_allow_ids_in_testing => [
            2001602, # IDM: Testing
            $TVM_DIRECT_AUTOTESTS,
            $TVM_DIRECT_DEVELOPER,
        ],
        tvm2_allow_ids_sandbox => [],
        tvm2_allow_ids_sandbox_in_testing => [],
    },
    CheckAgencyRightsForMetrika => {
        module => 'Export::AgencyCheckForMetrika',
        tvm2_allow_ids => [
            2000268, # faced
        ],
        tvm2_allow_ids_in_testing => [
            2000233, # faced
            $TVM_DIRECT_AUTOTESTS,
            $TVM_DIRECT_DEVELOPER,
        ],
        tvm2_allow_ids_sandbox => [],
        tvm2_allow_ids_sandbox_in_testing => [],
    },
    clients => {
        module => 'Intapi::Clients',
        allow_to => [
            'networks:custom_solutions.txt',
            'networks:geocontext.txt',
        ],
        tvm2_allow_ids_in_testing => [
            $TVM_DIRECT_AUTOTESTS,
        ],
    },
    GetRelatedUsersByClientID => {
        module => 'Intapi::GetRelatedUsersByClientID',
    },
    ClientData => {
        module => 'Intapi::ClientData',
        methods => [ 'get_by_ClientID' ],
        type => 'jsonrpc',
        tvm2_allow_ids => [
            2001277,    # CRM
            2010130,    # Metrika Schedulerd (prod)
        ],
        tvm2_allow_ids_in_testing => [
            2001283,    # CRM
            2010128,    # Metrika Schedulerd (test)
            $TVM_DIRECT_AUTOTESTS,
        ],
        tvm2_allow_ids_sandbox => [],
        tvm2_allow_ids_sandbox_in_testing => [],
        skip_tvm2_check => [
            @BSINT,
        ],
    },
    UserRole => {
        module => 'Export::UserRole',
        tvm2_allow_ids => [
            205,        # adv
            2001476,    # expert
            2001277,    # CRM
        ],
        tvm2_allow_ids_in_testing => [
            2000159,    # adv
            2001462,    # expert
            2001283,    # CRM
        ],
        tvm2_allow_ids_sandbox => [],
        tvm2_allow_ids_sandbox_in_testing => [],
    },
    ExportCmdLog => {
        module => 'Intapi::ExportCmdLog',
    },
    campaignsformetrica => {
        module => 'Intapi::CampaignsForMetrica',
        allow_to => [
            @METRICA,
        qw(
            metrika-load.yandex.ru
        ),
        ],
#        allow_to_in_testing => $ACL_IN_TESTING,
    },
    ExportCampaignsForMetrica => {
        module => 'Intapi::CampaignsForMetrica',
        tvm2_allow_ids => [
            2010134, # markedphoned
            2000268, # faced
            2000246, # Metrika Frontend (prod)
        ],
        tvm2_allow_ids_in_testing => [
            2010132, # markedphoned
            2000233, # faced
            2000234, # Metrika Frontend (test)
            $TVM_DIRECT_AUTOTESTS,
            $TVM_DIRECT_DEVELOPER,
        ],
        tvm2_allow_ids_sandbox => [],
        tvm2_allow_ids_sandbox_in_testing => [],
    },
    RequestBlockNumberDetailed => {
        module => 'Intapi::RequestBlockNumberDetailed',
        tvm2_allow_ids => [
            2001277,    # CRM
        ],
        tvm2_allow_ids_in_testing => [
            2001283,    # CRM
            $TVM_DIRECT_AUTOTESTS,
            $TVM_DIRECT_DEVELOPER,
        ],
        tvm2_allow_ids_sandbox => [],
        tvm2_allow_ids_sandbox_in_testing => [],
    },
    alive => {
        module => 'Intapi::Alive',
        ignore_stop_file => 1,
    },
    # + далее делается копия -- soap-utf8
    soap => {
        module => 'ServiceSOAP',
        group  => 'soap',
        type   => 'soap',
        # ТС и ppcdev для копирования сертификатов API с продакшна
        allow_to => [
        @PPCBACK,
        @YACOADS,
        @MODERATION,
        @BSSOAP,
        @PPCDEVS,
        qw(
            ppctest-ts1-front.yandex.ru
            ppctest-ts2-front.yandex.ru
            test-direct.yandex.ru

            bscoll.yandex.ru

            pi01f.yandex.ru
            pi02f.yandex.ru
            pi01i.yandex.ru
            pi02i.yandex.ru

            pi-rtb01e.yandex.ru
            pi-rtb01f.yandex.ru
        )],
#        allow_to_in_testing => $ACL_IN_TESTING,
    },
    SandboxService => {
        module => 'Sandbox::SandboxService',
        one_shard => 1,
        group => 'sandbox_services',
        tvm2_allow_ids => [],
        tvm2_allow_ids_in_testing => [],
        tvm2_allow_ids_sandbox => [
            2000769,    #direct-web
            2000389,    # direct-scripts
        ],
        tvm2_allow_ids_sandbox_in_testing => [
            2000771,    # direct-web-test
            2000767,    # direct-scripts-test
            $TVM_DIRECT_AUTOTESTS,
            $TVM_DIRECT_DEVELOPER,
        ],
        type => 'jsonrpc',
        methods => [qw(
            create_user
            get_current_state
            drop_user
            drop_fin_ops_counter
            save_app_access
        )],
        sandbox_only => 1,
        log_response => 1
    },
    FakeBlackbox => {
        module => 'Sandbox::FakeBlackbox',
        one_shard => 1,
        group => 'fake_services',
    },
    FakeBalance => {
        module => 'Sandbox::FakeBalanceXMLRPC',
        one_shard => 1,
        group => 'fake_services',
    },
    FakePassport => {
        module => 'Sandbox::FakePassport',
        one_shard => 1,
        group => 'fake_services',
    },
    FakeBS => {
        module => 'Sandbox::FakeBSSOAP',
        one_shard => 1,
        group => 'fake_services',
    },
    FakeMetrika => {
        module => 'Sandbox::FakeMetrika',
        one_shard => 1,
        group => 'fake_services',
    },
    FakeBSInfo => {
        module => 'Sandbox::FakeBSInfo',
        one_shard => 1,
        group => 'fake_services',
    },
    FakeYaMoney => {
        module => 'Sandbox::FakeYaMoney',
        one_shard => 1,
        group => 'fake_services',
    },
    AutobudgetAlerts => {
        module => "Intapi::AutobudgetAlerts",
        type => 'jsonrpc',
        group => 'main-jsonrpc',
        methods => [ qw/
            ordersNotExceededBudget
            ordersWithCpaWarnings
        /],
    },
    CampaignInfo => {
        module => "Intapi::CampaignInfo",
        type => 'jsonrpc',
        group => 'main-jsonrpc',
        methods => [ qw/
            getCampaignManager
            getCampaignInfo
        /],
        tvm2_allow_ids => [
            2001277,    # CRM
        ],
        tvm2_allow_ids_in_testing => [
            2001283,    # CRM
            $TVM_DIRECT_AUTOTESTS,
        ],
        tvm2_allow_ids_sandbox => [],
        tvm2_allow_ids_sandbox_in_testing => [],
        skip_tvm2_check => [
            # gorynych
            'gorynych01i.yandex.ru',
            'gorynych-vla-1.vla.yp-c.yandex.net',
            'gorynych-vla-2.vla.yp-c.yandex.net',
            'gorynych-sas-1.sas.yp-c.yandex.net',
            'gorynych-sas-2.sas.yp-c.yandex.net',
        ],
    },
    AutobudgetPrices => {
        module => "Intapi::AutobudgetPrices",
        type => 'jsonrpc',
        group => 'main-jsonrpc',
        methods => [ qw/
            set
        /],
        allow_to => [
            @BSSOAP,
        ],
        # ручка пишет все данные в AutobudgetPrices_set.log
        custom_log => 1,
    },
    AutobudgetOptimizedBy => {
        module => "Intapi::AutobudgetOptimizedBy",
        type => 'jsonrpc',
        group => 'main-jsonrpc',
        methods => [ qw/
            setOrder
            setFilter
        /],
        allow_to => [
            @BSSOAP,
        ],
        # ручка пишет отдельный лог на каждый метод в protected/setOrder.log и setFilter.log
        custom_log => 1,
    },
    TestDataGenerator => {
        module => 'Intapi::TestDataGenerator',
        type => 'jsonrpc',
        group => 'secret-jsonrpc',
        methods => [ qw/
            get_users
            get_campaigns
            get_banners
        /],
    },
    TestUsers => {
        module    => 'Intapi::TestUsers',
        type      => 'jsonrpc',
        group     => 'main-jsonrpc',
        methods   => [ 'get_all' ],
        tvm2_allow_ids => [
            2000389,    # direct-scripts
            2000767,    # direct-scripts-test
        ],
        tvm2_allow_ids_in_testing => [
            2000767,    # direct-scripts-test
        ],
        tvm2_allow_ids_sandbox => [],
        tvm2_allow_ids_sandbox_in_testing => [],
    },
    ExportCampaigns => {
        module => 'Intapi::ExportCampaigns',
        type => 'jsonrpc',
        group => 'jsonrpc',
        methods => [ 'get_managed_campaigns' ],

    },
    TestScriptRun => {
        module => 'Intapi::TestScriptRun',
        type => 'jsonrpc',
        group => 'secret-jsonrpc',
        methods => [ qw/
            get_whitelist
            run_script
        /],
    },
    FakeAdmin => {
        module => 'Intapi::FakeAdmin',
        type => 'jsonrpc',
        group => 'secret-jsonrpc',
        methods => [ qw/
            AddEvents
            GetUserShard
            ReshardUser
            FakeUpdateRetargetingGoals
            FakeConvertCurrencyClient
            FakeBalanceNotificationNDS
            FakeSetAllowBudgetAccountForAgency
            FakeClientParams
            FakeGetClientParams
            FakeCampaignParams
            FakeGetCampaignParams
            FakeCreateDynamicAdgroup
            FakeGroupParams
            FakeGetGroupParams
            FakeBannerParams
            FakeGetBannerParams
            FakePhrasesParams
            FakeGetPhrasesParams
            FakeCreateReportQueue
            FakeBalanceNotification
            FakeRemoveAgencyClientRelation
            SetDiscount
            FakeClientUnitsBalance
            FakeWithdrawClientUnits
            FakeClearClientSpentUnits
            ShowMailLogs
            ShowSmsLogs
            FakeCreateClient
            TakeRedisLock
            DropRedisLock
            SendCampaignToBalance
        /],
    },
    SearchBanners => {
        allow_to => \@MODERATION,
#        allow_to_in_testing => $ACL_IN_TESTING,
        group => 'main-jsonrpc',
        methods => [ qw/
            search
            advanced_search
        / ],
        module => 'Intapi::SearchBanners',
        type => 'jsonrpc',
    },
    SearchBannersTvm => {
        tvm2_allow_ids => [
            2002030,    # Супермодерация
        ],
        tvm2_allow_ids_in_testing => [
            2002032,    # Тестовая Супермодерация
            $TVM_DIRECT_AUTOTESTS,
            2010046,    # автотесты модерации
        ],
        tvm2_allow_ids_sandbox => [],
        tvm2_allow_ids_sandbox_in_testing => [],
        group => 'main-jsonrpc',
        methods => [ qw/
            search
            advanced_search
        / ],
        module => 'Intapi::SearchBanners',
        type => 'jsonrpc',
    },
    CampaignsSumsForBS => {
        allow_to => [
            @BSSOAP,
            @BSDEV,
            @BSINT,
        ],
#        allow_to_in_testing => $ACL_IN_TESTING,
        group => 'main-jsonrpc',
        methods => [ qw/
            get
        / ],
        module => 'Intapi::CampaignsSumsForBS',
        type => 'jsonrpc',
    },
    BSExportPreprodParams => {
        allow_to => [
            @BSSOAP,
        ],
        group => 'main-jsonrpc',
        methods => [ qw/
            get
        / ],
        module => 'Intapi::BSExportPreprodParams',
        type => 'jsonrpc',
    },
    ResyncCampaigns => {
        allow_to => [
            @BSINT,
        ],
        module => 'Intapi::ResyncCampaigns',
        type => 'jsonrpc',
#        allow_to_in_testing => $ACL_IN_TESTING,
        group => 'main-jsonrpc',
        methods => [ 'add' ],
    },
    Mirrors => {
        module => 'Intapi::Mirrors',
        type => 'jsonrpc',
        group => 'main-jsonrpc',
        methods => [
            'get_domain_filter',
            'get_domain_filter_for_bs',
        ],
    },
    BsFront => {
        module => 'Intapi::BsFront',
        type => 'jsonrpc',
        group => 'main-jsonrpc',
        methods => [
            'auth',
            'change_notify',
            'spawn_creatives',
            'get_client_feeds',
            'turbolanding_change_notify',
        ],
    },
    CurrencyRates => {
        module => 'Intapi::CurrencyRates',
        type => 'jsonrpc',
        group => 'main-jsonrpc',
        methods => [qw(
            get_YND_FIXED_rates
        )],
    },
    ExportRightsUidToCid => {
        module => 'Intapi::ExportRightsUidToCid',
    },
    CheckRightsUidToCidForMetrika => {
        module => 'Intapi::ExportRightsUidToCid',
        tvm2_allow_ids => [
            2010130, # schedulerd
        ],
        tvm2_allow_ids_in_testing => [
            2010128, # schedulerd
            $TVM_DIRECT_AUTOTESTS,
            $TVM_DIRECT_DEVELOPER,
        ],
        tvm2_allow_ids_sandbox => [],
        tvm2_allow_ids_sandbox_in_testing => [],
    },
    BannerLand => {
        module => 'Intapi::BannerLand',
        type => 'jsonrpc',
        group => 'main-jsonrpc',
        methods => [qw/
            setStatusForAdGroups
            setStatusForAdGroupsWithOrderID
        /],
    },
    BalanceRevise => {
        module => 'Intapi::BalanceRevise',
        type => 'jsonrpc',
        group => 'main-jsonrpc',
        methods => [qw/
            getPaidCidsNotInBS
        /],
        tvm2_allow_ids => [
            184, # balance-dcs
        ],
        tvm2_allow_ids_in_testing => [
            2002192, # paysys-balance-dcs-test
            $TVM_DIRECT_AUTOTESTS,
            $TVM_DIRECT_DEVELOPER,
        ],
        tvm2_allow_ids_sandbox => [],
        tvm2_allow_ids_sandbox_in_testing => [],
    },
    CurrenciesList => {
        # без TVM: DIRECT-93984
        module  => 'Intapi::CurrenciesList'
    },
    CampaignTypesList => {
        # без TVM: DIRECT-93985
        module  => 'Intapi::CampaignTypesList'
    },
    JavaExportGeoRegions => {
        # без TVM: DIRECT-93987
        module  => 'Intapi::JavaExport::GeoRegions'
    },
    Notification => {
        module => 'Intapi::Notification',
        tvm2_allow_ids => [
            2000389, # direct-scripts
            2000390, # direct-intapi
            2000769, # direct-web
            2000773, # direct-api
        ],
        tvm2_allow_ids_in_testing => [
            2000767, # direct-scripts-test
            2000693, # direct-intapi-test
            2000771, # direct-web
            2000775, # direct-api
        ],
        tvm2_allow_ids_sandbox => [],
        tvm2_allow_ids_sandbox_in_testing => [],
    },
    CampaignUnarc => {
        module => 'Intapi::CampaignUnarc',
        tvm2_allow_ids => [
            2000389, # direct-scripts
            2000390, # direct-intapi
        ],
        tvm2_allow_ids_in_testing => [
            2000767, # direct-scripts-test
            2000693, # direct-intapi-test
        ],
        tvm2_allow_ids_sandbox => [
            2000920, # direct-intapi-sandbox
        ],
        tvm2_allow_ids_sandbox_in_testing => [
            2000926, # direct-intapi-sandbox-test
        ],
    },
    CampaignsCopy => {
        module => 'Intapi::CampaignsCopy',
        tvm2_allow_ids => [
            $TVM_DIRECT_WEB_PROD,
        ],
        tvm2_allow_ids_in_testing => [
            $TVM_DIRECT_WEB_TEST,
            # для тестов из direct/apps/manual-tests
            $TVM_DIRECT_SCRIPTS_TEST,
            $TVM_DIRECT_DEVELOPER,
        ],
    },
    CalculateCampaignStatusModerate => {
        module => 'Intapi::CalculateCampaignStatusModerate',
        tvm2_allow_ids => [
            2000389, # direct-scripts
            2000390, # direct-intapi
        ],
        tvm2_allow_ids_in_testing => [
            2000767, # direct-scripts-test
            2000693, # direct-intapi-test
        ],
        tvm2_allow_ids_sandbox => [
            2000920, # direct-intapi-sandbox
        ],
        tvm2_allow_ids_sandbox_in_testing => [
            2000926, # direct-intapi-sandbox-test
        ],
    },
    CalculateCampaignStatusModerateReadOnly => {
        module => 'Intapi::CalculateCampaignStatusModerateReadOnly',
        tvm2_allow_ids => [
            2000389, # direct-scripts
            2000390, # direct-intapi
        ],
        tvm2_allow_ids_in_testing => [
            2000767, # direct-scripts-test
            2000693, # direct-intapi-test
        ],
        tvm2_allow_ids_sandbox => [
            2000920, # direct-intapi-sandbox
        ],
        tvm2_allow_ids_sandbox_in_testing => [
            2000926, # direct-intapi-sandbox-test
        ],
    },
    PrepareAndValidatePayCamp => {
        module => 'Intapi::PrepareAndValidatePayCamp',
        tvm2_allow_ids => [
            2000769, # direct-web
            2000390, # direct-intapi
        ],
        tvm2_allow_ids_in_testing => [
            2000771, # direct-web-test
            2000693, # direct-intapi-test
            $TVM_DIRECT_DEVELOPER,
        ],
        tvm2_allow_ids_sandbox => [],
        tvm2_allow_ids_sandbox_in_testing => [],
    },
    PayForAll => {
        module => 'Intapi::PayForAll',
        tvm2_allow_ids => [
            2000769, # direct-web
            2000390, # direct-intapi
        ],
        tvm2_allow_ids_in_testing => [
            2000771, # direct-web-test
            2000693, # direct-intapi-test
            $TVM_DIRECT_DEVELOPER,
        ],
        tvm2_allow_ids_sandbox => [],
        tvm2_allow_ids_sandbox_in_testing => [],
    },
    CampaignStatistics => {
        module => 'Intapi::CampaignStatistics',
        tvm2_allow_ids => [
            2000769, # direct-web
        ],
        tvm2_allow_ids_in_testing => [
            2000771, # direct-web-test
            $TVM_DIRECT_DEVELOPER,
        ],
        tvm2_allow_ids_sandbox => [],
        tvm2_allow_ids_sandbox_in_testing => [],
    },
);

$cmd{'soap-utf8'} = $cmd{soap};

# если доступ к ручке контроллируется через TVM - добавляем в тестовых конфигурациях доступ приложению "разработчик"
for my $conf (values %cmd) {
    for my $key (qw/tvm2_allow_ids_sandbox_in_testing tvm2_allow_ids_in_testing/) {
        if (exists $conf->{$key}) {
            push @{ $conf->{$key} }, $TVM_DIRECT_DEVELOPER;
        }
    }
}

=head2 INTAPI_LANG

    Язык сообщений/ошибок и прочих локализуемых текстов в INTAPI

=cut

use constant INTAPI_LANG => 'en';

=head2 execute_cmd

=cut
sub execute_cmd
{
    my ($r, $cmd, $multiform) = @_;

    # если существует стоп-файл - ничего не делаем
    # (если только в описании метода не сказано игнорировать стоп-файл)
    if (!$cmd{$cmd}->{ignore_stop_file} && -f $Settings::WEB_STOP_FLAG) {
        return {text => "Interface stopped", code => 500,};
    }

    cmd_require($cmd);

    # для сублокейшена /help всегда отдаем pod по соответствующему модулю
    if($r->path_info eq '/help'){
        (my $file = $cmd{$cmd}->{module}) =~ s!::!/!g;
        $file = $INC{"$file.pm"};
        return {code => 500, text => "can't find any documentation"} unless $file;

        my $text = yash_qx("podselect -section NAME/!.+ -section DESCRIPTION/!.+ ".yash_quote($file)."|pod2text-utf8");
        $text = Encode::decode_utf8($text);
        return {text => $text}
    }

    my $to_eval = $cmd{$cmd}->{module}.'::handler($r, $multiform)';
    my $response = eval $to_eval;
    die $@ if $@;

    return $response;
}


sub execute_cmd_soap
{
    my ($r, $cmd, $multiform) = @_;

    my $cmd_hash = $cmd{$cmd};
    cmd_require($cmd);

    $cmd{$cmd}->{server} ||= Yandex::SOAP::Transport::HTTP::Plack
    -> serializer(Yandex::SOAP::UTF8Serializer->new('utf8'))
    -> deserializer(Yandex::SOAP::UTF8Deserializer->new('utf8'))
    -> dispatch_to($cmd{$cmd}->{module})
    -> on_fault(sub {
        my ($self, @args) = @_;

        my $action = $self->action;
        my $content = "SOAP fault in action $action:\n" . join("\n", grep {$_} @args[1..$#args]);
        send_alert($content, "SOAP fault");
    });

    my $response = $cmd{$cmd}->{server}->handler(@_);
    if ($response && ref($response) eq 'ARRAY' && $response->[0] == 500) {
        # отдаём SOAP Fault c 200м кодом
        # если отдавать 500й, то вместо XMLки будет стандартная HTML-страница "Сервис временно недоступен"
        $response->[0] = 200;
    }

    return $response;
}

sub handler
{
    my ($r, $cmd, $multiform) = @_;

    die "unknown cmd $cmd" unless exists $cmd{$cmd} && $cmd{$cmd}->{module};
    die "unknown cmd $cmd" if $cmd{$cmd}->{sandbox_only} && !is_sandbox();

    # добавляем тип к имени метода, если он есть
    my $trace_method_name = current_trace()->method();
    $trace_method_name = join '.', grep { length } $cmd{$cmd}->{type}, $trace_method_name;
    current_trace()->method($trace_method_name);


    my $one_shard = exists $cmd{$cmd} && $cmd{$cmd}->{one_shard} ? 1 : 0;
    local $Yandex::DBShards::STRICT_SHARD_DBNAMES = $one_shard ? 0 : 1;
    Yandex::DBShards::clear_cache();
    Rbac::clear_cache();
    RBACDirect::clear_cache();
    my $lang_guard = Yandex::I18n::init_i18n_guard(INTAPI_LANG);

    my $response;
    my $type = $cmd{$cmd}->{type}||'';
    if ($type eq 'soap'){
        $response = execute_cmd_soap($r, $cmd, $multiform);
    } else {
        $response = execute_cmd($r, $cmd, $multiform);
    }

    return $response;
}


=head2 cmd_require

    загружает модуль, указанный для заданной команды

=cut
sub cmd_require(@)
{
    my @cmds = @_;

    for my $cmd (@cmds){
        # если указано -- отключаем предупреждения при загрузке модуля
        local $^W = $cmd{$cmd}->{no_warnings_on_require} ? 0 : 1;
        eval "require $cmd{$cmd}->{module}";
        die $@ if $@;
    }

    return;
}


=head2 get_group_cmds

=cut
sub get_group_cmds($)
{
    my ($group) = @_;

    return grep {($cmd{$_}->{group} || '') eq $group} keys %Intapi::cmd;
}


=head2 make_urlmap

    Параметры
    $baseurl
    @cmds -- список команд (контроллеров)

    Возвращает urlmap для переданных команд

=cut
sub make_urlmap($@)
{
    my ($baseurl, @cmds) = @_;
    my $acl_param_name = is_production() ? 'allow_to' : 'allow_to_in_testing';
    return [ map { ["$baseurl/$_", {cmd => $_, params => { $cmd{$_}->{$acl_param_name} ? (allow_to => $cmd{$_}->{$acl_param_name}) : () }}] } @cmds ];
}

=head2 get_commands_by_type

    Параметры
    $type - тип, команды которого нужно вернуть, например jsonrpc

    Возвращает ссылку на усечённый %cmd, в котором только команды с переданным типом

=cut
sub filter_cmd
{
    my %O = @_;

    die unless $O{type} || $O{group};

    my %filtered_cmd = %cmd;
    if ( $O{type} ){
        %filtered_cmd = map { $_ => $cmd{$_} } grep { ($cmd{$_}->{type}||'') eq $O{type} } keys %filtered_cmd;
    }
    if ( $O{group} ){
        %filtered_cmd = map { $_ => $cmd{$_} } grep { ($cmd{$_}->{group}||'') eq $O{group} } keys %filtered_cmd;
    }

    return \%filtered_cmd;
}

=head2 request_logger($cmd)

    Возвращает объект Yandex::Log для логгирования запросов к $cmd.

=cut

sub request_logger {
    my ($cmd) = @_;

    state $cmd_to_log = {};

    if (!defined $cmd_to_log->{$cmd}) {
        $cmd_to_log->{$cmd} = exists $cmd{$cmd} && exists $cmd{$cmd}->{custom_logger}
            ? $cmd{$cmd}->{custom_logger}
            : Yandex::Log->new(
                %LOG_SETTINGS,
                log_file_name => "$cmd.log",
            );
    }

    return $cmd_to_log->{$cmd};
}

1;
