package Intapi::Alive;

# $Id$

=head1 NAME

    Intapi::Alive

=head1 DESCRIPTION

    Проверить возможность коннекта к основным БД
    и заполненность раздела c логами на сервере

=cut

use strict;
use warnings;
use Filesys::Df qw/df/;

use Yandex::SendMail;

use Settings;
use Yandex::DBTools;

use utf8;

# партиции для проверки заполненности
my @PARTITIONS = ('/var', $Settings::LOG_ROOT);
# какая часть диска должна быть свободна (1%)
my $FREE_SPACE_BORDER = 0.01;
# какая часть inode должны быть свободны (1%)
my $FREE_INODES_BORDER = 0.01;
# файлик для проверки доступности фс на запись
# /tmp на прод машинах смонтирован на большой раздел (там же, где и логи) - должен обрабатывать ситуацию с read-only фс при развалившемся рейде.
my $FS_WRITABLE_CHECK_FILE = '/tmp/direct-fs-writable-check';

sub handler {
    my ($r, $multiform) = @_;

    my @errors;

    # проверяем заполненность диска
    for my $path (@PARTITIONS) {
        my $df = df($path, 1);
        if (!$df) {
            push @errors, {test => "df:$path", error => "no df data"};
            next;
        }
        my $space_proc = $df->{bfree} / ($df->{blocks}||1);
        if ($space_proc < $FREE_SPACE_BORDER) {
            push @errors, {test => "df:space:$path", error => "not enough space: $space_proc < $FREE_SPACE_BORDER"};
        }
        my $inodes_proc = $df->{ffree} / ($df->{files}||1);
        if ($inodes_proc < $FREE_INODES_BORDER) {
            push @errors, {test => "df:inodes:$path", error => "not enough inodes number: $inodes_proc < $FREE_INODES_BORDER"};
        }
    }

    eval {
        open(my $fh, '>', $FS_WRITABLE_CHECK_FILE . ".$$") or die "$!\n";
        print $fh "ok\n";
        close($fh) or die "$!\n";
        unlink($FS_WRITABLE_CHECK_FILE . ".$$") or die "$!\n";
    };
    if ($@) {
        push @errors, {test => "fs:writable:$FS_WRITABLE_CHECK_FILE.$$", error => "fs is not writable: $@"};
    }

    if (! eval { Yandex::DBTools::soft_reconnect_db(PPCDICT); } ) {
        push @errors, {test => "db:ppcdict", error => $@};
    }
    if (!@errors) {
        return {text => 'ok'};
    } else {
        my $text = join '', map {"$_->{test} - $_->{error}\n"} @errors;
        send_alert($text, "alive check");
        return {code => 500, text => "errors: ".join(", ", map {$_->{test}} @errors)};
    }
}

1;
